<?php
/**
 * Migration Script: Add Video Association to Course Materials
 * This script adds video_id column to course_materials table for associating materials with specific video lessons
 */

require_once __DIR__ . '/../includes/session_start.php';
require_once __DIR__ . '/../config/db_connect.php';

// Set execution time and memory limits for migration
ini_set('max_execution_time', 300);
ini_set('memory_limit', '256M');

echo "<h2>Course Materials Video Association Migration</h2>\n";
echo "<p>Adding video_id column to course_materials table...</p>\n";

try {
    // Start transaction for safety
    $conn->begin_transaction();
    
    // Check if video_id column already exists
    $check_column_query = "SHOW COLUMNS FROM course_materials LIKE 'video_id'";
    $column_result = $conn->query($check_column_query);
    
    if ($column_result->num_rows == 0) {
        echo "<p>✓ Adding video_id column to course_materials table...</p>\n";
        
        // Add video_id column after course_id
        $add_column_sql = "ALTER TABLE course_materials ADD COLUMN video_id INT(11) DEFAULT NULL AFTER course_id";
        if ($conn->query($add_column_sql)) {
            echo "<p style='color: green;'>✓ Successfully added video_id column</p>\n";
        } else {
            throw new Exception("Failed to add video_id column: " . $conn->error);
        }
        
        // Add index for video_id
        $add_index_sql = "ALTER TABLE course_materials ADD KEY `video_id` (`video_id`)";
        if ($conn->query($add_index_sql)) {
            echo "<p style='color: green;'>✓ Successfully added video_id index</p>\n";
        } else {
            throw new Exception("Failed to add video_id index: " . $conn->error);
        }
        
        // Check if videos table exists before adding foreign key
        $check_videos_table = "SHOW TABLES LIKE 'videos'";
        $videos_table_result = $conn->query($check_videos_table);
        
        if ($videos_table_result->num_rows > 0) {
            // Add foreign key constraint (videos table uses 'id' not 'video_id')
            $add_fk_sql = "ALTER TABLE course_materials ADD CONSTRAINT `course_materials_video_fk` FOREIGN KEY (`video_id`) REFERENCES `videos` (`id`) ON DELETE SET NULL";
            if ($conn->query($add_fk_sql)) {
                echo "<p style='color: green;'>✓ Successfully added video_id foreign key constraint</p>\n";
            } else {
                echo "<p style='color: orange;'>⚠ Warning: Could not add foreign key constraint (videos table may have different structure): " . $conn->error . "</p>\n";
                // Don't throw exception here as the column addition is more important
            }
        } else {
            echo "<p style='color: orange;'>⚠ Warning: Videos table not found. Foreign key constraint not added.</p>\n";
        }
        
    } else {
        echo "<p style='color: blue;'>ℹ video_id column already exists in course_materials table</p>\n";
    }
    
    // Check current table structure
    echo "<h3>Updated Course Materials Table Structure:</h3>\n";
    $describe_query = "DESCRIBE course_materials";
    $describe_result = $conn->query($describe_query);
    
    if ($describe_result) {
        echo "<table border='1' style='border-collapse: collapse; margin: 10px 0;'>\n";
        echo "<tr><th>Field</th><th>Type</th><th>Null</th><th>Key</th><th>Default</th><th>Extra</th></tr>\n";
        while ($row = $describe_result->fetch_assoc()) {
            $highlight = ($row['Field'] == 'video_id') ? 'style="background-color: #d4edda;"' : '';
            echo "<tr $highlight>";
            echo "<td>" . htmlspecialchars($row['Field']) . "</td>";
            echo "<td>" . htmlspecialchars($row['Type']) . "</td>";
            echo "<td>" . htmlspecialchars($row['Null']) . "</td>";
            echo "<td>" . htmlspecialchars($row['Key']) . "</td>";
            echo "<td>" . htmlspecialchars($row['Default'] ?? 'NULL') . "</td>";
            echo "<td>" . htmlspecialchars($row['Extra']) . "</td>";
            echo "</tr>\n";
        }
        echo "</table>\n";
    }
    
    // Show sample of existing materials
    echo "<h3>Sample Course Materials (showing video_id column):</h3>\n";
    $sample_query = "SELECT material_id, title, course_id, video_id, category, date_added FROM course_materials ORDER BY date_added DESC LIMIT 5";
    $sample_result = $conn->query($sample_query);
    
    if ($sample_result && $sample_result->num_rows > 0) {
        echo "<table border='1' style='border-collapse: collapse; margin: 10px 0;'>\n";
        echo "<tr><th>ID</th><th>Title</th><th>Course ID</th><th>Video ID</th><th>Category</th><th>Date Added</th></tr>\n";
        while ($row = $sample_result->fetch_assoc()) {
            echo "<tr>";
            echo "<td>" . htmlspecialchars($row['material_id']) . "</td>";
            echo "<td>" . htmlspecialchars($row['title']) . "</td>";
            echo "<td>" . htmlspecialchars($row['course_id'] ?? 'NULL') . "</td>";
            echo "<td>" . htmlspecialchars($row['video_id'] ?? 'NULL') . "</td>";
            echo "<td>" . htmlspecialchars($row['category']) . "</td>";
            echo "<td>" . htmlspecialchars($row['date_added']) . "</td>";
            echo "</tr>\n";
        }
        echo "</table>\n";
    } else {
        echo "<p>No course materials found in database.</p>\n";
    }
    
    // Commit transaction
    $conn->commit();
    
    echo "<h3 style='color: green;'>✅ Migration completed successfully!</h3>\n";
    echo "<p><strong>Next steps:</strong></p>\n";
    echo "<ul>\n";
    echo "<li>Update course-materials.php form to include video/lesson selection</li>\n";
    echo "<li>Update course-detail.php to display materials associated with each video lesson</li>\n";
    echo "<li>Test the video-material association functionality</li>\n";
    echo "</ul>\n";
    
} catch (Exception $e) {
    // Rollback transaction on error
    $conn->rollback();
    echo "<h3 style='color: red;'>❌ Migration failed!</h3>\n";
    echo "<p style='color: red;'>Error: " . htmlspecialchars($e->getMessage()) . "</p>\n";
    echo "<p>Transaction rolled back. No changes were made to the database.</p>\n";
}

echo "<hr>\n";
echo "<p><a href='../teacher/course-materials.php'>← Back to Course Materials</a></p>\n";
?>
