<?php
/**
 * Database Migration: Add Video-Specific Content Support
 * 
 * This migration adds video_id columns to assignments, quizzes, and course_materials tables
 * to enable teachers to create content specific to individual video lessons.
 */

require_once __DIR__ . '/../config/db_connect.php';

function runVideoContentMigration($conn) {
    $migrations = [];
    $results = [];
    
    // 1. Add video_id column to assignments table
    $migrations[] = [
        'name' => 'Add video_id to assignments table',
        'sql' => "ALTER TABLE assignments ADD COLUMN video_id INT NULL AFTER course_id, 
                  ADD INDEX idx_assignments_video_id (video_id),
                  ADD CONSTRAINT fk_assignments_video_id 
                  FOREIGN KEY (video_id) REFERENCES videos(video_id) ON DELETE SET NULL"
    ];
    
    // 2. Add video_id column to quizzes table
    $migrations[] = [
        'name' => 'Add video_id to quizzes table',
        'sql' => "ALTER TABLE quizzes ADD COLUMN video_id INT NULL AFTER course_id,
                  ADD INDEX idx_quizzes_video_id (video_id),
                  ADD CONSTRAINT fk_quizzes_video_id 
                  FOREIGN KEY (video_id) REFERENCES videos(video_id) ON DELETE SET NULL"
    ];
    
    // 3. Add video_id column to course_materials table
    $migrations[] = [
        'name' => 'Add video_id to course_materials table',
        'sql' => "ALTER TABLE course_materials ADD COLUMN video_id INT NULL AFTER course_id,
                  ADD INDEX idx_materials_video_id (video_id),
                  ADD CONSTRAINT fk_materials_video_id 
                  FOREIGN KEY (video_id) REFERENCES videos(video_id) ON DELETE SET NULL"
    ];
    
    // 4. Add content_type column to track what type of content is linked to videos
    $migrations[] = [
        'name' => 'Add content_type to videos table',
        'sql' => "ALTER TABLE videos ADD COLUMN linked_content_types JSON NULL COMMENT 'Tracks what content types are linked to this video'"
    ];
    
    echo "<h2>🚀 Video-Specific Content Support Migration</h2>";
    
    foreach ($migrations as $migration) {
        echo "<div class='migration-step'>";
        echo "<h4>⚡ " . $migration['name'] . "</h4>";
        
        try {
            $result = $conn->query($migration['sql']);
            if ($result) {
                echo "<p style='color: green;'>✅ Success: " . $migration['name'] . "</p>";
                $results[] = ['status' => 'success', 'migration' => $migration['name']];
            } else {
                echo "<p style='color: red;'>❌ Failed: " . $conn->error . "</p>";
                $results[] = ['status' => 'error', 'migration' => $migration['name'], 'error' => $conn->error];
            }
        } catch (Exception $e) {
            echo "<p style='color: orange;'>⚠️ Note: " . $e->getMessage() . " (may already exist)</p>";
            $results[] = ['status' => 'note', 'migration' => $migration['name'], 'note' => $e->getMessage()];
        }
        
        echo "</div><hr>";
    }
    
    return $results;
}

// Check if this is being run directly
if (basename(__FILE__) == basename($_SERVER['PHP_SELF'])) {
    echo "<!DOCTYPE html><html><head><title>Video Content Migration</title>";
    echo "<style>
        body { font-family: Arial, sans-serif; margin: 40px; }
        .migration-step { background: #f8f9fa; padding: 15px; border-radius: 8px; margin: 10px 0; }
        h2 { color: #2563eb; }
        h4 { color: #374151; }
        hr { margin: 20px 0; border: none; border-top: 1px solid #e5e7eb; }
    </style></head><body>";
    
    if (isset($conn)) {
        $results = runVideoContentMigration($conn);
        
        echo "<h3>📊 Migration Summary</h3>";
        $success_count = count(array_filter($results, function($r) { return $r['status'] === 'success'; }));
        $total_count = count($results);
        
        echo "<p><strong>Completed:</strong> {$success_count}/{$total_count} migrations</p>";
        
        if ($success_count === $total_count) {
            echo "<div style='background: #d1fae5; padding: 15px; border-radius: 8px; color: #065f46;'>";
            echo "<h4>🎉 All migrations completed successfully!</h4>";
            echo "<p>The database is now ready for video-specific content creation.</p>";
            echo "</div>";
        }
    } else {
        echo "<p style='color: red;'>❌ Database connection not available</p>";
    }
    
    echo "</body></html>";
}
?>
