<?php
/**
 * Enhanced SITA Assessment System
 * Adds support for flexible question types: Quiz + Document Submissions
 */

require_once '../config/db_connect.php';

if (!isset($conn)) {
    die("Database connection failed. Please check your database configuration.");
}

echo "<h2>🔧 Enhancing SITA Assessment System</h2>";
echo "<div style='font-family: Arial, sans-serif; max-width: 800px; margin: 0 auto; padding: 20px;'>";

$updates_applied = [];
$errors = [];

try {
    // 1. Enhance sita_assessment_questions table
    echo "<h3>📝 Updating Assessment Questions Table</h3>";
    
    // Add new columns for enhanced question types
    $columns_to_add = [
        "ADD COLUMN IF NOT EXISTS answer_type ENUM('quiz_only', 'document_only', 'quiz_and_document') DEFAULT 'quiz_only' AFTER question_type",
        "ADD COLUMN IF NOT EXISTS requires_file_upload TINYINT(1) DEFAULT 0 AFTER answer_type",
        "ADD COLUMN IF NOT EXISTS allowed_file_formats JSON AFTER requires_file_upload",
        "ADD COLUMN IF NOT EXISTS max_file_size_mb INT DEFAULT 10 AFTER allowed_file_formats",
        "ADD COLUMN IF NOT EXISTS min_files_required INT DEFAULT 0 AFTER max_file_size_mb",
        "ADD COLUMN IF NOT EXISTS max_files_allowed INT DEFAULT 1 AFTER min_files_required",
        "ADD COLUMN IF NOT EXISTS file_upload_instructions TEXT AFTER max_files_allowed",
        "ADD COLUMN IF NOT EXISTS quiz_weight_percentage DECIMAL(5,2) DEFAULT 100.00 AFTER file_upload_instructions",
        "ADD COLUMN IF NOT EXISTS document_weight_percentage DECIMAL(5,2) DEFAULT 0.00 AFTER quiz_weight_percentage"
    ];

    foreach ($columns_to_add as $column_sql) {
        $full_sql = "ALTER TABLE sita_assessment_questions $column_sql";
        if ($conn->query($full_sql) === TRUE) {
            echo "<p>✅ Added column: " . substr($column_sql, 18, 50) . "...</p>";
        } else {
            // Check if column already exists (ignore duplicate column errors)
            if (strpos($conn->error, 'Duplicate column') === false) {
                $errors[] = "Error adding column: " . $conn->error;
            }
        }
    }

    // 2. Enhance sita_assessments table for global settings
    echo "<h3>⚙️ Updating Assessment Settings Table</h3>";
    
    $assessment_columns = [
        "ADD COLUMN IF NOT EXISTS allow_mixed_questions TINYINT(1) DEFAULT 1 AFTER max_attempts",
        "ADD COLUMN IF NOT EXISTS global_file_formats JSON AFTER allow_mixed_questions",
        "ADD COLUMN IF NOT EXISTS require_all_questions TINYINT(1) DEFAULT 1 AFTER global_file_formats",
        "ADD COLUMN IF NOT EXISTS auto_grade_quiz_parts TINYINT(1) DEFAULT 1 AFTER require_all_questions",
        "ADD COLUMN IF NOT EXISTS admin_review_required TINYINT(1) DEFAULT 1 AFTER auto_grade_quiz_parts"
    ];

    foreach ($assessment_columns as $column_sql) {
        $full_sql = "ALTER TABLE sita_assessments $column_sql";
        if ($conn->query($full_sql) === TRUE) {
            echo "<p>✅ Added assessment setting: " . substr($column_sql, 18, 50) . "...</p>";
        } else {
            if (strpos($conn->error, 'Duplicate column') === false) {
                $errors[] = "Error adding assessment column: " . $conn->error;
            }
        }
    }

    // 3. Enhance sita_assessment_answers table for file uploads
    echo "<h3>📄 Updating Assessment Answers Table</h3>";
    
    $answer_columns = [
        "ADD COLUMN IF NOT EXISTS uploaded_files JSON AFTER file_upload",
        "ADD COLUMN IF NOT EXISTS quiz_score DECIMAL(5,2) DEFAULT 0.00 AFTER uploaded_files",
        "ADD COLUMN IF NOT EXISTS document_score DECIMAL(5,2) DEFAULT 0.00 AFTER quiz_score",
        "ADD COLUMN IF NOT EXISTS combined_score DECIMAL(5,2) DEFAULT 0.00 AFTER document_score",
        "ADD COLUMN IF NOT EXISTS document_feedback TEXT AFTER admin_feedback",
        "ADD COLUMN IF NOT EXISTS requires_manual_grading TINYINT(1) DEFAULT 0 AFTER document_feedback"
    ];

    foreach ($answer_columns as $column_sql) {
        $full_sql = "ALTER TABLE sita_assessment_answers $column_sql";
        if ($conn->query($full_sql) === TRUE) {
            echo "<p>✅ Added answer field: " . substr($column_sql, 18, 50) . "...</p>";
        } else {
            if (strpos($conn->error, 'Duplicate column') === false) {
                $errors[] = "Error adding answer column: " . $conn->error;
            }
        }
    }

    // 4. Create file upload tracking table
    echo "<h3>📁 Creating File Upload Tracking Table</h3>";
    
    $file_uploads_sql = "CREATE TABLE IF NOT EXISTS sita_assessment_files (
        file_id INT AUTO_INCREMENT PRIMARY KEY,
        answer_id INT NOT NULL,
        attempt_id INT NOT NULL,
        question_id INT NOT NULL,
        original_filename VARCHAR(255) NOT NULL,
        stored_filename VARCHAR(255) NOT NULL,
        file_path TEXT NOT NULL,
        file_size BIGINT NOT NULL,
        mime_type VARCHAR(100) NOT NULL,
        file_hash VARCHAR(64) NOT NULL,
        upload_timestamp TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        is_verified TINYINT(1) DEFAULT 0,
        admin_notes TEXT,
        
        INDEX idx_answer (answer_id),
        INDEX idx_attempt (attempt_id),
        INDEX idx_question (question_id),
        INDEX idx_file_hash (file_hash),
        
        FOREIGN KEY (answer_id) REFERENCES sita_assessment_answers(answer_id) ON DELETE CASCADE,
        FOREIGN KEY (attempt_id) REFERENCES sita_assessment_attempts(attempt_id) ON DELETE CASCADE,
        FOREIGN KEY (question_id) REFERENCES sita_assessment_questions(question_id) ON DELETE CASCADE
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";

    if ($conn->query($file_uploads_sql) === TRUE) {
        echo "<p>✅ Created table: <strong>sita_assessment_files</strong></p>";
        $updates_applied[] = "sita_assessment_files table";
    } else {
        $errors[] = "Error creating sita_assessment_files: " . $conn->error;
    }

    // 5. Create question templates table for admin presets
    echo "<h3>📋 Creating Question Templates Table</h3>";
    
    $templates_sql = "CREATE TABLE IF NOT EXISTS sita_question_templates (
        template_id INT AUTO_INCREMENT PRIMARY KEY,
        template_name VARCHAR(255) NOT NULL,
        category VARCHAR(100) NOT NULL,
        question_type ENUM('multiple_choice', 'true_false', 'short_answer', 'essay', 'practical') NOT NULL,
        answer_type ENUM('quiz_only', 'document_only', 'quiz_and_document') NOT NULL,
        question_text TEXT NOT NULL,
        instructions TEXT,
        allowed_file_formats JSON,
        max_file_size_mb INT DEFAULT 10,
        points DECIMAL(5,2) DEFAULT 1.00,
        created_by INT NOT NULL,
        created_date TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        is_active TINYINT(1) DEFAULT 1,
        
        INDEX idx_category (category),
        INDEX idx_created_by (created_by),
        INDEX idx_active (is_active),
        
        FOREIGN KEY (created_by) REFERENCES users(user_id) ON DELETE CASCADE
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";

    if ($conn->query($templates_sql) === TRUE) {
        echo "<p>✅ Created table: <strong>sita_question_templates</strong></p>";
        $updates_applied[] = "sita_question_templates table";
    } else {
        $errors[] = "Error creating sita_question_templates: " . $conn->error;
    }

    // 6. Insert default file format presets
    echo "<h3>🎯 Adding Default Configuration Presets</h3>";
    
    $default_formats = [
        'documents' => ['pdf', 'doc', 'docx', 'txt'],
        'images' => ['jpg', 'jpeg', 'png', 'gif'],
        'presentations' => ['ppt', 'pptx', 'pdf'],
        'spreadsheets' => ['xls', 'xlsx', 'csv'],
        'code' => ['zip', 'rar', 'txt', 'py', 'js', 'php'],
        'multimedia' => ['mp4', 'avi', 'mp3', 'wav'],
        'all_common' => ['pdf', 'doc', 'docx', 'jpg', 'png', 'zip', 'txt']
    ];

    // Create file format presets configuration
    $presets_sql = "CREATE TABLE IF NOT EXISTS sita_file_format_presets (
        preset_id INT AUTO_INCREMENT PRIMARY KEY,
        preset_name VARCHAR(100) NOT NULL,
        preset_description TEXT,
        file_formats JSON NOT NULL,
        max_file_size_mb INT DEFAULT 10,
        is_default TINYINT(1) DEFAULT 0,
        created_date TIMESTAMP DEFAULT CURRENT_TIMESTAMP
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";

    if ($conn->query($presets_sql) === TRUE) {
        echo "<p>✅ Created table: <strong>sita_file_format_presets</strong></p>";
        
        // Insert default presets
        foreach ($default_formats as $name => $formats) {
            $description = ucfirst(str_replace('_', ' ', $name)) . ' files';
            $formats_json = json_encode($formats);
            $insert_preset = "INSERT INTO sita_file_format_presets (preset_name, preset_description, file_formats) 
                             VALUES ('$name', '$description', '$formats_json')";
            $conn->query($insert_preset);
        }
        echo "<p>✅ Added " . count($default_formats) . " default file format presets</p>";
    }

    // 7. Create assessment directories
    $directories = [
        '../uploads/sita_assessments',
        '../uploads/sita_assessments/files',
        '../uploads/sita_assessments/temp'
    ];

    foreach ($directories as $dir) {
        if (!file_exists($dir)) {
            if (mkdir($dir, 0755, true)) {
                echo "<p>✅ Created directory: <strong>" . basename($dir) . "</strong></p>";
            } else {
                $errors[] = "Failed to create directory: $dir";
            }
        }
    }

    echo "<br><h3>📊 Enhancement Summary</h3>";
    echo "<div style='background: #d4edda; border: 1px solid #c3e6cb; color: #155724; padding: 15px; border-radius: 5px;'>";
    echo "<h4>🎉 Assessment System Enhanced!</h4>";
    echo "<p><strong>New Features Added:</strong></p>";
    echo "<ul>";
    echo "<li>✅ <strong>Flexible Question Types:</strong> Quiz-only, Document-only, or Both combined</li>";
    echo "<li>✅ <strong>File Upload Support:</strong> Multiple formats, size limits, quantity controls</li>";
    echo "<li>✅ <strong>Admin Configuration:</strong> Set allowed formats, requirements per question</li>";
    echo "<li>✅ <strong>Weighted Scoring:</strong> Separate weights for quiz vs document components</li>";
    echo "<li>✅ <strong>Question Templates:</strong> Reusable question patterns for admins</li>";
    echo "<li>✅ <strong>File Format Presets:</strong> Common format combinations</li>";
    echo "<li>✅ <strong>Mixed Assessments:</strong> Students can answer questions AND upload files</li>";
    echo "<li>✅ <strong>Auto + Manual Grading:</strong> Quiz parts auto-graded, documents manually reviewed</li>";
    echo "</ul>";
    echo "</div>";

    if (!empty($errors)) {
        echo "<h4 style='color: red;'>⚠️ Issues Encountered:</h4>";
        echo "<ul>";
        foreach ($errors as $error) {
            echo "<li style='color: orange;'>$error</li>";
        }
        echo "</ul>";
    }

    echo "<div style='background: #f8f9fa; border: 1px solid #dee2e6; padding: 15px; border-radius: 5px; margin: 20px 0;'>";
    echo "<h4>🔧 Admin Configuration Options</h4>";
    echo "<p>Admins can now configure assessments with:</p>";
    echo "<ul>";
    echo "<li><strong>Question Answer Types:</strong>";
    echo "<ul>";
    echo "<li>Quiz Only - Traditional multiple choice, true/false questions</li>";
    echo "<li>Document Only - File upload requirements</li>";
    echo "<li>Quiz + Document - Both correct answers AND file submissions</li>";
    echo "</ul></li>";
    echo "<li><strong>File Controls:</strong> Format restrictions, size limits, quantity requirements</li>";
    echo "<li><strong>Scoring Weights:</strong> Balance between quiz scores and document evaluations</li>";
    echo "<li><strong>Review Requirements:</strong> Auto-grade quiz parts, manual review documents</li>";
    echo "</ul>";
    echo "</div>";

} catch (Exception $e) {
    echo "<div style='background: #f8d7da; border: 1px solid #f5c6cb; color: #721c24; padding: 15px; border-radius: 5px;'>";
    echo "<h4>❌ Enhancement Failed</h4>";
    echo "<p>Error: " . $e->getMessage() . "</p>";
    echo "</div>";
}

echo "</div>";
echo "<p style='text-align: center; margin-top: 30px;'><a href='../index.php' style='background: #007bff; color: white; padding: 10px 20px; text-decoration: none; border-radius: 5px;'>← Back to Dashboard</a></p>";
?>
