<?php
/**
 * Fixed Enhanced SITA Assessment System
 * Properly adds columns by checking existence first
 */

require_once '../config/db_connect.php';

if (!isset($conn)) {
    die("Database connection failed. Please check your database configuration.");
}

echo "<h2>🔧 Fixing SITA Assessment System Columns</h2>";
echo "<div style='font-family: Arial, sans-serif; max-width: 800px; margin: 0 auto; padding: 20px;'>";

$updates_applied = [];
$errors = [];

// Function to check if column exists
function columnExists($conn, $table, $column) {
    $result = $conn->query("SHOW COLUMNS FROM `$table` LIKE '$column'");
    return $result->num_rows > 0;
}

try {
    // 1. Fix sita_assessment_questions table
    echo "<h3>📝 Adding Missing Columns to Assessment Questions</h3>";
    
    $question_columns = [
        'answer_type' => "ADD COLUMN answer_type ENUM('quiz_only', 'document_only', 'quiz_and_document') DEFAULT 'quiz_only' AFTER question_type",
        'requires_file_upload' => "ADD COLUMN requires_file_upload TINYINT(1) DEFAULT 0 AFTER answer_type",
        'allowed_file_formats' => "ADD COLUMN allowed_file_formats JSON AFTER requires_file_upload",
        'max_file_size_mb' => "ADD COLUMN max_file_size_mb INT DEFAULT 10 AFTER allowed_file_formats",
        'min_files_required' => "ADD COLUMN min_files_required INT DEFAULT 0 AFTER max_file_size_mb",
        'max_files_allowed' => "ADD COLUMN max_files_allowed INT DEFAULT 1 AFTER min_files_required",
        'file_upload_instructions' => "ADD COLUMN file_upload_instructions TEXT AFTER max_files_allowed",
        'quiz_weight_percentage' => "ADD COLUMN quiz_weight_percentage DECIMAL(5,2) DEFAULT 100.00 AFTER file_upload_instructions",
        'document_weight_percentage' => "ADD COLUMN document_weight_percentage DECIMAL(5,2) DEFAULT 0.00 AFTER quiz_weight_percentage"
    ];

    foreach ($question_columns as $column_name => $column_sql) {
        if (!columnExists($conn, 'sita_assessment_questions', $column_name)) {
            $full_sql = "ALTER TABLE sita_assessment_questions $column_sql";
            if ($conn->query($full_sql) === TRUE) {
                echo "<p>✅ Added column: <strong>$column_name</strong></p>";
                $updates_applied[] = $column_name;
            } else {
                $errors[] = "Error adding $column_name: " . $conn->error;
            }
        } else {
            echo "<p>ℹ️ Column already exists: <strong>$column_name</strong></p>";
        }
    }

    // 2. Fix sita_assessments table
    echo "<h3>⚙️ Adding Missing Columns to Assessment Settings</h3>";
    
    $assessment_columns = [
        'allow_mixed_questions' => "ADD COLUMN allow_mixed_questions TINYINT(1) DEFAULT 1 AFTER max_attempts",
        'global_file_formats' => "ADD COLUMN global_file_formats JSON AFTER allow_mixed_questions",
        'require_all_questions' => "ADD COLUMN require_all_questions TINYINT(1) DEFAULT 1 AFTER global_file_formats",
        'auto_grade_quiz_parts' => "ADD COLUMN auto_grade_quiz_parts TINYINT(1) DEFAULT 1 AFTER require_all_questions",
        'admin_review_required' => "ADD COLUMN admin_review_required TINYINT(1) DEFAULT 1 AFTER auto_grade_quiz_parts"
    ];

    foreach ($assessment_columns as $column_name => $column_sql) {
        if (!columnExists($conn, 'sita_assessments', $column_name)) {
            $full_sql = "ALTER TABLE sita_assessments $column_sql";
            if ($conn->query($full_sql) === TRUE) {
                echo "<p>✅ Added assessment column: <strong>$column_name</strong></p>";
                $updates_applied[] = $column_name;
            } else {
                $errors[] = "Error adding assessment $column_name: " . $conn->error;
            }
        } else {
            echo "<p>ℹ️ Assessment column already exists: <strong>$column_name</strong></p>";
        }
    }

    // 3. Fix sita_assessment_answers table
    echo "<h3>📄 Adding Missing Columns to Assessment Answers</h3>";
    
    $answer_columns = [
        'uploaded_files' => "ADD COLUMN uploaded_files JSON AFTER file_upload",
        'quiz_score' => "ADD COLUMN quiz_score DECIMAL(5,2) DEFAULT 0.00 AFTER uploaded_files",
        'document_score' => "ADD COLUMN document_score DECIMAL(5,2) DEFAULT 0.00 AFTER quiz_score",
        'combined_score' => "ADD COLUMN combined_score DECIMAL(5,2) DEFAULT 0.00 AFTER document_score",
        'document_feedback' => "ADD COLUMN document_feedback TEXT AFTER admin_feedback",
        'requires_manual_grading' => "ADD COLUMN requires_manual_grading TINYINT(1) DEFAULT 0 AFTER document_feedback"
    ];

    foreach ($answer_columns as $column_name => $column_sql) {
        if (!columnExists($conn, 'sita_assessment_answers', $column_name)) {
            $full_sql = "ALTER TABLE sita_assessment_answers $column_sql";
            if ($conn->query($full_sql) === TRUE) {
                echo "<p>✅ Added answer column: <strong>$column_name</strong></p>";
                $updates_applied[] = $column_name;
            } else {
                $errors[] = "Error adding answer $column_name: " . $conn->error;
            }
        } else {
            echo "<p>ℹ️ Answer column already exists: <strong>$column_name</strong></p>";
        }
    }

    // 4. Verify all tables exist and show structure
    echo "<h3>🔍 Verifying Assessment System Structure</h3>";
    
    $tables_to_check = [
        'accreditation_requests',
        'sita_assessments', 
        'sita_assessment_questions',
        'sita_assessment_options',
        'sita_assessment_attempts',
        'sita_assessment_answers',
        'sita_assessment_files',
        'sita_question_templates',
        'sita_file_format_presets',
        'accreditation_documents',
        'accredited_certificates'
    ];

    $existing_tables = [];
    foreach ($tables_to_check as $table) {
        $result = $conn->query("SHOW TABLES LIKE '$table'");
        if ($result->num_rows > 0) {
            $existing_tables[] = $table;
            echo "<p>✅ Table exists: <strong>$table</strong></p>";
        } else {
            echo "<p>❌ Table missing: <strong>$table</strong></p>";
            $errors[] = "Missing table: $table";
        }
    }

    echo "<br><h3>📊 Final System Status</h3>";
    echo "<div style='background: #d4edda; border: 1px solid #c3e6cb; color: #155724; padding: 15px; border-radius: 5px;'>";
    echo "<h4>🎉 Enhanced Assessment System Status</h4>";
    echo "<p><strong>Columns Added This Run:</strong> " . count($updates_applied) . "</p>";
    echo "<p><strong>Tables Available:</strong> " . count($existing_tables) . "/" . count($tables_to_check) . "</p>";
    
    if (count($updates_applied) > 0) {
        echo "<p><strong>New Columns:</strong></p>";
        echo "<ul>";
        foreach ($updates_applied as $column) {
            echo "<li>✅ $column</li>";
        }
        echo "</ul>";
    }

    echo "<p><strong>Assessment Features Now Available:</strong></p>";
    echo "<ul>";
    echo "<li>✅ <strong>Question Types:</strong> Quiz-only, Document-only, Quiz+Documents</li>";
    echo "<li>✅ <strong>File Upload Controls:</strong> Format restrictions, size limits, file quantity</li>";
    echo "<li>✅ <strong>Weighted Scoring:</strong> Separate scoring for quiz vs document components</li>";
    echo "<li>✅ <strong>Admin Templates:</strong> Reusable question patterns</li>";
    echo "<li>✅ <strong>Mixed Assessments:</strong> Students answer questions AND upload files</li>";
    echo "<li>✅ <strong>Flexible Grading:</strong> Auto-grade quizzes, manual review documents</li>";
    echo "</ul>";
    echo "</div>";

    if (!empty($errors)) {
        echo "<h4 style='color: red;'>⚠️ Issues to Address:</h4>";
        echo "<ul>";
        foreach ($errors as $error) {
            echo "<li style='color: orange;'>$error</li>";
        }
        echo "</ul>";
    }

    // 5. Sample configuration examples
    echo "<div style='background: #f8f9fa; border: 1px solid #dee2e6; padding: 15px; border-radius: 5px; margin: 20px 0;'>";
    echo "<h4>💡 Example Assessment Configurations</h4>";
    echo "<p><strong>1. Traditional Quiz Question:</strong></p>";
    echo "<code>answer_type: 'quiz_only', requires_file_upload: 0</code>";
    
    echo "<p><strong>2. Document Submission Only:</strong></p>";
    echo "<code>answer_type: 'document_only', allowed_file_formats: ['pdf','docx'], max_files_allowed: 3</code>";
    
    echo "<p><strong>3. Mixed Question (Quiz + Documents):</strong></p>";
    echo "<code>answer_type: 'quiz_and_document', quiz_weight: 60%, document_weight: 40%</code>";
    
    echo "<p><strong>4. Admin Controls:</strong></p>";
    echo "<ul>";
    echo "<li>Set allowed file formats per question</li>";
    echo "<li>Configure minimum/maximum file requirements</li>";
    echo "<li>Balance scoring between quiz and document parts</li>";
    echo "<li>Require manual admin review for documents</li>";
    echo "</ul>";
    echo "</div>";

} catch (Exception $e) {
    echo "<div style='background: #f8d7da; border: 1px solid #f5c6cb; color: #721c24; padding: 15px; border-radius: 5px;'>";
    echo "<h4>❌ Fix Failed</h4>";
    echo "<p>Error: " . $e->getMessage() . "</p>";
    echo "</div>";
}

echo "</div>";
echo "<p style='text-align: center; margin-top: 30px;'><a href='../index.php' style='background: #007bff; color: white; padding: 10px 20px; text-decoration: none; border-radius: 5px;'>← Back to Dashboard</a></p>";
?>
