<?php
/**
 * SITA-Compliant Accredited Certificate System Setup
 * Creates database tables for comprehensive accreditation workflow
 */

require_once '../config/db_connect.php';

// Check if connection exists
if (!isset($conn)) {
    die("Database connection failed. Please check your database configuration.");
}

echo "<h2>🎓 Setting up SITA-Compliant Accredited Certificate System</h2>";
echo "<div style='font-family: Arial, sans-serif; max-width: 800px; margin: 0 auto; padding: 20px;'>";

$tables_created = [];
$errors = [];

try {
    // 1. Accreditation Requests Table
    $sql1 = "CREATE TABLE IF NOT EXISTS accreditation_requests (
        request_id INT AUTO_INCREMENT PRIMARY KEY,
        student_id INT NOT NULL,
        course_id INT NOT NULL,
        request_date TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        status ENUM('pending', 'approved', 'assessment_created', 'assessment_completed', 'documents_pending', 'documents_submitted', 'under_review', 'approved_final', 'rejected', 'certificate_issued') DEFAULT 'pending',
        admin_notes TEXT,
        approved_by INT NULL,
        approved_date TIMESTAMP NULL,
        rejection_reason TEXT,
        certificate_issued_date TIMESTAMP NULL,
        certificate_number VARCHAR(100) UNIQUE,
        
        INDEX idx_student_course (student_id, course_id),
        INDEX idx_status (status),
        INDEX idx_request_date (request_date),
        
        FOREIGN KEY (student_id) REFERENCES users(user_id) ON DELETE CASCADE,
        FOREIGN KEY (course_id) REFERENCES courses(course_id) ON DELETE CASCADE,
        FOREIGN KEY (approved_by) REFERENCES users(user_id) ON DELETE SET NULL
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";

    if ($conn->query($sql1) === TRUE) {
        $tables_created[] = "accreditation_requests";
        echo "<p>✅ Created table: <strong>accreditation_requests</strong></p>";
    } else {
        $errors[] = "Error creating accreditation_requests: " . $conn->error;
    }

    // 2. SITA Assessment Questions Table
    $sql2 = "CREATE TABLE IF NOT EXISTS sita_assessments (
        assessment_id INT AUTO_INCREMENT PRIMARY KEY,
        request_id INT NOT NULL,
        course_id INT NOT NULL,
        created_by INT NOT NULL,
        title VARCHAR(255) NOT NULL,
        instructions TEXT,
        time_limit_minutes INT DEFAULT 180,
        pass_mark_percentage DECIMAL(5,2) DEFAULT 70.00,
        max_attempts INT DEFAULT 1,
        is_published TINYINT(1) DEFAULT 0,
        created_date TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        deadline TIMESTAMP NULL,
        
        INDEX idx_request (request_id),
        INDEX idx_course (course_id),
        INDEX idx_status (is_published),
        
        FOREIGN KEY (request_id) REFERENCES accreditation_requests(request_id) ON DELETE CASCADE,
        FOREIGN KEY (course_id) REFERENCES courses(course_id) ON DELETE CASCADE,
        FOREIGN KEY (created_by) REFERENCES users(user_id) ON DELETE CASCADE
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";

    if ($conn->query($sql2) === TRUE) {
        $tables_created[] = "sita_assessments";
        echo "<p>✅ Created table: <strong>sita_assessments</strong></p>";
    } else {
        $errors[] = "Error creating sita_assessments: " . $conn->error;
    }

    // 3. SITA Assessment Questions
    $sql3 = "CREATE TABLE IF NOT EXISTS sita_assessment_questions (
        question_id INT AUTO_INCREMENT PRIMARY KEY,
        assessment_id INT NOT NULL,
        question_number INT NOT NULL,
        question_text TEXT NOT NULL,
        question_type ENUM('multiple_choice', 'true_false', 'short_answer', 'essay', 'practical') NOT NULL,
        points DECIMAL(5,2) DEFAULT 1.00,
        required TINYINT(1) DEFAULT 1,
        
        INDEX idx_assessment (assessment_id),
        INDEX idx_question_number (assessment_id, question_number),
        
        FOREIGN KEY (assessment_id) REFERENCES sita_assessments(assessment_id) ON DELETE CASCADE
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";

    if ($conn->query($sql3) === TRUE) {
        $tables_created[] = "sita_assessment_questions";
        echo "<p>✅ Created table: <strong>sita_assessment_questions</strong></p>";
    } else {
        $errors[] = "Error creating sita_assessment_questions: " . $conn->error;
    }

    // 4. Question Options (for multiple choice questions)
    $sql4 = "CREATE TABLE IF NOT EXISTS sita_assessment_options (
        option_id INT AUTO_INCREMENT PRIMARY KEY,
        question_id INT NOT NULL,
        option_text TEXT NOT NULL,
        is_correct TINYINT(1) DEFAULT 0,
        option_order INT DEFAULT 1,
        
        INDEX idx_question (question_id),
        
        FOREIGN KEY (question_id) REFERENCES sita_assessment_questions(question_id) ON DELETE CASCADE
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";

    if ($conn->query($sql4) === TRUE) {
        $tables_created[] = "sita_assessment_options";
        echo "<p>✅ Created table: <strong>sita_assessment_options</strong></p>";
    } else {
        $errors[] = "Error creating sita_assessment_options: " . $conn->error;
    }

    // 5. Student Assessment Attempts
    $sql5 = "CREATE TABLE IF NOT EXISTS sita_assessment_attempts (
        attempt_id INT AUTO_INCREMENT PRIMARY KEY,
        assessment_id INT NOT NULL,
        student_id INT NOT NULL,
        request_id INT NOT NULL,
        attempt_number INT DEFAULT 1,
        start_time TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        end_time TIMESTAMP NULL,
        total_score DECIMAL(5,2) DEFAULT 0.00,
        percentage_score DECIMAL(5,2) DEFAULT 0.00,
        passed TINYINT(1) DEFAULT 0,
        status ENUM('in_progress', 'completed', 'submitted', 'graded') DEFAULT 'in_progress',
        time_taken_minutes INT DEFAULT 0,
        ip_address VARCHAR(45),
        user_agent TEXT,
        
        INDEX idx_assessment_student (assessment_id, student_id),
        INDEX idx_request (request_id),
        INDEX idx_status (status),
        
        FOREIGN KEY (assessment_id) REFERENCES sita_assessments(assessment_id) ON DELETE CASCADE,
        FOREIGN KEY (student_id) REFERENCES users(user_id) ON DELETE CASCADE,
        FOREIGN KEY (request_id) REFERENCES accreditation_requests(request_id) ON DELETE CASCADE
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";

    if ($conn->query($sql5) === TRUE) {
        $tables_created[] = "sita_assessment_attempts";
        echo "<p>✅ Created table: <strong>sita_assessment_attempts</strong></p>";
    } else {
        $errors[] = "Error creating sita_assessment_attempts: " . $conn->error;
    }

    // 6. Student Answers
    $sql6 = "CREATE TABLE IF NOT EXISTS sita_assessment_answers (
        answer_id INT AUTO_INCREMENT PRIMARY KEY,
        attempt_id INT NOT NULL,
        question_id INT NOT NULL,
        selected_option_id INT NULL,
        text_answer TEXT,
        file_upload TEXT,
        points_earned DECIMAL(5,2) DEFAULT 0.00,
        is_correct TINYINT(1) DEFAULT 0,
        admin_feedback TEXT,
        graded_by INT NULL,
        graded_at TIMESTAMP NULL,
        
        INDEX idx_attempt (attempt_id),
        INDEX idx_question (question_id),
        
        FOREIGN KEY (attempt_id) REFERENCES sita_assessment_attempts(attempt_id) ON DELETE CASCADE,
        FOREIGN KEY (question_id) REFERENCES sita_assessment_questions(question_id) ON DELETE CASCADE,
        FOREIGN KEY (selected_option_id) REFERENCES sita_assessment_options(option_id) ON DELETE SET NULL,
        FOREIGN KEY (graded_by) REFERENCES users(user_id) ON DELETE SET NULL
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";

    if ($conn->query($sql6) === TRUE) {
        $tables_created[] = "sita_assessment_answers";
        echo "<p>✅ Created table: <strong>sita_assessment_answers</strong></p>";
    } else {
        $errors[] = "Error creating sita_assessment_answers: " . $conn->error;
    }

    // 7. Document Submissions
    $sql7 = "CREATE TABLE IF NOT EXISTS accreditation_documents (
        document_id INT AUTO_INCREMENT PRIMARY KEY,
        request_id INT NOT NULL,
        student_id INT NOT NULL,
        document_type ENUM('cv', 'qualifications', 'id_document', 'assignment', 'other') NOT NULL,
        document_name VARCHAR(255) NOT NULL,
        file_path TEXT NOT NULL,
        file_size BIGINT DEFAULT 0,
        mime_type VARCHAR(100),
        upload_date TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        verification_status ENUM('pending', 'verified', 'rejected') DEFAULT 'pending',
        admin_notes TEXT,
        verified_by INT NULL,
        verified_date TIMESTAMP NULL,
        
        INDEX idx_request (request_id),
        INDEX idx_student (student_id),
        INDEX idx_document_type (document_type),
        INDEX idx_verification_status (verification_status),
        
        FOREIGN KEY (request_id) REFERENCES accreditation_requests(request_id) ON DELETE CASCADE,
        FOREIGN KEY (student_id) REFERENCES users(user_id) ON DELETE CASCADE,
        FOREIGN KEY (verified_by) REFERENCES users(user_id) ON DELETE SET NULL
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";

    if ($conn->query($sql7) === TRUE) {
        $tables_created[] = "accreditation_documents";
        echo "<p>✅ Created table: <strong>accreditation_documents</strong></p>";
    } else {
        $errors[] = "Error creating accreditation_documents: " . $conn->error;
    }

    // 8. Certificate Registry
    $sql8 = "CREATE TABLE IF NOT EXISTS accredited_certificates (
        certificate_id INT AUTO_INCREMENT PRIMARY KEY,
        request_id INT NOT NULL,
        student_id INT NOT NULL,
        course_id INT NOT NULL,
        certificate_number VARCHAR(100) UNIQUE NOT NULL,
        certificate_type ENUM('course_completion', 'sita_accredited', 'professional_certification') DEFAULT 'sita_accredited',
        issue_date TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        issued_by INT NOT NULL,
        certificate_data JSON,
        pdf_file_path TEXT,
        digital_signature TEXT,
        verification_code VARCHAR(100) UNIQUE,
        expiry_date DATE NULL,
        is_revoked TINYINT(1) DEFAULT 0,
        revocation_reason TEXT,
        revoked_date TIMESTAMP NULL,
        
        INDEX idx_certificate_number (certificate_number),
        INDEX idx_verification_code (verification_code),
        INDEX idx_student_course (student_id, course_id),
        INDEX idx_issue_date (issue_date),
        INDEX idx_status (is_revoked),
        
        FOREIGN KEY (request_id) REFERENCES accreditation_requests(request_id) ON DELETE CASCADE,
        FOREIGN KEY (student_id) REFERENCES users(user_id) ON DELETE CASCADE,
        FOREIGN KEY (course_id) REFERENCES courses(course_id) ON DELETE CASCADE,
        FOREIGN KEY (issued_by) REFERENCES users(user_id) ON DELETE CASCADE
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";

    if ($conn->query($sql8) === TRUE) {
        $tables_created[] = "accredited_certificates";
        echo "<p>✅ Created table: <strong>accredited_certificates</strong></p>";
    } else {
        $errors[] = "Error creating accredited_certificates: " . $conn->error;
    }

    // 9. System Notifications
    $sql9 = "CREATE TABLE IF NOT EXISTS accreditation_notifications (
        notification_id INT AUTO_INCREMENT PRIMARY KEY,
        request_id INT NOT NULL,
        recipient_id INT NOT NULL,
        notification_type ENUM('request_submitted', 'request_approved', 'assessment_ready', 'assessment_completed', 'documents_required', 'documents_submitted', 'certificate_issued', 'request_rejected') NOT NULL,
        title VARCHAR(255) NOT NULL,
        message TEXT NOT NULL,
        is_read TINYINT(1) DEFAULT 0,
        created_date TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        read_date TIMESTAMP NULL,
        email_sent TINYINT(1) DEFAULT 0,
        email_sent_date TIMESTAMP NULL,
        
        INDEX idx_recipient (recipient_id),
        INDEX idx_request (request_id),
        INDEX idx_type (notification_type),
        INDEX idx_read_status (is_read),
        
        FOREIGN KEY (request_id) REFERENCES accreditation_requests(request_id) ON DELETE CASCADE,
        FOREIGN KEY (recipient_id) REFERENCES users(user_id) ON DELETE CASCADE
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";

    if ($conn->query($sql9) === TRUE) {
        $tables_created[] = "accreditation_notifications";
        echo "<p>✅ Created table: <strong>accreditation_notifications</strong></p>";
    } else {
        $errors[] = "Error creating accreditation_notifications: " . $conn->error;
    }

    // 10. Audit Trail
    $sql10 = "CREATE TABLE IF NOT EXISTS accreditation_audit_log (
        log_id INT AUTO_INCREMENT PRIMARY KEY,
        request_id INT NOT NULL,
        user_id INT NOT NULL,
        action VARCHAR(100) NOT NULL,
        details TEXT,
        ip_address VARCHAR(45),
        user_agent TEXT,
        timestamp TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        
        INDEX idx_request (request_id),
        INDEX idx_user (user_id),
        INDEX idx_action (action),
        INDEX idx_timestamp (timestamp),
        
        FOREIGN KEY (request_id) REFERENCES accreditation_requests(request_id) ON DELETE CASCADE,
        FOREIGN KEY (user_id) REFERENCES users(user_id) ON DELETE CASCADE
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";

    if ($conn->query($sql10) === TRUE) {
        $tables_created[] = "accreditation_audit_log";
        echo "<p>✅ Created table: <strong>accreditation_audit_log</strong></p>";
    } else {
        $errors[] = "Error creating accreditation_audit_log: " . $conn->error;
    }

    // Create uploads directory for accreditation documents
    $upload_dir = '../uploads/accreditation';
    if (!file_exists($upload_dir)) {
        if (mkdir($upload_dir, 0755, true)) {
            echo "<p>✅ Created upload directory: <strong>uploads/accreditation</strong></p>";
        } else {
            $errors[] = "Failed to create uploads/accreditation directory";
        }
    }

    // Create certificates directory
    $certificates_dir = '../uploads/certificates';
    if (!file_exists($certificates_dir)) {
        if (mkdir($certificates_dir, 0755, true)) {
            echo "<p>✅ Created certificates directory: <strong>uploads/certificates</strong></p>";
        } else {
            $errors[] = "Failed to create uploads/certificates directory";
        }
    }

    echo "<br><h3>📊 Setup Summary</h3>";
    echo "<p><strong>Tables Created:</strong> " . count($tables_created) . "</p>";
    if (!empty($tables_created)) {
        echo "<ul>";
        foreach ($tables_created as $table) {
            echo "<li>✅ $table</li>";
        }
        echo "</ul>";
    }

    if (!empty($errors)) {
        echo "<h4 style='color: red;'>❌ Errors:</h4>";
        echo "<ul>";
        foreach ($errors as $error) {
            echo "<li style='color: red;'>$error</li>";
        }
        echo "</ul>";
    } else {
        echo "<div style='background: #d4edda; border: 1px solid #c3e6cb; color: #155724; padding: 15px; border-radius: 5px; margin: 20px 0;'>";
        echo "<h4>🎉 Setup Complete!</h4>";
        echo "<p>The SITA-Compliant Accredited Certificate System has been successfully installed.</p>";
        echo "<p><strong>System Features:</strong></p>";
        echo "<ul>";
        echo "<li>✅ Student accreditation requests</li>";
        echo "<li>✅ Admin approval workflow</li>";
        echo "<li>✅ SITA-compliant assessments</li>";
        echo "<li>✅ Document upload system</li>";
        echo "<li>✅ Certificate generation</li>";
        echo "<li>✅ Email notifications</li>";
        echo "<li>✅ Audit trail logging</li>";
        echo "<li>✅ Certificate verification</li>";
        echo "</ul>";
        echo "</div>";
    }

} catch (Exception $e) {
    echo "<div style='background: #f8d7da; border: 1px solid #f5c6cb; color: #721c24; padding: 15px; border-radius: 5px;'>";
    echo "<h4>❌ Setup Failed</h4>";
    echo "<p>Error: " . $e->getMessage() . "</p>";
    echo "</div>";
}

echo "</div>";
echo "<p style='text-align: center; margin-top: 30px;'><a href='../index.php' style='background: #007bff; color: white; padding: 10px 20px; text-decoration: none; border-radius: 5px;'>← Back to Dashboard</a></p>";
?>
