<?php
/**
 * Accreditation Table Debug & Migration Script
 * This script analyzes the existing accreditation_requests table and adds missing columns
 * for the complete PayFast payment integration and workflow
 */

require_once '../config/db_connect.php';

// Check if connection exists
if (!isset($conn)) {
    die("❌ Database connection failed. Please check your database configuration.");
}

echo "<!DOCTYPE html><html><head><title>Accreditation Table Debug</title>";
echo "<style>
    body { font-family: Arial, sans-serif; max-width: 1200px; margin: 20px auto; padding: 20px; }
    .success { color: #28a745; background: #d4edda; padding: 10px; border-radius: 5px; margin: 10px 0; }
    .error { color: #dc3545; background: #f8d7da; padding: 10px; border-radius: 5px; margin: 10px 0; }
    .info { color: #0c5460; background: #d1ecf1; padding: 10px; border-radius: 5px; margin: 10px 0; }
    .warning { color: #856404; background: #fff3cd; padding: 10px; border-radius: 5px; margin: 10px 0; }
    table { width: 100%; border-collapse: collapse; margin: 20px 0; }
    th, td { padding: 12px; text-align: left; border: 1px solid #ddd; }
    th { background-color: #f8f9fa; font-weight: bold; }
    .status-badge { padding: 4px 8px; border-radius: 12px; color: white; font-size: 12px; }
    .added { background: #28a745; }
    .exists { background: #6c757d; }
    .missing { background: #dc3545; }
</style></head><body>";

echo "<h1>🎓 Accreditation System Table Debug & Migration</h1>";
echo "<p><strong>Timestamp:</strong> " . date('Y-m-d H:i:s') . "</p>";

// 1. Check if accreditation_requests table exists
echo "<h2>📋 Step 1: Current Table Structure Analysis</h2>";

$tableExists = false;
$checkTable = $conn->query("SHOW TABLES LIKE 'accreditation_requests'");
if ($checkTable && $checkTable->num_rows > 0) {
    $tableExists = true;
    echo "<div class='success'>✅ Table 'accreditation_requests' exists</div>";
} else {
    echo "<div class='error'>❌ Table 'accreditation_requests' does not exist</div>";
}

// 2. Get current table structure if it exists
$currentColumns = [];
if ($tableExists) {
    echo "<h3>Current Columns:</h3>";
    $result = $conn->query("DESCRIBE accreditation_requests");
    if ($result) {
        echo "<table>";
        echo "<tr><th>Column</th><th>Type</th><th>Null</th><th>Key</th><th>Default</th><th>Extra</th></tr>";
        
        while ($row = $result->fetch_assoc()) {
            $currentColumns[$row['Field']] = $row;
            echo "<tr>";
            echo "<td><strong>" . $row['Field'] . "</strong></td>";
            echo "<td>" . $row['Type'] . "</td>";
            echo "<td>" . $row['Null'] . "</td>";
            echo "<td>" . $row['Key'] . "</td>";
            echo "<td>" . $row['Default'] . "</td>";
            echo "<td>" . $row['Extra'] . "</td>";
            echo "</tr>";
        }
        echo "</table>";
    }
}

// 3. Define required columns for complete PayFast workflow
echo "<h2>🔧 Step 2: Required Columns for Complete Workflow</h2>";

$requiredColumns = [
    // Payment Integration Columns
    'payment_amount' => [
        'type' => 'DECIMAL(10,2)',
        'default' => '0.00',
        'comment' => 'PayFast payment amount'
    ],
    'payment_reference' => [
        'type' => 'VARCHAR(255)',
        'null' => true,
        'comment' => 'PayFast payment reference/transaction ID'
    ],
    'payment_date' => [
        'type' => 'TIMESTAMP',
        'null' => true,
        'comment' => 'Date when payment was completed'
    ],
    'payment_status' => [
        'type' => "ENUM('pending', 'completed', 'failed', 'refunded')",
        'default' => 'pending',
        'comment' => 'PayFast payment status'
    ],
    
    // Enhanced Workflow Columns
    'motivation_statement' => [
        'type' => 'TEXT',
        'null' => true,
        'comment' => 'Student motivation statement'
    ],
    'review_deadline' => [
        'type' => 'DATE',
        'null' => true,
        'comment' => '3 business days deadline for admin review'
    ],
    'assignment_document_path' => [
        'type' => 'VARCHAR(500)',
        'null' => true,
        'comment' => 'Path to assignment document created by admin'
    ],
    'assignment_created_date' => [
        'type' => 'TIMESTAMP',
        'null' => true,
        'comment' => 'When admin created the assignment'
    ],
    'assignment_instructions' => [
        'type' => 'TEXT',
        'null' => true,
        'comment' => 'Instructions for the assignment'
    ],
    
    // Document Submission Tracking
    'documents_submitted_date' => [
        'type' => 'TIMESTAMP',
        'null' => true,
        'comment' => 'When student submitted all required documents'
    ],
    'final_admin_id' => [
        'type' => 'INT',
        'null' => true,
        'comment' => 'Admin who does final review'
    ],
    'final_review_date' => [
        'type' => 'TIMESTAMP',
        'null' => true,
        'comment' => 'Date of final document review'
    ],
    'final_comments' => [
        'type' => 'TEXT',
        'null' => true,
        'comment' => 'Final admin comments'
    ],
    
    // Certificate Information
    'certificate_path' => [
        'type' => 'VARCHAR(500)',
        'null' => true,
        'comment' => 'Path to generated certificate file'
    ],
    
    // Timestamps
    'created_at' => [
        'type' => 'TIMESTAMP',
        'default' => 'CURRENT_TIMESTAMP',
        'comment' => 'Record creation timestamp'
    ],
    'updated_at' => [
        'type' => 'TIMESTAMP',
        'default' => 'CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP',
        'comment' => 'Record update timestamp'
    ]
];

// 4. Enhanced status enum
$newStatusEnum = "ENUM(
    'pending_payment',      
    'payment_completed',    
    'under_review',         
    'approved',            
    'assignment_created',   
    'documents_submitted',  
    'final_review',        
    'approved_final',      
    'certificate_issued',  
    'rejected',            
    'cancelled'
)";

// 5. Analyze missing columns
echo "<h3>Column Analysis:</h3>";
echo "<table>";
echo "<tr><th>Column</th><th>Status</th><th>Action Needed</th><th>Description</th></tr>";

$missingColumns = [];
$columnsToAdd = [];

foreach ($requiredColumns as $columnName => $columnDef) {
    echo "<tr>";
    echo "<td><strong>$columnName</strong></td>";
    
    if (isset($currentColumns[$columnName])) {
        echo "<td><span class='status-badge exists'>EXISTS</span></td>";
        echo "<td>No action needed</td>";
    } else {
        echo "<td><span class='status-badge missing'>MISSING</span></td>";
        echo "<td>Will be added</td>";
        $missingColumns[] = $columnName;
        $columnsToAdd[] = $columnDef;
    }
    echo "<td>" . $columnDef['comment'] . "</td>";
    echo "</tr>";
}

echo "</table>";

// 6. Show what will be executed
echo "<h2>⚡ Step 3: Migration Commands</h2>";

if (!empty($missingColumns)) {
    echo "<div class='warning'>🔨 Found " . count($missingColumns) . " missing columns that need to be added.</div>";
    
    echo "<h3>SQL Commands to be executed:</h3>";
    echo "<pre style='background: #f8f9fa; padding: 15px; border-radius: 5px; overflow-x: auto;'>";
    
    // Add missing columns
    foreach ($requiredColumns as $columnName => $columnDef) {
        if (in_array($columnName, $missingColumns)) {
            $sql = "ALTER TABLE accreditation_requests ADD COLUMN $columnName " . $columnDef['type'];
            
            if (isset($columnDef['null']) && $columnDef['null']) {
                $sql .= " NULL";
            } else {
                $sql .= " NOT NULL";
            }
            
            if (isset($columnDef['default'])) {
                $sql .= " DEFAULT " . $columnDef['default'];
            }
            
            $sql .= ";";
            echo $sql . "\n";
        }
    }
    
    // Update status enum if needed
    if (isset($currentColumns['status'])) {
        echo "\n-- Update status enum to include new workflow states:\n";
        echo "ALTER TABLE accreditation_requests MODIFY COLUMN status $newStatusEnum DEFAULT 'pending_payment';\n";
    }
    
    // Add foreign keys if needed
    echo "\n-- Add foreign key for final_admin_id if not exists:\n";
    echo "ALTER TABLE accreditation_requests ADD CONSTRAINT fk_final_admin \n";
    echo "FOREIGN KEY (final_admin_id) REFERENCES users(user_id) ON DELETE SET NULL;\n";
    
    echo "</pre>";
} else {
    echo "<div class='success'>✅ All required columns are present!</div>";
}

// 7. Execute the migrations
if (!empty($missingColumns) && isset($_GET['execute']) && $_GET['execute'] === 'true') {
    echo "<h2>🚀 Step 4: Executing Migrations</h2>";
    
    $errors = [];
    $success = [];
    
    // Add missing columns
    foreach ($requiredColumns as $columnName => $columnDef) {
        if (in_array($columnName, $missingColumns)) {
            $sql = "ALTER TABLE accreditation_requests ADD COLUMN $columnName " . $columnDef['type'];
            
            if (isset($columnDef['null']) && $columnDef['null']) {
                $sql .= " NULL";
            } else {
                $sql .= " NOT NULL";
            }
            
            if (isset($columnDef['default'])) {
                $sql .= " DEFAULT " . $columnDef['default'];
            }
            
            if ($conn->query($sql) === TRUE) {
                $success[] = "✅ Added column: $columnName";
                echo "<div class='success'>✅ Added column: <strong>$columnName</strong></div>";
            } else {
                $errors[] = "❌ Error adding column $columnName: " . $conn->error;
                echo "<div class='error'>❌ Error adding column <strong>$columnName</strong>: " . $conn->error . "</div>";
            }
        }
    }
    
    // Update status enum
    $statusSql = "ALTER TABLE accreditation_requests MODIFY COLUMN status $newStatusEnum DEFAULT 'pending_payment'";
    if ($conn->query($statusSql) === TRUE) {
        $success[] = "✅ Updated status enum";
        echo "<div class='success'>✅ Updated status enum with new workflow states</div>";
    } else {
        $errors[] = "❌ Error updating status enum: " . $conn->error;
        echo "<div class='error'>❌ Error updating status enum: " . $conn->error . "</div>";
    }
    
    // Add foreign key for final_admin_id
    $fkSql = "ALTER TABLE accreditation_requests ADD CONSTRAINT fk_final_admin FOREIGN KEY (final_admin_id) REFERENCES users(user_id) ON DELETE SET NULL";
    if ($conn->query($fkSql) === TRUE) {
        $success[] = "✅ Added foreign key constraint";
        echo "<div class='success'>✅ Added foreign key constraint for final_admin_id</div>";
    } else {
        // This might fail if constraint already exists, which is okay
        if (strpos($conn->error, 'Duplicate key name') === false) {
            echo "<div class='warning'>⚠️ Foreign key constraint note: " . $conn->error . "</div>";
        }
    }
    
    echo "<h3>Migration Summary:</h3>";
    echo "<p><strong>Successful operations:</strong> " . count($success) . "</p>";
    echo "<p><strong>Errors:</strong> " . count($errors) . "</p>";
    
    if (empty($errors)) {
        echo "<div class='success'>🎉 All migrations completed successfully!</div>";
    }
    
} elseif (!empty($missingColumns)) {
    echo "<div class='info'>
        <strong>Ready to execute migrations?</strong><br>
        <a href='?execute=true' style='display: inline-block; margin-top: 10px; padding: 10px 20px; background: #007bff; color: white; text-decoration: none; border-radius: 5px;'>
            🚀 Execute Migrations Now
        </a>
    </div>";
}

// 8. Final table structure check
if (isset($_GET['execute']) && $_GET['execute'] === 'true') {
    echo "<h2>📊 Step 5: Final Table Structure</h2>";
    
    $result = $conn->query("DESCRIBE accreditation_requests");
    if ($result) {
        echo "<table>";
        echo "<tr><th>Column</th><th>Type</th><th>Null</th><th>Key</th><th>Default</th><th>Extra</th></tr>";
        
        while ($row = $result->fetch_assoc()) {
            $isNew = in_array($row['Field'], $missingColumns);
            $rowClass = $isNew ? "style='background-color: #d4edda;'" : "";
            
            echo "<tr $rowClass>";
            echo "<td><strong>" . $row['Field'] . "</strong>" . ($isNew ? " <span class='status-badge added'>NEW</span>" : "") . "</td>";
            echo "<td>" . $row['Type'] . "</td>";
            echo "<td>" . $row['Null'] . "</td>";
            echo "<td>" . $row['Key'] . "</td>";
            echo "<td>" . $row['Default'] . "</td>";
            echo "<td>" . $row['Extra'] . "</td>";
            echo "</tr>";
        }
        echo "</table>";
    }
}

echo "<hr>";
echo "<p><em>Debug completed at " . date('Y-m-d H:i:s') . "</em></p>";
echo "</body></html>";
?>
