<?php
// Advanced Session Diagnostics & Repair Tool
// Shows detailed information and provides fixes for common session issues

// Display all errors
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Log server path information
$server_paths = [
    'document_root' => $_SERVER['DOCUMENT_ROOT'] ?? 'not set',
    'script_filename' => $_SERVER['SCRIPT_FILENAME'] ?? 'not set',
    'server_name' => $_SERVER['SERVER_NAME'] ?? 'not set',
    'http_host' => $_SERVER['HTTP_HOST'] ?? 'not set',
    'request_uri' => $_SERVER['REQUEST_URI'] ?? 'not set',
    'php_self' => $_SERVER['PHP_SELF'] ?? 'not set'
];

// Functions for session diagnostics
function check_session_directory() {
    $session_path = session_save_path();
    $results = [
        'path' => $session_path,
        'exists' => is_dir($session_path),
        'writable' => is_writable($session_path),
        'free_space' => disk_free_space($session_path) ?? 'unknown'
    ];
    
    return $results;
}

function test_session_write() {
    $test_key = 'test_value_' . time();
    $_SESSION['test_key'] = $test_key;
    session_write_close();
    
    // Re-open session
    session_start();
    $result = [
        'write_requested' => $test_key,
        'read_back' => $_SESSION['test_key'] ?? 'not found',
        'success' => ($test_key === ($_SESSION['test_key'] ?? false))
    ];
    
    return $result;
}

function check_session_files() {
    $session_path = session_save_path();
    $session_id = session_id();
    $prefix = 'sess_';
    
    $file_path = $session_path . '/' . $prefix . $session_id;
    $alt_path = $session_path . '/' . $session_id;
    
    $results = [
        'expected_path' => $file_path,
        'file_exists' => file_exists($file_path),
        'alt_exists' => file_exists($alt_path),
        'readable' => is_readable($file_path) || is_readable($alt_path),
        'file_size' => file_exists($file_path) ? filesize($file_path) : (file_exists($alt_path) ? filesize($alt_path) : 0)
    ];
    
    return $results;
}

// Start output buffering to prevent headers already sent issues
ob_start();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>PHP Session Diagnostics</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0-alpha1/dist/css/bootstrap.min.css" rel="stylesheet">
    <style>
        pre {
            background-color: #f4f4f4;
            padding: 10px;
            border-radius: 5px;
        }
        .result-box {
            margin-bottom: 20px;
            padding: 15px;
            border-radius: 5px;
        }
        .result-success {
            background-color: #d4edda;
            border-color: #c3e6cb;
        }
        .result-error {
            background-color: #f8d7da;
            border-color: #f5c6cb;
        }
        .result-warning {
            background-color: #fff3cd;
            border-color: #ffeeba;
        }
    </style>
</head>
<body>
    <div class="container my-4">
        <h1>PHP Session Diagnostics & Repair</h1>
        
        <div class="card mb-4">
            <div class="card-header">
                <h2 class="card-title">Current Session Status</h2>
            </div>
            <div class="card-body">
                <?php
                // Check session status before starting
                $initial_status = session_status();
                $status_text = [
                    PHP_SESSION_DISABLED => 'Sessions are disabled',
                    PHP_SESSION_NONE => 'Session enabled but none exists',
                    PHP_SESSION_ACTIVE => 'Session enabled and exists'
                ][$initial_status];
                
                echo '<p><strong>Initial Session Status:</strong> ' . $status_text . ' (' . $initial_status . ')</p>';
                
                // Start session if not already active
                if ($initial_status === PHP_SESSION_NONE) {
                    session_start();
                    echo '<p class="text-success">✅ Session started successfully</p>';
                } elseif ($initial_status === PHP_SESSION_DISABLED) {
                    echo '<p class="text-danger">❌ Sessions are disabled on this server!</p>';
                } else {
                    echo '<p class="text-warning">⚠️ Session was already active</p>';
                }
                ?>
                
                <h3 class="mt-3">Session Information</h3>
                <ul>
                    <li><strong>Session ID:</strong> <?= session_id() ?></li>
                    <li><strong>Session Name:</strong> <?= session_name() ?></li>
                    <li><strong>Session Module:</strong> <?= session_module_name() ?></li>
                    <li><strong>Cookie Parameters:</strong>
                        <pre><?php print_r(session_get_cookie_params()); ?></pre>
                    </li>
                </ul>
            </div>
        </div>
        
        <div class="card mb-4">
            <div class="card-header">
                <h2 class="card-title">Session Storage Diagnostics</h2>
            </div>
            <div class="card-body">
                <?php
                $dir_check = check_session_directory();
                $class = ($dir_check['exists'] && $dir_check['writable']) ? 'result-success' : 'result-error';
                ?>
                
                <div class="result-box <?= $class ?>">
                    <h4>Session Directory Check</h4>
                    <ul>
                        <li><strong>Path:</strong> <?= $dir_check['path'] ?></li>
                        <li><strong>Directory Exists:</strong> <?= $dir_check['exists'] ? 'Yes' : 'No' ?></li>
                        <li><strong>Directory Writable:</strong> <?= $dir_check['writable'] ? 'Yes' : 'No' ?></li>
                        <li><strong>Free Space:</strong> <?= is_numeric($dir_check['free_space']) ? number_format($dir_check['free_space'] / (1024*1024), 2) . ' MB' : $dir_check['free_space'] ?></li>
                    </ul>
                    
                    <?php if (!$dir_check['exists'] || !$dir_check['writable']): ?>
                        <div class="alert alert-danger">
                            <strong>Critical Issue:</strong> The session directory doesn't exist or isn't writable. 
                            This will prevent sessions from working correctly.
                        </div>
                    <?php endif; ?>
                </div>
                
                <?php
                $file_check = check_session_files();
                $class = ($file_check['file_exists'] || $file_check['alt_exists']) ? 'result-success' : 'result-warning';
                ?>
                
                <div class="result-box <?= $class ?>">
                    <h4>Session File Check</h4>
                    <ul>
                        <li><strong>Expected File:</strong> <?= $file_check['expected_path'] ?></li>
                        <li><strong>File Exists:</strong> <?= $file_check['file_exists'] ? 'Yes' : 'No' ?></li>
                        <li><strong>Alternative Exists:</strong> <?= $file_check['alt_exists'] ? 'Yes' : 'No' ?></li>
                        <li><strong>File Readable:</strong> <?= $file_check['readable'] ? 'Yes' : 'No' ?></li>
                        <li><strong>File Size:</strong> <?= $file_check['file_size'] ?> bytes</li>
                    </ul>
                    
                    <?php if (!$file_check['file_exists'] && !$file_check['alt_exists']): ?>
                        <div class="alert alert-warning">
                            <strong>Warning:</strong> No session file was found. This might be normal if the session was just created
                            or if the server uses a different session handler.
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
        
        <div class="card mb-4">
            <div class="card-header">
                <h2 class="card-title">Current Session Data</h2>
            </div>
            <div class="card-body">
                <pre><?php print_r($_SESSION); ?></pre>
                
                <?php 
                $has_user_id = isset($_SESSION['user_id']);
                $has_user_role = isset($_SESSION['user_role']);
                
                if ($has_user_id && $has_user_role):
                ?>
                    <div class="alert alert-success">
                        <strong>Good:</strong> Your session has both 'user_id' and 'user_role' set.
                        This means you should be properly authenticated.
                    </div>
                <?php elseif (!$has_user_id && !$has_user_role): ?>
                    <div class="alert alert-danger">
                        <strong>Problem:</strong> Your session is missing both 'user_id' and 'user_role'.
                        This indicates you're not authenticated.
                    </div>
                <?php else: ?>
                    <div class="alert alert-warning">
                        <strong>Partial Auth:</strong> Your session has incomplete authentication data.
                        This might cause issues with certain features.
                    </div>
                <?php endif; ?>
            </div>
        </div>
        
        <div class="card mb-4">
            <div class="card-header">
                <h2 class="card-title">Session Write Test</h2>
            </div>
            <div class="card-body">
                <?php
                // Perform a session write test
                if (isset($_GET['test_write']) && $_GET['test_write'] === 'true') {
                    $write_test = test_session_write();
                    $class = $write_test['success'] ? 'result-success' : 'result-error';
                    ?>
                    
                    <div class="result-box <?= $class ?>">
                        <h4>Session Write/Read Test Results</h4>
                        <ul>
                            <li><strong>Value Requested to Write:</strong> <?= $write_test['write_requested'] ?></li>
                            <li><strong>Value Read Back:</strong> <?= $write_test['read_back'] ?></li>
                            <li><strong>Test Successful:</strong> <?= $write_test['success'] ? 'Yes' : 'No' ?></li>
                        </ul>
                        
                        <?php if (!$write_test['success']): ?>
                            <div class="alert alert-danger">
                                <strong>Critical Issue:</strong> Session data could not be saved and retrieved correctly.
                                This explains why your authentication data isn't persisting.
                            </div>
                        <?php else: ?>
                            <div class="alert alert-success">
                                <strong>Good:</strong> Session data is being saved and retrieved correctly.
                            </div>
                        <?php endif; ?>
                    </div>
                <?php } else { ?>
                    <p>Click the button below to test session write/read functionality:</p>
                    <a href="?test_write=true" class="btn btn-primary">Test Session Write</a>
                <?php } ?>
            </div>
        </div>
        
        <div class="card mb-4">
            <div class="card-header">
                <h2 class="card-title">Cookie Information</h2>
            </div>
            <div class="card-body">
                <h3>Current Cookies</h3>
                <pre><?php print_r($_COOKIE); ?></pre>
                
                <?php
                $session_cookie_exists = isset($_COOKIE[session_name()]);
                $class = $session_cookie_exists ? 'result-success' : 'result-error';
                ?>
                
                <div class="result-box <?= $class ?>">
                    <h4>Session Cookie Check</h4>
                    <?php if ($session_cookie_exists): ?>
                        <p class="text-success">✅ Session cookie exists with value: <?= $_COOKIE[session_name()] ?></p>
                    <?php else: ?>
                        <p class="text-danger">❌ Session cookie does not exist! This will prevent session persistence.</p>
                    <?php endif; ?>
                </div>
            </div>
        </div>
        
        <div class="card mb-4">
            <div class="card-header">
                <h2 class="card-title">Fix Session Issues</h2>
            </div>
            <div class="card-body">
                <div class="mb-4">
                    <h4>1. Update Session Cookie Parameters</h4>
                    <?php
                    if (isset($_GET['fix_cookie']) && $_GET['fix_cookie'] === 'true') {
                        // Update session cookie parameters
                        $params = session_get_cookie_params();
                        session_set_cookie_params([
                            'lifetime' => $params['lifetime'],
                            'path' => '/',       // Set path to root for site-wide access
                            'domain' => '',      // Current domain
                            'secure' => isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on',
                            'httponly' => true,  // Protect from JS access
                            'samesite' => 'Lax' // Protect from CSRF
                        ]);
                        
                        // Regenerate session with updated parameters
                        session_regenerate_id(true);
                        
                        echo '<div class="alert alert-success">
                            <strong>Cookie parameters updated!</strong> Session cookie now uses path="/" and other secure settings.
                        </div>';
                        
                        // Set a test value
                        $_SESSION['cookie_fixed_at'] = date('Y-m-d H:i:s');
                    } else {
                        echo '<p>Click the button below to update session cookie parameters for site-wide access:</p>
                        <a href="?fix_cookie=true" class="btn btn-warning">Fix Session Cookie</a>';
                    }
                    ?>
                </div>
                
                <div class="mb-4">
                    <h4>2. Set Test Authentication Data</h4>
                    <?php
                    if (isset($_GET['set_auth']) && $_GET['set_auth'] === 'true') {
                        // Set test auth data
                        $_SESSION['user_id'] = $_GET['user_id'] ?? 1;
                        $_SESSION['user_role'] = $_GET['user_role'] ?? 'student';
                        
                        echo '<div class="alert alert-success">
                            <strong>Auth data set!</strong> Your session now has user_id='.$_SESSION['user_id'].' and user_role='.$_SESSION['user_role'].'.
                        </div>';
                    } else {
                        echo '<form action="" method="get">
                            <input type="hidden" name="set_auth" value="true">
                            <div class="mb-3">
                                <label for="user_id" class="form-label">User ID</label>
                                <input type="number" class="form-control" id="user_id" name="user_id" value="1">
                            </div>
                            <div class="mb-3">
                                <label for="user_role" class="form-label">User Role</label>
                                <select class="form-control" id="user_role" name="user_role">
                                    <option value="student">student</option>
                                    <option value="teacher">teacher</option>
                                    <option value="admin">admin</option>
                                </select>
                            </div>
                            <button type="submit" class="btn btn-warning">Set Auth Data</button>
                        </form>';
                    }
                    ?>
                </div>
            </div>
        </div>
        
        <div class="card mb-4">
            <div class="card-header">
                <h2 class="card-title">Test API Access</h2>
            </div>
            <div class="card-body">
                <p>After fixing session issues, test if the API can access your session data:</p>
                
                <button id="testApiBtn" class="btn btn-primary">Test API Session Access</button>
                
                <div id="apiResult" class="mt-3" style="display: none;">
                    <h4>API Response:</h4>
                    <pre id="apiResponse"></pre>
                </div>
                
                <script>
                document.getElementById('testApiBtn').addEventListener('click', function() {
                    fetch('/student/api/send_message.php', {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/x-www-form-urlencoded',
                        },
                        body: 'message=test&recipient_id=2'
                    })
                    .then(response => response.json())
                    .then(data => {
                        document.getElementById('apiResponse').textContent = JSON.stringify(data, null, 2);
                        document.getElementById('apiResult').style.display = 'block';
                    })
                    .catch(error => {
                        document.getElementById('apiResponse').textContent = 'Error: ' + error;
                        document.getElementById('apiResult').style.display = 'block';
                    });
                });
                </script>
            </div>
        </div>
        
        <div class="card mb-4">
            <div class="card-header">
                <h2 class="card-title">Install Permanent Fix</h2>
            </div>
            <div class="card-body">
                <form action="" method="post">
                    <input type="hidden" name="install_fix" value="true">
                    <p>Apply a permanent fix to ensure consistent session handling across all files:</p>
                    <button type="submit" class="btn btn-danger">Install Permanent Fix</button>
                </form>
                
                <?php
                if (isset($_POST['install_fix']) && $_POST['install_fix'] === 'true') {
                    // Create the file fix
                    $api_session_fix_content = <<<'EOD'
<?php
/**
 * Advanced Session Fix for PHP Applications
 * 
 * This file handles session initialization properly to ensure
 * consistent session access across all application parts including API endpoints.
 */

// Only proceed if session isn't already active
if (session_status() == PHP_SESSION_NONE) {
    // 1. Set session save path if needed
    $server_tmp = '/tmp';
    if (is_dir($server_tmp) && is_writable($server_tmp)) {
        session_save_path($server_tmp); // Use Linux server path
    } else {
        session_save_path(sys_get_temp_dir()); // Fallback to system temp
    }
    
    // 2. Set secure session cookie parameters
    session_set_cookie_params([
        'lifetime' => 0,               // Until browser is closed
        'path' => '/',                 // Available across the entire site
        'domain' => '',                // Current domain only
        'secure' => isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on',
        'httponly' => true,            // Not accessible via JavaScript
        'samesite' => 'Lax'            // Prevent CSRF but allow normal links
    ]);
    
    // 3. If we have a cookie but session isn't started, use that session ID
    $session_name = session_name();
    if (isset($_COOKIE[$session_name]) && !empty($_COOKIE[$session_name])) {
        session_id($_COOKIE[$session_name]);
    }
    
    // 4. Now start the session with all the correct parameters
    session_start();
    
    // 5. Regenerate ID if this is a new session (security best practice)
    if (!isset($_SESSION['initialized'])) {
        session_regenerate_id(true);
        $_SESSION['initialized'] = true;
    }
}
EOD;
                    
                    $file_path = '../../includes/advanced_session.php';
                    if (file_put_contents($file_path, $api_session_fix_content)) {
                        // Now update send_message.php to use the new fix
                        $send_message_update = <<<'EOD'
<?php
// API endpoint to send a message

// Include advanced session fix at the very top
require_once '../../includes/advanced_session.php';

// Include authentication functions
require_once '../../auth/functions.php';

// Log session for debugging
error_log('API Session: ' . print_r($_SESSION, true));
EOD;
                        
                        $api_file_path = '../../student/api/send_message.php';
                        if (file_put_contents($api_file_path, $send_message_update)) {
                            echo '<div class="alert alert-success mt-3">
                                <strong>Permanent fix installed successfully!</strong> 
                                <ul>
                                    <li>Created advanced_session.php in the includes folder</li>
                                    <li>Updated send_message.php to use the advanced session handling</li>
                                </ul>
                            </div>';
                        } else {
                            echo '<div class="alert alert-danger mt-3">
                                <strong>Error:</strong> Could not update send_message.php file.
                            </div>';
                        }
                    } else {
                        echo '<div class="alert alert-danger mt-3">
                            <strong>Error:</strong> Could not create advanced_session.php file.
                        </div>';
                    }
                }
                ?>
            </div>
        </div>
        
        <div class="card mb-4">
            <div class="card-header">
                <h2 class="card-title">Server Environment</h2>
            </div>
            <div class="card-body">
                <h3>Server Paths</h3>
                <pre><?php print_r($server_paths); ?></pre>
                
                <h3>PHP Version</h3>
                <p><?= phpversion() ?></p>
                
                <h3>Additional Links</h3>
                <ul>
                    <li><a href="php_info.php" target="_blank">View PHP Session Info</a></li>
                    <li><a href="../student/messages.php" target="_blank">Go to Messages Page</a></li>
                </ul>
            </div>
        </div>
    </div>
</body>
</html>
<?php
// End output buffering and send output
ob_end_flush();
?>
