<?php
// Test script to create a sample teacher and student for testing the messaging system
require_once '../config/db_connect.php';

echo "<h1>Messaging System Test</h1>";

// Check if we need to create test data
$createTestData = isset($_GET['create']) && $_GET['create'] == 'true';

// Function to create a test user
function createTestUser($conn, $username, $email, $firstName, $lastName, $role) {
    // Check if user already exists
    $stmt = $conn->prepare("SELECT user_id FROM users WHERE username = ? OR email = ?");
    $stmt->bind_param("ss", $username, $email);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows > 0) {
        $row = $result->fetch_assoc();
        return $row['user_id'];
    }
    
    // Create new user
    $hashedPassword = password_hash("password123", PASSWORD_DEFAULT);
    $stmt = $conn->prepare("INSERT INTO users (username, email, password, first_name, last_name, user_role, profile_picture) 
                           VALUES (?, ?, ?, ?, ?, ?, 'default.jpg')");
    $stmt->bind_param("ssssss", $username, $email, $hashedPassword, $firstName, $lastName, $role);
    $stmt->execute();
    
    return $conn->insert_id;
}

// Function to create a teacher
function createTeacher($conn, $userId, $firstName, $lastName) {
    // Check if teacher already exists
    $stmt = $conn->prepare("SELECT teacher_id FROM teachers WHERE user_id = ?");
    $stmt->bind_param("i", $userId);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows > 0) {
        $row = $result->fetch_assoc();
        return $row['teacher_id'];
    }
    
    // Create new teacher
    $bio = "Experienced $firstName $lastName with expertise in teaching.";
    $stmt = $conn->prepare("INSERT INTO teachers (user_id, first_name, last_name, bio, expertise) 
                           VALUES (?, ?, ?, ?, 'Teaching')");
    $stmt->bind_param("isss", $userId, $firstName, $lastName, $bio);
    $stmt->execute();
    
    return $conn->insert_id;
}

// Function to create a student
function createStudent($conn, $userId, $firstName, $lastName) {
    // Check if student already exists
    $stmt = $conn->prepare("SELECT student_id FROM students WHERE user_id = ?");
    $stmt->bind_param("i", $userId);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows > 0) {
        $row = $result->fetch_assoc();
        return $row['student_id'];
    }
    
    // Create new student
    $stmt = $conn->prepare("INSERT INTO students (user_id, first_name, last_name, bio) 
                           VALUES (?, ?, ?, 'Test student account')");
    $stmt->bind_param("iss", $userId, $firstName, $lastName);
    $stmt->execute();
    
    return $conn->insert_id;
}

// Function to create a course
function createCourse($conn, $teacherId, $title, $price = 99.99) {
    // Check if course already exists
    $stmt = $conn->prepare("SELECT course_id FROM courses WHERE teacher_id = ? AND title = ?");
    $stmt->bind_param("is", $teacherId, $title);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows > 0) {
        $row = $result->fetch_assoc();
        return $row['course_id'];
    }
    
    // Get or create a category
    $categoryId = 1; // Default category
    $categoryCheck = $conn->query("SELECT category_id FROM categories LIMIT 1");
    if ($categoryCheck && $categoryCheck->num_rows > 0) {
        $row = $categoryCheck->fetch_assoc();
        $categoryId = $row['category_id'];
    } else {
        // Create a default category
        $conn->query("INSERT INTO categories (name, description) VALUES ('General', 'General courses')");
        $categoryId = $conn->insert_id;
    }
    
    // Create new course
    $description = "This is a test course for $title";
    $isPublished = 1;
    $isApproved = 1;
    $stmt = $conn->prepare("INSERT INTO courses (teacher_id, category_id, title, description, price, is_published, is_approved, status) 
                           VALUES (?, ?, ?, ?, ?, ?, ?, 'active')");
    $stmt->bind_param("iissdii", $teacherId, $categoryId, $title, $description, $price, $isPublished, $isApproved);
    $stmt->execute();
    
    return $conn->insert_id;
}

// Function to enroll student in a course
function enrollStudent($conn, $studentId, $userId, $courseId) {
    // Check if enrollment already exists
    $stmt = $conn->prepare("SELECT enrollment_id FROM enrollments WHERE student_id = ? AND course_id = ?");
    $stmt->bind_param("ii", $studentId, $courseId);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows > 0) {
        $row = $result->fetch_assoc();
        return $row['enrollment_id'];
    }
    
    // Create new enrollment
    $stmt = $conn->prepare("INSERT INTO enrollments (student_id, user_id, course_id, progress) 
                           VALUES (?, ?, ?, 0)");
    $stmt->bind_param("iii", $studentId, $userId, $courseId);
    $stmt->execute();
    
    return $conn->insert_id;
}

// Function to create a message between users
function createMessage($conn, $senderId, $recipientId, $messageText) {
    $stmt = $conn->prepare("INSERT INTO messages (sender_id, recipient_id, message_text, is_read, sent_at) 
                           VALUES (?, ?, ?, 0, NOW())");
    $stmt->bind_param("iis", $senderId, $recipientId, $messageText);
    $stmt->execute();
    
    return $conn->insert_id;
}

// Display existing users
echo "<h2>Existing Users</h2>";
$users = $conn->query("SELECT user_id, username, email, first_name, last_name, user_role FROM users ORDER BY user_role, user_id LIMIT 10");

if ($users && $users->num_rows > 0) {
    echo "<table border='1' cellpadding='5'>
        <tr>
            <th>ID</th>
            <th>Username</th>
            <th>Email</th>
            <th>Name</th>
            <th>Role</th>
        </tr>";
        
    while ($user = $users->fetch_assoc()) {
        echo "<tr>
            <td>{$user['user_id']}</td>
            <td>{$user['username']}</td>
            <td>{$user['email']}</td>
            <td>{$user['first_name']} {$user['last_name']}</td>
            <td>{$user['user_role']}</td>
        </tr>";
    }
    
    echo "</table>";
} else {
    echo "<p>No users found.</p>";
}

// Create test data if requested
if ($createTestData) {
    echo "<h2>Creating Test Data</h2>";
    
    // Create teacher
    $teacherUserId = createTestUser($conn, "testteacher", "teacher@example.com", "Test", "Teacher", "teacher");
    $teacherId = createTeacher($conn, $teacherUserId, "Test", "Teacher");
    echo "<p>Created teacher: ID $teacherId, User ID: $teacherUserId</p>";
    
    // Create student
    $studentUserId = createTestUser($conn, "teststudent", "student@example.com", "Test", "Student", "student");
    $studentId = createStudent($conn, $studentUserId, "Test", "Student");
    echo "<p>Created student: ID $studentId, User ID: $studentUserId</p>";
    
    // Create course
    $courseId = createCourse($conn, $teacherId, "Test Course");
    echo "<p>Created course: ID $courseId</p>";
    
    // Enroll student
    $enrollmentId = enrollStudent($conn, $studentId, $studentUserId, $courseId);
    echo "<p>Created enrollment: ID $enrollmentId</p>";
    
    // Create messages
    $messageId1 = createMessage($conn, $studentUserId, $teacherUserId, "Hello teacher, I have a question about the course");
    echo "<p>Created message from student to teacher: ID $messageId1</p>";
    
    $messageId2 = createMessage($conn, $teacherUserId, $studentUserId, "Hello student, how can I help you?");
    echo "<p>Created message from teacher to student: ID $messageId2</p>";
    
    echo "<h3>Test Data Created Successfully!</h3>";
    echo "<p>You can now log in with:</p>";
    echo "<ul>
        <li>Teacher: username 'testteacher', password 'password123'</li>
        <li>Student: username 'teststudent', password 'password123'</li>
    </ul>";
    
    echo "<p><a href='../student/login.php'>Go to Login Page</a></p>";
    echo "<p><a href='../student/messages.php'>Go to Messages (as student)</a></p>";
} else {
    echo "<p><a href='?create=true'>Create Test Data</a></p>";
}

// Display existing teachers
echo "<h2>Existing Teachers</h2>";
$teachers = $conn->query("SELECT t.teacher_id, t.user_id, u.first_name, u.last_name, u.email 
                         FROM teachers t 
                         JOIN users u ON t.user_id = u.user_id 
                         LIMIT 10");

if ($teachers && $teachers->num_rows > 0) {
    echo "<table border='1' cellpadding='5'>
        <tr>
            <th>Teacher ID</th>
            <th>User ID</th>
            <th>Name</th>
            <th>Email</th>
        </tr>";
        
    while ($teacher = $teachers->fetch_assoc()) {
        echo "<tr>
            <td>{$teacher['teacher_id']}</td>
            <td>{$teacher['user_id']}</td>
            <td>{$teacher['first_name']} {$teacher['last_name']}</td>
            <td>{$teacher['email']}</td>
        </tr>";
    }
    
    echo "</table>";
} else {
    echo "<p>No teachers found.</p>";
}

// Display existing courses
echo "<h2>Existing Courses</h2>";
$courses = $conn->query("SELECT c.course_id, c.title, c.price, t.teacher_id, u.first_name, u.last_name 
                       FROM courses c
                       JOIN teachers t ON c.teacher_id = t.teacher_id
                       JOIN users u ON t.user_id = u.user_id
                       LIMIT 10");

if ($courses && $courses->num_rows > 0) {
    echo "<table border='1' cellpadding='5'>
        <tr>
            <th>Course ID</th>
            <th>Title</th>
            <th>Price</th>
            <th>Teacher</th>
        </tr>";
        
    while ($course = $courses->fetch_assoc()) {
        echo "<tr>
            <td>{$course['course_id']}</td>
            <td>{$course['title']}</td>
            <td>\${$course['price']}</td>
            <td>{$course['first_name']} {$course['last_name']} (ID: {$course['teacher_id']})</td>
        </tr>";
    }
    
    echo "</table>";
} else {
    echo "<p>No courses found.</p>";
}

// Display existing enrollments
echo "<h2>Existing Enrollments</h2>";
$enrollments = $conn->query("SELECT e.enrollment_id, e.student_id, e.course_id, s.user_id as student_user_id,
                           u_student.first_name as student_first_name, u_student.last_name as student_last_name,
                           c.title as course_title, u_teacher.first_name as teacher_first_name, u_teacher.last_name as teacher_last_name
                           FROM enrollments e
                           JOIN students s ON e.student_id = s.student_id
                           JOIN users u_student ON s.user_id = u_student.user_id
                           JOIN courses c ON e.course_id = c.course_id
                           JOIN teachers t ON c.teacher_id = t.teacher_id
                           JOIN users u_teacher ON t.user_id = u_teacher.user_id
                           LIMIT 10");

if ($enrollments && $enrollments->num_rows > 0) {
    echo "<table border='1' cellpadding='5'>
        <tr>
            <th>Enrollment ID</th>
            <th>Student</th>
            <th>Course</th>
            <th>Teacher</th>
        </tr>";
        
    while ($enrollment = $enrollments->fetch_assoc()) {
        echo "<tr>
            <td>{$enrollment['enrollment_id']}</td>
            <td>{$enrollment['student_first_name']} {$enrollment['student_last_name']} (ID: {$enrollment['student_id']}, User ID: {$enrollment['student_user_id']})</td>
            <td>{$enrollment['course_title']} (ID: {$enrollment['course_id']})</td>
            <td>{$enrollment['teacher_first_name']} {$enrollment['teacher_last_name']}</td>
        </tr>";
    }
    
    echo "</table>";
} else {
    echo "<p>No enrollments found.</p>";
}

// Display existing messages
echo "<h2>Existing Messages</h2>";
$messages = $conn->query("SELECT m.message_id, m.sender_id, m.recipient_id, m.message_text, m.is_read, m.sent_at,
                        u_sender.first_name as sender_first_name, u_sender.last_name as sender_last_name,
                        u_recipient.first_name as recipient_first_name, u_recipient.last_name as recipient_last_name
                        FROM messages m
                        JOIN users u_sender ON m.sender_id = u_sender.user_id
                        JOIN users u_recipient ON m.recipient_id = u_recipient.user_id
                        ORDER BY m.sent_at DESC
                        LIMIT 10");

if ($messages && $messages->num_rows > 0) {
    echo "<table border='1' cellpadding='5'>
        <tr>
            <th>Message ID</th>
            <th>From</th>
            <th>To</th>
            <th>Message</th>
            <th>Read</th>
            <th>Time</th>
        </tr>";
        
    while ($message = $messages->fetch_assoc()) {
        echo "<tr>
            <td>{$message['message_id']}</td>
            <td>{$message['sender_first_name']} {$message['sender_last_name']} (ID: {$message['sender_id']})</td>
            <td>{$message['recipient_first_name']} {$message['recipient_last_name']} (ID: {$message['recipient_id']})</td>
            <td>{$message['message_text']}</td>
            <td>" . ($message['is_read'] ? 'Yes' : 'No') . "</td>
            <td>{$message['sent_at']}</td>
        </tr>";
    }
    
    echo "</table>";
} else {
    echo "<p>No messages found.</p>";
}
?>
