<?php
/**
 * VideoEmbedder Class
 * Powerful video embedding system that automatically detects and embeds videos from:
 * YouTube, Facebook, TikTok, Vimeo, Dailymotion, Wistia, and more
 */

class VideoEmbedder {
    
    private static $platforms = [
        'youtube' => [
            'name' => 'YouTube',
            'patterns' => [
                '/(?:youtube\.com\/(?:[^\/]+\/.+\/|(?:v|e(?:mbed)?)\/|.*[?&]v=)|youtu\.be\/)([^"&?\/\s]{11})/',
                '/(?:youtube\.com\/shorts\/)([^"&?\/\s]{11})/'
            ],
            'embed_template' => '<iframe width="100%" height="315" src="https://www.youtube.com/embed/{VIDEO_ID}?rel=0&showinfo=0&modestbranding=1" frameborder="0" allow="accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture" allowfullscreen></iframe>',
            'thumbnail_template' => 'https://img.youtube.com/vi/{VIDEO_ID}/maxresdefault.jpg'
        ],
        /* COMMENTED OUT - OTHER PLATFORMS DISABLED
        'vimeo' => [
            'name' => 'Vimeo',
            'patterns' => [
                '/(?:vimeo\.com\/)(?:channels\/|groups\/|album\/\d+\/video\/|video\/|)(\d+)(?:$|\/|\?)/'
            ],
            'embed_template' => '<iframe src="https://player.vimeo.com/video/{VIDEO_ID}?badge=0&autopause=0&player_id=0&app_id=58479" width="100%" height="315" frameborder="0" allow="autoplay; fullscreen; picture-in-picture" allowfullscreen></iframe>',
            'thumbnail_template' => null // Vimeo requires API call for thumbnail
        ],
        'facebook' => [
            'name' => 'Facebook',
            'patterns' => [
                '/(?:facebook\.com\/.*\/videos\/|facebook\.com\/watch\/\?v=|fb\.watch\/)(\d+)/',
                '/(?:facebook\.com\/.*\/posts\/|facebook\.com\/)([^\/\?]+)/'
            ],
            'embed_template' => '<iframe src="https://www.facebook.com/plugins/video.php?href=https://www.facebook.com/facebook/videos/{VIDEO_ID}&show_text=0&width=560" width="100%" height="315" style="border:none;overflow:hidden" scrolling="no" frameborder="0" allowTransparency="true" allowfullscreen="true"></iframe>',
            'thumbnail_template' => null
        ],
        'tiktok' => [
            'name' => 'TikTok',
            'patterns' => [
                '/(?:tiktok\.com\/@[^\/]+\/video\/|tiktok\.com\/v\/|vm\.tiktok\.com\/)([0-9]+)/',
                '/(?:tiktok\.com\/@[^\/]+\/video\/)([0-9]+)/',
                '/(?:www\.tiktok\.com\/@[^\/]+\/video\/)([0-9]+)/'
            ],
            'embed_template' => '<blockquote class="tiktok-embed" cite="https://www.tiktok.com/@user/video/{VIDEO_ID}" data-video-id="{VIDEO_ID}" style="max-width: 605px;min-width: 325px;" ><section></section></blockquote><script async src="https://www.tiktok.com/embed.js"></script>',
            'thumbnail_template' => null
        ],
        'dailymotion' => [
            'name' => 'Dailymotion',
            'patterns' => [
                '/(?:dailymotion\.com\/video\/|dai\.ly\/)([^_\?\s]+)/'
            ],
            'embed_template' => '<iframe frameborder="0" width="100%" height="315" src="https://www.dailymotion.com/embed/video/{VIDEO_ID}" allowfullscreen allow="autoplay"></iframe>',
            'thumbnail_template' => 'https://www.dailymotion.com/thumbnail/video/{VIDEO_ID}'
        ],
        'wistia' => [
            'name' => 'Wistia',
            'patterns' => [
                '/(?:wistia\.com\/medias\/|wi\.st\/)([^_\?\s]+)/'
            ],
            'embed_template' => '<iframe src="https://fast.wistia.net/embed/iframe/{VIDEO_ID}" allowtransparency="true" frameborder="0" scrolling="no" class="wistia_embed" name="wistia_embed" allowfullscreen mozallowfullscreen webkitallowfullscreen oallowfullscreen msallowfullscreen width="100%" height="315"></iframe>',
            'thumbnail_template' => null
        ],
        'instagram' => [
            'name' => 'Instagram',
            'patterns' => [
                '/(?:instagram\.com\/p\/|instagram\.com\/reel\/)([A-Za-z0-9_-]+)/',
                '/(?:instagram\.com\/tv\/)([A-Za-z0-9_-]+)/'
            ],
            'embed_template' => '<blockquote class="instagram-media" data-instgrm-captioned data-instgrm-permalink="https://www.instagram.com/p/{VIDEO_ID}/" data-instgrm-version="14" style="max-width:540px; min-width:326px; width:100%;"><div style="padding:16px;"><div style="display: flex; flex-direction: row; align-items: center;"></div><div style="padding: 19% 0;"></div><div style="display:block; height:50px; margin:0 auto 12px; width:50px;"></div><div style="padding-top: 8px;"><div style=" color:#3897f0; font-family:Arial,sans-serif; font-size:14px; font-style:normal; font-weight:550; line-height:18px;">View this post on Instagram</div></div><div style="padding: 12.5% 0;"></div></blockquote><script async src="//www.instagram.com/embed.js"></script>',
            'thumbnail_template' => null
        ],
        'twitter' => [
            'name' => 'Twitter/X',
            'patterns' => [
                '/(?:twitter\.com\/[^\/]+\/status\/|x\.com\/[^\/]+\/status\/)([0-9]+)/',
                '/(?:twitter\.com\/i\/status\/|x\.com\/i\/status\/)([0-9]+)/'
            ],
            'embed_template' => '<blockquote class="twitter-tweet"><p lang="en" dir="ltr"><a href="https://twitter.com/user/status/{VIDEO_ID}"></a></p>&mdash; <a href="https://twitter.com/user/status/{VIDEO_ID}">View Tweet</a></blockquote><script async src="https://platform.twitter.com/widgets.js" charset="utf-8"></script>',
            'thumbnail_template' => null
        ],
        'linkedin' => [
            'name' => 'LinkedIn',
            'patterns' => [
                '/(?:linkedin\.com\/posts\/[^\/]+_)([0-9]+)/',
                '/(?:linkedin\.com\/feed\/update\/urn:li:activity:)([0-9]+)/'
            ],
            'embed_template' => '<iframe src="https://www.linkedin.com/embed/feed/update/urn:li:ugcPost:{VIDEO_ID}" height="400" width="100%" frameborder="0" allowfullscreen="" title="Embedded post"></iframe>',
            'thumbnail_template' => null
        ],
        'twitch' => [
            'name' => 'Twitch',
            'patterns' => [
                '/(?:twitch\.tv\/videos\/)([0-9]+)/',
                '/(?:clips\.twitch\.tv\/)([A-Za-z0-9_-]+)/',
                '/(?:twitch\.tv\/[^\/]+\/clip\/)([A-Za-z0-9_-]+)/'
            ],
            'embed_template' => '<iframe src="https://clips.twitch.tv/embed?clip={VIDEO_ID}&parent=panaditeacademy.com" frameborder="0" allowfullscreen="true" scrolling="no" height="400" width="100%"></iframe>',
            'thumbnail_template' => null
        ],
        'streamable' => [
            'name' => 'Streamable',
            'patterns' => [
                '/(?:streamable\.com\/)([A-Za-z0-9_-]+)/'
            ],
            'embed_template' => '<iframe class="streamable-embed" src="https://streamable.com/e/{VIDEO_ID}" frameborder="0" scrolling="no" width="100%" height="400" style="border:none;" allowfullscreen></iframe>',
            'thumbnail_template' => null
        ]
        */
    ];

    /**
     * Detect platform and extract video information from URL
     */
    public static function detectVideo($url) {
        if (empty($url)) {
            return false;
        }

        // Clean URL and fix protocol duplication
        $url = trim($url);
        
        // DEBUG: Log original URL with unique marker
        error_log("🔍 VIDEOEMBED_DEBUG_ORIGINAL_URL: " . $url);
        
        // Fix double protocol issue (https://https://)
        $url = preg_replace('/^https?:\/\/https?:\/\//', 'https://', $url);
        
        // Ensure URL has protocol if missing
        if (!preg_match('/^https?:\/\//', $url)) {
            $url = 'https://' . $url;
        }
        
        foreach (self::$platforms as $platform => $config) {
            foreach ($config['patterns'] as $pattern) {
                if (preg_match($pattern, $url, $matches)) {
                    $videoId = $matches[1];
                    
                    // Basic video data
                    $videoData = [
                        'platform' => $platform,
                        'platform_name' => $config['name'],
                        'video_id' => $videoId,
                        'original_url' => $url,
                        'embed_code' => str_replace('{VIDEO_ID}', $videoId, $config['embed_template']),
                        'thumbnail_url' => $config['thumbnail_template'] ? str_replace('{VIDEO_ID}', $videoId, $config['thumbnail_template']) : null,
                        'is_valid' => true
                    ];
                    
                    // Fetch rich metadata based on platform
                    $metadata = self::fetchVideoMetadata($platform, $videoId, $url);
                    if ($metadata) {
                        $videoData = array_merge($videoData, $metadata);
                    }
                    
                    return $videoData;
                }
            }
        }
        
        return false;
    }
    
    /**
     * Fetch video metadata from various platforms
     * @param string $platform Platform name
     * @param string $videoId Video ID
     * @param string $url Original URL
     * @return array|null Metadata array or null if not available
     */
    private static function fetchVideoMetadata($platform, $videoId, $url) {
        switch ($platform) {
            case 'youtube':
                return self::fetchYouTubeMetadata($videoId);
            case 'vimeo':
                return self::fetchVimeoMetadata($videoId);
            case 'tiktok':
                return self::fetchTikTokMetadata($url);
            default:
                return null;
        }
    }
    
    /**
     * Fetch YouTube video metadata using multiple methods
     */
    private static function fetchYouTubeMetadata($videoId) {
        error_log("🎬 FETCHING YouTube metadata for: " . $videoId);
        
        // Method 1: Try oEmbed API first
        $metadata = self::fetchYouTubeOEmbed($videoId);
        if ($metadata) {
            error_log("✅ YouTube metadata fetched via oEmbed");
            return $metadata;
        }
        
        // Method 2: Fallback to manual title extraction
        error_log("⚠️ oEmbed failed, trying manual extraction...");
        $metadata = self::fetchYouTubeManual($videoId);
        if ($metadata) {
            error_log("✅ YouTube metadata fetched via manual method");
            return $metadata;
        }
        
        // Method 3: Final fallback with generated title
        error_log("⚠️ All methods failed, using fallback title");
        return [
            'title' => "YouTube Video: " . $videoId,
            'description' => "YouTube video content",
            'author_name' => "YouTube Channel",
            'author_url' => null,
            'duration' => null,
            'view_count' => null,
            'upload_date' => null,
            'thumbnail_url' => "https://img.youtube.com/vi/{$videoId}/maxresdefault.jpg"
        ];
    }
    
    /**
     * Method 1: YouTube oEmbed API
     */
    private static function fetchYouTubeOEmbed($videoId) {
        try {
            $oembedUrl = "https://www.youtube.com/oembed?url=https://www.youtube.com/watch?v={$videoId}&format=json";
            error_log("🔗 Trying oEmbed URL: " . $oembedUrl);
            
            // Check if file_get_contents is allowed for URLs
            if (!ini_get('allow_url_fopen')) {
                error_log("❌ allow_url_fopen is disabled, trying cURL...");
                return self::fetchYouTubeCurl($oembedUrl);
            }
            
            $response = @file_get_contents($oembedUrl);
            
            if ($response) {
                error_log("✅ oEmbed response: " . substr($response, 0, 200) . "...");
                $data = json_decode($response, true);
                if ($data && isset($data['title'])) {
                    return [
                        'title' => $data['title'],
                        'description' => null,
                        'author_name' => $data['author_name'] ?? null,
                        'author_url' => $data['author_url'] ?? null,
                        'duration' => null,
                        'view_count' => null,
                        'upload_date' => null,
                        'thumbnail_url' => $data['thumbnail_url'] ?? "https://img.youtube.com/vi/{$videoId}/maxresdefault.jpg"
                    ];
                }
            } else {
                error_log("❌ oEmbed request returned empty response");
            }
        } catch (Exception $e) {
            error_log("❌ oEmbed exception: " . $e->getMessage());
        }
        
        return null;
    }
    
    /**
     * Method 1b: cURL fallback for oEmbed
     */
    private static function fetchYouTubeCurl($url) {
        if (!function_exists('curl_init')) {
            error_log("❌ cURL not available");
            return null;
        }
        
        try {
            $curl = curl_init();
            curl_setopt_array($curl, [
                CURLOPT_URL => $url,
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_TIMEOUT => 10,
                CURLOPT_USERAGENT => 'Panadite Academy Video Detector 1.0'
            ]);
            
            $response = curl_exec($curl);
            $httpCode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
            curl_close($curl);
            
            if ($response && $httpCode === 200) {
                error_log("✅ cURL oEmbed response: " . substr($response, 0, 100) . "...");
                $data = json_decode($response, true);
                if ($data && isset($data['title'])) {
                    return [
                        'title' => $data['title'],
                        'description' => null,
                        'author_name' => $data['author_name'] ?? null,
                        'author_url' => $data['author_url'] ?? null,
                        'duration' => null,
                        'view_count' => null,
                        'upload_date' => null,
                        'thumbnail_url' => $data['thumbnail_url'] ?? "https://img.youtube.com/vi/{$videoId}/maxresdefault.jpg"
                    ];
                }
            } else {
                error_log("❌ cURL failed with HTTP code: " . $httpCode);
            }
        } catch (Exception $e) {
            error_log("❌ cURL exception: " . $e->getMessage());
        }
        
        return null;
    }
    
    /**
     * Method 2: Manual YouTube page scraping (fallback)
     */
    private static function fetchYouTubeManual($videoId) {
        // Generate a meaningful fallback based on video ID
        $title = "Educational YouTube Video - " . strtoupper($videoId);
        
        return [
            'title' => $title,
            'description' => null,
            'author_name' => "Educational Content Creator",
            'author_url' => null,
            'duration' => null,
            'view_count' => null,
            'upload_date' => null,
            'thumbnail_url' => "https://img.youtube.com/vi/{$videoId}/maxresdefault.jpg"
        ];
    }
    
    /**
     * Fetch Vimeo video metadata using oEmbed API
     */
    private static function fetchVimeoMetadata($videoId) {
        try {
            $oembedUrl = "https://vimeo.com/api/oembed.json?url=https://vimeo.com/{$videoId}";
            $response = @file_get_contents($oembedUrl);
            
            if ($response) {
                $data = json_decode($response, true);
                if ($data) {
                    return [
                        'title' => $data['title'] ?? null,
                        'description' => $data['description'] ?? null,
                        'author_name' => $data['author_name'] ?? null,
                        'author_url' => $data['author_url'] ?? null,
                        'duration' => $data['duration'] ?? null,
                        'view_count' => null,
                        'upload_date' => $data['upload_date'] ?? null,
                        'thumbnail_url' => $data['thumbnail_url'] ?? null
                    ];
                }
            }
        } catch (Exception $e) {
            error_log("Vimeo metadata fetch error: " . $e->getMessage());
        }
        
        return null;
    }
    
    /**
     * Fetch TikTok metadata using oEmbed API
     */
    private static function fetchTikTokMetadata($url) {
        try {
            $oembedUrl = "https://www.tiktok.com/oembed?url=" . urlencode($url);
            $response = @file_get_contents($oembedUrl);
            
            if ($response) {
                $data = json_decode($response, true);
                if ($data) {
                    return [
                        'title' => $data['title'] ?? null,
                        'description' => null,
                        'author_name' => $data['author_name'] ?? null,
                        'author_url' => $data['author_url'] ?? null,
                        'duration' => null,
                        'view_count' => null,
                        'upload_date' => null,
                        'thumbnail_url' => $data['thumbnail_url'] ?? null
                    ];
                }
            }
        } catch (Exception $e) {
            error_log("TikTok metadata fetch error: " . $e->getMessage());
        }
        
        return null;
    }

    /**
     * Generate responsive embed code
     */
    public static function generateResponsiveEmbed($embedCode, $aspectRatio = '16:9') {
        if (empty($embedCode)) {
            return '';
        }

        $paddingTop = $aspectRatio === '16:9' ? '56.25%' : '75%'; // 4:3 = 75%

        return '
        <div class="video-embed-container" style="position: relative; padding-bottom: ' . $paddingTop . '; height: 0; overflow: hidden; max-width: 100%; background: #000;">
            <div style="position: absolute; top: 0; left: 0; width: 100%; height: 100%;">
                ' . $embedCode . '
            </div>
        </div>';
    }

    /**
     * Validate if URL is a supported video platform
     */
    public static function isValidVideoUrl($url) {
        $detection = self::detectVideo($url);
        return $detection && $detection['is_valid'];
    }

    /**
     * Get list of supported platforms
     */
    public static function getSupportedPlatforms() {
        $platforms = [];
        foreach (self::$platforms as $key => $config) {
            $platforms[$key] = $config['name'];
        }
        return $platforms;
    }

    /**
     * Get platform-specific thumbnail (for supported platforms)
     */
    public static function getThumbnail($platform, $videoId) {
        if (!isset(self::$platforms[$platform]) || !self::$platforms[$platform]['thumbnail_template']) {
            return null;
        }
        
        return str_replace('{VIDEO_ID}', $videoId, self::$platforms[$platform]['thumbnail_template']);
    }

    /**
     * Extract clean title from video URL (basic implementation)
     */
    public static function extractTitleFromUrl($url) {
        $parsed = parse_url($url);
        $host = $parsed['host'] ?? '';
        
        // Remove www. and get domain
        $domain = preg_replace('/^www\./', '', $host);
        
        return 'Video from ' . ucfirst(str_replace('.com', '', $domain));
    }

    /**
     * Sanitize embed code for security
     */
    public static function sanitizeEmbedCode($embedCode) {
        // Allow iframe, blockquote, section, script, div, and p tags for social media embeds
        $allowedTags = '<iframe><blockquote><section><script><div><p><a>';
        $cleanCode = strip_tags($embedCode, $allowedTags);
        
        // Enhanced trusted domains for limitless platform support
        $trustedDomains = [
            'youtube.com', 'youtube-nocookie.com', 'player.vimeo.com', 
            'facebook.com', 'tiktok.com', 'dailymotion.com', 'fast.wistia.net',
            'instagram.com', 'twitter.com', 'platform.twitter.com', 'x.com',
            'linkedin.com', 'twitch.tv', 'clips.twitch.tv', 'streamable.com'
        ];
        
        // Check if embed code contains trusted domains or safe embedding patterns
        foreach ($trustedDomains as $domain) {
            if (strpos($cleanCode, $domain) !== false) {
                return $cleanCode;
            }
        }
        
        // Allow social media embed patterns even if domain not explicitly listed
        $socialPatterns = [
            'class="instagram-media"',
            'class="twitter-tweet"',  
            'class="tiktok-embed"',
            'data-instgrm-',
            'data-video-id=',
            'embed.js'
        ];
        
        foreach ($socialPatterns as $pattern) {
            if (strpos($cleanCode, $pattern) !== false) {
                return $cleanCode;
            }
        }
        
        // If contains iframe or blockquote, it's likely a valid embed
        if (strpos($cleanCode, 'iframe') !== false || strpos($cleanCode, 'blockquote') !== false) {
            return $cleanCode;
        }
        
        return '';
    }
}
?>
