<?php
/**
 * Pure PHP AI Grading Service
 * Handles automated grading without external dependencies
 */

// Load Composer dependencies if available
if (file_exists(__DIR__ . '/../vendor/autoload.php')) {
    require_once __DIR__ . '/../vendor/autoload.php';
}
require_once __DIR__ . '/robust_document_processor.php';

class PHPAIGradingService {
    private $conn;
    private $debug;
    private $documentProcessor;
    
    public function __construct($connection, $debug = false) {
        $this->conn = $connection;
        $this->debug = $debug;
        $this->documentProcessor = new RobustDocumentProcessor($debug);
        
        // Check if Composer autoloader is available
        if (!class_exists('Smalot\PdfParser\Parser')) {
            error_log('Warning: Smalot PDF Parser not available. Using fallback text extraction.');
        }
    }
    
    /**
     * Public method to extract text from files
     */
    public function extractText($filePath) {
        try {
            return $this->documentProcessor->extractText($filePath);
        } catch (Exception $e) {
            error_log('Document extraction failed: ' . $e->getMessage());
            throw $e;
        }
    }
    
    /**
     * Test document extraction and return detailed results
     */
    public function testDocumentExtraction($filePath) {
        return $this->documentProcessor->testExtraction($filePath);
    }
    
    /**
     * Main grading function - grades student submission against memorandum
     */
    public function gradeAssignment($studentText, $memorandumData, $assignmentId = null) {
        try {
            // Validate input
            if (empty($studentText) || trim($studentText) === '') {
                throw new Exception('Empty submission text provided');
            }
            
            if (empty($memorandumData) || trim($memorandumData) === '') {
                throw new Exception('No memorandum content available for grading');
            }
            
            // Parse memorandum if it's JSON
            if (is_string($memorandumData)) {
                $memorandumData = json_decode($memorandumData, true);
            }
            
            $memorandumText = $memorandumData['full_text'] ?? $memorandumData;
            
            // Process both texts
            $studentWords = $this->processText($studentText);
            $memoWords = $this->processText($memorandumText);
            
            if (empty($studentWords)) {
                throw new Exception('No valid words found in student submission');
            }
            
            if (empty($memoWords)) {
                throw new Exception('No valid words found in memorandum');
            }
            
            // Calculate similarity
            $similarity = $this->calculateSimilarity($studentText, $memorandumText);
            
            // Calculate keyword coverage
            $keywordCoverage = $this->calculateKeywordCoverage($studentWords, $memoWords);
            
            // Calculate quality metrics
            $qualityScore = $this->assessQuality($studentText);
            
            // Generate final grade
            $finalGrade = $this->generateGrade($similarity, $keywordCoverage, $qualityScore);
            
            return [
                'grade' => $finalGrade,
                'similarity' => round($similarity, 2),
                'keyword_coverage' => round($keywordCoverage, 2),
                'quality' => round($qualityScore, 2),
                'feedback' => $this->generateFeedback($similarity, $keywordCoverage, $qualityScore),
                'ai_confidence' => $this->calculateConfidence([
                    'similarity' => $similarity,
                    'keyword_coverage' => $keywordCoverage,
                    'quality' => $qualityScore
                ])
            ];
            
        } catch (Exception $e) {
            error_log('AI Grading error: ' . $e->getMessage());
            throw $e;
        }
    }
    
    /**
     * Process text and extract meaningful words
     */
    public function processText($text) {
        // Clean and normalize text
        $text = strtolower(trim($text));
        $text = preg_replace('/[^\w\s]/', ' ', $text);
        $text = preg_replace('/\s+/', ' ', $text);
        
        // Extract words
        $words = explode(' ', $text);
        
        // Filter stop words and short words
        $stopWords = $this->getStopWords();
        $processedWords = [];
        
        foreach ($words as $word) {
            $word = trim($word);
            if (strlen($word) > 2 && !in_array($word, $stopWords)) {
                $processedWords[] = $this->stemWord($word);
            }
        }
        
        return array_unique($processedWords);
    }
    
    /**
     * Calculate text similarity using multiple algorithms
     */
    public function calculateSimilarity($text1, $text2) {
        $text1 = strtolower(trim($text1));
        $text2 = strtolower(trim($text2));
        
        if (empty($text1) || empty($text2)) {
            return 0;
        }
        
        // Method 1: Simple word overlap
        $words1 = $this->processText($text1);
        $words2 = $this->processText($text2);
        
        $intersection = count(array_intersect($words1, $words2));
        $union = count(array_unique(array_merge($words1, $words2)));
        
        $wordSimilarity = $union > 0 ? ($intersection / $union) * 100 : 0;
        
        // Method 2: Character-based similarity
        similar_text($text1, $text2, $charSimilarity);
        
        // Combine both methods
        return ($wordSimilarity + $charSimilarity) / 2;
    }
    
    /**
     * Calculate keyword coverage
     */
    public function calculateKeywordCoverage($studentWords, $memoWords) {
        if (empty($memoWords)) return 0;
        
        $coveredKeywords = array_intersect($studentWords, $memoWords);
        return (count($coveredKeywords) / count($memoWords)) * 100;
    }
    
    /**
     * Assess text quality
     */
    public function assessQuality($text) {
        $score = 0;
        $factors = 0;
        
        // Length factor
        $length = strlen(trim($text));
        if ($length > 100) {
            $score += 25;
        } elseif ($length > 50) {
            $score += 15;
        } elseif ($length > 20) {
            $score += 10;
        }
        $factors++;
        
        // Word count factor
        $wordCount = str_word_count($text);
        if ($wordCount > 50) {
            $score += 25;
        } elseif ($wordCount > 25) {
            $score += 15;
        } elseif ($wordCount > 10) {
            $score += 10;
        }
        $factors++;
        
        // Sentence structure (basic)
        $sentences = preg_split('/[.!?]+/', $text);
        $validSentences = count(array_filter($sentences, function($s) { 
            return strlen(trim($s)) > 5; 
        }));
        
        if ($validSentences > 3) {
            $score += 25;
        } elseif ($validSentences > 1) {
            $score += 15;
        }
        $factors++;
        
        // Vocabulary diversity
        $words = $this->processText($text);
        $uniqueWords = count(array_unique($words));
        $totalWords = count($words);
        
        if ($totalWords > 0) {
            $diversity = ($uniqueWords / $totalWords) * 100;
            if ($diversity > 70) {
                $score += 25;
            } elseif ($diversity > 50) {
                $score += 15;
            } elseif ($diversity > 30) {
                $score += 10;
            }
        }
        $factors++;
        
        return $factors > 0 ? $score / $factors : 0;
    }
    
    /**
     * Generate final grade based on metrics
     */
    public function generateGrade($similarity, $keywordCoverage, $quality) {
        // Weighted scoring
        $weights = [
            'similarity' => 0.4,
            'keyword_coverage' => 0.4,
            'quality' => 0.2
        ];
        
        $totalScore = ($similarity * $weights['similarity']) + 
                     ($keywordCoverage * $weights['keyword_coverage']) + 
                     ($quality * $weights['quality']);
        
        return max(0, min(100, round($totalScore, 2)));
    }
    
    /**
     * Generate feedback based on metrics
     */
    public function generateFeedback($similarity, $keywordCoverage, $quality) {
        $feedback = [];
        
        if ($similarity < 30) {
            $feedback[] = "Your answer doesn't closely match the expected content. Consider reviewing the key concepts.";
        } elseif ($similarity < 60) {
            $feedback[] = "Your answer has some relevant content, but could be more comprehensive.";
        } else {
            $feedback[] = "Good job! Your answer demonstrates understanding of the key concepts.";
        }
        
        if ($keywordCoverage < 40) {
            $feedback[] = "Try to include more of the important keywords from the topic.";
        } elseif ($keywordCoverage < 70) {
            $feedback[] = "You've covered some important points, but consider adding more detail.";
        }
        
        if ($quality < 40) {
            $feedback[] = "Consider expanding your answer with more detailed explanations.";
        }
        
        return implode(' ', $feedback);
    }
    
    /**
     * Calculate AI confidence in the grading
     */
    public function calculateConfidence($metrics) {
        $similarity = $metrics['similarity'];
        $keywordCoverage = $metrics['keyword_coverage'];
        $quality = $metrics['quality'];
        
        // Higher confidence when metrics are consistent
        $variance = [
            abs($similarity - $keywordCoverage),
            abs($similarity - $quality),
            abs($keywordCoverage - $quality)
        ];
        
        $avgVariance = array_sum($variance) / count($variance);
        $consistency = max(0, 100 - $avgVariance);
        
        // Confidence decreases with extreme scores (very high or very low)
        $avgScore = ($similarity + $keywordCoverage + $quality) / 3;
        $extremeness = abs($avgScore - 50);
        $moderateness = max(0, 50 - $extremeness) * 2;
        
        $confidence = ($consistency + $moderateness) / 2;
        return round(min(100, max(0, $confidence)), 2);
    }
    
    /**
     * Get stop words for text processing
     */
    private function getStopWords() {
        return [
            'the', 'a', 'an', 'and', 'or', 'but', 'in', 'on', 'at', 'to', 'for', 'of', 'with',
            'by', 'is', 'are', 'was', 'were', 'be', 'been', 'have', 'has', 'had', 'do', 'does',
            'did', 'will', 'would', 'could', 'should', 'may', 'might', 'must', 'can', 'this',
            'that', 'these', 'those', 'i', 'you', 'he', 'she', 'it', 'we', 'they', 'me', 'him',
            'her', 'us', 'them', 'my', 'your', 'his', 'her', 'its', 'our', 'their'
        ];
    }
    
    /**
     * Basic word stemming
     */
    private function stemWord($word) {
        $suffixes = ['ing', 'ed', 'er', 'est', 'ly', 'ion', 'tion', 'ness', 'ment'];
        
        foreach ($suffixes as $suffix) {
            if (strlen($word) > strlen($suffix) + 2 && substr($word, -strlen($suffix)) === $suffix) {
                return substr($word, 0, -strlen($suffix));
            }
        }
        
        return $word;
    }

    /**
     * Extract keywords from text (used by AJAX endpoint)
     */
    public function extractKeywords($text) {
        if (empty($text)) {
            return [];
        }
        
        // Use processText to get meaningful keywords
        $words = $this->processText($text);
        
        // Filter for significant keywords (longer words are more meaningful)
        $keywords = array_filter($words, function($word) {
            return strlen($word) > 3; // Only words longer than 3 characters
        });
        
        return array_values($keywords);
    }
    
    /**
     * Check keyword matches between student text and keywords (used by AJAX endpoint)
     */
    public function checkKeywordMatches($studentText, $keywords) {
        if (empty($studentText) || empty($keywords)) {
            return [];
        }
        
        // Process student text to get comparable words
        $studentWords = $this->processText($studentText);
        
        // Find matches
        $matches = [];
        foreach ($keywords as $keyword) {
            if (in_array(strtolower($keyword), $studentWords)) {
                $matches[] = $keyword;
            }
        }
        
        return $matches;
    }
}
?>
