<?php
/**
 * Fresh AI Grading Service - Clean implementation
 */

class PHPAIGradingService {
    private $conn;
    private $debug;
    
    public function __construct($connection, $debug = false) {
        $this->conn = $connection;
        $this->debug = $debug;
    }
    
    /**
     * Main grading function
     */
    public function gradeAssignment($studentText, $memorandumData, $assignmentId = null) {
        try {
            // Debug parameters
            if ($this->debug) {
                error_log("DEBUG: gradeAssignment called");
                error_log("DEBUG: studentText type: " . gettype($studentText));
                error_log("DEBUG: memorandumData type: " . gettype($memorandumData));
                error_log("DEBUG: studentText length: " . strlen($studentText));
                error_log("DEBUG: memorandumData length: " . strlen($memorandumData));
            }
            
            // Validate input
            if (empty($studentText) || trim($studentText) === '') {
                throw new Exception('Empty submission text provided');
            }
            
            if (empty($memorandumData) || trim($memorandumData) === '') {
                throw new Exception('No memorandum content available for grading');
            }
            
            // Convert memorandum to plain text
            $memorandumText = $this->extractMemorandumText($memorandumData);
            
            if ($this->debug) {
                error_log("DEBUG: memorandumText after extraction: " . strlen($memorandumText));
            }
            
            // Process both texts
            $studentWords = $this->processText($studentText);
            $memoWords = $this->processText($memorandumText);
            
            if ($this->debug) {
                error_log("DEBUG: studentWords count: " . count($studentWords));
                error_log("DEBUG: memoWords count: " . count($memoWords));
            }
            
            if (empty($studentWords)) {
                throw new Exception('No valid words found in student submission');
            }
            
            if (empty($memoWords)) {
                throw new Exception('No valid words found in memorandum');
            }
            
            // Calculate metrics
            $similarity = $this->calculateSimilarity($studentText, $memorandumText);
            $keywordCoverage = $this->calculateKeywordCoverage($studentWords, $memoWords);
            $qualityScore = $this->assessQuality($studentText);
            
            // Generate final grade
            $finalGrade = $this->generateGrade($similarity, $keywordCoverage, $qualityScore);
            
            return [
                'ai_score' => $finalGrade,
                'ai_feedback' => $this->generateFeedback($similarity, $keywordCoverage, $qualityScore),
                'similarity_score' => round($similarity, 2),
                'plagiarism_score' => round(max(0, 100 - $similarity), 2),
                'keyword_match_score' => round($keywordCoverage, 2),
                'structure_score' => round($qualityScore, 2),
                'quality_score' => round($qualityScore, 2),
                'review_needed' => ($finalGrade < 50 || $similarity < 30) ? 1 : 0,
                'ai_confidence' => $this->calculateConfidence([
                    'similarity' => $similarity,
                    'keyword_coverage' => $keywordCoverage,
                    'quality' => $qualityScore
                ])
            ];
            
        } catch (Exception $e) {
            if ($this->debug) {
                error_log('AI Grading error: ' . $e->getMessage());
            }
            throw $e;
        }
    }
    
    /**
     * Extract memorandum text from various formats
     */
    private function extractMemorandumText($memorandumData) {
        if (is_string($memorandumData)) {
            // Try to decode as JSON first
            $decodedData = json_decode($memorandumData, true);
            if (json_last_error() === JSON_ERROR_NONE && is_array($decodedData)) {
                return $decodedData['full_text'] ?? $decodedData['content'] ?? $memorandumData;
            } else {
                return $memorandumData;
            }
        } else if (is_array($memorandumData)) {
            return $memorandumData['full_text'] ?? $memorandumData['content'] ?? (string)$memorandumData;
        } else {
            return (string)$memorandumData;
        }
    }
    
    /**
     * Process text and extract meaningful words
     */
    public function processText($text) {
        if (empty($text)) {
            return [];
        }
        
        // Clean and normalize text
        $text = strtolower(trim($text));
        $text = preg_replace('/[^\w\s]/', ' ', $text);
        $text = preg_replace('/\s+/', ' ', $text);
        
        // Extract words
        $words = explode(' ', $text);
        
        // Filter stop words and short words
        $stopWords = $this->getStopWords();
        $processedWords = [];
        
        foreach ($words as $word) {
            $word = trim($word);
            if (strlen($word) > 2 && !in_array($word, $stopWords)) {
                $processedWords[] = $this->stemWord($word);
            }
        }
        
        return array_unique($processedWords);
    }
    
    /**
     * Calculate text similarity
     */
    public function calculateSimilarity($text1, $text2) {
        $words1 = $this->processText($text1);
        $words2 = $this->processText($text2);
        
        if (empty($words1) || empty($words2)) {
            return 0;
        }
        
        $intersection = array_intersect($words1, $words2);
        $union = array_unique(array_merge($words1, $words2));
        
        return (count($intersection) / count($union)) * 100;
    }
    
    /**
     * Calculate keyword coverage
     */
    public function calculateKeywordCoverage($studentWords, $memoWords) {
        if (empty($studentWords) || empty($memoWords)) {
            return 0;
        }
        
        $matches = array_intersect($studentWords, $memoWords);
        return (count($matches) / count($memoWords)) * 100;
    }
    
    /**
     * Assess text quality
     */
    public function assessQuality($text) {
        $score = 0;
        $factors = 0;
        
        // Length factor
        $wordCount = str_word_count($text);
        if ($wordCount > 50) {
            $score += min(100, ($wordCount / 200) * 100);
        } else {
            $score += ($wordCount / 50) * 50;
        }
        $factors++;
        
        // Sentence structure
        $sentences = preg_split('/[.!?]+/', $text);
        $avgWordsPerSentence = $wordCount / max(1, count($sentences));
        if ($avgWordsPerSentence >= 8 && $avgWordsPerSentence <= 25) {
            $score += 80;
        } else {
            $score += max(20, 80 - abs($avgWordsPerSentence - 16) * 3);
        }
        $factors++;
        
        return $factors > 0 ? $score / $factors : 0;
    }
    
    /**
     * Generate final grade
     */
    public function generateGrade($similarity, $keywordCoverage, $quality) {
        $weights = [
            'similarity' => 0.4,
            'keyword_coverage' => 0.4,
            'quality' => 0.2
        ];
        
        $totalScore = ($similarity * $weights['similarity']) + 
                     ($keywordCoverage * $weights['keyword_coverage']) + 
                     ($quality * $weights['quality']);
        
        return max(0, min(100, round($totalScore, 2)));
    }
    
    /**
     * Generate feedback
     */
    public function generateFeedback($similarity, $keywordCoverage, $quality) {
        $feedback = [];
        
        if ($similarity < 30) {
            $feedback[] = "Your answer doesn't closely match the expected content. Consider reviewing the key concepts.";
        } elseif ($similarity < 60) {
            $feedback[] = "Your answer has some relevant content, but could be more comprehensive.";
        } else {
            $feedback[] = "Good job! Your answer demonstrates understanding of the key concepts.";
        }
        
        if ($keywordCoverage < 40) {
            $feedback[] = "Try to include more key terms from the course material.";
        } elseif ($keywordCoverage > 70) {
            $feedback[] = "Excellent use of key terminology!";
        }
        
        if ($quality < 50) {
            $feedback[] = "Consider expanding your answer with more detail and better structure.";
        }
        
        return implode(' ', $feedback);
    }
    
    /**
     * Calculate confidence
     */
    public function calculateConfidence($metrics) {
        $similarity = $metrics['similarity'];
        $keywordCoverage = $metrics['keyword_coverage'];
        $quality = $metrics['quality'];
        
        $variance = [
            abs($similarity - $keywordCoverage),
            abs($similarity - $quality),
            abs($keywordCoverage - $quality)
        ];
        
        $avgVariance = array_sum($variance) / count($variance);
        $consistency = max(0, 100 - $avgVariance);
        
        $avgScore = ($similarity + $keywordCoverage + $quality) / 3;
        $extremeness = abs($avgScore - 50);
        $moderateness = max(0, 50 - $extremeness) * 2;
        
        $confidence = ($consistency + $moderateness) / 2;
        return round(min(100, max(0, $confidence)), 2);
    }
    
    /**
     * Get stop words
     */
    private function getStopWords() {
        return [
            'the', 'a', 'an', 'and', 'or', 'but', 'in', 'on', 'at', 'to', 'for', 'of', 'with', 'by',
            'is', 'are', 'was', 'were', 'be', 'been', 'being', 'have', 'has', 'had', 'do', 'does', 'did',
            'will', 'would', 'should', 'could', 'can', 'may', 'might', 'must', 'shall',
            'this', 'that', 'these', 'those', 'i', 'you', 'he', 'she', 'it', 'we', 'they',
            'me', 'him', 'her', 'us', 'them', 'my', 'your', 'his', 'her', 'its', 'our', 'their'
        ];
    }
    
    /**
     * Basic word stemming
     */
    private function stemWord($word) {
        // Remove common suffixes
        $suffixes = ['ing', 'ed', 'er', 'est', 'ly', 'tion', 'sion', 'ness', 'ment', 'able', 'ible'];
        
        foreach ($suffixes as $suffix) {
            if (strlen($word) > strlen($suffix) + 3 && substr($word, -strlen($suffix)) === $suffix) {
                return substr($word, 0, -strlen($suffix));
            }
        }
        
        return $word;
    }
    
    /**
     * Extract keywords from text (for AJAX compatibility)
     */
    public function extractKeywords($text) {
        return $this->processText($text);
    }
    
    /**
     * Check keyword matches (for AJAX compatibility)
     */
    public function checkKeywordMatches($studentText, $keywords) {
        $studentWords = $this->processText($studentText);
        $matches = array_intersect($studentWords, $keywords);
        return count($matches);
    }
}
?>
