<?php
/**
 * Fresh AI Grading Service - Clean implementation
 */

class PHPAIGradingService {
    private $conn;
    private $debug;
    
    public function __construct($connection, $debug = false) {
        $this->conn = $connection;
        $this->debug = $debug;
    }
    
    /**
     * Main grading function
     */
    public function gradeAssignment($studentText, $memorandumData, $assignmentId = null) {
        try {
            // Debug parameters
            if ($this->debug) {
                error_log("DEBUG: gradeAssignment called");
                error_log("DEBUG: studentText type: " . gettype($studentText));
                error_log("DEBUG: memorandumData type: " . gettype($memorandumData));
                error_log("DEBUG: studentText length: " . strlen($studentText));
                error_log("DEBUG: memorandumData length: " . strlen($memorandumData));
            }
            
            // Validate input
            if (empty($studentText) || trim($studentText) === '') {
                throw new Exception('Empty submission text provided');
            }
            
            if (empty($memorandumData) || trim($memorandumData) === '') {
                throw new Exception('No memorandum content available for grading');
            }
            
            // Convert memorandum to plain text
            $memorandumText = $this->extractMemorandumText($memorandumData);
            
            if ($this->debug) {
                error_log("DEBUG: memorandumText after extraction: " . strlen($memorandumText));
            }
            
            // Process both texts
            $studentWords = $this->processText($studentText);
            $memoWords = $this->processText($memorandumText);
            
            if ($this->debug) {
                error_log("DEBUG: studentWords count: " . count($studentWords));
                error_log("DEBUG: memoWords count: " . count($memoWords));
            }
            
            if (empty($studentWords)) {
                throw new Exception('No valid words found in student submission');
            }
            
            if (empty($memoWords)) {
                throw new Exception('No valid words found in memorandum');
            }
            
            // Calculate metrics
            $similarity = $this->calculateSimilarity($studentText, $memorandumText);
            $keywordCoverage = $this->calculateKeywordCoverage($studentWords, $memoWords);
            $qualityScore = $this->assessQuality($studentText);
            
            // Generate final grade
            $finalGrade = $this->generateGrade($similarity, $keywordCoverage, $qualityScore);
            
            return [
                'ai_score' => $finalGrade,
                'ai_feedback' => $this->generateFeedback($similarity, $keywordCoverage, $qualityScore),
                'similarity_score' => round($similarity, 2),
                'plagiarism_score' => round(max(0, 100 - $similarity), 2),
                'keyword_match_score' => round($keywordCoverage, 2),
                'structure_score' => round($qualityScore, 2),
                'quality_score' => round($qualityScore, 2),
                'review_needed' => ($finalGrade < 50 || $similarity < 30) ? 1 : 0,
                'ai_confidence' => $this->calculateConfidence([
                    'similarity' => $similarity,
                    'keyword_coverage' => $keywordCoverage,
                    'quality' => $qualityScore
                ])
            ];
            
        } catch (Exception $e) {
            if ($this->debug) {
                error_log('AI Grading error: ' . $e->getMessage());
            }
            throw $e;
        }
    }
    
    /**
     * Extract memorandum text from various formats
     */
    private function extractMemorandumText($memorandumData) {
        if (is_string($memorandumData)) {
            // Try to decode as JSON first
            $decodedData = json_decode($memorandumData, true);
            if (json_last_error() === JSON_ERROR_NONE && is_array($decodedData)) {
                return $decodedData['full_text'] ?? $decodedData['content'] ?? $memorandumData;
            } else {
                return $memorandumData;
            }
        } else if (is_array($memorandumData)) {
            return $memorandumData['full_text'] ?? $memorandumData['content'] ?? (string)$memorandumData;
        } else {
            return (string)$memorandumData;
        }
    }
    
    /**
     * Process text and extract meaningful words
     */
    public function processText($text) {
        if (empty($text)) {
            return [];
        }
        
        // Clean and normalize text
        $text = strtolower(trim($text));
        $text = preg_replace('/[^\w\s]/', ' ', $text);
        $text = preg_replace('/\s+/', ' ', $text);
        
        // Extract words
        $words = explode(' ', $text);
        
        // Filter stop words and short words
        $stopWords = $this->getStopWords();
        $processedWords = [];
        
        foreach ($words as $word) {
            $word = trim($word);
            if (strlen($word) > 2 && !in_array($word, $stopWords)) {
                $processedWords[] = $this->stemWord($word);
            }
        }
        
        return array_unique($processedWords);
    }
    
    /**
     * Calculate intelligent text similarity using multiple advanced algorithms
     */
    public function calculateSimilarity($text1, $text2) {
        $words1 = $this->processText($text1);
        $words2 = $this->processText($text2);
        
        if (empty($words1) || empty($words2)) {
            return 0;
        }
        
        // Multi-factor similarity analysis
        $jaccardSim = $this->calculateJaccardSimilarity($words1, $words2);
        $cosineSim = $this->calculateCosineSimilarity($text1, $text2);
        $conceptSim = $this->calculateConceptualSimilarity($text1, $text2);
        $contextSim = $this->calculateContextualRelevance($text1, $text2);
        
        // Weighted combination for more accurate results
        $finalSimilarity = (
            $jaccardSim * 0.2 +     // Basic word overlap
            $cosineSim * 0.3 +      // TF-IDF similarity
            $conceptSim * 0.3 +     // Conceptual understanding
            $contextSim * 0.2       // Contextual relevance
        );
        
        if ($this->debug) {
            error_log("SIMILARITY DEBUG: Jaccard={$jaccardSim}, Cosine={$cosineSim}, Concept={$conceptSim}, Context={$contextSim}, Final={$finalSimilarity}");
        }
        
        return round($finalSimilarity, 2);
    }
    
    /**
     * Calculate Jaccard similarity (basic word overlap)
     */
    private function calculateJaccardSimilarity($words1, $words2) {
        $intersection = array_intersect($words1, $words2);
        $union = array_unique(array_merge($words1, $words2));
        return (count($intersection) / count($union)) * 100;
    }
    
    /**
     * Calculate cosine similarity using TF-IDF
     */
    private function calculateCosineSimilarity($text1, $text2) {
        $tfidf1 = $this->calculateTFIDF($text1, [$text1, $text2]);
        $tfidf2 = $this->calculateTFIDF($text2, [$text1, $text2]);
        
        $dotProduct = 0;
        $norm1 = 0;
        $norm2 = 0;
        
        $allTerms = array_unique(array_merge(array_keys($tfidf1), array_keys($tfidf2)));
        
        foreach ($allTerms as $term) {
            $val1 = $tfidf1[$term] ?? 0;
            $val2 = $tfidf2[$term] ?? 0;
            
            $dotProduct += $val1 * $val2;
            $norm1 += $val1 * $val1;
            $norm2 += $val2 * $val2;
        }
        
        if ($norm1 == 0 || $norm2 == 0) return 0;
        
        // Error-safe square root calculation
        $sqrt1 = sqrt(max(0, $norm1));
        $sqrt2 = sqrt(max(0, $norm2));
        
        if ($sqrt1 == 0 || $sqrt2 == 0) return 0;
        
        $result = ($dotProduct / ($sqrt1 * $sqrt2)) * 100;
        return is_finite($result) ? $result : 0;
    }
    
    /**
     * Calculate TF-IDF for text
     */
    private function calculateTFIDF($text, $corpus) {
        $words = $this->processText($text);
        $totalWords = count($words);
        $wordFreq = array_count_values($words);
        $tfidf = [];
        
        // Error-safe TF-IDF calculation
        if ($totalWords == 0 || empty($wordFreq)) {
            return [];
        }
        
        foreach ($wordFreq as $word => $freq) {
            $tf = $totalWords > 0 ? ($freq / $totalWords) : 0;
            $df = 0;
            
            // Count document frequency
            foreach ($corpus as $doc) {
                if (!empty($doc) && strpos(strtolower($doc), strtolower($word)) !== false) {
                    $df++;
                }
            }
            
            // Error-safe IDF calculation
            $corpusSize = count($corpus);
            if ($df > 0 && $corpusSize > 0) {
                $logValue = $corpusSize / $df;
                $idf = $logValue > 0 ? log($logValue) : 0;
            } else {
                $idf = 0;
            }
            
            $tfidfValue = $tf * $idf;
            $tfidf[$word] = is_finite($tfidfValue) ? $tfidfValue : 0;
        }
        
        return $tfidf;
    }
    
    /**
     * Calculate conceptual similarity based on academic concepts
     */
    private function calculateConceptualSimilarity($text1, $text2) {
        $concepts1 = $this->extractAcademicConcepts($text1);
        $concepts2 = $this->extractAcademicConcepts($text2);
        
        if (empty($concepts1) || empty($concepts2)) {
            return 10; // Low baseline for no concepts
        }
        
        $matchedConcepts = array_intersect($concepts1, $concepts2);
        $totalConcepts = array_unique(array_merge($concepts1, $concepts2));
        
        $conceptScore = (count($matchedConcepts) / count($totalConcepts)) * 100;
        
        // Bonus for concept depth and academic terminology
        $depthBonus = min(20, count($concepts1) * 2);
        
        return min(100, $conceptScore + $depthBonus);
    }
    
    /**
     * Extract academic concepts from text
     */
    private function extractAcademicConcepts($text) {
        $concepts = [];
        $text = strtolower($text);
        
        // Academic indicators
        $academicTerms = [
            'theory', 'concept', 'framework', 'methodology', 'analysis', 'research',
            'study', 'approach', 'principle', 'model', 'strategy', 'implementation',
            'evaluation', 'assessment', 'criteria', 'conclusion', 'recommendation',
            'evidence', 'data', 'results', 'findings', 'discussion', 'literature',
            'hypothesis', 'objective', 'goal', 'outcome', 'impact', 'significance'
        ];
        
        foreach ($academicTerms as $term) {
            if (strpos($text, $term) !== false) {
                $concepts[] = $term;
            }
        }
        
        // Extract quoted concepts and technical terms
        preg_match_all('/"([^"]+)"/', $text, $quoted);
        foreach ($quoted[1] as $quote) {
            if (strlen($quote) > 3 && strlen($quote) < 50) {
                $concepts[] = strtolower(trim($quote));
            }
        }
        
        // Extract capitalized terms (likely proper nouns/concepts)
        preg_match_all('/\b[A-Z][a-z]+(?:\s+[A-Z][a-z]+)*\b/', $text, $proper);
        foreach ($proper[0] as $term) {
            if (strlen($term) > 3) {
                $concepts[] = strtolower($term);
            }
        }
        
        return array_unique($concepts);
    }
    
    /**
     * Calculate contextual relevance to assignment topic
     */
    private function calculateContextualRelevance($studentText, $memorandumText) {
        $studentSentences = preg_split('/[.!?]+/', $studentText);
        $memoSentences = preg_split('/[.!?]+/', $memorandumText);
        
        $relevantSentences = 0;
        $totalSentences = count($studentSentences);
        
        foreach ($studentSentences as $sentence) {
            $sentence = trim($sentence);
            if (strlen($sentence) < 10) continue;
            
            $sentenceWords = $this->processText($sentence);
            $bestMatch = 0;
            
            foreach ($memoSentences as $memoSentence) {
                $memoWords = $this->processText($memoSentence);
                if (empty($memoWords)) continue;
                
                $overlap = array_intersect($sentenceWords, $memoWords);
                $relevance = count($overlap) / max(count($sentenceWords), count($memoWords));
                $bestMatch = max($bestMatch, $relevance);
            }
            
            if ($bestMatch > 0.2) { // At least 20% word overlap
                $relevantSentences++;
            }
        }
        
        if ($totalSentences == 0) return 0;
        
        $contextScore = ($relevantSentences / $totalSentences) * 100;
        
        // Penalty for very short responses
        if (str_word_count($studentText) < 50) {
            $contextScore *= 0.7; // 30% penalty
        }
        
        return round($contextScore, 2);
    }
    
    /**
     * Calculate keyword coverage
     */
    public function calculateKeywordCoverage($studentWords, $memoWords) {
        if (empty($studentWords) || empty($memoWords)) {
            return 0;
        }
        
        $matches = array_intersect($studentWords, $memoWords);
        return (count($matches) / count($memoWords)) * 100;
    }
    
    /**
     * Comprehensive quality assessment using multiple academic criteria
     */
    public function assessQuality($text) {
        $metrics = [];
        
        // 1. Academic Writing Quality (25%)
        $metrics['academic'] = $this->assessAcademicWriting($text);
        
        // 2. Content Depth & Analysis (30%)
        $metrics['depth'] = $this->assessContentDepth($text);
        
        // 3. Structure & Organization (20%)
        $metrics['structure'] = $this->assessStructure($text);
        
        // 4. Language & Clarity (15%)
        $metrics['language'] = $this->assessLanguageQuality($text);
        
        // 5. Evidence & Support (10%)
        $metrics['evidence'] = $this->assessEvidence($text);
        
        // Weighted final score
        $finalScore = (
            $metrics['academic'] * 0.25 +
            $metrics['depth'] * 0.30 +
            $metrics['structure'] * 0.20 +
            $metrics['language'] * 0.15 +
            $metrics['evidence'] * 0.10
        );
        
        if ($this->debug) {
            error_log("QUALITY DEBUG: Academic={$metrics['academic']}, Depth={$metrics['depth']}, Structure={$metrics['structure']}, Language={$metrics['language']}, Evidence={$metrics['evidence']}, Final={$finalScore}");
        }
        
        return round($finalScore, 2);
    }
    
    /**
     * Assess academic writing quality
     */
    private function assessAcademicWriting($text) {
        $score = 0;
        $wordCount = str_word_count($text);
        
        // Academic vocabulary usage
        $academicWords = [
            'analysis', 'evaluate', 'assess', 'examine', 'investigate', 'demonstrate',
            'illustrate', 'compare', 'contrast', 'discuss', 'explain', 'describe',
            'conclude', 'therefore', 'however', 'furthermore', 'moreover', 'consequently',
            'nevertheless', 'significant', 'substantial', 'comprehensive', 'extensive',
            'critical', 'essential', 'fundamental', 'primary', 'secondary', 'subsequent'
        ];
        
        $academicCount = 0;
        foreach ($academicWords as $word) {
            $academicCount += substr_count(strtolower($text), $word);
        }
        
        $academicRatio = $wordCount > 0 ? ($academicCount / $wordCount * 100) : 0;
        $score += min(40, $academicRatio * 20); // Up to 40 points
        
        // Sentence complexity and variation
        $sentences = preg_split('/[.!?]+/', $text);
        $avgLength = 0;
        $variation = 0;
        $lengths = [];
        
        foreach ($sentences as $sentence) {
            $sentence = trim($sentence);
            if (strlen($sentence) > 5) {
                $length = str_word_count($sentence);
                $lengths[] = $length;
                $avgLength += $length;
            }
        }
        
        if (count($lengths) > 0) {
            $avgLength /= count($lengths);
            $variance = 0;
            foreach ($lengths as $length) {
                $variance += pow($length - $avgLength, 2);
            }
            $variation = count($lengths) > 1 ? sqrt($variance / count($lengths)) : 0;
        }
        
        // Good academic writing has varied sentence lengths
        if ($avgLength >= 12 && $avgLength <= 25 && $variation >= 3) {
            $score += 30; // Up to 30 points
        } else {
            $score += max(10, 30 - abs($avgLength - 18) * 2);
        }
        
        // Professional tone indicators
        $professionalIndicators = [
            'research shows', 'studies indicate', 'evidence suggests', 'according to',
            'it can be argued', 'it is evident', 'this demonstrates', 'this illustrates',
            'in conclusion', 'to summarize', 'in summary', 'based on the analysis'
        ];
        
        $professionalCount = 0;
        foreach ($professionalIndicators as $indicator) {
            if (stripos($text, $indicator) !== false) {
                $professionalCount++;
            }
        }
        
        $score += min(30, $professionalCount * 5); // Up to 30 points
        
        return min(100, $score);
    }
    
    /**
     * Assess content depth and analytical thinking
     */
    private function assessContentDepth($text) {
        $score = 0;
        
        // Analytical language indicators
        $analyticalTerms = [
            'because', 'since', 'due to', 'as a result', 'consequently', 'leads to',
            'causes', 'results in', 'implies', 'suggests', 'indicates', 'demonstrates',
            'on the other hand', 'alternatively', 'conversely', 'in contrast',
            'similarly', 'likewise', 'compared to', 'whereas', 'although', 'despite'
        ];
        
        $analyticalCount = 0;
        foreach ($analyticalTerms as $term) {
            if (stripos($text, $term) !== false) {
                $analyticalCount++;
            }
        }
        
        $score += min(40, $analyticalCount * 4); // Up to 40 points for analytical thinking
        
        // Examples and elaboration
        $exampleIndicators = [
            'for example', 'for instance', 'such as', 'including', 'specifically',
            'particularly', 'namely', 'that is', 'in other words', 'to illustrate'
        ];
        
        $exampleCount = 0;
        foreach ($exampleIndicators as $indicator) {
            if (stripos($text, $indicator) !== false) {
                $exampleCount++;
            }
        }
        
        $score += min(30, $exampleCount * 6); // Up to 30 points for examples
        
        // Depth indicators (multiple perspectives, comprehensive coverage)
        $depthIndicators = [
            'furthermore', 'additionally', 'moreover', 'in addition', 'also',
            'another aspect', 'another consideration', 'it should be noted',
            'it is important to consider', 'from another perspective'
        ];
        
        $depthCount = 0;
        foreach ($depthIndicators as $indicator) {
            if (stripos($text, $indicator) !== false) {
                $depthCount++;
            }
        }
        
        $score += min(30, $depthCount * 5); // Up to 30 points for depth
        
        return min(100, $score);
    }
    
    /**
     * Assess text structure and organization
     */
    private function assessStructure($text) {
        $score = 0;
        
        // Paragraph structure
        $paragraphs = preg_split('/\n\s*\n/', trim($text));
        $paragraphCount = count($paragraphs);
        
        if ($paragraphCount >= 2 && $paragraphCount <= 6) {
            $score += 30; // Well-structured paragraphs
        } else if ($paragraphCount == 1) {
            $score += 15; // Single paragraph penalty
        } else {
            $score += 20; // Too many or too few paragraphs
        }
        
        // Logical flow indicators
        $transitionWords = [
            'first', 'second', 'third', 'finally', 'in conclusion',
            'next', 'then', 'subsequently', 'following this',
            'initially', 'furthermore', 'moreover', 'additionally'
        ];
        
        $transitionCount = 0;
        foreach ($transitionWords as $word) {
            if (stripos($text, $word) !== false) {
                $transitionCount++;
            }
        }
        
        $score += min(40, $transitionCount * 8); // Up to 40 points for transitions
        
        // Introduction and conclusion indicators
        $introIndicators = ['introduction', 'this essay', 'this paper', 'this analysis', 'this assignment'];
        $conclusionIndicators = ['conclusion', 'in summary', 'to conclude', 'in closing', 'finally'];
        
        $hasIntro = false;
        $hasConclusion = false;
        
        foreach ($introIndicators as $indicator) {
            if (stripos($text, $indicator) !== false) {
                $hasIntro = true;
                break;
            }
        }
        
        foreach ($conclusionIndicators as $indicator) {
            if (stripos($text, $indicator) !== false) {
                $hasConclusion = true;
                break;
            }
        }
        
        if ($hasIntro) $score += 15;
        if ($hasConclusion) $score += 15;
        
        return min(100, $score);
    }
    
    /**
     * Assess language quality and clarity
     */
    private function assessLanguageQuality($text) {
        $score = 60; // Base score
        
        // Check for common grammar patterns
        $wordCount = str_word_count($text);
        $sentences = preg_split('/[.!?]+/', $text);
        $avgWordsPerSentence = $wordCount / max(1, count($sentences));
        
        // Optimal sentence length
        if ($avgWordsPerSentence >= 10 && $avgWordsPerSentence <= 22) {
            $score += 25;
        } else {
            $score += max(5, 25 - abs($avgWordsPerSentence - 16) * 2);
        }
        
        // Vocabulary diversity
        $words = $this->processText($text);
        $uniqueWords = array_unique($words);
        $diversity = count($words) > 0 ? (count($uniqueWords) / count($words)) : 0;
        
        if ($diversity >= 0.6) {
            $score += 15; // Good vocabulary diversity
        } else {
            $score += max(5, $diversity * 25);
        }
        
        return min(100, $score);
    }
    
    /**
     * Assess evidence and support in the text
     */
    private function assessEvidence($text) {
        $score = 20; // Base score
        
        // Evidence indicators
        $evidenceTerms = [
            'research', 'study', 'data', 'statistics', 'evidence', 'findings',
            'results', 'survey', 'report', 'investigation', 'analysis',
            'according to', 'studies show', 'research indicates', 'data suggests'
        ];
        
        $evidenceCount = 0;
        foreach ($evidenceTerms as $term) {
            if (stripos($text, $term) !== false) {
                $evidenceCount++;
            }
        }
        
        $score += min(50, $evidenceCount * 8);
        
        // Reference to sources or authorities
        $authorityTerms = [
            'expert', 'scholar', 'researcher', 'professor', 'author',
            'specialist', 'authority', 'academic', 'scientist'
        ];
        
        $authorityCount = 0;
        foreach ($authorityTerms as $term) {
            if (stripos($text, $term) !== false) {
                $authorityCount++;
            }
        }
        
        $score += min(30, $authorityCount * 10);
        
        return min(100, $score);
    }
    
    /**
     * Generate final grade
     */
    public function generateGrade($similarity, $keywordCoverage, $quality) {
        $weights = [
            'similarity' => 0.4,
            'keyword_coverage' => 0.4,
            'quality' => 0.2
        ];
        
        $totalScore = ($similarity * $weights['similarity']) + 
                     ($keywordCoverage * $weights['keyword_coverage']) + 
                     ($quality * $weights['quality']);
        
        return max(0, min(100, round($totalScore, 2)));
    }
    
    /**
     * Generate feedback
     */
    public function generateFeedback($similarity, $keywordCoverage, $quality) {
        $feedback = [];
        
        if ($similarity < 30) {
            $feedback[] = "Your answer doesn't closely match the expected content. Consider reviewing the key concepts.";
        } elseif ($similarity < 60) {
            $feedback[] = "Your answer has some relevant content, but could be more comprehensive.";
        } else {
            $feedback[] = "Good job! Your answer demonstrates understanding of the key concepts.";
        }
        
        if ($keywordCoverage < 40) {
            $feedback[] = "Try to include more key terms from the course material.";
        } elseif ($keywordCoverage > 70) {
            $feedback[] = "Excellent use of key terminology!";
        }
        
        if ($quality < 50) {
            $feedback[] = "Consider expanding your answer with more detail and better structure.";
        }
        
        return implode(' ', $feedback);
    }
    
    /**
     * Calculate confidence
     */
    public function calculateConfidence($metrics) {
        $similarity = $metrics['similarity'];
        $keywordCoverage = $metrics['keyword_coverage'];
        $quality = $metrics['quality'];
        
        $variance = [
            abs($similarity - $keywordCoverage),
            abs($similarity - $quality),
            abs($keywordCoverage - $quality)
        ];
        
        $avgVariance = array_sum($variance) / count($variance);
        $consistency = max(0, 100 - $avgVariance);
        
        $avgScore = ($similarity + $keywordCoverage + $quality) / 3;
        $extremeness = abs($avgScore - 50);
        $moderateness = max(0, 50 - $extremeness) * 2;
        
        $confidence = ($consistency + $moderateness) / 2;
        return round(min(100, max(0, $confidence)), 2);
    }
    
    /**
     * Get stop words
     */
    private function getStopWords() {
        return [
            'the', 'a', 'an', 'and', 'or', 'but', 'in', 'on', 'at', 'to', 'for', 'of', 'with', 'by',
            'is', 'are', 'was', 'were', 'be', 'been', 'being', 'have', 'has', 'had', 'do', 'does', 'did',
            'will', 'would', 'should', 'could', 'can', 'may', 'might', 'must', 'shall',
            'this', 'that', 'these', 'those', 'i', 'you', 'he', 'she', 'it', 'we', 'they',
            'me', 'him', 'her', 'us', 'them', 'my', 'your', 'his', 'her', 'its', 'our', 'their'
        ];
    }
    
    /**
     * Basic word stemming
     */
    private function stemWord($word) {
        // Remove common suffixes
        $suffixes = ['ing', 'ed', 'er', 'est', 'ly', 'tion', 'sion', 'ness', 'ment', 'able', 'ible'];
        
        foreach ($suffixes as $suffix) {
            if (strlen($word) > strlen($suffix) + 3 && substr($word, -strlen($suffix)) === $suffix) {
                return substr($word, 0, -strlen($suffix));
            }
        }
        
        return $word;
    }
    
    /**
     * Extract keywords from text (for AJAX compatibility)
     */
    public function extractKeywords($text) {
        return $this->processText($text);
    }
    
    /**
     * Check keyword matches (for AJAX compatibility)
     */
    public function checkKeywordMatches($studentText, $keywords) {
        $studentWords = $this->processText($studentText);
        $matches = array_intersect($studentWords, $keywords);
        return count($matches);
    }
}
?>
