<?php
// Database connection handled by parent file

/**
 * Automated Certificate Generation System
 * Generates and emails certificates automatically when students complete courses
 */

class AutoCertificateGenerator {
    private $conn;
    
    public function __construct($connection) {
        $this->conn = $connection;
        $this->createCertificatesTable();
    }
    
    /**
     * Create certificates table if it doesn't exist
     */
    private function createCertificatesTable() {
        $sql = "
            CREATE TABLE IF NOT EXISTS certificates (
                certificate_id INT AUTO_INCREMENT PRIMARY KEY,
                course_id INT NOT NULL,
                student_id INT NOT NULL,
                certificate_number VARCHAR(50) UNIQUE NOT NULL,
                student_name VARCHAR(255) NOT NULL,
                student_email VARCHAR(255) NOT NULL,
                course_title VARCHAR(255) NOT NULL,
                issue_date DATE NOT NULL,
                completion_date DATE NOT NULL,
                status ENUM('active', 'revoked') DEFAULT 'active',
                email_sent BOOLEAN DEFAULT FALSE,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                UNIQUE KEY unique_student_course (course_id, student_id),
                INDEX idx_student (student_id),
                INDEX idx_course (course_id),
                INDEX idx_email_sent (email_sent)
            )
        ";
        $this->conn->query($sql);
    }
    
    /**
     * Generate certificate automatically when student completes course
     */
    public function generateCertificateOnCompletion($student_id, $course_id, $completion_date = null) {
        try {
            // Check if certificate already exists
            $checkStmt = $this->conn->prepare("SELECT certificate_id FROM certificates WHERE course_id = ? AND student_id = ?");
            $checkStmt->bind_param("ii", $course_id, $student_id);
            $checkStmt->execute();
            $existing = $checkStmt->get_result();
            
            if ($existing->num_rows > 0) {
                return ['success' => false, 'message' => 'Certificate already exists'];
            }
            
            // Get student details
            $studentStmt = $this->conn->prepare("SELECT first_name, last_name, email FROM users WHERE user_id = ? AND user_role = 'student'");
            $studentStmt->bind_param("i", $student_id);
            $studentStmt->execute();
            $studentResult = $studentStmt->get_result();
            $student = $studentResult->fetch_assoc();
            
            if (!$student) {
                return ['success' => false, 'message' => 'Student not found'];
            }
            
            // Get course details
            $courseStmt = $this->conn->prepare("SELECT course_title FROM courses WHERE course_id = ?");
            $courseStmt->bind_param("i", $course_id);
            $courseStmt->execute();
            $courseResult = $courseStmt->get_result();
            $course = $courseResult->fetch_assoc();
            
            if (!$course) {
                return ['success' => false, 'message' => 'Course not found'];
            }
            
            // Generate unique certificate number
            $certificate_number = 'PANADITE-' . date('Y') . '-' . sprintf('%04d', $course_id) . '-' . sprintf('%03d', $student_id);
            $student_name = $student['first_name'] . ' ' . $student['last_name'];
            $completion_date = $completion_date ?: date('Y-m-d');
            
            // Insert certificate record
            $insertStmt = $this->conn->prepare("
                INSERT INTO certificates (course_id, student_id, certificate_number, student_name, student_email, course_title, issue_date, completion_date, status, email_sent) 
                VALUES (?, ?, ?, ?, ?, ?, CURDATE(), ?, 'active', FALSE)
            ");
            $insertStmt->bind_param("iisssss", $course_id, $student_id, $certificate_number, $student_name, $student['email'], $course['course_title'], $completion_date);
            
            if ($insertStmt->execute()) {
                $certificate_id = $this->conn->insert_id;
                
                // Generate certificate HTML
                $certificateHtml = $this->generateCertificateHtml($certificate_number, $student_name, $course['course_title'], date('F j, Y'));
                
                // Send email to student
                $emailSent = $this->sendCertificateEmail($student['email'], $student_name, $course['course_title'], $certificateHtml, $certificate_number);
                
                // Update email sent status
                if ($emailSent) {
                    $updateStmt = $this->conn->prepare("UPDATE certificates SET email_sent = TRUE WHERE certificate_id = ?");
                    $updateStmt->bind_param("i", $certificate_id);
                    $updateStmt->execute();
                }
                
                return [
                    'success' => true,
                    'certificate_id' => $certificate_id,
                    'certificate_number' => $certificate_number,
                    'email_sent' => $emailSent
                ];
            } else {
                return ['success' => false, 'message' => 'Failed to generate certificate'];
            }
            
        } catch (Exception $e) {
            error_log("Certificate generation error: " . $e->getMessage());
            return ['success' => false, 'message' => 'Certificate generation error: ' . $e->getMessage()];
        }
    }
    
    /**
     * Generate universal certificate HTML design
     */
    private function generateCertificateHtml($certificate_number, $student_name, $course_title, $issue_date) {
        return "
        <!DOCTYPE html>
        <html>
        <head>
            <meta charset='UTF-8'>
            <title>Certificate of Completion - {$student_name}</title>
            <style>
                @import url('https://fonts.googleapis.com/css2?family=Playfair+Display:wght@400;700&family=Inter:wght@300;400;500;600&display=swap');
                
                .certificate {
                    width: 800px;
                    height: 600px;
                    margin: 20px auto;
                    background: linear-gradient(135deg, #f8faff 0%, #ffffff 100%);
                    border: 8px solid #4169E1;
                    border-radius: 20px;
                    position: relative;
                    font-family: 'Inter', sans-serif;
                    box-shadow: 0 20px 60px rgba(0,0,0,0.1);
                    padding: 40px;
                    box-sizing: border-box;
                }
                
                .certificate::before {
                    content: '';
                    position: absolute;
                    top: 20px;
                    left: 20px;
                    right: 20px;
                    bottom: 20px;
                    border: 2px solid #20B2AA;
                    border-radius: 12px;
                }
                
                .certificate-header {
                    text-align: center;
                    margin-bottom: 30px;
                }
                
                .logo {
                    width: 80px;
                    height: 80px;
                    background: linear-gradient(135deg, #4169E1, #20B2AA);
                    border-radius: 50%;
                    display: flex;
                    align-items: center;
                    justify-content: center;
                    font-size: 2rem;
                    font-weight: bold;
                    color: white;
                    margin: 0 auto 20px;
                    border: 4px solid #FFD700;
                }
                
                .certificate-title {
                    font-family: 'Playfair Display', serif;
                    font-size: 2.8rem;
                    font-weight: 700;
                    color: #4169E1;
                    margin: 0 0 10px 0;
                    text-transform: uppercase;
                    letter-spacing: 2px;
                }
                
                .subtitle {
                    font-size: 1.2rem;
                    color: #20B2AA;
                    font-weight: 500;
                    margin-bottom: 30px;
                }
                
                .certificate-body {
                    text-align: center;
                    margin: 40px 0;
                }
                
                .certificate-text {
                    font-size: 1.1rem;
                    color: #333;
                    line-height: 1.8;
                    margin: 15px 0;
                    font-weight: 400;
                }
                
                .student-name {
                    font-family: 'Playfair Display', serif;
                    font-size: 2.5rem;
                    font-weight: 700;
                    color: #4169E1;
                    margin: 20px 0;
                    text-decoration: underline;
                    text-decoration-color: #FFD700;
                    text-underline-offset: 8px;
                    text-decoration-thickness: 3px;
                }
                
                .course-title {
                    font-family: 'Playfair Display', serif;
                    font-size: 1.8rem;
                    color: #20B2AA;
                    font-weight: 600;
                    margin: 20px 0;
                    font-style: italic;
                }
                
                .certificate-footer {
                    display: flex;
                    justify-content: space-between;
                    align-items: end;
                    margin-top: 50px;
                    padding-top: 20px;
                    border-top: 2px solid #FFD700;
                }
                
                .signature-section {
                    text-align: center;
                    flex: 1;
                }
                
                .signature-line {
                    width: 150px;
                    height: 2px;
                    background: #4169E1;
                    margin: 0 auto 10px;
                }
                
                .signature-title {
                    font-weight: 600;
                    color: #333;
                    font-size: 0.9rem;
                }
                
                .certificate-details {
                    text-align: center;
                    flex: 1;
                }
                
                .cert-number {
                    font-weight: 600;
                    color: #4169E1;
                    font-size: 1rem;
                    margin-bottom: 8px;
                }
                
                .issue-date {
                    color: #666;
                    font-size: 0.9rem;
                }
                
                .watermark {
                    position: absolute;
                    top: 50%;
                    left: 50%;
                    transform: translate(-50%, -50%) rotate(-45deg);
                    font-size: 6rem;
                    color: rgba(65, 105, 225, 0.05);
                    font-weight: bold;
                    z-index: 0;
                    font-family: 'Playfair Display', serif;
                }
                
                .certificate-content {
                    position: relative;
                    z-index: 1;
                }
            </style>
        </head>
        <body>
            <div class='certificate'>
                <div class='watermark'>PANADITE</div>
                <div class='certificate-content'>
                    <div class='certificate-header'>
                        <div class='logo'>PA</div>
                        <h1 class='certificate-title'>Certificate</h1>
                        <p class='subtitle'>of Course Completion</p>
                    </div>
                    
                    <div class='certificate-body'>
                        <p class='certificate-text'>This is to certify that</p>
                        <div class='student-name'>{$student_name}</div>
                        <p class='certificate-text'>has successfully completed the course</p>
                        <div class='course-title'>{$course_title}</div>
                        <p class='certificate-text'>at Panadite Academy and has demonstrated proficiency in all required competencies.</p>
                    </div>
                    
                    <div class='certificate-footer'>
                        <div class='signature-section'>
                            <div class='signature-line'></div>
                            <p class='signature-title'>Director<br>Panadite Academy</p>
                        </div>
                        
                        <div class='certificate-details'>
                            <p class='cert-number'>Certificate No: {$certificate_number}</p>
                            <p class='issue-date'>Issued on {$issue_date}</p>
                        </div>
                        
                        <div class='signature-section'>
                            <div class='signature-line'></div>
                            <p class='signature-title'>Academic Registrar<br>Panadite Academy</p>
                        </div>
                    </div>
                </div>
            </div>
        </body>
        </html>";
    }
    
    /**
     * Send certificate email to student
     */
    private function sendCertificateEmail($email, $student_name, $course_title, $certificateHtml, $certificate_number) {
        try {
            $subject = "🎓 Congratulations! Your Certificate of Completion - {$course_title}";
            
            $message = "
            <div style='font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto; background: #f8faff; padding: 20px; border-radius: 10px;'>
                <div style='text-align: center; margin-bottom: 30px;'>
                    <div style='width: 60px; height: 60px; background: linear-gradient(135deg, #4169E1, #20B2AA); border-radius: 50%; display: inline-flex; align-items: center; justify-content: center; color: white; font-size: 1.5rem; font-weight: bold; margin-bottom: 15px;'>PA</div>
                    <h1 style='color: #4169E1; margin: 0;'>Panadite Academy</h1>
                </div>
                
                <div style='background: white; padding: 30px; border-radius: 10px; box-shadow: 0 5px 15px rgba(0,0,0,0.1);'>
                    <h2 style='color: #20B2AA; margin-top: 0;'>🎉 Congratulations, {$student_name}!</h2>
                    
                    <p style='color: #333; font-size: 16px; line-height: 1.6;'>
                        We are thrilled to inform you that you have successfully completed the course <strong style='color: #4169E1;'>{$course_title}</strong>!
                    </p>
                    
                    <p style='color: #333; font-size: 16px; line-height: 1.6;'>
                        Your dedication and hard work have paid off. As a testament to your achievement, we have generated your official Certificate of Completion.
                    </p>
                    
                    <div style='background: #f0f8ff; padding: 20px; border-radius: 8px; margin: 20px 0; border-left: 4px solid #4169E1;'>
                        <h3 style='color: #4169E1; margin-top: 0;'>📜 Certificate Details</h3>
                        <p style='margin: 5px 0; color: #333;'><strong>Student:</strong> {$student_name}</p>
                        <p style='margin: 5px 0; color: #333;'><strong>Course:</strong> {$course_title}</p>
                        <p style='margin: 5px 0; color: #333;'><strong>Certificate Number:</strong> {$certificate_number}</p>
                        <p style='margin: 5px 0; color: #333;'><strong>Issue Date:</strong> " . date('F j, Y') . "</p>
                    </div>
                    
                    <p style='color: #333; font-size: 16px; line-height: 1.6;'>
                        Your certificate is attached to this email and can be printed for your records. This certificate serves as official recognition of your course completion and can be used for professional development purposes.
                    </p>
                    
                    <div style='text-align: center; margin: 30px 0;'>
                        <div style='display: inline-block; background: linear-gradient(135deg, #4169E1, #20B2AA); color: white; padding: 12px 24px; border-radius: 8px; text-decoration: none; font-weight: bold;'>
                            🏆 Certificate Attached
                        </div>
                    </div>
                    
                    <p style='color: #666; font-size: 14px; text-align: center; margin-top: 30px; padding-top: 20px; border-top: 1px solid #eee;'>
                        Thank you for choosing Panadite Academy for your learning journey.<br>
                        Keep learning, keep growing! 🌟
                    </p>
                </div>
                
                <div style='text-align: center; margin-top: 20px; color: #666; font-size: 12px;'>
                    © " . date('Y') . " Panadite Academy. All rights reserved.
                </div>
            </div>";
            
            $headers = [
                'MIME-Version: 1.0',
                'Content-type: text/html; charset=UTF-8',
                'From: Panadite Academy <certificates@panaditeacademy.com>',
                'Reply-To: support@panaditeacademy.com',
                'X-Mailer: PHP/' . phpversion()
            ];
            
            // For now, use basic mail() function
            // In production, consider using PHPMailer for better email delivery
            $success = mail($email, $subject, $message, implode("\r\n", $headers));
            
            // Log certificate email
            error_log("Certificate email sent to {$email} for course {$course_title}: " . ($success ? 'SUCCESS' : 'FAILED'));
            
            return $success;
            
        } catch (Exception $e) {
            error_log("Certificate email error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Get all certificates for admin viewing
     */
    public function getAllCertificates($limit = 50) {
        try {
            // Ensure table exists first
            $this->createCertificatesTable();
            
            $query = "
                SELECT c.*, 
                       CASE WHEN c.email_sent = 1 THEN 'Sent' ELSE 'Pending' END as email_status
                FROM certificates c 
                ORDER BY c.created_at DESC 
                LIMIT ?
            ";
            
            $stmt = $this->conn->prepare($query);
            if (!$stmt) {
                error_log("Database prepare error: " . $this->conn->error);
                return [];
            }
            
            $stmt->bind_param("i", $limit);
            $stmt->execute();
            $result = $stmt->get_result();
            
            $certificates = [];
            while ($row = $result->fetch_assoc()) {
                $certificates[] = $row;
            }
            
            return $certificates;
            
        } catch (Exception $e) {
            error_log("Get certificates error: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Get certificate statistics
     */
    public function getStatistics() {
        $stats = [
            'total_certificates' => 0,
            'emails_sent' => 0,
            'emails_pending' => 0,
            'this_month' => 0
        ];
        
        try {
            // Ensure table exists first
            $this->createCertificatesTable();
            
            // Total certificates
            $result = $this->conn->query("SELECT COUNT(*) as count FROM certificates");
            if ($result) {
                $stats['total_certificates'] = $result->fetch_assoc()['count'];
            }
            
            // Emails sent
            $result = $this->conn->query("SELECT COUNT(*) as count FROM certificates WHERE email_sent = 1");
            if ($result) {
                $stats['emails_sent'] = $result->fetch_assoc()['count'];
            }
            
            // Emails pending
            $result = $this->conn->query("SELECT COUNT(*) as count FROM certificates WHERE email_sent = 0");
            if ($result) {
                $stats['emails_pending'] = $result->fetch_assoc()['count'];
            }
            
            // This month
            $result = $this->conn->query("SELECT COUNT(*) as count FROM certificates WHERE MONTH(created_at) = MONTH(CURRENT_DATE()) AND YEAR(created_at) = YEAR(CURRENT_DATE())");
            if ($result) {
                $stats['this_month'] = $result->fetch_assoc()['count'];
            }
            
        } catch (Exception $e) {
            error_log("Get statistics error: " . $e->getMessage());
        }
        
        return $stats;
    }
}

// Function to be called when student completes a course
function generateCertificateOnCourseCompletion($student_id, $course_id, $completion_date = null) {
    global $conn;
    $generator = new AutoCertificateGenerator($conn);
    return $generator->generateCertificateOnCompletion($student_id, $course_id, $completion_date);
}
?>
