<?php
/**
 * BackupScheduler Class
 * Handles scheduling and execution of automatic database backups
 */
class BackupScheduler {
    private $conn;
    private $settings;
    
    /**
     * Constructor
     */
    public function __construct() {
        global $conn;
        $this->conn = $conn;
        $this->loadSettings();
    }
    
    /**
     * Load backup settings from the database
     */
    private function loadSettings() {
        // Check if the backup_settings table exists, create it if not
        $this->createSettingsTableIfNotExists();
        
        // Load settings
        $query = "SELECT * FROM backup_settings WHERE id = 1";
        $result = $this->conn->query($query);
        
        if ($result && $result->num_rows > 0) {
            $this->settings = $result->fetch_assoc();
            
            // Unserialize tables array
            if ($this->settings['tables']) {
                $this->settings['tables'] = unserialize($this->settings['tables']);
            } else {
                $this->settings['tables'] = [];
            }
        } else {
            // Create default settings
            $this->settings = [
                'frequency' => 'weekly',
                'format' => 'sql',
                'tables' => [],
                'retention' => 30,
                'last_backup' => null,
                'next_backup' => date('Y-m-d H:i:s', strtotime('+1 week')),
                'interval_hours' => 168, // 7 days * 24 hours (weekly default)
                'enabled' => 1
            ];
            
            $this->saveSettings();
        }
    }
    
    /**
     * Create backup_settings table if it doesn't exist
     */
    private function createSettingsTableIfNotExists() {
        $query = "CREATE TABLE IF NOT EXISTS backup_settings (
            id INT PRIMARY KEY,
            frequency VARCHAR(20) NOT NULL,
            format VARCHAR(10) NOT NULL,
            tables TEXT,
            retention INT NOT NULL DEFAULT 30,
            last_backup DATETIME,
            next_backup DATETIME,
            interval_hours INT NOT NULL DEFAULT 24,
            enabled TINYINT(1) NOT NULL DEFAULT 1
        )";
        
        $this->conn->query($query);
        
        // Check if interval_hours column exists, if not add it
        $result = $this->conn->query("SHOW COLUMNS FROM backup_settings LIKE 'interval_hours'");
        if ($result->num_rows == 0) {
            $this->conn->query("ALTER TABLE backup_settings ADD COLUMN interval_hours INT NOT NULL DEFAULT 24");
        }
        
        // Check if there's a record, if not, create one
        $result = $this->conn->query("SELECT COUNT(*) as count FROM backup_settings");
        $row = $result->fetch_assoc();
        
        if ($row['count'] == 0) {
            // Set next backup to be one week from now
            $nextBackup = date('Y-m-d H:i:s', strtotime('+1 week'));
            
            $this->conn->query("INSERT INTO backup_settings (id, frequency, format, tables, retention, next_backup, enabled) 
                               VALUES (1, 'weekly', 'sql', '', 30, '$nextBackup', 1)");
        }
    }
    
    /**
     * Create backup_history table if it doesn't exist
     */
    public function createHistoryTableIfNotExists() {
        $query = "CREATE TABLE IF NOT EXISTS backup_history (
            id INT AUTO_INCREMENT PRIMARY KEY,
            filename VARCHAR(255) NOT NULL,
            format VARCHAR(10) NOT NULL,
            tables TEXT NOT NULL,
            size VARCHAR(20) NOT NULL,
            created_at DATETIME NOT NULL,
            type VARCHAR(10) NOT NULL,
            status VARCHAR(20) NOT NULL,
            file_path VARCHAR(255) NOT NULL
        )";
        
        return $this->conn->query($query);
    }
    
    /**
     * Save backup settings to database
     * @param array $newSettings (Optional) New settings to save
     */
    public function saveSettings($newSettings = null) {
        // If new settings provided, update current settings
        if ($newSettings) {
            foreach ($newSettings as $key => $value) {
                $this->settings[$key] = $value;
            }
        }
        
        // Calculate next backup time based on frequency
        if (!isset($this->settings['next_backup']) || $newSettings) {
            $this->calculateNextBackupTime();
        }
        
        // Serialize tables array
        $tables = $this->settings['tables'] ? serialize($this->settings['tables']) : '';
        
        // Update in database
        $query = "UPDATE backup_settings SET 
                  frequency = '{$this->settings['frequency']}',
                  format = '{$this->settings['format']}',
                  tables = '$tables',
                  retention = {$this->settings['retention']},
                  last_backup = " . ($this->settings['last_backup'] ? "'{$this->settings['last_backup']}'" : "NULL") . ",
                  next_backup = '{$this->settings['next_backup']}',
                  interval_hours = {$this->settings['interval_hours']},
                  enabled = {$this->settings['enabled']}
                  WHERE id = 1";
                  
        return $this->conn->query($query);
    }
    
    /**
     * Calculate the next backup time based on frequency and interval
     */
    private function calculateNextBackupTime() {
        $now = new DateTime();
        
        // First set a default interval_hours based on frequency if not already set
        if (!isset($this->settings['interval_hours']) || $this->settings['interval_hours'] <= 0) {
            switch ($this->settings['frequency']) {
                case 'daily':
                    $this->settings['interval_hours'] = 24;
                    break;
                case 'weekly':
                    $this->settings['interval_hours'] = 168; // 7 days * 24 hours
                    break;
                case 'biweekly':
                    $this->settings['interval_hours'] = 336; // 14 days * 24 hours
                    break;
                case 'monthly':
                    $this->settings['interval_hours'] = 720; // ~30 days * 24 hours
                    break;
                default:
                    $this->settings['interval_hours'] = 168; // Default to weekly
            }
        }
        
        // Now calculate next backup time using the interval hours
        $intervalHours = intval($this->settings['interval_hours']);
        $nextBackup = $now->modify("+{$intervalHours} hours");
        
        $this->settings['next_backup'] = $nextBackup->format('Y-m-d H:i:s');
    }
    
    /**
     * Check if it's time for a scheduled backup
     * @return bool True if backup should run
     */
    public function shouldRunBackup() {
        if (!$this->settings['enabled']) {
            return false;
        }
        
        $now = new DateTime();
        
        // Check if next_backup is set and is a valid date
        if (empty($this->settings['next_backup'])) {
            // If next_backup is not set, calculate it now and return false
            // (this will be the first run, so we'll set up the schedule for next time)
            $this->calculateNextBackupTime();
            $this->saveSettings();
            return false;
        }
        
        $nextBackup = new DateTime($this->settings['next_backup']);
        
        return $now >= $nextBackup;
    }
    
    /**
     * Run an automatic backup
     * @return bool True if backup was successful
     */
    public function runAutomaticBackup() {
        if (!$this->shouldRunBackup()) {
            return false;
        }
        
        try {
            // Create backup
            require_once __DIR__ . '/DatabaseBackup.php';
            $backup = new DatabaseBackup(
                $this->settings['tables'], 
                $this->settings['format']
            );
            
            $filePath = $backup->createBackup();
            
            // Update last backup time
            $now = new DateTime();
            $this->settings['last_backup'] = $now->format('Y-m-d H:i:s');
            
            // Calculate next backup time
            $this->calculateNextBackupTime();
            
            // Save settings
            $this->saveSettings();
            
            // Log backup in history
            $this->logBackupHistory($filePath, 'auto');
            
            // Clean up old backups
            $this->cleanupOldBackups();
            
            return true;
        } catch (Exception $e) {
            // Log error
            error_log("Automatic backup failed: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Log backup in history
     * @param string $filePath Path to backup file
     * @param string $type Type of backup (auto or manual)
     * @return bool Success status
     */
    public function logBackupHistory($filePath, $type = 'manual') {
        // Make sure history table exists
        $this->createHistoryTableIfNotExists();
        
        // Verify file exists
        if (!file_exists($filePath)) {
            error_log("Warning: Attempted to log non-existent backup file: $filePath");
            return false;
        }
        
        // Get file info
        $fileInfo = pathinfo($filePath);
        $filename = $fileInfo['basename'];
        $format = isset($fileInfo['extension']) ? $fileInfo['extension'] : '';
        $size = filesize($filePath);
        
        // Format size
        $formattedSize = DatabaseBackup::formatBytes($size);
        
        // Get tables - ensure we're using the tables that were actually backed up
        // For manual backups, the settings tables may not match what was selected in the form
        if ($type == 'manual' && isset($_POST['tables'])) {
            $tables = implode(',', $_POST['tables']);
        } else {
            $tables = $this->settings['tables'] ? implode(',', $this->settings['tables']) : 'all';
        }
        
        // Use absolute path to ensure file can be found later
        $absoluteFilePath = realpath($filePath);
        if (!$absoluteFilePath) {
            $absoluteFilePath = $filePath; // Fallback if realpath fails
        }
        
        // Log in database using prepared statement to prevent SQL injection
        $now = date('Y-m-d H:i:s');
        $stmt = $this->conn->prepare("INSERT INTO backup_history 
                 (filename, format, tables, size, created_at, type, status, file_path) 
                 VALUES 
                 (?, ?, ?, ?, ?, ?, 'complete', ?)");
                 
        if ($stmt) {
            $stmt->bind_param("sssssss", $filename, $format, $tables, $formattedSize, $now, $type, $absoluteFilePath);
            $result = $stmt->execute();
            $stmt->close();
            
            if ($result) {
                error_log("Successfully logged backup: $absoluteFilePath");
            } else {
                error_log("Failed to log backup in database: " . $this->conn->error);
            }
            
            return $result;
        }
        
        // Fallback to non-prepared statement if prepare fails
        error_log("Warning: Using non-prepared statement for backup logging. Error: " . $this->conn->error);
        $filename = $this->conn->real_escape_string($filename);
        $format = $this->conn->real_escape_string($format);
        $tables = $this->conn->real_escape_string($tables);
        $formattedSize = $this->conn->real_escape_string($formattedSize);
        $type = $this->conn->real_escape_string($type);
        $absoluteFilePath = $this->conn->real_escape_string($absoluteFilePath);
        
        $query = "INSERT INTO backup_history 
                 (filename, format, tables, size, created_at, type, status, file_path) 
                 VALUES 
                 ('$filename', '$format', '$tables', '$formattedSize', '$now', '$type', 'complete', '$absoluteFilePath')";
                 
        return $this->conn->query($query);
    }
    
    /**
     * Clean up old backups based on retention policy
     */
    public function cleanupOldBackups() {
        if ($this->settings['retention'] <= 0) {
            return; // No cleanup if retention is 0 or negative
        }
        
        // Calculate cutoff date
        $cutoffDate = date('Y-m-d H:i:s', strtotime("-{$this->settings['retention']} days"));
        
        // Get old auto backups
        $query = "SELECT id, file_path FROM backup_history 
                 WHERE type = 'auto' AND created_at < '$cutoffDate'";
        $result = $this->conn->query($query);
        
        if ($result && $result->num_rows > 0) {
            while ($row = $result->fetch_assoc()) {
                // Delete file if exists
                if (file_exists($row['file_path'])) {
                    unlink($row['file_path']);
                }
                
                // Delete from history
                $this->conn->query("DELETE FROM backup_history WHERE id = {$row['id']}");
            }
        }
    }
    
    /**
     * Get backup history
     * @param int $limit Number of records to return
     * @param int $offset Offset for pagination
     * @return array Backup history records
     */
    public function getBackupHistory($limit = 10, $offset = 0) {
        $this->createHistoryTableIfNotExists();
        
        $query = "SELECT * FROM backup_history ORDER BY created_at DESC LIMIT $offset, $limit";
        $result = $this->conn->query($query);
        
        $history = [];
        if ($result && $result->num_rows > 0) {
            while ($row = $result->fetch_assoc()) {
                $history[] = $row;
            }
        }
        
        return $history;
    }
    
    /**
     * Get current settings
     * @return array Current backup settings
     */
    public function getSettings() {
        return $this->settings;
    }
}
