<?php
/**
 * Individual Course Grade Calculator
 * 
 * Handles proper grade calculation for each course independently,
 * ensuring grades always add up to 100% regardless of which assignments/quizzes contribute.
 */

class CourseGradeCalculator {
    private $conn;
    
    public function __construct($db_connection) {
        $this->conn = $db_connection;
    }
    
    /**
     * Calculate final grade for a student in a specific course
     * Ensures proper weighting even with selective contributing assessments
     * 
     * @param int $student_id
     * @param int $course_id
     * @return array Complete grade breakdown
     */
    public function calculateFinalGrade($student_id, $course_id) {
        // Step 1: Get course weighting configuration
        $course_config = $this->getCourseConfiguration($course_id);
        
        // Step 2: Calculate assignment component average
        $assignment_data = $this->calculateAssignmentComponent($student_id, $course_id);
        
        // Step 3: Calculate quiz component average  
        $quiz_data = $this->calculateQuizComponent($student_id, $course_id);
        
        // Step 4: Apply course weighting with smart fallback
        $final_calculation = $this->applyCourseLevelWeighting(
            $assignment_data, 
            $quiz_data, 
            $course_config
        );
        
        return [
            'student_id' => $student_id,
            'course_id' => $course_id,
            'final_grade' => $final_calculation['final_grade'],
            'letter_grade' => $this->convertToLetterGrade($final_calculation['final_grade']),
            'assignment_component' => $assignment_data,
            'quiz_component' => $quiz_data,
            'weighting_applied' => $final_calculation['weighting'],
            'calculation_method' => $final_calculation['method'],
            'verification' => [
                'total_weight_used' => $final_calculation['total_weight'],
                'is_valid_100_percent' => abs($final_calculation['total_weight'] - 100) < 0.01
            ],
            'generated_at' => date('Y-m-d H:i:s')
        ];
    }
    
    /**
     * Get course configuration including assignment and quiz weights
     */
    private function getCourseConfiguration($course_id) {
        $query = "
            SELECT title, assignment_weight, quiz_weight 
            FROM courses 
            WHERE course_id = ?
        ";
        
        $stmt = $this->conn->prepare($query);
        if (!$stmt) {
            error_log("Database prepare failed in getCourseConfiguration: " . $this->conn->error);
            return [
                'title' => 'Unknown Course',
                'assignment_weight' => 70,
                'quiz_weight' => 30
            ];
        }
        
        $stmt->bind_param("i", $course_id);
        $stmt->execute();
        $result = $stmt->get_result()->fetch_assoc();
        
        return [
            'title' => $result['title'] ?? 'Unknown Course',
            'assignment_weight' => floatval($result['assignment_weight'] ?? 70),
            'quiz_weight' => floatval($result['quiz_weight'] ?? 30)
        ];
    }
    
    /**
     * Calculate assignment component - QUIZ-ONLY SYSTEM: Always returns empty/zero
     * Assignments are ignored for certificate generation in the quiz-only system
     */
    private function calculateAssignmentComponent($student_id, $course_id) {
        // QUIZ-ONLY CERTIFICATE SYSTEM: Assignments are completely ignored
        // This ensures only quizzes contribute to final grades and certificate eligibility
        
        return [
            'all_assignments' => [],
            'total_assignments' => 0,
            'contributing_assignments' => 0,
            'contributing_grades' => [],
            'average_grade' => 0,
            'has_grades' => false,
            'completion_status' => 'Quiz-only system - assignments ignored',
            'system_note' => 'This course uses quiz-only grading for certificate generation'
        ];
    }
    
    /**
     * Calculate quiz component - average of all contributing quizzes
     */
    private function calculateQuizComponent($student_id, $course_id) {
        $query = "
            SELECT 
                q.quiz_id,
                q.title,
                q.contributes_to_final_grade,
                qa.score,
                qa.attempt_id,
                qa.end_time
            FROM quizzes q
            LEFT JOIN quiz_attempts qa ON q.quiz_id = qa.quiz_id AND qa.student_id = ?
            WHERE q.course_id = ? AND q.is_published = 1
            ORDER BY q.created_at
        ";
        
        $stmt = $this->conn->prepare($query);
        if (!$stmt) {
            error_log("Database prepare failed in calculateQuizComponent: " . $this->conn->error);
            return [
                'all_quizzes' => [],
                'contributing_scores' => [],
                'average_score' => 0,
                'total_contributing' => 0
            ];
        }
        
        $stmt->bind_param("ii", $student_id, $course_id);
        $stmt->execute();
        $result = $stmt->get_result();
        
        $all_quizzes = [];
        $contributing_scores = [];
        $total_contributing = 0;
        
        while ($row = $result->fetch_assoc()) {
            $all_quizzes[] = [
                'quiz_id' => $row['quiz_id'],
                'title' => $row['title'],
                'contributes' => (bool)$row['contributes_to_final_grade'],
                'score' => $row['score'],
                'completed' => !empty($row['attempt_id']) && !empty($row['end_time']),
                'completed_at' => $row['end_time']
            ];
            
            // Only include scores from quizzes that contribute to final grade
            if ($row['contributes_to_final_grade'] && $row['score'] !== null) {
                $contributing_scores[] = floatval($row['score']);
                $total_contributing++;
            }
        }
        
        // Calculate average of contributing quizzes
        $average_score = 0;
        if (count($contributing_scores) > 0) {
            $average_score = array_sum($contributing_scores) / count($contributing_scores);
        }
        
        return [
            'all_quizzes' => $all_quizzes,
            'total_quizzes' => count($all_quizzes),
            'contributing_quizzes' => $total_contributing,
            'contributing_scores' => $contributing_scores,
            'average_score' => round($average_score, 2),
            'has_scores' => $total_contributing > 0,
            'completion_status' => $this->getCompletionStatus(count($all_quizzes), $total_contributing)
        ];
    }
    
    /**
     * Apply course-level weighting with intelligent fallback
     * Ensures total always equals 100%
     */
    private function applyCourseLevelWeighting($assignment_data, $quiz_data, $course_config) {
        $assignment_weight = $course_config['assignment_weight'];
        $quiz_weight = $course_config['quiz_weight'];
        
        $has_assignment_grades = $assignment_data['has_grades'];
        $has_quiz_grades = $quiz_data['has_scores'];
        
        if ($has_assignment_grades && $has_quiz_grades) {
            // CASE 1: Both assignments and quizzes have grades - use configured weights
            $assignment_contribution = ($assignment_data['average_grade'] * $assignment_weight) / 100;
            $quiz_contribution = ($quiz_data['average_score'] * $quiz_weight) / 100;
            $final_grade = $assignment_contribution + $quiz_contribution;
            
            return [
                'final_grade' => round($final_grade, 2),
                'weighting' => [
                    'assignment_weight' => $assignment_weight,
                    'quiz_weight' => $quiz_weight,
                    'assignment_contribution' => round($assignment_contribution, 2),
                    'quiz_contribution' => round($quiz_contribution, 2)
                ],
                'total_weight' => $assignment_weight + $quiz_weight,
                'method' => 'Standard weighted calculation using configured weights'
            ];
            
        } elseif ($has_assignment_grades && !$has_quiz_grades) {
            // CASE 2: Only assignments have grades - use 100% assignment weight
            $final_grade = $assignment_data['average_grade'];
            
            return [
                'final_grade' => round($final_grade, 2),
                'weighting' => [
                    'assignment_weight' => 100,
                    'quiz_weight' => 0,
                    'assignment_contribution' => round($final_grade, 2),
                    'quiz_contribution' => 0
                ],
                'total_weight' => 100,
                'method' => 'Assignment-only grading (no quiz grades available)'
            ];
            
        } elseif (!$has_assignment_grades && $has_quiz_grades) {
            // CASE 3: Only quizzes have grades - use 100% quiz weight
            $final_grade = $quiz_data['average_score'];
            
            return [
                'final_grade' => round($final_grade, 2),
                'weighting' => [
                    'assignment_weight' => 0,
                    'quiz_weight' => 100,
                    'assignment_contribution' => 0,
                    'quiz_contribution' => round($final_grade, 2)
                ],
                'total_weight' => 100,
                'method' => 'Quiz-only grading (no assignment grades available)'
            ];
            
        } else {
            // CASE 4: No grades available
            return [
                'final_grade' => 0,
                'weighting' => [
                    'assignment_weight' => 0,
                    'quiz_weight' => 0,
                    'assignment_contribution' => 0,
                    'quiz_contribution' => 0
                ],
                'total_weight' => 0,
                'method' => 'No grades available for calculation'
            ];
        }
    }
    
    /**
     * Get completion status description
     */
    private function getCompletionStatus($total_items, $completed_items) {
        if ($total_items == 0) {
            return 'No items available';
        } elseif ($completed_items == 0) {
            return 'Not started';
        } elseif ($completed_items == $total_items) {
            return 'All completed';
        } else {
            $percentage = round(($completed_items / $total_items) * 100);
            return "{$completed_items}/{$total_items} completed ({$percentage}%)";
        }
    }
    
    /**
     * Convert numerical grade to letter grade
     */
    private function convertToLetterGrade($numerical_grade) {
        if ($numerical_grade >= 90) return 'A';
        if ($numerical_grade >= 80) return 'B';
        if ($numerical_grade >= 70) return 'C';
        if ($numerical_grade >= 60) return 'D';
        return 'F';
    }
    
    /**
     * Generate a detailed grade report for display
     */
    public function generateGradeReport($student_id, $course_id) {
        $grade_data = $this->calculateFinalGrade($student_id, $course_id);
        $course_config = $this->getCourseConfiguration($course_id);
        
        return [
            'course_title' => $course_config['title'],
            'final_grade' => $grade_data['final_grade'],
            'letter_grade' => $grade_data['letter_grade'],
            'breakdown' => [
                'assignments' => [
                    'count' => $grade_data['assignment_component']['contributing_assignments'],
                    'average' => $grade_data['assignment_component']['average_grade'],
                    'weight_applied' => $grade_data['weighting_applied']['assignment_weight'],
                    'contribution' => $grade_data['weighting_applied']['assignment_contribution']
                ],
                'quizzes' => [
                    'count' => $grade_data['quiz_component']['contributing_quizzes'],
                    'average' => $grade_data['quiz_component']['average_score'],
                    'weight_applied' => $grade_data['weighting_applied']['quiz_weight'],
                    'contribution' => $grade_data['weighting_applied']['quiz_contribution']
                ]
            ],
            'calculation_method' => $grade_data['calculation_method'],
            'verification' => $grade_data['verification']
        ];
    }
}

/**
 * Helper function to get grade for a specific student/course
 */
function getStudentCourseGrade($student_id, $course_id, $db_connection = null) {
    global $conn;
    $connection = $db_connection ?? $conn;
    $calculator = new CourseGradeCalculator($connection);
    return $calculator->calculateFinalGrade($student_id, $course_id);
}

/**
 * Helper function to get grade report for display
 */
function getStudentGradeReport($student_id, $course_id, $db_connection = null) {
    global $conn;
    $connection = $db_connection ?? $conn;
    $calculator = new CourseGradeCalculator($connection);
    return $calculator->generateGradeReport($student_id, $course_id);
}
?>
