<?php
/**
 * Quiz-Focused Course Progress Tracking Service
 * QUIZ-ONLY SYSTEM: Course completion and grading based exclusively on quiz performance
 * Assignments are excluded from all progress and completion calculations
 */

class CourseProgressService {
    private $conn;
    
    public function __construct($database_connection) {
        $this->conn = $database_connection;
    }
    
    /**
     * Calculate overall course progress for a student
     * QUIZ-ONLY SYSTEM: Course completion based exclusively on quiz performance
     * @param int $student_id
     * @param int $course_id
     * @return array Progress breakdown with overall percentage
     */
    public function calculateCourseProgress($student_id, $course_id) {
        $progress = [
            'overall_percentage' => 0,
            'quizzes' => ['completed' => 0, 'total' => 0, 'percentage' => 0, 'average_score' => 0],
            'completion_method' => 'quiz_only_system',
            'completion_status' => 'not_started',
            'last_updated' => date('Y-m-d H:i:s')
        ];
        
        try {
            // QUIZ-ONLY CALCULATION: Only quiz performance determines course completion
            // Assignments and other activities are excluded from progress calculations
            
            $quizProgress = $this->calculateQuizProgress($student_id, $course_id);
            $progress['quizzes'] = $quizProgress;
            
            // QUIZ-ONLY SYSTEM: Overall percentage = Quiz percentage (100% weight)
            $progress['overall_percentage'] = $quizProgress['percentage'];
            
            // Determine completion status based on quiz performance only
            if ($progress['overall_percentage'] >= 100) {
                $progress['completion_status'] = 'completed';
            } elseif ($progress['overall_percentage'] >= 80) {
                $progress['completion_status'] = 'near_completion';
            } elseif ($progress['overall_percentage'] > 0) {
                $progress['completion_status'] = 'in_progress';
            } else {
                $progress['completion_status'] = 'not_started';
            }
            
            // Additional quiz-specific status information
            if ($quizProgress['total'] > 0) {
                $progress['quiz_details'] = [
                    'passed_quizzes' => $quizProgress['completed'],
                    'total_quizzes' => $quizProgress['total'],
                    'pass_rate' => round(($quizProgress['completed'] / $quizProgress['total']) * 100, 1),
                    'average_score' => $quizProgress['average_score']
                ];
            }
            
            // Update the enrollments table with the new progress
            $this->updateEnrollmentProgress($student_id, $course_id, $progress['overall_percentage']);
            
            return $progress;
            
        } catch (Exception $e) {
            error_log("Quiz-Only Progress Calculation Error: " . $e->getMessage());
            return $progress; // Return default progress on error
        }
    }
    
    /**
     * Calculate video completion progress
     */
    private function calculateVideoProgress($student_id, $course_id) {
        $progress = ['completed' => 0, 'total' => 0, 'percentage' => 0];
        
        // Check if videos table exists first
        $table_check = $this->conn->query("SHOW TABLES LIKE 'videos'");
        if (!$table_check || $table_check->num_rows === 0) {
            // Videos table doesn't exist, return default progress
            error_log("Videos table not found, returning default progress");
            return $progress;
        }
        
        try {
            // Get total videos for the course
            $totalStmt = $this->conn->prepare("
                SELECT COUNT(*) as total 
                FROM videos v
                JOIN courses c ON v.course_id = c.course_id
                WHERE c.course_id = ? AND v.status = 'active'
            ");
            
            if (!$totalStmt) {
                error_log("Prepare failed for total videos query: " . $this->conn->error);
                return $progress;
            }
            
            $totalStmt->bind_param("i", $course_id);
            $totalStmt->execute();
            $totalResult = $totalStmt->get_result();
            $totalData = $totalResult->fetch_assoc();
            $progress['total'] = (int)$totalData['total'];
            
            if ($progress['total'] > 0) {
                // Get completed videos (90% or more watched)
                $completedStmt = $this->conn->prepare("
                    SELECT COUNT(*) as completed
                    FROM video_progress vp
                    JOIN videos v ON vp.video_id = v.video_id
                    JOIN courses c ON v.course_id = c.course_id
                    WHERE vp.student_id = ? AND c.course_id = ? 
                    AND vp.completion_percentage >= 90
                ");
                
                if (!$completedStmt) {
                    error_log("Prepare failed for completed videos query: " . $this->conn->error);
                    return $progress;
                }
                
                $completedStmt->bind_param("ii", $student_id, $course_id);
                $completedStmt->execute();
                $completedResult = $completedStmt->get_result();
                $completedData = $completedResult->fetch_assoc();
                $progress['completed'] = (int)$completedData['completed'];
                
                $progress['percentage'] = round(($progress['completed'] / $progress['total']) * 100, 2);
            }
            
        } catch (Exception $e) {
            error_log("Video Progress Calculation Error: " . $e->getMessage());
        }
        
        return $progress;
    }
    
    /**
     * Calculate assignment completion progress
     */
    private function calculateAssignmentProgress($student_id, $course_id) {
        $progress = ['completed' => 0, 'total' => 0, 'percentage' => 0, 'average_score' => 0];
        
        // Check if assignments table exists first
        $table_check = $this->conn->query("SHOW TABLES LIKE 'assignments'");
        if (!$table_check || $table_check->num_rows === 0) {
            // Assignments table doesn't exist, return default progress
            error_log("Assignments table not found, returning default progress");
            return $progress;
        }
        
        try {
            // Get total assignments for the course
            $totalStmt = $this->conn->prepare("
                SELECT COUNT(*) as total 
                FROM assignments 
                WHERE course_id = ? AND status = 'active'
            ");
            
            if (!$totalStmt) {
                error_log("Prepare failed for total assignments query: " . $this->conn->error);
                return $progress;
            }
            
            $totalStmt->bind_param("i", $course_id);
            $totalStmt->execute();
            $totalResult = $totalStmt->get_result();
            $totalData = $totalResult->fetch_assoc();
            $progress['total'] = (int)$totalData['total'];
            
            if ($progress['total'] > 0) {
                // Get completed assignments (submitted and graded/scored)
                $completedStmt = $this->conn->prepare("
                    SELECT COUNT(*) as completed,
                           AVG(COALESCE(ai_score, 0)) as avg_ai_score,
                           AVG(CASE WHEN grade IS NOT NULL THEN (grade/a.points)*100 ELSE NULL END) as avg_manual_score
                    FROM assignment_submissions asub
                    JOIN assignments a ON asub.assignment_id = a.assignment_id
                    WHERE asub.student_id = ? AND a.course_id = ? 
                    AND asub.submitted_at IS NOT NULL
                ");
                
                if (!$completedStmt) {
                    error_log("Prepare failed for completed assignments query: " . $this->conn->error);
                    return $progress;
                }
                
                $completedStmt->bind_param("ii", $student_id, $course_id);
                $completedStmt->execute();
                $completedResult = $completedStmt->get_result();
                $completedData = $completedResult->fetch_assoc();
                $progress['completed'] = (int)$completedData['completed'];
                
                // Use manual score if available, otherwise AI score
                $avgScore = $completedData['avg_manual_score'] ?: $completedData['avg_ai_score'];
                $progress['average_score'] = round($avgScore ?: 0, 2);
                
                $progress['percentage'] = round(($progress['completed'] / $progress['total']) * 100, 2);
            }
            
        } catch (Exception $e) {
            error_log("Assignment Progress Calculation Error: " . $e->getMessage());
        }
     * Calculate quiz completion progress
     */
    private function calculateQuizProgress($student_id, $course_id) {
        $progress = ['completed' => 0, 'total' => 0, 'percentage' => 0, 'average_score' => 0];
        
        // Check if quizzes table exists first
        $table_check = $this->conn->query("SHOW TABLES LIKE 'quizzes'");
        if (!$table_check || $table_check->num_rows === 0) {
            // Quizzes table doesn't exist, return default progress
            error_log("Quizzes table not found, returning default progress");
            return $progress;
        }
        
        try {
            // Get total quizzes for the course
            $totalStmt = $this->conn->prepare("
                SELECT COUNT(*) as total 
                FROM quizzes 
                WHERE course_id = ? AND status = 'active'
            ");
            
            if (!$totalStmt) {
                error_log("Prepare failed for total quizzes query: " . $this->conn->error);
                return $progress;
            }
            
            $totalStmt->bind_param("i", $course_id);
            $totalStmt->execute();
            $totalResult = $totalStmt->get_result();
            $totalData = $totalResult->fetch_assoc();
            $progress['total'] = (int)$totalData['total'];
            
            if ($progress['total'] > 0) {
                // Get completed quizzes (attempted and scored)
                $completedStmt = $this->conn->prepare("
                    SELECT COUNT(DISTINCT q.quiz_id) as completed,
                           AVG(COALESCE(qa.ai_score, qa.score, 0)) as avg_score
                    FROM quiz_attempts qa
                    JOIN quizzes q ON qa.quiz_id = q.quiz_id
                    WHERE qa.student_id = ? AND q.course_id = ? 
                    AND qa.submitted_at IS NOT NULL
                ");
                
                if (!$completedStmt) {
                    error_log("Prepare failed for completed quizzes query: " . $this->conn->error);
                    return $progress;
                }
                
                $completedStmt->bind_param("ii", $student_id, $course_id);
                $completedStmt->execute();
                $completedResult = $completedStmt->get_result();
                $completedData = $completedResult->fetch_assoc();
                $progress['completed'] = (int)$completedData['completed'];
                $progress['average_score'] = round($completedData['avg_score'] ?: 0, 2);
                
                $progress['percentage'] = round(($progress['completed'] / $progress['total']) * 100, 2);
            }
            
        } catch (Exception $e) {
            error_log("Quiz Progress Calculation Error: " . $e->getMessage());
        }
        
        return $progress;
    }
    
    /**
     * Update enrollment progress in database
     */
    private function updateEnrollmentProgress($student_id, $course_id, $overall_percentage) {
        // Check if enrollments table exists first
        $table_check = $this->conn->query("SHOW TABLES LIKE 'enrollments'");
        if (!$table_check || $table_check->num_rows === 0) {
            // Enrollments table doesn't exist, skip update
            error_log("Enrollments table not found, skipping progress update");
            return;
        }
        
        try {
            $updateStmt = $this->conn->prepare("
                UPDATE enrollments 
                SET progress = ?, updated_at = NOW()
                WHERE student_id = ? AND course_id = ?
            ");
            
            if (!$updateStmt) {
                error_log("Prepare failed for enrollment progress update: " . $this->conn->error);
                return;
            }
            
            $updateStmt->bind_param("dii", $overall_percentage, $student_id, $course_id);
            $updateStmt->execute();
            
        } catch (Exception $e) {
            error_log("Enrollment Progress Update Error: " . $e->getMessage());
        }
    }
    
    /**
     * Get progress for all enrolled courses of a student
     */
    public function getAllCoursesProgress($student_id) {
        $coursesProgress = [];
        
        // Check if enrollments table exists first
        $table_check = $this->conn->query("SHOW TABLES LIKE 'enrollments'");
        if (!$table_check || $table_check->num_rows === 0) {
            // Enrollments table doesn't exist, return empty array
            error_log("Enrollments table not found, returning empty courses progress");
            return $coursesProgress;
        }
        
        try {
            // Get all enrolled courses
            $coursesStmt = $this->conn->prepare("
                SELECT e.course_id, c.title
                FROM enrollments e
                JOIN courses c ON e.course_id = c.course_id
                WHERE e.student_id = ?
            ");
            
            if (!$coursesStmt) {
                error_log("Prepare failed for all courses progress query: " . $this->conn->error);
                return $coursesProgress;
            }
            
            $coursesStmt->bind_param("i", $student_id);
            $coursesStmt->execute();
            $result = $coursesStmt->get_result();
            
            while ($course = $result->fetch_assoc()) {
                $progress = $this->calculateCourseProgress($student_id, $course['course_id']);
                $coursesProgress[$course['course_id']] = [
                    'course_title' => $course['title'],
                    'progress' => $progress
                ];
            }
            
        } catch (Exception $e) {
            error_log("All Courses Progress Calculation Error: " . $e->getMessage());
        }
        
        return $coursesProgress;
    }
    
    /**
     * Check if student is eligible for certificate
     */
    public function isCertificateEligible($student_id, $course_id, $minimum_percentage = 80) {
        $progress = $this->calculateCourseProgress($student_id, $course_id);
        
        return [
            'eligible' => $progress['overall_percentage'] >= $minimum_percentage,
            'current_percentage' => $progress['overall_percentage'],
            'minimum_required' => $minimum_percentage,
            'completion_details' => $progress
        ];
    }
}
?>
