<?php
/**
 * Quiz-Only Course Progress Tracking Service
 * QUIZ-ONLY SYSTEM: Course completion and grading based exclusively on quiz performance
 * Assignments are completely excluded from all progress and completion calculations
 */

class QuizOnlyProgressService {
    private $conn;
    
    public function __construct($database_connection) {
        $this->conn = $database_connection;
    }
    
    /**
     * Calculate overall course progress for a student
     * QUIZ-ONLY SYSTEM: Course completion based exclusively on quiz performance
     * CRITICAL FIX: Prevents empty courses from showing completion
     * @param int $student_id
     * @param int $course_id
     * @return array Progress breakdown with overall percentage
     */
    public function calculateCourseProgress($student_id, $course_id) {
        $progress = [
            'overall_percentage' => 0,
            'quizzes' => ['completed' => 0, 'total' => 0, 'percentage' => 0, 'average_score' => 0],
            'completion_method' => 'quiz_only_system',
            'completion_status' => 'not_started',
            'last_updated' => date('Y-m-d H:i:s'),
            'has_content' => false,
            'error_message' => null
        ];
        
        try {
            // CRITICAL FIX: Check if course has ANY content before calculating progress
            $quizProgress = $this->calculateQuizProgress($student_id, $course_id);
            $progress['quizzes'] = $quizProgress;
            
            // CRITICAL PROFESSIONAL FIX: Courses with NO CONTENT cannot be completed!
            if ($quizProgress['total'] === 0) {
                // Course has NO quizzes - cannot be completed or get certificate
                $progress['overall_percentage'] = 0;
                $progress['completion_status'] = 'no_content_available';
                $progress['has_content'] = false;
                $progress['error_message'] = 'Course has no quizzes or assessments available. Progress cannot be calculated until content is added by the instructor.';
                
                error_log("CRITICAL: Course ID $course_id has NO CONTENT but student $student_id tried to access progress");
                
                // Update enrollment with 0% progress for empty courses
                $this->updateEnrollmentProgress($student_id, $course_id, 0);
                
                return $progress;
            }
            
            // Course has content - proceed with normal progress calculation
            $progress['has_content'] = true;
            $progress['overall_percentage'] = $quizProgress['percentage'];
            
            // Determine completion status based on quiz performance only
            if ($progress['overall_percentage'] >= 100) {
                $progress['completion_status'] = 'completed';
            } elseif ($progress['overall_percentage'] >= 80) {
                $progress['completion_status'] = 'near_completion';
            } elseif ($progress['overall_percentage'] > 0) {
                $progress['completion_status'] = 'in_progress';
            } else {
                $progress['completion_status'] = 'not_started';
            }
            
            // Additional quiz-specific status information
            $progress['quiz_details'] = [
                'passed_quizzes' => $quizProgress['completed'],
                'total_quizzes' => $quizProgress['total'],
                'pass_rate' => round(($quizProgress['completed'] / $quizProgress['total']) * 100, 1),
                'average_score' => $quizProgress['average_score']
            ];
            
            // CRITICAL FIX: Only trigger certificate if course has content AND is completed
            if ($progress['overall_percentage'] >= 100 && $progress['has_content']) {
                $this->triggerCertificateGeneration($student_id, $course_id);
            }
            
            // Update the enrollments table with the new progress
            $this->updateEnrollmentProgress($student_id, $course_id, $progress['overall_percentage']);
            
            return $progress;
            
        } catch (Exception $e) {
            error_log("Quiz-Only Progress Calculation Error: " . $e->getMessage());
            $progress['error_message'] = 'Unable to calculate progress due to system error.';
            return $progress; // Return default progress on error
        }
    }
    
    /**
     * Calculate quiz completion progress - ENHANCED FOR QUIZ-ONLY SYSTEM
     */
    private function calculateQuizProgress($student_id, $course_id) {
        $progress = ['completed' => 0, 'total' => 0, 'percentage' => 0, 'average_score' => 0];
        
        // Check if quizzes table exists first
        $table_check = $this->conn->query("SHOW TABLES LIKE 'quizzes'");
        if (!$table_check || $table_check->num_rows === 0) {
            // Quizzes table doesn't exist, return default progress
            error_log("Quizzes table not found, returning default progress");
            return $progress;
        }
        
        try {
            // Get total quizzes for the course
            $totalStmt = $this->conn->prepare("
                SELECT COUNT(*) as total 
                FROM quizzes 
                WHERE course_id = ?
            ");
            
            if (!$totalStmt) {
                error_log("Prepare failed for total quizzes query: " . $this->conn->error);
                return $progress;
            }
            
            $totalStmt->bind_param("i", $course_id);
            $totalStmt->execute();
            $totalResult = $totalStmt->get_result();
            $totalData = $totalResult->fetch_assoc();
            $progress['total'] = (int)$totalData['total'];
            
            if ($progress['total'] > 0) {
                // Get passed quizzes (latest attempt with passing score)
                $completedStmt = $this->conn->prepare("
                    SELECT 
                        COUNT(DISTINCT q.quiz_id) as completed,
                        AVG(qa.score) as avg_score,
                        COUNT(DISTINCT qa.quiz_id) as attempted_quizzes
                    FROM quizzes q
                    LEFT JOIN (
                        SELECT qa1.quiz_id, qa1.student_id, qa1.score
                        FROM quiz_attempts qa1
                        INNER JOIN (
                            SELECT quiz_id, student_id, MAX(attempt_id) as max_attempt
                            FROM quiz_attempts
                            WHERE student_id = ?
                            GROUP BY quiz_id, student_id
                        ) qa2 ON qa1.quiz_id = qa2.quiz_id 
                        AND qa1.student_id = qa2.student_id 
                        AND qa1.attempt_id = qa2.max_attempt
                        WHERE qa1.score >= (
                            SELECT passing_score FROM quizzes WHERE quiz_id = qa1.quiz_id
                        )
                    ) qa ON q.quiz_id = qa.quiz_id
                    WHERE q.course_id = ?
                ");
                
                if (!$completedStmt) {
                    error_log("Prepare failed for completed quizzes query: " . $this->conn->error);
                    return $progress;
                }
                
                $completedStmt->bind_param("ii", $student_id, $course_id);
                $completedStmt->execute();
                $completedResult = $completedStmt->get_result();
                $completedData = $completedResult->fetch_assoc();
                
                $progress['completed'] = (int)$completedData['completed'];
                $progress['average_score'] = round($completedData['avg_score'] ?? 0, 2);
                
                // Calculate percentage: Must pass ALL quizzes to complete course
                $progress['percentage'] = $progress['total'] > 0 ? 
                    round(($progress['completed'] / $progress['total']) * 100, 2) : 0;
            }
            
        } catch (Exception $e) {
            error_log("Quiz Progress Calculation Error: " . $e->getMessage());
        }
        
        return $progress;
    }
    
    /**
     * Update enrollment progress in database
     */
    private function updateEnrollmentProgress($student_id, $course_id, $overall_percentage) {
        try {
            // Try course_enrollments first, then enrollments
            $tables_to_try = ['course_enrollments', 'enrollments'];
            
            foreach ($tables_to_try as $table) {
                $table_check = $this->conn->query("SHOW TABLES LIKE '{$table}'");
                if ($table_check && $table_check->num_rows > 0) {
                    
                    $updateStmt = $this->conn->prepare("
                        UPDATE {$table} 
                        SET progress = ?, updated_at = NOW()
                        WHERE user_id = ? AND course_id = ?
                    ");
                    
                    if (!$updateStmt) {
                        // Try with student_id column name
                        $updateStmt = $this->conn->prepare("
                            UPDATE {$table} 
                            SET progress = ?, updated_at = NOW()
                            WHERE student_id = ? AND course_id = ?
                        ");
                    }
                    
                    if ($updateStmt) {
                        $updateStmt->bind_param("dii", $overall_percentage, $student_id, $course_id);
                        $updateStmt->execute();
                        break; // Success, exit loop
                    }
                }
            }
            
        } catch (Exception $e) {
            error_log("Enrollment Progress Update Error: " . $e->getMessage());
        }
    }
    
    /**
     * Trigger certificate generation for completed courses
     */
    private function triggerCertificateGeneration($student_id, $course_id) {
        try {
            // Check if certificate already exists
            $certCheck = $this->conn->prepare("
                SELECT certificate_id FROM certificates 
                WHERE student_id = ? AND course_id = ?
            ");
            
            if ($certCheck) {
                $certCheck->bind_param("ii", $student_id, $course_id);
                $certCheck->execute();
                $existing = $certCheck->get_result()->fetch_assoc();
                
                if (!$existing) {
                    // Include certificate automation if it exists
                    $cert_automation_file = __DIR__ . '/certificate_automation.php';
                    if (file_exists($cert_automation_file)) {
                        require_once $cert_automation_file;
                        if (function_exists('autoCheckCourseCompletion')) {
                            autoCheckCourseCompletion($student_id, $course_id);
                        }
                    }
                }
            }
            
        } catch (Exception $e) {
            error_log("Certificate Generation Trigger Error: " . $e->getMessage());
        }
    }
    
    /**
     * Get progress for all enrolled courses of a student - QUIZ-ONLY
     */
    public function getAllCoursesProgress($student_id) {
        $coursesProgress = [];
        
        try {
            // Try different enrollment table names
            $enrollment_tables = ['course_enrollments', 'enrollments'];
            $coursesStmt = null;
            
            foreach ($enrollment_tables as $table) {
                $table_check = $this->conn->query("SHOW TABLES LIKE '{$table}'");
                if ($table_check && $table_check->num_rows > 0) {
                    
                    $coursesStmt = $this->conn->prepare("
                        SELECT e.course_id, c.title, c.course_title
                        FROM {$table} e
                        JOIN courses c ON e.course_id = c.course_id
                        WHERE e.user_id = ? OR e.student_id = ?
                    ");
                    
                    if ($coursesStmt) {
                        $coursesStmt->bind_param("ii", $student_id, $student_id);
                        $coursesStmt->execute();
                        $result = $coursesStmt->get_result();
                        
                        while ($course = $result->fetch_assoc()) {
                            $progress = $this->calculateCourseProgress($student_id, $course['course_id']);
                            $coursesProgress[$course['course_id']] = [
                                'course_title' => $course['title'] ?: $course['course_title'],
                                'progress' => $progress
                            ];
                        }
                        break; // Success, exit loop
                    }
                }
            }
            
        } catch (Exception $e) {
            error_log("All Courses Progress Calculation Error: " . $e->getMessage());
        }
        
        return $coursesProgress;
    }
    
    /**
     * Check if student is eligible for certificate - QUIZ-ONLY SYSTEM
     * CRITICAL FIX: Prevents certificate eligibility for courses with NO CONTENT
     */
    public function isCertificateEligible($student_id, $course_id, $minimum_percentage = null) {
        if ($minimum_percentage === null) {
            $minimum_percentage = $this->getCoursePassingGrade($course_id);
        }
        
        $progress = $this->calculateCourseProgress($student_id, $course_id);
        
        // CRITICAL PROFESSIONAL FIX: Course must have content AND student must pass
        if (!$progress['has_content']) {
            error_log("CERTIFICATE ELIGIBILITY DENIED: Course ID $course_id has NO CONTENT - Student $student_id cannot get certificate");
            return false;
        }
        
        if ($progress['completion_status'] === 'no_content_available') {
            error_log("CERTIFICATE ELIGIBILITY DENIED: Course ID $course_id marked as no content available");
            return false;
        }
        
        // Student is eligible only if:
        // 1. Course has content
        // 2. Student meets minimum percentage requirement  
        // 3. Student has actually completed quizzes
        return $progress['has_content'] && 
               $progress['overall_percentage'] >= $minimum_percentage &&
               $progress['quizzes']['total'] > 0 &&
               $progress['quizzes']['completed'] > 0;
    }

    /**
     * Get course passing grade from database
     */
    private function getCoursePassingGrade($course_id) {
        try {
            $stmt = $this->conn->prepare("SELECT course_passing_grade FROM courses WHERE course_id = ?");
            if (!$stmt) {
                error_log("Failed to prepare course passing grade query: " . $this->conn->error);
                return 60.00; // Default fallback
            }
            
            $stmt->bind_param("i", $course_id);
            $stmt->execute();
            $result = $stmt->get_result();
            $row = $result->fetch_assoc();
            
            return floatval($row['course_passing_grade'] ?? 60.00);
            
        } catch (Exception $e) {
            error_log("Error getting course passing grade: " . $e->getMessage());
            return 60.00; // Default fallback
        }
    }
    
    /**
     * Get detailed quiz performance report
     */
    public function getQuizPerformanceReport($student_id, $course_id) {
        $report = [
            'course_id' => $course_id,
            'student_id' => $student_id,
            'quiz_details' => [],
            'overall_stats' => [
                'total_quizzes' => 0,
                'passed_quizzes' => 0,
                'failed_quizzes' => 0,
                'average_score' => 0,
                'completion_percentage' => 0
            ]
        ];
        
        try {
            // Get all quizzes for the course with student's performance (FIXED COLUMN NAMES)
            $quizStmt = $this->conn->prepare("
                SELECT 
                    q.quiz_id,
                    q.title,
                    q.passing_score,
                    COALESCE(qa.score, 0) as score,
                    qa.end_time as attempt_date,
                    qa.attempt_id as attempt_number,
                    qa.pass_status,
                    CASE 
                        WHEN qa.score IS NULL THEN 'not_attempted'
                        WHEN qa.pass_status = 1 THEN 'passed'
                        WHEN qa.score >= q.passing_score THEN 'passed' 
                        ELSE 'failed' 
                    END as status
                FROM quizzes q
                LEFT JOIN (
                    SELECT 
                        qa1.quiz_id,
                        qa1.student_id, 
                        qa1.score, 
                        qa1.end_time, 
                        qa1.attempt_id,
                        qa1.pass_status
                    FROM quiz_attempts qa1
                    INNER JOIN (
                        SELECT quiz_id, student_id, MAX(attempt_id) as max_attempt_id
                        FROM quiz_attempts
                        WHERE student_id = ?
                        GROUP BY quiz_id, student_id
                    ) qa2 ON qa1.quiz_id = qa2.quiz_id 
                        AND qa1.student_id = qa2.student_id 
                        AND qa1.attempt_id = qa2.max_attempt_id
                ) qa ON q.quiz_id = qa.quiz_id AND qa.student_id = ?
                WHERE q.course_id = ?
                ORDER BY q.quiz_id
            ");
            
            if ($quizStmt) {
                $quizStmt->bind_param("iii", $student_id, $student_id, $course_id);
                $quizStmt->execute();
                $result = $quizStmt->get_result();
                
                $total_score = 0;
                $quiz_count = 0;
                
                while ($quiz = $result->fetch_assoc()) {
                    $report['quiz_details'][] = [
                        'quiz_id' => $quiz['quiz_id'],
                        'title' => $quiz['title'],
                        'passing_score' => $quiz['passing_score'],
                        'student_score' => $quiz['score'],
                        'status' => $quiz['status'] ?: 'not_attempted',
                        'attempt_date' => $quiz['attempt_date'],
                        'attempt_number' => $quiz['attempt_number']
                    ];
                    
                    $report['overall_stats']['total_quizzes']++;
                    
                    if ($quiz['score'] !== null) {
                        $total_score += $quiz['score'];
                        $quiz_count++;
                        
                        if ($quiz['status'] === 'passed') {
                            $report['overall_stats']['passed_quizzes']++;
                        } else {
                            $report['overall_stats']['failed_quizzes']++;
                        }
                    }
                }
                
                // Calculate overall statistics
                if ($quiz_count > 0) {
                    $report['overall_stats']['average_score'] = round($total_score / $quiz_count, 2);
                }
                
                if ($report['overall_stats']['total_quizzes'] > 0) {
                    $report['overall_stats']['completion_percentage'] = round(
                        ($report['overall_stats']['passed_quizzes'] / $report['overall_stats']['total_quizzes']) * 100, 
                        2
                    );
                }
            }
            
        } catch (Exception $e) {
            error_log("Quiz Performance Report Error: " . $e->getMessage());
        }
        
        return $report;
    }
}
?>
