<?php
/**
 * Panadite Academy Helper Functions
 * Contains common utility functions used throughout the application
 */

/**
 * Sanitize user input
 * @param string $input The input to sanitize
 * @return string The sanitized input
 */
function sanitize_input($input) {
    return htmlspecialchars(trim($input), ENT_QUOTES, 'UTF-8');
}

/**
 * Get user information by ID
 * @param mysqli $conn Database connection
 * @param int $user_id User ID
 * @return array|false User data as associative array or false if not found
 */
function getUserById($conn, $user_id) {
    $stmt = $conn->prepare("SELECT * FROM users WHERE user_id = ?");
    if (!$stmt) {
        return false;
    }
    
    $stmt->bind_param("i", $user_id);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows === 0) {
        return false;
    }
    
    return $result->fetch_assoc();
}

/**
 * Get teacher information by user ID
 * @param mysqli $conn Database connection
 * @param int $user_id User ID
 * @return array|false Teacher data or false if not found
 */
function get_teacher($conn, $user_id) {
    // Check if connection is valid
    if (!$conn) {
        return [
            'user_id' => $user_id,
            'username' => 'Unknown',
            'email' => 'unknown@example.com',
            'user_role' => 'teacher',
            'bio' => 'Connection error',
            'expertise' => '',
            'qualifications' => ''
        ];
    }
    
    try {
        $stmt = $conn->prepare("
            SELECT u.*, t.bio, t.expertise, t.qualifications
            FROM users u
            LEFT JOIN teacher_profiles t ON u.user_id = t.teacher_id
            WHERE u.user_id = ? AND u.user_role = 'teacher'
        ");
        
        if (!$stmt) {
            return false;
        }
    } catch (Exception $e) {
        // Return fallback data if query preparation fails
        return [
            'user_id' => $user_id,
            'username' => 'Unknown',
            'email' => 'unknown@example.com',
            'user_role' => 'teacher',
            'bio' => 'Database error: ' . $e->getMessage(),
            'expertise' => '',
            'qualifications' => ''
        ];
    }
    
    $stmt->bind_param("i", $user_id);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows === 0) {
        return false;
    }
    
    return $result->fetch_assoc();
}

/**
 * Check if a user has specific permission
 * @param mysqli $conn Database connection
 * @param int $user_id User ID
 * @param string $permission Permission name
 * @return bool True if user has permission, false otherwise
 */
function has_permission($conn, $user_id, $permission) {
    // First check if user is admin, admins have all permissions
    $stmt = $conn->prepare("SELECT user_role FROM users WHERE user_id = ? LIMIT 1");
    
    if (!$stmt) {
        return false;
    }
    
    $stmt->bind_param("i", $user_id);
    $stmt->execute();
    $result = $stmt->get_result();
    $user = $result->fetch_assoc();
    
    // If user is admin, they have all permissions
    if ($user && $user['user_role'] === 'admin') {
        return true;
    }
    
    // Otherwise check admin_profiles for specific permissions
    $stmt = $conn->prepare("
        SELECT permissions FROM admin_profiles 
        WHERE admin_id = ? 
        LIMIT 1
    ");
    
    if (!$stmt) {
        return false;
    }
    
    $stmt->bind_param("i", $user_id);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows === 0) {
        return false;
    }
    
    $row = $result->fetch_assoc();
    $permissions = $row['permissions'] ? json_decode($row['permissions'], true) : [];
    
    return in_array($permission, $permissions);
}

/**
 * Format date in a user-friendly way
 * @param string $date The date string
 * @param string $format The date format (default: 'M j, Y')
 * @return string Formatted date
 */
function format_date($date, $format = 'M j, Y') {
    return date($format, strtotime($date));
}

/**
 * Generate a random string (for tokens, etc.)
 * @param int $length Length of the string
 * @return string Random string
 */
function generate_random_string($length = 16) {
    $characters = '0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ';
    $string = '';
    
    for ($i = 0; $i < $length; $i++) {
        $string .= $characters[rand(0, strlen($characters) - 1)];
    }
    
    return $string;
}

/**
 * Redirect to another page
 * @param string $location URL to redirect to
 */
function redirect($location) {
    header("Location: $location");
    exit();
}

/**
 * Count total courses for a teacher
 * @param mysqli $conn Database connection
 * @param int $teacher_id Teacher ID
 * @return int Number of courses
 */
function count_teacher_courses($conn, $teacher_id) {
    $stmt = $conn->prepare("SELECT COUNT(*) as count FROM courses WHERE teacher_id = ?");
    if (!$stmt) {
        return 0;
    }
    
    $stmt->bind_param("i", $teacher_id);
    $stmt->execute();
    $result = $stmt->get_result();
    $data = $result->fetch_assoc();
    
    return $data['count'] ?? 0;
}

/**
 * Count total videos uploaded by a teacher
 * @param mysqli $conn Database connection
 * @param int $teacher_id Teacher ID
 * @return int Number of videos
 */
function count_teacher_videos($conn, $teacher_id) {
    $stmt = $conn->prepare("SELECT COUNT(*) as count FROM videos WHERE teacher_id = ?");
    if (!$stmt) {
        return 0;
    }
    
    $stmt->bind_param("i", $teacher_id);
    $stmt->execute();
    $result = $stmt->get_result();
    $data = $result->fetch_assoc();
    
    return $data['count'] ?? 0;
}

/**
 * Count students enrolled in a teacher's courses
 * @param mysqli $conn Database connection
 * @param int $teacher_id Teacher ID
 * @return int Number of students
 */
function count_teacher_students($conn, $teacher_id) {
    $stmt = $conn->prepare("
        SELECT COUNT(DISTINCT e.student_id) as count 
        FROM enrollments e
        JOIN courses c ON e.course_id = c.course_id
        WHERE c.teacher_id = ?
    ");
    
    if (!$stmt) {
        return 0;
    }
    
    $stmt->bind_param("i", $teacher_id);
    $stmt->execute();
    $result = $stmt->get_result();
    $data = $result->fetch_assoc();
    
    return $data['count'] ?? 0;
}

/**
 * Get recent courses for a teacher
 * @param mysqli $conn Database connection
 * @param int $teacher_id Teacher ID
 * @param int $limit Number of courses to return (default: 5)
 * @return mysqli_result Result set with courses
 */
function get_teacher_recent_courses($conn, $teacher_id, $limit = 5) {
    $stmt = $conn->prepare("
        SELECT * FROM courses 
        WHERE teacher_id = ? 
        ORDER BY created_at DESC 
        LIMIT ?
    ");
    
    if (!$stmt) {
        return false;
    }
    
    $stmt->bind_param("ii", $teacher_id, $limit);
    $stmt->execute();
    
    return $stmt->get_result();
}

/**
 * Get recent videos for a teacher
 * @param mysqli $conn Database connection
 * @param int $teacher_id Teacher ID
 * @param int $limit Number of videos to return (default: 5)
 * @return mysqli_result Result set with videos
 */
function get_teacher_recent_videos($conn, $teacher_id, $limit = 5) {
    $stmt = $conn->prepare("
        SELECT * FROM videos 
        WHERE teacher_id = ? 
        ORDER BY created_at DESC 
        LIMIT ?
    ");
    
    if (!$stmt) {
        return false;
    }
    
    $stmt->bind_param("ii", $teacher_id, $limit);
    $stmt->execute();
    
    return $stmt->get_result();
}
