<?php
/**
 * API: Get Student Enrolled Courses with Grade Data
 * Returns enrolled courses with calculated grades and progress
 */

header('Content-Type: application/json');
require_once '../../auth/functions.php';
requireRole('student');
require_once '../../config/db_connect.php';
require_once '../../includes/course_grade_calculator.php';

$student_id = $_SESSION['user_id'];

try {
    // Get enrolled courses with course details
    $query = "
        SELECT 
            c.course_id,
            c.title,
            c.thumbnail,
            c.assignment_weight,
            c.quiz_weight,
            u.first_name as instructor_first,
            u.last_name as instructor_last,
            e.enrolled_date,
            e.progress as enrollment_progress
        FROM courses c
        JOIN enrollments e ON c.course_id = e.course_id
        JOIN users u ON c.teacher_id = u.user_id
        WHERE e.student_id = ?
        ORDER BY e.enrolled_date DESC
    ";
    
    $stmt = $conn->prepare($query);
    $stmt->bind_param("i", $student_id);
    $stmt->execute();
    $result = $stmt->get_result();
    
    $courses = [];
    $calculator = new CourseGradeCalculator($conn);
    
    while ($course = $result->fetch_assoc()) {
        // Calculate grade for this course
        $grade_data = $calculator->calculateFinalGrade($student_id, $course['course_id']);
        
        // Calculate progress based on completed assignments and quizzes
        $progress = calculateCourseProgress($student_id, $course['course_id'], $conn);
        
        // Determine grade classification
        $final_grade = $grade_data['final_grade'];
        $grade_class = getGradeClass($final_grade);
        $grade_text = getGradeText($final_grade);
        
        // Check if eligible for certificate (50% or higher)
        $certificate_eligible = $final_grade >= 50;
        
        $courses[] = [
            'course_id' => $course['course_id'],
            'title' => $course['title'],
            'thumbnail' => $course['thumbnail'] ?: 'default-course.jpg',
            'instructor' => trim($course['instructor_first'] . ' ' . $course['instructor_last']),
            'enrolled_date' => $course['enrolled_date'],
            'enrollment_progress' => $course['enrollment_progress'],
            'final_grade' => $final_grade,
            'letter_grade' => $grade_data['letter_grade'],
            'grade_class' => $grade_class,
            'grade_text' => $grade_text,
            'progress_percentage' => $progress['percentage'],
            'progress_status' => $progress['status'],
            'assignment_count' => $grade_data['assignment_component']['contributing_assignments'],
            'quiz_count' => $grade_data['quiz_component']['contributing_quizzes'],
            'certificate_eligible' => $certificate_eligible,
            'course_weights' => [
                'assignment_weight' => $course['assignment_weight'],
                'quiz_weight' => $course['quiz_weight']
            ]
        ];
    }
    
    echo json_encode([
        'success' => true,
        'courses' => $courses,
        'total_enrolled' => count($courses)
    ]);
    
} catch (Exception $e) {
    echo json_encode([
        'success' => false,
        'error' => 'Failed to fetch courses: ' . $e->getMessage()
    ]);
}

/**
 * Calculate course progress based on completed assignments and quizzes
 */
function calculateCourseProgress($student_id, $course_id, $conn) {
    // Count total assignments and quizzes
    $total_query = "
        SELECT 
            (SELECT COUNT(*) FROM assignments WHERE course_id = ?) as total_assignments,
            (SELECT COUNT(*) FROM quizzes WHERE course_id = ? AND is_published = 1) as total_quizzes
    ";
    
    $stmt = $conn->prepare($total_query);
    $stmt->bind_param("ii", $course_id, $course_id);
    $stmt->execute();
    $totals = $stmt->get_result()->fetch_assoc();
    
    // Count completed assignments and quizzes
    $completed_query = "
        SELECT 
            (SELECT COUNT(*) FROM submissions s 
             JOIN assignments a ON s.assignment_id = a.assignment_id 
             WHERE a.course_id = ? AND s.student_id = ? AND s.submitted_at IS NOT NULL) as completed_assignments,
            (SELECT COUNT(*) FROM quiz_attempts qa 
             JOIN quizzes q ON qa.quiz_id = q.quiz_id 
             WHERE q.course_id = ? AND qa.student_id = ? AND qa.end_time IS NOT NULL) as completed_quizzes
    ";
    
    $stmt = $conn->prepare($completed_query);
    $stmt->bind_param("iiii", $course_id, $student_id, $course_id, $student_id);
    $stmt->execute();
    $completed = $stmt->get_result()->fetch_assoc();
    
    $total_items = $totals['total_assignments'] + $totals['total_quizzes'];
    $completed_items = $completed['completed_assignments'] + $completed['completed_quizzes'];
    
    $percentage = $total_items > 0 ? round(($completed_items / $total_items) * 100) : 0;
    
    // Determine status
    if ($percentage == 100) {
        $status = 'Completed';
    } elseif ($percentage >= 50) {
        $status = 'In Progress';
    } elseif ($percentage > 0) {
        $status = 'Started';
    } else {
        $status = 'Not Started';
    }
    
    return [
        'percentage' => $percentage,
        'status' => $status,
        'completed_items' => $completed_items,
        'total_items' => $total_items
    ];
}

/**
 * Get CSS class for grade display
 */
function getGradeClass($grade) {
    if ($grade >= 90) return 'grade-excellent';
    if ($grade >= 80) return 'grade-good';
    if ($grade >= 70) return 'grade-average';
    if ($grade >= 50) return 'grade-poor';
    return 'grade-none';
}

/**
 * Get grade text description
 */
function getGradeText($grade) {
    if ($grade >= 90) return 'Excellent';
    if ($grade >= 80) return 'Good';
    if ($grade >= 70) return 'Average';
    if ($grade >= 50) return 'Passing';
    if ($grade > 0) return 'Below Pass';
    return 'No Grade';
}
?>
