<?php
session_start();
require_once 'includes/db.php'; // Use the correct database connection
require_once 'auth/functions.php'; // Authentication functions including getCurrentUser
require_once 'config/functions.php';

// Check if database connection is working
$db_available = !isset($GLOBALS['db_connection_failed']) || !$GLOBALS['db_connection_failed'];

// Get current user if logged in and if DB is available
$currentUser = $db_available ? getCurrentUser() : null;

// Get featured videos
$featuredVideos = [];
if ($db_available) {
    try {
        // Direct MySQLi query instead of fetchAll to better diagnose any issues
        $videosQuery = "SELECT v.*, 
                        IFNULL(c.title, 'Untitled Course') as course_title, 
                        IFNULL(c.price, 0) as course_price, 
                        IFNULL(u.first_name, 'Instructor') as first_name, 
                        IFNULL(u.last_name, '') as last_name
                      FROM videos v
                      LEFT JOIN courses c ON v.course_id = c.course_id
                      LEFT JOIN users u ON c.teacher_id = u.user_id
                      ORDER BY v.created_at DESC
                      LIMIT 8";
                      
        $result = $conn->query($videosQuery);
        
        if ($result && $result->num_rows > 0) {
            while ($row = $result->fetch_assoc()) {
                $featuredVideos[] = $row;
            }
            // Add debug comment
            echo '<!--';
            echo 'DEBUG: Found ' . count($featuredVideos) . ' videos';
            echo '-->';
        } else {
            // Try a simpler query without the joins
            $simpleVideosQuery = "SELECT * FROM videos ORDER BY created_at DESC LIMIT 8";
            $result = $conn->query($simpleVideosQuery);
            
            if ($result && $result->num_rows > 0) {
                while ($row = $result->fetch_assoc()) {
                    $featuredVideos[] = $row;
                }
                echo '<!--';
                echo 'DEBUG: Found ' . count($featuredVideos) . ' videos with simple query';
                echo '-->';
            } else {
                echo '<!--';
                echo 'DEBUG: No videos found. SQL Error: ' . $conn->error;
                echo '-->';
            }
        }
    } catch (Exception $e) {
        // Log error but don't display it
        error_log('Database error fetching videos: ' . $e->getMessage());
    }
}

// Get featured courses with simplified approach
$featuredCourses = [];

if ($db_available) {
    try {
        // Get ALL courses - show everything, including drafts
        $basicQuery = "SELECT c.*, u.first_name, u.last_name 
                   FROM courses c 
                   LEFT JOIN users u ON c.teacher_id = u.user_id 
                   ORDER BY c.course_id DESC";
        $result = $conn->query($basicQuery);
        
        if ($result && $result->num_rows > 0) {
            // Explicitly create an array to hold courses
            $featuredCourses = array();
            
            while ($row = $result->fetch_assoc()) {
                // Add default values for missing fields
                if (empty($row['first_name'])) $row['first_name'] = 'Instructor';
                if (empty($row['last_name'])) $row['last_name'] = '';
                $row['category_name'] = 'General';
                $row['student_count'] = $row['student_count'] ?? 0;
                $row['rating'] = $row['rating'] ?? 4.5;
                $row['thumbnail'] = !empty($row['thumbnail']) ? $row['thumbnail'] : 'assets/images/course-placeholder.jpg';
                
                // Add to the array
                $featuredCourses[] = $row;
            }
            
            // Force debug output to see what we got
            echo '<!-- DEBUG: Found ' . count($featuredCourses) . ' courses -->';
            echo '<!-- DEBUG: First course: ' . json_encode(array_slice($featuredCourses, 0, 1)) . ' -->';
        } else {
            // Try an even simpler query
            $simpleQuery = "SELECT * FROM courses";
            $simpleResult = $conn->query($simpleQuery);
            
            if ($simpleResult && $simpleResult->num_rows > 0) {
                $featuredCourses = array();
                while ($row = $simpleResult->fetch_assoc()) {
                    $featuredCourses[] = $row;
                }
                echo '<!-- DEBUG: Found ' . count($featuredCourses) . ' courses with simple query -->';
            } else {
                echo '<!-- DEBUG: No courses found. SQL Error: ' . $conn->error . ' -->';
            }
        }
    } catch (Exception $e) {
        error_log('Error fetching courses: ' . $e->getMessage());
        echo '<!-- DEBUG: Exception: ' . $e->getMessage() . ' -->';
        $featuredCourses = [];
    }
}

// Get categories with video counts
$categories = [];
try {
    // Direct MySQLi query instead of fetchAll
    $categoriesQuery = "SELECT c.*, 
                        (SELECT COUNT(*) FROM videos WHERE category_name = c.category_name) as video_count 
                        FROM categories c 
                        ORDER BY c.category_name ASC";
    $result = $conn->query($categoriesQuery);
    
    if ($result && $result->num_rows > 0) {
        while ($row = $result->fetch_assoc()) {
            $categories[] = $row;
        }
        echo '<!-- DEBUG: Found ' . count($categories) . ' categories -->';
    } else {
        echo '<!-- DEBUG: No categories found. SQL Error: ' . $conn->error . ' -->';
    }
} catch (Exception $e) {
    // Log error but don't display it
    error_log('Database error fetching categories: ' . $e->getMessage());
    $categories = [];
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Panadite Academy - Get ahead of the curve</title>
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link rel="stylesheet" href="/assets/css/style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/animate.css/4.1.1/animate.min.css">
    <style>
        /* Clean Hero Section Styles */
        .modern-hero {
            position: relative;
            min-height: 80vh;
            display: flex;
            align-items: center;
            background: #f5f5f5; /* Whitesmoke background */
            overflow: hidden;
        }

        .hero-background {
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            z-index: 1;
        }

        .floating-icons {
            position: absolute;
            width: 100%;
            height: 100%;
            z-index: 2;
        }

        .floating-icon-card {
            position: absolute;
            width: 40px;
            height: 40px;
            background: transparent;
            border-radius: 8px;
            display: flex;
            align-items: center;
            justify-content: center;
            opacity: 0.1; /* Very faint icons */
        }

        .floating-icon-card i {
            font-size: 16px;
            color: #666;
        }

        .icon-1 {
            top: 20%;
            left: 10%;
        }

        .icon-2 {
            top: 30%;
            right: 15%;
        }

        .icon-3 {
            bottom: 35%;
            left: 15%;
        }

        .icon-4 {
            bottom: 25%;
            right: 25%;
        }

        .icon-5 {
            top: 60%;
            left: 8%;
        }

        .container {
            position: relative;
            z-index: 3;
        }

        .min-vh-80 {
            min-height: 80vh;
        }

        .hero-content-modern {
            padding: 2rem 0;
        }

        .hero-main-card {
            background: white;
            padding: 2.5rem 2rem;
            border-radius: 8px;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.08);
            max-width: 500px;
        }

        .hero-title {
            font-size: 2.8rem;
            font-weight: 700;
            line-height: 1.2;
            margin-bottom: 1.5rem;
            color: #2d3748;
        }

        .gradient-text {
            color: #276BE4;
        }

        .hero-description {
            font-size: 1.1rem;
            line-height: 1.6;
            color: #666;
            margin-bottom: 2rem;
        }

        .hero-cta-buttons {
            display: flex;
            gap: 1rem;
            flex-wrap: wrap;
        }

        .btn-modern-primary {
            background: #276BE4;
            border: none;
            color: white;
            padding: 0.875rem 1.75rem;
            font-size: 1rem;
            font-weight: 600;
            border-radius: 6px;
            text-decoration: none;
            transition: background 0.2s ease;
        }

        .btn-modern-primary:hover {
            background: #1e5bbf;
            color: white;
        }

        .btn-modern-secondary {
            background: transparent;
            border: 2px solid #276BE4;
            color: #276BE4;
            padding: 0.875rem 1.75rem;
            font-size: 1rem;
            font-weight: 600;
            border-radius: 6px;
            text-decoration: none;
            transition: all 0.2s ease;
        }

        .btn-modern-secondary:hover {
            background: #276BE4;
            color: white;
        }

        .hero-visual {
            display: flex;
            justify-content: center;
            align-items: center;
            height: 100%;
        }

        .hero-image-container {
            max-width: 450px;
            width: 100%;
        }

        .hero-image {
            width: 100%;
            height: auto;
            border-radius: 8px;
            box-shadow: 0 8px 24px rgba(0, 0, 0, 0.12);
        }

        /* Responsive Design */
        @media (max-width: 991px) {
            .hero-main-card {
                max-width: 100%;
                margin-bottom: 3rem;
            }
            
            .hero-title {
                font-size: 2.5rem;
            }
            
            .professional-figure {
                max-width: 400px;
            }
            
            .figure-bg-circle {
                width: 280px;
                height: 280px;
            }
            
            .professional-avatar {
                width: 240px;
                height: 240px;
            }
        }

        @media (max-width: 768px) {
            .modern-hero {
                min-height: 70vh;
                padding: 2rem 0;
            }
            
            .hero-title {
                font-size: 2rem;
            }
            
            .hero-description {
                font-size: 1.1rem;
            }
            
            .hero-cta-buttons {
                flex-direction: column;
            }
            
            .btn-modern-primary,
            .btn-modern-secondary {
                width: 100%;
                text-align: center;
            }
            
            .floating-icon-card {
                width: 40px;
                height: 40px;
            }
            
            .floating-icon-card i {
                font-size: 16px;
            }
            
            .skill-badge {
                display: none;
            }
        }

        /* Enhanced Floating Education Icons */
        .floating-icon {
            position: absolute;
            z-index: 1;
            pointer-events: none;
            transition: all 1.5s ease;
            color: rgba(100, 100, 100, 0.08); /* Default very light color */
            text-shadow: 0 0 10px rgba(100, 100, 100, 0.05);
        }
        
        /* Modern Search Form Styles */
        .hero-search-container {
            max-width: 600px;
            margin: 0 auto;
        }

        .search-input-group {
            display: flex;
            border-radius: 12px;
            overflow: hidden;
            box-shadow: 0 8px 20px rgba(0, 0, 0, 0.1);
            transition: all 0.3s ease;
            background: #fff;
            position: relative;
        }

        .search-input-group:hover {
            box-shadow: 0 10px 25px rgba(39, 107, 228, 0.15);
            transform: translateY(-2px);
        }

        .search-input {
            padding: 16px 20px;
            border: none;
            font-size: 16px;
            background: white;
            flex-grow: 1;
            transition: all 0.3s;
            border-radius: 12px 0 0 12px;
        }

        .search-input:focus {
            outline: none;
            box-shadow: none;
        }

        .search-input::placeholder {
            color: #aab7c4;
            font-weight: 400;
        }

        .search-button {
            padding: 0 25px;
            background-color: #276BE4;
            border: none;
            color: white;
            font-weight: 500;
            font-size: 16px;
            border-radius: 0 12px 12px 0;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            justify-content: center;
        }

        .search-button:hover {
            background-color: #1854c4;
            box-shadow: 0 4px 10px rgba(39, 107, 228, 0.3);
        }

        .search-tags {
            display: flex;
            align-items: center;
            gap: 10px;
            padding: 5px 0;
            font-size: 14px;
            color: rgba(255, 255, 255, 0.8);
        }

        .search-tag {
            color: rgba(255, 255, 255, 0.9);
            text-decoration: none;
            padding: 3px 12px;
            border-radius: 20px;
            background-color: rgba(255, 255, 255, 0.15);
            transition: all 0.2s ease;
            font-size: 13px;
            font-weight: 500;
        }

        .search-tag:hover {
            background-color: rgba(255, 255, 255, 0.25);
            color: white;
            transform: translateY(-1px);
        }

        .search-form {
            position: relative;
            z-index: 10;
            margin: 1.5rem 0;
        }

        @media (max-width: 768px) {
            .hero-search-container {
                width: 90%;
                max-width: 450px;
            }
            
            .search-input-group {
                display: flex;
                flex-direction: row;
                border-radius: 10px;
                overflow: hidden;
            }
            
            .search-input {
                border-radius: 10px 0 0 10px;
                font-size: 14px;
                padding: 14px 15px;
                width: 70%;
            }
            
            .search-button {
                border-radius: 0 10px 10px 0;
                padding: 0 15px;
                font-size: 14px;
                width: 30%;
                white-space: nowrap;
            }
            
            .search-tags {
                flex-wrap: wrap;
                justify-content: center;
                margin-top: 8px;
                gap: 8px;
            }
            
            .search-tag {
                padding: 2px 10px;
                font-size: 12px;
            }
        }
        
        .floating-icon.light {
            color: rgba(255, 255, 255, 0.2); /* Much lighter */
            text-shadow: 0 0 15px rgba(255, 255, 255, 0.15);
        }
        
        .floating-icon.accent {
            color: rgba(255, 169, 21, 0.15); /* Much lighter */
            text-shadow: 0 0 10px rgba(255, 169, 21, 0.1);
        }
        
        .floating-icon.primary {
            color: rgba(39, 107, 228, 0.12); /* Much lighter */
            text-shadow: 0 0 5px rgba(39, 107, 228, 0.05);
        }
        
        .hero-icon-container {
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            overflow: hidden;
            z-index: 1;
        }
        
        /* Footer styling */
        footer {
            position: relative;
        }
        
        footer::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 4px;
            background: linear-gradient(to right, #276BE4, #FFA915);
        }
        
        footer h5 {
            position: relative;
        }
        
        footer h5::after {
            content: '';
            position: absolute;
            left: 0;
            bottom: 0;
            width: 40px;
            height: 2px;
            background-color: #FFA915;
        }
        
        footer a {
            transition: all 0.3s ease;
        }
        
        footer a:hover {
            color: #276BE4 !important;
            padding-left: 5px;
        }
        
        footer .social-icons a {
            display: inline-block;
            width: 36px;
            height: 36px;
            background-color: rgba(39, 107, 228, 0.1);
            text-align: center;
            line-height: 36px;
            border-radius: 50%;
            transition: all 0.3s ease;
        }
        
        footer .social-icons a:hover {
            background-color: #276BE4;
            color: white !important;
            transform: translateY(-3px);
            padding-left: 0;
        }
        
        /* Category cards */
        .category-card {
            transition: all 0.3s ease;
            border-radius: 12px;
            overflow: hidden;
        }
        
        .category-card:hover {
            transform: translateY(-7px);
            box-shadow: 0 10px 20px rgba(0, 0, 0, 0.1) !important;
        }
        
        .category-icon {
            display: inline-block;
            background: rgba(39, 107, 228, 0.1);
            width: 60px;
            height: 60px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            margin-bottom: 1rem;
        }
        
        /* Removed old navbar styling as we're using the include now */
        
        /* Modern clean hero section - Minimalist design */
        .hero {
            background: #ffffff;
            color: #333;
            padding: 80px 0 70px;
            position: relative;
            overflow: hidden;
            margin-bottom: 1.5rem;
            border-bottom: 1px solid rgba(0,0,0,0.02);
        }
        
        .hero::after {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            width: 100px;
            height: 3px;
            background: #276BE4;
            border-radius: 0 0 2px 0;
        }
        
        .hero-overlay {
            position: absolute;
            top: 0;
            right: 0;
            height: 100%;
            width: 40%;
            background: linear-gradient(135deg, rgba(39, 107, 228, 0.01) 0%, rgba(39, 107, 228, 0.03) 100%);
            clip-path: polygon(25% 0, 100% 0, 100% 100%, 5% 100%);
            z-index: 1;
        }
        
        .hero-content {
            position: relative;
            z-index: 3;
        }
        
        .hero-left {
            position: relative;
            z-index: 4;
        }
        
        .hero h1 {
            font-weight: 700;
            margin-bottom: 1.2rem;
            font-size: 2.6rem;
            color: #333;
            line-height: 1.2;
            letter-spacing: -0.5px;
        }
        
        .hero h1 span {
            color: #276BE4;
            position: relative;
            display: inline-block;
        }
        
        .hero p {
            font-size: 1.1rem;
            margin-bottom: 2rem;
            max-width: 520px;
            color: #666;
            line-height: 1.6;
            font-weight: 400;
            opacity: 0.9;
        }
        
        .btn-hero {
            padding: 12px 24px;
            font-weight: 500;
            border-radius: 8px;
            letter-spacing: 0.3px;
            transition: all 0.3s ease;
            margin: 5px 10px 5px 0;
            position: relative;
            z-index: 1;
            box-shadow: 0 4px 10px rgba(39, 107, 228, 0.05);
        }
        
        .btn-hero::after {
            content: '';
            position: absolute;
            bottom: 0;
            left: 0;
            width: 100%;
            height: 0;
            background: rgba(0,0,0,0.1);
            z-index: -1;
            transition: all 0.4s ease;
        }
        
        .btn-hero:hover::after {
            height: 100%;
        }
        
        .btn-hero.btn-primary {
            background-color: #276BE4;
            border-color: #276BE4;
            box-shadow: 0 4px 10px rgba(39, 107, 228, 0.15);
        }
        
        .btn-hero.btn-secondary {
            background-color: #FFA915;
            border-color: #FFA915;
            box-shadow: 0 4px 10px rgba(255, 169, 21, 0.3);
            box-shadow: 0 4px 15px rgba(255, 169, 21, 0.3);
        }
        
        .btn-hero.btn-outline {
            background-color: transparent;
            border: 1px solid #276BE4;
            color: #276BE4;
        }
        
        .btn-hero:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 15px rgba(39, 107, 228, 0.12);
        }
        
        /* Modern search input styles */
        .search-container {
            margin-bottom: 1.5rem;
            max-width: 520px;
        }
        
        .hero-search-form {
            width: 100%;
        }
        
        .search-input {
            height: 46px;
            border-radius: 8px 0 0 8px;
            border: 1px solid rgba(39, 107, 228, 0.15);
            box-shadow: 0 2px 6px rgba(0, 0, 0, 0.03);
            padding-left: 15px;
            font-size: 0.95rem;
            background-color: #ffffff;
            transition: all 0.3s ease;
        }
        
        .search-input:focus {
            border-color: rgba(39, 107, 228, 0.3);
            box-shadow: 0 2px 10px rgba(39, 107, 228, 0.08);
            outline: none;
        }
        
        .btn-search {
            background-color: #276BE4;
            border-color: #276BE4;
            color: white;
            border-radius: 0 8px 8px 0;
            padding: 0 15px;
            height: 46px;
            transition: all 0.3s ease;
        }
        
        .btn-search:hover {
            background-color: #1c5acd;
            box-shadow: 0 4px 10px rgba(39, 107, 228, 0.15);
        }
        
        .hero-buttons {
            display: flex;
            flex-wrap: wrap;
        }
        
        .hero-illustration {
            position: relative;
            height: 100%;
            display: flex;
            align-items: center;
            justify-content: center;
        }
        
        .hero-image {
            position: relative;
            z-index: 2;
            max-width: 100%;
            max-height: 300px;
            filter: drop-shadow(0 8px 15px rgba(0, 0, 0, 0.05));
            animation: gentleFloat 10s ease-in-out infinite;
            transform-origin: center center;
            margin: 0 auto;
            display: block;
        }
        
        @keyframes gentleFloat {
            0% { transform: translateY(0px); }
            50% { transform: translateY(-8px); }
            100% { transform: translateY(0px); }
        }
        
        /* Video card enhancements */
        .video-card {
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.05);
            border-radius: 12px;
            overflow: hidden;
            transition: all 0.3s ease;
            height: 100%;
            position: relative;
            background-color: #fff;
        }
        
        .video-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 15px 30px rgba(39, 107, 228, 0.15);
        }
        
        .video-thumbnail {
            position: relative;
            overflow: hidden;
        }
        
        .svg-bg {
            width: 100%;
            aspect-ratio: 16 / 9;
            display: flex;
            align-items: center;
            justify-content: center;
            background: linear-gradient(45deg, #f8fafc, #f1f5f9);
            transition: all 0.5s ease;
        }
        
        .svg-icon-container {
            width: 80px;
            height: 80px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            transition: transform 0.5s ease;
        }
        
        .svg-icon-container i {
            font-size: 40px;
            filter: drop-shadow(0 4px 6px rgba(0,0,0,0.1));
        }
        
        .video-card:hover .svg-icon-container {
            transform: scale(1.2) rotate(5deg);
        }
        
        .play-overlay {
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0, 0, 0, 0.3);
            display: flex;
            align-items: center;
            justify-content: center;
            opacity: 0;
            transition: opacity 0.3s ease;
        }
        
        .play-overlay i {
            text-shadow: 0 2px 10px rgba(0, 0, 0, 0.3);
            transform: scale(0.8);
            transition: transform 0.3s ease;
        }
        
        .video-card:hover .play-overlay {
            opacity: 1;
        }
        
        .video-card:hover .play-overlay i {
            transform: scale(1);
        }
        
        .video-info {
            padding: 15px;
            position: relative;
        }
        
        .video-title {
            font-weight: 600;
            margin-bottom: 0.5rem;
            font-size: 1.1rem;
            line-height: 1.4;
            color: #333;
        }
        
        .video-meta {
            font-size: 0.85rem;
            color: #6c757d;
        }
        
        .video-duration {
            border-radius: 20px;
            font-size: 0.75rem;
            font-weight: 500;
            z-index: 2;
        }
        
        /* Section headings */
        .section-heading {
            position: relative;
            margin-bottom: 2.5rem;
            font-weight: 700;
        }
        
        .section-heading::after {
            content: '';
            position: absolute;
            left: 0;
            bottom: -12px;
            width: 50px;
            height: 4px;
            background-color: #FFA915;
            border-radius: 2px;
        }
        
        /* Become teacher section */
        .become-teacher {
            background-color: #f8f9fa;
            border-radius: 15px;
            padding: 3rem 0;
            margin: 3rem 0;
            position: relative;
            overflow: hidden;
        }
        
        /* Custom animations */
        .fade-in-up {
            animation: fadeInUp 1s ease-out;
        }
        
        .fade-in {
            animation: fadeIn 1.2s ease-out;
        }
        
        /* Minimal Floating Animation Effects */
        .floating-element {
            position: absolute;
            border-radius: 50%;
            background: rgba(255,255,255,0.9);
            border: 1px solid rgba(39, 107, 228, 0.08);
            pointer-events: none;
            z-index: 2;
        }
        
        @keyframes floatCircle {
            0% { transform: translate(0, 0); }
            50% { transform: translate(0, -10px); }
            100% { transform: translate(0, 0); }
        }
        
        @keyframes pulse {
            0% { opacity: 0.4; transform: scale(1); }
            50% { opacity: 0.7; transform: scale(1.03); }
            100% { opacity: 0.4; transform: scale(1); }
        }
    </style>
</head>
<body>
    <!-- Floating Icons -->
    <div class="floating-icon" style="top: 15%; left: 10%; font-size: 50px;"><i class="fas fa-graduation-cap"></i></div>
    <div class="floating-icon" style="top: 25%; right: 8%; font-size: 45px;"><i class="fas fa-book"></i></div>
    <div class="floating-icon" style="bottom: 35%; left: 12%; font-size: 40px;"><i class="fas fa-laptop-code"></i></div>
    <div class="floating-icon" style="bottom: 25%; right: 10%; font-size: 35px;"><i class="fas fa-brain"></i></div>
    
    <!-- Include the modern navbar -->
    <?php include 'includes/navbar.php'; ?>
    
    <!-- Clean Hero Section -->
    <section class="modern-hero">
        <!-- Subtle Background Icons -->
        <div class="floating-icons">
            <div class="floating-icon-card icon-1">
                <i class="fas fa-graduation-cap"></i>
            </div>
            <div class="floating-icon-card icon-2">
                <i class="fas fa-certificate"></i>
            </div>
            <div class="floating-icon-card icon-3">
                <i class="fas fa-laptop-code"></i>
            </div>
            <div class="floating-icon-card icon-4">
                <i class="fas fa-chart-line"></i>
            </div>
            <div class="floating-icon-card icon-5">
                <i class="fas fa-brain"></i>
            </div>
        </div>
        
        <div class="container">
            <div class="row align-items-center min-vh-80">
                <!-- Left Content -->
                <div class="col-lg-6">
                    <div class="hero-content-modern">
                        <div class="hero-main-card">
                            <h1 class="hero-title">
                                Master tomorrow's skills 
                                <span class="gradient-text">today</span>
                            </h1>
                            <p class="hero-description">
                                Power up your career and life skills with the most up-to-date, 
                                expert-led learning from Panadite Academy.
                            </p>
                            
                            <div class="hero-cta-buttons">
                                <a href="/student/courses.php" class="btn btn-modern-primary">
                                    Get started
                                </a>
                                <a href="/student/courses.php?category=trending" class="btn btn-modern-secondary">
                                    Explore trending
                                </a>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Right Visual -->
                <div class="col-lg-6">
                    <div class="hero-visual">
                        <div class="hero-image-container">
                            <img src="/landing_images/download (15).jpeg" 
                                 alt="Learning and Education" 
                                 class="hero-image"
                                 onerror="this.src='/landing_images/images (3).jpeg'">
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </section>

    <!-- Featured Videos Section -->
    <section class="container py-5">
        <div class="d-flex justify-content-between align-items-center mb-4">
            <h2 class="section-heading animate__animated animate__fadeIn">Featured Videos</h2>
            <a href="/videos.php" class="btn btn-outline-primary animate__animated animate__fadeIn">View All</a>
        </div>
        
        <div class="row">
            <?php if (empty($featuredVideos)): ?>
                <div class="col-12 text-center py-5">
                    <p class="lead">No videos available yet. Check back soon!</p>
                    <?php if (!$currentUser || $currentUser['user_role'] === 'teacher'): ?>
                        <a href="/auth/register.php?role=teacher" class="btn btn-primary mt-3">Become a Facilitator</a>
                    <?php endif; ?>
                </div>
            <?php else: ?>
                <?php foreach ($featuredVideos as $video): ?>
                    <div class="col-md-6 col-lg-3 mb-4 animate__animated animate__fadeIn">
                        <div class="video-card shadow-lg rounded-4 overflow-hidden">
                            <div class="video-thumbnail position-relative">
                                <?php
                                // AGGRESSIVE VIDEO THUMBNAIL PATH HANDLING
                                $thumbnail = $video['thumbnail'] ?? '';
                                
                                // Clean up path
                                if (strpos($thumbnail, '../') === 0) {
                                    $thumbnail = substr($thumbnail, 3);
                                }
                                if (substr($thumbnail, 0, 1) !== '/') {
                                    $thumbnail = '/' . $thumbnail;
                                }
                                $thumbnail = str_replace('//', '/', $thumbnail);
                                
                                // Ensure it's valid or use default
                                if (empty($thumbnail) || $thumbnail == '/') {
                                    // Use absolute path to placeholder
                                    $thumbnail = '/assets/images/default-thumbnail.jpg';
                                }
                                
                                // Add cache buster
                                $img_src = $thumbnail . '?' . time();
                                ?>
                                
                                <!-- ACTUAL VIDEO THUMBNAIL IMAGE -->
                                <img 
                                    src="<?php echo htmlspecialchars($img_src); ?>" 
                                    class="video-thumbnail-img"
                                    alt="<?php echo htmlspecialchars($video['title'] ?? 'Video'); ?>"
                                    onerror="this.onerror=null; this.src='/assets/images/default-thumbnail.jpg';"
                                    style="width: 100% !important; 
                                          height: 100% !important; 
                                          object-fit: cover !important; 
                                          display: block !important; 
                                          position: absolute !important;
                                          top: 0 !important;
                                          left: 0 !important;
                                          z-index: 5 !important;
                                          visibility: visible !important;"
                                >
                                <div class="video-duration position-absolute bottom-0 end-0 bg-dark text-white small py-1 px-2 m-2 rounded-pill">
                                    <i class="far fa-clock me-1"></i> <?php echo !empty($video['duration']) ? formatDuration($video['duration']) : '10:00'; ?>
                                </div>
                                <?php
                                // PRICE DISPLAY LOGIC
                                if (isset($video['is_free']) && $video['is_free']): 
                                ?>
                                <span class="badge bg-success position-absolute top-0 start-0 m-2 px-2 py-1" style="z-index: 10 !important; font-weight: bold !important; font-size: 0.8rem !important;">Free</span>
                                <?php 
                                else:
                                    // If video price is zero/null but it's part of a paid course, show the course price
                                    $displayPrice = 0;
                                    if ((!isset($video['price']) || $video['price'] <= 0) && isset($video['course_price'])): 
                                        $displayPrice = $video['course_price'];
                                    elseif (isset($video['price']) && $video['price'] > 0):
                                        $displayPrice = $video['price'];
                                    endif;
                                ?>
                                <span class="badge bg-primary position-absolute top-0 start-0 m-2 px-2 py-1" style="z-index: 10 !important; font-weight: bold !important; font-size: 0.8rem !important;">R<?php echo number_format($displayPrice, 2); ?></span>
                                <?php endif; ?>
                                <div class="play-overlay">
                                    <i class="fas fa-play-circle fa-3x text-white"></i>
                                </div>
                            </div>
                            <div class="video-info p-3">
                                <h5 class="video-title fw-bold mb-2 text-truncate"><?php echo $video['title']; ?></h5>
                                <div class="video-meta d-flex flex-column gap-1 mb-2">
                                    <div class="d-flex align-items-center">
                                        <i class="fas fa-graduation-cap me-2 text-primary"></i>
                                        <span class="small text-truncate"><?php echo $video['course_title']; ?></span>
                                    </div>
                                    <div class="d-flex align-items-center">
                                        <i class="fas fa-user me-2 text-primary"></i>
                                        <span class="small text-truncate"><?php echo $video['first_name'] . ' ' . $video['last_name']; ?></span>
                                    </div>
                                </div>
                                <div class="d-flex justify-content-between align-items-center mt-3">
                                    <span class="small text-muted">
                                        <i class="fas fa-eye me-1"></i> <?php echo number_format($video['views'] ?? 0); ?>
                                    </span>
                                    <a href="/video.php?id=<?php echo $video['video_id']; ?>" class="btn btn-sm btn-primary rounded-pill px-3">
                                        <i class="fas fa-play-circle me-1"></i> Watch Now
                                    </a>
                                </div>
                            </div>
                        </div>
                    </div>
                <?php endforeach; ?>
            <?php endif; ?>
        </div>
    </section>

    <!-- Featured Courses Section -->
    <section class="container py-5">
        <div class="d-flex justify-content-between align-items-center mb-4">
            <h2 class="section-heading animate__animated animate__fadeIn">Featured Courses</h2>
            <a href="/courses-list.php" class="btn btn-outline-primary animate__animated animate__fadeIn">View All</a>
        </div>
        
        <div class="row">
            <?php if (empty($featuredCourses)): ?>
                <div class="col-12">
                    <div class="alert alert-info">
                        <p class="mb-0">No courses available yet.</p>
                        <pre style="font-size:10px;"><?php print_r($conn->error ?? 'No DB error'); ?></pre>
                        <?php if (!empty($featuredCourses)) echo '<p>But we have courses in array!</p>'; ?>
                    </div>
                </div>
            <?php else: ?>
                <?php foreach ($featuredCourses as $course): ?>
                <div class="col-md-6 col-lg-3 mb-4">
                    <!-- CARD START - FULLY REBUILT -->
                    <div class="card" style="border-radius: 15px; box-shadow: 0 5px 15px rgba(0,0,0,0.1); overflow: hidden;">
                        <!-- THUMBNAIL CONTAINER -->
                        <div style="position: relative; height: 180px; overflow: hidden; background-color: #f1f1f1;">
                            <?php
                            // SUPER AGGRESSIVE THUMBNAIL PATH HANDLING
                            $thumbnail = $course['thumbnail'] ?? '';
                            
                            // Clean up path
                            if (strpos($thumbnail, '../') === 0) {
                                $thumbnail = substr($thumbnail, 3);
                            }
                            if (substr($thumbnail, 0, 1) !== '/') {
                                $thumbnail = '/' . $thumbnail;
                            }
                            $thumbnail = str_replace('//', '/', $thumbnail);
                            
                            // Ensure it's valid or use default
                            if (empty($thumbnail) || $thumbnail == '/' || strpos($thumbnail, '/uploads/') === false) {
                                // Use absolute path to placeholder
                                $thumbnail = '/assets/images/default-thumbnail.jpg';
                            }
                            
                            // Add cache buster
                            $img_src = $thumbnail . '?' . time();
                            ?>
                            
                            <!-- ACTUAL IMAGE TAG (NO SVG ALTERNATIVE) -->
                            <img 
                                src="<?php echo htmlspecialchars($img_src); ?>" 
                                class="course-thumbnail"
                                alt="<?php echo htmlspecialchars($course['title'] ?? 'Course'); ?>"
                                onerror="this.onerror=null; this.src='/assets/images/default-thumbnail.jpg';"
                                style="width: 100% !important; 
                                      height: 100% !important; 
                                      object-fit: cover !important; 
                                      display: block !important; 
                                      position: absolute !important;
                                      top: 0 !important;
                                      left: 0 !important;
                                      z-index: 5 !important;
                                      visibility: visible !important;"
                            >
                            
                            <!-- PRICE BADGE -->
                            <?php if (isset($course['price']) && $course['price'] == 0): ?>
                                <span style="position: absolute; top: 10px; left: 10px; z-index: 10; background-color: #28a745; color: white; padding: 5px 10px; border-radius: 4px; font-weight: bold;">Free</span>
                            <?php elseif (isset($course['price'])): ?>
                                <span style="position: absolute; top: 10px; left: 10px; z-index: 10; background-color: #007bff; color: white; padding: 5px 10px; border-radius: 4px; font-weight: bold;">R<?php echo number_format($course['price'], 2); ?></span>
                            <?php endif; ?>
                        </div>
                        
                        <!-- COURSE INFO -->
                        <div style="padding: 15px;">
                            <h5 style="font-weight: 700; margin-bottom: 10px; overflow: hidden; text-overflow: ellipsis; white-space: nowrap;">
                                <?php echo htmlspecialchars($course['title'] ?? 'Unnamed Course'); ?>
                            </h5>
                            
                            <!-- CATEGORY & STUDENTS -->
                            <div style="display: flex; align-items: center; margin-bottom: 10px;">
                                <span style="background-color: #f1f1f1; color: #555; padding: 3px 8px; border-radius: 4px; margin-right: 10px; font-size: 12px;">
                                    <i class="fas fa-bookmark" style="margin-right: 5px;"></i>
                                    <?php echo htmlspecialchars($course['category_name'] ?? 'General'); ?>
                                </span>
                                
                                <span style="color: #777; font-size: 12px;">
                                    <i class="fas fa-users" style="margin-right: 5px;"></i>
                                    <?php echo intval($course['student_count'] ?? 0); ?> student<?php echo (intval($course['student_count'] ?? 0) != 1) ? 's' : ''; ?>
                                </span>
                            </div>
                            
                            <!-- INSTRUCTOR & VIEW BUTTON -->
                            <div style="display: flex; justify-content: space-between; align-items: center;">
                                <div style="font-size: 13px; color: #555;">
                                    <i class="fas fa-user-circle" style="margin-right: 5px;"></i>
                                    <?php 
                                    $instructorName = trim(($course['first_name'] ?? '') . ' ' . ($course['last_name'] ?? ''));
                                    echo !empty($instructorName) ? htmlspecialchars($instructorName) : 'Instructor';
                                    ?>
                                </div>
                                
                                <a href="course.php?id=<?php echo $course['course_id'] ?? 0; ?>" 
                                   style="text-decoration: none; background-color: #007bff; color: white; padding: 5px 15px; border-radius: 4px; font-size: 13px;">
                                    View
                                </a>
                            </div>
                        </div>
                    </div>
                    <!-- CARD END -->
                </div>
                <?php endforeach; ?>
            <?php endif; ?>
        </div>
    </section>

    <!-- Categories Section -->
    <section class="py-5 bg-light">
        <div class="container">
            <h2 class="section-heading animate__animated animate__fadeIn">Courses Categories</h2>
            <div class="row" id="categoriesList">
                <!-- End-User Computing & Office Productivity -->
                <div class="col-md-6 col-lg-4 mb-4 animate__animated animate__fadeIn">
                    <a href="/category.php?category=office-productivity" class="text-decoration-none">
                        <div class="card category-card h-100 border-0 shadow-lg rounded-4 overflow-hidden">
                            <div class="card-body p-4">
                                <div class="category-icon mb-3">
                                    <i class="fas fa-desktop text-primary" style="font-size: 2rem;"></i>
                                </div>
                                <h5 class="card-title fw-bold mb-3">End-User Computing & Office Productivity</h5>
                                <p class="card-text text-muted small">
                                    <strong>• Basic-Advanced Computer and MS Office 365</strong><br>
                                    Covers: Computer, Excel, Word, PowerPoint, Outlook<br>
                                    <strong>• Data Capturing</strong><br>
                                    Covers: Intermediate End-User Computing, Data Management, Typing
                                </p>
                                <div class="d-flex justify-content-between align-items-center mt-3">
                                    <span class="badge rounded-pill bg-primary px-3 py-2">2 courses</span>
                                    <i class="fas fa-chevron-circle-right text-primary fs-5"></i>
                                </div>
                            </div>
                        </div>
                    </a>
                </div>
                
                <!-- Programming & Software Development -->
                <div class="col-md-6 col-lg-4 mb-4 animate__animated animate__fadeIn">
                    <a href="/category.php?category=programming" class="text-decoration-none">
                        <div class="card category-card h-100 border-0 shadow-lg rounded-4 overflow-hidden">
                            <div class="card-body p-4">
                                <div class="category-icon mb-3">
                                    <i class="fas fa-code text-primary" style="font-size: 2rem;"></i>
                                </div>
                                <h5 class="card-title fw-bold mb-3">Programming & Software Development</h5>
                                <p class="card-text text-muted small">
                                    <strong>• Python Programming</strong><br>
                                    Covers: Python Programming, Programming Principles, REST API, GUI in Python<br>
                                    <strong>• IT Software Development</strong><br>
                                    Covers: Programming, Database Management, OOP Design, Web Graphic Design
                                </p>
                                <div class="d-flex justify-content-between align-items-center mt-3">
                                    <span class="badge rounded-pill bg-primary px-3 py-2">2 courses</span>
                                    <i class="fas fa-chevron-circle-right text-primary fs-5"></i>
                                </div>
                            </div>
                        </div>
                    </a>
                </div>
                
                <!-- Web & Front-End Development -->
                <div class="col-md-6 col-lg-4 mb-4 animate__animated animate__fadeIn">
                    <a href="/category.php?category=web-development" class="text-decoration-none">
                        <div class="card category-card h-100 border-0 shadow-lg rounded-4 overflow-hidden">
                            <div class="card-body p-4">
                                <div class="category-icon mb-3">
                                    <i class="fas fa-laptop-code text-primary" style="font-size: 2rem;"></i>
                                </div>
                                <h5 class="card-title fw-bold mb-3">Web & Front-End Development</h5>
                                <p class="card-text text-muted small">
                                    <strong>• Front-End Web Development</strong><br>
                                    Covers: HTML, CSS, Bootstrap, JavaScript
                                </p>
                                <div class="d-flex justify-content-between align-items-center mt-3">
                                    <span class="badge rounded-pill bg-primary px-3 py-2">1 course</span>
                                    <i class="fas fa-chevron-circle-right text-primary fs-5"></i>
                                </div>
                            </div>
                        </div>
                    </a>
                </div>
                
                <!-- Data Analysis & Business Intelligence -->
                <div class="col-md-6 col-lg-4 mb-4 animate__animated animate__fadeIn">
                    <a href="/category.php?category=data-analysis" class="text-decoration-none">
                        <div class="card category-card h-100 border-0 shadow-lg rounded-4 overflow-hidden">
                            <div class="card-body p-4">
                                <div class="category-icon mb-3">
                                    <i class="fas fa-chart-bar text-primary" style="font-size: 2rem;"></i>
                                </div>
                                <h5 class="card-title fw-bold mb-3">Data Analysis & Business Intelligence</h5>
                                <p class="card-text text-muted small">
                                    <strong>• Power BI Analytics</strong><br>
                                    Covers: Advanced Excel, Power Query, DAX, Dashboards<br>
                                    <strong>• Data Analytics</strong><br>
                                    Covers: Advanced End-User Computing, Power BI, Intro to Python Programming
                                </p>
                                <div class="d-flex justify-content-between align-items-center mt-3">
                                    <span class="badge rounded-pill bg-primary px-3 py-2">2 courses</span>
                                    <i class="fas fa-chevron-circle-right text-primary fs-5"></i>
                                </div>
                            </div>
                        </div>
                    </a>
                </div>
                
                <!-- IT Support & Infrastructure -->
                <div class="col-md-6 col-lg-4 mb-4 animate__animated animate__fadeIn">
                    <a href="/category.php?category=it-support" class="text-decoration-none">
                        <div class="card category-card h-100 border-0 shadow-lg rounded-4 overflow-hidden">
                            <div class="card-body p-4">
                                <div class="category-icon mb-3">
                                    <i class="fas fa-network-wired text-primary" style="font-size: 2rem;"></i>
                                </div>
                                <h5 class="card-title fw-bold mb-3">IT Support & Infrastructure</h5>
                                <p class="card-text text-muted small">
                                    <strong>• IT Technical Support</strong><br>
                                    Covers: Maths Literacy, PC Hardware, Infrastructure Support, Network Support
                                </p>
                                <div class="d-flex justify-content-between align-items-center mt-3">
                                    <span class="badge rounded-pill bg-primary px-3 py-2">1 course</span>
                                    <i class="fas fa-chevron-circle-right text-primary fs-5"></i>
                                </div>
                            </div>
                        </div>
                    </a>
                </div>
            </div>
        </div>
    </section>

    <!-- Become a Teacher CTA -->
    <section class="teacher-section position-relative overflow-hidden py-5">
        <!-- Modern background elements -->
        <div class="position-absolute top-0 start-0 w-100 h-100" style="background: linear-gradient(135deg, rgba(248,249,250,1) 0%, rgba(255,255,255,1) 100%); z-index: -2;"></div>
        <div class="position-absolute top-0 end-0 h-100" style="width: 40%; background: linear-gradient(135deg, rgba(39,107,228,0.03) 0%, rgba(39,107,228,0.08) 100%); clip-path: polygon(30% 0, 100% 0, 100% 100%, 0% 100%); z-index: -1;"></div>
        
        <!-- Floating education icons with new styling -->
        <div class="position-absolute top-0 start-0 w-100 h-100" style="overflow: hidden; z-index: 0;">
            <!-- Light floating icons -->
            <div class="floating-icon light" style="top: 20%; left: 5%; font-size: 22px;"><i class="fas fa-chart-line"></i></div>
            <div class="floating-icon light" style="bottom: 15%; right: 10%; font-size: 18px;"><i class="fas fa-school"></i></div>
            
            <!-- Accent floating icons -->
            <div class="floating-icon accent" style="top: 30%; right: 15%; font-size: 26px;"><i class="fas fa-chalkboard"></i></div>
            <div class="floating-icon accent" style="bottom: 25%; left: 12%; font-size: 24px;"><i class="fas fa-user-graduate"></i></div>
        </div>
        
        <div class="container py-4">
            <div class="row align-items-center">
                <div class="col-lg-6 mb-5 mb-lg-0">
                    <div class="pe-lg-4">
                        <!-- Section heading with animated underline -->
                        <h2 class="display-4 fw-bold mb-2 text-primary animate__animated animate__fadeIn">Share Your <span style="color: #FFA915;">Knowledge</span></h2>
                        <div class="animate__animated animate__fadeIn animate__delay-1s">
                            <div class="accent-bar mb-4" style="width: 80px; height: 4px; background: linear-gradient(90deg, #276BE4, #FFA915); border-radius: 2px;"></div>
                            <p class="lead text-dark mb-4">Become a facilitator at Panadite Academy and help students get ahead of the curve</p>
                            
                            <!-- Feature cards -->
                            <div class="mb-4">
                                <div class="d-flex align-items-center mb-3 p-3 rounded-3 animate__animated animate__fadeInUp animate__delay-1s" 
                                     style="background-color: rgba(255,255,255,0.8); box-shadow: 0 3px 10px rgba(0,0,0,0.05);">
                                    <div class="me-3 rounded-circle p-3" style="background-color: rgba(39,107,228,0.1);">
                                        <i class="fas fa-video text-primary"></i>
                                    </div>
                                    <div>
                                        <h5 class="mb-1 fw-bold">Create & Upload Courses</h5>
                                        <p class="text-muted mb-0 small">Share your expertise with quality educational content</p>
                                    </div>
                                </div>
                                
                                <div class="d-flex align-items-center mb-3 p-3 rounded-3 animate__animated animate__fadeInUp animate__delay-2s"
                                     style="background-color: rgba(255,255,255,0.8); box-shadow: 0 3px 10px rgba(0,0,0,0.05);">
                                    <div class="me-3 rounded-circle p-3" style="background-color: rgba(255,169,21,0.1);">
                                        <i class="fas fa-coins text-warning"></i>
                                    </div>
                                    <div>
                                        <h5 class="mb-1 fw-bold">Earn Revenue</h5>
                                        <p class="text-muted mb-0 small">Get paid for referrals</p>
                                    </div>
                                </div>
                            </div>
                            
                            <!-- CTA Button with pulsing animation -->
                            <a href="/auth/register.php?role=teacher" class="btn btn-lg btn-primary animate__animated animate__pulse animate__infinite animate__slow"
                               style="border-radius: 50px; padding: 12px 30px; box-shadow: 0 5px 15px rgba(39,107,228,0.3);">
                                <i class="fas fa-chalkboard-teacher me-2"></i> Become a Facilitator
                            </a>
                        </div>
                    </div>
                </div>
                
                <!-- Visual section with SVG illustration -->
                <div class="col-lg-6 animate__animated animate__fadeIn animate__delay-1s">
                    <div class="position-relative" style="height: 400px;">
                        <!-- Logo background circle -->
                        <div class="position-absolute start-50 translate-middle-x" style="bottom: 0; width: 120px; height: 120px; background: linear-gradient(135deg, #276BE4, #1a4ba8); border-radius: 50%; z-index: 1;"></div>
                        
                        <!-- Main illustration -->
                        <div class="position-absolute top-0 start-50 translate-middle-x" style="width: 100%; max-width: 400px; z-index: 2;">
                            <img src="/assets/images/teacher-illustration.svg" alt="Teacher" class="img-fluid" style="animation: float 6s ease-in-out infinite;">
                        </div>
                        
                        <!-- Panadite logo at bottom -->
                        <div class="position-absolute start-50 translate-middle-x" style="bottom: 0; width: 100px; height: 100px; z-index: 3;">
                            <img src="/assets/images/logo.jpeg" alt="Panadite Logo" class="img-fluid" style="border-radius: 50%;">
                        </div>
                        
                        <!-- Tagline -->
                        <div class="position-absolute start-50 translate-middle-x" style="bottom: -30px; z-index: 4; width: 80%;">
                            <div class="py-2 px-4 rounded-pill text-center" style="background: linear-gradient(90deg, #276BE4, #FFA915); box-shadow: 0 3px 10px rgba(0,0,0,0.1);">
                                <span class="text-white fw-bold">Get ahead of the curve</span>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </section>

<!-- Footer -->
<footer class="mt-5 position-relative">
        <!-- Top gradient bar -->
        <div class="position-absolute top-0 start-0 w-100" style="height: 4px; background: linear-gradient(90deg, #276BE4, #FFA915);"></div>
        
        <!-- Main footer content -->
        <div class="bg-white py-5">
            <div class="container">
                <div class="row gy-4">
                    <!-- Logo and description -->
                    <div class="col-lg-4 col-md-6 mb-4">
                        <div class="mb-4">
                            <img src="/assets/images/logo.jpeg" alt="Panadite Academy" style="height: 50px;">
                        </div>
                        <p class="mb-2" style="color: #333; font-weight: 500;">Empowering learners through high-quality educational content.</p>
                        <p style="color: #333;">Get ahead of the curve with our premium videos and courses.</p>
                    </div>
                    
                    <!-- Quick links -->
                    <div class="col-lg-2 col-md-6 mb-4">
                        <h5 class="fw-bold mb-3 pb-2 position-relative" style="color: #222;">
                            Quick Links
                            <span class="position-absolute bottom-0 start-0" style="height: 2px; width: 40px; background: #FFA915;"></span>
                        </h5>
                        <ul class="list-unstyled">
                            <li class="mb-2"><a href="/index.php" class="text-decoration-none" style="color: #333;"><i class="fas fa-angle-right me-2" style="color: #276BE4;"></i>Home</a></li>
                            <li class="mb-2"><a href="/courses.php" class="text-decoration-none" style="color: #333;"><i class="fas fa-angle-right me-2" style="color: #276BE4;"></i>Courses</a></li>
                            <li class="mb-2"><a href="/videos.php" class="text-decoration-none" style="color: #333;"><i class="fas fa-angle-right me-2" style="color: #276BE4;"></i>Videos</a></li>
                            <li class="mb-2"><a href="/about.php" class="text-decoration-none" style="color: #333;"><i class="fas fa-angle-right me-2" style="color: #276BE4;"></i>About Us</a></li>
                            <li class="mb-2"><a href="/contact.php" class="text-decoration-none" style="color: #333;"><i class="fas fa-angle-right me-2" style="color: #276BE4;"></i>Contact</a></li>
                        </ul>
                    </div>
                    
                    <!-- Facilitator links -->
                    <div class="col-lg-2 col-md-6 mb-4">
                        <h5 class="fw-bold mb-3 pb-2 position-relative" style="color: #222;">
                            Facilitator
                            <span class="position-absolute bottom-0 start-0" style="height: 2px; width: 40px; background: #FFA915;"></span>
                        </h5>
                        <ul class="list-unstyled">
                            <li class="mb-2"><a href="/auth/register.php?role=teacher" class="text-decoration-none" style="color: #333;"><i class="fas fa-angle-right me-2" style="color: #276BE4;"></i>Become a Facilitator</a></li>
                            <li class="mb-2"><a href="/teacher/dashboard.php" class="text-decoration-none" style="color: #333;"><i class="fas fa-angle-right me-2" style="color: #276BE4;"></i>Teacher Dashboard</a></li>
                            <li class="mb-2"><a href="/teacher/sales.php" class="text-decoration-none" style="color: #333;"><i class="fas fa-angle-right me-2" style="color: #276BE4;"></i>Your Sales</a></li>
                            <li class="mb-2"><a href="/teacher/help.php" class="text-decoration-none" style="color: #333;"><i class="fas fa-angle-right me-2" style="color: #276BE4;"></i>Teacher Help</a></li>
                        </ul>
                    </div>
                    
                    <!-- Contact info -->
                    <div class="col-lg-4 col-md-6">
                        <h5 class="fw-bold mb-3 pb-2 position-relative" style="color: #222;">
                            Contact
                            <span class="position-absolute bottom-0 start-0" style="height: 2px; width: 40px; background: #FFA915;"></span>
                        </h5>
                        <ul class="list-unstyled mb-4">
                            <li class="mb-3 d-flex align-items-center">
                                <span class="d-flex align-items-center justify-content-center me-3" style="width: 36px; height: 36px; background: rgba(39,107,228,0.1); border-radius: 50%;">
                                    <i class="fas fa-map-marker-alt" style="color: #276BE4;"></i>
                                </span>
                                <span style="color: #333;">296 Pretorius Street, Corner Lilian Ngoyi, Pretoria Central, 0001, South Africa</span>
                            </li>
                            <li class="mb-3 d-flex align-items-center">
                                <span class="d-flex align-items-center justify-content-center me-3" style="width: 36px; height: 36px; background: rgba(39,107,228,0.1); border-radius: 50%;">
                                    <i class="fas fa-envelope" style="color: #276BE4;"></i>
                                </span>
                                <a href="mailto:admin@panaditeacademy.com" class="text-decoration-none" style="color: #333;">admin@panaditeacademy.com</a>
                            </li>
                            <li class="mb-3 d-flex align-items-center">
                                <span class="d-flex align-items-center justify-content-center me-3" style="width: 36px; height: 36px; background: rgba(39,107,228,0.1); border-radius: 50%;">
                                    <i class="fas fa-phone" style="color: #276BE4;"></i>
                                </span>
                                <a href="tel:+27747223141" class="text-decoration-none" style="color: #333;">+27 74 722 3141</a>
                            </li>
                        </ul>
                        
                        <!-- Social icons -->
                        <div class="d-flex">
                            <a href="#" class="d-flex align-items-center justify-content-center me-2" style="width: 40px; height: 40px; background: rgba(39,107,228,0.1); border-radius: 50%; color: #276BE4; transition: all 0.3s ease;">
                                <i class="fab fa-facebook-f"></i>
                            </a>
                            <a href="#" class="d-flex align-items-center justify-content-center me-2" style="width: 40px; height: 40px; background: rgba(39,107,228,0.1); border-radius: 50%; color: #276BE4; transition: all 0.3s ease;">
                                <i class="fab fa-twitter"></i>
                            </a>
                            <a href="#" class="d-flex align-items-center justify-content-center me-2" style="width: 40px; height: 40px; background: rgba(39,107,228,0.1); border-radius: 50%; color: #276BE4; transition: all 0.3s ease;">
                                <i class="fab fa-instagram"></i>
                            </a>
                            <a href="#" class="d-flex align-items-center justify-content-center" style="width: 40px; height: 40px; background: rgba(39,107,228,0.1); border-radius: 50%; color: #276BE4; transition: all 0.3s ease;">
                                <i class="fab fa-linkedin-in"></i>
                            </a>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Copyright section -->
        <div style="background: #f8f9fa; border-top: 1px solid #eee;" class="py-3">
            <div class="container">
                <div class="row align-items-center">
                    <div class="col-md-6 mb-3 mb-md-0">
                        <p class="mb-0" style="color: #333; font-weight: 500;">&copy; <?php echo date('Y'); ?> Panadite Academy. All rights reserved.</p>
                    </div>
                    <div class="col-md-6 text-md-end">
                        <a href="/privacy-policy.php" class="text-decoration-none me-3" style="color: #333;">Privacy Policy</a>
                        <a href="/terms.php" class="text-decoration-none me-3" style="color: #333;">Terms of Use</a>
                        <a href="/sitemap.php" class="text-decoration-none" style="color: #333;">Sitemap</a>
                    </div>
                </div>
            </div>
        </div>
    </footer>
    <script src="https://cdn.jsdelivr.net/npm/jquery@3.6.4/dist/jquery.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/gsap@3.11.4/dist/gsap.min.js"></script>
    <script src="/force-image.js"></script>
    <script>
        // Advanced floating icon animations
        function animateFloatingIcons() {
            $('.floating-icon').each(function(index) {
                var icon = $(this);
                var speedFactor = Math.random() * 0.5 + 0.5; // 0.5 to 1.0 speed variation
                
                // Set initial positions and scale
                var startScale = 0.9 + Math.random() * 0.2; // 0.9 to 1.1
                
                // Advanced animation properties
                var baseX = parseInt(icon.css('left'));
                var baseY = parseInt(icon.css('top'));
                if (isNaN(baseX)) baseX = parseInt(icon.css('right')) || 0;
                if (isNaN(baseY)) baseY = parseInt(icon.css('bottom')) || 0;
                
                var amplitude = 15 + Math.random() * 20; // 15-35px movement range
                var period = 5 + Math.random() * 5; // 5-10 second period
                
                // Function to create fluid floating effect using sine waves
                function animateIcon(timestamp) {
                    if (!icon.data('startTime')) {
                        icon.data('startTime', timestamp);
                    }
                    
                    // Calculate elapsed time for continuity
                    var elapsed = timestamp - icon.data('startTime');
                    
                    // Calculate position using sine/cosine waves with phase shift for each icon
                    var offsetX = amplitude * Math.sin((elapsed/1000) * speedFactor + index);
                    var offsetY = (amplitude * 0.7) * Math.cos((elapsed/1000) * speedFactor + index + 2);
                    
                    // Calculate scale oscillation (subtle)
                    var scale = startScale + 0.05 * Math.sin((elapsed/1000) * speedFactor * 0.5);
                    
                    // Calculate rotation (very subtle)
                    var rotation = 2 * Math.sin((elapsed/1000) * speedFactor * 0.3 + index);
                    
                    // Set transform with hardware acceleration
                    icon.css('transform', 
                        'translate3d(' + offsetX + 'px, ' + offsetY + 'px, 0) ' +
                        'scale(' + scale + ') ' +
                        'rotate(' + rotation + 'deg)'
                    );
                    
                    // Opacity pulsing for light variants
                    if (icon.hasClass('light')) {
                        var opacity = 0.6 + 0.2 * Math.sin((elapsed/1000) * 0.5);
                        icon.css('opacity', opacity);
                    }
                    
                    // Continue animation
                    requestAnimationFrame(animateIcon);
                }
                
                // Start with a delay based on index
                setTimeout(function() {
                    // Start the animation loop
                    requestAnimationFrame(animateIcon);
                }, index * 120);
            });
        }
        
        $(document).ready(function() {
            // Initialize the floating icon animation
            animateFloatingIcons();
            
            // Add smooth scrolling
            $("a.nav-link").on('click', function(event) {
                if (this.hash !== "") {
                    event.preventDefault();
                    var hash = this.hash;
                    $('html, body').animate({
                        scrollTop: $(hash).offset().top - 70
                    }, 800);
                }
            });
            
            // Initialize animations for elements that are in view
            function initAnimations() {
                $('.fade-in-element:not(.animated)').each(function() {
                    var elementTop = $(this).offset().top;
                    var viewportTop = $(window).scrollTop();
                    var windowHeight = $(window).height();
                    
                    if (elementTop < viewportTop + windowHeight - 100) {
                        $(this).addClass('animated');
                    }
                });
            }
            
            // Run animations on scroll
            $(window).on('scroll', function() {
                initAnimations();
            });
            
            // Run animations on page load
            initAnimations();
            
            // EMERGENCY COURSE THUMBNAIL FIX
            console.log('Applying emergency thumbnail fix...');
            setTimeout(function() {
                var courseImages = document.querySelectorAll('.card img, .video-card img');
                console.log('Found ' + courseImages.length + ' course images');
                
                courseImages.forEach(function(img) {
                    console.log('Fixing image: ' + img.src);
                    img.style.cssText = 'display: block !important; visibility: visible !important; opacity: 1 !important; width: 100% !important; height: 100% !important; object-fit: cover !important;';
                    
                    // If image is broken, try reloading with timestamp
                    if (!img.complete || img.naturalWidth === 0) {
                        var originalSrc = img.src.split('?')[0];
                        img.src = originalSrc + '?t=' + new Date().getTime();
                        console.log('Reloading image: ' + img.src);
                    }
                });
                
                // Hide any SVG containers
                var svgContainers = document.querySelectorAll('.svg-bg, .svg-icon-container');
                svgContainers.forEach(function(container) {
                    container.style.display = 'none';
                });
            }, 500);
        });
    </script>
    
    <?php include 'debug-js.php'; ?>
</body>
</html>
