<?php
/**
 * Panadite Academy - Enhanced Login System
 * Integrated with SessionManager and AuthManager for robust session handling
 */

// Turn off output buffering to prevent header issues
ob_end_clean();
if (ob_get_level() == 0) ob_start();

// Start error reporting for debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);

// ==========================================
// ENHANCED SESSION AND AUTH MANAGEMENT
// ==========================================
// Integrated session management directly to avoid external dependencies

// Start secure session configuration
if (session_status() !== PHP_SESSION_ACTIVE) {
    // Enhanced session configuration for security and Linux server compatibility
    ini_set('session.cookie_httponly', 1);
    ini_set('session.use_only_cookies', 1);
    // Adjust secure flag based on HTTPS availability
    ini_set('session.cookie_secure', isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 1 : 0);
    ini_set('session.cookie_samesite', 'Lax');
    ini_set('session.use_strict_mode', 1);
    ini_set('session.cookie_lifetime', 7200); // 2 hours
    ini_set('session.gc_maxlifetime', 7200); // 2 hours
    ini_set('session.cookie_domain', ''); // Let PHP determine the domain
    ini_set('session.cookie_path', '/'); // Ensure cookie works for entire site

    // Set proper session save path for Linux server
    $temp_dir = sys_get_temp_dir();
    if (!is_writable($temp_dir)) {
        // Try alternative paths for Linux server
        $alternative_paths = ['/tmp', '/var/tmp', dirname(__FILE__) . '/tmp'];
        foreach ($alternative_paths as $path) {
            if (is_dir($path) && is_writable($path)) {
                $temp_dir = $path;
                break;
            }
        }
    }
    ini_set('session.save_path', $temp_dir);
    

    
    // Start session
    session_start();
}

// Initialize session security
if (!isset($_SESSION['created_at'])) {
    $_SESSION['created_at'] = time();
    $_SESSION['last_regeneration'] = time();
    session_regenerate_id(true);
} else {
    // Regenerate session ID periodically for security
    if (time() - ($_SESSION['last_regeneration'] ?? 0) > 300) { // 5 minutes
        $_SESSION['last_regeneration'] = time();
        session_regenerate_id(true);
    }
    
    // Check session lifetime
    if (time() - $_SESSION['created_at'] > 7200) { // 2 hours
        session_destroy();
        session_start();
    }
}

require_once __DIR__ . '/functions.php';
require_once __DIR__ . '/../includes/site_config.php';

// Try to include database connection with error handling
try {
    require_once __DIR__ . '/../includes/db_connection.php';
    if (isset($conn) && $conn) {
        
    }
} catch (Exception $e) {
    // Continue with emergency credentials fallback
}

// Handle session clearing if requested
if (isset($_GET['clear_session']) && $_GET['clear_session'] === 'true') {
    error_log('DEBUG - Clearing session as requested');
    // Clear session completely
    $_SESSION = array();
    if (ini_get("session.use_cookies")) {
        $params = session_get_cookie_params();
        setcookie(session_name(), '', time() - 42000,
            $params["path"], $params["domain"],
            $params["secure"], $params["httponly"]
        );
    }
    session_destroy();
    header('Location: login.php');
    exit();
}

// Check for redirect parameter
$redirect = '';
if (isset($_GET['redirect'])) {
    $redirect = $_GET['redirect'];
    // Store redirect in session to persist through session clearing
    $_SESSION['stored_redirect'] = $redirect;
} else if (isset($_SESSION['stored_redirect'])) {
    // Retrieve redirect from session if it was stored before
    $redirect = $_SESSION['stored_redirect'];
}

// Define isLoggedIn function inline if it doesn't exist
if (!function_exists('isLoggedIn')) {
    function isLoggedIn() {
        return isset($_SESSION['logged_in']) && $_SESSION['logged_in'] === true && 
               isset($_SESSION['user_id']) && !empty($_SESSION['user_id']);
    }
}

// Check if already logged in
if (isset($_SESSION['logged_in']) && $_SESSION['logged_in'] === true && isset($_SESSION['user_id'])) {
    $username = $_SESSION['username'] ?? 'unknown';
    $role = $_SESSION['user_role'] ?? 'unknown';
    error_log('DEBUG - User already logged in: ' . $username . ' (Role: ' . $role . ')');
    
    // Handle redirect logic
    if (!empty($redirect)) {
        header('Location: ' . $redirect);
    } else {
        // Redirect to role-based dashboard
        if ($role === 'admin') {
            $dashboard_url = $site_url . '/admin/dashboard.php';
        } else if ($role === 'teacher') {
            $dashboard_url = $site_url . '/teacher/dashboard.php';
        } else {
            $dashboard_url = $site_url . '/student/dashboard.php';
        }
        header('Location: ' . $dashboard_url);
    }
    exit();
}

$error = '';
$success = '';

// Check for logout success message
if (isset($_GET['logout']) && $_GET['logout'] == '1') {
    $success = 'You have been logged out successfully';
}



// Handle login form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $username = trim($_POST['username'] ?? '');
    $password = trim($_POST['password'] ?? '');
    

    
    if (empty($username) || empty($password)) {
        $error = 'Username and password are required';

    } else {

        
        // Check emergency credentials
        if (($username === 'admin' && $password === 'admin123') || 
            ($username === 'teacher' && $password === 'teacher123') || 
            ($username === 'student' && $password === 'student123')) {
            
            // Create emergency session
            error_log('DEBUG - Creating emergency session for: ' . $username);
            error_log('DEBUG - Before session_regenerate_id - Session ID: ' . session_id());
            error_log('DEBUG - Current session data before clear: ' . print_r($_SESSION, true));
            
            session_regenerate_id(true);
            $_SESSION = array();
            $_SESSION['user_id'] = 999;
            $_SESSION['username'] = $username;
            $_SESSION['first_name'] = ucfirst($username);
            $_SESSION['last_name'] = 'User';
            $_SESSION['user_role'] = $username;
            $_SESSION['logged_in'] = true;
            $_SESSION['is_logged_in'] = true;
            $_SESSION['login_time'] = time();
            $_SESSION['created_at'] = time();
            $_SESSION['last_regeneration'] = time();
            
            error_log('DEBUG - Session data set: ' . print_r($_SESSION, true));
            error_log('DEBUG - After setting session - Session ID: ' . session_id());
            
            // Force session to be written and immediately check
            session_write_close();
            session_start();
            
            error_log('DEBUG - After session_write_close/start - Session ID: ' . session_id());
            error_log('DEBUG - Emergency session final check: ' . print_r($_SESSION, true));
            
            // Test session directly
            $session_check = isset($_SESSION['logged_in']) && $_SESSION['logged_in'] === true;
            error_log('DEBUG - Session check returns: ' . ($session_check ? 'true' : 'false'));
            
            // Check cookie headers
            error_log('DEBUG - Headers already sent: ' . (headers_sent() ? 'YES' : 'NO'));
            error_log('DEBUG - Session cookie params: ' . print_r(session_get_cookie_params(), true));
            
            // Manual cookie setting as backup (in case automatic cookie isn't working)
            if (!headers_sent()) {
                $cookie_params = session_get_cookie_params();
                setcookie(
                    session_name(),
                    session_id(),
                    time() + 7200, // 2 hours
                    $cookie_params['path'],
                    $cookie_params['domain'],
                    $cookie_params['secure'],
                    $cookie_params['httponly']
                );
                error_log('DEBUG - Manual cookie set for session: ' . session_name() . '=' . session_id());
            }
            
            // Redirect to dashboard
            $dashboard_url = '';
            if ($username === 'admin') {
                $dashboard_url = $site_url . '/admin/dashboard.php';
            } else if ($username === 'teacher') {
                $dashboard_url = $site_url . '/teacher/dashboard.php';
            } else {
                $dashboard_url = 'https://panaditeacademy.com/student/dashboard.php';
            }
            
            error_log('DEBUG - Redirecting to: ' . $dashboard_url);
            
            header('Location: ' . $dashboard_url);
            exit();
            
        } else {
            // Database authentication
            try {
                // Check database connection with more specific error handling
                if (!isset($conn) || !$conn) {
                    error_log('DEBUG - Database connection not available');
                    throw new Exception('Database connection not available');
                }
                
                // Test the connection
                if ($conn->connect_error) {
                    error_log('DEBUG - Database connection error: ' . $conn->connect_error);
                    throw new Exception('Database connection failed: ' . $conn->connect_error);
                }
                
                // Prepare statement with error checking
                $sql = "SELECT user_id, username, email, password, first_name, last_name, user_role FROM users WHERE (username = ? OR email = ?)";
                $stmt = $conn->prepare($sql);
                
                if (!$stmt) {
                    error_log('DEBUG - Prepare failed: ' . $conn->error);
                    throw new Exception('Database query preparation failed: ' . $conn->error);
                }
                
                $stmt->bind_param("ss", $username, $username);
                
                if (!$stmt->execute()) {
                    error_log('DEBUG - Execute failed: ' . $stmt->error);
                    throw new Exception('Database query execution failed: ' . $stmt->error);
                }
                
                $result = $stmt->get_result();
                
                if ($result->num_rows === 0) {
                    $error = 'Invalid username/email or password';
                } else {
                    $user = $result->fetch_assoc();
                    
                    if (password_verify($password, $user['password']) || $user['password'] === $password) {
                        // Create user session (same logic as emergency credentials)
                        // Enhanced session data creation
                        $sessionData = [
                            'user_id' => $user['user_id'],
                            'username' => $user['username'],
                            'email' => $user['email'],
                            'first_name' => $user['first_name'],
                            'last_name' => $user['last_name'],
                            'user_role' => $user['user_role'],
                            'logged_in' => true,
                            'is_logged_in' => true,
                            'login_time' => time(),
                            'created_at' => time(),
                            'last_regeneration' => time()
                        ];
                        
                        // Set all session data at once
                        foreach ($sessionData as $key => $value) {
                            $_SESSION[$key] = $value;
                        }


                        
                        // Manual cookie setting as backup (same as emergency credentials)
                        if (!headers_sent()) {
                            session_write_close();
                            session_start();
                            $session_id = session_id();
                            
                            // Force cookie settings with multiple fallbacks for cross-directory access
                            $cookie_options = [
                                'expires' => time() + 7200,
                                'path' => '/',  // CRITICAL: Root path for entire site
                                'domain' => $_SERVER['HTTP_HOST'] ?? $_SERVER['SERVER_NAME'] ?? '',
                                'secure' => isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on',
                                'httponly' => true,
                                'samesite' => 'Lax'
                            ];
                            
                            // Set cookie multiple ways to ensure it sticks
                            setcookie('PHPSESSID', $session_id, $cookie_options);
                            setcookie(session_name(), $session_id, $cookie_options);
                            
                            // Also set via header as additional fallback
                            $cookie_header = session_name() . '=' . $session_id . '; Path=/; HttpOnly; SameSite=Lax';
                            if ($cookie_options['secure']) {
                                $cookie_header .= '; Secure';
                            }
                            header('Set-Cookie: ' . $cookie_header, false);
                            

                        }
                        
                        // Redirect to dashboard
                        $role = $user['user_role'];
                        if ($role === 'admin') {
                            $dashboard_url = $site_url . '/admin/dashboard.php';
                        } else if ($role === 'teacher') {
                            $dashboard_url = $site_url . '/teacher/dashboard.php';
                        } else {
                            $dashboard_url = $site_url . '/student/dashboard.php';
                        }
                        
                        error_log('DEBUG - Redirecting to: ' . $dashboard_url);
                        
                        header('Location: ' . $dashboard_url);
                        exit();
                    } else {
                        $error = 'Invalid username/email or password';
                    }
                }
                $stmt->close();
                
            } catch (Exception $e) {
                // If database fails, log the error but still allow emergency credentials
                error_log('DEBUG - Database authentication failed: ' . $e->getMessage());
                $error = 'Database authentication unavailable. Please use emergency credentials or try again later.';
            }
        }
    }
}

?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Login - Panadite Academy</title>
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css">
    <!-- Updated Font Awesome with integrity check to ensure proper loading -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css" integrity="sha512-iecdLmaskl7CVkqkXNQ/ZH/XLlvWZOJyj7Yy7tcenmpD1ypASozpmT/E0iPtmFIB46ZmdtAc9eNBvH0H/ZpiBw==" crossorigin="anonymous" referrerpolicy="no-referrer" />
    <link rel="stylesheet" href="/assets/css/style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/animate.css/4.1.1/animate.min.css">
    <style>
        body {
            background-color: #f8f9fa;
            min-height: 100vh;
        }
        
        .login-container {
            max-width: 480px;
            margin: 40px auto;
            padding: 40px;
            background-color: #fff;
            border-radius: 20px;
            box-shadow: 0 8px 40px rgba(39, 107, 228, 0.1);
            border: 1px solid rgba(39, 107, 228, 0.05);
        }
        
        .logo-container {
            text-align: center;
            margin-bottom: 35px;
        }
        
        .logo-container img {
            max-height: 100px;
            margin-bottom: 20px;
            filter: drop-shadow(0 2px 4px rgba(39, 107, 228, 0.1));
        }
        
        .form-label {
            font-weight: 600;
            color: #2d3436;
            margin-bottom: 10px;
            font-size: 0.95rem;
        }
        
        .form-control {
            border: 2px solid #e9ecef;
            border-radius: 12px;
            padding: 14px 16px;
            transition: all 0.3s ease;
            font-size: 16px;
            background-color: #fafbfc;
            height: 52px;
        }
        
        .form-control:focus {
            border-color: #276BE4;
            box-shadow: 0 0 0 0.25rem rgba(39, 107, 228, 0.15);
            background-color: #ffffff;
            transform: translateY(-1px);
        }
        
        .input-group-text {
            background-color: #fafbfc;
            border: 2px solid #e9ecef;
            border-radius: 12px 0 0 12px;
            color: #276BE4;
            font-size: 18px;
            padding: 14px 16px;
            width: 52px;
            display: flex;
            align-items: center;
            justify-content: center;
        }
        
        .input-group .form-control {
            border-left: none;
            border-radius: 0 12px 12px 0;
        }
        
        .input-group:focus-within .input-group-text {
            border-color: #276BE4;
            background-color: rgba(39, 107, 228, 0.05);
            color: #1e5bcc;
        }
        
        .btn-login {
            background: linear-gradient(135deg, #276BE4, #1e5bcc);
            border: none;
            border-radius: 12px;
            color: white;
            font-weight: 600;
            padding: 16px 24px;
            font-size: 17px;
            transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
            width: 100%;
            height: 52px;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 8px;
            margin-top: 8px;
        }
        
        .btn-login:hover {
            background: linear-gradient(135deg, #1e5bcc, #1752b8);
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(39, 107, 228, 0.3);
        }
        
        .login-links {
            text-align: center;
            margin-top: 25px;
            padding-top: 20px;
            border-top: 1px solid #f1f3f4;
        }
        
        .login-links a {
            color: #276BE4;
            text-decoration: none;
            font-weight: 500;
            font-size: 15px;
            transition: all 0.3s ease;
        }
        
        .login-links a:hover {
            color: #1e5bcc;
            text-decoration: underline;
        }
        
        .tagline {
            text-align: center;
            color: #FFA915;
            font-style: italic;
            margin-bottom: 25px;
            letter-spacing: 0.5px;
            font-size: 1.1em;
            font-weight: 500;
        }
        
        .password-toggle {
            position: absolute;
            right: 10px;
            top: 50%;
            transform: translateY(-50%);
            cursor: pointer;
            color: #6c757d;
            z-index: 10;
        }
        
        .password-container {
            position: relative;
        }
            transition: all 0.3s ease;
        }
        
        .input-group:focus-within + .form-label::after,
        .form-control:focus ~ .form-label::after {
            width: 30px;
        }
        
        .input-group {
            transition: all 0.3s ease;
        }
        
        .input-group:focus-within {
            transform: translateY(-3px);
        }
        
        @keyframes pulseGlow {
            0% { box-shadow: 0 0 0 0 rgba(39, 107, 228, 0.4); }
            70% { box-shadow: 0 0 0 10px rgba(39, 107, 228, 0); }
            100% { box-shadow: 0 0 0 0 rgba(39, 107, 228, 0); }
        }
        
        .pulse-animation {
            animation: pulseGlow 2s infinite;
        }
    </style>
</head>
<body>
    <!-- Include the modern navbar -->
    <?php include '../includes/navbar.php'; ?>
    
    <div class="container">
        <div class="login-container">
            <div class="logo-container animate__animated animate__fadeIn">
                <img src="/assets/images/logo.jpeg" alt="Panadite Academy Logo" class="img-fluid">
                <p class="tagline animate__animated animate__fadeIn animate__delay-1s">Get ahead of the curve</p>
            </div>
            
            <?php if ($error): ?>
            <div class="alert alert-danger" role="alert">
                <?php echo $error; ?>
            </div>
            <?php endif; ?>
            
            <?php if ($success): ?>
            <div class="alert alert-success" role="alert">
                <?php echo $success; ?>
            </div>
            <?php endif; ?>
            
            <?php if (!empty($debug_info)): ?>
            <div class="alert alert-info" role="alert">
                <strong>Debug Information:</strong>
                <ul>
                    <?php foreach($debug_info as $info): ?>
                    <li><?php echo htmlspecialchars($info); ?></li>
                    <?php endforeach; ?>
                </ul>
                <p><strong>SESSION Data:</strong> <?php echo htmlspecialchars(print_r($_SESSION, true)); ?></p>
            </div>
            <?php endif; ?>
            

            
            <form class="login-form animate__animated animate__fadeIn animate__delay-1s" method="POST" action="<?php echo htmlspecialchars($_SERVER['PHP_SELF']) . (!empty($redirect) ? '?redirect=' . urlencode($redirect) : ''); ?>" id="loginForm">
                <div class="mb-3">
                    <label for="username" class="form-label">Username</label>
                    <div class="input-group">
                        <span class="input-group-text"><i class="fas fa-user"></i></span>
                        <input type="text" class="form-control" id="username" name="username" required>
                    </div>
                </div>
                
                <div class="mb-3">
                    <label for="password" class="form-label">Password</label>
                    <div class="input-group password-container">
                        <span class="input-group-text"><i class="fas fa-lock"></i></span>
                        <input type="password" class="form-control" id="password" name="password" required>
                        <span class="password-toggle" onclick="togglePassword('password')">
                            <i class="far fa-eye"></i>
                        </span>
                    </div>
                </div>
                

                
                <button type="submit" class="btn btn-login w-100 animate__animated animate__fadeIn animate__delay-2s" id="loginBtn">
                    Login <i class="fas fa-arrow-right-to-bracket ms-2"></i>
                </button>
            </form>
            
            
            <div class="login-links animate__animated animate__fadeIn animate__delay-1s">
                <a href="register.php">Don't have an account? Register</a><br>
                <a href="/index.php">Back to Home</a>
            </div>
        </div>
    </div>





    <footer class="mt-5 position-relative">
        <!-- Top gradient bar -->
        <div class="position-absolute top-0 start-0 w-100" style="height: 4px; background: linear-gradient(90deg, #276BE4, #FFA915);"></div>
        
        <!-- Main footer content -->
        <div class="bg-white py-5">
            <div class="container">
                <div class="row gy-4">
                    <!-- Logo and description -->
                    <div class="col-lg-4 col-md-6 mb-4">
                        <div class="mb-4">
                            <img src="/assets/images/logo.jpeg" alt="Panadite Academy" style="height: 50px;">
                        </div>
                        <p class="mb-2" style="color: #333; font-weight: 500;">Empowering learners through high-quality educational content.</p>
                        <p style="color: #333;">Get ahead of the curve with our premium videos and courses.</p>
                    </div>
                    
                    <!-- Quick links -->
                    <div class="col-lg-2 col-md-6 mb-4">
                        <h5 class="fw-bold mb-3 pb-2 position-relative" style="color: #222;">
                            Quick Links
                            <span class="position-absolute bottom-0 start-0" style="height: 2px; width: 40px; background: #FFA915;"></span>
                        </h5>
                        <ul class="list-unstyled">
                            <li class="mb-2"><a href="/index.php" class="text-decoration-none" style="color: #333;"><i class="fas fa-angle-right me-2" style="color: #276BE4;"></i>Home</a></li>
                            <li class="mb-2"><a href="/courses.php" class="text-decoration-none" style="color: #333;"><i class="fas fa-angle-right me-2" style="color: #276BE4;"></i>Courses</a></li>
                            <li class="mb-2"><a href="/videos.php" class="text-decoration-none" style="color: #333;"><i class="fas fa-angle-right me-2" style="color: #276BE4;"></i>Videos</a></li>
                            <li class="mb-2"><a href="/about.php" class="text-decoration-none" style="color: #333;"><i class="fas fa-angle-right me-2" style="color: #276BE4;"></i>About Us</a></li>
                            <li class="mb-2"><a href="/contact.php" class="text-decoration-none" style="color: #333;"><i class="fas fa-angle-right me-2" style="color: #276BE4;"></i>Contact</a></li>
                        </ul>
                    </div>
                    
                    <!-- Facilitator links -->
                    <div class="col-lg-2 col-md-6 mb-4">
                        <h5 class="fw-bold mb-3 pb-2 position-relative" style="color: #222;">
                            Facilitator
                            <span class="position-absolute bottom-0 start-0" style="height: 2px; width: 40px; background: #FFA915;"></span>
                        </h5>
                        <ul class="list-unstyled">
                            <li class="mb-2"><a href="/auth/register.php?role=teacher" class="text-decoration-none" style="color: #333;"><i class="fas fa-angle-right me-2" style="color: #276BE4;"></i>Become a Facilitator</a></li>
                            <li class="mb-2"><a href="/teacher/dashboard.php" class="text-decoration-none" style="color: #333;"><i class="fas fa-angle-right me-2" style="color: #276BE4;"></i>Teacher Dashboard</a></li>
                            <li class="mb-2"><a href="/teacher/sales.php" class="text-decoration-none" style="color: #333;"><i class="fas fa-angle-right me-2" style="color: #276BE4;"></i>Your Sales</a></li>
                            <li class="mb-2"><a href="/teacher/help.php" class="text-decoration-none" style="color: #333;"><i class="fas fa-angle-right me-2" style="color: #276BE4;"></i>Teacher Help</a></li>
                        </ul>
                    </div>
                    
                    <!-- Contact info -->
                    <div class="col-lg-4 col-md-6">
                        <h5 class="fw-bold mb-3 pb-2 position-relative" style="color: #222;">
                            Contact
                            <span class="position-absolute bottom-0 start-0" style="height: 2px; width: 40px; background: #FFA915;"></span>
                        </h5>
                        <ul class="list-unstyled mb-4">
                            <li class="mb-3 d-flex align-items-center">
                                <span class="d-flex align-items-center justify-content-center me-3" style="width: 36px; height: 36px; background: rgba(39,107,228,0.1); 
    </script>    
border-radius: 50%;">
                                    <i class="fas fa-map-marker-alt" style="color: #276BE4;"></i>
                                </span>
                                <span style="color: #333;">296 Pretorius Street, Corner Lilian Ngoyi, Pretoria Central, 0001, South Africa</span>
                            </li>
                            <li class="mb-3 d-flex align-items-center">
                                <span class="d-flex align-items-center justify-content-center me-3" style="width: 36px; height: 36px; background: rgba(39,107,228,0.1); border-radius: 50%;">
                                    <i class="fas fa-envelope" style="color: #276BE4;"></i>
                                </span>
                                <a href="mailto:admin@panaditeacademy.com" class="text-decoration-none" style="color: #333;">admin@panaditeacademy.com</a>
                            </li>
                            <li class="mb-3 d-flex align-items-center">
                                <span class="d-flex align-items-center justify-content-center me-3" style="width: 36px; height: 36px; background: rgba(39,107,228,0.1); border-radius: 50%;">
                                    <i class="fas fa-phone" style="color: #276BE4;"></i>
                                </span>
                                <a href="tel:+27747223141" class="text-decoration-none" style="color: #333;">+27 74 722 3141</a>
                            </li>
                        </ul>
                        
                        <!-- Social icons -->
                        <div class="d-flex">
                            <a href="#" class="d-flex align-items-center justify-content-center me-2" style="width: 40px; height: 40px; background: rgba(39,107,228,0.1); border-radius: 50%; color: #276BE4; transition: all 0.3s ease;">
                                <i class="fab fa-facebook-f"></i>
                            </a>
                            <a href="#" class="d-flex align-items-center justify-content-center me-2" style="width: 40px; height: 40px; background: rgba(39,107,228,0.1); border-radius: 50%; color: #276BE4; transition: all 0.3s ease;">
                                <i class="fab fa-twitter"></i>
                            </a>
                            <a href="#" class="d-flex align-items-center justify-content-center me-2" style="width: 40px; height: 40px; background: rgba(39,107,228,0.1); border-radius: 50%; color: #276BE4; transition: all 0.3s ease;">
                                <i class="fab fa-instagram"></i>
                            </a>
                            <a href="#" class="d-flex align-items-center justify-content-center" style="width: 40px; height: 40px; background: rgba(39,107,228,0.1); border-radius: 50%; color: #276BE4; transition: all 0.3s ease;">
                                <i class="fab fa-linkedin-in"></i>
                            </a>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Copyright section -->
        <div style="background: #f8f9fa; border-top: 1px solid #eee;" class="py-3">
            <div class="container">
                <div class="row align-items-center">
                    <div class="col-md-6 mb-3 mb-md-0">
                        <p class="mb-0" style="color: #333; font-weight: 500;">&copy; <?php echo date('Y'); ?> Panadite Academy. All rights reserved.</p>
                    </div>
                    <div class="col-md-6 text-md-end">
                        <a href="/privacy-policy.php" class="text-decoration-none me-3" style="color: #333;">Privacy Policy</a>
                        <a href="/terms.php" class="text-decoration-none me-3" style="color: #333;">Terms of Use</a>
                        <a href="/sitemap.php" class="text-decoration-none" style="color: #333;">Sitemap</a>
                    </div>
                </div>
            </div>
        </div>
    </footer>

    <script src="https://cdn.jsdelivr.net/npm/jquery@3.6.4/dist/jquery.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        function togglePassword(inputId) {
            const input = document.getElementById(inputId);
            const icon = input.nextElementSibling.querySelector('i');
            
            if (input.type === 'password') {
                input.type = 'text';
                icon.classList.remove('fa-eye');
                icon.classList.add('fa-eye-slash');
            } else {
                input.type = 'password';
                icon.classList.remove('fa-eye-slash');
                icon.classList.add('fa-eye');
            }
        }
        

        
        // Add form submission animation
        document.getElementById('loginForm').addEventListener('submit', function(e) {
            document.getElementById('loginBtn').innerHTML = 'Logging in... <i class="fas fa-spinner fa-spin ms-2"></i>';
            document.getElementById('loginBtn').disabled = true;
        });
        
        $(document).ready(function() {
            // Animate floating icons and shapes
            animateFloatingElements();
            
            // Add staggered animations to form elements
            $('.form-control').each(function(index) {
                $(this).addClass('animate__animated animate__fadeInUp')
                       .css('animation-delay', (index * 100 + 300) + 'ms');
            });
            
            // Add focus animations
            $('.form-control').on('focus', function() {
                $(this).parent().addClass('pulse-animation');
            }).on('blur', function() {
                $(this).parent().removeClass('pulse-animation');
            });
            
            // Add floating effect to login container
            $('.login-container').on('mousemove', function(e) {
                const container = $(this);
                const containerWidth = container.width();
                const containerHeight = container.height();
                const centerX = container.offset().left + containerWidth/2;
                const centerY = container.offset().top + containerHeight/2;
                const mouseX = e.pageX - centerX;
                const mouseY = e.pageY - centerY;
                const rotateY = (mouseX / containerWidth) * 5;
                const rotateX = ((mouseY / containerHeight) * -5);
                
                container.css('transform', 
                    `translateY(-5px) perspective(1000px) rotateX(${rotateX}deg) rotateY(${rotateY}deg)`
                );
            }).on('mouseleave', function() {
                $(this).css('transform', 'translateY(-5px)');
            });
            
            // Add shine effect on button hover
            $('.btn-login').on('mouseenter', function() {
                $(this).addClass('animate__animated animate__pulse');
                $(this).find('i').css({'opacity': '1', 'transform': 'translateX(3px)'});
            }).on('mouseleave', function() {
                $(this).removeClass('animate__animated animate__pulse');
                $(this).find('i').css({'transform': 'translateX(0)'});
            });
            
            // Ensure the icon is visible with a small animation to draw attention
            $('.btn-login i').css({'opacity': '1', 'transition': 'transform 0.3s ease'});
            
            // Add a quick flicker animation to ensure the icon is noticed
            setTimeout(function() {
                $('.btn-login i').css({'transform': 'scale(1.2)'});
                setTimeout(function() {
                    $('.btn-login i').css({'transform': 'scale(1)'});
                }, 300);
            }, 1000);
            
            // Validate login form before submission
            $('.login-form').on('submit', function() {
                const username = $('#username').val();
                const password = $('#password').val();
                
                if (!username || !password) {
                    return false;
                }
                
                $('.btn-login').html('<span class="spinner-border spinner-border-sm" role="status" aria-hidden="true"></span> Signing in...');
                return true;
            });
        });
        // Function to animate floating elements
        function animateFloatingElements() {
            $('.floating-icon, .shape').each(function(index) {
                var element = $(this);
                // Set random initial positions
                var startX = Math.random() * 10 - 5; // -5 to +5
                var startY = Math.random() * 10 - 5; // -5 to +5
                
                // Animate with random duration between 5-10 seconds
                var duration = 5000 + Math.random() * 5000;
                
                // Function to create floating animation
                function floatElement() {
                    // Generate new random positions
                    var moveX = Math.random() * 20 - 10; // -10 to +10
                    var moveY = Math.random() * 20 - 10; // -10 to +10
                    var rotate = Math.random() * 10 - 5; // -5 to +5 degrees
                    
                    element.animate({
                        transform: 'translate(' + moveX + 'px, ' + moveY + 'px) rotate(' + rotate + 'deg)'
                    }, duration, 'linear', function() {
                        floatElement(); // Infinite loop
                    });
                }
                
                // Start with a delay based on index
                setTimeout(function() {
                    element.css('transform', 'translate(' + startX + 'px, ' + startY + 'px)');
                    floatElement();
                }, index * 300);
            });
        }
    </script>
</body>
</html>

       