<?php
/**
 * Secure Referral Code Generator
 * Generates cryptographically secure, unpredictable referral codes
 */

class SecureReferralGenerator {
    
    // Secret key for encryption (should be in environment variables in production)
    private static $SECRET_KEY = 'PanaditeAcademy_2024_SecureReferrals_v2.1';
    
    /**
     * Generate a secure referral code for a user
     * @param int $userId
     * @param string $userRole
     * @param string $email
     * @return string
     */
    public static function generateSecureCode($userId, $userRole, $email) {
        // Create unique data string
        $timestamp = time();
        $salt = bin2hex(random_bytes(8)); // 16 character random salt
        
        // Combine user data with salt and timestamp
        $dataString = $userId . '|' . $userRole . '|' . $email . '|' . $timestamp . '|' . $salt;
        
        // Create HMAC hash for integrity
        $hash = hash_hmac('sha256', $dataString, self::$SECRET_KEY);
        
        // Take first 12 characters of hash for the code
        $codeBase = strtoupper(substr($hash, 0, 12));
        
        // Add role prefix for easy identification
        $rolePrefix = ($userRole === 'teacher') ? 'TCH' : 'STU';
        
        // Final secure code format: TCH123ABC4DEF or STU456GHI7JKL
        return $rolePrefix . $codeBase;
    }
    
    /**
     * Validate a secure referral code and extract user information
     * @param string $code
     * @param mysqli $conn Database connection
     * @return array|false
     */
    public static function validateSecureCode($code, $conn) {
        // Check if it's a new secure format (starts with TCH or STU)
        if (!preg_match('/^(TCH|STU)[A-F0-9]{12}$/', $code)) {
            return false; // Not a secure format
        }
        
        $rolePrefix = substr($code, 0, 3);
        $hashPart = substr($code, 3);
        $userRole = ($rolePrefix === 'TCH') ? 'teacher' : 'student';
        
        // Look up in secure codes table (we'll create this)
        $query = "SELECT user_id, code_hash, created_at FROM secure_referral_codes 
                  WHERE code_hash = ? AND user_role = ? AND is_active = 1";
        
        $stmt = $conn->prepare($query);
        if (!$stmt) return false;
        
        $stmt->bind_param("ss", $hashPart, $userRole);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result->num_rows === 0) {
            return false;
        }
        
        $codeData = $result->fetch_assoc();
        
        // Get user information
        $userQuery = "SELECT user_id, username, first_name, last_name, user_role, email 
                      FROM users WHERE user_id = ?";
        $userStmt = $conn->prepare($userQuery);
        $userStmt->bind_param("i", $codeData['user_id']);
        $userStmt->execute();
        $userResult = $userStmt->get_result();
        
        if ($userResult->num_rows === 0) {
            return false;
        }
        
        return $userResult->fetch_assoc();
    }
    
    /**
     * Store secure referral code in database
     * @param int $userId
     * @param string $userRole
     * @param string $fullCode
     * @param mysqli $conn
     * @return bool
     */
    public static function storeSecureCode($userId, $userRole, $fullCode, $conn) {
        $hashPart = substr($fullCode, 3); // Remove TCH/STU prefix
        
        // Check if table exists, create if not
        self::createSecureCodesTable($conn);
        
        // Deactivate old codes for this user
        $deactivateQuery = "UPDATE secure_referral_codes SET is_active = 0 WHERE user_id = ?";
        $deactivateStmt = $conn->prepare($deactivateQuery);
        $deactivateStmt->bind_param("i", $userId);
        $deactivateStmt->execute();
        
        // Insert new secure code
        $insertQuery = "INSERT INTO secure_referral_codes (user_id, user_role, code_hash, full_code, created_at, is_active) 
                        VALUES (?, ?, ?, ?, NOW(), 1)";
        $insertStmt = $conn->prepare($insertQuery);
        $insertStmt->bind_param("isss", $userId, $userRole, $hashPart, $fullCode);
        
        return $insertStmt->execute();
    }
    
    /**
     * Create secure referral codes table if it doesn't exist
     */
    private static function createSecureCodesTable($conn) {
        $createTableSQL = "
        CREATE TABLE IF NOT EXISTS secure_referral_codes (
            id INT AUTO_INCREMENT PRIMARY KEY,
            user_id INT NOT NULL,
            user_role ENUM('student', 'teacher') NOT NULL,
            code_hash VARCHAR(12) NOT NULL UNIQUE,
            full_code VARCHAR(15) NOT NULL UNIQUE,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            is_active BOOLEAN DEFAULT 1,
            INDEX idx_user (user_id),
            INDEX idx_code_hash (code_hash),
            INDEX idx_active (is_active)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
        
        $conn->query($createTableSQL);
    }
    
    /**
     * Generate and store secure referral code for user
     * @param int $userId
     * @param string $userRole
     * @param string $email
     * @param mysqli $conn
     * @return string|false
     */
    public static function generateAndStore($userId, $userRole, $email, $conn) {
        $secureCode = self::generateSecureCode($userId, $userRole, $email);
        
        if (self::storeSecureCode($userId, $userRole, $secureCode, $conn)) {
            return $secureCode;
        }
        
        return false;
    }
    
    /**
     * Migrate existing user to secure referral code
     * @param int $userId
     * @param mysqli $conn
     * @return string|false
     */
    public static function migrateUserToSecure($userId, $conn) {
        try {
            // Get user data
            $userQuery = "SELECT user_id, username, first_name, last_name, user_role, email 
                          FROM users WHERE user_id = ?";
            $stmt = $conn->prepare($userQuery);
            
            if (!$stmt) {
                error_log("Failed to prepare user query: " . $conn->error);
                return false;
            }
            
            $stmt->bind_param("i", $userId);
            $stmt->execute();
            $result = $stmt->get_result();
            
            if (!$result || $result->num_rows === 0) {
                error_log("User not found: " . $userId);
                return false;
            }
            
            $user = $result->fetch_assoc();
            return self::generateAndStore($user['user_id'], $user['user_role'], $user['email'], $conn);
            
        } catch (Exception $e) {
            error_log("Error migrating user to secure code: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Get or create secure referral code for user
     * @param int $userId
     * @param mysqli $conn
     * @return string|false
     */
    public static function getOrCreateSecureCode($userId, $conn) {
        // Ensure table exists first
        self::createSecureCodesTable($conn);
        
        // Check if user already has a secure code
        $checkQuery = "SELECT full_code FROM secure_referral_codes 
                       WHERE user_id = ? AND is_active = 1";
        $stmt = $conn->prepare($checkQuery);
        
        if (!$stmt) {
            // If prepare fails, fall back to generating a new code
            error_log("Failed to prepare secure code check query: " . $conn->error);
            return self::migrateUserToSecure($userId, $conn);
        }
        
        $stmt->bind_param("i", $userId);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result && $result->num_rows > 0) {
            $row = $result->fetch_assoc();
            return $row['full_code'];
        }
        
        // Generate new secure code
        return self::migrateUserToSecure($userId, $conn);
    }
}

/**
 * Legacy referral code functions for backward compatibility
 */
class LegacyReferralValidator {
    
    /**
     * Validate old format referral codes (like ALKI2315)
     * @param string $code
     * @param mysqli $conn
     * @return array|false
     */
    public static function validateLegacyCode($code, $conn) {
        // First check database for stored codes
        $dbQuery = "SELECT u.user_id, u.username, u.user_role, u.first_name, u.last_name
                   FROM referrals r 
                   JOIN users u ON r.referrer_id = u.user_id 
                   WHERE r.referral_code = ?";
                   
        $stmt = $conn->prepare($dbQuery);
        if ($stmt) {
            $stmt->bind_param("s", $code);
            $stmt->execute();
            $result = $stmt->get_result();
            
            if ($result->num_rows > 0) {
                return $result->fetch_assoc();
            }
        }
        
        // Validate against legacy pattern for backward compatibility
        if (strlen($code) >= 6) {
            $firstNamePrefix = substr($code, 0, 2);
            $lastNamePrefix = substr($code, 2, 2);
            $userIdAndRandom = substr($code, 4);
            
            for ($userIdLength = 1; $userIdLength <= 4; $userIdLength++) {
                if (strlen($userIdAndRandom) > $userIdLength) {
                    $potentialUserId = substr($userIdAndRandom, 0, $userIdLength);
                    $randomPart = substr($userIdAndRandom, $userIdLength);
                    
                    if (strlen($randomPart) === 2 && is_numeric($randomPart) && $randomPart >= 10 && $randomPart <= 99) {
                        $userQuery = "SELECT user_id, username, user_role, first_name, last_name 
                                     FROM users 
                                     WHERE user_id = ? 
                                     AND UPPER(SUBSTRING(first_name, 1, 2)) = ?
                                     AND UPPER(SUBSTRING(last_name, 1, 2)) = ?";
                        
                        $userStmt = $conn->prepare($userQuery);
                        if ($userStmt) {
                            $userStmt->bind_param("iss", $potentialUserId, $firstNamePrefix, $lastNamePrefix);
                            $userStmt->execute();
                            $userResult = $userStmt->get_result();
                            
                            if ($userResult->num_rows > 0) {
                                return $userResult->fetch_assoc();
                            }
                        }
                    }
                }
            }
        }
        
        return false;
    }
}
?>
