<?php
// Include database connection
require_once '../config/db_connect.php';

// Function to run database setup
function setup_teacher_settings($conn) {
    $output = [];
    
    // Step 1: Create the teacher_settings table if it doesn't exist
    $create_table_sql = "CREATE TABLE IF NOT EXISTS teacher_settings (
        id INT PRIMARY KEY AUTO_INCREMENT,
        user_id INT NOT NULL,
        email_notifications TINYINT(1) DEFAULT 1,
        sms_notifications TINYINT(1) DEFAULT 0,
        dashboard_theme VARCHAR(20) DEFAULT 'light',
        timezone VARCHAR(50) DEFAULT 'UTC',
        language VARCHAR(10) DEFAULT 'en',
        enable_2fa TINYINT(1) DEFAULT 0,
        FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
    )";
    
    if ($conn->query($create_table_sql) === TRUE) {
        $output[] = "✅ Table 'teacher_settings' created successfully or already exists.";
        
        // Step 2: Find teachers without settings
        $find_teachers_sql = "SELECT u.id, CONCAT(u.first_name, ' ', u.last_name) as name 
                           FROM users u 
                           LEFT JOIN teacher_settings ts ON u.id = ts.user_id 
                           WHERE u.role = 'teacher' AND ts.id IS NULL";
        
        $result = $conn->query($find_teachers_sql);
        
        if ($result && $result->num_rows > 0) {
            $output[] = "ℹ️ Found " . $result->num_rows . " teachers without settings entries.";
            
            // Step 3: Create default settings for each teacher
            $insert_stmt = $conn->prepare("INSERT INTO teacher_settings 
                                        (user_id, email_notifications, sms_notifications, dashboard_theme, timezone, language, enable_2fa) 
                                        VALUES (?, 1, 0, 'light', 'UTC', 'en', 0)");
            
            $insert_stmt->bind_param("i", $teacher_id);
            $success_count = 0;
            
            while ($row = $result->fetch_assoc()) {
                $teacher_id = $row['id'];
                if ($insert_stmt->execute()) {
                    $output[] = "  ✓ Created settings for teacher: " . htmlspecialchars($row['name']) . " (ID: " . $teacher_id . ")";
                    $success_count++;
                } else {
                    $output[] = "  ✗ Failed to create settings for teacher ID: " . $teacher_id . " - Error: " . $insert_stmt->error;
                }
            }
            
            $output[] = "✅ Default settings created for " . $success_count . " teachers.";
            $insert_stmt->close();
        } else {
            $output[] = "ℹ️ All teachers already have settings entries.";
        }
    } else {
        $output[] = "❌ Error creating table: " . $conn->error;
    }
    
    return $output;
}

// Check if this is a CLI execution or web request
if (php_sapi_name() === 'cli') {
    // Running from command line
    $output = setup_teacher_settings($conn);
    echo implode(PHP_EOL, $output) . PHP_EOL;
} else {
    // Running from web browser
    echo "<!DOCTYPE html>
<html>
<head>
    <title>Teacher Settings Table Setup</title>
    <style>
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            line-height: 1.6;
            max-width: 800px;
            margin: 0 auto;
            padding: 20px;
            background-color: #f9fafb;
        }
        .container {
            background-color: #fff;
            border-radius: 10px;
            padding: 20px;
            box-shadow: 0 1px 3px rgba(0,0,0,0.1);
        }
        h1 {
            color: #6366f1;
        }
        .output-line {
            margin: 10px 0;
        }
        .back-link {
            display: inline-block;
            margin-top: 20px;
            background-color: #6366f1;
            color: white;
            padding: 8px 20px;
            text-decoration: none;
            border-radius: 5px;
            font-weight: 500;
        }
        .back-link:hover {
            background-color: #4f46e5;
        }
    </style>
</head>
<body>
    <div class='container'>
        <h1>Teacher Settings Table Setup</h1>";
    
    $output = setup_teacher_settings($conn);
    foreach ($output as $line) {
        echo "<div class='output-line'>" . $line . "</div>";
    }
    
    echo "<a href='../teacher/settings.php' class='back-link'>Return to Settings Page</a>
    </div>
</body>
</html>";
}

// Close the database connection
$conn->close();
?>
