<?php
/**
 * Simple and Reliable PDF Certificate Generator using FPDF
 * No external dependencies - works immediately
 */

class SimplePDFCertificate {
    
    private $conn;
    private $academy_name = "PANADITE ACADEMY";
    
    public function __construct($database_connection) {
        $this->conn = $database_connection;
    }
    
    /**
     * Generate certificate using built-in PHP capabilities
     */
    public function generateCertificate($user_id, $course_id, $download = false) {
        
        // Validate course completion
        if (!$this->isStudentEligibleForCertificate($user_id, $course_id)) {
            throw new Exception("Student has not completed this course yet.");
        }
        
        // Get data
        $student_data = $this->getStudentData($user_id);
        $course_data = $this->getCourseData($course_id);
        
        if (!$student_data || !$course_data) {
            throw new Exception("Invalid student or course data.");
        }
        
        // Use FPDF (lightweight, no dependencies)
        require_once __DIR__ . '/fpdf/fpdf.php';
        
        $pdf = new FPDF('L', 'mm', 'A4'); // Landscape
        $pdf->AddPage();
        
        // Generate certificate content
        $this->generateCertificateContent($pdf, $student_data, $course_data);
        
        // Generate filename
        $safe_course_title = preg_replace('/[^A-Za-z0-9\-_]/', '_', $course_data['title']);
        $safe_student_name = preg_replace('/[^A-Za-z0-9\-_]/', '_', $student_data['first_name'] . '_' . $student_data['last_name']);
        $filename = "Certificate_{$safe_course_title}_{$safe_student_name}_" . date('Y-m-d') . ".pdf";
        
        // Save record
        $this->saveCertificateRecord($user_id, $course_id, $filename);
        
        if ($download) {
            $pdf->Output('D', $filename);
            exit;
        } else {
            return $pdf->Output('S');
        }
    }
    
    /**
     * Create beautiful certificate content
     */
    private function generateCertificateContent($pdf, $student_data, $course_data) {
        
        // Set background color (cream)
        $pdf->SetFillColor(255, 253, 250);
        $pdf->Rect(0, 0, 297, 210, 'F');
        
        // Main border (blue)
        $pdf->SetLineWidth(3);
        $pdf->SetDrawColor(70, 115, 223);
        $pdf->Rect(10, 10, 277, 190);
        
        // Inner border (light gray)
        $pdf->SetLineWidth(1);
        $pdf->SetDrawColor(200, 200, 200);
        $pdf->Rect(15, 15, 267, 180);
        
        // Academy name
        $pdf->SetFont('Arial', 'B', 32);
        $pdf->SetTextColor(70, 115, 223);
        $pdf->SetXY(0, 30);
        $pdf->Cell(297, 15, $this->academy_name, 0, 1, 'C');
        
        // Tagline
        $pdf->SetFont('Arial', 'I', 14);
        $pdf->SetTextColor(100, 100, 100);
        $pdf->SetXY(0, 50);
        $pdf->Cell(297, 8, 'Excellence in Digital Education', 0, 1, 'C');
        
        // Certificate title
        $pdf->SetFont('Arial', 'B', 28);
        $pdf->SetTextColor(50, 50, 50);
        $pdf->SetXY(0, 70);
        $pdf->Cell(297, 15, 'CERTIFICATE OF COMPLETION', 0, 1, 'C');
        
        // Decorative line
        $pdf->SetLineWidth(2);
        $pdf->SetDrawColor(70, 115, 223);
        $pdf->Line(80, 90, 217, 90);
        
        // "This is to certify that"
        $pdf->SetFont('Arial', '', 16);
        $pdf->SetTextColor(80, 80, 80);
        $pdf->SetXY(0, 105);
        $pdf->Cell(297, 8, 'This is to certify that', 0, 1, 'C');
        
        // Student name (main focus)
        $pdf->SetFont('Arial', 'B', 36);
        $pdf->SetTextColor(0, 0, 0);
        $student_name = trim($student_data['first_name'] . ' ' . $student_data['last_name']);
        $pdf->SetXY(0, 120);
        $pdf->Cell(297, 15, $student_name, 0, 1, 'C');
        
        // Underline name
        $pdf->SetLineWidth(1);
        $pdf->SetDrawColor(0, 0, 0);
        $pdf->Line(100, 138, 197, 138);
        
        // "has successfully completed"
        $pdf->SetFont('Arial', '', 16);
        $pdf->SetTextColor(80, 80, 80);
        $pdf->SetXY(0, 145);
        $pdf->Cell(297, 8, 'has successfully completed the course', 0, 1, 'C');
        
        // Course title
        $pdf->SetFont('Arial', 'B', 24);
        $pdf->SetTextColor(70, 115, 223);
        $pdf->SetXY(0, 160);
        $pdf->Cell(297, 12, $course_data['title'], 0, 1, 'C');
        
        // Completion date
        $pdf->SetFont('Arial', '', 14);
        $pdf->SetTextColor(80, 80, 80);
        $pdf->SetXY(0, 175);
        $completion_date = date('F j, Y');
        $pdf->Cell(297, 8, "Date of Completion: {$completion_date}", 0, 1, 'C');
        
        // Footer - Academy signature
        $pdf->SetXY(50, 185);
        $pdf->SetFont('Arial', 'B', 12);
        $pdf->SetTextColor(0, 0, 0);
        $pdf->Cell(80, 6, 'Panadite Academy', 0, 0, 'C');
        
        // Date issued
        $pdf->SetXY(167, 185);
        $pdf->Cell(80, 6, 'Date Issued: ' . date('F j, Y'), 0, 0, 'C');
        
        // Certificate ID
        $certificate_id = 'CERT-' . strtoupper(substr(md5($student_data['user_id'] . $course_data['course_id'] . time()), 0, 8));
        $pdf->SetXY(15, 195);
        $pdf->SetFont('Arial', '', 10);
        $pdf->SetTextColor(150, 150, 150);
        $pdf->Cell(0, 6, "Certificate ID: {$certificate_id}", 0, 1, 'L');
        
        // Website
        $pdf->SetXY(15, 200);
        $pdf->Cell(0, 6, "Verify at: panaditeacademy.com", 0, 1, 'L');
    }
    
    /**
     * Check course completion eligibility
     */
    private function isStudentEligibleForCertificate($user_id, $course_id) {
        
        // Check if student is enrolled
        $enrollment_query = "SELECT * FROM enrollments WHERE user_id = ? AND course_id = ?";
        $stmt = $this->conn->prepare($enrollment_query);
        if ($stmt) {
            $stmt->bind_param("ii", $user_id, $course_id);
            $stmt->execute();
            $result = $stmt->get_result();
            
            if ($result->num_rows > 0) {
                // Student is enrolled
                
                // Check video completion (if videos exist)
                $video_check = "
                    SELECT 
                        COUNT(v.video_id) as total_videos,
                        COUNT(vp.video_id) as watched_videos
                    FROM videos v
                    LEFT JOIN video_progress vp ON v.video_id = vp.video_id AND vp.user_id = ?
                    WHERE v.course_id = ?
                ";
                
                $video_stmt = $this->conn->prepare($video_check);
                if ($video_stmt) {
                    $video_stmt->bind_param("ii", $user_id, $course_id);
                    $video_stmt->execute();
                    $video_result = $video_stmt->get_result();
                    
                    if ($video_row = $video_result->fetch_assoc()) {
                        $total_videos = intval($video_row['total_videos']);
                        $watched_videos = intval($video_row['watched_videos']);
                        
                        if ($total_videos == 0) {
                            // Course has no videos, consider complete upon enrollment
                            return true;
                        } else {
                            // Must watch at least 80% of videos
                            $completion_rate = ($watched_videos / $total_videos) * 100;
                            return $completion_rate >= 80;
                        }
                    }
                    $video_stmt->close();
                }
                
                return true; // Default to eligible if enrolled
            }
            $stmt->close();
        }
        
        return false;
    }
    
    private function getStudentData($user_id) {
        $query = "SELECT user_id, first_name, last_name, email FROM users WHERE user_id = ?";
        $stmt = $this->conn->prepare($query);
        if ($stmt) {
            $stmt->bind_param("i", $user_id);
            $stmt->execute();
            $result = $stmt->get_result();
            return $result->fetch_assoc();
        }
        return null;
    }
    
    private function getCourseData($course_id) {
        $query = "SELECT course_id, title, description FROM courses WHERE course_id = ?";
        $stmt = $this->conn->prepare($query);
        if ($stmt) {
            $stmt->bind_param("i", $course_id);
            $stmt->execute();
            $result = $stmt->get_result();
            return $result->fetch_assoc();
        }
        return null;
    }
    
    private function saveCertificateRecord($user_id, $course_id, $filename) {
        // Create table if not exists
        $create_table = "
            CREATE TABLE IF NOT EXISTS student_certificates (
                id INT AUTO_INCREMENT PRIMARY KEY,
                user_id INT NOT NULL,
                course_id INT NOT NULL,
                certificate_id VARCHAR(50) NOT NULL,
                filename VARCHAR(255) NOT NULL,
                issued_date TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                status ENUM('active', 'revoked') DEFAULT 'active',
                UNIQUE KEY unique_user_course (user_id, course_id)
            )
        ";
        $this->conn->query($create_table);
        
        $certificate_id = 'CERT-' . strtoupper(substr(md5($user_id . $course_id . time()), 0, 8));
        
        $query = "
            INSERT INTO student_certificates (user_id, course_id, certificate_id, filename)
            VALUES (?, ?, ?, ?)
            ON DUPLICATE KEY UPDATE 
                certificate_id = VALUES(certificate_id),
                filename = VALUES(filename),
                issued_date = CURRENT_TIMESTAMP
        ";
        
        $stmt = $this->conn->prepare($query);
        if ($stmt) {
            $stmt->bind_param("iiss", $user_id, $course_id, $certificate_id, $filename);
            $stmt->execute();
            $stmt->close();
        }
        
        return $certificate_id;
    }
}

?>
