<?php
session_start();
require_once 'includes/db.php'; // Use the correct database connection
require_once 'auth/functions.php'; // Authentication functions including getCurrentUser
require_once 'config/functions.php';
require_once 'includes/id_encryption.php'; // Include ID encryption utility
// Updated on 2025-11-18 - Encrypted IDs version

// Check if database connection is working
$db_available = !isset($GLOBALS['db_connection_failed']) || !$GLOBALS['db_connection_failed'];

// Get current user if logged in and if DB is available
$currentUser = $db_available ? getCurrentUser() : null;

// Get featured videos
$featuredVideos = [];
if ($db_available) {
    try {
        // Direct MySQLi query instead of fetchAll to better diagnose any issues
        $videosQuery = "SELECT v.*, 
                        IFNULL(c.title, 'Untitled Course') as course_title, 
                        IFNULL(c.price, 0) as course_price, 
                        IFNULL(u.first_name, 'Instructor') as first_name, 
                        IFNULL(u.last_name, '') as last_name
                      FROM videos v
                      LEFT JOIN courses c ON v.course_id = c.course_id
                      LEFT JOIN users u ON c.teacher_id = u.user_id
                      ORDER BY v.created_at DESC
                      LIMIT 8";
                      
        $result = $conn->query($videosQuery);
        
        if ($result && $result->num_rows > 0) {
            while ($row = $result->fetch_assoc()) {
                $featuredVideos[] = $row;
            }
            // Add debug comment
            echo '<!--';
            echo 'DEBUG: Found ' . count($featuredVideos) . ' videos';
            echo '-->';
        } else {
            // Try a simpler query without the joins
            $simpleVideosQuery = "SELECT * FROM videos ORDER BY created_at DESC LIMIT 8";
            $result = $conn->query($simpleVideosQuery);
            
            if ($result && $result->num_rows > 0) {
                while ($row = $result->fetch_assoc()) {
                    $featuredVideos[] = $row;
                }
                echo '<!--';
                echo 'DEBUG: Found ' . count($featuredVideos) . ' videos with simple query';
                echo '-->';
            } else {
                echo '<!--';
                echo 'DEBUG: No videos found. SQL Error: ' . $conn->error;
                echo '-->';
            }
        }
    } catch (Exception $e) {
        // Log error but don't display it
        error_log('Database error fetching videos: ' . $e->getMessage());
    }
}

// Get featured courses with simplified approach
$featuredCourses = [];

if ($db_available) {
    try {
        // Get ALL courses - show everything, including drafts
        $basicQuery = "SELECT c.*, u.first_name, u.last_name 
                   FROM courses c 
                   LEFT JOIN users u ON c.teacher_id = u.user_id 
                   ORDER BY c.course_id DESC";
        $result = $conn->query($basicQuery);
        
        if ($result && $result->num_rows > 0) {
            // Explicitly create an array to hold courses
            $featuredCourses = array();
            
            while ($row = $result->fetch_assoc()) {
                // Add default values for missing fields
                if (empty($row['first_name'])) $row['first_name'] = 'Instructor';
                if (empty($row['last_name'])) $row['last_name'] = '';
                $row['category_name'] = 'General';
                $row['student_count'] = $row['student_count'] ?? 0;
                $row['rating'] = $row['rating'] ?? 4.5;
                $row['thumbnail'] = !empty($row['thumbnail']) ? $row['thumbnail'] : 'assets/images/course-placeholder.jpg';
                
                // Add to the array
                $featuredCourses[] = $row;
            }
            
            // Force debug output to see what we got
            echo '<!-- DEBUG: Found ' . count($featuredCourses) . ' courses -->';
            echo '<!-- DEBUG: First course: ' . json_encode(array_slice($featuredCourses, 0, 1)) . ' -->';
        } else {
            // Try an even simpler query
            $simpleQuery = "SELECT * FROM courses";
            $simpleResult = $conn->query($simpleQuery);
            
            if ($simpleResult && $simpleResult->num_rows > 0) {
                $featuredCourses = array();
                while ($row = $simpleResult->fetch_assoc()) {
                    $featuredCourses[] = $row;
                }
                echo '<!-- DEBUG: Found ' . count($featuredCourses) . ' courses with simple query -->';
            } else {
                echo '<!-- DEBUG: No courses found. SQL Error: ' . $conn->error . ' -->';
            }
        }
    } catch (Exception $e) {
        error_log('Error fetching courses: ' . $e->getMessage());
        echo '<!-- DEBUG: Exception: ' . $e->getMessage() . ' -->';
        $featuredCourses = [];
    }
}

// Get categories with course counts
$categories = [];
$coursesByCategory = [];

// Get categories with course counts (using the working query from debug)
$categoriesQuery = "SELECT 
                        cat.category_name,
                        COUNT(c.course_id) as course_count
                    FROM categories cat
                    LEFT JOIN courses c ON cat.category_id = c.category_id AND c.is_published = 1
                    GROUP BY cat.category_id, cat.category_name 
                    HAVING course_count > 0
                    ORDER BY cat.category_name ASC";

$result = $conn->query($categoriesQuery);

if ($result && $result->num_rows > 0) {
    while ($row = $result->fetch_assoc()) {
        $categories[] = $row;
    }
}

// Debug output to verify categories are loaded correctly
echo '<!-- DEBUG: Found ' . count($categories) . ' categories -->';
echo '<!-- DEBUG: Categories: ' . json_encode(array_column($categories, 'category_name')) . ' -->';
echo '<!-- DEBUG: Courses by category count: ' . count($coursesByCategory) . ' -->';

// Get courses grouped by category (using the working query from debug)
$coursesQuery = "SELECT c.*, 
                 cat.category_name,
                 IFNULL(u.first_name, 'Instructor') as first_name, 
                 IFNULL(u.last_name, '') as last_name
                 FROM courses c
                 LEFT JOIN categories cat ON c.category_id = cat.category_id
                 LEFT JOIN users u ON c.teacher_id = u.user_id
                 WHERE c.is_published = 1 AND cat.category_name IS NOT NULL
                 ORDER BY cat.category_name ASC, c.title ASC";

$coursesResult = $conn->query($coursesQuery);

if ($coursesResult && $coursesResult->num_rows > 0) {
    while ($course = $coursesResult->fetch_assoc()) {
        $categoryName = $course['category_name'];
        if (!empty($categoryName)) {
            if (!isset($coursesByCategory[$categoryName])) {
                $coursesByCategory[$categoryName] = [];
            }
            $coursesByCategory[$categoryName][] = $course;
        }
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <meta http-equiv="Cache-Control" content="no-cache, no-store, must-revalidate">
    <meta http-equiv="Pragma" content="no-cache">
    <meta http-equiv="Expires" content="0">
    <title>Panadite Academy - Get ahead of the curve</title>
    <!-- Preload critical hero image for faster loading -->
    <link rel="preload" href="/landing_images/laptop-3087585.jpg" as="image" fetchpriority="high">
    
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link rel="stylesheet" href="/assets/css/style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/animate.css/4.1.1/animate.min.css">
    <style>
        /* Clean Hero Section Styles */
        .modern-hero {
            position: relative;
            min-height: 80vh;
            display: flex;
            align-items: center;
            background: #f5f5f5; /* Whitesmoke background */
            overflow: hidden;
        }

        .hero-background {
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            z-index: 1;
        }

        .floating-icons {
            position: absolute;
            width: 100%;
            height: 100%;
            z-index: 2;
        }

        .floating-icon-card {
            position: absolute;
            width: 40px;
            height: 40px;
            background: transparent;
            border-radius: 8px;
            display: flex;
            align-items: center;
            justify-content: center;
            opacity: 0.1; /* Very faint icons */
        }

        .floating-icon-card i {
            font-size: 16px;
            color: #666;
        }

        .icon-1 {
            top: 20%;
            left: 10%;
        }

        .icon-2 {
            top: 30%;
            right: 15%;
        }

        .icon-3 {
            bottom: 35%;
            left: 15%;
        }

        .icon-4 {
            bottom: 25%;
            right: 25%;
        }

        .icon-5 {
         /* FORCE HIDE OLD CATEGORIES SECTION AND SHOW NEW ONE */
        section:has([class*="categor"]) {
            display: none !important;
        }
        
        section[class*="bg-light"] {
            display: none !important;
        }
        
        div:has([class*="nav-pill"]) {
            display: none !important;
        }
        
        /* Force show my new categories section */
        div[style*="linear-gradient(135deg, #667eea 0%, #764ba2 100%)"] {
            display: block !important;
            visibility: visible !important;
            position: relative !important;
            z-index: 9999 !important;
        }
        }

        .category-tab {
            background: white !important;
            color: #6a6f73 !important;
            border: 1px solid #e8e8e8 !important;
            padding: 12px 20px !important;
            border-radius: 25px !important;
            font-weight: 600 !important;
            font-size: 14px !important;
            white-space: nowrap !important;
            flex-shrink: 0 !important;
            transition: all 0.3s ease !important;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1) !important;
            transform: none !important;
            margin: 0 !important;
        }

        .category-tab.active {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%) !important;
            color: white !important;
            border: 1px solid transparent !important;
            box-shadow: 0 4px 15px rgba(102, 126, 234, 0.4) !important;
            transform: translateY(-2px) !important;
        }

        .category-tab:hover {
            transform: translateY(-1px) !important;
            box-shadow: 0 6px 20px rgba(0,0,0,0.15) !important;
        }

        .category-tab.active:hover {
            transform: translateY(-3px) !important;
        }

        .course-card {
            width: 220px !important;
            height: 320px !important;
            background: white !important;
            border: none !important;
            border-radius: 12px !important;
            overflow: hidden !important;
            transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1) !important;
            box-shadow: 0 4px 20px rgba(0,0,0,0.08) !important;
            display: flex !important;
            flex-direction: column !important;
            cursor: pointer !important;
        }

        .course-card:hover {
            transform: translateY(-8px) !important;
            box-shadow: 0 12px 40px rgba(0,0,0,0.15) !important;
        }

        .course-slide {
            min-width: 220px !important;
            max-width: 220px !important;
            width: 220px !important;
            flex-shrink: 0 !important;
            flex-grow: 0 !important;
        }

        .carousel-nav-btn {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%) !important;
            border: none !important;
            border-radius: 50% !important;
            width: 44px !important;
            height: 44px !important;
            display: flex !important;
            align-items: center !important;
            justify-content: center !important;
            cursor: pointer !important;
            box-shadow: 0 4px 15px rgba(102, 126, 234, 0.4) !important;
            transition: all 0.3s ease !important;
            opacity: 0.9 !important;
        }

        .carousel-nav-btn:hover {
            transform: scale(1.05) !important;
            opacity: 1 !important;
        }

        .carousel-prev {
            position: absolute !important;
            left: -15px !important;
            top: 50% !important;
            transform: translateY(-50%) !important;
            z-index: 10 !important;
        }

        .carousel-next {
            position: absolute !important;
            right: -15px !important;
            top: 50% !important;
            transform: translateY(-50%) !important;
            z-index: 10 !important;
        }

        .carousel-prev:hover {
            transform: translateY(-50%) scale(1.05) !important;
        }

        .carousel-next:hover {
            transform: translateY(-50%) scale(1.05) !important;
        }

        .container {
            position: relative;
            z-index: 3;
        }

        .min-vh-80 {
            min-height: 80vh;
        }

        .hero-content-modern {
            padding: 2rem 0;
        }

        .hero-main-card {
            background: white;
            padding: 2.5rem 2rem;
            border-radius: 8px;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.08);
            max-width: 500px;
        }

        .hero-title {
            font-size: 2.8rem;
            font-weight: 700;
            line-height: 1.2;
            margin-bottom: 1.5rem;
            color: #2d3748;
        }

        .gradient-text {
            color: #276BE4;
        }

        .hero-description {
            font-size: 1.1rem;
            line-height: 1.6;
            color: #666;
            margin-bottom: 2rem;
        }

        .hero-cta-buttons {
            display: flex;
            gap: 1rem;
            flex-wrap: wrap;
        }

        .btn-modern-primary {
            background: #276BE4;
            border: none;
            color: white;
            padding: 0.875rem 1.75rem;
            font-size: 1rem;
            font-weight: 600;
            border-radius: 6px;
            text-decoration: none;
            transition: background 0.2s ease;
        }

        .btn-modern-primary:hover {
            background: #1e5bbf;
            color: white;
        }

        .btn-modern-secondary {
            background: transparent;
            border: 2px solid #276BE4;
            color: #276BE4;
            padding: 0.875rem 1.75rem;
            font-size: 1rem;
            font-weight: 600;
            border-radius: 6px;
            text-decoration: none;
            transition: all 0.2s ease;
        }

        .btn-modern-secondary:hover {
            background: #276BE4;
            color: white;
        }

        .hero-visual {
            display: flex;
            justify-content: center;
            align-items: center;
            height: 100%;
        }

        .hero-image-container {
            max-width: 450px;
            width: 100%;
        }

        .hero-image {
            width: 100%;
            height: auto;
            border-radius: 8px;
            box-shadow: 0 8px 24px rgba(0, 0, 0, 0.12);
        }

        /* Responsive Design */
        @media (max-width: 991px) {
            .hero-main-card {
                max-width: 100%;
                margin-bottom: 3rem;
            }
            
            .hero-title {
                font-size: 2.5rem;
            }
            
            .professional-figure {
                max-width: 400px;
            }
            
            .figure-bg-circle {
                width: 280px;
                height: 280px;
            }
            
            .professional-avatar {
                width: 240px;
                height: 240px;
            }
        }

        @media (max-width: 768px) {
            .modern-hero {
                min-height: 70vh;
                padding: 2rem 0;
            }
            
            .hero-title {
                font-size: 2rem;
            }
            
            .hero-description {
                font-size: 1.1rem;
            }
            
            .hero-cta-buttons {
                flex-direction: column;
            }
            
            .btn-modern-primary,
            .btn-modern-secondary {
                width: 100%;
                text-align: center;
            }
            
            .floating-icon-card {
                width: 40px;
                height: 40px;
            }
            
            .floating-icon-card i {
                font-size: 16px;
            }
            
            .skill-badge {
                display: none;
            }
        }

        /* Enhanced Floating Education Icons */
        .floating-icon {
            position: absolute;
            z-index: 1;
            pointer-events: none;
            transition: all 1.5s ease;
            color: rgba(100, 100, 100, 0.08); /* Default very light color */
            text-shadow: 0 0 10px rgba(100, 100, 100, 0.05);
        }
        
        /* Modern Search Form Styles */
        .hero-search-container {
            max-width: 600px;
            margin: 0 auto;
        }

        .search-input-group {
            display: flex;
            border-radius: 12px;
            overflow: hidden;
            box-shadow: 0 8px 20px rgba(0, 0, 0, 0.1);
            transition: all 0.3s ease;
            background: #fff;
            position: relative;
        }

        .search-input-group:hover {
            box-shadow: 0 10px 25px rgba(39, 107, 228, 0.15);
            transform: translateY(-2px);
        }

        .search-input {
            padding: 16px 20px;
            border: none;
            font-size: 16px;
            background: white;
            flex-grow: 1;
            transition: all 0.3s;
            border-radius: 12px 0 0 12px;
        }

        .search-input:focus {
            outline: none;
            box-shadow: none;
        }

        .search-input::placeholder {
            color: #aab7c4;
            font-weight: 400;
        }

        .search-button {
            padding: 0 25px;
            background-color: #276BE4;
            border: none;
            color: white;
            font-weight: 500;
            font-size: 16px;
            border-radius: 0 12px 12px 0;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            justify-content: center;
        }

        .search-button:hover {
            background-color: #1854c4;
            box-shadow: 0 4px 10px rgba(39, 107, 228, 0.3);
        }

        .search-tags {
            display: flex;
            align-items: center;
            gap: 10px;
            padding: 5px 0;
            font-size: 14px;
            color: rgba(255, 255, 255, 0.8);
        }

        .search-tag {
            color: rgba(255, 255, 255, 0.9);
            text-decoration: none;
            padding: 3px 12px;
            border-radius: 20px;
            background-color: rgba(255, 255, 255, 0.15);
            transition: all 0.2s ease;
            font-size: 13px;
            font-weight: 500;
        }

        .search-tag:hover {
            background-color: rgba(255, 255, 255, 0.25);
            color: white;
            transform: translateY(-1px);
        }

        .search-form {
            position: relative;
            z-index: 10;
            margin: 1.5rem 0;
        }

        @media (max-width: 768px) {
            .hero-search-container {
                width: 90%;
                max-width: 450px;
            }
            
            .search-input-group {
                display: flex;
                flex-direction: row;
                border-radius: 10px;
                overflow: hidden;
            }
            
            .search-input {
                border-radius: 10px 0 0 10px;
                font-size: 14px;
                padding: 14px 15px;
                width: 70%;
            }
            
            .search-button {
                border-radius: 0 10px 10px 0;
                padding: 0 15px;
                font-size: 14px;
                width: 30%;
                white-space: nowrap;
            }
            
            .search-tags {
                flex-wrap: wrap;
                justify-content: center;
                margin-top: 8px;
                gap: 8px;
            }
            
            .search-tag {
                padding: 2px 10px;
                font-size: 12px;
            }
        }
        
        .floating-icon.light {
            color: rgba(255, 255, 255, 0.2); /* Much lighter */
            text-shadow: 0 0 15px rgba(255, 255, 255, 0.15);
        }
        
        .floating-icon.accent {
            color: rgba(255, 169, 21, 0.15); /* Much lighter */
            text-shadow: 0 0 10px rgba(255, 169, 21, 0.1);
        }
        
        .floating-icon.primary {
            color: rgba(39, 107, 228, 0.12); /* Much lighter */
            text-shadow: 0 0 5px rgba(39, 107, 228, 0.05);
        }
        
        .hero-icon-container {
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            overflow: hidden;
            z-index: 1;
        }
        
        /* Footer styling */
        footer {
            position: relative;
        }
        
        footer::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 4px;
            background: linear-gradient(to right, #276BE4, #FFA915);
        }
        
        footer h5 {
            position: relative;
        }
        
        footer h5::after {
            content: '';
            position: absolute;
            left: 0;
            bottom: 0;
            width: 40px;
            height: 2px;
            background-color: #FFA915;
        }
        
        footer a {
            transition: all 0.3s ease;
        }
        
        footer a:hover {
            color: #276BE4 !important;
            padding-left: 5px;
        }
        
        footer .social-icons a {
            display: inline-block;
            width: 36px;
            height: 36px;
            background-color: rgba(39, 107, 228, 0.1);
            text-align: center;
            line-height: 36px;
            border-radius: 50%;
            transition: all 0.3s ease;
        }
        
        footer .social-icons a:hover {
            background-color: #276BE4;
            color: white !important;
            transform: translateY(-3px);
            padding-left: 0;
        }
        
        /* Category cards */
        .category-card {
            transition: all 0.3s ease;
            border-radius: 12px;
            overflow: hidden;
        }
        
        .category-card:hover {
            transform: translateY(-7px);
            box-shadow: 0 10px 20px rgba(0, 0, 0, 0.1) !important;
        }
        
        .category-icon {
            display: inline-block;
            background: rgba(39, 107, 228, 0.1);
            width: 60px;
            height: 60px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            margin-bottom: 1rem;
        }
        
        /* Removed old navbar styling as we're using the include now */
        
        /* Modern clean hero section - Minimalist design */
        .hero {
            background: #ffffff;
            color: #333;
            padding: 80px 0 70px;
            position: relative;
            overflow: hidden;
            margin-bottom: 1.5rem;
            border-bottom: 1px solid rgba(0,0,0,0.02);
        }
        
        .hero::after {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            width: 100px;
            height: 3px;
            background: #276BE4;
            border-radius: 0 0 2px 0;
        }
        
        .hero-overlay {
            position: absolute;
            top: 0;
            right: 0;
            height: 100%;
            width: 40%;
            background: linear-gradient(135deg, rgba(39, 107, 228, 0.01) 0%, rgba(39, 107, 228, 0.03) 100%);
            clip-path: polygon(25% 0, 100% 0, 100% 100%, 5% 100%);
            z-index: 1;
        }
        
        .hero-content {
            position: relative;
            z-index: 3;
        }
        
        .hero-left {
            position: relative;
            z-index: 4;
        }
        
        .hero h1 {
            font-weight: 700;
            margin-bottom: 1.2rem;
            font-size: 2.6rem;
            color: #333;
            line-height: 1.2;
            letter-spacing: -0.5px;
        }
        
        .hero h1 span {
            color: #276BE4;
            position: relative;
            display: inline-block;
        }
        
        .hero p {
            font-size: 1.1rem;
            margin-bottom: 2rem;
            max-width: 520px;
            color: #666;
            line-height: 1.6;
            font-weight: 400;
            opacity: 0.9;
        }
        
        .btn-hero {
            padding: 12px 24px;
            font-weight: 500;
            border-radius: 8px;
            letter-spacing: 0.3px;
            transition: all 0.3s ease;
            margin: 5px 10px 5px 0;
            position: relative;
            z-index: 1;
            box-shadow: 0 4px 10px rgba(39, 107, 228, 0.05);
        }
        
        .btn-hero::after {
            content: '';
            position: absolute;
            bottom: 0;
            left: 0;
            width: 100%;
            height: 0;
            background: rgba(0,0,0,0.1);
            z-index: -1;
            transition: all 0.4s ease;
        }
        
        .btn-hero:hover::after {
            height: 100%;
        }
        
        .btn-hero.btn-primary {
            background-color: #276BE4;
            border-color: #276BE4;
            box-shadow: 0 4px 10px rgba(39, 107, 228, 0.15);
        }
        
        .btn-hero.btn-secondary {
            background-color: #FFA915;
            border-color: #FFA915;
            box-shadow: 0 4px 10px rgba(255, 169, 21, 0.3);
            box-shadow: 0 4px 15px rgba(255, 169, 21, 0.3);
        }
        
        .btn-hero.btn-outline {
            background-color: transparent;
            border: 1px solid #276BE4;
            color: #276BE4;
        }
        
        .btn-hero:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 15px rgba(39, 107, 228, 0.12);
        }
        
        /* Modern search input styles */
        .search-container {
            margin-bottom: 1.5rem;
            max-width: 520px;
        }
        
        .hero-search-form {
            width: 100%;
        }
        
        .search-input {
            height: 46px;
            border-radius: 8px 0 0 8px;
            border: 1px solid rgba(39, 107, 228, 0.15);
            box-shadow: 0 2px 6px rgba(0, 0, 0, 0.03);
            padding-left: 15px;
            font-size: 0.95rem;
            background-color: #ffffff;
            transition: all 0.3s ease;
        }
        
        .search-input:focus {
            border-color: rgba(39, 107, 228, 0.3);
            box-shadow: 0 2px 10px rgba(39, 107, 228, 0.08);
            outline: none;
        }
        
        .btn-search {
            background-color: #276BE4;
            border-color: #276BE4;
            color: white;
            border-radius: 0 8px 8px 0;
            padding: 0 15px;
            height: 46px;
            transition: all 0.3s ease;
        }
        
        .btn-search:hover {
            background-color: #1c5acd;
            box-shadow: 0 4px 10px rgba(39, 107, 228, 0.15);
        }
        
        .hero-buttons {
            display: flex;
            flex-wrap: wrap;
        }
        
        .hero-illustration {
            position: relative;
            height: 100%;
            display: flex;
            align-items: center;
            justify-content: center;
        }
        
        .hero-image {
            position: relative;
            z-index: 2;
            max-width: 100%;
            max-height: 300px;
            filter: drop-shadow(0 8px 15px rgba(0, 0, 0, 0.05));
            animation: gentleFloat 10s ease-in-out infinite;
            transform-origin: center center;
            margin: 0 auto;
            display: block;
        }
        
        @keyframes gentleFloat {
            0% { transform: translateY(0px); }
            50% { transform: translateY(-8px); }
            100% { transform: translateY(0px); }
        }
        
        /* Video card enhancements */
        .video-card {
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.05);
            border-radius: 12px;
            overflow: hidden;
            transition: all 0.3s ease;
            height: 100%;
            position: relative;
            background-color: #fff;
        }
        
        .video-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 15px 30px rgba(39, 107, 228, 0.15);
        }
        
        .video-thumbnail {
            position: relative;
            overflow: hidden;
        }
        
        .svg-bg {
            width: 100%;
            aspect-ratio: 16 / 9;
            display: flex;
            align-items: center;
            justify-content: center;
            background: linear-gradient(45deg, #f8fafc, #f1f5f9);
            transition: all 0.5s ease;
        }
        
        .svg-icon-container {
            width: 80px;
            height: 80px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            transition: transform 0.5s ease;
        }
        
        .svg-icon-container i {
            font-size: 40px;
            filter: drop-shadow(0 4px 6px rgba(0,0,0,0.1));
        }
        
        .video-card:hover .svg-icon-container {
            transform: scale(1.2) rotate(5deg);
        }
        
        .play-overlay {
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0, 0, 0, 0.3);
            display: flex;
            align-items: center;
            justify-content: center;
            opacity: 0;
            transition: opacity 0.3s ease;
        }
        
        .play-overlay i {
            text-shadow: 0 2px 10px rgba(0, 0, 0, 0.3);
            transform: scale(0.8);
            transition: transform 0.3s ease;
        }
        
        .video-card:hover .play-overlay {
            opacity: 1;
        }
        
        .video-card:hover .play-overlay i {
            transform: scale(1);
        }
        
        .video-info {
            padding: 15px;
            position: relative;
        }
        
        .video-title {
            font-weight: 600;
            margin-bottom: 0.5rem;
            font-size: 1.1rem;
            line-height: 1.4;
            color: #333;
        }
        
        .video-meta {
            font-size: 0.85rem;
            color: #6c757d;
        }
        
        .video-duration {
            border-radius: 20px;
            font-size: 0.75rem;
            font-weight: 500;
            z-index: 2;
        }
        
        /* Section headings */
        .section-heading {
            position: relative;
            margin-bottom: 2.5rem;
            font-weight: 700;
        }
        
        .section-heading::after {
            content: '';
            position: absolute;
            left: 0;
            bottom: -12px;
            width: 50px;
            height: 4px;
            background-color: #FFA915;
            border-radius: 2px;
        }
        
        /* Become teacher section */
        .become-teacher {
            background-color: #f8f9fa;
            border-radius: 15px;
            padding: 3rem 0;
            margin: 3rem 0;
            position: relative;
            overflow: hidden;
        }
        
        /* Custom animations */
        .fade-in-up {
            animation: fadeInUp 1s ease-out;
        }
        
        .fade-in {
            animation: fadeIn 1.2s ease-out;
        }
        
        /* Minimal Floating Animation Effects */
        .floating-element {
            position: absolute;
            border-radius: 50%;
            background: rgba(255,255,255,0.9);
            border: 1px solid rgba(39, 107, 228, 0.08);
            pointer-events: none;
            z-index: 2;
        }
        
        @keyframes floatCircle {
            0% { transform: translate(0, 0); }
            50% { transform: translate(0, -10px); }
            100% { transform: translate(0, 0); }
        }
        
        @keyframes pulse {
            0% { opacity: 0.4; transform: scale(1); }
            50% { opacity: 0.7; transform: scale(1.03); }
            100% { opacity: 0.4; transform: scale(1); }
        }
        
        /* FORCE HIDE EXISTING CATEGORIES SECTION AND SHOW NEW ONE */
        section:has(.nav-pills),
        section:has([id*="category"]),
        section:has([class*="bg-light"]):has([class*="categor"]),
        div:has(.nav-pills) {
            display: none !important;
            visibility: hidden !important;
        }
        
        /* Force show my new gradient categories section */
        div[style*="linear-gradient(135deg, #667eea 0%, #764ba2 100%)"] {
            display: block !important;
            visibility: visible !important;
            position: relative !important;
            z-index: 9999 !important;
        }
        
        /* Additional aggressive hiding of any categories content */
        *:has-text("Browse by Categories"),
        *:has-text("Cyber Security"),
        *:has-text("Programming"),
        *:has-text("Assessment Practitioner") {
            display: none !important;
        }
        
        /* Hide any section with bg-light that contains categories */
        section.bg-light:has(*[class*="nav"]) {
            display: none !important;
        }
        
        /* Animation for floating icons */
        @keyframes floatIcon {
            0%, 100% { transform: translateY(0px) rotate(0deg); }
            25% { transform: translateY(-10px) rotate(1deg); }
            50% { transform: translateY(-5px) rotate(-1deg); }
            75% { transform: translateY(-15px) rotate(0.5deg); }
        }

        /* Browse by Categories - Isolated Styles with PA prefix */
        .pa-category-browse-section {
            background: #fafbfc;
            position: relative;
        }

        .pa-category-header {
            margin-bottom: 2rem;
        }

        .pa-category-main-title {
            font-size: 2.2rem;
            font-weight: 700;
            color: #2563eb;
            margin-bottom: 0.5rem;
        }

        .pa-category-subtitle {
            font-size: 1rem;
            color: #64748b;
            margin: 0;
            font-weight: 400;
        }

        .pa-category-tabs-wrapper {
            margin-bottom: 1.5rem;
        }

        .pa-category-tabs-nav {
            display: flex;
            gap: 0.5rem;
            justify-content: center;
            padding: 0.4rem;
            background: white;
            border-radius: 8px;
            box-shadow: 0 1px 4px rgba(0, 0, 0, 0.06);
            border: 1px solid #e2e8f0;
            overflow-x: auto;
            scroll-behavior: smooth;
            scrollbar-width: none;
            -ms-overflow-style: none;
        }

        .pa-category-tabs-nav::-webkit-scrollbar {
            display: none;
        }

        .pa-category-tab-btn {
            background: transparent;
            border: none;
            padding: 0.6rem 1rem;
            border-radius: 6px;
            font-size: 0.85rem;
            font-weight: 500;
            color: #64748b;
            cursor: pointer;
            transition: all 0.2s ease;
            display: flex;
            align-items: center;
            gap: 0.4rem;
            white-space: nowrap;
        }

        .pa-category-tab-btn:hover {
            background: #f8fafc;
            color: #2563eb;
        }

        .pa-category-tab-btn.pa-active {
            background: #2563eb;
            color: white;
            box-shadow: 0 2px 6px rgba(37, 99, 235, 0.2);
        }

        .pa-category-count {
            background: rgba(255, 255, 255, 0.25);
            padding: 0.15rem 0.4rem;
            border-radius: 10px;
            font-size: 0.7rem;
            font-weight: 600;
            min-width: 1.2rem;
            text-align: center;
        }

        .pa-category-tab-btn:not(.pa-active) .pa-category-count {
            background: #f59e0b;
            color: white;
        }

        .pa-category-content-wrapper {
            position: relative;
        }

        .pa-category-panel {
            display: none;
            animation: paFadeIn 0.3s ease-in-out;
        }

        .pa-category-panel.pa-active {
            display: block;
        }

        @keyframes paFadeIn {
            from {
                opacity: 0;
                transform: translateY(5px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }

        .pa-category-scroll-container {
            position: relative;
            margin: 0 20px;
        }

        .pa-category-courses-scroll {
            display: flex;
            overflow-x: auto;
            scroll-behavior: smooth;
            gap: 1rem;
            padding: 10px 0;
            scrollbar-width: none;
            -ms-overflow-style: none;
        }

        .pa-category-courses-scroll::-webkit-scrollbar {
            display: none;
        }

        .pa-scroll-btn {
            position: absolute;
            top: 50%;
            transform: translateY(-50%);
            z-index: 10;
            background: white;
            border: 1px solid #e2e8f0;
            border-radius: 50%;
            width: 40px;
            height: 40px;
            display: flex;
            align-items: center;
            justify-content: center;
            cursor: pointer;
            box-shadow: 0 2px 8px rgba(0, 0, 0, 0.1);
            transition: all 0.2s ease;
            color: #2563eb;
        }

        .pa-scroll-btn:hover {
            background: #2563eb;
            color: white;
            transform: translateY(-50%) scale(1.1);
        }

        .pa-category-course-card {
            background: white;
            border-radius: 8px;
            overflow: hidden;
            box-shadow: 0 2px 6px rgba(0, 0, 0, 0.06);
            border: 1px solid #e2e8f0;
            transition: all 0.2s ease;
            min-width: 220px;
            max-width: 220px;
            width: 220px;
            flex-shrink: 0;
            display: flex;
            flex-direction: column;
        }

        .pa-category-course-card:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 16px rgba(0, 0, 0, 0.12);
            border-color: #2563eb;
        }

        .pa-course-thumbnail {
            position: relative;
            height: 120px;
            overflow: hidden;
            background: #f8fafc;
        }

        .pa-course-thumbnail img {
            width: 100%;
            height: 100%;
            object-fit: cover;
            transition: transform 0.2s ease;
        }

        .pa-category-course-card:hover .pa-course-thumbnail img {
            transform: scale(1.03);
        }

        .pa-course-overlay {
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0, 0, 0, 0.5);
            display: flex;
            align-items: center;
            justify-content: center;
            opacity: 0;
            transition: opacity 0.2s ease;
        }

        .pa-category-course-card:hover .pa-course-overlay {
            opacity: 1;
        }

        .pa-course-actions {
            display: flex;
            gap: 0.5rem;
        }

        .pa-course-view-btn,
        .pa-course-cart-btn {
            width: 36px;
            height: 36px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            text-decoration: none;
            transition: all 0.2s ease;
            font-size: 0.9rem;
            border: none;
            cursor: pointer;
        }

        .pa-course-view-btn {
            background: white;
            color: #2563eb;
        }

        .pa-course-view-btn:hover {
            background: #f8fafc;
            transform: scale(1.1);
        }

        .pa-course-cart-btn {
            background: #f59e0b;
            color: white;
        }

        .pa-course-cart-btn:hover {
            background: #d97706;
            transform: scale(1.1);
        }

        .pa-course-content {
            padding: 0.9rem;
            flex: 1;
            display: flex;
            flex-direction: column;
        }

        .pa-course-level-badge {
            font-size: 0.65rem;
            font-weight: 600;
            text-transform: uppercase;
            letter-spacing: 0.3px;
            margin-bottom: 0.6rem;
            padding: 0.2rem 0.6rem;
            border-radius: 12px;
            width: fit-content;
        }

        .pa-level-beginner {
            background: rgba(34, 197, 94, 0.1);
            color: #15803d;
        }

        .pa-level-intermediate {
            background: rgba(245, 158, 11, 0.1);
            color: #d97706;
        }

        .pa-level-advanced {
            background: rgba(239, 68, 68, 0.1);
            color: #dc2626;
        }

        .pa-course-title {
            font-size: 0.95rem;
            font-weight: 600;
            color: #1a202c;
            margin: 0 0 0.4rem 0;
            line-height: 1.3;
            display: -webkit-box;
            -webkit-line-clamp: 2;
            -webkit-box-orient: vertical;
            overflow: hidden;
            min-height: 2.4rem;
        }

        .pa-course-instructor {
            font-size: 0.75rem;
            color: #64748b;
            margin-bottom: 0.8rem;
            font-weight: 400;
        }

        .pa-course-price {
            margin-top: auto;
            font-size: 0.9rem;
            font-weight: 700;
            color: #2563eb;
        }

        .pa-price-free {
            color: #22c55e;
        }

        .pa-category-empty-state {
            padding: 3rem 1rem;
            text-align: center;
        }

        .pa-empty-title {
            color: #64748b;
            font-size: 1.2rem;
            margin-bottom: 0.5rem;
        }

        .pa-empty-text {
            color: #9ca3af;
            font-size: 0.9rem;
            margin: 0;
        }

        /* Responsive Design */
        @media (max-width: 768px) {
            .pa-category-main-title {
                font-size: 1.8rem;
            }

            .pa-category-tabs-nav {
                justify-content: flex-start;
                flex-wrap: nowrap;
                padding: 0.3rem;
                margin: 0 10px;
            }

            .pa-category-tab-btn {
                flex-shrink: 0;
                padding: 0.6rem 1rem;
                font-size: 0.8rem;
            }

            .pa-category-course-card {
                min-width: 200px;
                max-width: 200px;
                width: 200px;
            }

            .pa-course-thumbnail {
                height: 110px;
            }

            .pa-scroll-btn {
                width: 36px;
                height: 36px;
            }
        }

        @media (max-width: 480px) {
            .pa-category-main-title {
                font-size: 1.6rem;
            }

            .pa-course-content {
                padding: 0.8rem;
            }

            .pa-category-course-card {
                min-width: 180px;
                max-width: 180px;
                width: 180px;
            }
        }
    </style>
    
    <!-- Typing Animation Styles -->
    <style>
        /* Typing Animation for Hero Text */
        .pa-typing-text {
            position: relative;
            display: inline-block;
            text-decoration: none !important;
        }

        .pa-typed-text {
            color: #f59e0b !important;
            font-weight: 700;
            text-decoration: none !important;
        }

        /* Override any underline from parent elements */
        .pa-hero-title .pa-typing-text,
        .pa-hero-title .pa-typed-text {
            text-decoration: none !important;
            border-bottom: none !important;
        }

        .pa-typing-cursor {
            color: #f59e0b;
            font-weight: 400;
            animation: paBlinkCursor 1s infinite;
        }

        @keyframes paBlinkCursor {
            0%, 50% { opacity: 1; }
            51%, 100% { opacity: 0; }
        }

        .pa-typing-complete .pa-typing-cursor {
            display: none;
        }

        /* Why Panadite Academy Section */
        .why-panadite-section {
            position: relative;
            overflow: hidden;
        }

        .why-title {
            font-size: 2.2rem;
            font-weight: 700;
            color: #1a202c;
            line-height: 1.2;
            margin-bottom: 1rem;
            letter-spacing: -0.01em;
        }

        .brand-highlight {
            color: #f59e0b;
            position: relative;
            font-weight: 700;
        }

        .why-description {
            font-size: 1rem;
            color: #475569;
            line-height: 1.6;
            margin-bottom: 1.5rem;
            font-weight: 400;
            max-width: 95%;
        }

        .features-list {
            margin-top: 1rem;
        }

        .feature-item {
            background: white;
            padding: 1.2rem;
            border-radius: 12px;
            box-shadow: 0 2px 8px rgba(0, 0, 0, 0.06);
            border: 1px solid #f1f5f9;
            transition: all 0.3s ease;
            margin-bottom: 1rem;
            position: relative;
            overflow: hidden;
        }

        .feature-item::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            width: 4px;
            height: 100%;
            background: #f59e0b;
            transition: width 0.3s ease;
        }

        .feature-item:hover {
            transform: translateY(-4px);
            box-shadow: 0 12px 32px rgba(0, 0, 0, 0.12);
        }

        .feature-item:hover::before {
            width: 8px;
        }

        .feature-icon {
            width: 40px;
            height: 40px;
            background: #2563eb;
            border-radius: 10px;
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-size: 1rem;
            flex-shrink: 0;
            margin-bottom: 0.5rem;
        }

        .feature-content {
            padding-left: 0;
        }

        .feature-title {
            font-size: 1.1rem;
            font-weight: 600;
            color: #1a202c;
            margin-bottom: 0.5rem;
            line-height: 1.3;
        }

        .feature-title .text-muted {
            font-size: 0.85rem;
            font-weight: 400;
            color: #64748b;
        }

        .feature-text {
            font-size: 0.9rem;
            color: #64748b;
            line-height: 1.5;
            margin: 0;
            font-weight: 400;
        }

        .why-content {
            padding-right: 2rem;
        }

        /* Right Content Layout */
        .right-content-wrapper {
            display: flex;
            flex-direction: column;
            gap: 1.5rem;
        }

        .side-feature {
            margin-top: 0;
            margin-bottom: 0;
        }

        /* Visual Mockup Styles */
        .why-visual-wrapper {
            position: relative;
            height: 400px;
        }

        .platform-mockup {
            position: relative;
            width: 100%;
            height: 100%;
        }

        .mockup-bg-circle {
            display: none; /* Completely hide the background circle */
        }

        .laptop-mockup {
            position: relative;
            z-index: 2;
            max-width: 380px;
            margin: 0 auto;
            top: 50%;
            transform: translateY(-50%);
        }

        .laptop-screen {
            background: #1f2937;
            border-radius: 8px 8px 0 0;
            padding: 4px;
            border: 2px solid #374151;
        }

        /* Dashboard Header */
        .dashboard-header {
            background: #f8f9fa;
            border-radius: 6px 6px 0 0;
            padding: 8px 12px;
            border-bottom: 1px solid #e5e7eb;
        }

        .dashboard-nav {
            display: flex;
            align-items: center;
            justify-content: space-between;
        }

        .logo-section {
            display: flex;
            align-items: center;
            gap: 6px;
        }

        .dashboard-logo {
            width: 20px;
            height: 20px;
            border-radius: 4px;
            object-fit: cover;
        }

        .dashboard-title {
            font-size: 0.75rem;
            font-weight: 600;
            color: #2563eb;
        }

        .nav-menu {
            display: flex;
            gap: 12px;
        }

        .nav-item {
            font-size: 0.7rem;
            color: #64748b;
            padding: 4px 8px;
            border-radius: 4px;
            cursor: pointer;
        }

        .nav-item.active {
            background: #2563eb;
            color: white;
        }

        /* Dashboard Content */
        .dashboard-content {
            background: #f8f9fa;
            padding: 12px;
            border-radius: 0 0 6px 6px;
            min-height: 180px;
        }

        /* Welcome Section */
        .welcome-section {
            display: flex;
            align-items: center;
            justify-content: space-between;
            margin-bottom: 12px;
            padding: 8px;
            background: white;
            border-radius: 6px;
            border: 1px solid #e5e7eb;
        }

        .welcome-text h6 {
            font-size: 0.8rem;
            font-weight: 600;
            color: #1a202c;
            margin: 0 0 2px 0;
        }

        .welcome-text p {
            font-size: 0.65rem;
            color: #64748b;
            margin: 0;
        }

        .profile-avatar {
            width: 24px;
            height: 24px;
            border-radius: 50%;
            overflow: hidden;
        }

        .profile-img {
            width: 100%;
            height: 100%;
            object-fit: cover;
        }

        /* Current Course Card */
        .current-course-card {
            display: flex;
            background: white;
            border-radius: 6px;
            padding: 8px;
            border: 1px solid #e5e7eb;
            margin-bottom: 12px;
        }

        .course-thumbnail {
            width: 48px;
            height: 36px;
            border-radius: 4px;
            overflow: hidden;
            margin-right: 8px;
            flex-shrink: 0;
        }

        .course-img {
            width: 100%;
            height: 100%;
            object-fit: cover;
        }

        .course-details {
            flex: 1;
        }

        .course-details h6 {
            font-size: 0.75rem;
            font-weight: 600;
            color: #1a202c;
            margin: 0 0 2px 0;
        }

        .course-instructor {
            font-size: 0.6rem;
            color: #64748b;
            margin: 0 0 6px 0;
        }

        .progress-section {
            margin-bottom: 6px;
        }

        .progress-bar {
            width: 100%;
            height: 4px;
            background: #f1f5f9;
            border-radius: 2px;
            overflow: hidden;
            margin-bottom: 3px;
        }

        .progress-fill {
            height: 100%;
            background: #f59e0b;
            border-radius: 2px;
        }

        .progress-label {
            font-size: 0.6rem;
            color: #64748b;
        }

        .continue-btn {
            background: #2563eb;
            color: white;
            border: none;
            padding: 3px 8px;
            border-radius: 4px;
            font-size: 0.6rem;
            font-weight: 500;
            cursor: pointer;
        }

        /* Stats Row */
        .stats-row {
            display: flex;
            gap: 8px;
        }

        .stat-item {
            flex: 1;
            background: white;
            padding: 8px;
            border-radius: 6px;
            border: 1px solid #e5e7eb;
            text-align: center;
        }

        .stat-number {
            display: block;
            font-size: 0.9rem;
            font-weight: 700;
            color: #f59e0b;
            margin-bottom: 2px;
        }

        .stat-label {
            font-size: 0.6rem;
            color: #64748b;
        }

        .laptop-base {
            height: 12px;
            background: #9ca3af;
            border-radius: 0 0 20px 20px;
            margin: 0 auto;
            width: 90%;
            position: relative;
        }

        .laptop-base::after {
            content: '';
            position: absolute;
            bottom: -4px;
            left: 50%;
            transform: translateX(-50%);
            width: 60px;
            height: 4px;
            background: #6b7280;
            border-radius: 2px;
        }

        /* Floating Badges */
        .floating-badge {
            position: absolute;
            background: white;
            padding: 12px;
            border-radius: 12px;
            box-shadow: 0 4px 16px rgba(0, 0, 0, 0.1);
            display: flex;
            align-items: center;
            gap: 8px;
            font-size: 0.75rem;
            font-weight: 600;
            z-index: 3;
            animation: floatBadge 3s ease-in-out infinite;
        }

        .badge-1 {
            top: 20%;
            right: 10%;
            color: #22c55e;
        }

        .badge-2 {
            bottom: 25%;
            left: 5%;
            color: #2563eb;
        }

        @keyframes floatBadge {
            0%, 100% { transform: translateY(0px); }
            50% { transform: translateY(-10px); }
        }

        /* Responsive Design */
        @media (max-width: 768px) {
            .why-title {
                font-size: 2rem;
            }

            .why-visual-wrapper {
                height: 400px;
                margin-top: 2rem;
            }

            .laptop-mockup {
                max-width: 320px;
            }

            .mockup-bg-circle {
                width: 300px;
                height: 300px;
            }
        }

        /* Referral Program Carousel Styles */
        .referral-carousel-section {
            position: relative;
            overflow: hidden;
            padding: 2rem 0 !important;
        }

        .referral-main-title {
            font-size: 1.6rem;
            font-weight: 700;
            color: #1a202c;
            margin-bottom: 0.3rem;
        }

        .highlight-amber {
            color: #f59e0b;
            font-weight: 800;
        }

        .referral-subtitle {
            font-size: 0.9rem;
            color: #64748b;
            margin-bottom: 0;
        }

        .referral-carousel-container {
            max-width: 1000px;
            margin: 0 auto;
            position: relative;
        }

        .carousel-wrapper {
            background: white;
            border-radius: 16px;
            box-shadow: 0 4px 20px rgba(0, 0, 0, 0.08);
            padding: 2rem 1.5rem;
            position: relative;
            overflow: hidden;
        }

        .carousel-slides {
            position: relative;
            width: 100%;
        }

        .carousel-slide {
            display: none;
            opacity: 0;
            transition: opacity 0.5s ease-in-out;
        }

        .carousel-slide.active {
            display: block;
            opacity: 1;
        }

        .slide-content {
            padding-right: 1rem;
        }

        .slide-badge {
            display: inline-block;
            background: #2563eb;
            color: white;
            padding: 0.3rem 0.6rem;
            border-radius: 12px;
            font-size: 0.65rem;
            font-weight: 600;
            margin-bottom: 0.6rem;
        }

        .slide-title {
            font-size: 1.2rem;
            font-weight: 700;
            color: #1a202c;
            margin-bottom: 0.5rem;
            line-height: 1.2;
        }

        .amount-highlight {
            color: #f59e0b;
            font-weight: 800;
        }

        .slide-description {
            font-size: 0.85rem;
            color: #64748b;
            line-height: 1.4;
            margin-bottom: 1rem;
        }

        .benefits-list, .process-steps {
            margin-bottom: 1rem;
        }

        .benefit-item, .step-item {
            display: flex;
            align-items: center;
            margin-bottom: 0.6rem;
            padding: 0.4rem 0;
        }

        .benefit-item i {
            width: 16px;
            height: 16px;
            background: #f59e0b;
            color: white;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 0.6rem;
            margin-right: 0.8rem;
            flex-shrink: 0;
        }

        .step-number {
            width: 24px;
            height: 24px;
            background: #2563eb;
            color: white;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: 700;
            margin-right: 0.8rem;
            flex-shrink: 0;
            font-size: 0.8rem;
        }

        .step-content {
            display: flex;
            flex-direction: column;
        }

        .step-content strong {
            color: #1a202c;
            margin-bottom: 0.15rem;
            font-size: 0.85rem;
        }

        .step-content span {
            color: #64748b;
            font-size: 0.75rem;
        }

        .cta-button {
            background: linear-gradient(135deg, #2563eb, #3b82f6);
            color: white;
            border: none;
            padding: 0.6rem 1.2rem;
            border-radius: 8px;
            font-size: 0.8rem;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            box-shadow: 0 2px 8px rgba(37, 99, 235, 0.3);
        }

        .cta-button:hover {
            transform: translateY(-1px);
            box-shadow: 0 4px 12px rgba(37, 99, 235, 0.4);
        }

        /* Graphics Styles */
        .slide-graphic {
            display: flex;
            align-items: center;
            justify-content: center;
            height: 220px;
        }

        .graphic-container {
            position: relative;
            width: 100%;
            height: 100%;
            display: flex;
            align-items: center;
            justify-content: center;
        }

        /* Teacher Referral Graphics */
        .teacher-avatar, .student-avatar {
            background: white;
            border: 2px solid #e5e7eb;
            border-radius: 12px;
            padding: 1rem;
            text-align: center;
            box-shadow: 0 2px 8px rgba(0, 0, 0, 0.08);
            position: relative;
        }

        .teacher-avatar i, .student-avatar i {
            font-size: 1.4rem;
            color: #2563eb;
            margin-bottom: 0.3rem;
        }

        .student-avatar i {
            color: #f59e0b;
        }

        .teacher-name, .student-name {
            font-weight: 700;
            color: #1a202c;
            font-size: 0.8rem;
        }

        .teacher-role, .student-progress {
            color: #64748b;
            font-size: 0.65rem;
        }

        .main-teacher, .main-student {
            margin-right: 1.5rem;
        }

        .referred-teacher, .referred-student {
            margin-left: 1.5rem;
        }

        .referral-arrow {
            color: #f59e0b;
            font-size: 1.2rem;
            animation: bounce 2s infinite;
        }

        .earning-popup, .discount-popup {
            position: absolute;
            background: #22c55e;
            color: white;
            padding: 0.3rem 0.6rem;
            border-radius: 12px;
            font-size: 0.65rem;
            font-weight: 600;
            box-shadow: 0 2px 6px rgba(34, 197, 94, 0.3);
            animation: pulse 2s infinite;
        }

        .earning-popup {
            top: -15px;
            right: -15px;
        }

        .discount-popup {
            background: #f59e0b;
            box-shadow: 0 2px 6px rgba(245, 158, 11, 0.3);
        }

        .discount-popup.emma {
            top: -15px;
            right: -15px;
        }

        .discount-popup.alex {
            top: -15px;
            left: -15px;
            background: #22c55e;
            box-shadow: 0 2px 6px rgba(34, 197, 94, 0.3);
        }

        .course-icon {
            position: absolute;
            bottom: -20px;
            left: 50%;
            transform: translateX(-50%);
            background: #2563eb;
            color: white;
            padding: 0.3rem 0.6rem;
            border-radius: 8px;
            font-size: 0.65rem;
            display: flex;
            align-items: center;
            gap: 0.3rem;
        }

        /* Process Graphics */
        .process-flow {
            display: flex;
            flex-direction: column;
            align-items: center;
            gap: 0.8rem;
        }

        .flow-item {
            text-align: center;
        }

        .flow-icon {
            width: 45px;
            height: 45px;
            background: #2563eb;
            color: white;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 1.1rem;
            margin: 0 auto 0.3rem;
        }

        .flow-text {
            color: #1a202c;
            font-weight: 600;
            font-size: 0.8rem;
        }

        .flow-arrow {
            color: #f59e0b;
            font-size: 1rem;
        }

        .success-celebration {
            position: absolute;
            top: 20px;
            right: 20px;
            display: flex;
            gap: 0.25rem;
        }

        .success-celebration i {
            color: #fbbf24;
            font-size: 1rem;
            animation: twinkle 1.5s infinite alternate;
        }

        /* Carousel Navigation */
        .carousel-nav {
            display: flex;
            align-items: center;
            justify-content: center;
            margin-top: 1.5rem;
        }

        .nav-btn {
            position: absolute;
            top: 50%;
            transform: translateY(-50%);
            width: 32px;
            height: 32px;
            background: #2563eb;
            color: white;
            border: none;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            cursor: pointer;
            transition: all 0.3s ease;
            font-size: 0.75rem;
            box-shadow: 0 1px 4px rgba(37, 99, 235, 0.3);
            z-index: 10;
        }

        .nav-btn:hover {
            background: #1d4ed8;
            transform: translateY(-50%) scale(1.1);
            box-shadow: 0 4px 12px rgba(37, 99, 235, 0.4);
        }

        .prev-btn {
            left: -20px;
        }

        .next-btn {
            right: -20px;
        }

        .carousel-dots {
            display: flex;
            gap: 0.5rem;
            justify-content: center;
        }

        .dot {
            width: 8px;
            height: 8px;
            border-radius: 50%;
            background: #cbd5e1;
            cursor: pointer;
            transition: all 0.3s ease;
        }

        .dot.active {
            background: #2563eb;
            transform: scale(1.3);
        }

        /* Animations */
        @keyframes bounce {
            0%, 20%, 50%, 80%, 100% { transform: translateY(0); }
            40% { transform: translateY(-10px); }
            60% { transform: translateY(-5px); }
        }

        @keyframes pulse {
            0%, 100% { transform: scale(1); }
            50% { transform: scale(1.05); }
        }

        @keyframes twinkle {
            0% { opacity: 0.5; transform: scale(0.8); }
            100% { opacity: 1; transform: scale(1.2); }
        }

        /* Responsive Design */
        @media (max-width: 768px) {
            .referral-main-title {
                font-size: 2rem;
            }

            .slide-title {
                font-size: 1.5rem;
            }

            .carousel-wrapper {
                padding: 1.5rem 1rem;
                overflow: hidden;
            }

            .slide-content {
                padding-right: 0;
                margin-bottom: 2rem;
            }

            .slide-graphic {
                height: 300px;
            }

            .main-teacher, .main-student {
                margin-right: 1rem;
            }

            .referred-teacher, .referred-student {
                margin-left: 1rem;
            }
        }
    </style>
</head>
<body>
    <!-- Floating Icons -->
    <div class="floating-icon" style="top: 15%; left: 10%; font-size: 50px;"><i class="fas fa-graduation-cap"></i></div>
    <div class="floating-icon" style="top: 25%; right: 8%; font-size: 45px;"><i class="fas fa-book"></i></div>
    <div class="floating-icon" style="bottom: 35%; left: 12%; font-size: 40px;"><i class="fas fa-laptop-code"></i></div>
    <div class="floating-icon" style="bottom: 25%; right: 10%; font-size: 35px;"><i class="fas fa-brain"></i></div>
    
    <!-- Include the modern navbar -->
    <?php include 'includes/navbar.php'; ?>
    
    <!-- Modern Hero Section - Panadite Academy -->
    <section class="pa-hero-section">
        <!-- Background Elements -->
        <div class="pa-hero-bg-elements">
            <div class="pa-bg-circle pa-bg-circle-1"></div>
            <div class="pa-bg-circle pa-bg-circle-2"></div>
            <div class="pa-bg-circle pa-bg-circle-3"></div>
        </div>
        
        <!-- Floating Educational Icons -->
        <div class="pa-floating-icons">
            <div class="pa-float-icon pa-icon-1"><i class="fas fa-graduation-cap"></i></div>
            <div class="pa-float-icon pa-icon-2"><i class="fas fa-book-open"></i></div>
            <div class="pa-float-icon pa-icon-3"><i class="fas fa-laptop-code"></i></div>
            <div class="pa-float-icon pa-icon-4"><i class="fas fa-brain"></i></div>
            <div class="pa-float-icon pa-icon-5"><i class="fas fa-chart-line"></i></div>
            <div class="pa-float-icon pa-icon-6"><i class="fas fa-lightbulb"></i></div>
        </div>
        
        <div class="container">
            <div class="row align-items-center pa-hero-content-row">
                <!-- Left Content -->
                <div class="col-lg-6 col-md-12">
                    <div class="pa-hero-content">
                        <!-- Welcome Badge -->
                        <div class="pa-welcome-badge">
                            <i class="fas fa-star"></i>
                            <span>Welcome to Panadite Academy</span>
                        </div>
                        
                        <!-- Main Heading -->
                        <h1 class="pa-hero-title">
                            Master tomorrow's<br>
                            <span class="pa-hero-highlight pa-typing-text" id="typingText">
                                <span class="pa-typed-text"></span>
                                <span class="pa-typing-cursor">|</span>
                            </span>
                        </h1>
                        
                        <!-- Description -->
                        <p class="pa-hero-description">
                            Power up your career and life skills with the most up-to-date, 
                            expert-led learning from Panadite Academy. Join thousands of learners 
                            advancing their futures.
                        </p>
                        
                        <!-- Stats Row -->
                       
                        
                        <!-- CTA Buttons -->
                        <div class="pa-hero-buttons">
                            <a href="/student/courses.php" class="pa-btn pa-btn-primary">
                                <i class="fas fa-play-circle"></i>
                                <span>Get Started</span>
                            </a>
                            <a href="/student/courses.php?category=trending" class="pa-btn pa-btn-secondary">
                                <i class="fas fa-trending-up"></i>
                                <span>Explore Trending</span>
                            </a>
                        </div>
                        
                        <!-- Trust Indicators -->
                        <div class="pa-trust-indicators">
                            <div class="pa-trust-item">
                                <i class="fas fa-shield-alt"></i>
                                <span>Certified Courses</span>
                            </div>
                            <div class="pa-trust-item">
                                <i class="fas fa-clock"></i>
                                <span>Lifetime Access</span>
                            </div>
                            <div class="pa-trust-item">
                                <i class="fas fa-medal"></i>
                                <span>Expert Instructors</span>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Right Visual -->
                <div class="col-lg-6 col-md-12">
                    <div class="pa-hero-visual">
                        <div class="pa-hero-image-container">
                            <!-- Hero Image Card -->
                            <div class="pa-hero-image-card">
                                <img src="/landing_images/laptop-3087585.jpg" 
                                     alt="Professional Learning Experience" 
                                     class="pa-hero-image"
                                     loading="eager"
                                     onerror="this.src='/landing_images/download (15).jpeg'">
                                
                                <!-- Floating Course Card -->
                                <div class="pa-floating-course-card">
                                    <div class="pa-course-icon">
                                        <i class="fas fa-code"></i>
                                    </div>
                                    <div class="pa-course-info">
                                        <div class="pa-course-title">Web Development</div>
                                        <div class="pa-course-progress">
                                            <div class="pa-progress-bar">
                                                <div class="pa-progress-fill"></div>
                                            </div>
                                            <span class="pa-progress-text">87% Complete</span>
                                        </div>
                                    </div>
                                </div>
                                
                                <!-- Floating Achievement Badge -->
                                <div class="pa-floating-achievement">
                                    <div class="pa-achievement-icon">
                                        <i class="fas fa-trophy"></i>
                                    </div>
                                    <div class="pa-achievement-text">
                                        <div>Certificate</div>
                                        <div>Earned!</div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </section>

    <!-- Hero Section Styles -->
    <style>
        /* Panadite Academy Hero Section - Isolated Styles */
        .pa-hero-section {
            background: #f8f9fa;
            position: relative;
            padding: 60px 0 50px;
            overflow: hidden;
            min-height: 60vh;
            display: flex;
            align-items: center;
        }

        /* Background Elements */
        .pa-hero-bg-elements {
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            z-index: 1;
        }

        .pa-bg-circle {
            position: absolute;
            border-radius: 50%;
            background: rgba(255, 255, 255, 0.05);
            backdrop-filter: blur(10px);
        }

        .pa-bg-circle-1 {
            width: 300px;
            height: 300px;
            top: -150px;
            right: -100px;
            background: rgba(37, 99, 235, 0.05);
        }

        .pa-bg-circle-2 {
            width: 200px;
            height: 200px;
            bottom: -100px;
            left: -50px;
            background: rgba(245, 158, 11, 0.04);
        }

        .pa-bg-circle-3 {
            width: 150px;
            height: 150px;
            top: 30%;
            left: 20%;
            background: rgba(37, 99, 235, 0.03);
        }

        /* Floating Icons */
        .pa-floating-icons {
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            z-index: 2;
            pointer-events: none;
        }

        .pa-float-icon {
            position: absolute;
            color: rgba(37, 99, 235, 0.08);
            font-size: 24px;
            animation: paFloatIcon 8s ease-in-out infinite;
        }

        .pa-icon-1 { top: 20%; left: 8%; animation-delay: 0s; }
        .pa-icon-2 { top: 10%; right: 15%; animation-delay: 1s; }
        .pa-icon-3 { bottom: 30%; left: 5%; animation-delay: 2s; }
        .pa-icon-4 { bottom: 15%; right: 8%; animation-delay: 3s; }
        .pa-icon-5 { top: 45%; left: 15%; animation-delay: 4s; }
        .pa-icon-6 { top: 60%; right: 20%; animation-delay: 5s; }

        @keyframes paFloatIcon {
            0%, 100% { transform: translateY(0px) rotate(0deg); opacity: 0.08; }
            25% { transform: translateY(-15px) rotate(2deg); opacity: 0.12; }
            50% { transform: translateY(-8px) rotate(-1deg); opacity: 0.08; }
            75% { transform: translateY(-20px) rotate(1deg); opacity: 0.1; }
        }

        /* Content Container */
        .pa-hero-content-row {
            position: relative;
            z-index: 10;
            min-height: 60vh;
        }

        /* Hero Content */
        .pa-hero-content {
            color: #1f2937;
            position: relative;
            z-index: 10;
        }

        .pa-welcome-badge {
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
            background: rgba(37, 99, 235, 0.1);
            backdrop-filter: blur(10px);
            color: #2563eb;
            padding: 0.5rem 1rem;
            border-radius: 25px;
            font-size: 0.85rem;
            font-weight: 500;
            margin-bottom: 1.5rem;
            border: 1px solid rgba(37, 99, 235, 0.2);
        }

        .pa-welcome-badge i {
            color: #f59e0b;
        }

        .pa-hero-title {
            font-size: 3.2rem;
            font-weight: 800;
            line-height: 1.2;
            margin-bottom: 1.5rem;
            color: #1f2937;
        }

        .pa-hero-highlight {
            color: #f59e0b;
            position: relative;
        }

        .pa-hero-highlight::after {
            content: '';
            position: absolute;
            bottom: 0;
            left: 0;
            right: 0;
            height: 3px;
            background: #f59e0b;
            border-radius: 2px;
        }

        .pa-hero-accent {
            color: #fbbf24;
            display: inline-block;
            animation: paGlowPulse 2s ease-in-out infinite;
        }

        @keyframes paGlowPulse {
            0%, 100% { text-shadow: 0 0 20px rgba(251, 191, 36, 0.5); }
            50% { text-shadow: 0 0 30px rgba(251, 191, 36, 0.8); }
        }

        .pa-hero-description {
            font-size: 1.1rem;
            line-height: 1.7;
            color: #6b7280;
            margin-bottom: 2rem;
            max-width: 500px;
            font-weight: 400;
        }

        /* Stats Row */
        .pa-stats-row {
            display: flex;
            gap: 2rem;
            margin-bottom: 2.5rem;
            flex-wrap: wrap;
        }

        .pa-stat-item {
            text-align: center;
        }

        .pa-stat-number {
            font-size: 1.8rem;
            font-weight: 700;
            color: #f59e0b;
            line-height: 1;
        }

        .pa-stat-label {
            font-size: 0.85rem;
            color: #6b7280;
            font-weight: 500;
            margin-top: 0.25rem;
        }

        /* Hero Buttons */
        .pa-hero-buttons {
            display: flex;
            gap: 1rem;
            margin-bottom: 2.5rem;
            flex-wrap: wrap;
        }

        .pa-btn {
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
            padding: 1rem 2rem;
            border-radius: 8px;
            font-weight: 600;
            text-decoration: none;
            transition: all 0.3s ease;
            border: none;
            font-size: 1rem;
            position: relative;
            overflow: hidden;
        }

        .pa-btn::before {
            content: '';
            position: absolute;
            top: 0;
            left: -100%;
            width: 100%;
            height: 100%;
            background: linear-gradient(90deg, transparent, rgba(255, 255, 255, 0.2), transparent);
            transition: left 0.5s ease;
        }

        .pa-btn:hover::before {
            left: 100%;
        }

        .pa-btn-primary {
            background: #f59e0b;
            color: white;
            box-shadow: 0 4px 15px rgba(245, 158, 11, 0.4);
        }

        .pa-btn-primary:hover {
            background: #d97706;
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(245, 158, 11, 0.6);
            color: white;
        }

        .pa-btn-secondary {
            background: rgba(37, 99, 235, 0.1);
            color: #2563eb;
            border: 2px solid rgba(37, 99, 235, 0.3);
            backdrop-filter: blur(10px);
        }

        .pa-btn-secondary:hover {
            background: #2563eb;
            border-color: #2563eb;
            transform: translateY(-2px);
            color: white;
        }

        /* Trust Indicators */
        .pa-trust-indicators {
            display: flex;
            gap: 1.5rem;
            flex-wrap: wrap;
        }

        .pa-trust-item {
            display: flex;
            align-items: center;
            gap: 0.5rem;
            color: #6b7280;
            font-size: 0.85rem;
            font-weight: 500;
        }

        .pa-trust-item i {
            color: #f59e0b;
            font-size: 1rem;
        }

        /* Hero Visual */
        .pa-hero-visual {
            position: relative;
            z-index: 10;
            display: flex;
            justify-content: center;
            align-items: center;
            height: 100%;
        }

        .pa-hero-image-container {
            position: relative;
            max-width: 500px;
            width: 100%;
        }

        .pa-hero-image-card {
            position: relative;
            border-radius: 20px;
            overflow: hidden;
            box-shadow: 0 20px 40px rgba(0, 0, 0, 0.3);
            background: white;
            padding: 0.5rem;
        }

        .pa-hero-image {
            width: 100%;
            height: auto;
            border-radius: 15px;
            display: block;
            transition: transform 0.3s ease;
        }

        .pa-hero-image-card:hover .pa-hero-image {
            transform: scale(1.05);
        }

        /* Floating Course Card */
        .pa-floating-course-card {
            position: absolute;
            top: 20px;
            right: -40px;
            background: white;
            border-radius: 12px;
            padding: 1rem;
            box-shadow: 0 10px 25px rgba(0, 0, 0, 0.15);
            display: flex;
            align-items: center;
            gap: 0.75rem;
            animation: paFloatUp 3s ease-in-out infinite;
            min-width: 200px;
        }

        .pa-course-icon {
            width: 40px;
            height: 40px;
            background: #2563eb;
            border-radius: 8px;
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            flex-shrink: 0;
        }

        .pa-course-info {
            flex: 1;
        }

        .pa-course-title {
            font-size: 0.9rem;
            font-weight: 600;
            color: #1f2937;
            margin-bottom: 0.25rem;
        }

        .pa-course-progress {
            display: flex;
            align-items: center;
            gap: 0.5rem;
        }

        .pa-progress-bar {
            flex: 1;
            height: 6px;
            background: #e5e7eb;
            border-radius: 3px;
            overflow: hidden;
        }

        .pa-progress-fill {
            height: 100%;
            width: 87%;
            background: #f59e0b;
            border-radius: 3px;
        }

        .pa-progress-text {
            font-size: 0.75rem;
            color: #6b7280;
            font-weight: 500;
        }

        /* Floating Achievement */
        .pa-floating-achievement {
            position: absolute;
            bottom: 30px;
            left: -30px;
            background: #10b981;
            border-radius: 12px;
            padding: 1rem;
            box-shadow: 0 10px 25px rgba(16, 185, 129, 0.3);
            display: flex;
            align-items: center;
            gap: 0.75rem;
            color: white;
            animation: paFloatUp 3s ease-in-out infinite 1.5s;
        }

        .pa-achievement-icon {
            width: 40px;
            height: 40px;
            background: rgba(255, 255, 255, 0.2);
            border-radius: 8px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 1.2rem;
        }

        .pa-achievement-text {
            font-size: 0.85rem;
            font-weight: 600;
            line-height: 1.2;
        }

        @keyframes paFloatUp {
            0%, 100% { transform: translateY(0px); }
            50% { transform: translateY(-10px); }
        }

        /* Responsive Design */
        @media (max-width: 992px) {
            .pa-hero-section {
                padding: 50px 0 40px;
                min-height: 50vh;
            }

            .pa-hero-title {
                font-size: 2.5rem;
            }

            .pa-stats-row {
                justify-content: center;
                margin-bottom: 2rem;
            }

            .pa-hero-buttons {
                justify-content: center;
            }

            .pa-floating-course-card,
            .pa-floating-achievement {
                display: none;
            }
        }

        @media (max-width: 768px) {
            .pa-hero-section {
                padding: 40px 0 30px;
                min-height: 45vh;
                text-align: center;
            }

            .pa-hero-title {
                font-size: 2.2rem;
            }

            .pa-hero-description {
                max-width: 100%;
            }

            .pa-stats-row {
                gap: 1rem;
            }

            .pa-hero-buttons {
                flex-direction: column;
                align-items: center;
            }

            .pa-btn {
                width: 100%;
                max-width: 300px;
                justify-content: center;
            }

            .pa-trust-indicators {
                justify-content: center;
                gap: 1rem;
            }

            .pa-bg-circle-1,
            .pa-bg-circle-2,
            .pa-bg-circle-3 {
                display: none;
            }
        }

        @media (max-width: 480px) {
            .pa-hero-title {
                font-size: 1.9rem;
            }

            .pa-hero-description {
                font-size: 1rem;
            }

            .pa-stats-row {
                flex-direction: column;
                gap: 0.5rem;
            }

            .pa-trust-indicators {
                flex-direction: column;
                align-items: center;
                gap: 0.75rem;
            }
        }
    </style>

    <!-- Featured Courses Section -->
    <section class="fc-featured-section py-5" style="background: white;">
        <div class="container">
            <!-- Section Header -->
            <div class="fc-header text-center mb-4">
                <h2 class="fc-main-title">Featured Courses</h2>
                <p class="fc-subtitle">Handpicked courses recommended by our experts</p>
            </div>

            <?php if (!empty($featuredCourses)): ?>
            <!-- Featured Courses Container -->
            <div class="fc-scroll-container" style="position: relative; margin: 0 20px;">
                <!-- Navigation Arrows -->
                <button class="fc-scroll-btn fc-scroll-left" onclick="fcScrollCourses('left')" style="left: -15px;">
                    <i class="fas fa-chevron-left"></i>
                </button>
                <button class="fc-scroll-btn fc-scroll-right" onclick="fcScrollCourses('right')" style="right: -15px;">
                    <i class="fas fa-chevron-right"></i>
                </button>
                
                <div class="fc-courses-scroll" id="fc-scroll-container">
                    <?php foreach ($featuredCourses as $course): 
                        $courseLevels = ['Beginner', 'Intermediate', 'Advanced'];
                        $courseLevel = $courseLevels[array_rand($courseLevels)];
                    ?>
                    <div class="fc-course-card" data-course-id="<?php echo htmlspecialchars($encrypted_id); ?>">
                        <div class="fc-course-thumbnail">
                            <?php
                            $thumbnail = $course['thumbnail'] ?? '';
                            if (strpos($thumbnail, '../') === 0) {
                                $thumbnail = substr($thumbnail, 3);
                            }
                            if (substr($thumbnail, 0, 1) !== '/') {
                                $thumbnail = '/' . $thumbnail;
                            }
                            $thumbnail = str_replace('//', '/', $thumbnail);
                            
                            if (empty($thumbnail) || $thumbnail == '/') {
                                $thumbnail = '/assets/images/default-thumbnail.jpg';
                            }
                            ?>
                            <img src="<?php echo htmlspecialchars($thumbnail); ?>" 
                                 alt="<?php echo htmlspecialchars($course['title'] ?? 'Course'); ?>"
                                 onerror="this.src='/assets/images/default-thumbnail.jpg';">
                        </div>
                        <div class="fc-course-content">
                            <div class="fc-course-level-badge fc-level-<?php echo strtolower($courseLevel); ?> course-level" data-level="<?php echo $courseLevel; ?>">
                                <?php echo $courseLevel; ?>
                            </div>
                            <h3 class="fc-course-title course-title">
                                <?php echo htmlspecialchars($course['title'] ?? 'Unnamed Course'); ?>
                            </h3>
                            <div class="fc-course-instructor course-instructor">
                                <?php 
                                $instructorName = trim(($course['first_name'] ?? '') . ' ' . ($course['last_name'] ?? ''));
                                echo !empty($instructorName) ? htmlspecialchars($instructorName) : 'Instructor Name';
                                ?>
                            </div>
                            <div class="fc-course-price course-price" data-price="<?php echo $course['price'] ?? 1000; ?>">
                                <?php if (isset($course['price']) && $course['price'] == 0): ?>
                                    <span class="fc-price-free">Free</span>
                                <?php else: ?>
                                    <span class="fc-price">R<?php echo number_format($course['price'] ?? 1000, 2); ?></span>
                                <?php endif; ?>
                            </div>
                            <div class="fc-course-actions">
                                <?php 
                                $course_id = $course['course_id'] ?? 0;
                                try {
                                    if (class_exists('IdEncryption')) {
                                        $encrypted_id = IdEncryption::encrypt($course_id);
                                        if ($encrypted_id && $encrypted_id !== false) {
                                            echo "<!-- DEBUG: Course ID $course_id encrypted to: $encrypted_id -->";
                                        } else {
                                            echo "<!-- DEBUG: Encryption failed for course ID $course_id, using fallback -->";
                                            $encrypted_id = $course_id; // Fallback to numeric ID
                                        }
                                    } else {
                                        echo "<!-- DEBUG: IdEncryption class not found, using fallback -->";
                                        $encrypted_id = $course_id; // Fallback to numeric ID
                                    }
                                } catch (Exception $e) {
                                    echo "<!-- DEBUG: Encryption error: " . $e->getMessage() . " -->";
                                    $encrypted_id = $course_id; // Fallback to numeric ID
                                }
                                ?>
                                <a href="course.php?id=<?php echo urlencode($encrypted_id); ?>" class="fc-course-btn fc-course-view-btn">
                                    <i class="fas fa-eye"></i> View
                                </a>
                                <button class="fc-course-btn fc-course-cart-btn pa-course-cart-btn" data-course-id="<?php echo htmlspecialchars($encrypted_id); ?>">
                                    <i class="fas fa-shopping-cart"></i> Cart
                                </button>
                            </div>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
            </div>

            <?php else: ?>
            <!-- Empty State -->
            <div class="fc-empty-state text-center py-5">
                <div class="fc-empty-icon mb-3">
                    <i class="fas fa-star" style="font-size: 4rem; color: #e0e7ff;"></i>
                </div>
                <h3 class="fc-empty-title">No Featured Courses Available</h3>
                <p class="fc-empty-text">Featured courses will appear here once added to the system.</p>
            </div>
            <?php endif; ?>
        </div>
    </section>

    <!-- Isolated CSS for Featured Courses Section -->
    <style>
        /* Featured Courses - Isolated Styles with FC prefix */
        .fc-featured-section {
            background: white;
            position: relative;
        }

        /* Section Header */
        .fc-header {
            margin-bottom: 2rem;
        }

        .fc-main-title {
            font-size: 2.2rem;
            font-weight: 700;
            color: #2563eb;
            margin-bottom: 0.5rem;
        }

        .fc-subtitle {
            font-size: 1rem;
            color: #64748b;
            margin: 0;
            font-weight: 400;
        }

        /* Horizontal Scroll Container */
        .fc-scroll-container {
            position: relative;
            margin: 0 20px;
        }

        .fc-courses-scroll {
            display: flex;
            overflow-x: auto;
            scroll-behavior: smooth;
            gap: 1rem;
            padding: 10px 0;
            scrollbar-width: none;
            -ms-overflow-style: none;
        }

        .fc-courses-scroll::-webkit-scrollbar {
            display: none;
        }

        /* Navigation Arrows */
        .fc-scroll-btn {
            position: absolute;
            top: 50%;
            transform: translateY(-50%);
            z-index: 10;
            background: white;
            border: 1px solid #e2e8f0;
            border-radius: 50%;
            width: 40px;
            height: 40px;
            display: flex;
            align-items: center;
            justify-content: center;
            cursor: pointer;
            box-shadow: 0 2px 8px rgba(0, 0, 0, 0.1);
            transition: all 0.2s ease;
            color: #2563eb;
        }

        .fc-scroll-btn:hover {
            background: #2563eb;
            color: white;
            transform: translateY(-50%) scale(1.1);
        }

        /* Course Card - Same size as categories */
        .fc-course-card {
            background: white;
            border-radius: 8px;
            overflow: hidden;
            box-shadow: 0 2px 6px rgba(0, 0, 0, 0.06);
            border: 1px solid #e2e8f0;
            transition: all 0.2s ease;
            min-width: 220px;
            max-width: 220px;
            width: 220px;
            flex-shrink: 0;
            display: flex;
            flex-direction: column;
        }

        .fc-course-card:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 16px rgba(0, 0, 0, 0.12);
            border-color: #2563eb;
        }

        /* Course Thumbnail */
        .fc-course-thumbnail {
            position: relative;
            height: 120px;
            overflow: hidden;
            background: #f8fafc;
        }

        .fc-course-thumbnail img {
            width: 100%;
            height: 100%;
            object-fit: cover;
            transition: transform 0.2s ease;
        }

        .fc-course-card:hover .fc-course-thumbnail img {
            transform: scale(1.03);
        }

        .fc-course-overlay {
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0, 0, 0, 0.5);
            display: flex;
            align-items: center;
            justify-content: center;
            opacity: 0;
            transition: opacity 0.2s ease;
        }

        .fc-course-card:hover .fc-course-overlay {
            opacity: 1;
        }

        .fc-course-actions {
            display: flex;
            gap: 0.4rem;
            margin-top: 0.75rem;
            justify-content: space-between;
        }

        .fc-course-btn {
            flex: 1;
            padding: 0.4rem 0.6rem;
            border-radius: 6px;
            display: flex;
            align-items: center;
            justify-content: center;
            text-decoration: none;
            transition: all 0.2s ease;
            font-size: 0.75rem;
            font-weight: 500;
            border: none;
            cursor: pointer;
            gap: 0.3rem;
        }

        .fc-course-view-btn {
            background: #f1f5f9;
            color: #2563eb;
            border: 1px solid #e2e8f0;
        }

        .fc-course-view-btn:hover {
            background: #e2e8f0;
            color: #1d4ed8;
            transform: translateY(-1px);
        }

        .fc-course-cart-btn {
            background: #f59e0b !important;
            color: white !important;
            border: 1px solid #f59e0b !important;
        }

        .fc-course-cart-btn:hover {
            background: #d97706 !important;
            border-color: #d97706 !important;
            transform: translateY(-1px);
        }

        /* Course Content */
        .fc-course-content {
            padding: 0.9rem;
            flex: 1;
            display: flex;
            flex-direction: column;
        }

        .fc-course-level-badge {
            font-size: 0.65rem;
            font-weight: 600;
            text-transform: uppercase;
            letter-spacing: 0.3px;
            margin-bottom: 0.6rem;
            padding: 0.2rem 0.6rem;
            border-radius: 12px;
            width: fit-content;
        }

        .fc-level-beginner {
            background: rgba(34, 197, 94, 0.1);
            color: #15803d;
        }

        .fc-level-intermediate {
            background: rgba(245, 158, 11, 0.1);
            color: #d97706;
        }

        .fc-level-advanced {
            background: rgba(239, 68, 68, 0.1);
            color: #dc2626;
        }

        .fc-course-title {
            font-size: 0.95rem;
            font-weight: 600;
            color: #1a202c;
            margin: 0 0 0.4rem 0;
            line-height: 1.3;
            display: -webkit-box;
            -webkit-line-clamp: 2;
            -webkit-box-orient: vertical;
            overflow: hidden;
            min-height: 2.4rem;
        }

        .fc-course-instructor {
            font-size: 0.75rem;
            color: #64748b;
            margin-bottom: 0.8rem;
            font-weight: 400;
        }

        .fc-course-price {
            margin-top: auto;
            font-weight: 600;
        }

        .fc-price {
            color: #2563eb;
            font-size: 1rem;
        }

        .fc-price-free {
            color: #10b981;
            font-size: 0.9rem;
            font-weight: 700;
        }

        /* Empty State */
        .fc-empty-state {
            padding: 3rem 1rem;
        }

        .fc-empty-title {
            font-size: 1.3rem;
            font-weight: 600;
            color: #64748b;
            margin-bottom: 0.5rem;
        }

        .fc-empty-text {
            font-size: 0.9rem;
            color: #94a3b8;
            margin: 0;
        }

        /* Responsive Design */
        @media (max-width: 768px) {
            .fc-main-title {
                font-size: 1.8rem;
            }

            .fc-course-card {
                min-width: 200px;
                max-width: 200px;
                width: 200px;
            }

            .fc-course-thumbnail {
                height: 110px;
            }

            .fc-scroll-btn {
                width: 36px;
                height: 36px;
            }
        }

        @media (max-width: 480px) {
            .fc-main-title {
                font-size: 1.6rem;
            }

            .fc-course-content {
                padding: 0.8rem;
            }

            .fc-course-card {
                min-width: 180px;
                max-width: 180px;
                width: 180px;
            }
        }
    </style>

    <!-- JavaScript for Featured Courses Scrolling -->
    <script>
        // Featured courses horizontal scrolling function
        function fcScrollCourses(direction) {
            const scrollContainer = document.getElementById('fc-scroll-container');
            if (!scrollContainer) return;
            
            const cardWidth = 220; // Card width + gap
            const scrollAmount = cardWidth + 16; // Include gap
            const currentScroll = scrollContainer.scrollLeft;
            
            if (direction === 'left') {
                scrollContainer.scrollTo({
                    left: currentScroll - scrollAmount,
                    behavior: 'smooth'
                });
            } else if (direction === 'right') {
                scrollContainer.scrollTo({
                    left: currentScroll + scrollAmount,
                    behavior: 'smooth'
                });
            }
        }

        // Initialize featured courses functionality
        document.addEventListener('DOMContentLoaded', function() {
            const scrollContainer = document.querySelector('.fc-courses-scroll');
            if (!scrollContainer) return;

            // Handle scroll button visibility for featured courses
            function updateFCScrollButtons() {
                const scrollWrapper = scrollContainer.parentElement;
                const leftBtn = scrollWrapper.querySelector('.fc-scroll-left');
                const rightBtn = scrollWrapper.querySelector('.fc-scroll-right');
                
                if (scrollContainer.scrollWidth <= scrollContainer.clientWidth) {
                    // Hide buttons if content doesn't overflow
                    if (leftBtn) leftBtn.style.display = 'none';
                    if (rightBtn) rightBtn.style.display = 'none';
                } else {
                    // Show buttons if content overflows
                    if (leftBtn) leftBtn.style.display = 'flex';
                    if (rightBtn) rightBtn.style.display = 'flex';
                    
                    // Update button states based on scroll position
                    function updateButtonStates() {
                        const isAtStart = scrollContainer.scrollLeft <= 0;
                        const isAtEnd = scrollContainer.scrollLeft >= scrollContainer.scrollWidth - scrollContainer.clientWidth - 1;
                        
                        if (leftBtn) {
                            leftBtn.style.opacity = isAtStart ? '0.5' : '1';
                            leftBtn.style.pointerEvents = isAtStart ? 'none' : 'auto';
                        }
                        
                        if (rightBtn) {
                            rightBtn.style.opacity = isAtEnd ? '0.5' : '1';
                            rightBtn.style.pointerEvents = isAtEnd ? 'none' : 'auto';
                        }
                    }
                    
                    // Initial button state
                    updateButtonStates();
                    
                    // Update button states on scroll
                    scrollContainer.addEventListener('scroll', updateButtonStates);
                }
            }

            // Initial update
            updateFCScrollButtons();
            
            // Update on window resize
            window.addEventListener('resize', updateFCScrollButtons);

            // Mouse wheel horizontal scrolling for featured courses
            scrollContainer.addEventListener('wheel', function(e) {
                if (e.deltaY !== 0) {
                    e.preventDefault();
                    scrollContainer.scrollLeft += e.deltaY > 0 ? 50 : -50;
                }
            });
            
            // Touch/swipe support for featured courses
            let isDown = false;
            let startX;
            let scrollLeft;
            
            scrollContainer.addEventListener('mousedown', (e) => {
                if (e.target.closest('.fc-course-overlay')) return;
                isDown = true;
                scrollContainer.style.cursor = 'grabbing';
                startX = e.pageX - scrollContainer.offsetLeft;
                scrollLeft = scrollContainer.scrollLeft;
            });
            
            scrollContainer.addEventListener('mouseleave', () => {
                isDown = false;
                scrollContainer.style.cursor = 'grab';
            });
            
            scrollContainer.addEventListener('mouseup', () => {
                isDown = false;
                scrollContainer.style.cursor = 'grab';
            });
            
            scrollContainer.addEventListener('mousemove', (e) => {
                if (!isDown) return;
                e.preventDefault();
                const x = e.pageX - scrollContainer.offsetLeft;
                const walk = (x - startX) * 2;
                scrollContainer.scrollLeft = scrollLeft - walk;
            });
            
            // Set initial cursor
            scrollContainer.style.cursor = 'grab';
        });

        // Expose function globally
        window.fcScrollCourses = fcScrollCourses;
        
        // Initialize Cart System Event Listeners
        document.addEventListener('DOMContentLoaded', function() {
            // Bind cart buttons to addToCart function
            document.addEventListener('click', function(e) {
                if (e.target.closest('.pa-course-cart-btn')) {
                    e.preventDefault();
                    const button = e.target.closest('.pa-course-cart-btn');
                    const courseId = button.getAttribute('data-course-id');
                    
                    if (courseId) {
                        console.log('Adding course to cart:', courseId);
                        addToCart(courseId);
                    } else {
                        console.error('No course ID found on button');
                        showNotification('Error: Course ID not found!', 'error');
                    }
                }
            });
            
            // Update cart counter on page load
            updateCartBadge();
        });
    </script>

    <!-- Browse by Categories Section -->
    <section class="pa-category-browse-section py-5" style="background: white;">
        <div class="container">
            <!-- Section Header -->
            <div class="pa-category-header text-center mb-4">
                <h2 class="pa-category-main-title">Browse by Categories</h2>
                <p class="pa-category-subtitle">Discover courses across various fields of expertise</p>
            </div>

            <!-- Category Tabs Navigation -->
            <div class="pa-category-tabs-wrapper mb-4">
                <div class="pa-category-tabs-nav">
                        <button class="pa-category-tab-btn pa-active" data-category="all" onclick="paShowCategory('all')">
                            All Courses <span class="pa-category-count"><?php 
                            $totalCount = 0;
                            if (!empty($categories)) {
                                foreach($categories as $cat) {
                                    $totalCount += isset($cat['course_count']) ? intval($cat['course_count']) : 0;
                                }
                            }
                            echo $totalCount;
                            ?></span>
                        </button>
                        <?php foreach ($categories as $category): ?>
                            <?php if ($category['course_count'] > 0): ?>
                            <button class="pa-category-tab-btn" data-category="<?php echo htmlspecialchars($category['category_name']); ?>" onclick="paShowCategory('<?php echo htmlspecialchars($category['category_name']); ?>')">
                                <?php echo htmlspecialchars($category['category_name']); ?> <span class="pa-category-count"><?php echo $category['course_count']; ?></span>
                            </button>
                            <?php endif; ?>
                        <?php endforeach; ?>
                </div>
            </div>

            <!-- Category Content Panels -->
            <div class="pa-category-content-wrapper">
                <!-- All Courses Panel -->
                <div class="pa-category-panel pa-active" data-category-panel="all">
                    <div class="pa-category-scroll-container" style="position: relative;">
                        <!-- Navigation Arrows -->
                        <button class="pa-scroll-btn pa-scroll-left" onclick="paScrollCategory('all', 'left')" style="left: -15px;">
                            <i class="fas fa-chevron-left"></i>
                        </button>
                        <button class="pa-scroll-btn pa-scroll-right" onclick="paScrollCategory('all', 'right')" style="right: -15px;">
                            <i class="fas fa-chevron-right"></i>
                        </button>
                        
                        <div class="pa-category-courses-scroll" id="pa-scroll-all">
                            <?php 
                            $allCourses = [];
                            foreach ($coursesByCategory as $categoryName => $courses) {
                                $allCourses = array_merge($allCourses, $courses);
                            }
                            foreach ($allCourses as $course): 
                                $courseLevels = ['Beginner', 'Intermediate', 'Advanced'];
                                $courseLevel = $courseLevels[array_rand($courseLevels)];
                                $course_id = $course['course_id'] ?? 0;
                                $encrypted_id = class_exists('IdEncryption') ? IdEncryption::encrypt($course_id) : $course_id;
                            ?>
                            <div class="pa-category-course-card" data-course-id="<?php echo htmlspecialchars($encrypted_id); ?>">
                                <div class="pa-course-thumbnail">
                                    <?php
                                    $thumbnail = $course['thumbnail'] ?? '';
                                    if (strpos($thumbnail, '../') === 0) {
                                        $thumbnail = substr($thumbnail, 3);
                                    }
                                    if (substr($thumbnail, 0, 1) !== '/') {
                                        $thumbnail = '/' . $thumbnail;
                                    }
                                    $thumbnail = str_replace('//', '/', $thumbnail);
                                    
                                    if (empty($thumbnail) || $thumbnail == '/') {
                                        $thumbnail = '/assets/images/default-thumbnail.jpg';
                                    }
                                    ?>
                                    <img src="<?php echo htmlspecialchars($thumbnail); ?>" 
                                         alt="<?php echo htmlspecialchars($course['title'] ?? 'Course'); ?>"
                                         onerror="this.src='/assets/images/default-thumbnail.jpg';">
                                </div>
                                <div class="pa-course-content">
                                    <div class="pa-course-level-badge pa-level-<?php echo strtolower($courseLevel); ?> course-level" data-level="<?php echo $courseLevel; ?>">
                                        <?php echo $courseLevel; ?>
                                    </div>
                                    <h3 class="pa-course-title course-title">
                                        <?php echo htmlspecialchars($course['title'] ?? 'Unnamed Course'); ?>
                                    </h3>
                                    <div class="pa-course-instructor course-instructor">
                                        <?php 
                                        $instructorName = trim(($course['first_name'] ?? '') . ' ' . ($course['last_name'] ?? ''));
                                        echo !empty($instructorName) ? htmlspecialchars($instructorName) : 'Instructor Name';
                                        ?>
                                    </div>
                                    <div class="pa-course-price course-price" data-price="<?php echo $course['price'] ?? 1000; ?>">
                                        <?php if (isset($course['price']) && $course['price'] == 0): ?>
                                            <span class="pa-price-free">Free</span>
                                        <?php else: ?>
                                            <span class="pa-price">R<?php echo number_format($course['price'] ?? 1000, 2); ?></span>
                                        <?php endif; ?>
                                    </div>
                                    <div class="pa-course-actions">
                                        <a href="course.php?id=<?php echo urlencode($encrypted_id); ?>" class="pa-course-btn pa-course-view-btn">
                                            <i class="fas fa-eye"></i> View
                                        </a>
                                        <button class="pa-course-btn pa-course-cart-btn pa-course-cart-btn" data-course-id="<?php echo htmlspecialchars($encrypted_id); ?>">
                                            <i class="fas fa-shopping-cart"></i> Cart
                                        </button>
                                    </div>
                                </div>
                            </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                </div>

                <!-- Individual Category Panels -->
                <?php foreach ($coursesByCategory as $categoryName => $courses): ?>
                <div class="pa-category-panel" data-category-panel="<?php echo htmlspecialchars($categoryName); ?>">
                    <div class="pa-category-scroll-container" style="position: relative;">
                        <!-- Navigation Arrows -->
                        <button class="pa-scroll-btn pa-scroll-left" onclick="paScrollCategory('<?php echo htmlspecialchars($categoryName); ?>', 'left')" style="left: -15px;">
                            <i class="fas fa-chevron-left"></i>
                        </button>
                        <button class="pa-scroll-btn pa-scroll-right" onclick="paScrollCategory('<?php echo htmlspecialchars($categoryName); ?>', 'right')" style="right: -15px;">
                            <i class="fas fa-chevron-right"></i>
                        </button>
                                                <div class="pa-category-courses-scroll" id="pa-scroll-<?php echo htmlspecialchars($categoryName); ?>">
                            <?php foreach ($courses as $course): 
                                $courseLevels = ['Beginner', 'Intermediate', 'Advanced'];
                                $courseLevel = $courseLevels[array_rand($courseLevels)];
                                $course_id = $course['course_id'] ?? 0;
                                $encrypted_id = class_exists('IdEncryption') ? IdEncryption::encrypt($course_id) : $course_id;
                            ?>
                            <div class="pa-category-course-card" data-course-id="<?php echo htmlspecialchars($encrypted_id); ?>">
                                <div class="pa-course-thumbnail">
                                    <?php
                                    $thumbnail = $course['thumbnail'] ?? '';
                                    if (strpos($thumbnail, '../') === 0) {
                                        $thumbnail = substr($thumbnail, 3);
                                    }
                                    if (substr($thumbnail, 0, 1) !== '/') {
                                        $thumbnail = '/' . $thumbnail;
                                    }
                                    $thumbnail = str_replace('//', '/', $thumbnail);
                                    
                                    if (empty($thumbnail) || $thumbnail == '/') {
                                        $thumbnail = '/assets/images/default-thumbnail.jpg';
                                    }
                                    ?>
                                    <img src="<?php echo htmlspecialchars($thumbnail); ?>" 
                                         alt="<?php echo htmlspecialchars($course['title'] ?? 'Course'); ?>"
                                         onerror="this.src='/assets/images/default-thumbnail.jpg';">
                                </div>
                                <div class="pa-course-content">
                                    <div class="pa-course-level-badge pa-level-<?php echo strtolower($courseLevel); ?> course-level" data-level="<?php echo $courseLevel; ?>">
                                        <?php echo $courseLevel; ?>
                                    </div>
                                    <h3 class="pa-course-title course-title">
                                        <?php echo htmlspecialchars($course['title'] ?? 'Unnamed Course'); ?>
                                    </h3>
                                    <div class="pa-course-instructor course-instructor">
                                        <?php 
                                        $instructorName = trim(($course['first_name'] ?? '') . ' ' . ($course['last_name'] ?? ''));
                                        echo !empty($instructorName) ? htmlspecialchars($instructorName) : 'Instructor Name';
                                        ?>
                                    </div>
                                    <div class="pa-course-price course-price" data-price="<?php echo $course['price'] ?? 1000; ?>">
                                        <?php if (isset($course['price']) && $course['price'] == 0): ?>
                                            <span class="pa-price-free">Free</span>
                                        <?php else: ?>
                                            <span class="pa-price">R<?php echo number_format($course['price'] ?? 1000, 2); ?></span>
                                        <?php endif; ?>
                                    </div>
                                    <div class="pa-course-actions">
                                        <a href="course.php?id=<?php echo urlencode($encrypted_id); ?>" class="pa-course-btn pa-course-view-btn">
                                            <i class="fas fa-eye" ></i> View
                                        </a>
                                        <button class="pa-course-btn pa-course-cart-btn pa-course-cart-btn" data-course-id="<?php echo htmlspecialchars($encrypted_id); ?>">
                                            <i class="fas fa-shopping-cart" ></i> Cart
                                        </button>
                                    </div>
                                </div>
                            </div>
                            
                            <?php endforeach; ?>
                        </div>
                    </div>
                </div>

                <?php endforeach; ?>
            </div>
       
            <?php if (empty($categories) && empty($coursesByCategory)): ?>
            <!-- Empty State -->
            <div class="pa-category-empty-state text-center py-5">
                <div class="pa-empty-icon mb-3">
                    <i class="fas fa-graduation-cap" style="font-size: 4rem; color: #e0e7ff;"></i>
                </div>
                <h3 class="pa-empty-title">No Categories Available</h3>
                <p class="pa-empty-text">Categories will appear here once courses are added to the system.</p>
            </div>
            <?php endif; ?>
        </div>
    </section>

    <!-- Why Panadite Academy Section -->
    <section class="why-panadite-section py-5" style="background: linear-gradient(135deg, #f8fafc 0%, #ffffff 100%);">
        <div class="container">
            <div class="row align-items-center">
                <div class="col-lg-6 mb-5 mb-lg-0">
                    <div class="why-content pe-lg-4">
                        <!-- Section Header -->
                        <h2 class="why-title mb-3">
                            Why <span class="brand-highlight">Panadite Academy</span>?
                        </h2>
                        
                        <!-- Description -->
                        <p class="why-description mb-4">
                            We have a multitude of courses tailored to your career goals and busy schedule. 
                            These courses have been developed to enhance your knowledge and critical thinking 
                            abilities and make you an expert in your domain.
                        </p>
                        
                        <!-- Features List -->
                        <div class="features-list">
                            <!-- Feature 1: Flexible Learning -->
                            <div class="feature-item">
                                <div class="feature-icon">
                                    <i class="fas fa-layer-group"></i>
                                </div>
                                <div class="feature-content">
                                    <h5 class="feature-title">Flexible & Stackable</h5>
                                    <p class="feature-text">
                                        At Panadite Academy, you are in charge of your learning! You decide your 
                                        pace of learning based on your convenience. The learning structure is modular - 
                                        Take one module at a time and stack up credits to your Dream Qualification.
                                    </p>
                                </div>
                            </div>
                            
                            <!-- Feature 2: Affordable -->
                            <div class="feature-item">
                                <div class="feature-icon">
                                    <i class="fas fa-wallet"></i>
                                </div>
                                <div class="feature-content">
                                    <h5 class="feature-title">Affordable <span class="text-muted">(Pay Per Module - Easy Installments)</span></h5>
                                    <p class="feature-text">
                                        With Panadite Academy as your learning partner, you follow a 'Pay as you 
                                        Go Scheme' for your program, that means no upfront bulk payment of your fee. 
                                        Instead your fee will be divided into small pocket-friendly installments.
                                    </p>
                                </div>
                            </div>
                            
                        </div>
                    </div>
                </div>
                
                <!-- Right Column: Visual + Feature -->
                <div class="col-lg-6">
                    <div class="right-content-wrapper">
                        <!-- Laptop Mockup -->
                        <div class="why-visual-wrapper">
                            <div class="platform-mockup">
                                <!-- Background Gradient Circle -->
                                <div class="mockup-bg-circle"></div>
                                
                                <!-- Laptop Mockup -->
                                <div class="laptop-mockup">
                                    <div class="laptop-screen">
                                        <!-- Dashboard Header with Logo -->
                                        <div class="dashboard-header">
                                            <div class="dashboard-nav">
                                                <div class="logo-section">
                                                    <img src="logo.jpeg" alt="Panadite Academy" class="dashboard-logo">
                                                    <span class="dashboard-title">Panadite Academy</span>
                                                </div>
                                                <div class="nav-menu">
                                                    <span class="nav-item">Dashboard</span>
                                                    <span class="nav-item active">Courses</span>
                                                </div>
                                            </div>
                                        </div>
                                        
                                        <!-- Dashboard Content -->
                                        <div class="dashboard-content">
                                            <!-- Welcome Section -->
                                            <div class="welcome-section">
                                                <div class="welcome-text">
                                                    <h6>Welcome back, Student!</h6>
                                                    <p>Continue your learning journey</p>
                                                </div>
                                                <div class="profile-avatar">
                                                    <img src="landing_images/images (3).jpeg" alt="Profile" class="profile-img">
                                                </div>
                                            </div>
                                            
                                            <!-- Current Course Card -->
                                            <div class="current-course-card">
                                                <div class="course-thumbnail">
                                                    <img src="landing_images/download (15).jpeg" alt="Course" class="course-img">
                                                </div>
                                                <div class="course-details">
                                                    <h6>Leadership & Strategy</h6>
                                                    <p class="course-instructor">Expert Instructor</p>
                                                    <div class="progress-section">
                                                        <div class="progress-bar">
                                                            <div class="progress-fill" style="width: 75%;"></div>
                                                        </div>
                                                        <span class="progress-label">75% Complete</span>
                                                    </div>
                                                    <button class="continue-btn">Continue Learning</button>
                                                </div>
                                            </div>
                                            
                                            <!-- Stats Row -->
                                            <div class="stats-row">
                                                <div class="stat-item">
                                                    <span class="stat-number">3</span>
                                                    <span class="stat-label">Completed</span>
                                                </div>
                                                <div class="stat-item">
                                                    <span class="stat-number">2</span>
                                                    <span class="stat-label">In Progress</span>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                    <div class="laptop-base"></div>
                                </div>
                                
                                <!-- Floating Elements -->
                                <div class="floating-badge badge-1">
                                    <i class="fas fa-certificate"></i>
                                    <span>Certificate<br>Earned</span>
                                </div>
                                
                                <div class="floating-badge badge-2">
                                    <i class="fas fa-users"></i>
                                    <span>1000+<br>Students</span>
                                </div>
                            </div>
                        </div>

                        <!-- Feature 3: Accredited - Moved to right side -->
                        <div class="feature-item side-feature">
                            <div class="feature-icon">
                                <i class="fas fa-award"></i>
                            </div>
                            <div class="feature-content">
                                <h5 class="feature-title">Accredited & Globally Accepted</h5>
                                <p class="feature-text">
                                    We have partnered with reputed Universities and Awarding bodies, 
                                    which are accredited by governmental regulatory bodies in their respective 
                                    countries. The universities are SAQA Approved and are recognized worldwide.
                                </p>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </section>

    <!-- Referral Program Carousel Section -->
    <section class="referral-carousel-section" style="background: #f8fafc;">
        <div class="container">
            <!-- Section Header -->
            <div class="text-center mb-3">
                <h2 class="referral-main-title">
                    Transform Your Network into <span class="highlight-amber">Rewards</span>
                </h2>
                <p class="referral-subtitle">
                    Join our referral program and earn while helping others advance their careers
                </p>
            </div>

            <!-- Carousel Container -->
            <div class="referral-carousel-container">
                <!-- Navigation Arrows -->
                <button class="nav-btn prev-btn" onclick="changeSlide(-1)">
                    <i class="fas fa-chevron-left"></i>
                </button>
                <button class="nav-btn next-btn" onclick="changeSlide(1)">
                    <i class="fas fa-chevron-right"></i>
                </button>
                
                <div class="carousel-wrapper">
                    <!-- Carousel Slides -->
                    <div class="carousel-slides" id="referralCarousel">
                        
                        <!-- Slide 1: Teacher Referral Benefits -->
                        <div class="carousel-slide active" data-slide="1">
                            <div class="row align-items-center">
                                <div class="col-lg-6">
                                    <div class="slide-content">
                                        <div class="slide-badge">For Educators</div>
                                        <h3 class="slide-title">Earn <span class="amount-highlight">$50</span> for Every Teacher Referral</h3>
                                        <p class="slide-description">
                                            Share your expertise and get rewarded! Refer qualified teachers to join 
                                            Panadite Academy and earn $50 for each successful referral who creates a course.
                                        </p>
                                        <div class="benefits-list">
                                            <div class="benefit-item">
                                                <i class="fas fa-dollar-sign"></i>
                                                <span>Instant payout after course publication</span>
                                            </div>
                                            <div class="benefit-item">
                                                <i class="fas fa-users"></i>
                                                <span>No limit on referral earnings</span>
                                            </div>
                                            <div class="benefit-item">
                                                <i class="fas fa-chart-line"></i>
                                                <span>Track referrals in your dashboard</span>
                                            </div>
                                        </div>
                                        <button class="cta-button">Start Referring Teachers</button>
                                    </div>
                                </div>
                                <div class="col-lg-6">
                                    <div class="slide-graphic">
                                        <div class="graphic-container teacher-referral">
                                            <div class="teacher-avatar main-teacher">
                                                <i class="fas fa-user-tie"></i>
                                                <div class="teacher-name">Sarah</div>
                                                <div class="teacher-role">Lead Educator</div>
                                            </div>
                                            <div class="referral-arrow">
                                                <i class="fas fa-arrow-right"></i>
                                            </div>
                                            <div class="teacher-avatar referred-teacher">
                                                <i class="fas fa-graduation-cap"></i>
                                                <div class="teacher-name">Mike</div>
                                                <div class="teacher-role">New Teacher</div>
                                            </div>
                                            <div class="earning-popup">
                                                <i class="fas fa-dollar-sign"></i>
                                                <span>+$50 Earned!</span>
                                            </div>
                                            <div class="course-icon">
                                                <i class="fas fa-book"></i>
                                                <span>Course Created</span>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Slide 2: Student Referral Benefits -->
                        <div class="carousel-slide" data-slide="2">
                            <div class="row align-items-center">
                                <div class="col-lg-6">
                                    <div class="slide-content">
                                        <div class="slide-badge">For Students</div>
                                        <h3 class="slide-title">Get <span class="amount-highlight">20% Off</span> Your Next Course</h3>
                                        <p class="slide-description">
                                            Refer friends to Panadite Academy and both of you save! Get 20% discount 
                                            on your next course enrollment for every successful student referral.
                                        </p>
                                        <div class="benefits-list">
                                            <div class="benefit-item">
                                                <i class="fas fa-percentage"></i>
                                                <span>20% discount per referral</span>
                                            </div>
                                            <div class="benefit-item">
                                                <i class="fas fa-gift"></i>
                                                <span>Your friend gets 10% off too</span>
                                            </div>
                                            <div class="benefit-item">
                                                <i class="fas fa-infinity"></i>
                                                <span>Unlimited referral discounts</span>
                                            </div>
                                        </div>
                                        <button class="cta-button">Refer Students Now</button>
                                    </div>
                                </div>
                                <div class="col-lg-6">
                                    <div class="slide-graphic">
                                        <div class="graphic-container student-referral">
                                            <div class="student-avatar main-student">
                                                <i class="fas fa-user"></i>
                                                <div class="student-name">Emma</div>
                                                <div class="student-progress">3 Courses</div>
                                            </div>
                                            <div class="referral-arrow">
                                                <i class="fas fa-share-alt"></i>
                                            </div>
                                            <div class="student-avatar referred-student">
                                                <i class="fas fa-user-plus"></i>
                                                <div class="student-name">Alex</div>
                                                <div class="student-progress">New Student</div>
                                            </div>
                                            <div class="discount-popup emma">
                                                <i class="fas fa-tag"></i>
                                                <span>20% Off!</span>
                                            </div>
                                            <div class="discount-popup alex">
                                                <i class="fas fa-gift"></i>
                                                <span>10% Off!</span>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Slide 3: Program Overview -->
                        <div class="carousel-slide" data-slide="3">
                            <div class="row align-items-center">
                                <div class="col-lg-6">
                                    <div class="slide-content">
                                        <div class="slide-badge">How It Works</div>
                                        <h3 class="slide-title">Simple <span class="amount-highlight">3-Step</span> Process</h3>
                                        <p class="slide-description">
                                            Our referral program is designed to be straightforward and rewarding 
                                            for everyone involved in the Panadite Academy community.
                                        </p>
                                        <div class="process-steps">
                                            <div class="step-item">
                                                <div class="step-number">1</div>
                                                <div class="step-content">
                                                    <strong>Share Your Link</strong>
                                                    <span>Get your unique referral link from your dashboard</span>
                                                </div>
                                            </div>
                                            <div class="step-item">
                                                <div class="step-number">2</div>
                                                <div class="step-content">
                                                    <strong>Friend Joins</strong>
                                                    <span>They sign up and enroll or create a course</span>
                                                </div>
                                            </div>
                                            <div class="step-item">
                                                <div class="step-number">3</div>
                                                <div class="step-content">
                                                    <strong>You Both Win</strong>
                                                    <span>Earn rewards and help grow our community</span>
                                                </div>
                                            </div>
                                        </div>
                                        <button class="cta-button">Get Your Referral Link</button>
                                    </div>
                                </div>
                                <div class="col-lg-6">
                                    <div class="slide-graphic">
                                        <div class="graphic-container process-graphic">
                                            <div class="process-flow">
                                                <div class="flow-item">
                                                    <div class="flow-icon">
                                                        <i class="fas fa-link"></i>
                                                    </div>
                                                    <div class="flow-text">Share Link</div>
                                                </div>
                                                <div class="flow-arrow">
                                                    <i class="fas fa-arrow-down"></i>
                                                </div>
                                                <div class="flow-item">
                                                    <div class="flow-icon">
                                                        <i class="fas fa-user-check"></i>
                                                    </div>
                                                    <div class="flow-text">Friend Joins</div>
                                                </div>
                                                <div class="flow-arrow">
                                                    <i class="fas fa-arrow-down"></i>
                                                </div>
                                                <div class="flow-item">
                                                    <div class="flow-icon">
                                                        <i class="fas fa-trophy"></i>
                                                    </div>
                                                    <div class="flow-text">Both Win</div>
                                                </div>
                                            </div>
                                            <div class="success-celebration">
                                                <i class="fas fa-star"></i>
                                                <i class="fas fa-star"></i>
                                                <i class="fas fa-star"></i>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Carousel Dots Navigation -->
                <div class="carousel-nav">
                    <div class="carousel-dots">
                        <span class="dot active" onclick="currentSlide(1)"></span>
                        <span class="dot" onclick="currentSlide(2)"></span>
                        <span class="dot" onclick="currentSlide(3)"></span>
                    </div>
                </div>
            </div>
        </div>
    </section>

    <!-- Panadite Library Section -->
    <section class="library-section py-5" style="background: #ffffff;">
        <div class="container">
            <!-- Section Header -->
            <div class="row align-items-center mb-4">
                <div class="col-md-8">
                    <h2 class="library-main-title">
                        From Panadite's <span class="highlight-library">Resource Library</span>
                    </h2>
                    <p class="library-subtitle">
                        Your comprehensive gateway to Panadite Academy's world of learning. Explore our diverse collection of 
                        educational videos, webinars, course previews, student success stories and much more.
                    </p>
                </div>
                <div class="col-md-4 text-end">
                    <a href="#" class="view-more-link">
                        View More <i class="fas fa-arrow-right ms-1"></i>
                    </a>
                </div>
            </div>

            <!-- Library Content Grid -->
            <div class="row g-4">
                <!-- Video Item 1 -->
                <div class="col-lg-4 col-md-6">
                    <div class="library-item">
                        <div class="library-thumbnail">
                            <img src="landing_images/course1.jpg" alt="Advanced Business Analytics" class="library-image">
                            <div class="play-overlay">
                                <i class="fas fa-play"></i>
                            </div>
                            <div class="video-duration">12:34</div>
                        </div>
                        <div class="library-content">
                            <h4 class="library-item-title">Advanced Business Analytics in Action</h4>
                            <p class="library-item-description">
                                Master data-driven decision making with real-world case studies and practical applications 
                                in modern business environments.
                            </p>
                            <div class="library-meta">
                                <span class="video-type">Course Preview</span>
                                <span class="view-count"><i class="fas fa-eye"></i> 2.3k views</span>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Video Item 2 -->
                <div class="col-lg-4 col-md-6">
                    <div class="library-item">
                        <div class="library-thumbnail">
                            <img src="landing_images/course2.jpg" alt="Digital Marketing Mastery" class="library-image">
                            <div class="play-overlay">
                                <i class="fas fa-play"></i>
                            </div>
                            <div class="video-duration">18:27</div>
                        </div>
                        <div class="library-content">
                            <h4 class="library-item-title">Digital Marketing Mastery: Complete Guide</h4>
                            <p class="library-item-description">
                                From social media strategies to conversion optimization - everything you need to succeed 
                                in digital marketing today.
                            </p>
                            <div class="library-meta">
                                <span class="video-type">Webinar</span>
                                <span class="view-count"><i class="fas fa-eye"></i> 4.7k views</span>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Video Item 3 -->
                <div class="col-lg-4 col-md-6">
                    <div class="library-item">
                        <div class="library-thumbnail">
                            <img src="landing_images/course3.jpg" alt="Student Success Story" class="library-image">
                            <div class="play-overlay">
                                <i class="fas fa-play"></i>
                            </div>
                            <div class="video-duration">8:15</div>
                        </div>
                        <div class="library-content">
                            <h4 class="library-item-title">Student Success: From Learning to Leading</h4>
                            <p class="library-item-description">
                                Inspiring journey of our graduate Sarah who transformed her career through our 
                                comprehensive program and strategic mentorship.
                            </p>
                            <div class="library-meta">
                                <span class="video-type">Success Story</span>
                                <span class="view-count"><i class="fas fa-eye"></i> 1.8k views</span>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </section>

    <!-- Library Section Styles -->
    <style>
        /* Library Section Styles */
        .library-section {
            position: relative;
        }

        .library-main-title {
            font-size: 2rem;
            font-weight: 700;
            color: #1a202c;
            margin-bottom: 0.5rem;
            line-height: 1.2;
        }

        .highlight-library {
            color: #f59e0b;
            font-weight: 800;
            position: relative;
        }

        .library-subtitle {
            font-size: 1rem;
            color: #64748b;
            line-height: 1.6;
            margin: 0;
            max-width: 90%;
        }

        .view-more-link {
            color: #2563eb;
            text-decoration: none;
            font-weight: 600;
            font-size: 1rem;
            transition: all 0.3s ease;
            display: inline-flex;
            align-items: center;
        }

        .view-more-link:hover {
            color: #1d4ed8;
            transform: translateX(3px);
        }

        .view-more-link i {
            transition: transform 0.3s ease;
        }

        .view-more-link:hover i {
            transform: translateX(3px);
        }

        /* Library Items */
        .library-item {
            background: white;
            border-radius: 16px;
            overflow: hidden;
            box-shadow: 0 4px 16px rgba(0, 0, 0, 0.08);
            transition: all 0.3s ease;
            height: 100%;
        }

        .library-item:hover {
            transform: translateY(-4px);
            box-shadow: 0 8px 32px rgba(0, 0, 0, 0.12);
        }

        .library-thumbnail {
            position: relative;
            aspect-ratio: 16/9;
            overflow: hidden;
        }

        .library-image {
            width: 100%;
            height: 100%;
            object-fit: cover;
            transition: transform 0.3s ease;
        }

        .library-item:hover .library-image {
            transform: scale(1.05);
        }

        .play-overlay {
            position: absolute;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%);
            width: 60px;
            height: 60px;
            background: rgba(37, 99, 235, 0.9);
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-size: 1.2rem;
            transition: all 0.3s ease;
            backdrop-filter: blur(4px);
        }

        .library-item:hover .play-overlay {
            background: rgba(37, 99, 235, 1);
            transform: translate(-50%, -50%) scale(1.1);
        }

        .video-duration {
            position: absolute;
            bottom: 12px;
            right: 12px;
            background: rgba(0, 0, 0, 0.8);
            color: white;
            padding: 0.25rem 0.5rem;
            border-radius: 4px;
            font-size: 0.8rem;
            font-weight: 500;
        }

        .library-content {
            padding: 1.5rem;
        }

        .library-item-title {
            font-size: 1.1rem;
            font-weight: 700;
            color: #1a202c;
            margin-bottom: 0.75rem;
            line-height: 1.3;
            display: -webkit-box;
            -webkit-line-clamp: 2;
            -webkit-box-orient: vertical;
            overflow: hidden;
        }

        .library-item-description {
            font-size: 0.9rem;
            color: #64748b;
            line-height: 1.5;
            margin-bottom: 1rem;
            display: -webkit-box;
            -webkit-line-clamp: 3;
            -webkit-box-orient: vertical;
            overflow: hidden;
        }

        .library-meta {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding-top: 0.75rem;
            border-top: 1px solid #e5e7eb;
        }

        .video-type {
            background: #eff6ff;
            color: #2563eb;
            padding: 0.25rem 0.75rem;
            border-radius: 20px;
            font-size: 0.75rem;
            font-weight: 600;
        }

        .view-count {
            color: #64748b;
            font-size: 0.8rem;
            display: flex;
            align-items: center;
            gap: 0.25rem;
        }

        .view-count i {
            font-size: 0.7rem;
        }

        /* Responsive Design */
        @media (max-width: 768px) {
            .library-main-title {
                font-size: 1.6rem;
            }

            .library-subtitle {
                font-size: 0.9rem;
                max-width: 100%;
            }

            .view-more-link {
                margin-top: 1rem;
            }

            .library-content {
                padding: 1rem;
            }

            .play-overlay {
                width: 50px;
                height: 50px;
                font-size: 1rem;
            }
        }
    </style>

    <!-- Isolated CSS for Browse by Categories Section -->
    <style>
        /* Browse by Categories - Isolated Styles with PA prefix */
        .pa-category-browse-section {
            background: white;
            position: relative;
        }

        /* Section Header */
        .pa-category-header {
            margin-bottom: 2rem;
        }

        .pa-category-main-title {
            font-size: 2.2rem;
            font-weight: 700;
            color: #2563eb;
            margin-bottom: 0.5rem;
        }

        .pa-category-subtitle {
            font-size: 1rem;
            color: #64748b;
            margin: 0;
            font-weight: 400;
        }

        /* Category Tabs Navigation */
        .pa-category-tabs-wrapper {
            margin-bottom: 1.5rem;
        }

        .pa-tabs-scroll-container {
            position: relative;
            max-width: 100%;
        }

        .pa-tabs-scroll-btn {
            position: absolute;
            top: 50%;
            transform: translateY(-50%);
            z-index: 15;
            background: white;
            border: 1px solid #e2e8f0;
            border-radius: 50%;
            width: 32px;
            height: 32px;
            display: none;
            align-items: center;
            justify-content: center;
            cursor: pointer;
            box-shadow: 0 2px 8px rgba(0, 0, 0, 0.1);
            transition: all 0.2s ease;
            color: #2563eb;
            font-size: 0.8rem;
        }

        .pa-tabs-scroll-left {
            left: -10px;
        }

        .pa-tabs-scroll-right {
            right: -10px;
        }

        .pa-tabs-scroll-btn:hover {
            background: #2563eb;
            color: white;
            transform: translateY(-50%) scale(1.1);
        }

        .pa-category-tabs-nav {
            display: flex;
            gap: 0.5rem;
            justify-content: center;
            padding: 0.4rem;
            background: white;
            border-radius: 8px;
            box-shadow: 0 1px 4px rgba(0, 0, 0, 0.06);
            border: 1px solid #e2e8f0;
            overflow-x: auto;
            scroll-behavior: smooth;
            scrollbar-width: none;
            -ms-overflow-style: none;
        }

        .pa-category-tabs-nav::-webkit-scrollbar {
            display: none;
        }

        .pa-category-tab-btn {
            background: transparent;
            border: none;
            padding: 0.6rem 1rem;
            border-radius: 6px;
            font-size: 0.85rem;
            font-weight: 500;
            color: #64748b;
            cursor: pointer;
            transition: all 0.2s ease;
            display: flex;
            align-items: center;
            gap: 0.4rem;
            white-space: nowrap;
        }

        .pa-category-tab-btn:hover {
            background: #f8fafc;
            color: #2563eb;
        }

        .pa-category-tab-btn.pa-active {
            background: #2563eb;
            color: white;
            box-shadow: 0 2px 6px rgba(37, 99, 235, 0.2);
        }

        .pa-category-count {
            background: rgba(255, 255, 255, 0.25);
            padding: 0.15rem 0.4rem;
            border-radius: 10px;
            font-size: 0.7rem;
            font-weight: 600;
            min-width: 1.2rem;
            text-align: center;
        }

        .pa-category-tab-btn:not(.pa-active) .pa-category-count {
            background: #f59e0b;
            color: white;
        }

        /* Content Panels */
        .pa-category-content-wrapper {
            position: relative;
        }

        .pa-category-panel {
            display: none;
            animation: paFadeIn 0.3s ease-in-out;
        }

        .pa-category-panel.pa-active {
            display: block;
        }

        @keyframes paFadeIn {
            from {
                opacity: 0;
                transform: translateY(5px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }

        /* Horizontal Scroll Container */
        .pa-category-scroll-container {
            position: relative;
            margin: 0 20px;
        }

        .pa-category-courses-scroll {
            display: flex;
            overflow-x: auto;
            scroll-behavior: smooth;
            gap: 1rem;
            padding: 10px 0;
            scrollbar-width: none;
            -ms-overflow-style: none;
        }

        .pa-category-courses-scroll::-webkit-scrollbar {
            display: none;
        }

        /* Navigation Arrows */
        .pa-scroll-btn {
            position: absolute;
            top: 50%;
            transform: translateY(-50%);
            z-index: 10;
            background: white;
            border: 1px solid #e2e8f0;
            border-radius: 50%;
            width: 40px;
            height: 40px;
            display: flex;
            align-items: center;
            justify-content: center;
            cursor: pointer;
            box-shadow: 0 2px 8px rgba(0, 0, 0, 0.1);
            transition: all 0.2s ease;
            color: #2563eb;
        }

        .pa-scroll-btn:hover {
            background: #2563eb;
            color: white;
            transform: translateY(-50%) scale(1.1);
        }

        /* Course Card - Smaller Size */
        .pa-category-course-card {
            background: white;
            border-radius: 8px;
            overflow: hidden;
            box-shadow: 0 2px 6px rgba(0, 0, 0, 0.06);
            border: 1px solid #e2e8f0;
            transition: all 0.2s ease;
            min-width: 220px;
            max-width: 220px;
            width: 220px;
            flex-shrink: 0;
            display: flex;
            flex-direction: column;
        }

        .pa-category-course-card:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 16px rgba(0, 0, 0, 0.12);
            border-color: #2563eb;
        }

        /* Course Thumbnail - Smaller */
        .pa-course-thumbnail {
            position: relative;
            height: 120px;
            overflow: hidden;
            background: #f8fafc;
        }

        .pa-course-thumbnail img {
            width: 100%;
            height: 100%;
            object-fit: cover;
            transition: transform 0.2s ease;
        }

        .pa-category-course-card:hover .pa-course-thumbnail img {
            transform: scale(1.03);
        }

        .pa-course-overlay {
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0, 0, 0, 0.5);
            display: flex;
            align-items: center;
            justify-content: center;
            opacity: 0;
            transition: opacity 0.2s ease;
        }

        .pa-category-course-card:hover .pa-course-overlay {
            opacity: 1;
        }

        .pa-course-actions {
            display: flex;
            gap: 0.5rem;
        }

        .pa-course-view-btn,
        .pa-course-cart-btn {
            width: 36px;
            height: 36px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            text-decoration: none;
            transition: all 0.2s ease;
            font-size: 0.9rem;
            border: none;
            cursor: pointer;
        }

        .pa-course-view-btn {
            background: white;
            color: #2563eb;
        }

        .pa-course-view-btn:hover {
            background: #f8fafc;
            transform: scale(1.1);
        }

        .pa-course-cart-btn {
            background: #f59e0b;
            color: white;
        }

        .pa-course-cart-btn:hover {
            background: #d97706;
            transform: scale(1.1);
        }

        /* Course Content - Compact */
        .pa-course-content {
            padding: 0.9rem;
            flex: 1;
            display: flex;
            flex-direction: column;
        }

        .pa-course-level-badge {
            font-size: 0.65rem;
            font-weight: 600;
            text-transform: uppercase;
            letter-spacing: 0.3px;
            margin-bottom: 0.6rem;
            padding: 0.2rem 0.6rem;
            border-radius: 12px;
            width: fit-content;
        }

        .pa-level-beginner {
            background: rgba(34, 197, 94, 0.1);
            color: #15803d;
        }

        .pa-level-intermediate {
            background: rgba(245, 158, 11, 0.1);
            color: #d97706;
        }

        .pa-level-advanced {
            background: rgba(239, 68, 68, 0.1);
            color: #dc2626;
        }

        .pa-course-title {
            font-size: 0.95rem;
            font-weight: 600;
            color: #1a202c;
            margin: 0 0 0.4rem 0;
            line-height: 1.3;
            display: -webkit-box;
            -webkit-line-clamp: 2;
            -webkit-box-orient: vertical;
            overflow: hidden;
            min-height: 2.4rem;
        }

        .pa-course-instructor {
            font-size: 0.75rem;
            color: #64748b;
            margin-bottom: 0.8rem;
            font-weight: 400;
        }

        .pa-course-price {
            margin-top: auto;
            font-weight: 600;
        }

        .pa-price {
            color: #2563eb;
            font-size: 1rem;
        }

        .pa-price-free {
            color: #10b981;
            font-size: 0.9rem;
            font-weight: 700;
        }

        /* Course Action Buttons in Visible Content Area */
        .pa-course-actions {
            display: flex;
            gap: 0.5rem;
            margin-top: 0.8rem;
            justify-content: space-between;
        }

        .pa-course-btn {
            flex: 1;
            padding: 0.4rem 0.6rem;
            font-size: 0.75rem;
            font-weight: 600;
            border: none;
            border-radius: 6px;
            text-decoration: none;
            text-align: center;
            cursor: pointer;
            transition: all 0.2s ease;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 0.3rem;
        }

        .pa-course-view-btn {
            background: #f1f5f9;
            color: #475569;
            border: 1px solid #e2e8f0;
        }

        .pa-course-view-btn:hover {
            background: #e2e8f0;
            color: #334155;
            text-decoration: none;
        }

        .pa-course-cart-btn {
            background: #f59e0b !important;
            color: white !important;
            border: 1px solid #f59e0b !important;
        }

        .pa-course-cart-btn:hover {
            background: #d97706 !important;
            border-color: #d97706 !important;
        }

        .pa-course-btn i {
            font-size: 0.7rem;
        }

        /* Empty State */
        .pa-category-empty-state {
            padding: 3rem 1rem;
        }

        .pa-empty-title {
            font-size: 1.3rem;
            font-weight: 600;
            color: #64748b;
            margin-bottom: 0.5rem;
        }

        .pa-empty-text {
            font-size: 0.9rem;
            color: #94a3b8;
            margin: 0;
        }

        /* Responsive Design */
        @media (max-width: 768px) {
            .pa-category-main-title {
                font-size: 1.8rem;
            }

            .pa-tabs-scroll-container {
                margin: 0 15px;
            }

            .pa-tabs-scroll-btn {
                display: flex;
            }

            .pa-category-tabs-nav {
                justify-content: flex-start;
                flex-wrap: nowrap;
                padding: 0.3rem;
                margin: 0 10px;
            }

            .pa-category-tab-btn {
                flex-shrink: 0;
                padding: 0.6rem 1rem;
                font-size: 0.8rem;
            }

            .pa-category-course-card {
                min-width: 200px;
                max-width: 200px;
                width: 200px;
            }

            .pa-course-thumbnail {
                height: 110px;
            }

            .pa-scroll-btn {
                width: 36px;
                height: 36px;
            }
        }

        @media (max-width: 480px) {
            .pa-category-main-title {
                font-size: 1.6rem;
            }

            .pa-course-content {
                padding: 0.8rem;
            }

            .pa-category-course-card {
                min-width: 180px;
                max-width: 180px;
                width: 180px;
            }
        }
    </style>

    <!-- JavaScript for Category Switching and Horizontal Scrolling -->
    <script>
        // Category switching function with PA prefix to avoid conflicts
        function paShowCategory(categoryName) {
            console.log('Switching to category:', categoryName);
            
            // Remove active class from all tabs
            document.querySelectorAll('.pa-category-tab-btn').forEach(btn => {
                btn.classList.remove('pa-active');
            });
            
            // Add active class to clicked tab
            const activeTab = document.querySelector(`[data-category="${categoryName}"]`);
            if (activeTab) {
                activeTab.classList.add('pa-active');
                console.log('Found and activated tab:', categoryName);
            } else {
                console.log('Tab not found for category:', categoryName);
            }
            
            // Hide all panels
            document.querySelectorAll('.pa-category-panel').forEach(panel => {
                panel.classList.remove('pa-active');
                panel.style.display = 'none';
            });
            
            // Show selected panel
            const activePanel = document.querySelector(`[data-category-panel="${categoryName}"]`);
            if (activePanel) {
                activePanel.classList.add('pa-active');
                activePanel.style.display = 'block';
                console.log('Found and activated panel:', categoryName);
            } else {
                console.log('Panel not found for category:', categoryName);
            }
        }

        // Category tabs horizontal scrolling function
        function paScrollTabs(direction) {
            const tabsContainer = document.getElementById('pa-category-tabs-scroll');
            if (!tabsContainer) return;
            
            const tabWidth = 150; // Approximate tab width + gap
            const scrollAmount = tabWidth + 20;
            const currentScroll = tabsContainer.scrollLeft;
            
            if (direction === 'left') {
                tabsContainer.scrollTo({
                    left: currentScroll - scrollAmount,
                    behavior: 'smooth'
                });
            } else if (direction === 'right') {
                tabsContainer.scrollTo({
                    left: currentScroll + scrollAmount,
                    behavior: 'smooth'
                });
            }
        }

        // Horizontal scrolling function for course cards
        function paScrollCategory(categoryName, direction) {
            const scrollContainer = document.getElementById(`pa-scroll-${categoryName}`);
            if (!scrollContainer) return;
            
            const cardWidth = 220; // Card width + gap
            const scrollAmount = cardWidth + 16; // Include gap
            const currentScroll = scrollContainer.scrollLeft;
            
            if (direction === 'left') {
                scrollContainer.scrollTo({
                    left: currentScroll - scrollAmount,
                    behavior: 'smooth'
                });
            } else if (direction === 'right') {
                scrollContainer.scrollTo({
                    left: currentScroll + scrollAmount,
                    behavior: 'smooth'
                });
            }
        }

        // Initialize scroll button visibility and functionality
        document.addEventListener('DOMContentLoaded', function() {
            console.log('Initializing category system...');
            
            // Debug: Log available categories and panels
            console.log('Available category tabs:', document.querySelectorAll('.pa-category-tab-btn').length);
            console.log('Available category panels:', document.querySelectorAll('.pa-category-panel').length);
            
            // Log each panel's data attribute
            document.querySelectorAll('.pa-category-panel').forEach(panel => {
                console.log('Panel found:', panel.getAttribute('data-category-panel'));
            });
            
            // Initialize category tabs - show "All Courses" by default
            paShowCategory('all');
            
            // Add hover effects for course cards
            const courseCards = document.querySelectorAll('.pa-category-course-card');
            
            courseCards.forEach(card => {
                card.addEventListener('mouseenter', function() {
                    this.style.transform = 'translateY(-2px)';
                });
                
                card.addEventListener('mouseleave', function() {
                    this.style.transform = 'translateY(0)';
                });
            });

            // Handle scroll button visibility
            function updateScrollButtons() {
                const scrollContainers = document.querySelectorAll('.pa-category-courses-scroll');
                
                scrollContainers.forEach(container => {
                    const scrollWrapper = container.parentElement;
                    const leftBtn = scrollWrapper.querySelector('.pa-scroll-left');
                    const rightBtn = scrollWrapper.querySelector('.pa-scroll-right');
                    
                    if (container.scrollWidth <= container.clientWidth) {
                        // Hide buttons if content doesn't overflow
                        if (leftBtn) leftBtn.style.display = 'none';
                        if (rightBtn) rightBtn.style.display = 'none';
                    } else {
                        // Show buttons if content overflows
                        if (leftBtn) leftBtn.style.display = 'flex';
                        if (rightBtn) rightBtn.style.display = 'flex';
                        
                        // Update button states based on scroll position
                        function updateButtonStates() {
                            const isAtStart = container.scrollLeft <= 0;
                            const isAtEnd = container.scrollLeft >= container.scrollWidth - container.clientWidth - 1;
                            
                            if (leftBtn) {
                                leftBtn.style.opacity = isAtStart ? '0.5' : '1';
                                leftBtn.style.pointerEvents = isAtStart ? 'none' : 'auto';
                            }
                            
                            if (rightBtn) {
                                rightBtn.style.opacity = isAtEnd ? '0.5' : '1';
                                rightBtn.style.pointerEvents = isAtEnd ? 'none' : 'auto';
                            }
                        }
                        
                        // Initial button state
                        updateButtonStates();
                        
                        // Update button states on scroll
                        container.addEventListener('scroll', updateButtonStates);
                    }
                });
            }

            // Initialize tab scroll buttons
            function updateTabScrollButtons() {
                const tabsContainer = document.getElementById('pa-category-tabs-scroll');
                const tabsWrapper = document.querySelector('.pa-tabs-scroll-container');
                
                if (!tabsContainer || !tabsWrapper) return;
                
                const leftBtn = tabsWrapper.querySelector('.pa-tabs-scroll-left');
                const rightBtn = tabsWrapper.querySelector('.pa-tabs-scroll-right');
                
                if (window.innerWidth <= 768) {
                    // On mobile, check if tabs overflow
                    if (tabsContainer.scrollWidth > tabsContainer.clientWidth) {
                        if (leftBtn) leftBtn.style.display = 'flex';
                        if (rightBtn) rightBtn.style.display = 'flex';
                        
                        // Update button states
                        function updateTabButtonStates() {
                            const isAtStart = tabsContainer.scrollLeft <= 0;
                            const isAtEnd = tabsContainer.scrollLeft >= tabsContainer.scrollWidth - tabsContainer.clientWidth - 1;
                            
                            if (leftBtn) {
                                leftBtn.style.opacity = isAtStart ? '0.5' : '1';
                                leftBtn.style.pointerEvents = isAtStart ? 'none' : 'auto';
                            }
                            
                            if (rightBtn) {
                                rightBtn.style.opacity = isAtEnd ? '0.5' : '1';
                                rightBtn.style.pointerEvents = isAtEnd ? 'none' : 'auto';
                            }
                        }
                        
                        updateTabButtonStates();
                        tabsContainer.addEventListener('scroll', updateTabButtonStates);
                    } else {
                        if (leftBtn) leftBtn.style.display = 'none';
                        if (rightBtn) rightBtn.style.display = 'none';
                    }
                } else {
                    // Hide buttons on desktop
                    if (leftBtn) leftBtn.style.display = 'none';
                    if (rightBtn) rightBtn.style.display = 'none';
                }
            }

            // Initial update
            updateScrollButtons();
            updateTabScrollButtons();
            
            // Update on window resize
            window.addEventListener('resize', function() {
                updateScrollButtons();
                updateTabScrollButtons();
            });
            
            // Update when category changes
            const originalShowCategory = paShowCategory;
            paShowCategory = function(categoryName) {
                originalShowCategory(categoryName);
                setTimeout(updateScrollButtons, 100);
            };

            // Add touch/mouse wheel support for horizontal scrolling
            const scrollContainers = document.querySelectorAll('.pa-category-courses-scroll');
            
            scrollContainers.forEach(container => {
                // Mouse wheel horizontal scrolling
                container.addEventListener('wheel', function(e) {
                    if (e.deltaY !== 0) {
                        e.preventDefault();
                        container.scrollLeft += e.deltaY > 0 ? 50 : -50;
                    }
                });
                
                // Touch/swipe support
                let isDown = false;
                let startX;
                let scrollLeft;
                
                container.addEventListener('mousedown', (e) => {
                    if (e.target.closest('.pa-course-overlay')) return; // Don't interfere with button clicks
                    isDown = true;
                    container.style.cursor = 'grabbing';
                    startX = e.pageX - container.offsetLeft;
                    scrollLeft = container.scrollLeft;
                });
                
                container.addEventListener('mouseleave', () => {
                    isDown = false;
                    container.style.cursor = 'grab';
                });
                
                container.addEventListener('mouseup', () => {
                    isDown = false;
                    container.style.cursor = 'grab';
                });
                
                container.addEventListener('mousemove', (e) => {
                    if (!isDown) return;
                    e.preventDefault();
                    const x = e.pageX - container.offsetLeft;
                    const walk = (x - startX) * 2;
                    container.scrollLeft = scrollLeft - walk;
                });
                
                // Set initial cursor
                container.style.cursor = 'grab';
            });
        });

        // Expose functions globally to maintain backward compatibility
        window.paShowCategory = paShowCategory;
        window.paScrollCategory = paScrollCategory;
        window.paScrollTabs = paScrollTabs;
    </script>

    <!-- Typing Animation Script -->
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            const typingElement = document.querySelector('.pa-typed-text');
            const cursor = document.querySelector('.pa-typing-cursor');
            
            if (typingElement && cursor) {
                const words = ['skills today', 'future ready', 'career goals'];
                let currentWordIndex = 0;
                let currentCharIndex = 0;
                let isDeleting = false;
                const typingSpeed = 100;
                const deletingSpeed = 50;
                const pauseTime = 2000;

                function type() {
                    const currentWord = words[currentWordIndex];
                    
                    if (isDeleting) {
                        typingElement.textContent = currentWord.substring(0, currentCharIndex - 1);
                        currentCharIndex--;
                        
                        if (currentCharIndex === 0) {
                            isDeleting = false;
                            currentWordIndex = (currentWordIndex + 1) % words.length;
                            setTimeout(type, 500);
                            return;
                        }
                        setTimeout(type, deletingSpeed);
                    } else {
                        typingElement.textContent = currentWord.substring(0, currentCharIndex + 1);
                        currentCharIndex++;
                        
                        if (currentCharIndex === currentWord.length) {
                            setTimeout(() => {
                                isDeleting = true;
                                type();
                            }, pauseTime);
                            return;
                        }
                        setTimeout(type, typingSpeed);
                    }
                }

                // Start the typing animation after a brief delay
                setTimeout(type, 1000);
            }
        });

        // Referral Carousel Functionality
        let currentSlide = 1;
        const totalSlides = 3;
        let autoSlideInterval;

        // Function to change slide
        function changeSlide(direction) {
            const slides = document.querySelectorAll('.carousel-slide');
            const dots = document.querySelectorAll('.dot');
            
            // Remove active class from current slide
            slides[currentSlide - 1].classList.remove('active');
            dots[currentSlide - 1].classList.remove('active');
            
            // Calculate new slide
            if (direction === 1) {
                currentSlide = currentSlide === totalSlides ? 1 : currentSlide + 1;
            } else {
                currentSlide = currentSlide === 1 ? totalSlides : currentSlide - 1;
            }
            
            // Add active class to new slide
            slides[currentSlide - 1].classList.add('active');
            dots[currentSlide - 1].classList.add('active');
            
            // Reset auto-slide timer
            resetAutoSlide();
        }

        // Function to go to specific slide
        function currentSlideFunc(slideNumber) {
            const slides = document.querySelectorAll('.carousel-slide');
            const dots = document.querySelectorAll('.dot');
            
            // Remove active class from current slide
            slides[currentSlide - 1].classList.remove('active');
            dots[currentSlide - 1].classList.remove('active');
            
            // Set new slide
            currentSlide = slideNumber;
            
            // Add active class to new slide
            slides[currentSlide - 1].classList.add('active');
            dots[currentSlide - 1].classList.add('active');
            
            // Reset auto-slide timer
            resetAutoSlide();
        }

        // Auto-slide functionality
        function startAutoSlide() {
            autoSlideInterval = setInterval(() => {
                changeSlide(1);
            }, 5000); // Change slide every 5 seconds
        }

        function resetAutoSlide() {
            clearInterval(autoSlideInterval);
            startAutoSlide();
        }

        function stopAutoSlide() {
            clearInterval(autoSlideInterval);
        }

        // Initialize carousel when DOM is loaded
        document.addEventListener('DOMContentLoaded', function() {
            const carousel = document.querySelector('.referral-carousel-container');
            if (carousel) {
                // Start auto-slide
                startAutoSlide();
                
                // Pause auto-slide on hover
                carousel.addEventListener('mouseenter', stopAutoSlide);
                carousel.addEventListener('mouseleave', startAutoSlide);
                
                // Add click event listeners to dots
                document.querySelectorAll('.dot').forEach((dot, index) => {
                    dot.addEventListener('click', () => currentSlideFunc(index + 1));
                });
                
                // Add click event listeners to navigation buttons
                const prevBtn = document.querySelector('.prev-btn');
                const nextBtn = document.querySelector('.next-btn');
                
                if (prevBtn) {
                    prevBtn.addEventListener('click', () => changeSlide(-1));
                }
                
                if (nextBtn) {
                    nextBtn.addEventListener('click', () => changeSlide(1));
                }
                
                // Add keyboard navigation
                document.addEventListener('keydown', function(e) {
                    if (carousel.querySelector('.carousel-slide.active')) {
                        if (e.key === 'ArrowLeft') {
                            changeSlide(-1);
                        } else if (e.key === 'ArrowRight') {
                            changeSlide(1);
                        }
                    }
                });

                // Add touch/swipe support for mobile
                let startX = 0;
                let endX = 0;
                
                carousel.addEventListener('touchstart', function(e) {
                    startX = e.changedTouches[0].screenX;
                });
                
                carousel.addEventListener('touchend', function(e) {
                    endX = e.changedTouches[0].screenX;
                    handleSwipe();
                });
                
                function handleSwipe() {
                    const swipeThreshold = 50;
                    const diff = startX - endX;
                    
                    if (Math.abs(diff) > swipeThreshold) {
                        if (diff > 0) {
                            // Swipe left - next slide
                            changeSlide(1);
                        } else {
                            // Swipe right - previous slide
                            changeSlide(-1);
                        }
                    }
                }
            }
        });

        // Global functions for onclick handlers
        window.changeSlide = changeSlide;
        window.currentSlide = currentSlideFunc;
    </script>

    <!-- Our Partners Section -->
    <section class="partners-section py-5" style="background: #ffffff;">
        <div class="container">
            <!-- Section Header -->
            <div class="text-center mb-5">
                <h2 class="partners-main-title">
                    Our Students <span class="highlight-partners">Work at</span>
                </h2>
                <p class="partners-subtitle">
                    Join the League of Achievers and <strong>Make Your Way</strong> to these Global Brands.
                </p>
                <div class="text-end mt-3">
                    <a href="#" class="view-all-link">View All</a>
                </div>
            </div>

            <!-- Partners Scrolling Banner -->
            <div class="partners-scrolling-wrapper">
                <div class="partners-scrolling-track">
                    <!-- First set of logos -->
                    <div class="partner-logo-container">
                        <img src="https://upload.wikimedia.org/wikipedia/commons/4/44/Microsoft_logo.svg" alt="Microsoft" class="partner-logo">
                    </div>
                    <div class="partner-logo-container">
                        <img src="https://upload.wikimedia.org/wikipedia/commons/2/2f/Google_2015_logo.svg" alt="Google" class="partner-logo">
                    </div>
                    <div class="partner-logo-container">
                        <img src="https://upload.wikimedia.org/wikipedia/commons/a/a9/Amazon_logo.svg" alt="Amazon" class="partner-logo">
                    </div>
                    <div class="partner-logo-container">
                        <img src="https://upload.wikimedia.org/wikipedia/commons/f/fa/Apple_logo_black.svg" alt="Apple" class="partner-logo">
                    </div>
                    <div class="partner-logo-container">
                        <img src="https://upload.wikimedia.org/wikipedia/commons/5/51/IBM_logo.svg" alt="IBM" class="partner-logo">
                    </div>
                    <div class="partner-logo-container">
                        <img src="https://upload.wikimedia.org/wikipedia/commons/5/50/Oracle_logo.svg" alt="Oracle" class="partner-logo">
                    </div>
                    <div class="partner-logo-container">
                        <img src="https://upload.wikimedia.org/wikipedia/commons/c/cd/Accenture.svg" alt="Accenture" class="partner-logo">
                    </div>
                    <div class="partner-logo-container">
                        <img src="https://upload.wikimedia.org/wikipedia/commons/1/15/Deloitte_Logo.svg" alt="Deloitte" class="partner-logo">
                    </div>
                    <div class="partner-logo-container">
                        <img src="https://upload.wikimedia.org/wikipedia/commons/0/08/Netflix_2015_logo.svg" alt="Netflix" class="partner-logo">
                    </div>
                    <div class="partner-logo-container">
                        <img src="https://upload.wikimedia.org/wikipedia/commons/b/bb/Tesla_T_symbol.svg" alt="Tesla" class="partner-logo">
                    </div>
                    <div class="partner-logo-container">
                        <img src="https://upload.wikimedia.org/wikipedia/commons/c/cc/Uber_logo_2018.svg" alt="Uber" class="partner-logo">
                    </div>
                    <div class="partner-logo-container">
                        <img src="https://upload.wikimedia.org/wikipedia/commons/7/7b/Meta_Platforms_Inc._logo.svg" alt="Meta" class="partner-logo">
                    </div>
                    
                    <!-- Duplicate set for seamless scrolling -->
                    <div class="partner-logo-container">
                        <img src="https://upload.wikimedia.org/wikipedia/commons/4/44/Microsoft_logo.svg" alt="Microsoft" class="partner-logo">
                    </div>
                    <div class="partner-logo-container">
                        <img src="https://upload.wikimedia.org/wikipedia/commons/2/2f/Google_2015_logo.svg" alt="Google" class="partner-logo">
                    </div>
                    <div class="partner-logo-container">
                        <img src="https://upload.wikimedia.org/wikipedia/commons/a/a9/Amazon_logo.svg" alt="Amazon" class="partner-logo">
                    </div>
                    <div class="partner-logo-container">
                        <img src="https://upload.wikimedia.org/wikipedia/commons/f/fa/Apple_logo_black.svg" alt="Apple" class="partner-logo">
                    </div>
                    <div class="partner-logo-container">
                        <img src="https://upload.wikimedia.org/wikipedia/commons/5/51/IBM_logo.svg" alt="IBM" class="partner-logo">
                    </div>
                    <div class="partner-logo-container">
                        <img src="https://upload.wikimedia.org/wikipedia/commons/5/50/Oracle_logo.svg" alt="Oracle" class="partner-logo">
                    </div>
                    <div class="partner-logo-container">
                        <img src="https://upload.wikimedia.org/wikipedia/commons/c/cd/Accenture.svg" alt="Accenture" class="partner-logo">
                    </div>
                    <div class="partner-logo-container">
                        <img src="https://upload.wikimedia.org/wikipedia/commons/1/15/Deloitte_Logo.svg" alt="Deloitte" class="partner-logo">
                    </div>
                    <div class="partner-logo-container">
                        <img src="https://upload.wikimedia.org/wikipedia/commons/0/08/Netflix_2015_logo.svg" alt="Netflix" class="partner-logo">
                    </div>
                    <div class="partner-logo-container">
                        <img src="https://upload.wikimedia.org/wikipedia/commons/b/bb/Tesla_T_symbol.svg" alt="Tesla" class="partner-logo">
                    </div>
                    <div class="partner-logo-container">
                        <img src="https://upload.wikimedia.org/wikipedia/commons/c/cc/Uber_logo_2018.svg" alt="Uber" class="partner-logo">
                    </div>
                    <div class="partner-logo-container">
                        <img src="https://upload.wikimedia.org/wikipedia/commons/7/7b/Meta_Platforms_Inc._logo.svg" alt="Meta" class="partner-logo">
                    </div>
                </div>
            </div>

            <!-- Disclaimer -->
            <div class="text-center mt-4">
                <p class="disclaimer-text">
                    <strong>Disclaimer:</strong> The companies listed here are provided for informational purposes only and do not constitute endorsement or affiliation with our platform.
                </p>
            </div>
        </div>
    </section>

    <!-- Partners Section Styles -->
    <style>
        /* Partners Section Styles */
        .partners-section {
            position: relative;
        }

        .partners-main-title {
            font-size: 2.5rem;
            font-weight: 700;
            color: #1a202c;
            margin-bottom: 1rem;
            line-height: 1.2;
        }

        .highlight-partners {
            color: #f59e0b;
            font-weight: 800;
            position: relative;
        }

        .partners-subtitle {
            font-size: 1.1rem;
            color: #64748b;
            line-height: 1.6;
            margin: 0;
            max-width: 600px;
            margin: 0 auto;
        }

        .view-all-link {
            color: #f59e0b;
            text-decoration: none;
            font-weight: 600;
            font-size: 1rem;
            transition: all 0.3s ease;
        }

        .view-all-link:hover {
            color: #d97706;
            text-decoration: underline;
        }

        .partners-scrolling-wrapper {
            overflow: hidden;
            width: 100%;
            margin-top: 3rem;
            position: relative;
        }
        
        .partners-scrolling-wrapper::before,
        .partners-scrolling-wrapper::after {
            content: '';
            position: absolute;
            top: 0;
            width: 100px;
            height: 100%;
            z-index: 2;
            pointer-events: none;
        }
        
        .partners-scrolling-wrapper::before {
            left: 0;
            background: linear-gradient(90deg, rgba(255,255,255,1) 0%, rgba(255,255,255,0) 100%);
        }
        
        .partners-scrolling-wrapper::after {
            right: 0;
            background: linear-gradient(270deg, rgba(255,255,255,1) 0%, rgba(255,255,255,0) 100%);
        }

        .partners-scrolling-track {
            display: flex;
            width: calc(200px * 24); /* 24 logos × 200px each */
            animation: scroll-partners 60s linear infinite;
        }

        .partners-scrolling-track:hover {
            animation-play-state: paused;
        }

        @keyframes scroll-partners {
            0% {
                transform: translateX(0);
            }
            100% {
                transform: translateX(calc(-200px * 12)); /* Move by half the width (12 logos) */
            }
        }

        .partner-logo-container {
            display: flex;
            align-items: center;
            justify-content: center;
            width: 200px;
            height: 100px;
            padding: 1.5rem 1rem;
            background: #ffffff;
            border-radius: 12px;
            box-shadow: 0 2px 8px rgba(0, 0, 0, 0.06);
            transition: all 0.3s ease;
            border: 1px solid #f1f5f9;
            margin-right: 2rem;
            flex-shrink: 0;
        }

        .partner-logo-container:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 16px rgba(0, 0, 0, 0.1);
            border-color: #e2e8f0;
        }

        .partner-logo {
            max-width: 120px;
            max-height: 60px;
            width: auto;
            height: auto;
            object-fit: contain;
            filter: grayscale(100%) opacity(0.7);
            transition: all 0.3s ease;
        }

        .partner-logo-container:hover .partner-logo {
            filter: grayscale(0%) opacity(1);
            transform: scale(1.05);
        }

        .disclaimer-text {
            font-size: 0.85rem;
            color: #64748b;
            font-style: italic;
            max-width: 800px;
            margin: 0 auto;
            line-height: 1.4;
        }

        /* Responsive Design */
        @media (max-width: 992px) {
            .partners-main-title {
                font-size: 2rem;
            }

            .partners-subtitle {
                font-size: 1rem;
            }
        }

        @media (max-width: 768px) {
            .partners-main-title {
                font-size: 1.75rem;
            }

            .partners-subtitle {
                font-size: 0.95rem;
            }

            .partner-logo-container {
                height: 80px;
                padding: 1rem 0.5rem;
            }

            .partner-logo {
                max-height: 45px;
            }

            .partners-grid {
                margin-top: 2rem;
            }
        }

        @media (max-width: 576px) {
            .partner-logo-container {
                height: 70px;
                padding: 0.75rem 0.25rem;
            }

            .partner-logo {
                max-height: 35px;
            }
        }
    </style>

    <!-- FAQ Section -->
    <section class="faq-section py-5" style="background: #f8fafc;">
        <div class="container">
            <!-- Section Header -->
            <div class="text-center mb-5">
                <h2 class="faq-main-title">
                    Frequently Asked <span class="highlight-questions">Questions</span>
                </h2>
            </div>

            <!-- FAQ Content -->
            <div class="row justify-content-center">
                <div class="col-lg-8">
                    <div class="faq-accordion">
                        <!-- FAQ Item 1 -->
                        <div class="faq-item">
                            <div class="faq-question" onclick="toggleFAQ(this)">
                                <span>How do we learn with Panadite Academy?</span>
                                <div class="faq-toggle">
                                    <i class="fas fa-plus"></i>
                                </div>
                            </div>
                            <div class="faq-answer">
                                <p>Panadite Academy offers flexible online learning through our comprehensive platform. You can access high-quality courses, interactive content, live sessions, and personalized mentorship from industry experts. Our stackable learning approach allows you to build your skills progressively at your own pace.</p>
                            </div>
                        </div>

                        <!-- FAQ Item 2 -->
                        <div class="faq-item">
                            <div class="faq-question" onclick="toggleFAQ(this)">
                                <span>Are Panadite Academy's programs Accredited?</span>
                                <div class="faq-toggle">
                                    <i class="fas fa-plus"></i>
                                </div>
                            </div>
                            <div class="faq-answer">
                                <p>Yes, our programs are globally recognized and accredited by leading educational bodies. Upon successful completion, you'll receive certificates that are valued by employers worldwide and can significantly enhance your career prospects.</p>
                            </div>
                        </div>

                        <!-- FAQ Item 3 -->
                        <div class="faq-item">
                            <div class="faq-question" onclick="toggleFAQ(this)">
                                <span>What is the eligibility for our Professional Development Programs?</span>
                                <div class="faq-toggle">
                                    <i class="fas fa-plus"></i>
                                </div>
                            </div>
                            <div class="faq-answer">
                                <p>Our programs are designed for learners at all levels. Whether you're a recent graduate, working professional, or looking to switch careers, we have courses suitable for you. Some advanced programs may require basic prerequisites, which are clearly mentioned in the course descriptions.</p>
                            </div>
                        </div>

                        <!-- FAQ Item 4 -->
                        <div class="faq-item">
                            <div class="faq-question" onclick="toggleFAQ(this)">
                                <span>How will Upskilling help grow my career?</span>
                                <div class="faq-toggle">
                                    <i class="fas fa-plus"></i>
                                </div>
                            </div>
                            <div class="faq-answer">
                                <p>Upskilling with Panadite Academy keeps you relevant in today's rapidly evolving job market. Our industry-aligned curriculum ensures you learn in-demand skills, making you more valuable to employers and opening doors to promotions, salary increases, and new career opportunities.</p>
                            </div>
                        </div>

                        <!-- FAQ Item 5 -->
                        <div class="faq-item">
                            <div class="faq-question" onclick="toggleFAQ(this)">
                                <span>Who does Panadite Academy partner with?</span>
                                <div class="faq-toggle">
                                    <i class="fas fa-plus"></i>
                                </div>
                            </div>
                            <div class="faq-answer">
                                <p>We partner with leading corporations, educational institutions, and industry experts to ensure our curriculum meets current market demands. Our partnerships with companies like Microsoft, Google, Amazon, and others help provide real-world insights and potential career opportunities for our graduates.</p>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </section>

    <!-- FAQ Section Styles -->
    <style>
        /* FAQ Section Styles */
        .faq-section {
            position: relative;
        }

        .faq-main-title {
            font-size: 2.5rem;
            font-weight: 700;
            color: #1a202c;
            margin-bottom: 1rem;
            line-height: 1.2;
        }

        .highlight-questions {
            color: #f59e0b;
            font-weight: 800;
            position: relative;
        }

        .faq-accordion {
            background: white;
            border-radius: 16px;
            box-shadow: 0 4px 16px rgba(0, 0, 0, 0.08);
            overflow: hidden;
        }

        .faq-item {
            border-bottom: 1px solid #f1f5f9;
        }

        .faq-item:last-child {
            border-bottom: none;
        }

        .faq-question {
            padding: 1.5rem;
            display: flex;
            align-items: center;
            justify-content: space-between;
            cursor: pointer;
            transition: all 0.3s ease;
            background: white;
        }

        .faq-question:hover {
            background: #f8fafc;
        }

        .faq-question span {
            font-size: 1.1rem;
            font-weight: 600;
            color: #1a202c;
            flex: 1;
            margin-right: 1rem;
        }

        .faq-toggle {
            width: 32px;
            height: 32px;
            background: #f59e0b;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-size: 0.9rem;
            transition: all 0.3s ease;
            flex-shrink: 0;
        }

        .faq-item.active .faq-toggle {
            background: #2563eb;
            transform: rotate(45deg);
        }

        .faq-answer {
            max-height: 0;
            overflow: hidden;
            transition: max-height 0.3s ease;
            background: #f8fafc;
        }

        .faq-item.active .faq-answer {
            max-height: 200px;
        }

        .faq-answer p {
            padding: 1.5rem;
            margin: 0;
            color: #64748b;
            line-height: 1.6;
            font-size: 1rem;
        }

        /* Responsive Design */
        @media (max-width: 768px) {
            .faq-main-title {
                font-size: 2rem;
            }

            .faq-question span {
                font-size: 1rem;
            }

            .faq-question {
                padding: 1rem;
            }

            .faq-answer p {
                padding: 1rem;
                font-size: 0.95rem;
            }
        }
    </style>

    <!-- FAQ JavaScript -->
    <script>
        function toggleFAQ(element) {
            const faqItem = element.parentElement;
            const isActive = faqItem.classList.contains('active');
            
            // Close all other FAQ items
            document.querySelectorAll('.faq-item').forEach(item => {
                item.classList.remove('active');
            });
            
            // Toggle current item
            if (!isActive) {
                faqItem.classList.add('active');
            }
        }
    </script>

    <!-- Call to Action Banner -->
    <section class="cta-banner-section py-5" style="background: #ffffff;">
        <div class="container">
            <div class="cta-banner">
                <div class="row align-items-center">
                    <div class="col-lg-8">
                        <div class="cta-content">
                            <h2 class="cta-title">
                                It's Time to Start<br>
                                <span class="highlight-investing">Investing In Yourself</span>
                            </h2>
                        </div>
                    </div>
                    <div class="col-lg-4 text-end">
                        <a href="#" class="cta-join-btn">Join Now</a>
                    </div>
                </div>
            </div>
        </div>
    </section>

    <!-- CTA Banner Styles -->
    <style>
        /* CTA Banner Styles */
        .cta-banner-section {
            position: relative;
        }

        .cta-banner {
            background: linear-gradient(135deg, #f59e0b 0%, #ea580c 50%, #dc2626 100%);
            border-radius: 20px;
            padding: 3rem 2rem;
            position: relative;
            overflow: hidden;
            min-height: 200px;
        }

        .cta-banner::before {
            content: '';
            position: absolute;
            top: 0;
            right: 0;
            width: 50%;
            height: 100%;
            background: url('data:image/svg+xml,<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 400 200"><rect fill="%23ffffff" opacity="0.1" width="20" height="200" x="20"/><rect fill="%23ffffff" opacity="0.1" width="20" height="200" x="50"/><rect fill="%23ffffff" opacity="0.1" width="20" height="200" x="80"/><rect fill="%23ffffff" opacity="0.1" width="20" height="200" x="110"/><rect fill="%23ffffff" opacity="0.1" width="20" height="200" x="140"/><rect fill="%23ffffff" opacity="0.1" width="20" height="200" x="170"/><rect fill="%23ffffff" opacity="0.1" width="20" height="200" x="200"/><rect fill="%23ffffff" opacity="0.1" width="20" height="200" x="230"/><rect fill="%23ffffff" opacity="0.1" width="20" height="200" x="260"/><rect fill="%23ffffff" opacity="0.1" width="20" height="200" x="290"/></svg>') center/cover;
            opacity: 0.3;
            z-index: 1;
        }

        .cta-banner::after {
            content: '';
            position: absolute;
            top: -20px;
            right: -20px;
            width: 100px;
            height: 100px;
            background: rgba(255, 255, 255, 0.1);
            border-radius: 50%;
            z-index: 1;
        }

        .cta-content {
            position: relative;
            z-index: 2;
        }

        .cta-title {
            font-size: 2.5rem;
            font-weight: 700;
            color: white;
            line-height: 1.2;
            margin: 0;
        }

        .highlight-investing {
            color: #fff3cd;
            text-shadow: 2px 2px 4px rgba(0, 0, 0, 0.3);
        }

        .cta-join-btn {
            background: white;
            color: #f59e0b;
            padding: 1rem 2.5rem;
            border-radius: 50px;
            font-size: 1.1rem;
            font-weight: 700;
            text-decoration: none;
            display: inline-block;
            transition: all 0.3s ease;
            box-shadow: 0 4px 16px rgba(0, 0, 0, 0.2);
            position: relative;
            z-index: 2;
        }

        .cta-join-btn:hover {
            background: #f8fafc;
            color: #ea580c;
            transform: translateY(-2px);
            box-shadow: 0 8px 24px rgba(0, 0, 0, 0.3);
        }

        /* Responsive Design */
        @media (max-width: 992px) {
            .cta-banner {
                text-align: center;
            }

            .cta-title {
                font-size: 2rem;
                margin-bottom: 2rem;
            }
        }

        @media (max-width: 768px) {
            .cta-banner {
                padding: 2rem 1.5rem;
            }

            .cta-title {
                font-size: 1.75rem;
            }

            .cta-join-btn {
                padding: 0.875rem 2rem;
                font-size: 1rem;
            }
        }
    </style>

<!-- Footer -->
<footer class="mt-5 position-relative">
        <!-- Top gradient bar -->
        <div class="position-absolute top-0 start-0 w-100" style="height: 4px; background: linear-gradient(90deg, #276BE4, #FFA915);"></div>
        
        <!-- Main footer content -->
        <div class="bg-white py-5">
            <div class="container">
                <div class="row gy-4">
                    <!-- Logo and description -->
                    <div class="col-lg-4 col-md-6 mb-4">
                        <div class="mb-4">
                            <img src="/assets/images/logo.jpeg" alt="Panadite Academy" style="height: 50px;">
                        </div>
                        <p class="mb-2" style="color: #333; font-weight: 500;">Empowering learners through high-quality educational content.</p>
                        <p style="color: #333;">Get ahead of the curve with our premium videos and courses.</p>
                    </div>
                    
                    <!-- Quick links -->
                    <div class="col-lg-2 col-md-6 mb-4">
                        <h5 class="fw-bold mb-3 pb-2 position-relative" style="color: #222;">
                            Quick Links
                            <span class="position-absolute bottom-0 start-0" style="height: 2px; width: 40px; background: #FFA915;"></span>
                        </h5>
                        <ul class="list-unstyled">
                            <li class="mb-2"><a href="/index.php" class="text-decoration-none" style="color: #333;"><i class="fas fa-angle-right me-2" style="color: #276BE4;"></i>Home</a></li>
                            <li class="mb-2"><a href="/courses.php" class="text-decoration-none" style="color: #333;"><i class="fas fa-angle-right me-2" style="color: #276BE4;"></i>Courses</a></li>
                            <li class="mb-2"><a href="/videos.php" class="text-decoration-none" style="color: #333;"><i class="fas fa-angle-right me-2" style="color: #276BE4;"></i>Videos</a></li>
                            <li class="mb-2"><a href="/about.php" class="text-decoration-none" style="color: #333;"><i class="fas fa-angle-right me-2" style="color: #276BE4;"></i>About Us</a></li>
                            <li class="mb-2"><a href="/contact.php" class="text-decoration-none" style="color: #333;"><i class="fas fa-angle-right me-2" style="color: #276BE4;"></i>Contact</a></li>
                        </ul>
                    </div>
                    
                    <!-- Facilitator links -->
                    <div class="col-lg-2 col-md-6 mb-4">
                        <h5 class="fw-bold mb-3 pb-2 position-relative" style="color: #222;">
                            Facilitator
                            <span class="position-absolute bottom-0 start-0" style="height: 2px; width: 40px; background: #FFA915;"></span>
                        </h5>
                        <ul class="list-unstyled">
                            <li class="mb-2"><a href="/auth/register.php?role=teacher" class="text-decoration-none" style="color: #333;"><i class="fas fa-angle-right me-2" style="color: #276BE4;"></i>Become a Facilitator</a></li>
                            <li class="mb-2"><a href="/teacher/dashboard.php" class="text-decoration-none" style="color: #333;"><i class="fas fa-angle-right me-2" style="color: #276BE4;"></i>Teacher Dashboard</a></li>
                            <li class="mb-2"><a href="/teacher/sales.php" class="text-decoration-none" style="color: #333;"><i class="fas fa-angle-right me-2" style="color: #276BE4;"></i>Your Sales</a></li>
                            <li class="mb-2"><a href="/teacher/help.php" class="text-decoration-none" style="color: #333;"><i class="fas fa-angle-right me-2" style="color: #276BE4;"></i>Teacher Help</a></li>
                        </ul>
                    </div>
                    
                    <!-- Contact info -->
                    <div class="col-lg-4 col-md-6">
                        <h5 class="fw-bold mb-3 pb-2 position-relative" style="color: #222;">
                            Contact
                            <span class="position-absolute bottom-0 start-0" style="height: 2px; width: 40px; background: #FFA915;"></span>
                        </h5>
                        <ul class="list-unstyled mb-4">
                            <li class="mb-3 d-flex align-items-center">
                                <span class="d-flex align-items-center justify-content-center me-3" style="width: 36px; height: 36px; background: rgba(39,107,228,0.1); border-radius: 50%;">
                                    <i class="fas fa-map-marker-alt" style="color: #276BE4;"></i>
                                </span>
                                <span style="color: #333;">296 Pretorius Street, Corner Lilian Ngoyi, Pretoria Central, 0001, South Africa</span>
                            </li>
                            <li class="mb-3 d-flex align-items-center">
                                <span class="d-flex align-items-center justify-content-center me-3" style="width: 36px; height: 36px; background: rgba(39,107,228,0.1); border-radius: 50%;">
                                    <i class="fas fa-envelope" style="color: #276BE4;"></i>
                                </span>
                                <a href="mailto:admin@panaditeacademy.com" class="text-decoration-none" style="color: #333;">admin@panaditeacademy.com</a>
                            </li>
                            <li class="mb-3 d-flex align-items-center">
                                <span class="d-flex align-items-center justify-content-center me-3" style="width: 36px; height: 36px; background: rgba(39,107,228,0.1); border-radius: 50%;">
                                    <i class="fas fa-phone" style="color: #276BE4;"></i>
                                </span>
                                <a href="tel:+27747223141" class="text-decoration-none" style="color: #333;">+27 74 722 3141</a>
                            </li>
                        </ul>
                        
                        <!-- Social icons -->
                        <div class="d-flex">
                            <a href="#" class="d-flex align-items-center justify-content-center me-2" style="width: 40px; height: 40px; background: rgba(39,107,228,0.1); border-radius: 50%; color: #276BE4; transition: all 0.3s ease;">
                                <i class="fab fa-facebook-f"></i>
                            </a>
                            <a href="#" class="d-flex align-items-center justify-content-center me-2" style="width: 40px; height: 40px; background: rgba(39,107,228,0.1); border-radius: 50%; color: #276BE4; transition: all 0.3s ease;">
                                <i class="fab fa-twitter"></i>
                            </a>
                            <a href="#" class="d-flex align-items-center justify-content-center me-2" style="width: 40px; height: 40px; background: rgba(39,107,228,0.1); border-radius: 50%; color: #276BE4; transition: all 0.3s ease;">
                                <i class="fab fa-instagram"></i>
                            </a>
                            <a href="#" class="d-flex align-items-center justify-content-center" style="width: 40px; height: 40px; background: rgba(39,107,228,0.1); border-radius: 50%; color: #276BE4; transition: all 0.3s ease;">
                                <i class="fab fa-linkedin-in"></i>
                            </a>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Copyright section -->
        <div style="background: #f8f9fa; border-top: 1px solid #eee;" class="py-3">
            <div class="container">
                <div class="row align-items-center">
                    <div class="col-md-6 mb-3 mb-md-0">
                        <p class="mb-0" style="color: #333; font-weight: 500;">&copy; <?php echo date('Y'); ?> Panadite Academy. All rights reserved.</p>
                    </div>
                    <div class="col-md-6 text-md-end">
                        <a href="privacy-policy.php" class="text-decoration-none me-3" style="color: #333;">Privacy Policy</a>
                        <a href="terms-of-use.php" class="text-decoration-none me-3" style="color: #333;">Terms of Use</a>
                        <a href="/sitemap.php" class="text-decoration-none" style="color: #333;">Sitemap</a>
                    </div>
                </div>
            </div>
        </div>
    </footer>
    <script src="https://cdn.jsdelivr.net/npm/jquery@3.6.4/dist/jquery.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/gsap@3.11.4/dist/gsap.min.js"></script>
    <script src="/force-image.js"></script>
    
    <!-- Panadite Academy Guest Cart System -->
    <script src="/js/guest-cart-system.js"></script>
    <script>
        // Advanced floating icon animations
        function animateFloatingIcons() {
            $('.floating-icon').each(function(index) {
                var icon = $(this);
                var speedFactor = Math.random() * 0.5 + 0.5; // 0.5 to 1.0 speed variation
                
                // Set initial positions and scale
                var startScale = 0.9 + Math.random() * 0.2; // 0.9 to 1.1
                
                // Advanced animation properties
                var baseX = parseInt(icon.css('left'));
                var baseY = parseInt(icon.css('top'));
                if (isNaN(baseX)) baseX = parseInt(icon.css('right')) || 0;
                if (isNaN(baseY)) baseY = parseInt(icon.css('bottom')) || 0;
                
                var amplitude = 15 + Math.random() * 20; // 15-35px movement range
                var period = 5 + Math.random() * 5; // 5-10 second period
                
                // Function to create fluid floating effect using sine waves
                function animateIcon(timestamp) {
                    if (!icon.data('startTime')) {
                        icon.data('startTime', timestamp);
                    }
                    
                    // Calculate elapsed time for continuity
                    var elapsed = timestamp - icon.data('startTime');
                    
                    // Calculate position using sine/cosine waves with phase shift for each icon
                    var offsetX = amplitude * Math.sin((elapsed/1000) * speedFactor + index);
                    var offsetY = (amplitude * 0.7) * Math.cos((elapsed/1000) * speedFactor + index + 2);
                    
                    // Calculate scale oscillation (subtle)
                    var scale = startScale + 0.05 * Math.sin((elapsed/1000) * speedFactor * 0.5);
                    
                    // Calculate rotation (very subtle)
                    var rotation = 2 * Math.sin((elapsed/1000) * speedFactor * 0.3 + index);
                    
                    // Set transform with hardware acceleration
                    icon.css('transform', 
                        'translate3d(' + offsetX + 'px, ' + offsetY + 'px, 0) ' +
                        'scale(' + scale + ') ' +
                        'rotate(' + rotation + 'deg)'
                    );
                    
                    // Opacity pulsing for light variants
                    if (icon.hasClass('light')) {
                        var opacity = 0.6 + 0.2 * Math.sin((elapsed/1000) * 0.5);
                        icon.css('opacity', opacity);
                    }
                    
                    // Continue animation
                    requestAnimationFrame(animateIcon);
                }
                
                // Start with a delay based on index
                setTimeout(function() {
                    // Start the animation loop
                    requestAnimationFrame(animateIcon);
                }, index * 120);
            });
        }
        
        $(document).ready(function() {
            // Initialize the floating icon animation
            animateFloatingIcons();
            
            // Add smooth scrolling
            $("a.nav-link").on('click', function(event) {
                if (this.hash !== "") {
                    event.preventDefault();
                    var hash = this.hash;
                    $('html, body').animate({
                        scrollTop: $(hash).offset().top - 70
                    }, 800);
                }
            });
            
            // Initialize animations for elements that are in view
            function initAnimations() {
                $('.fade-in-element:not(.animated)').each(function() {
                    var elementTop = $(this).offset().top;
                    var viewportTop = $(window).scrollTop();
                    var windowHeight = $(window).height();
                    
                    if (elementTop < viewportTop + windowHeight - 100) {
                        $(this).addClass('animated');
                    }
                });
            }
            
            // Run animations on scroll
            $(window).on('scroll', function() {
                initAnimations();
            });
            
            // Run animations on page load
            initAnimations();
            
            // Modern Course Card Hover Effects
            $('.course-card').hover(
                function() {
                    $(this).css({
                        'transform': 'translateY(-4px)',
                        'box-shadow': '0 8px 25px rgba(0,0,0,0.15)'
                    });
                },
                function() {
                    $(this).css({
                        'transform': 'translateY(0)',
                        'box-shadow': '0 2px 8px rgba(0,0,0,0.08)'
                    });
                }
            );
            
            // EMERGENCY COURSE THUMBNAIL FIX
            console.log('Applying emergency thumbnail fix...');
            setTimeout(function() {
                var courseImages = document.querySelectorAll('.card img, .video-card img, .course-card img');
                console.log('Found ' + courseImages.length + ' course images');
                
                courseImages.forEach(function(img) {
                    console.log('Fixing image: ' + img.src);
                    img.style.cssText = 'display: block !important; visibility: visible !important; opacity: 1 !important; width: 100% !important; height: 100% !important; object-fit: cover !important;';
                    
                    // If image is broken, try reloading with timestamp
                    if (!img.complete || img.naturalWidth === 0) {
                        var originalSrc = img.src.split('?')[0];
                        img.src = originalSrc + '?t=' + new Date().getTime();
                        console.log('Reloading image: ' + img.src);
                    }
                });
                
                // Hide any SVG containers
                var svgContainers = document.querySelectorAll('.svg-bg, .svg-icon-container');
                svgContainers.forEach(function(container) {
                    container.style.display = 'none';
                });
            }, 500);
        });
        
        // Course Carousel Scrolling Functionality
        function scrollCourseCarousel(direction) {
            const container = document.getElementById('courseScrollContainer');
            const scrollAmount = 300; // Pixels to scroll
            
            if (direction === 'left') {
                container.scrollLeft -= scrollAmount;
            } else {
                container.scrollLeft += scrollAmount;
            }
            
            // Update navigation buttons visibility
            setTimeout(() => {
                updateCarouselButtons();
            }, 100);
        }
        
        // Update carousel navigation buttons based on scroll position
        function updateCarouselButtons() {
            const container = document.getElementById('courseScrollContainer');
            const prevBtn = document.querySelector('.carousel-prev');
            const nextBtn = document.querySelector('.carousel-next');
            
            if (!container || !prevBtn || !nextBtn) return;
            
            // Check if at the beginning
            if (container.scrollLeft <= 10) {
                prevBtn.style.opacity = '0.5';
                prevBtn.style.cursor = 'not-allowed';
            } else {
                prevBtn.style.opacity = '1';
                prevBtn.style.cursor = 'pointer';
            }
            
            // Check if at the end
            if (container.scrollLeft >= (container.scrollWidth - container.clientWidth - 10)) {
                nextBtn.style.opacity = '0.5';
                nextBtn.style.cursor = 'not-allowed';
            } else {
                nextBtn.style.opacity = '1';
                nextBtn.style.cursor = 'pointer';
            }
        }
        
        // Initialize carousel on page load
        $(document).ready(function() {
            // Hide scrollbars for course container
            const style = document.createElement('style');
            style.textContent = `
                .course-scroll-container::-webkit-scrollbar {
                    display: none;
                }
                .carousel-nav-btn:hover {
                    background: #f7f9fa !important;
                    box-shadow: 0 4px 12px rgba(0,0,0,0.15) !important;
                }
                .carousel-nav-btn:active {
                    transform: translateY(-50%) scale(0.95) !important;
                }
            `;
            document.head.appendChild(style);
            
            // Set initial button states
            setTimeout(() => {
                updateCarouselButtons();
            }, 500);
            
            // Update buttons on scroll
            const container = document.getElementById('courseScrollContainer');
            if (container) {
                container.addEventListener('scroll', updateCarouselButtons);
            }
        });
        
        // Category Carousel Scrolling Functionality
        function scrollCategoryCarousel(categoryName, direction) {
            const container = document.getElementById('categoryScrollContainer-' + categoryName);
            const scrollAmount = 300; // Pixels to scroll
            
            if (container) {
                if (direction === 'left') {
                    container.scrollLeft -= scrollAmount;
                } else {
                    container.scrollLeft += scrollAmount;
                }
            }
        }
        
        // Add to Cart Functionality
        function addToCart(courseId) {
            // Get existing cart from localStorage
            let cart = JSON.parse(localStorage.getItem('cart') || '[]');
            
            // Check if course is already in cart
            if (cart.find(item => item.courseId === courseId)) {
                showNotification('Course already in cart!', 'info');
                return;
            }
            
            // Extract course data from DOM
            const courseCard = document.querySelector(`[data-course-id="${courseId}"]`);
            if (!courseCard) {
                console.error('Course card not found for ID:', courseId);
                showNotification('Error: Course not found!', 'error');
                return;
            }
            
            // Extract course information from the DOM
            const courseTitle = courseCard.querySelector('.course-title')?.textContent?.trim() || 'Course';
            const courseInstructor = courseCard.querySelector('.course-instructor')?.textContent?.trim() || 'Instructor';
            const coursePriceElement = courseCard.querySelector('.course-price');
            const courseLevelElement = courseCard.querySelector('.course-level');
            const courseThumbnail = courseCard.querySelector('img')?.src || '/assets/images/default-thumbnail.jpg';
            
            // Extract price
            let coursePrice = 0;
            if (coursePriceElement) {
                const priceAttr = coursePriceElement.getAttribute('data-price');
                if (priceAttr) {
                    coursePrice = parseFloat(priceAttr) || 0;
                } else {
                    const priceText = coursePriceElement.textContent;
                    if (priceText.includes('Free')) {
                        coursePrice = 0;
                    } else {
                        const priceMatch = priceText.match(/R?([\d,]+\.?\d*)/);
                        coursePrice = priceMatch ? parseFloat(priceMatch[1].replace(',', '')) : 0;
                    }
                }
            }
            
            // Extract level
            const courseLevel = courseLevelElement?.getAttribute('data-level') || 
                               courseLevelElement?.textContent?.trim() || 'Beginner';
            
            // Add course to cart with all data
            cart.push({
                courseId: courseId,
                title: courseTitle,
                instructor: courseInstructor,
                price: coursePrice,
                level: courseLevel,
                thumbnail: courseThumbnail,
                addedAt: new Date().toISOString()
            });
            
            // Save to localStorage
            localStorage.setItem('cart', JSON.stringify(cart));
            
            // Update cart badge in navbar
            updateCartBadge();
            
            // Show success notification
            showNotification(`"${courseTitle}" added to cart!`, 'success');
        }
        
        // Update cart badge count
        function updateCartBadge() {
            const cart = JSON.parse(localStorage.getItem('cart') || '[]');
            const badge = document.querySelector('.cart-badge');
            if (badge) {
                badge.textContent = cart.length;
                badge.style.display = cart.length > 0 ? 'inline' : 'none';
            }
        }
        
        // Initialize cart badge on page load
        updateCartBadge();

        // Modern Category Tab Switching Function
        function switchCategoryTab(activeIndex) {
            // Hide all content panels
            document.querySelectorAll('.category-content').forEach(panel => {
                panel.style.display = 'none';
                panel.classList.remove('active');
            });
            
            // Show active panel
            const activePanel = document.getElementById('category-content-' + activeIndex);
            if (activePanel) {
                activePanel.style.display = 'block';
                activePanel.classList.add('active');
            }
            
            // Update tab styles
            document.querySelectorAll('.category-tab').forEach((tab, index) => {
                const isActive = index === activeIndex;
                
                if (isActive) {
                    // Active tab styling
                    tab.style.background = 'linear-gradient(135deg, #3b82f6 0%, #1d4ed8 100%)';
                    tab.style.color = 'white';
                    tab.style.boxShadow = '0 4px 12px rgba(59, 130, 246, 0.3)';
                    tab.style.transform = 'translateY(-1px)';
                    tab.classList.add('active');
                    
                    // Update count badge for active tab
                    const countBadge = tab.querySelector('.category-count');
                    if (countBadge) {
                        countBadge.style.background = 'rgba(255, 255, 255, 0.2)';
                        countBadge.style.color = 'white';
                    }
                } else {
                    // Inactive tab styling
                    tab.style.background = 'transparent';
                    tab.style.color = '#6b7280';
                    tab.style.boxShadow = 'none';
                    tab.style.transform = 'translateY(0)';
                    tab.classList.remove('active');
                    
                    // Update count badge for inactive tab
                    const countBadge = tab.querySelector('.category-count');
                    if (countBadge) {
                        countBadge.style.background = 'rgba(107, 114, 128, 0.1)';
                        countBadge.style.color = '#6b7280';
                    }
                }
            });
                        
                        // Update badge in active tab
                        const badge = tab.querySelector('span:last-child');
                        if (badge) {
                            badge.style.background = '#667eea';
                            badge.style.color = 'white';
                        }
                    } else {
                        // Inactive ultra tab styling
                        tab.style.background = 'transparent';
                        tab.style.color = 'rgba(255,255,255,0.9)';
                        tab.style.boxShadow = 'none';
                        tab.style.transform = 'scale(1)';
                        
                        // Update badge in inactive tab
                        const badge = tab.querySelector('span:last-child');
                        if (badge) {
                            badge.style.background = 'rgba(255,255,255,0.2)';
                            badge.style.color = 'rgba(255,255,255,0.9)';
                        }
                    }
                }
            }
        }

        // Initialize modern categories section
        document.addEventListener('DOMContentLoaded', function() {
            // Initialize the first tab as active
            switchCategoryTab(0);
            
            // Add mobile touch support for horizontal scrolling
            const tabsWrapper = document.querySelector('.categories-tabs-wrapper');
            if (tabsWrapper) {
                let isScrolling = false;
                let startX;
                
                tabsWrapper.addEventListener('touchstart', function(e) {
                    isScrolling = true;
                    startX = e.touches[0].clientX;
                });
                
                tabsWrapper.addEventListener('touchmove', function(e) {
                    if (!isScrolling) return;
                    
                    const currentX = e.touches[0].clientX;
                    const diffX = startX - currentX;
                    
                    tabsWrapper.scrollLeft += diffX;
                    startX = currentX;
                });
                
                tabsWrapper.addEventListener('touchend', function() {
                    isScrolling = false;
                });
            }
        });
        
        // Add notification functionality for cart operations
        function showNotification(message, type = 'info') {
            const notification = document.createElement('div');
            notification.style.cssText = `
                position: fixed;
                top: 20px;
                right: 20px;
                background: ${type === 'success' ? '#10b981' : type === 'error' ? '#ef4444' : '#3b82f6'};
                color: white;
                padding: 12px 20px;
                border-radius: 8px;
                box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
                z-index: 10000;
                font-weight: 600;
                transform: translateX(100%);
                transition: transform 0.3s ease;
            `;
            notification.textContent = message;
            
            document.body.appendChild(notification);
            
            // Slide in
            setTimeout(() => {
                notification.style.transform = 'translateX(0)';
            }, 100);
            
            // Slide out and remove
            setTimeout(() => {
                notification.style.transform = 'translateX(100%)';
                setTimeout(() => {
                    if (notification.parentNode) {
                        notification.parentNode.removeChild(notification);
                    }
                }, 300);
            }, 3000);
        }
    </script>
    
    <?php include 'debug-js.php'; ?>
</body>
</html>
