/**
 * Video API Integration Module
 * Provides automatic duration detection and metadata for supported platforms
 * Integrates with existing upload form WITHOUT breaking current functionality
 */

class VideoAPIIntegration {
    constructor() {
        this.apiLoadCallbacks = new Map();
        this.loadedAPIs = new Set();
        this.initializeAPIs();
    }

    /**
     * Initialize API loaders for supported platforms
     */
    initializeAPIs() {
        console.log('🎬 Initializing Video API Integration...');
        
        // YouTube API
        this.loadYouTubeAPI();
        
        // Vimeo API  
        this.loadVimeoAPI();
        
        // TikTok uses postMessage, no external loading needed
        this.loadedAPIs.add('tiktok');
        
        console.log('✅ Video API Integration initialized');
    }

    /**
     * Load YouTube IFrame API
     */
    loadYouTubeAPI() {
        if (window.YT && window.YT.Player) {
            this.loadedAPIs.add('youtube');
            console.log('✅ YouTube API already loaded');
            return;
        }

        // Load YouTube API script
        const script = document.createElement('script');
        script.src = 'https://www.youtube.com/iframe_api';
        script.async = true;
        document.head.appendChild(script);

        // Setup callback for when API loads
        window.onYouTubeIframeAPIReady = () => {
            this.loadedAPIs.add('youtube');
            console.log('✅ YouTube API loaded successfully');
            this.executeCallbacks('youtube');
        };
    }

    /**
     * Load Vimeo Player API
     */
    loadVimeoAPI() {
        if (window.Vimeo && window.Vimeo.Player) {
            this.loadedAPIs.add('vimeo');
            console.log('✅ Vimeo API already loaded');
            return;
        }

        // Load Vimeo API script
        const script = document.createElement('script');
        script.src = 'https://player.vimeo.com/api/player.js';
        script.async = true;
        script.onload = () => {
            this.loadedAPIs.add('vimeo');
            console.log('✅ Vimeo API loaded successfully');
            this.executeCallbacks('vimeo');
        };
        document.head.appendChild(script);
    }

    /**
     * Execute pending callbacks for a platform
     */
    executeCallbacks(platform) {
        if (this.apiLoadCallbacks.has(platform)) {
            this.apiLoadCallbacks.get(platform).forEach(callback => callback());
            this.apiLoadCallbacks.delete(platform);
        }
    }

    /**
     * Wait for API to be loaded before executing callback
     */
    whenAPIReady(platform, callback) {
        if (this.loadedAPIs.has(platform)) {
            callback();
        } else {
            if (!this.apiLoadCallbacks.has(platform)) {
                this.apiLoadCallbacks.set(platform, []);
            }
            this.apiLoadCallbacks.get(platform).push(callback);
        }
    }

    /**
     * Detect video platform and extract video ID
     */
    detectPlatformAndExtractId(url) {
        const patterns = {
            youtube: [
                /(?:https?:\/\/)?(?:www\.)?(?:youtube\.com\/watch\?v=|youtu\.be\/)([a-zA-Z0-9_-]+)/,
                /(?:https?:\/\/)?(?:www\.)?youtube\.com\/embed\/([a-zA-Z0-9_-]+)/,
                /(?:https?:\/\/)?(?:www\.)?youtube\.com\/shorts\/([a-zA-Z0-9_-]+)/
            ],
            vimeo: [
                /(?:https?:\/\/)?(?:www\.)?vimeo\.com\/([0-9]+)/,
                /(?:https?:\/\/)?player\.vimeo\.com\/video\/([0-9]+)/
            ],
            tiktok: [
                /(?:https?:\/\/)?(?:www\.)?tiktok\.com\/@[^\/]+\/video\/([0-9]+)/,
                /(?:https?:\/\/)?(?:vm\.)?tiktok\.com\/([a-zA-Z0-9]+)/
            ],
            dailymotion: [
                /(?:https?:\/\/)?(?:www\.)?dailymotion\.com\/video\/([^_\?\s]+)/,
                /(?:https?:\/\/)?(?:www\.)?dai\.ly\/([^_\?\s]+)/
            ],
            twitch: [
                /(?:https?:\/\/)?(?:www\.)?twitch\.tv\/videos\/([0-9]+)/,
                /(?:https?:\/\/)?clips\.twitch\.tv\/([A-Za-z0-9_-]+)/,
                /(?:https?:\/\/)?(?:www\.)?twitch\.tv\/[^\/]+\/clip\/([A-Za-z0-9_-]+)/
            ],
            streamable: [
                /(?:https?:\/\/)?(?:www\.)?streamable\.com\/([A-Za-z0-9_-]+)/
            ]
        };

        for (const [platform, platformPatterns] of Object.entries(patterns)) {
            for (const pattern of platformPatterns) {
                const match = url.match(pattern);
                if (match) {
                    return {
                        platform: platform,
                        videoId: match[1],
                        originalUrl: url
                    };
                }
            }
        }

        return null;
    }

    /**
     * Get video duration for supported platforms
     */
    async getVideoDuration(url, onSuccess, onError) {
        console.log('🔍 Getting video duration for:', url);
        
        const detection = this.detectPlatformAndExtractId(url);
        if (!detection) {
            onError && onError('Platform not supported or invalid URL');
            return;
        }

        console.log(`📱 Detected platform: ${detection.platform}, Video ID: ${detection.videoId}`);

        switch (detection.platform) {
            case 'youtube':
                this.getYouTubeDuration(detection.videoId, onSuccess, onError);
                break;
            case 'vimeo':
                this.getVimeoDuration(detection.videoId, onSuccess, onError);
                break;
            case 'tiktok':
                this.getTikTokDuration(detection.videoId, detection.originalUrl, onSuccess, onError);
                break;
            case 'dailymotion':
                this.getDailymotionDuration(detection.videoId, onSuccess, onError);
                break;
            case 'twitch':
                this.getTwitchDuration(detection.videoId, detection.originalUrl, onSuccess, onError);
                break;
            case 'streamable':
                this.getStreamableDuration(detection.videoId, onSuccess, onError);
                break;
            default:
                onError && onError(`Duration detection not yet supported for ${detection.platform}`);
        }
    }

    /**
     * Get YouTube video duration
     */
    getYouTubeDuration(videoId, onSuccess, onError) {
        this.whenAPIReady('youtube', () => {
            try {
                console.log('📺 Getting YouTube duration for:', videoId);
                
                // Create a hidden player to get video info
                const tempContainer = document.createElement('div');
                tempContainer.style.display = 'none';
                document.body.appendChild(tempContainer);

                const player = new YT.Player(tempContainer, {
                    height: '1',
                    width: '1',
                    videoId: videoId,
                    events: {
                        onReady: (event) => {
                            try {
                                const duration = event.target.getDuration();
                                const formattedDuration = this.formatDuration(duration);
                                
                                console.log(`✅ YouTube duration: ${duration}s (${formattedDuration})`);
                                
                                // Safe cleanup with error handling
                                setTimeout(() => {
                                    try {
                                        if (tempContainer && tempContainer.parentNode) {
                                            tempContainer.parentNode.removeChild(tempContainer);
                                        }
                                    } catch (cleanupError) {
                                        console.warn('YouTube cleanup warning (non-critical):', cleanupError);
                                    }
                                }, 100);
                                
                                onSuccess && onSuccess({
                                    duration: duration,
                                    formattedDuration: formattedDuration,
                                    platform: 'youtube',
                                    method: 'YouTube IFrame API'
                                });
                            } catch (error) {
                                console.error('❌ YouTube duration extraction failed:', error);
                                // Safe cleanup
                                setTimeout(() => {
                                    try {
                                        if (tempContainer && tempContainer.parentNode) {
                                            tempContainer.parentNode.removeChild(tempContainer);
                                        }
                                    } catch (cleanupError) {
                                        console.warn('YouTube cleanup warning:', cleanupError);
                                    }
                                }, 100);
                                onError && onError('Failed to extract duration from YouTube API');
                            }
                        },
                        onError: (error) => {
                            console.error('❌ YouTube player error:', error);
                            // Safe cleanup
                            setTimeout(() => {
                                try {
                                    if (tempContainer && tempContainer.parentNode) {
                                        tempContainer.parentNode.removeChild(tempContainer);
                                    }
                                } catch (cleanupError) {
                                    console.warn('YouTube cleanup warning:', cleanupError);
                                }
                            }, 100);
                            onError && onError('YouTube video unavailable or restricted');
                        }
                    }
                });
            } catch (error) {
                console.error('❌ YouTube API error:', error);
                onError && onError('YouTube API initialization failed');
            }
        });
    }

    /**
     * Get Vimeo video duration
     */
    getVimeoDuration(videoId, onSuccess, onError) {
        this.whenAPIReady('vimeo', () => {
            try {
                console.log('🎥 Getting Vimeo duration for:', videoId);
                
                // Try multiple approaches for Vimeo duration
                this.tryVimeoApproaches(videoId, onSuccess, onError);
                    
            } catch (error) {
                console.error('❌ Vimeo API error:', error);
                onError && onError('Vimeo API initialization failed');
            }
        });
    }

    /**
     * Try multiple approaches for Vimeo duration detection
     */
    tryVimeoApproaches(videoId, onSuccess, onError) {
        console.log('🔄 Trying Vimeo approach 1: oEmbed API');
        
        // Approach 1: Try oEmbed with JSONP to avoid CORS
        const script = document.createElement('script');
        const callbackName = `vimeo_callback_${Date.now()}`;
        
        // Setup global callback
        window[callbackName] = (data) => {
            try {
                // Cleanup
                document.head.removeChild(script);
                delete window[callbackName];
                
                if (data && data.duration) {
                    const duration = data.duration;
                    const formattedDuration = this.formatDuration(duration);
                    
                    console.log(`✅ Vimeo duration (JSONP): ${duration}s (${formattedDuration})`);
                    
                    onSuccess && onSuccess({
                        duration: duration,
                        formattedDuration: formattedDuration,
                        platform: 'vimeo',
                        method: 'Vimeo oEmbed JSONP',
                        title: data.title,
                        thumbnail: data.thumbnail_url
                    });
                } else {
                    console.warn('⚠️ Vimeo JSONP: no duration data');
                    this.tryVimeoPlayerAPI(videoId, onSuccess, onError);
                }
            } catch (error) {
                console.error('❌ Vimeo JSONP callback error:', error);
                this.tryVimeoPlayerAPI(videoId, onSuccess, onError);
            }
        };
        
        // JSONP request
        script.src = `https://vimeo.com/api/oembed.json?url=https://vimeo.com/${videoId}&callback=${callbackName}`;
        script.onerror = () => {
            console.warn('⚠️ Vimeo JSONP failed, trying Player API');
            // Cleanup
            try {
                document.head.removeChild(script);
                delete window[callbackName];
            } catch (e) {}
            this.tryVimeoPlayerAPI(videoId, onSuccess, onError);
        };
        
        // Timeout for JSONP
        setTimeout(() => {
            if (window[callbackName]) {
                console.warn('⚠️ Vimeo JSONP timeout');
                try {
                    document.head.removeChild(script);
                    delete window[callbackName];
                } catch (e) {}
                this.tryVimeoPlayerAPI(videoId, onSuccess, onError);
            }
        }, 5000);
        
        document.head.appendChild(script);
    }

    /**
     * Fallback: Try Vimeo Player API approach
     */
    tryVimeoPlayerAPI(videoId, onSuccess, onError) {
        console.log('🔄 Trying Vimeo approach 2: Player API');
        
        try {
            // Create hidden iframe with Vimeo player
            const iframe = document.createElement('iframe');
            iframe.src = `https://player.vimeo.com/video/${videoId}?api=1`;
            iframe.width = '1';
            iframe.height = '1';
            iframe.style.display = 'none';
            
            let resolved = false;
            let timeoutId;
            
            // Listen for Vimeo player messages
            const messageHandler = (event) => {
                if (event.origin !== 'https://player.vimeo.com') return;
                
                try {
                    const data = JSON.parse(event.data);
                    
                    if (data.event === 'ready' && !resolved) {
                        // Request duration from player
                        iframe.contentWindow.postMessage('{"method":"getDuration"}', 'https://player.vimeo.com');
                    } else if (data.method === 'getDuration' && data.value && !resolved) {
                        resolved = true;
                        clearTimeout(timeoutId);
                        
                        const duration = Math.round(data.value);
                        const formattedDuration = this.formatDuration(duration);
                        
                        console.log(`✅ Vimeo duration (Player API): ${duration}s (${formattedDuration})`);
                        
                        // Cleanup
                        window.removeEventListener('message', messageHandler);
                        if (iframe.parentNode) {
                            iframe.parentNode.removeChild(iframe);
                        }
                        
                        onSuccess && onSuccess({
                            duration: duration,
                            formattedDuration: formattedDuration,
                            platform: 'vimeo',
                            method: 'Vimeo Player API'
                        });
                    }
                } catch (e) {
                    // Not JSON or not relevant
                }
            };
            
            window.addEventListener('message', messageHandler);
            document.body.appendChild(iframe);
            
            // Timeout
            timeoutId = setTimeout(() => {
                if (!resolved) {
                    console.warn('⚠️ Vimeo Player API timeout');
                    window.removeEventListener('message', messageHandler);
                    if (iframe.parentNode) {
                        iframe.parentNode.removeChild(iframe);
                    }
                    onError && onError('Vimeo duration detection timed out');
                }
            }, 8000);
            
        } catch (error) {
            console.error('❌ Vimeo Player API error:', error);
            onError && onError('Vimeo Player API failed');
        }
    }

    /**
     * Get TikTok video duration (using embed method)
     */
    getTikTokDuration(videoId, originalUrl, onSuccess, onError) {
        console.log('🎵 Getting TikTok duration for:', videoId);
        
        // For now, TikTok's PostMessage API is unreliable due to security restrictions
        // Provide a more user-friendly fallback
        console.warn('⚠️ TikTok duration detection: API restrictions prevent automatic detection');
        
        // Estimate based on TikTok's typical video lengths
        const estimatedDuration = this.estimateTikTokDuration();
        
        console.log(`📊 TikTok estimated duration: ${estimatedDuration}s`);
        
        // Return estimated duration with clear method indication
        onSuccess && onSuccess({
            duration: estimatedDuration,
            formattedDuration: this.formatDuration(estimatedDuration),
            platform: 'tiktok',
            method: 'TikTok Estimated Duration',
            note: 'Estimated - manual verification recommended'
        });
        
        // Also try the PostMessage approach in background (for future improvement)
        this.tryTikTokPostMessage(videoId, originalUrl, null, null);
    }

    /**
     * Estimate TikTok video duration based on typical lengths
     */
    estimateTikTokDuration() {
        // TikTok videos are typically 15-60 seconds, with most being around 30 seconds
        // Return a reasonable estimate
        return 30; // 30 seconds as default estimate
    }

    /**
     * Try TikTok PostMessage (background attempt for future improvement)
     */
    tryTikTokPostMessage(videoId, originalUrl, onSuccess, onError) {
        try {
            console.log('🔄 Attempting TikTok PostMessage (background)...');
            
            // Create a hidden iframe to communicate with TikTok player
            const tempContainer = document.createElement('div');
            tempContainer.style.display = 'none';
            tempContainer.style.position = 'absolute';
            tempContainer.style.width = '1px';
            tempContainer.style.height = '1px';
            tempContainer.style.overflow = 'hidden';
            document.body.appendChild(tempContainer);

            const iframe = document.createElement('iframe');
            iframe.src = `https://www.tiktok.com/player/v1/${videoId}?autoplay=0&controls=1`;
            iframe.width = '1';
            iframe.height = '1';
            iframe.style.border = 'none';
            tempContainer.appendChild(iframe);

            let timeoutId;
            let messageHandler;

            // Listen for messages from TikTok iframe
            messageHandler = (event) => {
                if (event.origin !== 'https://www.tiktok.com') return;
                
                try {
                    const message = event.data;
                    console.log('📨 TikTok PostMessage received:', message);
                    
                    if (message && message['x-tiktok-player'] && message.type === 'onCurrentTime') {
                        const duration = message.value.duration;
                        
                        if (duration > 0) {
                            const formattedDuration = this.formatDuration(duration);
                            
                            console.log(`✅ TikTok PostMessage success: ${duration}s (${formattedDuration})`);
                            
                            // Cleanup
                            clearTimeout(timeoutId);
                            window.removeEventListener('message', messageHandler);
                            if (tempContainer.parentNode) {
                                tempContainer.parentNode.removeChild(tempContainer);
                            }
                        }
                    }
                } catch (error) {
                    console.error('❌ TikTok message parsing error:', error);
                }
            };

            window.addEventListener('message', messageHandler);

            // Set timeout for TikTok API response
            timeoutId = setTimeout(() => {
                console.warn('⚠️ TikTok PostMessage timeout (expected)');
                window.removeEventListener('message', messageHandler);
                if (tempContainer.parentNode) {
                    tempContainer.parentNode.removeChild(tempContainer);
                }
            }, 5000); // Shorter timeout since this is just a background attempt

        } catch (error) {
            console.error('❌ TikTok PostMessage error:', error);
        }
    }

    /**
     * Format seconds to MM:SS or HH:MM:SS format
     */
    formatDuration(seconds) {
        if (!seconds || isNaN(seconds)) return '0:00';
        
        const hours = Math.floor(seconds / 3600);
        const minutes = Math.floor((seconds % 3600) / 60);
        const secs = Math.floor(seconds % 60);
        
        if (hours > 0) {
            return `${hours}:${minutes.toString().padStart(2, '0')}:${secs.toString().padStart(2, '0')}`;
        } else {
            return `${minutes}:${secs.toString().padStart(2, '0')}`;
        }
    }

    /**
     * Detect video platform and extract video ID from URL
     * @param {string} url - Video URL to detect
     * @returns {Object|null} - {platform, videoId} or null if not supported
     */
    detectVideo(url) {
        if (!url || typeof url !== 'string') {
            return null;
        }

        console.log('🔍 Detecting video platform for URL:', url);

        // YouTube patterns
        const youtubePatterns = [
            /(?:youtube\.com\/watch\?v=|youtu\.be\/|youtube\.com\/embed\/)([a-zA-Z0-9_-]{11})/,
            /youtube\.com\/v\/([a-zA-Z0-9_-]{11})/
        ];

        for (const pattern of youtubePatterns) {
            const match = url.match(pattern);
            if (match) {
                console.log('✅ YouTube detected:', match[1]);
                return { platform: 'youtube', videoId: match[1] };
            }
        }

        // Vimeo patterns
        const vimeoPatterns = [
            /vimeo\.com\/(?:video\/)?(\d+)/,
            /player\.vimeo\.com\/video\/(\d+)/
        ];

        for (const pattern of vimeoPatterns) {
            const match = url.match(pattern);
            if (match) {
                console.log('✅ Vimeo detected:', match[1]);
                return { platform: 'vimeo', videoId: match[1] };
            }
        }

        // TikTok patterns
        const tiktokPatterns = [
            /tiktok\.com\/@[^\/]+\/video\/(\d+)/,
            /tiktok\.com\/v\/(\d+)/,
            /tiktok\.com\/embed\/v2\/(\d+)/
        ];

        for (const pattern of tiktokPatterns) {
            const match = url.match(pattern);
            if (match) {
                console.log('✅ TikTok detected:', match[1]);
                return { platform: 'tiktok', videoId: match[1] };
            }
        }

        // Dailymotion patterns
        const dailymotionPatterns = [
            /dailymotion\.com\/video\/([a-zA-Z0-9]+)/,
            /dai\.ly\/([a-zA-Z0-9]+)/,
            /dailymotion\.com\/embed\/video\/([a-zA-Z0-9]+)/
        ];

        for (const pattern of dailymotionPatterns) {
            const match = url.match(pattern);
            if (match) {
                console.log('✅ Dailymotion detected:', match[1]);
                return { platform: 'dailymotion', videoId: match[1] };
            }
        }

        // Twitch patterns
        const twitchPatterns = [
            /twitch\.tv\/videos\/(\d+)/,
            /player\.twitch\.tv\/\?video=(\d+)/
        ];

        for (const pattern of twitchPatterns) {
            const match = url.match(pattern);
            if (match) {
                console.log('✅ Twitch detected:', match[1]);
                return { platform: 'twitch', videoId: match[1] };
            }
        }

        // Streamable patterns
        const streamablePatterns = [
            /streamable\.com\/([a-zA-Z0-9]+)/,
            /streamable\.com\/e\/([a-zA-Z0-9]+)/
        ];

        for (const pattern of streamablePatterns) {
            const match = url.match(pattern);
            if (match) {
                console.log('✅ Streamable detected:', match[1]);
                return { platform: 'streamable', videoId: match[1] };
            }
        }

        console.log('❌ Video platform not supported or not detected for URL:', url);
        return null;
    }

    // ================================
    // VIDEO PROGRESS TRACKING EXTENSION
    // ================================

    /**
     * Initialize automatic progress tracking for supported video platforms
     * Integrates with existing video_progress system
     * @param {string} videoUrl - The video URL
     * @param {number} videoId - Database video ID
     * @param {number} courseId - Database course ID
     * @param {Object} options - Tracking options
     */
    initializeProgressTracking(videoUrl, videoId, courseId, options = {}) {
        console.log('🎬 Initializing progress tracking for existing system:', { videoUrl, videoId, courseId });
        
        const detection = this.detectVideo(videoUrl);
        if (!detection || !detection.platform) {
            console.log('❌ Video platform not supported for automatic progress tracking:', videoUrl);
            return false;
        }

        const trackingOptions = {
            updateInterval: 5000, // Update every 5 seconds
            progressThreshold: 5,  // Minimum progress change to trigger update
            autoStart: true,
            ...options
        };

        console.log('🔧 Setting up automatic progress tracking for platform:', detection.platform);

        // Initialize platform-specific tracking
        switch (detection.platform) {
            case 'youtube':
                return this.initYouTubeTracking(detection.videoId, videoId, courseId, trackingOptions);
            case 'vimeo':
                return this.initVimeoTracking(detection.videoId, videoId, courseId, trackingOptions);
            case 'tiktok':
                return this.initTikTokTracking(detection.videoId, videoId, courseId, trackingOptions);
            case 'dailymotion':
                return this.initDailymotionTracking(detection.videoId, videoId, courseId, trackingOptions);
            case 'twitch':
                return this.initTwitchTracking(detection.videoId, videoId, courseId, trackingOptions);
            case 'streamable':
                return this.initStreamableTracking(detection.videoId, videoId, courseId, trackingOptions);
            default:
                console.log('❌ Progress tracking not implemented for platform:', detection.platform);
                return false;
        }
    }

    /**
     * Update progress using existing video_progress system
     */
    updateVideoProgress(videoId, courseId, currentTime, duration, progressPercentage, completed = false) {
        const formData = new FormData();
        formData.append('video_id', videoId);
        formData.append('course_id', courseId);
        formData.append('progress_percentage', Math.round(progressPercentage));
        formData.append('current_time', Math.round(currentTime));
        formData.append('completed', completed ? 1 : 0);

        console.log('📤 Updating progress:', { videoId, courseId, progressPercentage, currentTime, completed });

        return fetch('ajax/update_video_progress.php', {
            method: 'POST',
            body: formData
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                console.log('✅ Progress updated successfully:', data);
                
                // Trigger progress update event for UI
                window.dispatchEvent(new CustomEvent('videoProgressUpdate', { 
                    detail: {
                        ...data,
                        currentTime,
                        duration,
                        progressPercentage
                    }
                }));

                // 🎯 CRITICAL: Update accordion badges if we're in course-detail.php
                if (typeof updateAccordionBadges === 'function') {
                    // Get current lesson ID from the page context
                    const currentLessonId = window.currentLessonId || document.querySelector('[data-current-lesson-id]')?.dataset.currentLessonId;
                    
                    if (currentLessonId) {
                        console.log('🏷️ Updating accordion badges for lesson:', currentLessonId);
                        
                        // Determine badge status based on progress
                        const badgeStatus = completed ? 'completed' : (progressPercentage > 0 ? 'watching' : 'idle');
                        
                        // Update accordion badges to show "Currently Watching" and progress
                        updateAccordionBadges(currentLessonId, badgeStatus, progressPercentage);
                        
                        // Also update the accordion progress bar and text
                        const accordionProgressBar = document.getElementById(`accordion-progress-bar-${currentLessonId}`);
                        const accordionProgressText = document.getElementById(`accordion-progress-text-${currentLessonId}`);
                        
                        if (accordionProgressBar) {
                            accordionProgressBar.style.width = progressPercentage + '%';
                        }
                        
                        if (accordionProgressText) {
                            accordionProgressText.textContent = `Progress: ${Math.round(progressPercentage)}% complete`;
                        }
                        
                        console.log('✅ Accordion badges and progress updated for URL video');
                    }
                }
                
                return data;
            } else {
                console.error('❌ Progress update failed:', data);
                throw new Error(data.error || 'Progress update failed');
            }
        })
        .catch(error => {
            console.error('❌ Progress update error:', error);
            throw error;
        });
    }

    /**
     * Initialize YouTube progress tracking using YouTube IFrame API
     */
    initYouTubeTracking(youtubeVideoId, videoId, courseId, options) {
        console.log('🎬 Initializing YouTube progress tracking...', { youtubeVideoId, videoId, courseId });
        console.log('🔍 YouTube API Check: window.YT =', typeof window.YT, ', window.YT.Player =', typeof (window.YT && window.YT.Player));
        
        const initializePlayer = () => {
            try {
                // Find the existing YouTube iframe
                const iframe = document.querySelector('iframe[src*="youtube.com/embed"]');
                if (!iframe) {
                    console.error('❌ YouTube iframe not found');
                    return false;
                }

                console.log('✅ YouTube iframe found, setting up dynamic API integration...');
                
                // Give iframe a unique ID if it doesn't have one
                if (!iframe.id) {
                    iframe.id = 'youtube-player-' + Date.now();
                }
                
                console.log(`🔧 Creating YouTube API player for iframe: ${iframe.id}`);
                
                // Create YouTube player from existing iframe
                const player = new YT.Player(iframe.id, {
                    events: {
                        'onReady': (event) => {
                            console.log('✅ YouTube player ready for DYNAMIC progress tracking');
                            
                            // Start progress tracking interval
                            const trackingInterval = setInterval(() => {
                                try {
                                    const currentTime = player.getCurrentTime();
                                    const duration = player.getDuration();
                                    const progressPercentage = duration > 0 ? (currentTime / duration) * 100 : 0;
                                    
                                    console.log(`🔍 YouTube API Check: Time=${Math.round(currentTime)}s, Duration=${Math.round(duration)}s, Progress=${Math.round(progressPercentage)}%`);
                                    
                                    // Update even with small progress (more responsive)
                                    if (currentTime >= 0 && duration > 0) {
                                        console.log(`📊 DYNAMIC YouTube Progress: ${Math.round(progressPercentage)}% (${Math.round(currentTime)}s / ${Math.round(duration)}s)`);
                                        
                                        // Update database and UI
                                        this.updateVideoProgress(
                                            videoId, 
                                            courseId, 
                                            currentTime, 
                                            duration, 
                                            progressPercentage,
                                            progressPercentage >= 90 // Mark as completed at 90%
                                        );
                                    }
                                } catch (error) {
                                    console.error('❌ YouTube tracking error:', error);
                                }
                            }, options.updateInterval || 2000); // More frequent updates

                            // Store interval for cleanup
                            this.youtubeTrackingInterval = trackingInterval;
                            
                            console.log('🎬 YouTube dynamic progress tracking ACTIVE!');
                        },
                        'onStateChange': (event) => {
                            // Track play/pause events with detailed logging
                            console.log(`🎮 YouTube State Change: ${event.data}`);
                            
                            if (event.data == YT.PlayerState.PLAYING) {
                                console.log('▶️ YouTube video STARTED playing - progress tracking active');
                            } else if (event.data == YT.PlayerState.PAUSED) {
                                console.log('⏸️ YouTube video PAUSED - progress tracking continues');
                            } else if (event.data == YT.PlayerState.ENDED) {
                                console.log('🏁 YouTube video ENDED - marking as completed');
                                
                                // Mark as completed
                                this.updateVideoProgress(
                                    videoId, 
                                    courseId, 
                                    player.getDuration(), 
                                    player.getDuration(), 
                                    100,
                                    true
                                );
                            }
                        },
                        'onError': (event) => {
                            console.error('❌ YouTube player error:', event.data);
                        }
                    }
                });

                console.log('🎯 YouTube API player created successfully');
                return true;

            } catch (error) {
                console.error('❌ Error initializing YouTube tracking:', error);
                return false;
            }
        };

        // Check if YouTube API is loaded
        if (window.YT && window.YT.Player) {
            console.log('✅ YouTube API available, using API method');
            return initializePlayer();
        } else {
            console.log('⚠️ YouTube API not available, using PostMessage fallback');
            
            try {
                // FALLBACK: Use PostMessage method for YouTube progress tracking
                const result = this.initYouTubePostMessageTracking(youtubeVideoId, videoId, courseId, options);
                console.log('🔧 PostMessage method result:', result);
                return result;
            } catch (error) {
                console.error('❌ Error calling PostMessage method:', error);
                return false;
            }
        }
    }

    /**
     * 🚀 FALLBACK YouTube Progress Tracking using PostMessage
     * This works when YouTube IFrame API is not available
     */
    initYouTubePostMessageTracking(youtubeVideoId, videoId, courseId, options = {}) {
        console.log('🚀 Initializing YouTube PostMessage progress tracking...');
        console.log('🔍 PostMessage Parameters:', { youtubeVideoId, videoId, courseId, options });
        
        const iframe = document.querySelector('iframe[src*="youtube.com/embed"]');
        if (!iframe) {
            console.error('❌ YouTube iframe not found for PostMessage tracking');
            return false;
        }

        console.log('✅ YouTube iframe found for PostMessage:', iframe.src);

        // Enable YouTube postMessage API by adding parameter
        const currentSrc = iframe.src;
        if (!currentSrc.includes('enablejsapi=1')) {
            const separator = currentSrc.includes('?') ? '&' : '?';
            const newSrc = currentSrc + separator + 'enablejsapi=1&origin=' + encodeURIComponent(window.location.origin);
            console.log('🔧 Enhancing iframe src from:', currentSrc);
            console.log('🔧 Enhancing iframe src to:', newSrc);
            iframe.src = newSrc;
            console.log('✅ Enhanced YouTube iframe with PostMessage API');
        } else {
            console.log('✅ YouTube iframe already has enablejsapi=1');
        }

        // Track the last progress to avoid unnecessary updates
        let lastProgressPercent = -1;
        let lastCurrentTime = -1;
        let lastUpdateTime = 0;
        
        // Listen for YouTube PostMessage events
        window.addEventListener('message', (event) => {
            if (event.origin !== 'https://www.youtube.com' && event.origin !== 'https://youtube.com') {
                return;
            }
            
            try {
                const data = typeof event.data === 'string' ? JSON.parse(event.data) : event.data;
                console.log('🎬 YouTube PostMessage Received:', JSON.stringify(data));
                
                if (data.info && typeof data.info.currentTime !== 'undefined' && typeof data.info.duration !== 'undefined') {
                    const currentTime = Math.floor(data.info.currentTime);
                    const duration = Math.floor(data.info.duration);
                    const progressPercent = duration > 0 ? Math.round((currentTime / duration) * 100) : 0;
                    const now = Date.now();
                    
                    console.log(`📊 YouTube PostMessage Progress: ${progressPercent}% (${currentTime}s / ${duration}s)`);
                    
                    // Only update if progress has meaningfully changed (debounced)
                    const progressChanged = Math.abs(progressPercent - lastProgressPercent) >= 1;
                    const timeElapsed = (now - lastUpdateTime) >= 2000; // 2 seconds
                    const currentTimeAdvanced = currentTime > lastCurrentTime;
                    
                    if ((progressChanged && currentTimeAdvanced) || (timeElapsed && currentTimeAdvanced)) {
                        lastProgressPercent = progressPercent;
                        lastCurrentTime = currentTime;
                        lastUpdateTime = now;
                        
                        // Update progress in database
                        this.updateVideoProgress(videoId, courseId, currentTime, duration, progressPercent, progressPercent >= 100)
                            .then(() => {
                                console.log(`✅ YouTube PostMessage Progress Updated: ${progressPercent}% (${currentTime}s / ${duration}s)`);
                            })
                            .catch(error => console.error('❌ PostMessage progress update failed:', error));
                    }
                }
            } catch (error) {
                // Ignore non-JSON messages
            }
        });
        
        // Request progress data every 2 seconds
        const progressInterval = setInterval(() => {
            console.log('📤 Requesting YouTube PostMessage progress data');
            if (iframe && iframe.contentWindow) {
                iframe.contentWindow.postMessage('{"event":"command","func":"getCurrentTime","args":""}', '*');
                iframe.contentWindow.postMessage('{"event":"command","func":"getDuration","args":""}', '*');
            }
        }, options.updateInterval || 2000);
        
        console.log('🎬 YouTube PostMessage tracking activated - polling every 2 seconds');
        
        return progressInterval;
    }

    setupYouTubeProgressTracking(youtubeVideoId, videoId, courseId, options) {
        let lastProgress = 0;
        let progressInterval;

        const trackProgress = (player) => {
            try {
                const currentTime = player.getCurrentTime();
                const duration = player.getDuration();
                
                if (duration > 0) {
                    const progressPercentage = (currentTime / duration) * 100;
                    const completed = progressPercentage >= 90; // Mark as completed at 90%

                    // Only update if progress changed significantly
                    if (Math.abs(progressPercentage - lastProgress) >= options.progressThreshold) {
                        this.updateVideoProgress(videoId, courseId, currentTime, duration, progressPercentage, completed);
                        lastProgress = progressPercentage;
                    }
                }
            } catch (err) {
                console.error('❌ YouTube progress tracking error:', err);
            }
        };

        // Find YouTube iframe and set up tracking
        const iframe = document.querySelector('iframe[src*="youtube.com"]');
        if (iframe && iframe.id) {
            const player = new YT.Player(iframe.id, {
                events: {
                    'onStateChange': (event) => {
                        if (event.data === YT.PlayerState.PLAYING) {
                            console.log('▶️ YouTube video playing - starting progress tracking');
                            progressInterval = setInterval(() => trackProgress(event.target), options.updateInterval);
                        } else if (event.data === YT.PlayerState.PAUSED || event.data === YT.PlayerState.ENDED) {
                            console.log('⏸️ YouTube video paused/ended - stopping progress tracking');
                            clearInterval(progressInterval);
                            
                            if (event.data === YT.PlayerState.ENDED) {
                                // Mark as completed when video ends
                                trackProgress(event.target);
                            }
                        }
                    }
                }
            });
        }
        
        console.log('✅ YouTube progress tracking initialized');
        return true;
    }

    /**
     * Vimeo Progress Tracking
     */
    initVimeoTracking(vimeoVideoId, videoId, courseId, options) {
        console.log('🎭 Setting up Vimeo progress tracking...');
        
        if (!window.Vimeo) {
            this.loadVimeoSDK(() => {
                this.setupVimeoProgressTracking(vimeoVideoId, videoId, courseId, options);
            });
            return true;
        } else {
            return this.setupVimeoProgressTracking(vimeoVideoId, videoId, courseId, options);
        }
    }

    setupVimeoProgressTracking(vimeoVideoId, videoId, courseId, options) {
        const iframe = document.querySelector('iframe[src*="vimeo.com"]');
        if (iframe) {
            const player = new Vimeo.Player(iframe);
            let lastProgress = 0;

            player.on('timeupdate', (data) => {
                const currentTime = data.seconds;
                const duration = data.duration;
                const progressPercentage = (currentTime / duration) * 100;
                const completed = progressPercentage >= 90;

                // Only update if progress changed significantly
                if (Math.abs(progressPercentage - lastProgress) >= options.progressThreshold) {
                    this.updateVideoProgress(videoId, courseId, currentTime, duration, progressPercentage, completed);
                    lastProgress = progressPercentage;
                }
            });

            player.on('ended', () => {
                console.log('🎭 Vimeo video ended - marking as completed');
                player.getCurrentTime().then(currentTime => {
                    player.getDuration().then(duration => {
                        this.updateVideoProgress(videoId, courseId, currentTime, duration, 100, true);
                    });
                });
            });

            console.log('✅ Vimeo progress tracking initialized');
            return true;
        }
        
        console.log('❌ Vimeo iframe not found');
        return false;
    }

    /**
     * TikTok Progress Tracking
     */
    initTikTokTracking(tiktokVideoId, videoId, courseId, options) {
        console.log('🎵 Setting up TikTok progress tracking...');
        
        const iframe = document.querySelector('iframe[src*="tiktok.com"]');
        if (iframe) {
            let lastProgress = 0;

            // Set up message listener for TikTok embed communication
            const messageHandler = (event) => {
                if (event.origin.includes('tiktok.com') && event.data) {
                    try {
                        const data = typeof event.data === 'string' ? JSON.parse(event.data) : event.data;
                        
                        if (data.type === 'onCurrentTime' && data.currentTime !== undefined && data.duration !== undefined) {
                            const currentTime = data.currentTime;
                            const duration = data.duration;
                            const progressPercentage = duration > 0 ? (currentTime / duration) * 100 : 0;
                            const completed = progressPercentage >= 90;

                            // Only update if progress changed significantly
                            if (Math.abs(progressPercentage - lastProgress) >= options.progressThreshold) {
                                this.updateVideoProgress(videoId, courseId, currentTime, duration, progressPercentage, completed);
                                lastProgress = progressPercentage;
                            }
                        }
                    } catch (err) {
                        console.error('❌ TikTok message parsing error:', err);
                    }
                }
            };

            window.addEventListener('message', messageHandler);
            console.log('✅ TikTok progress tracking initialized');
            return true;
        }
        
        console.log('❌ TikTok iframe not found');
        return false;
    }

    /**
     * Dailymotion Progress Tracking
     */
    initDailymotionTracking(dailymotionVideoId, videoId, courseId, options) {
        console.log('📺 Setting up Dailymotion progress tracking...');
        
        const iframe = document.querySelector('iframe[src*="dailymotion.com"]');
        if (iframe) {
            let lastProgress = 0;

            // Set up message listener for Dailymotion player
            const messageHandler = (event) => {
                if (event.origin.includes('dailymotion.com') && event.data) {
                    try {
                        const data = typeof event.data === 'string' ? JSON.parse(event.data) : event.data;
                        
                        if (data.event === 'timeupdate' && data.currentTime !== undefined && data.duration !== undefined) {
                            const currentTime = data.currentTime;
                            const duration = data.duration;
                            const progressPercentage = duration > 0 ? (currentTime / duration) * 100 : 0;
                            const completed = progressPercentage >= 90;

                            // Only update if progress changed significantly
                            if (Math.abs(progressPercentage - lastProgress) >= options.progressThreshold) {
                                this.updateVideoProgress(videoId, courseId, currentTime, duration, progressPercentage, completed);
                                lastProgress = progressPercentage;
                            }
                        }
                    } catch (err) {
                        console.error('❌ Dailymotion message parsing error:', err);
                    }
                }
            };

            window.addEventListener('message', messageHandler);
            console.log('✅ Dailymotion progress tracking initialized');
            return true;
        }
        
        console.log('❌ Dailymotion iframe not found');
        return false;
    }

    /**
     * Twitch Progress Tracking
     */
    initTwitchTracking(twitchVideoId, videoId, courseId, options) {
        console.log('🎮 Setting up Twitch progress tracking...');
        
        if (!window.Twitch || !window.Twitch.Player) {
            console.log('⏳ Loading Twitch Player SDK...');
            this.loadTwitchSDK(() => {
                this.setupTwitchProgressTracking(twitchVideoId, videoId, courseId, options);
            });
            return true;
        } else {
            return this.setupTwitchProgressTracking(twitchVideoId, videoId, courseId, options);
        }
    }

    setupTwitchProgressTracking(twitchVideoId, videoId, courseId, options) {
        const iframe = document.querySelector('iframe[src*="twitch.tv"]');
        if (iframe && iframe.id) {
            try {
                const player = new Twitch.Player(iframe.id);
                let lastProgress = 0;
                let progressInterval;

                const trackProgress = () => {
                    try {
                        const currentTime = player.getCurrentTime();
                        const duration = player.getDuration();
                        
                        if (duration > 0) {
                            const progressPercentage = (currentTime / duration) * 100;
                            const completed = progressPercentage >= 90;

                            // Only update if progress changed significantly
                            if (Math.abs(progressPercentage - lastProgress) >= options.progressThreshold) {
                                this.updateVideoProgress(videoId, courseId, currentTime, duration, progressPercentage, completed);
                                lastProgress = progressPercentage;
                            }
                        }
                    } catch (err) {
                        console.error('❌ Twitch progress tracking error:', err);
                    }
                };

                player.addEventListener(Twitch.Player.PLAY, () => {
                    console.log('▶️ Twitch video playing - starting progress tracking');
                    progressInterval = setInterval(trackProgress, options.updateInterval);
                });

                player.addEventListener(Twitch.Player.PAUSE, () => {
                    console.log('⏸️ Twitch video paused - stopping progress tracking');
                    clearInterval(progressInterval);
                });

                player.addEventListener(Twitch.Player.ENDED, () => {
                    console.log('🎮 Twitch video ended - marking as completed');
                    clearInterval(progressInterval);
                    trackProgress();
                });

                console.log('✅ Twitch progress tracking initialized');
                return true;
            } catch (err) {
                console.error('❌ Twitch player initialization error:', err);
                return false;
            }
        }
        
        console.log('❌ Twitch iframe not found');
        return false;
    }

    /**
     * Streamable Progress Tracking (HTML5 Video fallback)
     */
    initStreamableTracking(streamableVideoId, videoId, courseId, options) {
        console.log('🎞️ Setting up Streamable progress tracking...');
        
        const video = document.querySelector('video');
        if (video) {
            let lastProgress = 0;

            const trackProgress = () => {
                const currentTime = video.currentTime;
                const duration = video.duration;
                
                if (duration > 0) {
                    const progressPercentage = (currentTime / duration) * 100;
                    const completed = progressPercentage >= 90;

                    // Only update if progress changed significantly
                    if (Math.abs(progressPercentage - lastProgress) >= options.progressThreshold) {
                        this.updateVideoProgress(videoId, courseId, currentTime, duration, progressPercentage, completed);
                        lastProgress = progressPercentage;
                    }
                }
            };

            video.addEventListener('timeupdate', trackProgress);
            video.addEventListener('ended', () => {
                console.log('🎞️ Streamable video ended - marking as completed');
                trackProgress();
            });

            console.log('✅ Streamable progress tracking initialized');
            return true;
        }
        
        console.log('❌ Streamable video element not found');
        return false;
    }

    /**
     * Load Twitch Player SDK
     */
    loadTwitchSDK(callback) {
        if (window.Twitch && window.Twitch.Player) {
            callback && callback();
            return;
        }

        const script = document.createElement('script');
        script.src = 'https://player.twitch.tv/js/embed/v1.js';
        script.onload = () => {
            console.log('✅ Twitch Player SDK loaded');
            callback && callback();
        };
        script.onerror = () => {
            console.error('❌ Failed to load Twitch Player SDK');
        };
        document.head.appendChild(script);
    }

    /**
     * Get Dailymotion video duration
     */
    getDailymotionDuration(videoId, onSuccess, onError) {
        console.log('🎥 Getting Dailymotion duration for:', videoId);
        // ... (rest of the code remains the same)
        try {
            // Use JSONP to avoid CORS issues
            const script = document.createElement('script');
            const callbackName = `dailymotion_callback_${Date.now()}`;
            
            // Setup global callback
            window[callbackName] = (data) => {
                try {
                    // Cleanup
                    document.head.removeChild(script);
                    delete window[callbackName];
                    
                    if (data && data.duration) {
                        const duration = data.duration;
                        const formattedDuration = this.formatDuration(duration);
                        
                        console.log(`✅ Dailymotion duration: ${duration}s (${formattedDuration})`);
                        
                        onSuccess && onSuccess({
                            duration: duration,
                            formattedDuration: formattedDuration,
                            platform: 'dailymotion',
                            method: 'Dailymotion API',
                            title: data.title
                        });
                    } else {
                        console.warn('⚠️ Dailymotion: no duration data');
                        onError && onError('Duration not available for this Dailymotion video');
                    }
                } catch (error) {
                    console.error('❌ Dailymotion callback error:', error);
                    onError && onError('Failed to process Dailymotion response');
                }
            };
            
            // JSONP request
            script.src = `https://api.dailymotion.com/video/${videoId}?fields=duration,title&callback=${callbackName}`;
            script.onerror = () => {
                console.error('❌ Dailymotion API request failed');
                try {
                    document.head.removeChild(script);
                    delete window[callbackName];
                } catch (e) {}
                onError && onError('Failed to load Dailymotion API');
            };
            
            // Timeout
            setTimeout(() => {
                if (window[callbackName]) {
                    console.warn('⚠️ Dailymotion API timeout');
                    try {
                        document.head.removeChild(script);
                        delete window[callbackName];
                    } catch (e) {}
                    onError && onError('Dailymotion API request timed out');
                }
            }, 8000);
            
            document.head.appendChild(script);
            
        } catch (error) {
            console.error('❌ Dailymotion API error:', error);
            onError && onError('Dailymotion API initialization failed');
        }
    }

    /**
     * Get Twitch video/clip duration
     */
    getTwitchDuration(videoId, originalUrl, onSuccess, onError) {
        console.log('🎮 Getting Twitch duration for:', videoId);
        
        // Twitch API requires authentication, but we can try oEmbed as fallback
        try {
            const script = document.createElement('script');
            const callbackName = `twitch_callback_${Date.now()}`;
            
            // For Twitch clips, try to estimate based on typical lengths
            // Most Twitch clips are 30-60 seconds
            const estimatedDuration = 45; // 45 seconds default
            
            console.log(`📊 Twitch estimated duration: ${estimatedDuration}s`);
            
            onSuccess && onSuccess({
                duration: estimatedDuration,
                formattedDuration: this.formatDuration(estimatedDuration),
                platform: 'twitch',
                method: 'Twitch Estimated Duration',
                note: 'Estimated - Twitch API requires authentication'
            });
            
        } catch (error) {
            console.error('❌ Twitch API error:', error);
            onError && onError('Twitch duration detection failed');
        }
    }

    /**
     * Get Streamable video duration
     */
    getStreamableDuration(videoId, onSuccess, onError) {
        console.log('🎬 Getting Streamable duration for:', videoId);
        
        try {
            // Use direct API call (Streamable allows CORS)
            fetch(`https://api.streamable.com/videos/${videoId}`)
                .then(response => {
                    if (!response.ok) {
                        throw new Error(`HTTP ${response.status}`);
                    }
                    return response.json();
                })
                .then(data => {
                    if (data && data.files && data.files.mp4 && data.files.mp4.duration) {
                        const duration = Math.round(data.files.mp4.duration);
                        const formattedDuration = this.formatDuration(duration);
                        
                        console.log(`✅ Streamable duration: ${duration}s (${formattedDuration})`);
                        
                        onSuccess && onSuccess({
                            duration: duration,
                            formattedDuration: formattedDuration,
                            platform: 'streamable',
                            method: 'Streamable API',
                            title: data.title,
                            thumbnail: data.thumbnail_url
                        });
                    } else {
                        console.warn('⚠️ Streamable: no duration data');
                        onError && onError('Duration not available for this Streamable video');
                    }
                })
                .catch(error => {
                    console.error('❌ Streamable API error:', error);
                    onError && onError('Failed to fetch Streamable video information');
                });
                
        } catch (error) {
            console.error('❌ Streamable API error:', error);
            onError && onError('Streamable API initialization failed');
        }
    }

    /**
     * Get video metadata (title, thumbnail, etc.)
     */
    async getVideoMetadata(url, onSuccess, onError) {
        const detection = this.detectPlatformAndExtractId(url);
        if (!detection) {
            onError && onError('Platform not supported');
            return;
        }

        // For now, duration detection also provides some metadata
        this.getVideoDuration(url, onSuccess, onError);
    }
}

// Global instance
window.VideoAPIIntegration = new VideoAPIIntegration();

console.log('🎬 Video API Integration module loaded successfully');
