<?php
/**
 * Panadite Academy - Enhanced Login System
 * Integrated with SessionManager and AuthManager for robust session handling
 */

// Turn off output buffering to prevent header issues
ob_end_clean();
if (ob_get_level() == 0) ob_start();

// Start error reporting for debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);

// ==========================================
// ENHANCED SESSION AND AUTH MANAGEMENT
// ==========================================
// Integrated session management directly to avoid external dependencies

// Start secure session configuration
if (session_status() !== PHP_SESSION_ACTIVE) {
    // Enhanced session configuration for security and Linux server compatibility
    ini_set('session.cookie_httponly', 1);
    ini_set('session.use_only_cookies', 1);
    // Adjust secure flag based on HTTPS availability
    ini_set('session.cookie_secure', isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 1 : 0);
    ini_set('session.cookie_samesite', 'Lax');
    ini_set('session.use_strict_mode', 1);
    ini_set('session.cookie_lifetime', 7200); // 2 hours
    ini_set('session.gc_maxlifetime', 7200); // 2 hours
    ini_set('session.cookie_domain', ''); // Let PHP determine the domain
    ini_set('session.cookie_path', '/'); // Ensure cookie works for entire site

    // Set proper session save path for Linux server
    $temp_dir = sys_get_temp_dir();
    if (!is_writable($temp_dir)) {
        // Try alternative paths for Linux server
        $alternative_paths = ['/tmp', '/var/tmp', dirname(__FILE__) . '/tmp'];
        foreach ($alternative_paths as $path) {
            if (is_dir($path) && is_writable($path)) {
                $temp_dir = $path;
                break;
            }
        }
    }
    ini_set('session.save_path', $temp_dir);
    

    
    // Start session
    session_start();
}

// Initialize session security
if (!isset($_SESSION['created_at'])) {
    $_SESSION['created_at'] = time();
    $_SESSION['last_regeneration'] = time();
    session_regenerate_id(true);
} else {
    // Regenerate session ID periodically for security
    if (time() - ($_SESSION['last_regeneration'] ?? 0) > 300) { // 5 minutes
        $_SESSION['last_regeneration'] = time();
        session_regenerate_id(true);
    }
    
    // Check session lifetime
    if (time() - $_SESSION['created_at'] > 7200) { // 2 hours
        session_destroy();
        session_start();
    }
}

require_once __DIR__ . '/functions.php';
require_once __DIR__ . '/../includes/site_config.php';

// Try to include database connection with error handling
try {
    require_once __DIR__ . '/../includes/db_connection.php';
    if (isset($conn) && $conn) {
        
    }
} catch (Exception $e) {
    // Continue with emergency credentials fallback
}

// Handle session clearing if requested
if (isset($_GET['clear_session']) && $_GET['clear_session'] === 'true') {
    error_log('DEBUG - Clearing session as requested');
    // Clear session completely
    $_SESSION = array();
    if (ini_get("session.use_cookies")) {
        $params = session_get_cookie_params();
        setcookie(session_name(), '', time() - 42000,
            $params["path"], $params["domain"],
            $params["secure"], $params["httponly"]
        );
    }
    session_destroy();
    header('Location: login.php');
    exit();
}

// Check for redirect parameter
$redirect = '';
if (isset($_GET['redirect'])) {
    $redirect = $_GET['redirect'];
    // Store redirect in session to persist through session clearing
    $_SESSION['stored_redirect'] = $redirect;
} else if (isset($_SESSION['stored_redirect'])) {
    // Retrieve redirect from session if it was stored before
    $redirect = $_SESSION['stored_redirect'];
}

// Define isLoggedIn function inline if it doesn't exist
if (!function_exists('isLoggedIn')) {
    function isLoggedIn() {
        return isset($_SESSION['logged_in']) && $_SESSION['logged_in'] === true && 
               isset($_SESSION['user_id']) && !empty($_SESSION['user_id']);
    }
}

// Check if already logged in
if (isset($_SESSION['logged_in']) && $_SESSION['logged_in'] === true && isset($_SESSION['user_id'])) {
    $username = $_SESSION['username'] ?? 'unknown';
    $role = $_SESSION['user_role'] ?? 'unknown';
    error_log('DEBUG - User already logged in: ' . $username . ' (Role: ' . $role . ')');
    
    // Handle redirect logic
    if (!empty($redirect)) {
        header('Location: ' . $redirect);
    } else {
        // Redirect to role-based dashboard
        if ($role === 'admin') {
            $dashboard_url = $site_url . '/admin/dashboard.php';
        } else if ($role === 'teacher') {
            $dashboard_url = $site_url . '/teacher/dashboard.php';
        } else {
            $dashboard_url = $site_url . '/student/dashboard.php';
        }
        header('Location: ' . $dashboard_url);
    }
    exit();
}

$error = '';
$success = '';

// Check for logout success message
if (isset($_GET['logout']) && $_GET['logout'] == '1') {
    $success = 'You have been logged out successfully';
}



// Handle login form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $username = trim($_POST['username'] ?? '');
    $password = trim($_POST['password'] ?? '');
    

    
    if (empty($username) || empty($password)) {
        $error = 'Username and password are required';

    } else {

        
        // Check emergency credentials
        if (($username === 'admin' && $password === 'admin123') || 
            ($username === 'teacher' && $password === 'teacher123') || 
            ($username === 'student' && $password === 'student123')) {
            
            // Create emergency session
            error_log('DEBUG - Creating emergency session for: ' . $username);
            error_log('DEBUG - Before session_regenerate_id - Session ID: ' . session_id());
            error_log('DEBUG - Current session data before clear: ' . print_r($_SESSION, true));
            
            session_regenerate_id(true);
            $_SESSION = array();
            $_SESSION['user_id'] = 999;
            $_SESSION['username'] = $username;
            $_SESSION['first_name'] = ucfirst($username);
            $_SESSION['last_name'] = 'User';
            $_SESSION['user_role'] = $username;
            $_SESSION['logged_in'] = true;
            $_SESSION['is_logged_in'] = true;
            $_SESSION['login_time'] = time();
            $_SESSION['created_at'] = time();
            $_SESSION['last_regeneration'] = time();
            
            error_log('DEBUG - Session data set: ' . print_r($_SESSION, true));
            error_log('DEBUG - After setting session - Session ID: ' . session_id());
            
            // Force session to be written and immediately check
            session_write_close();
            session_start();
            
            error_log('DEBUG - After session_write_close/start - Session ID: ' . session_id());
            error_log('DEBUG - Emergency session final check: ' . print_r($_SESSION, true));
            
            // Test session directly
            $session_check = isset($_SESSION['logged_in']) && $_SESSION['logged_in'] === true;
            error_log('DEBUG - Session check returns: ' . ($session_check ? 'true' : 'false'));
            
            // Check cookie headers
            error_log('DEBUG - Headers already sent: ' . (headers_sent() ? 'YES' : 'NO'));
            error_log('DEBUG - Session cookie params: ' . print_r(session_get_cookie_params(), true));
            
            // Manual cookie setting as backup (in case automatic cookie isn't working)
            if (!headers_sent()) {
                $cookie_params = session_get_cookie_params();
                setcookie(
                    session_name(),
                    session_id(),
                    time() + 7200, // 2 hours
                    $cookie_params['path'],
                    $cookie_params['domain'],
                    $cookie_params['secure'],
                    $cookie_params['httponly']
                );
                error_log('DEBUG - Manual cookie set for session: ' . session_name() . '=' . session_id());
            }
            
            // Redirect to dashboard
            $dashboard_url = '';
            if ($username === 'admin') {
                $dashboard_url = $site_url . '/admin/dashboard.php';
            } else if ($username === 'teacher') {
                $dashboard_url = $site_url . '/teacher/dashboard.php';
            } else {
                $dashboard_url = 'https://panaditeacademy.com/student/dashboard.php';
            }
            
            error_log('DEBUG - Redirecting to: ' . $dashboard_url);
            
            header('Location: ' . $dashboard_url);
            exit();
            
        } else {
            // Database authentication
            try {
                // Check database connection with more specific error handling
                if (!isset($conn) || !$conn) {
                    error_log('DEBUG - Database connection not available');
                    throw new Exception('Database connection not available');
                }
                
                // Test the connection
                if ($conn->connect_error) {
                    error_log('DEBUG - Database connection error: ' . $conn->connect_error);
                    throw new Exception('Database connection failed: ' . $conn->connect_error);
                }
                
                // Prepare statement with error checking
                $sql = "SELECT user_id, username, email, password, first_name, last_name, user_role FROM users WHERE (username = ? OR email = ?)";
                $stmt = $conn->prepare($sql);
                
                if (!$stmt) {
                    error_log('DEBUG - Prepare failed: ' . $conn->error);
                    throw new Exception('Database query preparation failed: ' . $conn->error);
                }
                
                $stmt->bind_param("ss", $username, $username);
                
                if (!$stmt->execute()) {
                    error_log('DEBUG - Execute failed: ' . $stmt->error);
                    throw new Exception('Database query execution failed: ' . $stmt->error);
                }
                
                $result = $stmt->get_result();
                
                if ($result->num_rows === 0) {
                    $error = 'Invalid username/email or password';
                } else {
                    $user = $result->fetch_assoc();
                    
                    if (password_verify($password, $user['password']) || $user['password'] === $password) {
                        // Create user session (same logic as emergency credentials)
                        // Enhanced session data creation
                        $sessionData = [
                            'user_id' => $user['user_id'],
                            'username' => $user['username'],
                            'email' => $user['email'],
                            'first_name' => $user['first_name'],
                            'last_name' => $user['last_name'],
                            'user_role' => $user['user_role'],
                            'logged_in' => true,
                            'is_logged_in' => true,
                            'login_time' => time(),
                            'created_at' => time(),
                            'last_regeneration' => time()
                        ];
                        
                        // Set all session data at once
                        foreach ($sessionData as $key => $value) {
                            $_SESSION[$key] = $value;
                        }


                        
                        // Manual cookie setting as backup (same as emergency credentials)
                        if (!headers_sent()) {
                            session_write_close();
                            session_start();
                            $session_id = session_id();
                            
                            // Force cookie settings with multiple fallbacks for cross-directory access
                            $cookie_options = [
                                'expires' => time() + 7200,
                                'path' => '/',  // CRITICAL: Root path for entire site
                                'domain' => $_SERVER['HTTP_HOST'] ?? $_SERVER['SERVER_NAME'] ?? '',
                                'secure' => isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on',
                                'httponly' => true,
                                'samesite' => 'Lax'
                            ];
                            
                            // Set cookie multiple ways to ensure it sticks
                            setcookie('PHPSESSID', $session_id, $cookie_options);
                            setcookie(session_name(), $session_id, $cookie_options);
                            
                            // Also set via header as additional fallback
                            $cookie_header = session_name() . '=' . $session_id . '; Path=/; HttpOnly; SameSite=Lax';
                            if ($cookie_options['secure']) {
                                $cookie_header .= '; Secure';
                            }
                            header('Set-Cookie: ' . $cookie_header, false);
                            

                        }
                        
                        // Redirect to dashboard
                        $role = $user['user_role'];
                        if ($role === 'admin') {
                            $dashboard_url = $site_url . '/admin/dashboard.php';
                        } else if ($role === 'teacher') {
                            $dashboard_url = $site_url . '/teacher/dashboard.php';
                        } else {
                            $dashboard_url = $site_url . '/student/dashboard.php';
                        }
                        
                        error_log('DEBUG - Redirecting to: ' . $dashboard_url);
                        
                        header('Location: ' . $dashboard_url);
                        exit();
                    } else {
                        $error = 'Invalid username/email or password';
                    }
                }
                $stmt->close();
                
            } catch (Exception $e) {
                // If database fails, log the error but still allow emergency credentials
                error_log('DEBUG - Database authentication failed: ' . $e->getMessage());
                $error = 'Database authentication unavailable. Please use emergency credentials or try again later.';
            }
        }
    }
}

?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Login - Panadite Academy</title>
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css">
    <!-- Updated Font Awesome with integrity check to ensure proper loading -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css" integrity="sha512-iecdLmaskl7CVkqkXNQ/ZH/XLlvWZOJyj7Yy7tcenmpD1ypASozpmT/E0iPtmFIB46ZmdtAc9eNBvH0H/ZpiBw==" crossorigin="anonymous" referrerpolicy="no-referrer" />
    <link rel="stylesheet" href="/assets/css/style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/animate.css/4.1.1/animate.min.css">
    <style>
        /* Enhanced Form Animation Styles */
        body {
            background-color: #f8f9fa;
            position: relative;
            overflow-x: hidden;
            min-height: 100vh;
        }
        
        /* Floating Education Icons */
        .floating-icon {
            position: absolute;
            opacity: 0.05; /* Reduced opacity */
            z-index: 0;
            pointer-events: none;
            color: #276BE4;
            filter: drop-shadow(0 3px 5px rgba(39, 107, 228, 0.1)); /* Lighter shadow */
        }
        
        /* SVG Shapes */
        .shape {
            position: absolute;
            z-index: 0;
            opacity: 0.04;
            pointer-events: none;
        }
        
        .shape-circle {
            border-radius: 50%;
            background: #276BE4;
        }
        
        .shape-donut {
            border-radius: 50%;
            border: 8px solid #FFA915;
            background: transparent;
        }
        
        .shape-square {
            background: #276BE4;
            transform: rotate(45deg);
        }
        
        .shape-triangle {
            width: 0;
            height: 0;
            border-left: 50px solid transparent;
            border-right: 50px solid transparent;
            border-bottom: 86px solid #FFA915;
            background: transparent;
        }
        
        .navbar {
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
            background-color: white;
        }
        
        .navbar-brand img {
            height: 40px;
        }
        .login-container {
            max-width: 450px;
            margin: 50px auto;
            padding: 30px;
            background-color: #fff;
            border-radius: 15px;
            box-shadow: 0 5px 30px rgba(0,0,0,0.08);
            transition: all 0.5s ease;
            position: relative;
            overflow: hidden;
        }
        
        .login-container::before {
            content: '';
            position: absolute;
            top: -100%;
            left: -100%;
            width: 300%;
            height: 300%;
            background: linear-gradient(135deg, rgba(39,107,228,0.05) 0%, rgba(39,107,228,0) 100%);
            transform: rotate(45deg);
            z-index: 0;
            transition: all 0.8s ease;
        }
        
        .login-container:hover {
            transform: translateY(-5px);
            box-shadow: 0 15px 35px rgba(39, 107, 228, 0.1);
        }
        
        .login-container:hover::before {
            top: -150%;
            left: -150%;
            transition: all 1.5s ease;
        }
        .logo-container {
            text-align: center;
            margin-bottom: 30px;
        }
        .logo-container img {
            max-width: 200px;
            object-fit: contain;
            transition: transform 0.5s;
        }
        
        .logo-container img:hover {
            transform: scale(1.05);
        }
        .login-form .form-control {
            height: 50px;
            border-radius: 8px;
            box-shadow: none;
            margin-bottom: 20px;
            border: 1px solid #e1e1e1;
            padding-left: 15px;
            transition: all 0.4s ease;
            position: relative;
            z-index: 1;
        }
        
        .login-form .form-control:focus {
            border-color: #276BE4;
            box-shadow: 0 0 10px rgba(39, 107, 228, 0.2);
            transform: translateY(-3px);
        }
        
        .input-group-text {
            background-color: #f8f9fa;
            border-color: #e1e1e1;
            color: #276BE4;
        }
        
        .input-group-text i {
            width: 16px;
            text-align: center;
        }
        .btn-login {
            background-color: #276BE4;
            border-color: #276BE4;
            color: #fff;
            height: 50px;
            border-radius: 8px;
            font-weight: 600;
            margin-top: 10px;
            transition: all 0.4s ease;
            box-shadow: 0 4px 6px rgba(39, 107, 228, 0.15);
            letter-spacing: 0.5px;
            position: relative;
            overflow: hidden;
            z-index: 1;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 8px;
        }
        
        .btn-login i {
            font-size: 1.15rem;
            display: inline-block;
            opacity: 1;
            visibility: visible;
        }
        
        .btn-login::before {
            content: '';
            position: absolute;
            top: 0;
            left: -100%;
            width: 100%;
            height: 100%;
            background: rgba(255, 255, 255, 0.2);
            transition: all 0.4s ease;
            z-index: -1;
        }
        
        .btn-login:hover::before {
            left: 0;
            width: 100%;
        }
        .btn-login:hover {
            background-color: #1c54b8;
            transform: translateY(-3px);
            box-shadow: 0 7px 14px rgba(39, 107, 228, 0.25);
        }
        .login-links {
            text-align: center;
            margin-top: 20px;
        }
        .login-links a {
            color: #276BE4;
            text-decoration: none;
        }
        .login-links a:hover {
            text-decoration: underline;
        }
        .tagline {
            text-align: center;
            color: #FFA915;
            font-style: italic;
            margin-bottom: 20px;
            letter-spacing: 1px;
            font-size: 1.1em;
        }
        
        /* Password toggle button */
        .password-toggle {
            position: absolute;
            right: 10px;
            top: 50%;
            transform: translateY(-50%);
            cursor: pointer;
            color: #6c757d;
            z-index: 10;
        }
        
        .password-container {
            position: relative;
        }
        
        /* Enhanced animations */
        .animate-fade {
            animation: fadeIn 0.8s ease-out;
        }
        
        .animate-slide-up {
            animation: slideInUp 0.5s ease-out;
        }
        
        .form-label {
            transition: all 0.3s ease;
            position: relative;
            margin-bottom: 10px;
            font-weight: 500;
            color: #495057;
        }
        
        .form-label::after {
            content: '';
            position: absolute;
            left: 0;
            bottom: -4px;
            width: 0;
            height: 2px;
            background: #276BE4;
            transition: all 0.3s ease;
        }
        
        .input-group:focus-within + .form-label::after,
        .form-control:focus ~ .form-label::after {
            width: 30px;
        }
        
        .input-group {
            transition: all 0.3s ease;
        }
        
        .input-group:focus-within {
            transform: translateY(-3px);
        }
        
        @keyframes pulseGlow {
            0% { box-shadow: 0 0 0 0 rgba(39, 107, 228, 0.4); }
            70% { box-shadow: 0 0 0 10px rgba(39, 107, 228, 0); }
            100% { box-shadow: 0 0 0 0 rgba(39, 107, 228, 0); }
        }
        
        .pulse-animation {
            animation: pulseGlow 2s infinite;
        }
    </style>
</head>
<body>
    <!-- Floating Education Icons -->
    <div class="floating-icon" style="top: 10%; left: 5%; font-size: 40px;"><i class="fas fa-book"></i></div>
    <div class="floating-icon" style="top: 25%; right: 8%; font-size: 35px;"><i class="fas fa-graduation-cap"></i></div>
    <div class="floating-icon" style="top: 60%; left: 8%; font-size: 28px;"><i class="fas fa-lightbulb"></i></div>
    <div class="floating-icon" style="bottom: 15%; right: 12%; font-size: 32px;"><i class="fas fa-laptop-code"></i></div>
    <div class="floating-icon" style="bottom: 40%; left: 15%; font-size: 30px;"><i class="fas fa-brain"></i></div>
    
    <!-- Abstract Shapes -->
    <div class="shape shape-circle" style="top: 20%; left: 20%; width: 80px; height: 80px;"></div>
    <div class="shape shape-donut" style="bottom: 25%; right: 20%; width: 100px; height: 100px;"></div>
    <div class="shape shape-square" style="top: 40%; right: 10%; width: 60px; height: 60px;"></div>
    <div class="shape shape-triangle" style="bottom: 20%; left: 10%;"></div>
    
    <!-- Include the modern navbar -->
    <?php include '../includes/navbar.php'; ?>
    
    <div class="container">
        <div class="login-container">
            <div class="logo-container animate__animated animate__fadeIn">
                <img src="/assets/images/logo.jpeg" alt="Panadite Academy Logo" class="img-fluid">
                <p class="tagline animate__animated animate__fadeIn animate__delay-1s">Get ahead of the curve</p>
            </div>
            
            <?php if ($error): ?>
            <div class="alert alert-danger" role="alert">
                <?php echo $error; ?>
            </div>
            <?php endif; ?>
            
            <?php if ($success): ?>
            <div class="alert alert-success" role="alert">
                <?php echo $success; ?>
            </div>
            <?php endif; ?>
            
            <?php if (!empty($debug_info)): ?>
            <div class="alert alert-info" role="alert">
                <strong>Debug Information:</strong>
                <ul>
                    <?php foreach($debug_info as $info): ?>
                    <li><?php echo htmlspecialchars($info); ?></li>
                    <?php endforeach; ?>
                </ul>
                <p><strong>SESSION Data:</strong> <?php echo htmlspecialchars(print_r($_SESSION, true)); ?></p>
            </div>
            <?php endif; ?>
            

            
            <form class="login-form animate__animated animate__fadeIn animate__delay-1s" method="POST" action="<?php echo htmlspecialchars($_SERVER['PHP_SELF']) . (!empty($redirect) ? '?redirect=' . urlencode($redirect) : ''); ?>" id="loginForm">
                <div class="mb-3">
                    <label for="username" class="form-label">Username</label>
                    <div class="input-group">
                        <span class="input-group-text"><i class="fas fa-user"></i></span>
                        <input type="text" class="form-control" id="username" name="username" required>
                    </div>
                </div>
                
                <div class="mb-3">
                    <label for="password" class="form-label">Password</label>
                    <div class="input-group password-container">
                        <span class="input-group-text"><i class="fas fa-lock"></i></span>
                        <input type="password" class="form-control" id="password" name="password" required>
                        <span class="password-toggle" onclick="togglePassword('password')">
                            <i class="far fa-eye"></i>
                        </span>
                    </div>
                </div>
                

                
                <button type="submit" class="btn btn-login w-100 animate__animated animate__fadeIn animate__delay-2s" id="loginBtn">
                    Login <i class="fas fa-arrow-right-to-bracket ms-2"></i>
                </button>
            </form>
            
            <div class="login-links animate__animated animate__fadeIn animate__delay-1s">
                <a href="register.php">Don't have an account? Register</a><br>
               <!-- <a href="forgot_password.php">Forgot Password?</a>--><br>
                <a href="/index.php">Back to Home</a>
lass="alert alert-danger" role="alert" style="border-radius: 12px;">
                        <?php echo $error; ?>
                    </div>
                    <?php endif; ?>
                    
                    <?php if ($success): ?>
                    <div class="alert alert-success" role="alert" style="border-radius: 12px;">
                        <?php echo $success; ?>
                    </div>
                    <?php endif; ?>
                    
                    <form class="login-form" method="POST" action="<?php echo htmlspecialchars($_SERVER['PHP_SELF']) . (!empty($redirect) ? '?redirect=' . urlencode($redirect) : ''); ?>" id="loginForm">
                        <div class="mb-3">
                            <label for="username" class="form-label fw-medium" style="color: #2d3436;">Username</label>
                            <div class="input-group">
                                <span class="input-group-text" style="background: #f8f9fa; border-color: #e9ecef;">
                                    <i class="fas fa-user" style="color: #276BE4;"></i>
                                </span>
                                <input type="text" class="form-control" id="username" name="username" required 
                                       style="border-left: none; padding: 12px 16px; border-color: #e9ecef;"
                                       placeholder="Enter your username">
                            </div>
                        </div>
                        
                        <div class="mb-4">
                            <label for="password" class="form-label fw-medium" style="color: #2d3436;">Password</label>
                            <div class="input-group password-container">
                                <span class="input-group-text" style="background: #f8f9fa; border-color: #e9ecef;">
                                    <i class="fas fa-lock" style="color: #276BE4;"></i>
                                </span>
                                <input type="password" class="form-control" id="password" name="password" required 
                                       style="border-left: none; border-right: none; padding: 12px 16px; border-color: #e9ecef;"
                                       placeholder="Enter your password">
                                <span class="password-toggle" onclick="togglePassword('password')" 
                                      style="background: #f8f9fa; border-color: #e9ecef; cursor: pointer; padding: 12px 16px; border-left: none;">
                                    <i class="far fa-eye" style="color: #636e72;"></i>
                                </span>
                            </div>
                        </div>
                        
                        <button type="submit" class="btn w-100 py-3 mb-3" id="loginBtn"
                                style="background: linear-gradient(135deg, #276BE4, #1e5bcc); border: none; border-radius: 12px; color: white; font-weight: 600; font-size: 1.1rem; transition: all 0.3s ease;">
                            Sign In <i class="fas fa-arrow-right ms-2"></i>
                        </button>
                    </form>
                    
                    <div class="text-center">
                        <p style="color: #636e72; margin-bottom: 0.5rem;">
                            <a href="register.php" style="color: #276BE4; text-decoration: none; font-weight: 500;">
                                Don't have an account? Register
                            </a>
                        </p>
                        <p style="color: #636e72;">
                            <a href="/index.php" style="color: #636e72; text-decoration: none;">
                                <i class="fas fa-arrow-left me-2"></i>Back to Home
                            </a>
                        </p>
                    </div>
                </div>
            </div>
            
            <!-- Right Panel - Brand Content -->
            <div class="col-lg-6 login-right-panel">
                <!-- Animated Technology Icons -->
                <div class="tech-icon tech-icon-1"><i class="fas fa-laptop-code"></i></div>
                <div class="tech-icon tech-icon-2"><i class="fab fa-react"></i></div>
                <div class="tech-icon tech-icon-3"><i class="fas fa-database"></i></div>
                <div class="tech-icon tech-icon-4"><i class="fab fa-js-square"></i></div>
                <div class="tech-icon tech-icon-5"><i class="fas fa-cloud"></i></div>
                <div class="tech-icon tech-icon-6"><i class="fab fa-python"></i></div>
                <div class="tech-icon tech-icon-7"><i class="fas fa-mobile-alt"></i></div>
                <div class="tech-icon tech-icon-8"><i class="fas fa-brain"></i></div>
                
                <!-- Floating Particles -->
                <div class="particle particle-1"></div>
                <div class="particle particle-2"></div>
                <div class="particle particle-3"></div>
                <div class="particle particle-4"></div>
                

            </div>
        </div>
    </div>





    <footer class="mt-5 position-relative">
        <!-- Top gradient bar -->
        <div class="position-absolute top-0 start-0 w-100" style="height: 4px; background: linear-gradient(90deg, #276BE4, #FFA915);"></div>
        
        <!-- Main footer content -->
        <div class="bg-white py-5">
            <div class="container">
                <div class="row gy-4">
                    <!-- Logo and description -->
                    <div class="col-lg-4 col-md-6 mb-4">
                        <div class="mb-4">
                            <img src="/assets/images/logo.jpeg" alt="Panadite Academy" style="height: 50px;">
                        </div>
                        <p class="mb-2" style="color: #333; font-weight: 500;">Empowering learners through high-quality educational content.</p>
                        <p style="color: #333;">Get ahead of the curve with our premium videos and courses.</p>
                    </div>
                    
                    <!-- Quick links -->
                    <div class="col-lg-2 col-md-6 mb-4">
                        <h5 class="fw-bold mb-3 pb-2 position-relative" style="color: #222;">
                            Quick Links
                            <span class="position-absolute bottom-0 start-0" style="height: 2px; width: 40px; background: #FFA915;"></span>
                        </h5>
                        <ul class="list-unstyled">
                            <li class="mb-2"><a href="/index.php" class="text-decoration-none" style="color: #333;"><i class="fas fa-angle-right me-2" style="color: #276BE4;"></i>Home</a></li>
                            <li class="mb-2"><a href="/courses.php" class="text-decoration-none" style="color: #333;"><i class="fas fa-angle-right me-2" style="color: #276BE4;"></i>Courses</a></li>
                            <li class="mb-2"><a href="/videos.php" class="text-decoration-none" style="color: #333;"><i class="fas fa-angle-right me-2" style="color: #276BE4;"></i>Videos</a></li>
                            <li class="mb-2"><a href="/about.php" class="text-decoration-none" style="color: #333;"><i class="fas fa-angle-right me-2" style="color: #276BE4;"></i>About Us</a></li>
                            <li class="mb-2"><a href="/contact.php" class="text-decoration-none" style="color: #333;"><i class="fas fa-angle-right me-2" style="color: #276BE4;"></i>Contact</a></li>
                        </ul>
                    </div>
                    
                    <!-- Facilitator links -->
                    <div class="col-lg-2 col-md-6 mb-4">
                        <h5 class="fw-bold mb-3 pb-2 position-relative" style="color: #222;">
                            Facilitator
                            <span class="position-absolute bottom-0 start-0" style="height: 2px; width: 40px; background: #FFA915;"></span>
                        </h5>
                        <ul class="list-unstyled">
                            <li class="mb-2"><a href="/auth/register.php?role=teacher" class="text-decoration-none" style="color: #333;"><i class="fas fa-angle-right me-2" style="color: #276BE4;"></i>Become a Facilitator</a></li>
                            <li class="mb-2"><a href="/teacher/dashboard.php" class="text-decoration-none" style="color: #333;"><i class="fas fa-angle-right me-2" style="color: #276BE4;"></i>Teacher Dashboard</a></li>
                            <li class="mb-2"><a href="/teacher/sales.php" class="text-decoration-none" style="color: #333;"><i class="fas fa-angle-right me-2" style="color: #276BE4;"></i>Your Sales</a></li>
                            <li class="mb-2"><a href="/teacher/help.php" class="text-decoration-none" style="color: #333;"><i class="fas fa-angle-right me-2" style="color: #276BE4;"></i>Teacher Help</a></li>
                        </ul>
                    </div>
                    
                    <!-- Contact info -->
                    <div class="col-lg-4 col-md-6">
                        <h5 class="fw-bold mb-3 pb-2 position-relative" style="color: #222;">
                            Contact
                            <span class="position-absolute bottom-0 start-0" style="height: 2px; width: 40px; background: #FFA915;"></span>
                        </h5>
                        <ul class="list-unstyled mb-4">
                            <li class="mb-3 d-flex align-items-center">
                                <span class="d-flex align-items-center justify-content-center me-3" style="width: 36px; height: 36px; background: rgba(39,107,228,0.1); 
    </script>    
border-radius: 50%;">
                                    <i class="fas fa-map-marker-alt" style="color: #276BE4;"></i>
                                </span>
                                <span style="color: #333;">296 Pretorius Street, Corner Lilian Ngoyi, Pretoria Central, 0001, South Africa</span>
                            </li>
                            <li class="mb-3 d-flex align-items-center">
                                <span class="d-flex align-items-center justify-content-center me-3" style="width: 36px; height: 36px; background: rgba(39,107,228,0.1); border-radius: 50%;">
                                    <i class="fas fa-envelope" style="color: #276BE4;"></i>
                                </span>
                                <a href="mailto:admin@panaditeacademy.com" class="text-decoration-none" style="color: #333;">admin@panaditeacademy.com</a>
                            </li>
                            <li class="mb-3 d-flex align-items-center">
                                <span class="d-flex align-items-center justify-content-center me-3" style="width: 36px; height: 36px; background: rgba(39,107,228,0.1); border-radius: 50%;">
                                    <i class="fas fa-phone" style="color: #276BE4;"></i>
                                </span>
                                <a href="tel:+27747223141" class="text-decoration-none" style="color: #333;">+27 74 722 3141</a>
                            </li>
                        </ul>
                        
                        <!-- Social icons -->
                        <div class="d-flex">
                            <a href="#" class="d-flex align-items-center justify-content-center me-2" style="width: 40px; height: 40px; background: rgba(39,107,228,0.1); border-radius: 50%; color: #276BE4; transition: all 0.3s ease;">
                                <i class="fab fa-facebook-f"></i>
                            </a>
                            <a href="#" class="d-flex align-items-center justify-content-center me-2" style="width: 40px; height: 40px; background: rgba(39,107,228,0.1); border-radius: 50%; color: #276BE4; transition: all 0.3s ease;">
                                <i class="fab fa-twitter"></i>
                            </a>
                            <a href="#" class="d-flex align-items-center justify-content-center me-2" style="width: 40px; height: 40px; background: rgba(39,107,228,0.1); border-radius: 50%; color: #276BE4; transition: all 0.3s ease;">
                                <i class="fab fa-instagram"></i>
                            </a>
                            <a href="#" class="d-flex align-items-center justify-content-center" style="width: 40px; height: 40px; background: rgba(39,107,228,0.1); border-radius: 50%; color: #276BE4; transition: all 0.3s ease;">
                                <i class="fab fa-linkedin-in"></i>
                            </a>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Copyright section -->
        <div style="background: #f8f9fa; border-top: 1px solid #eee;" class="py-3">
            <div class="container">
                <div class="row align-items-center">
                    <div class="col-md-6 mb-3 mb-md-0">
                        <p class="mb-0" style="color: #333; font-weight: 500;">&copy; <?php echo date('Y'); ?> Panadite Academy. All rights reserved.</p>
                    </div>
                    <div class="col-md-6 text-md-end">
                        <a href="/privacy-policy.php" class="text-decoration-none me-3" style="color: #333;">Privacy Policy</a>
                        <a href="/terms.php" class="text-decoration-none me-3" style="color: #333;">Terms of Use</a>
                        <a href="/sitemap.php" class="text-decoration-none" style="color: #333;">Sitemap</a>
                    </div>
                </div>
            </div>
        </div>
    </footer>

    <script src="https://cdn.jsdelivr.net/npm/jquery@3.6.4/dist/jquery.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        function togglePassword(inputId) {
            const input = document.getElementById(inputId);
            const icon = input.nextElementSibling.querySelector('i');
            
            if (input.type === 'password') {
                input.type = 'text';
                icon.classList.remove('fa-eye');
                icon.classList.add('fa-eye-slash');
            } else {
                input.type = 'password';
                icon.classList.remove('fa-eye-slash');
                icon.classList.add('fa-eye');
            }
        }
        

        
        // Add form submission animation
        document.getElementById('loginForm').addEventListener('submit', function(e) {
            document.getElementById('loginBtn').innerHTML = 'Logging in... <i class="fas fa-spinner fa-spin ms-2"></i>';
            document.getElementById('loginBtn').disabled = true;
        });
        
        $(document).ready(function() {
            // Animate floating icons and shapes
            animateFloatingElements();
            
            // Add staggered animations to form elements
            $('.form-control').each(function(index) {
                $(this).addClass('animate__animated animate__fadeInUp')
                       .css('animation-delay', (index * 100 + 300) + 'ms');
            });
            
            // Add focus animations
            $('.form-control').on('focus', function() {
                $(this).parent().addClass('pulse-animation');
            }).on('blur', function() {
                $(this).parent().removeClass('pulse-animation');
            });
            
            // Add floating effect to login container
            $('.login-container').on('mousemove', function(e) {
                const container = $(this);
                const containerWidth = container.width();
                const containerHeight = container.height();
                const centerX = container.offset().left + containerWidth/2;
                const centerY = container.offset().top + containerHeight/2;
                const mouseX = e.pageX - centerX;
                const mouseY = e.pageY - centerY;
                const rotateY = (mouseX / containerWidth) * 5;
                const rotateX = ((mouseY / containerHeight) * -5);
                
                container.css('transform', 
                    `translateY(-5px) perspective(1000px) rotateX(${rotateX}deg) rotateY(${rotateY}deg)`
                );
            }).on('mouseleave', function() {
                $(this).css('transform', 'translateY(-5px)');
            });
            
            // Add shine effect on button hover
            $('.btn-login').on('mouseenter', function() {
                $(this).addClass('animate__animated animate__pulse');
                $(this).find('i').css({'opacity': '1', 'transform': 'translateX(3px)'});
            }).on('mouseleave', function() {
                $(this).removeClass('animate__animated animate__pulse');
                $(this).find('i').css({'transform': 'translateX(0)'});
            });
            
            // Ensure the icon is visible with a small animation to draw attention
            $('.btn-login i').css({'opacity': '1', 'transition': 'transform 0.3s ease'});
            
            // Add a quick flicker animation to ensure the icon is noticed
            setTimeout(function() {
                $('.btn-login i').css({'transform': 'scale(1.2)'});
                setTimeout(function() {
                    $('.btn-login i').css({'transform': 'scale(1)'});
                }, 300);
            }, 1000);
            
            // Validate login form before submission
            $('.login-form').on('submit', function() {
                const username = $('#username').val();
                const password = $('#password').val();
                
                if (!username || !password) {
                    return false;
                }
                
                $('.btn-login').html('<span class="spinner-border spinner-border-sm" role="status" aria-hidden="true"></span> Signing in...');
                return true;
            });
        });
        // Function to animate floating elements
        function animateFloatingElements() {
            $('.floating-icon, .shape').each(function(index) {
                var element = $(this);
                // Set random initial positions
                var startX = Math.random() * 10 - 5; // -5 to +5
                var startY = Math.random() * 10 - 5; // -5 to +5
                
                // Animate with random duration between 5-10 seconds
                var duration = 5000 + Math.random() * 5000;
                
                // Function to create floating animation
                function floatElement() {
                    // Generate new random positions
                    var moveX = Math.random() * 20 - 10; // -10 to +10
                    var moveY = Math.random() * 20 - 10; // -10 to +10
                    var rotate = Math.random() * 10 - 5; // -5 to +5 degrees
                    
                    element.animate({
                        transform: 'translate(' + moveX + 'px, ' + moveY + 'px) rotate(' + rotate + 'deg)'
                    }, duration, 'linear', function() {
                        floatElement(); // Infinite loop
                    });
                }
                
                // Start with a delay based on index
                setTimeout(function() {
                    element.css('transform', 'translate(' + startX + 'px, ' + startY + 'px)');
                    floatElement();
                }, index * 300);
            });
        }
    </script>
</body>
</html>

        
        .btn-primary {
            background: linear-gradient(135deg, #276BE4, #1e5bcc);
            border: none;
            border-radius: 10px;
            padding: 0.75rem 2rem;
            font-weight: 600;
            font-size: 0.95rem;
            transition: all 0.3s ease;
        }
        
        .login-form-container .btn-link {
            color: #276BE4;
            text-decoration: none;
            font-weight: 500;
            font-size: 0.9rem;
        }
        
        .login-form-container .btn-link:hover {
            color: #1e5bcc;
            text-decoration: underline;
        }
        
        .text-center a {
            color: #636e72;
            text-decoration: none;
            font-size: 0.9rem;
            transition: color 0.3s ease;
        }
        
        .text-center a:hover {
            color: #276BE4;
        }
        
        /* Loading Animation for Sign In Button */
        .btn-loading {
            position: relative;
        }
        
        .btn-loading .spinner {
            position: absolute;
            left: 50%;
            top: 50%;
            transform: translate(-50%, -50%);
        }
    </style>
</head>
<body>
    <!-- Include the modern navbar -->
    <?php include '../includes/navbar.php'; ?>
    
    <div class="container-fluid p-0">
        <div class="row g-0 login-split-container">
            <!-- Left Panel - Login Form -->
            <div class="col-lg-6 login-left-panel">
                <div class="login-form-container">
                    <div class="text-center mb-4">
                        <img src="/assets/images/logo.jpeg" alt="Panadite Academy Logo" class="img-fluid mb-3" style="height: 80px;">
                        <h3 style="color: #276BE4; font-weight: 600;">Welcome Back</h3>
                        <p style="color: #636e72;">Sign in to continue your learning journey</p>
                    </div>
                    
                    <?php if ($error): ?>
                    <div class="alert alert-danger" role="alert" style="border-radius: 12px;">
                        <?php echo $error; ?>
                    </div>
                    <?php endif; ?>
                    
                    <?php if ($success): ?>
                    <div class="alert alert-success" role="alert" style="border-radius: 12px;">
                        <?php echo $success; ?>
                    </div>
                    <?php endif; ?>
                    
                    <form class="login-form" method="POST" action="<?php echo htmlspecialchars($_SERVER['PHP_SELF']) . (!empty($redirect) ? '?redirect=' . urlencode($redirect) : ''); ?>" id="loginForm">
                        <div class="mb-3">
                            <label for="username" class="form-label fw-medium" style="color: #2d3436;">Username</label>
                            <div class="input-group">
                                <span class="input-group-text" style="background: #f8f9fa; border-color: #e9ecef;">
                                    <i class="fas fa-user" style="color: #276BE4;"></i>
                                </span>
                                <input type="text" class="form-control" id="username" name="username" required 
                                       style="border-left: none; padding: 12px 16px; border-color: #e9ecef;"
                                       placeholder="Enter your username">
                            </div>
                        </div>
                        
                        <div class="mb-4">
                            <label for="password" class="form-label fw-medium" style="color: #2d3436;">Password</label>
                            <div class="input-group password-container">
                                <span class="input-group-text" style="background: #f8f9fa; border-color: #e9ecef;">
                                    <i class="fas fa-lock" style="color: #276BE4;"></i>
                                </span>
                                <input type="password" class="form-control" id="password" name="password" required 
                                       style="border-left: none; border-right: none; padding: 12px 16px; border-color: #e9ecef;"
                                       placeholder="Enter your password">
                                <span class="password-toggle" onclick="togglePassword('password')" 
                                      style="background: #f8f9fa; border-color: #e9ecef; cursor: pointer; padding: 12px 16px; border-left: none;">
                                    <i class="far fa-eye" style="color: #636e72;"></i>
                                </span>
                            </div>
                        </div>
                        
                        <button type="submit" class="btn w-100 py-3 mb-3" id="loginBtn"
                                style="background: linear-gradient(135deg, #276BE4, #1e5bcc); border: none; border-radius: 12px; color: white; font-weight: 600; font-size: 1.1rem; transition: all 0.3s ease;">
                            Sign In <i class="fas fa-arrow-right ms-2"></i>
                        </button>
                    </form>
                    
                    <div class="text-center">
                        <p style="color: #636e72; margin-bottom: 0.5rem;">
                            <a href="register.php" style="color: #276BE4; text-decoration: none; font-weight: 500;">
                                Don't have an account? Register
                            </a>
                        </p>
                        <p style="color: #636e72;">
                            <a href="/index.php" style="color: #636e72; text-decoration: none;">
                                <i class="fas fa-arrow-left me-2"></i>Back to Home
                            </a>
                        </p>
                    </div>
                </div>
            </div>
            
            <!-- Right Panel - Brand Content -->
            <div class="col-lg-6 login-right-panel">
                <!-- Animated Technology Icons -->
                <div class="tech-icon tech-icon-1"><i class="fas fa-laptop-code"></i></div>
                <div class="tech-icon tech-icon-2"><i class="fab fa-react"></i></div>
                <div class="tech-icon tech-icon-3"><i class="fas fa-database"></i></div>
                <div class="tech-icon tech-icon-4"><i class="fab fa-js-square"></i></div>
                <div class="tech-icon tech-icon-5"><i class="fas fa-cloud"></i></div>
                <div class="tech-icon tech-icon-6"><i class="fab fa-python"></i></div>
                <div class="tech-icon tech-icon-7"><i class="fas fa-mobile-alt"></i></div>
                <div class="tech-icon tech-icon-8"><i class="fas fa-brain"></i></div>
                
                <!-- Floating Particles -->
                <div class="particle particle-1"></div>
                <div class="particle particle-2"></div>
                <div class="particle particle-3"></div>
                <div class="particle particle-4"></div>
                

            </div>
        </div>
    </div>





    <footer class="mt-5 position-relative">
        <!-- Top gradient bar -->
        <div class="position-absolute top-0 start-0 w-100" style="height: 4px; background: linear-gradient(90deg, #276BE4, #FFA915);"></div>
        
        <!-- Main footer content -->
        <div class="bg-white py-5">
            <div class="container">
                <div class="row gy-4">
                    <!-- Logo and description -->
                    <div class="col-lg-4 col-md-6 mb-4">
                        <div class="mb-4">
                            <img src="/assets/images/logo.jpeg" alt="Panadite Academy" style="height: 50px;">
                        </div>
                        <p class="mb-2" style="color: #333; font-weight: 500;">Empowering learners through high-quality educational content.</p>
                        <p style="color: #333;">Get ahead of the curve with our premium videos and courses.</p>
                    </div>
                    
                    <!-- Quick links -->
                    <div class="col-lg-2 col-md-6 mb-4">
                        <h5 class="fw-bold mb-3 pb-2 position-relative" style="color: #222;">
                            Quick Links
                            <span class="position-absolute bottom-0 start-0" style="height: 2px; width: 40px; background: #FFA915;"></span>
                        </h5>
                        <ul class="list-unstyled">
                            <li class="mb-2"><a href="/index.php" class="text-decoration-none" style="color: #333;"><i class="fas fa-angle-right me-2" style="color: #276BE4;"></i>Home</a></li>
                            <li class="mb-2"><a href="/courses.php" class="text-decoration-none" style="color: #333;"><i class="fas fa-angle-right me-2" style="color: #276BE4;"></i>Courses</a></li>
                            <li class="mb-2"><a href="/videos.php" class="text-decoration-none" style="color: #333;"><i class="fas fa-angle-right me-2" style="color: #276BE4;"></i>Videos</a></li>
                            <li class="mb-2"><a href="/about.php" class="text-decoration-none" style="color: #333;"><i class="fas fa-angle-right me-2" style="color: #276BE4;"></i>About Us</a></li>
                            <li class="mb-2"><a href="/contact.php" class="text-decoration-none" style="color: #333;"><i class="fas fa-angle-right me-2" style="color: #276BE4;"></i>Contact</a></li>
                        </ul>
                    </div>
                    
                    <!-- Facilitator links -->
                    <div class="col-lg-2 col-md-6 mb-4">
                        <h5 class="fw-bold mb-3 pb-2 position-relative" style="color: #222;">
                            Facilitator
                            <span class="position-absolute bottom-0 start-0" style="height: 2px; width: 40px; background: #FFA915;"></span>
                        </h5>
                        <ul class="list-unstyled">
                            <li class="mb-2"><a href="/auth/register.php?role=teacher" class="text-decoration-none" style="color: #333;"><i class="fas fa-angle-right me-2" style="color: #276BE4;"></i>Become a Facilitator</a></li>
                            <li class="mb-2"><a href="/teacher/dashboard.php" class="text-decoration-none" style="color: #333;"><i class="fas fa-angle-right me-2" style="color: #276BE4;"></i>Teacher Dashboard</a></li>
                            <li class="mb-2"><a href="/teacher/sales.php" class="text-decoration-none" style="color: #333;"><i class="fas fa-angle-right me-2" style="color: #276BE4;"></i>Your Sales</a></li>
                            <li class="mb-2"><a href="/teacher/help.php" class="text-decoration-none" style="color: #333;"><i class="fas fa-angle-right me-2" style="color: #276BE4;"></i>Teacher Help</a></li>
                        </ul>
                    </div>
                    
                    <!-- Contact info -->
                    <div class="col-lg-4 col-md-6">
                        <h5 class="fw-bold mb-3 pb-2 position-relative" style="color: #222;">
                            Contact
                            <span class="position-absolute bottom-0 start-0" style="height: 2px; width: 40px; background: #FFA915;"></span>
                        </h5>
                        <ul class="list-unstyled mb-4">
                            <li class="mb-3 d-flex align-items-center">
                                <span class="d-flex align-items-center justify-content-center me-3" style="width: 36px; height: 36px; background: rgba(39,107,228,0.1); border-radius: 50%;">
                                    <i class="fas fa-map-marker-alt" style="color: #276BE4;"></i>
                                </span>
                                <span style="color: #333;">296 Pretorius Street, Corner Lilian Ngoyi, Pretoria Central, 0001, South Africa</span>
                            </li>
                            <li class="mb-3 d-flex align-items-center">
                                <span class="d-flex align-items-center justify-content-center me-3" style="width: 36px; height: 36px; background: rgba(39,107,228,0.1); border-radius: 50%;">
                                    <i class="fas fa-envelope" style="color: #276BE4;"></i>
                                </span>
                                <a href="mailto:admin@panaditeacademy.com" class="text-decoration-none" style="color: #333;">admin@panaditeacademy.com</a>
                            </li>
                            <li class="mb-3 d-flex align-items-center">
                                <span class="d-flex align-items-center justify-content-center me-3" style="width: 36px; height: 36px; background: rgba(39,107,228,0.1); border-radius: 50%;">
                                    <i class="fas fa-phone" style="color: #276BE4;"></i>
                                </span>
                                <a href="tel:+27747223141" class="text-decoration-none" style="color: #333;">+27 74 722 3141</a>
                            </li>
                        </ul>
                        
                        <!-- Social icons -->
                        <div class="d-flex">
                            <a href="#" class="d-flex align-items-center justify-content-center me-2" style="width: 40px; height: 40px; background: rgba(39,107,228,0.1); border-radius: 50%; color: #276BE4; transition: all 0.3s ease;">
                                <i class="fab fa-facebook-f"></i>
                            </a>
                            <a href="#" class="d-flex align-items-center justify-content-center me-2" style="width: 40px; height: 40px; background: rgba(39,107,228,0.1); border-radius: 50%; color: #276BE4; transition: all 0.3s ease;">
                                <i class="fab fa-twitter"></i>
                            </a>
                            <a href="#" class="d-flex align-items-center justify-content-center me-2" style="width: 40px; height: 40px; background: rgba(39,107,228,0.1); border-radius: 50%; color: #276BE4; transition: all 0.3s ease;">
                                <i class="fab fa-instagram"></i>
                            </a>
                            <a href="#" class="d-flex align-items-center justify-content-center" style="width: 40px; height: 40px; background: rgba(39,107,228,0.1); border-radius: 50%; color: #276BE4; transition: all 0.3s ease;">
                                <i class="fab fa-linkedin-in"></i>
                            </a>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Copyright section -->
        <div style="background: #f8f9fa; border-top: 1px solid #eee;" class="py-3">
            <div class="container">
                <div class="row align-items-center">
                    <div class="col-md-6 mb-3 mb-md-0">
                        <p class="mb-0" style="color: #333; font-weight: 500;">&copy; <?php echo date('Y'); ?> Panadite Academy. All rights reserved.</p>
                    </div>
                    <div class="col-md-6 text-md-end">
                        <a href="/privacy-policy.php" class="text-decoration-none me-3" style="color: #333;">Privacy Policy</a>
                        <a href="/terms.php" class="text-decoration-none me-3" style="color: #333;">Terms of Use</a>
                        <a href="/sitemap.php" class="text-decoration-none" style="color: #333;">Sitemap</a>
                    </div>
                </div>
            </div>
        </div>
    </footer>

    <script src="https://cdn.jsdelivr.net/npm/jquery@3.6.4/dist/jquery.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        function togglePassword(inputId) {
            const input = document.getElementById(inputId);
            const icon = input.nextElementSibling.querySelector('i');
            
            if (input.type === 'password') {
                input.type = 'text';
                icon.classList.remove('fa-eye');
                icon.classList.add('fa-eye-slash');
            } else {
                input.type = 'password';
                icon.classList.remove('fa-eye-slash');
                icon.classList.add('fa-eye');
            }
        }
        

        
        // Add form submission animation
        document.getElementById('loginForm').addEventListener('submit', function(e) {
            document.getElementById('loginBtn').innerHTML = 'Logging in... <i class="fas fa-spinner fa-spin ms-2"></i>';
            document.getElementById('loginBtn').disabled = true;
        });
        
        $(document).ready(function() {
            // Animate floating icons and shapes
            animateFloatingElements();
            
            // Add staggered animations to form elements
            $('.form-control').each(function(index) {
                $(this).addClass('animate__animated animate__fadeInUp')
                       .css('animation-delay', (index * 100 + 300) + 'ms');
            });
            
            // Add focus animations
            $('.form-control').on('focus', function() {
                $(this).parent().addClass('pulse-animation');
            }).on('blur', function() {
                $(this).parent().removeClass('pulse-animation');
            });
            
            // Add floating effect to login container
            $('.login-container').on('mousemove', function(e) {
                const container = $(this);
                const containerWidth = container.width();
                const containerHeight = container.height();
                const centerX = container.offset().left + containerWidth/2;
                const centerY = container.offset().top + containerHeight/2;
                const mouseX = e.pageX - centerX;
                const mouseY = e.pageY - centerY;
                const rotateY = (mouseX / containerWidth) * 5;
                const rotateX = ((mouseY / containerHeight) * -5);
                
                container.css('transform', 
                    `translateY(-5px) perspective(1000px) rotateX(${rotateX}deg) rotateY(${rotateY}deg)`
                );
            }).on('mouseleave', function() {
                $(this).css('transform', 'translateY(-5px)');
            });
            
            // Add shine effect on button hover
            $('.btn-login').on('mouseenter', function() {
                $(this).addClass('animate__animated animate__pulse');
                $(this).find('i').css({'opacity': '1', 'transform': 'translateX(3px)'});
            }).on('mouseleave', function() {
                $(this).removeClass('animate__animated animate__pulse');
                $(this).find('i').css({'transform': 'translateX(0)'});
            });
            
            // Ensure the icon is visible with a small animation to draw attention
            $('.btn-login i').css({'opacity': '1', 'transition': 'transform 0.3s ease'});
            
            // Add a quick flicker animation to ensure the icon is noticed
            setTimeout(function() {
                $('.btn-login i').css({'transform': 'scale(1.2)'});
                setTimeout(function() {
                    $('.btn-login i').css({'transform': 'scale(1)'});
                }, 300);
            }, 1000);
            
            // Validate login form before submission
            $('.login-form').on('submit', function() {
                const username = $('#username').val();
                const password = $('#password').val();
                
                if (!username || !password) {
                    return false;
                }
                
                $('.btn-login').html('<span class="spinner-border spinner-border-sm" role="status" aria-hidden="true"></span> Signing in...');
                return true;
            });
        });
        // Function to animate floating elements
        function animateFloatingElements() {
            $('.floating-icon, .shape').each(function(index) {
                var element = $(this);
                // Set random initial positions
                var startX = Math.random() * 10 - 5; // -5 to +5
                var startY = Math.random() * 10 - 5; // -5 to +5
                
                // Animate with random duration between 5-10 seconds
                var duration = 5000 + Math.random() * 5000;
                
                // Function to create floating animation
                function floatElement() {
                    // Generate new random positions
                    var moveX = Math.random() * 20 - 10; // -10 to +10
                    var moveY = Math.random() * 20 - 10; // -10 to +10
                    var rotate = Math.random() * 10 - 5; // -5 to +5 degrees
                    
                    element.animate({
                        transform: 'translate(' + moveX + 'px, ' + moveY + 'px) rotate(' + rotate + 'deg)'
                    }, duration, 'linear', function() {
                        floatElement(); // Infinite loop
                    });
                }
                
                // Start with a delay based on index
                setTimeout(function() {
                    element.css('transform', 'translate(' + startX + 'px, ' + startY + 'px)');
                    floatElement();
                }, index * 300);
            });
        }
    </script>
</body>
</html>

                    </div>
                    
                    <!-- Facilitator links -->
                    <div class="col-lg-2 col-md-6 mb-4">
                        <h5 class="fw-bold mb-3 pb-2 position-relative" style="color: #222;">
                            Facilitator
                            <span class="position-absolute bottom-0 start-0" style="height: 2px; width: 40px; background: #FFA915;"></span>
                        </h5>
                        <ul class="list-unstyled">
                            <li class="mb-2"><a href="/auth/register.php?role=teacher" class="text-decoration-none" style="color: #333;"><i class="fas fa-angle-right me-2" style="color: #276BE4;"></i>Become a Facilitator</a></li>
                            <li class="mb-2"><a href="/teacher/dashboard.php" class="text-decoration-none" style="color: #333;"><i class="fas fa-angle-right me-2" style="color: #276BE4;"></i>Teacher Dashboard</a></li>
                            <li class="mb-2"><a href="/teacher/sales.php" class="text-decoration-none" style="color: #333;"><i class="fas fa-angle-right me-2" style="color: #276BE4;"></i>Your Sales</a></li>
                            <li class="mb-2"><a href="/teacher/help.php" class="text-decoration-none" style="color: #333;"><i class="fas fa-angle-right me-2" style="color: #276BE4;"></i>Teacher Help</a></li>
                        </ul>
                    </div>
                    
                    <!-- Contact info -->
                    <div class="col-lg-4 col-md-6">
                        <h5 class="fw-bold mb-3 pb-2 position-relative" style="color: #222;">
                            Contact
                            <span class="position-absolute bottom-0 start-0" style="height: 2px; width: 40px; background: #FFA915;"></span>
                        </h5>
                        <ul class="list-unstyled mb-4">
                            <li class="mb-3 d-flex align-items-center">
                                <span class="d-flex align-items-center justify-content-center me-3" style="width: 36px; height: 36px; background: rgba(39,107,228,0.1); border-radius: 50%;">
                                    <i class="fas fa-map-marker-alt" style="color: #276BE4;"></i>
                                </span>
                                <span style="color: #333;">296 Pretorius Street, Corner Lilian Ngoyi, Pretoria Central, 0001, South Africa</span>
                            </li>
                            <li class="mb-3 d-flex align-items-center">
                                <span class="d-flex align-items-center justify-content-center me-3" style="width: 36px; height: 36px; background: rgba(39,107,228,0.1); border-radius: 50%;">
                                    <i class="fas fa-envelope" style="color: #276BE4;"></i>
                                </span>
                                <a href="mailto:admin@panaditeacademy.com" class="text-decoration-none" style="color: #333;">admin@panaditeacademy.com</a>
                            </li>
                            <li class="mb-3 d-flex align-items-center">
                                <span class="d-flex align-items-center justify-content-center me-3" style="width: 36px; height: 36px; background: rgba(39,107,228,0.1); border-radius: 50%;">
                                    <i class="fas fa-phone" style="color: #276BE4;"></i>
                                </span>
                                <a href="tel:+27747223141" class="text-decoration-none" style="color: #333;">+27 74 722 3141</a>
                            </li>
                        </ul>
                        
                        <!-- Social icons -->
                        <div class="d-flex">
                            <a href="#" class="d-flex align-items-center justify-content-center me-2" style="width: 40px; height: 40px; background: rgba(39,107,228,0.1); border-radius: 50%; color: #276BE4; transition: all 0.3s ease;">
                                <i class="fab fa-facebook-f"></i>
                            </a>
                            <a href="#" class="d-flex align-items-center justify-content-center me-2" style="width: 40px; height: 40px; background: rgba(39,107,228,0.1); border-radius: 50%; color: #276BE4; transition: all 0.3s ease;">
                                <i class="fab fa-twitter"></i>
                            </a>
                            <a href="#" class="d-flex align-items-center justify-content-center me-2" style="width: 40px; height: 40px; background: rgba(39,107,228,0.1); border-radius: 50%; color: #276BE4; transition: all 0.3s ease;">
                                <i class="fab fa-instagram"></i>
                            </a>
                            <a href="#" class="d-flex align-items-center justify-content-center" style="width: 40px; height: 40px; background: rgba(39,107,228,0.1); border-radius: 50%; color: #276BE4; transition: all 0.3s ease;">
                                <i class="fab fa-linkedin-in"></i>
                            </a>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Copyright section -->
        <div style="background: #f8f9fa; border-top: 1px solid #eee;" class="py-3">
            <div class="container">
                <div class="row align-items-center">
                    <div class="col-md-6 mb-3 mb-md-0">
                        <p class="mb-0" style="color: #333; font-weight: 500;">&copy; <?php echo date('Y'); ?> Panadite Academy. All rights reserved.</p>
                    </div>
                    <div class="col-md-6 text-md-end">
                        <a href="/privacy-policy.php" class="text-decoration-none me-3" style="color: #333;">Privacy Policy</a>
                        <a href="/terms.php" class="text-decoration-none me-3" style="color: #333;">Terms of Use</a>
                        <a href="/sitemap.php" class="text-decoration-none" style="color: #333;">Sitemap</a>
                    </div>
                </div>
            </div>
        </div>
    </footer>

    <script src="https://cdn.jsdelivr.net/npm/jquery@3.6.4/dist/jquery.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        function togglePassword(inputId) {
            const input = document.getElementById(inputId);
            const icon = input.nextElementSibling.querySelector('i');
            
            if (input.type === 'password') {
                input.type = 'text';
                icon.classList.remove('fa-eye');
                icon.classList.add('fa-eye-slash');
            } else {
                input.type = 'password';
                icon.classList.remove('fa-eye-slash');
                icon.classList.add('fa-eye');
            }
        }
        

        
        // Add form submission animation
        document.getElementById('loginForm').addEventListener('submit', function(e) {
            document.getElementById('loginBtn').innerHTML = 'Logging in... <i class="fas fa-spinner fa-spin ms-2"></i>';
            document.getElementById('loginBtn').disabled = true;
        });
        
        $(document).ready(function() {
            // Animate floating icons and shapes
            animateFloatingElements();
            
            // Add staggered animations to form elements
            $('.form-control').each(function(index) {
                $(this).addClass('animate__animated animate__fadeInUp')
                       .css('animation-delay', (index * 100 + 300) + 'ms');
            });
            
            // Add focus animations
            $('.form-control').on('focus', function() {
                $(this).parent().addClass('pulse-animation');
            }).on('blur', function() {
                $(this).parent().removeClass('pulse-animation');
            });
            
            // Add floating effect to login container
            $('.login-container').on('mousemove', function(e) {
                const container = $(this);
                const containerWidth = container.width();
                const containerHeight = container.height();
                const centerX = container.offset().left + containerWidth/2;
                const centerY = container.offset().top + containerHeight/2;
                const mouseX = e.pageX - centerX;
                const mouseY = e.pageY - centerY;
                const rotateY = (mouseX / containerWidth) * 5;
                const rotateX = ((mouseY / containerHeight) * -5);
                
                container.css('transform', 
                    `translateY(-5px) perspective(1000px) rotateX(${rotateX}deg) rotateY(${rotateY}deg)`
                );
            }).on('mouseleave', function() {
                $(this).css('transform', 'translateY(-5px)');
            });
            
            // Add shine effect on button hover
            $('.btn-login').on('mouseenter', function() {
                $(this).addClass('animate__animated animate__pulse');
                $(this).find('i').css({'opacity': '1', 'transform': 'translateX(3px)'});
            }).on('mouseleave', function() {
                $(this).removeClass('animate__animated animate__pulse');
                $(this).find('i').css({'transform': 'translateX(0)'});
            });
            
            // Ensure the icon is visible with a small animation to draw attention
            $('.btn-login i').css({'opacity': '1', 'transition': 'transform 0.3s ease'});
            
            // Add a quick flicker animation to ensure the icon is noticed
            setTimeout(function() {
                $('.btn-login i').css({'transform': 'scale(1.2)'});
                setTimeout(function() {
                    $('.btn-login i').css({'transform': 'scale(1)'});
                }, 300);
            }, 1000);
            
            // Validate login form before submission
            $('.login-form').on('submit', function() {
                const username = $('#username').val();
                const password = $('#password').val();
                
                if (!username || !password) {
                    return false;
                }
                
                $('.btn-login').html('<span class="spinner-border spinner-border-sm" role="status" aria-hidden="true"></span> Signing in...');
                return true;
            });
        });
        // Function to animate floating elements
        function animateFloatingElements() {
            $('.floating-icon, .shape').each(function(index) {
                var element = $(this);
                // Set random initial positions
                var startX = Math.random() * 10 - 5; // -5 to +5
                var startY = Math.random() * 10 - 5; // -5 to +5
                
                // Animate with random duration between 5-10 seconds
                var duration = 5000 + Math.random() * 5000;
                
                // Function to create floating animation
                function floatElement() {
                    // Generate new random positions
                    var moveX = Math.random() * 20 - 10; // -10 to +10
                    var moveY = Math.random() * 20 - 10; // -10 to +10
                    var rotate = Math.random() * 10 - 5; // -5 to +5 degrees
                    
                    element.animate({
                        transform: 'translate(' + moveX + 'px, ' + moveY + 'px) rotate(' + rotate + 'deg)'
                    }, duration, 'linear', function() {
                        floatElement(); // Infinite loop
                    });
                }
                
                // Start with a delay based on index
                setTimeout(function() {
                    element.css('transform', 'translate(' + startX + 'px, ' + startY + 'px)');
                    floatElement();
                }, index * 300);
            });
        }
    </script>
</body>
</html>
