<?php
/**
 * PayFast Integration Functions
 * Panadite Academy
 */
require_once __DIR__ . '/../includes/config.php';

// Include fast email service for purchase confirmation notifications
require_once __DIR__ . '/../config/fast_email_service.php';

// Bridge database connections - Email service needs PDO, but PayFast uses MySQLi
if (!isset($pdo)) {
    try {
        // Create PDO connection using the same credentials as MySQLi
        $pdo = new PDO("mysql:host=" . DB_SERVER . ";dbname=" . DB_NAME, DB_USERNAME, DB_PASSWORD);
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        $pdo->setAttribute(PDO::ATTR_DEFAULT_FETCH_MODE, PDO::FETCH_ASSOC);
        error_log("PDO connection created successfully for email service");
    } catch (Exception $e) {
        error_log("Failed to create PDO connection for email service: " . $e->getMessage());
        $pdo = null;
    }
}

// Get current PayFast settings
$payfastSettings = getPayfastSettings();

/**
 * Send purchase confirmation notification to student
 */
function sendPurchaseConfirmationEmail($user_id, $course_id, $payment_id, $amount) {
    global $conn;
    
    try {
        // Get course and student details - SIMPLIFIED QUERY TO AVOID SQL ERRORS
        $stmt = $conn->prepare("
            SELECT c.title, c.description, c.price, 
                   u.first_name, u.last_name, u.email, u.user_id,
                   t.first_name as teacher_first_name, t.last_name as teacher_last_name
            FROM courses c 
            JOIN users u ON u.user_id = ?
            JOIN users t ON c.teacher_id = t.user_id 
            WHERE c.course_id = ?
        ");
        
        if (!$stmt) {
            error_log("SQL prepare failed in sendPurchaseConfirmationEmail: " . $conn->error);
            return false;
        }
        
        $stmt->bind_param("ii", $user_id, $course_id);
        $stmt->execute();
        $result = $stmt->get_result();
        $purchase = $result->fetch_assoc();
        
        if (!$purchase) {
            error_log("Course or user not found for purchase confirmation email: $course_id, $user_id");
            return false;
        }
        
        $student_name = $purchase['first_name'] . ' ' . $purchase['last_name'];
        $student_email = $purchase['email'];
        $course_title = $purchase['title'];
        $teacher_name = $purchase['teacher_first_name'] . ' ' . $purchase['teacher_last_name'];
        $purchase_amount = number_format(floatval($amount), 2);
        $purchase_date = date('M j, Y \a\t g:i A');
        
        // Create email content
        $subject = "🎉 Purchase Confirmed: \"$course_title\" - Access Now Available!";
        $message = "
        <div style='max-width:600px; margin:0 auto; font-family:Arial,sans-serif;'>
            <div style='background:#28a745; color:white; padding:20px; text-align:center;'>
                <h1>🎉 Purchase Confirmed!</h1>
            </div>
            <div style='padding:20px; background:white;'>
                <h2>Congratulations, " . htmlspecialchars($student_name) . "! 🎓</h2>
                <p>Your purchase of <strong>\"" . htmlspecialchars($course_title) . "\"</strong> has been confirmed and you now have full access!</p>
                <div style='background:#f8f9fa; padding:15px; border-radius:5px; margin:20px 0;'>
                    <h3>Purchase Details:</h3>
                    <p><strong>Course:</strong> " . htmlspecialchars($course_title) . "</p>
                    <p><strong>Instructor:</strong> " . htmlspecialchars($teacher_name) . "</p>
                    <p><strong>Amount Paid:</strong> R" . $purchase_amount . "</p>
                    <p><strong>Purchase Date:</strong> " . $purchase_date . "</p>
                    <p><strong>Payment ID:</strong> " . htmlspecialchars($payment_id) . "</p>
                </div>
                <div style='background:#e8f5e8; padding:15px; border-radius:5px; margin:20px 0;'>
                    <h4>🚀 Ready to Start Learning?</h4>
                    <p>You can now access all course materials, videos, and resources immediately!</p>
                    <p>Start your learning journey right away and unlock your potential.</p>
                </div>
                <p style='text-align:center;'>
                    <a href='https://panaditeacademy.com/student/my-courses.php' 
                       style='background:#667eea; color:white; text-decoration:none; padding:15px 25px; border-radius:8px; font-weight:bold; display:inline-block;'>
                        Access Your Course Now
                    </a>
                </p>
            </div>
        </div>";
        
        // Try direct email sending
        if (function_exists('sendEmailNow')) {
            $result = sendEmailNow($student_email, $subject, $message);
            if ($result) {
                return true;
            }
        }
        
        // Fallback to native PHP mail
        $headers = "MIME-Version: 1.0" . "\r\n";
        $headers .= "Content-type:text/html;charset=UTF-8" . "\r\n";
        $headers .= "From: noreply@panaditeacademy.com" . "\r\n";
        
        return mail($student_email, $subject, $message, $headers);
        
    } catch (Exception $e) {
        error_log("Error sending purchase confirmation email: " . $e->getMessage());
        return false;
    }
}

/**
 * Generate payment form data for PayFast
 * 
 * @param array $item Item details (title, amount, item_type, item_id)
 * @param array $user User details
 * @param string $returnUrl URL to return to after payment
 * @param string $cancelUrl URL to return to if payment is cancelled
 * @param string $notifyUrl URL for ITN notifications
 * @return array Payment form data
 */
function generatePayfastFormData($item, $user, $returnUrl, $cancelUrl, $notifyUrl) {
    $payfastSettings = getPayfastSettings();
    
    if (empty($payfastSettings['merchant_id']) || empty($payfastSettings['merchant_key'])) {
        throw new Exception('PayFast merchant settings not configured');
    }
    
    // Generate unique payment ID and custom fields based on item type
    $paymentId = 'PAY_' . time() . '_' . $user['user_id'] . '_' . $item['item_id'];
    
    // Set merchant payment ID and custom fields based on item type
    if ($item['item_type'] === 'accreditation') {
        // For accreditation payments
        $merchant_payment_id = $item['item_id']; // Use the payment reference (ACC_4_xxxxx)
        $custom_str1 = 'accreditation';
        $custom_int1 = intval(str_replace('acc_', '', $item['item_id'])); // Extract request_id from 'acc_4'
        $custom_int2 = intval($user['user_id'] ?? 0);
    } else {
        // For course/video payments (existing logic)
        $merchant_payment_id = 'course_' . $item['item_id'];
        $custom_str1 = $item['item_type']; // 'course' or 'video'
        $custom_int1 = intval($item['item_id']);
        $custom_int2 = intval($user['user_id'] ?? 0);
    }
    
    // Use ONLY the minimal required fields that work (exactly like working test)
    $data = [
        'merchant_id' => $payfastSettings['merchant_id'],
        'merchant_key' => $payfastSettings['merchant_key'],
        'return_url' => $returnUrl,
        'cancel_url' => $cancelUrl,
        'notify_url' => $notifyUrl,
        'amount' => number_format($item['amount'], 2, '.', ''),
        'item_name' => substr($item['title'], 0, 100),
        'm_payment_id' => $merchant_payment_id,
        'custom_str1' => $custom_str1,
        'custom_int1' => $custom_int1,
        'custom_int2' => $custom_int2
    ];
    
    // Since PayFast account has no passphrase configured, don't include signature field at all
    // PayFast expects NO signature field when account has no signature verification enabled
    
    // Determine PayFast URL
    $payfastUrl = $payfastSettings['sandbox_mode'] 
        ? 'https://sandbox.payfast.co.za/eng/process'
        : 'https://www.payfast.co.za/eng/process';
    
    return [
        'url' => $payfastUrl,
        'data' => $data,
        'payment_id' => $paymentId
    ];
}

/**
 * Generate PayFast signature
 * 
 * @param array $data Payment data
 * @param string $passphrase Optional passphrase
 * @return string Generated signature
 */
function generateSignature($data, $passphrase = null) {
    // Remove signature field if it exists
    $tempData = $data;
    if (isset($tempData['signature'])) {
        unset($tempData['signature']);
    }
    
    // Sort the array by key
    ksort($tempData);
    
    // Create parameter string according to PayFast requirements
    $pfOutput = '';
    foreach ($tempData as $key => $val) {
        // Skip empty values - be more strict about what we consider empty
        if ($val !== '' && $val !== null && $val !== false) {
            $pfOutput .= $key . '=' . urlencode(trim($val)) . '&';
        }
    }
    
    // Remove last ampersand
    $pfOutput = substr($pfOutput, 0, -1);
    
    // Add passphrase if provided and not empty
    if ($passphrase !== null && $passphrase !== '') {
        $pfOutput .= '&passphrase=' . urlencode(trim($passphrase));
    }
    
    // Generate MD5 hash
    return md5($pfOutput);
}

/**
 * Verify PayFast ITN (Instant Transaction Notification)
 * 
 * @param array $postData POST data from PayFast
 * @return bool True if verification passes
 */
function verifyPayfastItn($postData) {
    global $payfastSettings;
    
    $pfHost = $payfastSettings['sandbox_mode'] ? 'sandbox.payfast.co.za' : 'www.payfast.co.za';
    $pfValid = true;
    
    // Step 1: Verify data received is valid PayFast data
    $validHosts = array(
        'www.payfast.co.za',
        'sandbox.payfast.co.za',
        'w1w.payfast.co.za',
        'w2w.payfast.co.za'
    );

    $validIps = array();
    foreach ($validHosts as $pfHostname) {
        $ips = gethostbynamel($pfHostname);
        if ($ips !== false) {
            $validIps = array_merge($validIps, $ips);
        }
    }

    // Remove duplicates
    $validIps = array_unique($validIps);

    // Verify source IP
    if (!in_array($_SERVER['REMOTE_ADDR'], $validIps)) {
        error_log('Payfast ITN: Invalid IP ' . $_SERVER['REMOTE_ADDR']);
        $pfValid = false;
    }

    // Step 2: Verify data has not been tampered with
    $receivedSignature = $postData['signature'];
    unset($postData['signature']);
    
    // Regenerate signature
    $signature = generateSignature($postData, $payfastSettings['passphrase']);
    
    if ($signature !== $receivedSignature) {
        error_log('Payfast ITN: Invalid signature');
        $pfValid = false;
    }

    // Step 3: Verify the payment status
    if ($postData['payment_status'] !== 'COMPLETE') {
        error_log('Payfast ITN: Payment not complete');
        $pfValid = false;
    }

    // Step 4: Verify amount
    $itemType = $postData['custom_str1'];
    $itemId = intval($postData['custom_int1']);
    $amount = false;
    
    if ($itemType === 'video') {
        // Get video price
        $stmt = $GLOBALS['conn']->prepare("SELECT price FROM videos WHERE video_id = ?");
        $stmt->bind_param("i", $itemId);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result->num_rows === 1) {
            $row = $result->fetch_assoc();
            $amount = $row['price'];
        }
    } elseif ($itemType === 'course') {
        // Get course price
        $stmt = $GLOBALS['conn']->prepare("SELECT price FROM courses WHERE course_id = ?");
        $stmt->bind_param("i", $itemId);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result->num_rows === 1) {
            $row = $result->fetch_assoc();
            $amount = $row['price'];
        }
    }
    
    // Verify amount matches
    if ($amount !== false && number_format($amount, 2, '.', '') !== number_format($postData['amount_gross'], 2, '.', '')) {
        error_log('Payfast ITN: Amount mismatch');
        $pfValid = false;
    }
    
    return $pfValid;
}

/**
 * Process successful payment
 * 
 * @param array $postData POST data from PayFast
 * @return bool Success status
 */
function processSuccessfulPayment($postData) {
    global $conn;
    
    $itemType = $postData['custom_str1'];
    $itemId = intval($postData['custom_int1']);
    $userId = intval($postData['custom_int2']);
    $paymentId = $postData['pf_payment_id'];
    $amount = $postData['amount_gross'];
    
    try {
        if ($itemType === 'video') {
            // Insert video purchase record
            $stmt = $conn->prepare("
                INSERT INTO video_purchases (user_id, video_id, amount, payment_id, payment_status)
                VALUES (?, ?, ?, ?, 'completed')
                ON DUPLICATE KEY UPDATE payment_status = 'completed', payment_id = ?, amount = ?
            ");
            $stmt->bind_param("iidsss", $userId, $itemId, $amount, $paymentId, $paymentId, $amount);
            $stmt->execute();
            
            // Update video views
            $stmt = $conn->prepare("UPDATE videos SET views = views + 1 WHERE video_id = ?");
            $stmt->bind_param("i", $itemId);
            $stmt->execute();
            
            return true;
        } elseif ($itemType === 'course') {
            // Insert course purchase record
            $stmt = $conn->prepare("
                INSERT INTO course_purchases (user_id, course_id, amount, payment_id, payment_status)
                VALUES (?, ?, ?, ?, 'completed')
                ON DUPLICATE KEY UPDATE payment_status = 'completed', payment_id = ?, amount = ?
            ");
            $stmt->bind_param("iidsss", $userId, $itemId, $amount, $paymentId, $paymentId, $amount);
            $stmt->execute();
            
            // Create enrollment record if it doesn't exist
            $stmt = $conn->prepare("
                INSERT IGNORE INTO enrollments (student_id, course_id)
                VALUES (?, ?)
            ");
            $stmt->bind_param("ii", $userId, $itemId);
            $stmt->execute();
            
            // Send purchase confirmation email - USING EXACT SAME APPROACH AS OTHER NOTIFICATIONS
            $email_result = sendPurchaseConfirmationEmail($userId, $itemId, $paymentId, $amount);
            error_log("Purchase confirmation email sent: " . ($email_result ? "SUCCESS" : "FAILED") . " for course $itemId to user $userId");
            
            return true;
        } elseif ($itemType === 'accreditation') {
            // Handle accreditation payment
            $stmt = $conn->prepare("
                UPDATE accreditation_requests 
                SET payment_status = 'completed',
                    payment_date = NOW(),
                    status = 'under_review',
                    updated_at = NOW()
                WHERE request_id = ? AND student_id = ?
            ");
            $stmt->bind_param("ii", $itemId, $userId);
            $stmt->execute();
            
            if ($stmt->affected_rows > 0) {
                error_log("Accreditation payment completed: Request ID $itemId, User $userId, Payment $paymentId");
                
                // TODO: Send accreditation payment confirmation email
                // $email_result = sendAccreditationConfirmationEmail($userId, $itemId, $paymentId, $amount);
                
                return true;
            } else {
                error_log("Failed to update accreditation request: Request ID $itemId, User $userId");
                return false;
            }
        }
    } catch (Exception $e) {
        error_log('Payfast Processing Error: ' . $e->getMessage());
        return false;
    }
    
    return false;
}

/**
 * Check if user has purchased a video
 * 
 * @param int $userId User ID
 * @param int $videoId Video ID
 * @return bool True if purchased
 */
function hasUserPurchasedVideo($userId, $videoId) {
    global $conn;
    
    // Check if video is free
    $stmt = $conn->prepare("SELECT is_free FROM videos WHERE video_id = ?");
    $stmt->bind_param("i", $videoId);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows === 1) {
        $row = $result->fetch_assoc();
        if ($row['is_free']) {
            return true; // Video is free, so user can watch
        }
    }
    
    // Check if user has purchased the video directly
    $stmt = $conn->prepare("
        SELECT * FROM video_purchases 
        WHERE user_id = ? AND video_id = ? AND payment_status = 'completed'
    ");
    $stmt->bind_param("ii", $userId, $videoId);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows > 0) {
        return true;
    }
    
    // Check if user has purchased the course containing this video
    $stmt = $conn->prepare("
        SELECT cp.* FROM course_purchases cp
        JOIN videos v ON v.course_id = cp.course_id
        WHERE cp.user_id = ? AND v.video_id = ? AND cp.payment_status = 'completed'
    ");
    $stmt->bind_param("ii", $userId, $videoId);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows > 0) {
        return true;
    }
    
    return false;
}

/**
 * Check if user has purchased a course
 * 
 * @param int $userId User ID
 * @param int $courseId Course ID
 * @return bool True if purchased
 */
function hasUserPurchasedCourse($userId, $courseId) {
    global $conn;
    
    $stmt = $conn->prepare("
        SELECT * FROM course_purchases 
        WHERE user_id = ? AND course_id = ? AND payment_status = 'completed'
    ");
    $stmt->bind_param("ii", $userId, $courseId);
    $stmt->execute();
    $result = $stmt->get_result();
    
    return $result->num_rows > 0;
}
?>
