<?php
require_once __DIR__ . '/functions.php';
require_once __DIR__ . '/../config/email_verification_service.php';
require_once __DIR__ . '/../includes/referral_processor.php';

// Handle cart/checkout redirect parameter
$redirect = '';
if (isset($_GET['redirect_to'])) {
    if ($_GET['redirect_to'] === 'cart') {
        $redirect = '/student/cart.php?cart_redirect=1';
    } else if ($_GET['redirect_to'] === 'student/checkout.php') {
        $redirect = '/student/checkout.php?from_guest_cart=1';
    } else {
        $redirect = '/' . $_GET['redirect_to'];
    }
}

// Check if already logged in
if (isLoggedIn()) {
    // If coming from cart, redirect to cart page
    if ($redirect) {
        header('Location: ' . $redirect);
        exit();
    }
    
    // Redirect based on user role
    $role = $_SESSION['user_role'];
    if ($role === 'admin') {
        header('Location: /admin/dashboard.php');
    } else if ($role === 'teacher') {
        header('Location: /teacher/dashboard.php');
    } else {
        header('Location: /student/dashboard.php');
    }
    exit();
}

$error = '';
$success = '';
$pendingVerification = false;

// Handle referral code from URL parameter
$referralCode = isset($_GET['ref']) ? htmlspecialchars($_GET['ref']) : '';

// Include database connection for referral processor
require_once __DIR__ . '/../includes/db_connection.php';
$referralProcessor = new ReferralProcessor($conn);

// Handle registration form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $username = trim($_POST['username']);
    $email = trim($_POST['email']);
    $password = $_POST['password'];
    $confirmPassword = $_POST['confirm_password'];
    $firstName = trim($_POST['first_name']);
    $lastName = trim($_POST['last_name']);
    $role = $_POST['role']; // student or teacher
    $referralCodeInput = trim($_POST['referral_code'] ?? '');
    
    // Validate input
    if (empty($username) || empty($email) || empty($password) || empty($firstName) || empty($lastName)) {
        $error = 'All fields are required';
    } elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $error = 'Invalid email format';
    } elseif ($password !== $confirmPassword) {
        $error = 'Passwords do not match';
    } elseif (strlen($password) < 6) {
        $error = 'Password must be at least 6 characters long';
    } else {
        // Check if username or email already exists
        $checkResult = checkExistingUser($username, $email);
        
        if (!$checkResult['success']) {
            $error = $checkResult['message'];
        } else {
            // Validate referral code if provided using our secure validation system
            $validReferral = null;
            if (!empty($referralCodeInput)) {
                try {
                    // Include secure referral generator for validation
                    require_once __DIR__ . '/../includes/secure_referral_generator.php';
                    
                    // Try secure referral code validation first
                    if (preg_match('/^(TCH|STU)[A-F0-9]{12}$/', $referralCodeInput)) {
                        $validReferral = SecureReferralGenerator::validateSecureCode($referralCodeInput, $conn);
                    }
                    
                    // If not secure format, try legacy validation
                    if (!$validReferral) {
                        $validReferral = LegacyReferralValidator::validateLegacyCode($referralCodeInput, $conn);
                    }
                    
                    if (!$validReferral) {
                        $error = 'Invalid or expired referral code';
                    }
                } catch (Exception $e) {
                    error_log("Referral validation error during registration: " . $e->getMessage());
                    $error = 'Error validating referral code';
                }
            }
            
            // Collect additional data based on role
            $additionalData = null;
            if ($role === 'teacher') {
                $additionalData = [
                    'course_title' => trim($_POST['course_title'] ?? ''),
                    'course_description' => trim($_POST['course_description'] ?? ''),
                    'qualifications' => trim($_POST['qualifications'] ?? '')
                ];
            } elseif ($role === 'student') {
                // Collect all student-specific fields
                $additionalData = [
                    'title' => trim($_POST['title'] ?? ''),
                    'cell_number' => trim($_POST['cell_number'] ?? ''),
                    'id_number' => trim($_POST['id_number'] ?? ''),
                    'suburb' => trim($_POST['suburb'] ?? ''),
                    'province' => trim($_POST['province'] ?? ''),
                    'communication_method' => trim($_POST['communication_method'] ?? ''),
                    'owns_laptop' => trim($_POST['owns_laptop'] ?? ''),
                    'training_preference' => trim($_POST['training_preference'] ?? ''),
                    'job_field_interest' => trim($_POST['job_field_interest'] ?? ''),
                    'job_field_other' => trim($_POST['job_field_other'] ?? ''),
                    'education_status' => trim($_POST['education_status'] ?? ''),
                    'employment_status' => trim($_POST['employment_status'] ?? ''),
                    'heard_about_us' => trim($_POST['heard_about_us'] ?? ''),
                    'referral_person_name' => trim($_POST['referral_person_name'] ?? '')
                ];
            }
            
            // Store pending registration (with referral code if valid)
            if (empty($error)) {
                // Ensure email verification service is loaded
                if (!function_exists('storePendingRegistration')) {
                    require_once __DIR__ . '/../config/email_verification_service.php';
                }
                
                // Log registration attempt for debugging
                error_log("REGISTRATION ATTEMPT: Email={$email}, Referral={$referralCodeInput}, Role={$role}");
                
                $result = storePendingRegistration($username, $email, $password, $firstName, $lastName, $role, $additionalData, $referralCodeInput);
                
                if ($result['success']) {
                    error_log("PENDING REGISTRATION SUCCESS: Email={$email}, VerificationID={$result['verification_id']}");
                    
                    // Send verification email
                    $emailResult = sendVerificationEmail($email, $firstName, $lastName, $result['token']);
                    
                    if ($emailResult['success']) {
                        $referralMessage = '';
                        if (!empty($referralCodeInput) && $validReferral) {
                            $referralMessage = ' Your referral will be processed once your account is verified.';
                        }
                        $success = 'Registration initiated! Please check your email and click the verification link to complete your account setup.' . $referralMessage;
                        $pendingVerification = true;
                        error_log("EMAIL VERIFICATION SENT: Email={$email}");
                    } else {
                        $error = 'Registration failed: ' . $emailResult['message'];
                        error_log("EMAIL VERIFICATION FAILED: Email={$email}, Error={$emailResult['message']}");
                    }
                } else {
                    error_log("PENDING REGISTRATION FAILED: Email={$email}, Error={$result['message']}");
                    
                    // FALLBACK: Create user directly with referral processing
                    error_log("ATTEMPTING DIRECT REGISTRATION FALLBACK: Email={$email}");
                    
                    try {
                        // Include database connection and functions
                        require_once __DIR__ . '/functions.php';
                        require_once __DIR__ . '/../includes/referral_processor.php';
                        require_once __DIR__ . '/../includes/db_connection.php';
                        
                        // Create user directly
                        $directResult = registerUser($username, $email, $password, $firstName, $lastName, $role);
                        
                        if ($directResult['success']) {
                            $userId = $directResult['user_id'];
                            error_log("DIRECT USER CREATION SUCCESS: UserID={$userId}");
                            
                            // Process referral if provided
                            if (!empty($referralCodeInput) && $validReferral) {
                                $referralProcessor = new ReferralProcessor($conn);
                                $referralResult = $referralProcessor->processReferralRegistration($referralCodeInput, $userId, $email);
                                
                                if ($referralResult['success']) {
                                    error_log("REFERRAL PROCESSING SUCCESS: UserID={$userId}, Code={$referralCodeInput}");
                                } else {
                                    error_log("REFERRAL PROCESSING FAILED: UserID={$userId}, Code={$referralCodeInput}, Error={$referralResult['message']}");
                                }
                            }
                            
                            // Insert student-specific data
                            if ($role === 'student' && !empty($additionalData)) {
                                $updateSql = "UPDATE users SET ";
                                $updateFields = [];
                                $updateValues = [];
                                
                                foreach ($additionalData as $field => $value) {
                                    if (!empty($value)) {
                                        $updateFields[] = "{$field} = ?";
                                        $updateValues[] = $value;
                                    }
                                }
                                
                                if (!empty($updateFields)) {
                                    $updateSql .= implode(', ', $updateFields) . " WHERE user_id = ?";
                                    $updateValues[] = $userId;
                                    
                                    $updateStmt = $conn->prepare($updateSql);
                                    $updateStmt->execute($updateValues);
                                    error_log("STUDENT DATA UPDATED: UserID={$userId}");
                                }
                            }
                            
                            // Log in the user immediately
                            $_SESSION['user_id'] = $userId;
                            $_SESSION['username'] = $username;
                            $_SESSION['user_role'] = $role;
                            $_SESSION['first_name'] = $firstName;
                            $_SESSION['last_name'] = $lastName;
                            
                            $success = 'Registration successful! Welcome to Panadite Academy.';
                            if (!empty($referralCodeInput)) {
                                $success .= ' Your referral has been processed.';
                            }
                            
                            // Redirect based on role and original destination
                            if ($redirect) {
                                header('Location: ' . $redirect);
                                exit();
                            } else {
                                $dashboardUrl = ($role === 'teacher') ? '/teacher/dashboard.php' : '/student/dashboard.php';
                                header('Location: ' . $dashboardUrl);
                                exit();
                            }
                            
                        } else {
                            $error = 'Registration failed: ' . $directResult['message'];
                            error_log("DIRECT REGISTRATION FAILED: Email={$email}, Error={$directResult['message']}");
                        }
                        
                    } catch (Exception $e) {
                        $error = 'Registration failed due to technical issues. Please try again.';
                        error_log("FALLBACK REGISTRATION EXCEPTION: Email={$email}, Error={$e->getMessage()}");
                    }
                }
            }
        }
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Register - Panadite Academy</title>
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link rel="stylesheet" href="/assets/css/style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/animate.css/4.1.1/animate.min.css">
    <style>
        body {
            background-color: #f8f9fa;
            position: relative;
            overflow-x: hidden;
        }
        
        /* Floating Education Icons */
        .floating-icon {
            position: absolute;
            opacity: 0.05; /* Reduced opacity to make them much lighter */
            z-index: 0;
            pointer-events: none;
            color: #276BE4;
            filter: drop-shadow(0 3px 5px rgba(39, 107, 228, 0.1)); /* Light shadow */
        }
        
        .register-container {
            max-width: 600px;
            margin: 40px auto;
            padding: 25px;
            background-color: #fff;
            border-radius: 10px;
            box-shadow: 0 0 20px rgba(0,0,0,0.1);
        }
        .logo-container {
            text-align: center;
            margin-bottom: 20px;
        }
        .logo-container img {
            max-width: 180px;
        }
        .register-form .form-control {
            height: 45px;
            border-radius: 5px;
            box-shadow: none;
            margin-bottom: 15px;
        }
        .btn-register {
            background-color: #276BE4;
            border-color: #276BE4;
            color: #fff;
            height: 45px;
            border-radius: 5px;
            font-weight: 600;
        }
        .btn-register:hover {
            background-color: #1c54b8;
        }
        .register-links {
            text-align: center;
            margin-top: 20px;
        }
        .register-links a {
            color: #276BE4;
            text-decoration: none;
        }
        .register-links a:hover {
            text-decoration: underline;
        }
        .tagline {
            text-align: center;
            color: #FFA915;
            font-style: italic;
            margin-bottom: 20px;
        }
        .role-selector {
            display: flex;
            justify-content: center;
            margin-bottom: 20px;
        }
        .role-option {
            flex: 1;
            text-align: center;
            padding: 15px;
            border: 2px solid #e9ecef;
            border-radius: 5px;
            margin: 0 10px;
            cursor: pointer;
            transition: all 0.3s;
        }
        .role-option:hover {
            border-color: #FFA915;
        }
        .role-option.active {
            border-color: #276BE4;
            background-color: rgba(39, 107, 228, 0.1);
        }
        .role-option i {
            font-size: 24px;
            margin-bottom: 10px;
            color: #276BE4;
        }
        
        /* Password strength styles */
        .password-strength-meter {
            height: 5px;
            background-color: #e9ecef;
            margin-top: -10px;
            margin-bottom: 15px;
            border-radius: 3px;
            overflow: hidden;
        }
        
        .password-strength-meter-fill {
            height: 100%;
            border-radius: 3px;
            transition: width 0.3s ease, background-color 0.3s ease;
            width: 0%;
        }
        
        .strength-weak { background-color: #ff4d4d; }
        .strength-medium { background-color: #ffa500; }
        .strength-good { background-color: #2ecc71; }
        
        /* Password toggles */
        .password-container {
            position: relative;
        }
        
        .toggle-password {
            position: absolute;
            right: 10px;
            top: 13px;
            cursor: pointer;
            color: #6c757d;
        }
        
        /* Input field hover effect */
        .register-form .form-control:focus {
            border-color: #276BE4;
            box-shadow: 0 0 5px rgba(39, 107, 228, 0.2);
        }
        
        /* Animation styles */
        .fade-in {
            animation: fadeIn 0.8s;
        }
        
        .slide-up {
            animation: slideInUp 0.5s;
        }
        
        .logo-wrapper {
            position: relative;
            height: 120px;
            display: flex;
            justify-content: center;
            align-items: center;
            overflow: hidden;
        }
        
        .logo-wrapper img {
            max-height: 100%;
            max-width: 100%;
            object-fit: contain;
        }
        
        /* Referrer Display Styling */
        .referrer-info {
            opacity: 0;
            animation: fadeInUp 0.5s ease-out forwards;
        }
        
        .referrer-card {
            background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);
            border: 1px solid #28a745;
            border-radius: 8px;
            padding: 12px 15px;
            display: flex;
            flex-direction: column;
            gap: 4px;
        }
        
        .referrer-text {
            display: flex;
            align-items: center;
            font-size: 14px;
            color: #495057;
        }
        
        .referrer-text strong {
            color: #28a745;
            font-weight: 600;
        }
        
        .reward-info {
            padding-left: 20px;
        }
        
        .reward-info small {
            color: #6c757d;
            font-size: 12px;
        }
        
        @keyframes fadeInUp {
            from {
                opacity: 0;
                transform: translateY(10px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }
        
        #referral-loading .fa-spin {
            animation: spin 1s linear infinite;
        }
        
        @keyframes spin {
            from { transform: rotate(0deg); }
            to { transform: rotate(360deg); }
        }
    </style>
</head>
<body>
    <div class="floating-icon" style="top: 10%; left: 10%; font-size: 40px;"><i class="fas fa-book"></i></div>
    <div class="floating-icon" style="top: 20%; right: 15%; font-size: 35px;"><i class="fas fa-graduation-cap"></i></div>
    <div class="floating-icon" style="top: 60%; left: 5%; font-size: 45px;"><i class="fas fa-pencil-alt"></i></div>
    <div class="floating-icon" style="bottom: 20%; right: 10%; font-size: 50px;"><i class="fas fa-lightbulb"></i></div>
    <div class="floating-icon" style="bottom: 35%; left: 40%; font-size: 30px;"><i class="fas fa-brain"></i></div>
    
    <!-- Include the modern navbar -->
    <?php include '../includes/navbar.php'; ?>
    
    <div class="container">
        <div class="register-container animate__animated animate__fadeIn">
            <div class="logo-container">
                <div class="logo-wrapper">
                    <img src="/assets/images/logo.jpeg" alt="Panadite Academy Logo" class="animate__animated animate__zoomIn">
                </div>
                <p class="tagline animate__animated animate__fadeIn animate__delay-1s">Get ahead of the curve</p>
            </div>
            
            <?php if ($error): ?>
            <div class="alert alert-danger animate__animated animate__shakeX" role="alert">
                <i class="bx bx-error-circle me-2"></i>
                <?php echo $error; ?>
            </div>
            <?php endif; ?>
            
            <?php if ($success && $pendingVerification): ?>
            <div class="alert alert-success animate__animated animate__bounceIn" role="alert">
                <div class="d-flex align-items-start">
                    <i class="bx bx-check-circle me-2 fs-4 text-success"></i>
                    <div class="flex-grow-1">
                        <h6 class="alert-heading mb-2">
                            <i class="bx bx-envelope"></i> Check Your Email!
                        </h6>
                        <p class="mb-2"><?php echo $success; ?></p>
                        <div class="verification-instructions mt-3 p-3" style="background: rgba(16, 185, 129, 0.1); border-radius: 8px; border-left: 4px solid #10b981;">
                            <h6 class="text-success mb-2">
                                <i class="bx bx-info-circle"></i> What's Next?
                            </h6>
                            <ol class="mb-2 small">
                                <li>Check your email inbox (and spam folder)</li>
                                <li>Click the verification link in the email</li>
                                <li>Your account will be created automatically</li>
                                <li>Login with your credentials</li>
                            </ol>
                            <div class="mt-3">
                                <small class="text-muted">
                                    <i class="bx bx-time"></i> Didn't receive the email? 
                                    <a href="/resend-verification.php" class="text-decoration-none">Resend verification email</a>
                                </small>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            
            <div class="text-center mt-3">
                <a href="/auth/login.php" class="btn btn-outline-primary">
                    <i class="bx bx-log-in"></i> Already verified? Login here
                </a>
            </div>
            
            <?php elseif ($success): ?>
            <div class="alert alert-success animate__animated animate__bounceIn" role="alert">
                <i class="bx bx-check-circle me-2"></i>
                <?php echo $success; ?>
            </div>
            <?php endif; ?>
            
            <?php if (!$pendingVerification): ?>
            <form class="register-form" method="POST" action="">
                <div class="role-selector">
                    <div class="role-option active" data-role="student">
                        <i class="fas fa-user-graduate"></i>
                        <h5>Student</h5>
                        <p class="small">Learn from our courses</p>
                    </div>
                    <div class="role-option" data-role="teacher">
                        <i class="fas fa-chalkboard-teacher"></i>
                        <h5>Facilitator</h5>
                        <p class="small">Create and teach courses</p>
                    </div>
                </div>
                <input type="hidden" name="role" id="role" value="student">
                
                <div class="row">
                    <div class="col-md-6">
                        <div class="mb-3">
                            <label for="first_name" class="form-label">First Name</label>
                            <input type="text" class="form-control" id="first_name" name="first_name" required>
                        </div>
                    </div>
                    <div class="col-md-6">
                        <div class="mb-3">
                            <label for="last_name" class="form-label">Last Name</label>
                            <input type="text" class="form-control" id="last_name" name="last_name" required>
                        </div>
                    </div>
                </div>
                
                <div class="mb-3">
                    <label for="username" class="form-label">Username</label>
                    <input type="text" class="form-control" id="username" name="username" required>
                </div>
                
                <div class="mb-3">
                    <label for="email" class="form-label">Email</label>
                    <input type="email" class="form-control" id="email" name="email" required>
                </div>
                
                <div class="mb-3">
                    <label for="referral_code" class="form-label">Referral Code <span class="text-muted">(Optional)</span></label>
                    <input type="text" class="form-control" id="referral_code" name="referral_code" 
                           value="<?= htmlspecialchars($referralCode) ?>" 
                           placeholder="Enter referral code if you have one"
                           oninput="validateReferralCode(this.value)">
                    
                    <!-- Dynamic Referrer Display -->
                    <div id="referrer-info" class="referrer-info mt-2" style="display: none;">
                        <div class="referrer-card">
                            <i class="fas fa-user-check text-success me-2"></i>
                            <span class="referrer-text">
                                Referred by: <strong id="referrer-name"></strong> 
                                <span class="badge bg-primary ms-1" id="referrer-role"></span>
                            </span>
                            <div class="reward-info mt-1">
                                <i class="fas fa-gift text-warning me-1"></i>
                                <small id="reward-text" class="text-muted"></small>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Loading/Error States -->
                    <div id="referral-loading" class="mt-2" style="display: none;">
                        <small class="text-muted">
                            <i class="fas fa-spinner fa-spin me-1"></i> Validating referral code...
                        </small>
                    </div>
                    <div id="referral-error" class="mt-2" style="display: none;">
                        <small class="text-danger">
                            <i class="fas fa-exclamation-triangle me-1"></i>
                            <span id="error-message"></span>
                        </small>
                    </div>
                    
                    <div class="form-text">
                        <i class="fas fa-gift text-primary"></i> Get rewards when you use a referral code!
                    </div>
                </div>
                
                <div class="mb-3">
                    <label for="password" class="form-label">Password</label>
                    <div class="password-container">
                        <input type="password" class="form-control" id="password" name="password" required>
                        <span class="toggle-password" data-target="password">
                            <i class="far fa-eye"></i>
                        </span>
                    </div>
                    <div class="password-strength-meter mt-2">
                        <div class="password-strength-meter-fill"></div>
                    </div>
                    <small id="passwordHelp" class="form-text text-muted">Use at least 8 characters with letters, numbers and symbols</small>
                </div>
                
                <div class="mb-3">
                    <label for="confirm_password" class="form-label">Confirm Password</label>
                    <div class="password-container">
                        <input type="password" class="form-control" id="confirm_password" name="confirm_password" required>
                        <span class="toggle-password" data-target="confirm_password">
                            <i class="far fa-eye"></i>
                        </span>
                    </div>
                    <small id="passwordMatch" class="form-text"></small>
                </div>
                
                <!-- Student specific fields - shown by default -->
                <div id="student-fields" class="border rounded p-3 mb-3 bg-light">
                    <h5 class="mb-3"><i class="fas fa-user-graduate me-2"></i>Student Information</h5>
                    
                    <!-- Title -->
                    <div class="mb-3">
                        <label for="title" class="form-label">Title</label>
                        <select class="form-select" id="title" name="title" required>
                            <option value="">Select Title</option>
                            <option value="Mr">Mr</option>
                            <option value="Mrs">Mrs</option>
                            <option value="Miss">Miss</option>
                            <option value="Other">Other</option>
                        </select>
                    </div>
                    
                    <!-- Contact Information Row -->
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="cell_number" class="form-label">Cell Number</label>
                                <input type="tel" class="form-control" id="cell_number" name="cell_number" 
                                       placeholder="e.g., 0821234567" required>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="id_number" class="form-label">ID Number</label>
                                <input type="text" class="form-control" id="id_number" name="id_number" 
                                       placeholder="Your South African ID number" required>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Location Information Row -->
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="suburb" class="form-label">Suburb</label>
                                <input type="text" class="form-control" id="suburb" name="suburb" 
                                       placeholder="e.g., Midrand" required>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="province" class="form-label">Province</label>
                                <select class="form-select" id="province" name="province" required>
                                    <option value="">Select Province</option>
                                    <option value="Eastern Cape">Eastern Cape</option>
                                    <option value="Free State">Free State</option>
                                    <option value="Gauteng">Gauteng</option>
                                    <option value="KwaZulu-Natal">KwaZulu-Natal</option>
                                    <option value="Limpopo">Limpopo</option>
                                    <option value="Mpumalanga">Mpumalanga</option>
                                    <option value="Northern Cape">Northern Cape</option>
                                    <option value="North West">North West</option>
                                    <option value="Western Cape">Western Cape</option>
                                    <option value="Other">Other</option>
                                </select>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Communication Preferences -->
                    <div class="mb-3">
                        <label class="form-label">Which communication method is the fastest way to reach you?</label>
                        <div class="mt-2">
                            <div class="form-check form-check-inline">
                                <input class="form-check-input" type="radio" name="communication_method" 
                                       id="comm_whatsapp" value="WhatsApp" required>
                                <label class="form-check-label" for="comm_whatsapp">
                                    <i class="fab fa-whatsapp text-success me-1"></i> WhatsApp
                                </label>
                            </div>
                            <div class="form-check form-check-inline">
                                <input class="form-check-input" type="radio" name="communication_method" 
                                       id="comm_email" value="Email" required>
                                <label class="form-check-label" for="comm_email">
                                    <i class="fas fa-envelope text-primary me-1"></i> Email
                                </label>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Technology Access -->
                    <div class="mb-3">
                        <label class="form-label">Do you own a laptop/PC?</label>
                        <div class="mt-2">
                            <div class="form-check form-check-inline">
                                <input class="form-check-input" type="radio" name="owns_laptop" 
                                       id="laptop_yes" value="Yes" required>
                                <label class="form-check-label" for="laptop_yes">
                                    <i class="fas fa-laptop text-success me-1"></i> Yes
                                </label>
                            </div>
                            <div class="form-check form-check-inline">
                                <input class="form-check-input" type="radio" name="owns_laptop" 
                                       id="laptop_no" value="No" required>
                                <label class="form-check-label" for="laptop_no">
                                    <i class="fas fa-times text-danger me-1"></i> No
                                </label>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Training Preference -->
                    <div class="mb-3">
                        <label class="form-label">How would you like to attend the training?</label>
                        <div class="mt-2">
                            <div class="form-check form-check-inline">
                                <input class="form-check-input" type="radio" name="training_preference" 
                                       id="training_onsite" value="Onsite" required>
                                <label class="form-check-label" for="training_onsite">
                                    <i class="fas fa-building text-primary me-1"></i> Onsite
                                </label>
                            </div>
                            <div class="form-check form-check-inline">
                                <input class="form-check-input" type="radio" name="training_preference" 
                                       id="training_online" value="Online" required>
                                <label class="form-check-label" for="training_online">
                                    <i class="fas fa-wifi text-success me-1"></i> Online
                                </label>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Job Field Interest -->
                    <div class="mb-3">
                        <label for="job_field_interest" class="form-label">Job Field Interest</label>
                        <select class="form-select" id="job_field_interest" name="job_field_interest" required>
                            <option value="">Select Job Field</option>
                            <option value="Data Analysis">Data Analysis</option>
                            <option value="Data Capturing">Data Capturing</option>
                            <option value="Data Clerk">Data Clerk</option>
                            <option value="Administration">Administration</option>
                            <option value="Receptionist">Receptionist</option>
                            <option value="Librarian">Librarian</option>
                            <option value="Front-end Web Developer">Front-end Web Developer</option>
                            <option value="Other">Other</option>
                        </select>
                        <input type="text" class="form-control mt-2" id="job_field_other" name="job_field_other" 
                               placeholder="Please specify if 'Other'" style="display: none;">
                    </div>
                    
                    <!-- Education Status -->
                    <div class="mb-3">
                        <label for="education_status" class="form-label">Highest Education Status</label>
                        <select class="form-select" id="education_status" name="education_status" required>
                            <option value="">Select Education Level</option>
                            <option value="University Degree/Diploma">University Degree/Diploma</option>
                            <option value="TVET NATED">TVET NATED</option>
                            <option value="TVET National Certificate Vocational">TVET National Certificate Vocational</option>
                            <option value="Higher Certificate">Higher Certificate</option>
                            <option value="Matric">Matric</option>
                            <option value="Certificate Only">Certificate Only</option>
                        </select>
                    </div>
                    
                    <!-- Employment Status -->
                    <div class="mb-3">
                        <label class="form-label">What best describes your current employment status?</label>
                        <div class="mt-2">
                            <div class="form-check">
                                <input class="form-check-input" type="radio" name="employment_status" 
                                       id="unemployed" value="Unemployed and looking for new skills and a job" required>
                                <label class="form-check-label" for="unemployed">
                                    Unemployed and looking for new skills and a job
                                </label>
                            </div>
                            <div class="form-check">
                                <input class="form-check-input" type="radio" name="employment_status" 
                                       id="employed_same" value="Employed and want to be equipped with new skills for the same job" required>
                                <label class="form-check-label" for="employed_same">
                                    Employed and want to be equipped with new skills for the same job
                                </label>
                            </div>
                            <div class="form-check">
                                <input class="form-check-input" type="radio" name="employment_status" 
                                       id="employed_new" value="Employed and want to acquire new skills and get a new job" required>
                                <label class="form-check-label" for="employed_new">
                                    Employed and want to acquire new skills and get a new job
                                </label>
                            </div>
                        </div>
                    </div>
                    
                    <!-- How did you hear about us -->
                    <div class="mb-3">
                        <label for="heard_about_us" class="form-label">How did you hear about us?</label>
                        <select class="form-select" id="heard_about_us" name="heard_about_us" required>
                            <option value="">Select Option</option>
                            <option value="Referral">Referral</option>
                            <option value="LinkedIn">LinkedIn</option>
                            <option value="Facebook">Facebook</option>
                            <option value="TikTok">TikTok</option>
                            <option value="Instagram">Instagram</option>
                            <option value="Twitter">Twitter</option>
                            <option value="Brochure handout">Brochure handout</option>
                            <option value="WhatsApp Status">WhatsApp Status</option>
                            <option value="Other">Other</option>
                        </select>
                        <input type="text" class="form-control mt-2" id="referral_person_name" name="referral_person_name" 
                               placeholder="Name of person who referred you" style="display: none;">
                    </div>
                </div>

                <!-- Facilitator specific fields - hidden by default -->
                <div id="facilitator-fields" style="display: none;" class="border rounded p-3 mb-3 bg-light">
                    <h5 class="mb-3">Facilitator Information</h5>
                    
                    <div class="mb-3">
                        <label for="course_title" class="form-label">Course Title</label>
                        <input type="text" class="form-control" id="course_title" name="course_title" placeholder="What course do you want to teach?">
                    </div>
                    
                    <div class="mb-3">
                        <label for="course_description" class="form-label">Course Description</label>
                        <textarea class="form-control" id="course_description" name="course_description" rows="3" placeholder="Brief description of your proposed course"></textarea>
                    </div>
                    
                    <div class="mb-3">
                        <label for="qualifications" class="form-label">Your Qualifications</label>
                        <textarea class="form-control" id="qualifications" name="qualifications" rows="3" placeholder="Your qualifications related to teaching this course"></textarea>
                    </div>
                    
                    <div class="alert alert-info" role="alert">
                        <i class="fas fa-info-circle me-2"></i> Note: All facilitator applications require approval from our admin team before you can upload content.
                    </div>
                </div>
                
                <div class="d-grid gap-2">
                    <button type="submit" class="btn btn-register">Register</button>
                </div>
            </form>
            
            <div class="register-links">
                <a href="login.php">Already have an account? Login</a><br>
                <a href="/index.php">Back to Home</a>
            </div>
            <?php endif; ?>
        </div>
    </div>



    <footer class="mt-5 position-relative">
        <!-- Top gradient bar -->
        <div class="position-absolute top-0 start-0 w-100" style="height: 4px; background: linear-gradient(90deg, #276BE4, #FFA915);"></div>
        
        <!-- Main footer content -->
        <div class="bg-white py-5">
            <div class="container">
                <div class="row gy-4">
                    <!-- Logo and description -->
                    <div class="col-lg-4 col-md-6 mb-4">
                        <div class="mb-4">
                            <img src="/assets/images/logo.jpeg" alt="Panadite Academy" style="height: 50px;">
                        </div>
                        <p class="mb-2" style="color: #333; font-weight: 500;">Empowering learners through high-quality educational content.</p>
                        <p style="color: #333;">Get ahead of the curve with our premium videos and courses.</p>
                    </div>
                    
                    <!-- Quick links -->
                    <div class="col-lg-2 col-md-6 mb-4">
                        <h5 class="fw-bold mb-3 pb-2 position-relative" style="color: #222;">
                            Quick Links
                            <span class="position-absolute bottom-0 start-0" style="height: 2px; width: 40px; background: #FFA915;"></span>
                        </h5>
                        <ul class="list-unstyled">
                            <li class="mb-2"><a href="/index.php" class="text-decoration-none" style="color: #333;"><i class="fas fa-angle-right me-2" style="color: #276BE4;"></i>Home</a></li>
                            <li class="mb-2"><a href="/courses.php" class="text-decoration-none" style="color: #333;"><i class="fas fa-angle-right me-2" style="color: #276BE4;"></i>Courses</a></li>
                            <li class="mb-2"><a href="/videos.php" class="text-decoration-none" style="color: #333;"><i class="fas fa-angle-right me-2" style="color: #276BE4;"></i>Videos</a></li>
                            <li class="mb-2"><a href="/about.php" class="text-decoration-none" style="color: #333;"><i class="fas fa-angle-right me-2" style="color: #276BE4;"></i>About Us</a></li>
                            <li class="mb-2"><a href="/contact.php" class="text-decoration-none" style="color: #333;"><i class="fas fa-angle-right me-2" style="color: #276BE4;"></i>Contact</a></li>
                        </ul>
                    </div>
                    
                    <!-- Facilitator links -->
                    <div class="col-lg-2 col-md-6 mb-4">
                        <h5 class="fw-bold mb-3 pb-2 position-relative" style="color: #222;">
                            Facilitator
                            <span class="position-absolute bottom-0 start-0" style="height: 2px; width: 40px; background: #FFA915;"></span>
                        </h5>
                        <ul class="list-unstyled">
                            <li class="mb-2"><a href="/auth/register.php?role=teacher" class="text-decoration-none" style="color: #333;"><i class="fas fa-angle-right me-2" style="color: #276BE4;"></i>Become a Facilitator</a></li>
                            <li class="mb-2"><a href="/teacher/dashboard.php" class="text-decoration-none" style="color: #333;"><i class="fas fa-angle-right me-2" style="color: #276BE4;"></i>Teacher Dashboard</a></li>
                            <li class="mb-2"><a href="/teacher/sales.php" class="text-decoration-none" style="color: #333;"><i class="fas fa-angle-right me-2" style="color: #276BE4;"></i>Your Sales</a></li>
                            <li class="mb-2"><a href="/teacher/help.php" class="text-decoration-none" style="color: #333;"><i class="fas fa-angle-right me-2" style="color: #276BE4;"></i>Teacher Help</a></li>
                        </ul>
                    </div>
                    
                    <!-- Contact info -->
                    <div class="col-lg-4 col-md-6">
                        <h5 class="fw-bold mb-3 pb-2 position-relative" style="color: #222;">
                            Contact
                            <span class="position-absolute bottom-0 start-0" style="height: 2px; width: 40px; background: #FFA915;"></span>
                        </h5>
                        <ul class="list-unstyled mb-4">
                            <li class="mb-3 d-flex align-items-center">
                                <span class="d-flex align-items-center justify-content-center me-3" style="width: 36px; height: 36px; background: rgba(39,107,228,0.1); border-radius: 50%;">
                                    <i class="fas fa-map-marker-alt" style="color: #276BE4;"></i>
                                </span>
                                <span style="color: #333;">296 Pretorius Street, Corner Lilian Ngoyi, Pretoria Central, 0001, South Africa</span>
                            </li>
                            <li class="mb-3 d-flex align-items-center">
                                <span class="d-flex align-items-center justify-content-center me-3" style="width: 36px; height: 36px; background: rgba(39,107,228,0.1); border-radius: 50%;">
                                    <i class="fas fa-envelope" style="color: #276BE4;"></i>
                                </span>
                                <a href="mailto:admin@panaditeacademy.com" class="text-decoration-none" style="color: #333;">admin@panaditeacademy.com</a>
                            </li>
                            <li class="mb-3 d-flex align-items-center">
                                <span class="d-flex align-items-center justify-content-center me-3" style="width: 36px; height: 36px; background: rgba(39,107,228,0.1); border-radius: 50%;">
                                    <i class="fas fa-phone" style="color: #276BE4;"></i>
                                </span>
                                <a href="tel:+27747223141" class="text-decoration-none" style="color: #333;">+27 74 722 3141</a>
                            </li>
                        </ul>
                        
                        <!-- Social icons -->
                        <div class="d-flex">
                            <a href="#" class="d-flex align-items-center justify-content-center me-2" style="width: 40px; height: 40px; background: rgba(39,107,228,0.1); border-radius: 50%; color: #276BE4; transition: all 0.3s ease;">
                                <i class="fab fa-facebook-f"></i>
                            </a>
                            <a href="#" class="d-flex align-items-center justify-content-center me-2" style="width: 40px; height: 40px; background: rgba(39,107,228,0.1); border-radius: 50%; color: #276BE4; transition: all 0.3s ease;">
                                <i class="fab fa-twitter"></i>
                            </a>
                            <a href="#" class="d-flex align-items-center justify-content-center me-2" style="width: 40px; height: 40px; background: rgba(39,107,228,0.1); border-radius: 50%; color: #276BE4; transition: all 0.3s ease;">
                                <i class="fab fa-instagram"></i>
                            </a>
                            <a href="#" class="d-flex align-items-center justify-content-center" style="width: 40px; height: 40px; background: rgba(39,107,228,0.1); border-radius: 50%; color: #276BE4; transition: all 0.3s ease;">
                                <i class="fab fa-linkedin-in"></i>
                            </a>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Copyright section -->
        <div style="background: #f8f9fa; border-top: 1px solid #eee;" class="py-3">
            <div class="container">
                <div class="row align-items-center">
                    <div class="col-md-6 mb-3 mb-md-0">
                        <p class="mb-0" style="color: #333; font-weight: 500;">&copy; <?php echo date('Y'); ?> Panadite Academy. All rights reserved.</p>
                    </div>
                    <div class="col-md-6 text-md-end">
                        <a href="/privacy-policy.php" class="text-decoration-none me-3" style="color: #333;">Privacy Policy</a>
                        <a href="/terms.php" class="text-decoration-none me-3" style="color: #333;">Terms of Use</a>
                        <a href="/sitemap.php" class="text-decoration-none" style="color: #333;">Sitemap</a>
                    </div>
                </div>
            </div>
        </div>
    </footer>
    <script src="https://cdn.jsdelivr.net/npm/jquery@3.6.4/dist/jquery.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/zxcvbn/4.4.2/zxcvbn.js"></script>
    <script>
        $(document).ready(function() {
            // Animate floating icons
            animateFloatingIcons();
            
            // Handle role selection with animation
            $('.role-option').on('click', function() {
                $('.role-option').removeClass('active');
                $(this).addClass('active');
                
                const role = $(this).data('role');
                $('#role').val(role);
                
                // Show/hide role-specific fields with animation
                if (role === 'teacher') {
                    $('#student-fields').slideUp(300);
                    $('#facilitator-fields').slideDown(300);
                    // Make student fields not required
                    $('#student-fields input, #student-fields select').prop('required', false);
                    // Make facilitator fields required (if any)
                } else {
                    $('#facilitator-fields').slideUp(300);
                    $('#student-fields').slideDown(300);
                    // Make student fields required
                    $('#student-fields input[required], #student-fields select[required]').prop('required', true);
                }
            });
            
            // Handle conditional "Other" fields
            $('#job_field_interest').on('change', function() {
                if ($(this).val() === 'Other') {
                    $('#job_field_other').show().prop('required', true);
                } else {
                    $('#job_field_other').hide().prop('required', false);
                }
            });
            
            $('#heard_about_us').on('change', function() {
                if ($(this).val() === 'Referral') {
                    $('#referral_person_name').show().prop('required', true);
                } else {
                    $('#referral_person_name').hide().prop('required', false);
                }
            });
            
            // Password strength meter
            $('#password').on('keyup', function() {
                var password = $(this).val();
                var result = zxcvbn(password);
                var $meter = $('.password-strength-meter-fill');
                
                // Update the strength meter
                $meter.removeClass('strength-weak strength-medium strength-good');
                
                if (password.length === 0) {
                    $meter.css('width', '0%');
                } else if (result.score < 2) {
                    $meter.addClass('strength-weak').css('width', '33%');
                } else if (result.score < 4) {
                    $meter.addClass('strength-medium').css('width', '66%');
                } else {
                    $meter.addClass('strength-good').css('width', '100%');
                }
            });
            
            // Password matching validation
            $('#confirm_password').on('keyup', function() {
                var password = $('#password').val();
                var confirmPassword = $(this).val();
                var $feedback = $('#passwordMatch');
                
                if (confirmPassword.length > 0) {
                    if (password === confirmPassword) {
                        $feedback.text('Passwords match').removeClass('text-danger').addClass('text-success');
                    } else {
                        $feedback.text('Passwords do not match').removeClass('text-success').addClass('text-danger');
                    }
                } else {
                    $feedback.text('');
                }
            });
            
            // Toggle password visibility
            $('.toggle-password').click(function() {
                var targetId = $(this).data('target');
                var input = $('#' + targetId);
                var icon = $(this).find('i');
                
                if (input.attr('type') === 'password') {
                    input.attr('type', 'text');
                    icon.removeClass('fa-eye').addClass('fa-eye-slash');
                } else {
                    input.attr('type', 'password');
                    icon.removeClass('fa-eye-slash').addClass('fa-eye');
                }
            });
            
            // Form validation before submit
            $('.register-form').on('submit', function(e) {
                var password = $('#password').val();
                var confirmPassword = $('#confirm_password').val();
                
                if (password !== confirmPassword) {
                    e.preventDefault();
                    $('#passwordMatch').text('Passwords do not match').removeClass('text-success').addClass('text-danger');
                    $('#confirm_password').focus();
                    return false;
                }
                
                if (password.length < 8) {
                    e.preventDefault();
                    $('#passwordHelp').text('Password must be at least 8 characters long').addClass('text-danger');
                    $('#password').focus();
                    return false;
                }
                
                // Add animation to the submit button
                $('.btn-register').addClass('animate__animated animate__pulse');
                return true;
            });
            
            // Custom username validation
            $('#username').on('blur', function() {
                var username = $(this).val();
                if (username && username.length < 5) {
                    $(this).addClass('is-invalid');
                    $(this).after('<div class="invalid-feedback">Username must be at least 5 characters long</div>');
                } else {
                    $(this).removeClass('is-invalid').addClass('is-valid');
                    $(this).next('.invalid-feedback').remove();
                }
            });
        });
        
        // Function to animate the floating icons
        function animateFloatingIcons() {
            $('.floating-icon').each(function(index) {
                var icon = $(this);
                // Set random initial positions
                var startX = Math.random() * 10 - 5; // -5 to +5
                var startY = Math.random() * 10 - 5; // -5 to +5
                
                // Animate with random duration between 5-10 seconds
                var duration = 5000 + Math.random() * 5000;
                
                // Function to create floating animation
                function floatIcon() {
                    // Generate new random positions
                    var moveX = Math.random() * 20 - 10; // -10 to +10
                    var moveY = Math.random() * 20 - 10; // -10 to +10
                    
                    icon.animate({
                        transform: 'translate(' + moveX + 'px, ' + moveY + 'px)'
                    }, duration, 'linear', function() {
                        floatIcon(); // Infinite loop
                    });
                }
                
                // Start with a delay based on index
                setTimeout(function() {
                    icon.css('transform', 'translate(' + startX + 'px, ' + startY + 'px)');
                    floatIcon();
                }, index * 500);
            });
        }
        
        // Referral Code Validation Function
        let referralTimeout;
        function validateReferralCode(code) {
            // Clear previous timeout
            clearTimeout(referralTimeout);
            
            // Hide all states
            hideReferralStates();
            
            // If code is empty, don't validate
            if (!code || code.trim().length === 0) {
                return;
            }
            
            // Show loading state after a short delay (debounce)
            referralTimeout = setTimeout(() => {
                showReferralLoading();
                
                // Make AJAX request to validate referral code
                fetch('validate_referral.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        code: code.trim()
                    })
                })
                .then(response => response.json())
                .then(data => {
                    hideReferralStates();
                    
                    if (data.success) {
                        showReferrerInfo(data.referrer, data.reward);
                    } else {
                        showReferralError(data.message);
                    }
                })
                .catch(error => {
                    hideReferralStates();
                    showReferralError('Error validating referral code');
                    console.error('Referral validation error:', error);
                });
            }, 800); // 800ms debounce
        }
        
        function hideReferralStates() {
            document.getElementById('referrer-info').style.display = 'none';
            document.getElementById('referral-loading').style.display = 'none';
            document.getElementById('referral-error').style.display = 'none';
        }
        
        function showReferralLoading() {
            document.getElementById('referral-loading').style.display = 'block';
        }
        
        function showReferrerInfo(referrer, reward) {
            // Update referrer information
            document.getElementById('referrer-name').textContent = referrer.name;
            document.getElementById('referrer-role').textContent = referrer.role;
            
            // Update reward information
            let rewardText = '';
            if (reward.discount && reward.discount !== '0%' && reward.discount !== '$0.00') {
                rewardText += `Get ${reward.discount} discount`;
            }
            if (reward.points && reward.points > 0) {
                if (rewardText) rewardText += ' + ';
                rewardText += `${reward.points} bonus points`;
            }
            if (!rewardText) {
                rewardText = 'Referral rewards available';
            }
            
            document.getElementById('reward-text').textContent = rewardText;
            
            // Show referrer info with animation
            const referrerInfo = document.getElementById('referrer-info');
            referrerInfo.style.display = 'block';
            referrerInfo.style.opacity = '0';
            
            // Trigger animation
            setTimeout(() => {
                referrerInfo.style.opacity = '1';
            }, 10);
        }
        
        function showReferralError(message) {
            document.getElementById('error-message').textContent = message;
            document.getElementById('referral-error').style.display = 'block';
        }
        
        // Initialize referral validation on page load if code is pre-filled
        $(document).ready(function() {
            const initialCode = $('#referral_code').val();
            if (initialCode && initialCode.trim().length > 0) {
                validateReferralCode(initialCode);
            }
        });
    </script>
</body>
</html>
