<?php
/**
 * Session Trace Diagnostic
 * Upload this to your live server to trace session handling step by step
 */

error_reporting(E_ALL);
ini_set('display_errors', 1);

echo "<h1>Session Trace Diagnostic</h1><pre>";
echo "=== STEP 1: Check File Existence ===\n";

$files_to_check = [
    'includes/session_start.php',
    'includes/advanced_session.php', 
    'includes/site_config.php',
    'auth/functions.php',
    'config/db_connect.php'
];

foreach ($files_to_check as $file) {
    $full_path = __DIR__ . '/' . $file;
    if (file_exists($full_path)) {
        echo "✅ EXISTS: $file (at $full_path)\n";
    } else {
        echo "❌ MISSING: $file (looked at $full_path)\n";
    }
}

echo "\n=== STEP 2: System Information ===\n";
echo "Document Root: " . $_SERVER['DOCUMENT_ROOT'] . "\n";
echo "Current Directory: " . __DIR__ . "\n";
echo "Script Name: " . $_SERVER['SCRIPT_NAME'] . "\n";
echo "sys_get_temp_dir(): " . sys_get_temp_dir() . "\n";
echo "Default session path: " . session_save_path() . "\n";

echo "\n=== STEP 3: Test Session Start Process ===\n";

// Test 1: Direct session start
if (session_status() == PHP_SESSION_NONE) {
    echo "Session not started yet. Testing direct start...\n";
    
    // Try Linux paths first
    $temp_dirs = ['/tmp', '/var/tmp', sys_get_temp_dir()];
    $chosen_path = null;
    
    foreach ($temp_dirs as $temp_dir) {
        echo "Testing directory: $temp_dir\n";
        if (is_dir($temp_dir)) {
            echo "  - Directory exists: YES\n";
            if (is_writable($temp_dir)) {
                echo "  - Directory writable: YES\n";
                $chosen_path = $temp_dir;
                break;
            } else {
                echo "  - Directory writable: NO\n";
            }
        } else {
            echo "  - Directory exists: NO\n";
        }
    }
    
    if ($chosen_path) {
        echo "Setting session path to: $chosen_path\n";
        session_save_path($chosen_path);
        
        // Set session parameters
        session_set_cookie_params([
            'lifetime' => 0,
            'path' => '/',
            'domain' => '',
            'secure' => isset($_SERVER['HTTPS']),
            'httponly' => true,
            'samesite' => 'Lax'
        ]);
        
        echo "Starting session...\n";
        session_start();
        echo "✅ Session started successfully!\n";
        echo "Session ID: " . session_id() . "\n";
        echo "Session save path: " . session_save_path() . "\n";
        
        // Test session write
        $_SESSION['test_data'] = 'Session working at ' . date('Y-m-d H:i:s');
        echo "✅ Session data written\n";
        
    } else {
        echo "❌ No writable temp directory found!\n";
    }
} else {
    echo "Session already started\n";
    echo "Session ID: " . session_id() . "\n";
    echo "Session save path: " . session_save_path() . "\n";
}

echo "\n=== STEP 4: Test Include Chain ===\n";

// Test including session_start.php if it exists
if (file_exists('includes/session_start.php')) {
    echo "Testing includes/session_start.php...\n";
    ob_start();
    try {
        require_once 'includes/session_start.php';
        $output = ob_get_clean();
        echo "✅ session_start.php included successfully\n";
        if ($output) echo "Output: $output\n";
        echo "Session path after include: " . session_save_path() . "\n";
    } catch (Exception $e) {
        ob_end_clean();
        echo "❌ Error including session_start.php: " . $e->getMessage() . "\n";
    }
} else {
    echo "❌ includes/session_start.php not found\n";
}

echo "\n=== STEP 5: Current Session State ===\n";
echo "Session Status: " . session_status() . " (1=disabled, 2=active, 3=none)\n";
echo "Session ID: " . session_id() . "\n";
echo "Session Save Path: " . session_save_path() . "\n";
echo "Session Data: " . print_r($_SESSION, true) . "\n";

echo "\n=== STEP 6: Authentication Function Test ===\n";

if (file_exists('auth/functions.php')) {
    echo "Loading auth/functions.php...\n";
    try {
        require_once 'auth/functions.php';
        echo "✅ Auth functions loaded\n";
        
        if (function_exists('isLoggedIn')) {
            echo "Testing isLoggedIn(): " . (isLoggedIn() ? 'true' : 'false') . "\n";
        } else {
            echo "❌ isLoggedIn() function not found\n";
        }
        
        if (function_exists('hasPermission')) {
            echo "Testing hasPermission('student'): " . (hasPermission('student') ? 'true' : 'false') . "\n";
        } else {
            echo "❌ hasPermission() function not found\n";
        }
        
    } catch (Exception $e) {
        echo "❌ Error loading auth functions: " . $e->getMessage() . "\n";
    }
} else {
    echo "❌ auth/functions.php not found\n";
}

echo "</pre>";

echo "<h2>Recommended Actions</h2>";
echo "<ul>";
echo "<li>Upload missing files shown above</li>";
echo "<li>Ensure session save path is writable</li>";
echo "<li>Check that paths are relative to server root</li>";
echo "<li>Test the corrected dashboard after fixes</li>";
echo "</ul>";

echo "<p><a href='student/dashboard.php'>Test Dashboard After Fixes</a></p>";
?>
