<?php
require_once __DIR__ . '/../config/db_connect.php';

// Create Certificate Templates table
$certificate_templates_sql = "CREATE TABLE IF NOT EXISTS certificate_templates (
    template_id INT AUTO_INCREMENT PRIMARY KEY,
    template_name VARCHAR(255) NOT NULL,
    certificate_type ENUM('completion', 'accreditation', 'competence', 'participation', 'achievement', 'custom') NOT NULL,
    template_file VARCHAR(500) NOT NULL,
    preview_image VARCHAR(500),
    is_active BOOLEAN DEFAULT TRUE,
    is_default BOOLEAN DEFAULT FALSE,
    created_by INT NOT NULL,
    created_date TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_date TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    description TEXT,
    FOREIGN KEY (created_by) REFERENCES users(user_id)
)";

// Create Certificate Template Fields table
$template_fields_sql = "CREATE TABLE IF NOT EXISTS certificate_template_fields (
    field_id INT AUTO_INCREMENT PRIMARY KEY,
    template_id INT NOT NULL,
    field_name VARCHAR(100) NOT NULL,
    field_type ENUM('student_name', 'student_id', 'course_title', 'completion_date', 'issue_date', 'certificate_number', 'director_name', 'director_signature', 'academy_name', 'grade', 'duration', 'instructor_name', 'custom_text') NOT NULL,
    x_position INT NOT NULL,
    y_position INT NOT NULL,
    width INT DEFAULT 200,
    height INT DEFAULT 50,
    font_family VARCHAR(100) DEFAULT 'Arial',
    font_size INT DEFAULT 16,
    font_weight ENUM('normal', 'bold', '100', '200', '300', '400', '500', '600', '700', '800', '900') DEFAULT 'normal',
    font_color VARCHAR(7) DEFAULT '#000000',
    text_align ENUM('left', 'center', 'right', 'justify') DEFAULT 'center',
    custom_text TEXT,
    created_date TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (template_id) REFERENCES certificate_templates(template_id) ON DELETE CASCADE
)";

// Create Student Certificates table
$student_certificates_sql = "CREATE TABLE IF NOT EXISTS student_certificates (
    certificate_id INT AUTO_INCREMENT PRIMARY KEY,
    student_id INT NOT NULL,
    template_id INT NOT NULL,
    certificate_type ENUM('completion', 'accreditation', 'competence', 'participation', 'achievement', 'custom') NOT NULL,
    course_id INT,
    certificate_number VARCHAR(100) UNIQUE NOT NULL,
    issue_date TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    generated_file VARCHAR(500),
    status ENUM('generated', 'downloaded', 'shared') DEFAULT 'generated',
    metadata JSON,
    created_date TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (student_id) REFERENCES users(user_id),
    FOREIGN KEY (template_id) REFERENCES certificate_templates(template_id),
    FOREIGN KEY (course_id) REFERENCES courses(course_id)
)";

// Create Certificate Settings table
$certificate_settings_sql = "CREATE TABLE IF NOT EXISTS certificate_settings (
    setting_id INT AUTO_INCREMENT PRIMARY KEY,
    setting_key VARCHAR(100) UNIQUE NOT NULL,
    setting_value TEXT,
    description TEXT,
    created_date TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_date TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
)";

try {
    // Execute table creations
    if ($conn->query($certificate_templates_sql) === TRUE) {
        echo "✅ Certificate Templates table created successfully.\n";
    } else {
        echo "❌ Error creating Certificate Templates table: " . $conn->error . "\n";
    }

    if ($conn->query($template_fields_sql) === TRUE) {
        echo "✅ Certificate Template Fields table created successfully.\n";
    } else {
        echo "❌ Error creating Certificate Template Fields table: " . $conn->error . "\n";
    }

    if ($conn->query($student_certificates_sql) === TRUE) {
        echo "✅ Student Certificates table created successfully.\n";
    } else {
        echo "❌ Error creating Student Certificates table: " . $conn->error . "\n";
    }

    if ($conn->query($certificate_settings_sql) === TRUE) {
        echo "✅ Certificate Settings table created successfully.\n";
    } else {
        echo "❌ Error creating Certificate Settings table: " . $conn->error . "\n";
    }

    // Insert default certificate settings
    $default_settings = [
        ['academy_name', 'Panadite Academy', 'Default academy name for certificates'],
        ['director_name', 'Dr. Academy Director', 'Default director name for certificates'],
        ['certificate_prefix', 'PA', 'Prefix for certificate numbers'],
        ['default_font_family', 'Arial', 'Default font family for certificates'],
        ['certificate_quality', 'high', 'Certificate generation quality (low, medium, high)'],
        ['auto_generate_completion', '1', 'Automatically generate completion certificates'],
        ['auto_generate_accreditation', '1', 'Automatically generate accreditation certificates']
    ];

    $settings_stmt = $conn->prepare("INSERT IGNORE INTO certificate_settings (setting_key, setting_value, description) VALUES (?, ?, ?)");
    
    foreach ($default_settings as $setting) {
        $settings_stmt->bind_param("sss", $setting[0], $setting[1], $setting[2]);
        $settings_stmt->execute();
    }

    echo "✅ Default certificate settings inserted.\n";
    
    // Create upload directory structure
    $upload_dirs = [
        '../uploads/certificates/templates/',
        '../uploads/certificates/generated/',
        '../uploads/certificates/previews/'
    ];
    
    foreach ($upload_dirs as $dir) {
        if (!is_dir(__DIR__ . '/' . $dir)) {
            if (mkdir(__DIR__ . '/' . $dir, 0755, true)) {
                echo "✅ Created directory: $dir\n";
            } else {
                echo "❌ Failed to create directory: $dir\n";
            }
        } else {
            echo "✅ Directory already exists: $dir\n";
        }
    }

    echo "\n🎉 Certificate Template System setup completed successfully!\n";
    echo "\n📋 Next Steps:\n";
    echo "1. Navigate to Admin Dashboard -> Certificate Templates\n";
    echo "2. Upload your certificate designs from Canva or other design tools\n";
    echo "3. Configure field positions using the drag-and-drop editor\n";
    echo "4. Set default templates for each certificate type\n";
    echo "5. Test certificate generation\n";

} catch (Exception $e) {
    echo "❌ Error setting up Certificate Template System: " . $e->getMessage() . "\n";
}

$conn->close();
?>
