<?php
/**
 * Setup script for SITA Assessment tables
 * This script creates the necessary database tables for SITA accreditation assessments
 */

require_once '../config/db_connect.php';

echo "<h2>Creating SITA Assessment Tables...</h2>";

try {
    // Create sita_assessments table
    $sita_assessments_sql = "
    CREATE TABLE IF NOT EXISTS sita_assessments (
        assessment_id INT AUTO_INCREMENT PRIMARY KEY,
        student_id INT NOT NULL,
        course_id INT NOT NULL,
        accreditation_request_id INT NOT NULL,
        title VARCHAR(255) NOT NULL,
        description TEXT,
        status ENUM('pending', 'in_progress', 'submitted', 'graded', 'passed', 'failed') DEFAULT 'pending',
        created_date TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        due_date TIMESTAMP NULL,
        submitted_date TIMESTAMP NULL,
        graded_date TIMESTAMP NULL,
        score DECIMAL(5,2) NULL,
        passing_score DECIMAL(5,2) DEFAULT 70.00,
        graded_by INT NULL,
        feedback TEXT NULL,
        FOREIGN KEY (student_id) REFERENCES users(user_id) ON DELETE CASCADE,
        FOREIGN KEY (course_id) REFERENCES courses(course_id) ON DELETE CASCADE,
        FOREIGN KEY (accreditation_request_id) REFERENCES accreditation_requests(request_id) ON DELETE CASCADE,
        FOREIGN KEY (graded_by) REFERENCES users(user_id) ON DELETE SET NULL,
        INDEX idx_student_course (student_id, course_id),
        INDEX idx_status (status),
        INDEX idx_created_date (created_date)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
    
    if ($conn->query($sita_assessments_sql) === TRUE) {
        echo "<p style='color: green;'>✓ Table 'sita_assessments' created successfully</p>";
    } else {
        throw new Exception("Error creating sita_assessments table: " . $conn->error);
    }
    
    // Create sita_assessment_questions table
    $sita_questions_sql = "
    CREATE TABLE IF NOT EXISTS sita_assessment_questions (
        question_id INT AUTO_INCREMENT PRIMARY KEY,
        assessment_id INT NOT NULL,
        question_text TEXT NOT NULL,
        question_order INT NOT NULL,
        question_type ENUM('essay', 'multiple_choice', 'short_answer', 'file_upload') DEFAULT 'essay',
        max_words INT DEFAULT 500,
        points INT DEFAULT 20,
        required BOOLEAN DEFAULT TRUE,
        created_date TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        FOREIGN KEY (assessment_id) REFERENCES sita_assessments(assessment_id) ON DELETE CASCADE,
        INDEX idx_assessment_order (assessment_id, question_order)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
    
    if ($conn->query($sita_questions_sql) === TRUE) {
        echo "<p style='color: green;'>✓ Table 'sita_assessment_questions' created successfully</p>";
    } else {
        throw new Exception("Error creating sita_assessment_questions table: " . $conn->error);
    }
    
    // Create sita_assessment_answers table
    $sita_answers_sql = "
    CREATE TABLE IF NOT EXISTS sita_assessment_answers (
        answer_id INT AUTO_INCREMENT PRIMARY KEY,
        assessment_id INT NOT NULL,
        question_id INT NOT NULL,
        student_id INT NOT NULL,
        answer_text TEXT NULL,
        file_path VARCHAR(500) NULL,
        submitted_date TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        score DECIMAL(5,2) NULL,
        feedback TEXT NULL,
        FOREIGN KEY (assessment_id) REFERENCES sita_assessments(assessment_id) ON DELETE CASCADE,
        FOREIGN KEY (question_id) REFERENCES sita_assessment_questions(question_id) ON DELETE CASCADE,
        FOREIGN KEY (student_id) REFERENCES users(user_id) ON DELETE CASCADE,
        UNIQUE KEY unique_answer (assessment_id, question_id, student_id),
        INDEX idx_assessment_student (assessment_id, student_id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
    
    if ($conn->query($sita_answers_sql) === TRUE) {
        echo "<p style='color: green;'>✓ Table 'sita_assessment_answers' created successfully</p>";
    } else {
        throw new Exception("Error creating sita_assessment_answers table: " . $conn->error);
    }
    
    // Create sita_assessment_audit_log table
    $sita_audit_sql = "
    CREATE TABLE IF NOT EXISTS sita_assessment_audit_log (
        log_id INT AUTO_INCREMENT PRIMARY KEY,
        assessment_id INT NOT NULL,
        student_id INT NOT NULL,
        action VARCHAR(100) NOT NULL,
        action_date TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        performed_by INT NULL,
        details TEXT NULL,
        ip_address VARCHAR(45) NULL,
        FOREIGN KEY (assessment_id) REFERENCES sita_assessments(assessment_id) ON DELETE CASCADE,
        FOREIGN KEY (student_id) REFERENCES users(user_id) ON DELETE CASCADE,
        FOREIGN KEY (performed_by) REFERENCES users(user_id) ON DELETE SET NULL,
        INDEX idx_assessment_date (assessment_id, action_date),
        INDEX idx_student_action (student_id, action)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
    
    if ($conn->query($sita_audit_sql) === TRUE) {
        echo "<p style='color: green;'>✓ Table 'sita_assessment_audit_log' created successfully</p>";
    } else {
        throw new Exception("Error creating sita_assessment_audit_log table: " . $conn->error);
    }
    
    // Update accreditation_requests table if needed (add new status values)
    $update_accreditation_status = "
    ALTER TABLE accreditation_requests 
    MODIFY COLUMN status ENUM(
        'pending', 
        'approved', 
        'assessment_created', 
        'assessment_in_progress',
        'assessment_completed', 
        'assessment_passed',
        'assessment_failed',
        'documents_pending', 
        'documents_submitted', 
        'under_review', 
        'approved_final', 
        'rejected', 
        'certificate_issued'
    ) DEFAULT 'pending'";
    
    if ($conn->query($update_accreditation_status) === TRUE) {
        echo "<p style='color: green;'>✓ Updated 'accreditation_requests' table with new status values</p>";
    } else {
        // This might fail if the enum already has these values, which is okay
        echo "<p style='color: orange;'>⚠ Note: accreditation_requests status enum may already be updated</p>";
    }
    
    echo "<br><h3 style='color: green;'>🎉 SITA Assessment Tables Setup Complete!</h3>";
    echo "<p>All necessary tables for SITA assessments have been created successfully.</p>";
    
    echo "<h4>Created Tables:</h4>";
    echo "<ul>";
    echo "<li><strong>sita_assessments:</strong> Main assessment records</li>";
    echo "<li><strong>sita_assessment_questions:</strong> Assessment questions</li>";
    echo "<li><strong>sita_assessment_answers:</strong> Student answers</li>";
    echo "<li><strong>sita_assessment_audit_log:</strong> Audit trail</li>";
    echo "</ul>";
    
    echo "<p><a href='../admin/accreditation_requests.php' style='background: #2563eb; color: white; padding: 10px 20px; text-decoration: none; border-radius: 5px;'>Go to Admin Panel</a></p>";
    
} catch (Exception $e) {
    echo "<p style='color: red; background: #fef2f2; padding: 15px; border: 1px solid #f87171; border-radius: 5px;'>";
    echo "<strong>Error:</strong> " . $e->getMessage();
    echo "</p>";
    
    echo "<h4>Troubleshooting:</h4>";
    echo "<ul>";
    echo "<li>Make sure the database connection is working</li>";
    echo "<li>Ensure you have proper database permissions</li>";
    echo "<li>Check if the referenced tables (users, courses, accreditation_requests) exist</li>";
    echo "</ul>";
}

if (isset($conn)) {
    $conn->close();
}
?>

<!DOCTYPE html>
<html>
<head>
    <title>SITA Assessment Tables Setup</title>
    <style>
        body { font-family: Arial, sans-serif; max-width: 800px; margin: 0 auto; padding: 20px; }
        h2 { color: #2563eb; }
        .success { color: green; }
        .error { color: red; }
        .warning { color: orange; }
    </style>
</head>
<body>
    <!-- Content is echoed above -->
</body>
</html>
