<?php
/**
 * Deep Email Debugging Tool - Panadite Academy
 * Comprehensive step-by-step email sending debug
 */

require_once __DIR__ . '/../config/fast_email_service.php';

// Enhanced debugging functions
function debugLog($message, $level = 'INFO') {
    $timestamp = date('Y-m-d H:i:s');
    echo "<div class='debug-line debug-{$level}'>";
    echo "<span class='debug-time'>[$timestamp]</span>";
    echo "<span class='debug-level'>[$level]</span>";
    echo "<span class='debug-message'>$message</span>";
    echo "</div>";
    flush();
    ob_flush();
}

function testSMTPConnection($settings) {
    debugLog("Starting SMTP Connection Test", 'INFO');
    
    $host = $settings['smtp_host'];
    $port = intval($settings['smtp_port']);
    $timeout = 10;
    
    debugLog("Attempting connection to {$host}:{$port}", 'INFO');
    
    $socket = @fsockopen($host, $port, $errno, $errstr, $timeout);
    
    if (!$socket) {
        debugLog("FAILED: Cannot connect to SMTP server - $errstr ($errno)", 'ERROR');
        return false;
    }
    
    $response = fgets($socket, 512);
    debugLog("Server greeting: " . trim($response), 'SUCCESS');
    
    // Send EHLO
    fputs($socket, "EHLO " . $_SERVER['HTTP_HOST'] . "\r\n");
    $response = '';
    while ($line = fgets($socket, 512)) {
        $response .= $line;
        if (substr($line, 3, 1) == ' ') break;
    }
    debugLog("EHLO response: " . trim($response), 'INFO');
    
    // Test STARTTLS if enabled
    if ($settings['smtp_encryption'] === 'tls') {
        fputs($socket, "STARTTLS\r\n");
        $response = fgets($socket, 512);
        debugLog("STARTTLS response: " . trim($response), 'INFO');
    }
    
    fclose($socket);
    debugLog("Connection test completed successfully", 'SUCCESS');
    return true;
}

function debugSendEmailNative($to, $subject, $body, $settings) {
    debugLog("=== TESTING NATIVE PHP MAIL ===", 'INFO');
    
    $headers = [
        "MIME-Version: 1.0",
        "Content-Type: text/html; charset=UTF-8",
        "From: " . $settings['from_name'] . " <" . $settings['from_email'] . ">",
        "Reply-To: " . $settings['from_email'],
        "X-Mailer: Panadite Academy Debug System",
        "X-Debug-Test: " . date('Y-m-d H:i:s')
    ];
    
    debugLog("Mail function available: " . (function_exists('mail') ? 'YES' : 'NO'), 'INFO');
    debugLog("To: $to", 'INFO');
    debugLog("Subject: $subject", 'INFO');
    debugLog("Headers: " . implode(" | ", $headers), 'INFO');
    debugLog("Body length: " . strlen($body) . " characters", 'INFO');
    
    $result = mail($to, $subject, $body, implode("\r\n", $headers));
    
    debugLog("PHP mail() result: " . ($result ? 'TRUE' : 'FALSE'), $result ? 'SUCCESS' : 'ERROR');
    
    if (!$result) {
        debugLog("Native mail failed - this suggests server mail configuration issues", 'ERROR');
    }
    
    return $result;
}

function debugSMTPSend($to, $subject, $body, $settings) {
    debugLog("=== TESTING SMTP SEND ===", 'INFO');
    
    $host = $settings['smtp_host'];
    $port = intval($settings['smtp_port']);
    $username = $settings['smtp_username'];
    $password = $settings['smtp_password'];
    $from = $settings['from_email'];
    
    debugLog("SMTP Host: $host", 'INFO');
    debugLog("SMTP Port: $port", 'INFO');
    debugLog("SMTP Username: $username", 'INFO');
    debugLog("SMTP Password: " . (empty($password) ? 'NOT SET' : '***SET***'), 'INFO');
    debugLog("From Email: $from", 'INFO');
    
    $socket = @fsockopen($host, $port, $errno, $errstr, 30);
    
    if (!$socket) {
        debugLog("FATAL: Cannot establish socket connection - $errstr ($errno)", 'ERROR');
        return false;
    }
    
    debugLog("Socket connection established", 'SUCCESS');
    
    // Read greeting
    $response = fgets($socket, 512);
    debugLog("Server Greeting: " . trim($response), 'INFO');
    
    // EHLO
    fputs($socket, "EHLO " . $_SERVER['HTTP_HOST'] . "\r\n");
    $ehlo_response = '';
    while ($line = fgets($socket, 512)) {
        $ehlo_response .= $line;
        debugLog("EHLO Line: " . trim($line), 'INFO');
        if (substr($line, 3, 1) == ' ') break;
    }
    
    // STARTTLS if needed
    if ($settings['smtp_encryption'] === 'tls') {
        debugLog("Initiating STARTTLS", 'INFO');
        fputs($socket, "STARTTLS\r\n");
        $response = fgets($socket, 512);
        debugLog("STARTTLS Response: " . trim($response), 'INFO');
        
        if (substr($response, 0, 3) != '220') {
            debugLog("STARTTLS failed", 'ERROR');
            fclose($socket);
            return false;
        }
        
        // Enable crypto
        if (!stream_socket_enable_crypto($socket, true, STREAM_CRYPTO_METHOD_TLS_CLIENT)) {
            debugLog("Failed to enable TLS encryption", 'ERROR');
            fclose($socket);
            return false;
        }
        debugLog("TLS encryption enabled successfully", 'SUCCESS');
        
        // Send EHLO again after TLS
        fputs($socket, "EHLO " . $_SERVER['HTTP_HOST'] . "\r\n");
        while ($line = fgets($socket, 512)) {
            debugLog("Post-TLS EHLO: " . trim($line), 'INFO');
            if (substr($line, 3, 1) == ' ') break;
        }
    }
    
    // Authentication
    if ($settings['smtp_auth'] === '1' && !empty($username)) {
        debugLog("Starting SMTP Authentication", 'INFO');
        
        fputs($socket, "AUTH LOGIN\r\n");
        $response = fgets($socket, 512);
        debugLog("AUTH LOGIN Response: " . trim($response), 'INFO');
        
        if (substr($response, 0, 3) != '334') {
            debugLog("AUTH LOGIN failed", 'ERROR');
            fclose($socket);
            return false;
        }
        
        // Send username
        fputs($socket, base64_encode($username) . "\r\n");
        $response = fgets($socket, 512);
        debugLog("Username Response: " . trim($response), 'INFO');
        
        // Send password
        fputs($socket, base64_encode($password) . "\r\n");
        $response = fgets($socket, 512);
        debugLog("Password Response: " . trim($response), 'INFO');
        
        if (substr($response, 0, 3) != '235') {
            debugLog("SMTP Authentication FAILED - check username/password", 'ERROR');
            fclose($socket);
            return false;
        }
        
        debugLog("SMTP Authentication SUCCESS", 'SUCCESS');
    }
    
    // MAIL FROM
    fputs($socket, "MAIL FROM:<$from>\r\n");
    $response = fgets($socket, 512);
    debugLog("MAIL FROM Response: " . trim($response), 'INFO');
    
    if (substr($response, 0, 3) != '250') {
        debugLog("MAIL FROM rejected", 'ERROR');
        fclose($socket);
        return false;
    }
    
    // RCPT TO
    fputs($socket, "RCPT TO:<$to>\r\n");
    $response = fgets($socket, 512);
    debugLog("RCPT TO Response: " . trim($response), 'INFO');
    
    if (substr($response, 0, 3) != '250') {
        debugLog("RCPT TO rejected - recipient not accepted", 'ERROR');
        fclose($socket);
        return false;
    }
    
    // DATA
    fputs($socket, "DATA\r\n");
    $response = fgets($socket, 512);
    debugLog("DATA Response: " . trim($response), 'INFO');
    
    if (substr($response, 0, 3) != '354') {
        debugLog("DATA command rejected", 'ERROR');
        fclose($socket);
        return false;
    }
    
    // Email headers and body
    $email_data = "From: {$settings['from_name']} <$from>\r\n";
    $email_data .= "To: $to\r\n";
    $email_data .= "Subject: $subject\r\n";
    $email_data .= "MIME-Version: 1.0\r\n";
    $email_data .= "Content-Type: text/html; charset=UTF-8\r\n";
    $email_data .= "X-Mailer: Panadite Academy Debug\r\n";
    $email_data .= "X-Debug-Timestamp: " . date('r') . "\r\n";
    $email_data .= "\r\n";
    $email_data .= $body;
    $email_data .= "\r\n.\r\n";
    
    debugLog("Sending email data (" . strlen($email_data) . " bytes)", 'INFO');
    fputs($socket, $email_data);
    
    $response = fgets($socket, 512);
    debugLog("Email Data Response: " . trim($response), 'INFO');
    
    if (substr($response, 0, 3) != '250') {
        debugLog("Email data rejected", 'ERROR');
        fclose($socket);
        return false;
    }
    
    // QUIT
    fputs($socket, "QUIT\r\n");
    $response = fgets($socket, 512);
    debugLog("QUIT Response: " . trim($response), 'INFO');
    
    fclose($socket);
    debugLog("SMTP send completed successfully", 'SUCCESS');
    return true;
}

// Handle form submission
$testResult = null;
if ($_POST && isset($_POST['test_email'])) {
    $testEmail = filter_var($_POST['test_email'], FILTER_VALIDATE_EMAIL);
    
    if ($testEmail) {
        ob_start();
        ?>
        <!DOCTYPE html>
        <html lang="en">
        <head>
            <meta charset="UTF-8">
            <meta name="viewport" content="width=device-width, initial-scale=1.0">
            <title>Deep Email Debug - Panadite Academy</title>
            <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css" rel="stylesheet">
            <style>
                .debug-console { 
                    background: #1a1a1a; 
                    color: #00ff00; 
                    padding: 20px; 
                    border-radius: 8px; 
                    font-family: 'Courier New', monospace; 
                    max-height: 600px; 
                    overflow-y: auto; 
                    margin: 20px 0;
                }
                .debug-line { 
                    margin: 2px 0; 
                    padding: 2px 5px; 
                    border-radius: 3px;
                }
                .debug-INFO { color: #00ff00; }
                .debug-SUCCESS { color: #00ff88; background: rgba(0,255,136,0.1); }
                .debug-ERROR { color: #ff4444; background: rgba(255,68,68,0.1); }
                .debug-WARNING { color: #ffaa00; background: rgba(255,170,0,0.1); }
                .debug-time { color: #888; }
                .debug-level { font-weight: bold; margin: 0 5px; }
                .debug-message { }
            </style>
        </head>
        <body class="bg-dark text-white">
            <div class="container-fluid">
                <div class="row">
                    <div class="col-12">
                        <h2 class="text-center mb-4">🔍 Deep Email Debug Console</h2>
                        <div class="debug-console">
        <?php
        
        debugLog("=== STARTING COMPREHENSIVE EMAIL DEBUG ===", 'INFO');
        debugLog("Target Email: $testEmail", 'INFO');
        debugLog("Server: " . $_SERVER['HTTP_HOST'], 'INFO');
        debugLog("Time: " . date('Y-m-d H:i:s T'), 'INFO');
        
        // Get email settings
        debugLog("Loading email settings...", 'INFO');
        $settings = getEmailSettings();
        
        debugLog("SMTP Enabled: " . ($settings['smtp_enabled'] === '1' ? 'YES' : 'NO'), 'INFO');
        debugLog("SMTP Host: " . ($settings['smtp_host'] ?: 'NOT SET'), 'INFO');
        debugLog("SMTP Port: " . ($settings['smtp_port'] ?: 'NOT SET'), 'INFO');
        debugLog("SMTP Encryption: " . ($settings['smtp_encryption'] ?: 'NONE'), 'INFO');
        debugLog("From Email: " . ($settings['from_email'] ?: 'NOT SET'), 'INFO');
        
        // Test basic connectivity
        if ($settings['smtp_enabled'] === '1' && !empty($settings['smtp_host'])) {
            testSMTPConnection($settings);
        }
        
        // Test email content
        debugLog("=== PREPARING TEST EMAIL ===", 'INFO');
        $subject = "DEBUG TEST - Panadite Academy Email System";
        $body = "<html><body style='font-family:Arial;padding:20px;'>";
        $body .= "<h2>🔧 Email Debug Test</h2>";
        $body .= "<p>This is a debug test email sent at: " . date('Y-m-d H:i:s T') . "</p>";
        $body .= "<p>If you receive this email, the system is working correctly.</p>";
        $body .= "<div style='background:#f0f0f0;padding:10px;margin:20px 0;'>";
        $body .= "<strong>Debug Info:</strong><br>";
        $body .= "Server: " . $_SERVER['HTTP_HOST'] . "<br>";
        $body .= "Method: " . ($settings['smtp_enabled'] === '1' ? 'SMTP' : 'Native PHP Mail') . "<br>";
        $body .= "Timestamp: " . time() . "<br>";
        $body .= "</div></body></html>";
        
        debugLog("Email subject: $subject", 'INFO');
        debugLog("Email body length: " . strlen($body) . " characters", 'INFO');
        
        // Try different sending methods
        if ($settings['smtp_enabled'] === '1' && !empty($settings['smtp_host'])) {
            debugLog("Attempting SMTP send...", 'INFO');
            $smtpResult = debugSMTPSend($testEmail, $subject, $body, $settings);
            
            if (!$smtpResult) {
                debugLog("SMTP failed, falling back to native mail", 'WARNING');
                debugSendEmailNative($testEmail, $subject, $body, $settings);
            }
        } else {
            debugLog("SMTP not configured, using native PHP mail", 'INFO');
            debugSendEmailNative($testEmail, $subject, $body, $settings);
        }
        
        debugLog("=== DEBUG SESSION COMPLETED ===", 'SUCCESS');
        debugLog("Check your email now (including spam folder)", 'INFO');
        debugLog("If still no email, the issue is with server/provider configuration", 'WARNING');
        
        ?>
                        </div>
                        <div class="text-center mt-4">
                            <a href="?" class="btn btn-primary">Run Another Test</a>
                            <a href="/admin/dashboard.php" class="btn btn-warning">Email Settings</a>
                        </div>
                    </div>
                </div>
            </div>
        </body>
        </html>
        <?php
        exit;
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Deep Email Debug - Panadite Academy</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://unpkg.com/boxicons@2.1.4/css/boxicons.min.css" rel="stylesheet">
</head>
<body class="bg-dark text-white">
    <div class="container mt-5">
        <div class="row justify-content-center">
            <div class="col-md-8">
                <div class="card bg-dark border-light">
                    <div class="card-header bg-danger text-white">
                        <h4 class="mb-0"><i class="bx bx-bug"></i> Deep Email Debug Console</h4>
                    </div>
                    <div class="card-body">
                        
                        <div class="alert alert-warning">
                            <h6><i class="bx bx-shield-x"></i> COMPREHENSIVE EMAIL DEBUG</h6>
                            <p>This tool will perform deep debugging of the email system including:</p>
                            <ul class="mb-0">
                                <li><strong>SMTP Connection Testing</strong> - Real-time server connection</li>
                                <li><strong>Authentication Testing</strong> - Username/password verification</li>
                                <li><strong>Step-by-Step SMTP Conversation</strong> - Every command and response</li>
                                <li><strong>Email Content Analysis</strong> - Headers, body, encoding</li>
                                <li><strong>Server Response Codes</strong> - Detailed error analysis</li>
                            </ul>
                        </div>
                        
                        <form method="POST" action="">
                            <div class="mb-3">
                                <label for="test_email" class="form-label">Email Address for Deep Debug Test</label>
                                <input type="email" class="form-control" id="test_email" name="test_email" 
                                       placeholder="Enter your email address" required 
                                       value="alainkitumaini8@gmail.com">
                                <div class="form-text">This will show EVERY step of the email sending process</div>
                            </div>
                            
                            <button type="submit" class="btn btn-danger btn-lg w-100">
                                <i class="bx bx-bug"></i> START DEEP EMAIL DEBUG
                            </button>
                        </form>
                        
                        <div class="mt-4">
                            <h6><i class="bx bx-info-circle"></i> What This Debug Will Show:</h6>
                            <ul class="small">
                                <li>Real-time SMTP server connection attempt</li>
                                <li>Server greeting and capabilities</li>
                                <li>TLS/SSL encryption negotiation</li>
                                <li>Authentication success/failure details</li>
                                <li>Recipient acceptance/rejection</li>
                                <li>Email data transmission status</li>
                                <li>Final delivery confirmation</li>
                            </ul>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</body>
</html>
