<?php
require_once '../includes/session_start.php';
require_once '../auth/functions.php';
requireRole('student');
require_once '../config/db_connect.php';

// Get current student ID
$student_id = $_SESSION['user_id'];

// Get student's enrolled courses with activities
$courses_query = "
    SELECT DISTINCT 
        c.course_id,
        c.title as course_title,
        c.description,
        c.image,
        COUNT(DISTINCT q.quiz_id) as quiz_count,
        COUNT(DISTINCT a.assignment_id) as assignment_count,
        COUNT(DISTINCT qr.quiz_result_id) as completed_quizzes,
        COUNT(DISTINCT ar.assignment_result_id) as completed_assignments
    FROM enrollments e
    JOIN courses c ON e.course_id = c.course_id
    LEFT JOIN quizzes q ON c.course_id = q.course_id
    LEFT JOIN assignments a ON c.course_id = a.course_id
    LEFT JOIN quiz_results qr ON q.quiz_id = qr.quiz_id AND qr.student_id = ?
    LEFT JOIN assignment_results ar ON a.assignment_id = ar.assignment_id AND ar.student_id = ?
    WHERE e.student_id = ?
    GROUP BY c.course_id, c.title, c.description, c.image
    ORDER BY c.title ASC
";

$stmt = $pdo->prepare($courses_query);
$stmt->execute([$student_id, $student_id, $student_id]);
$enrolled_courses = $stmt->fetchAll(PDO::FETCH_ASSOC);

$page_title = "Student Activities";
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= $page_title ?> - Panadite Academy</title>
    <link href="https://cdn.jsdelivr.net/npm/boxicons@2.1.4/css/boxicons.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    
    <style>
        :root {
            --panadite-primary: #2563eb;
            --panadite-secondary: #1d4ed8;
            --panadite-light: #dbeafe;
            --panadite-dark: #1e40af;
            --panadite-accent: #f59e0b;
            --text-primary: #1f2937;
            --text-secondary: #6b7280;
            --border-color: #e5e7eb;
            --success-color: #10b981;
            --warning-color: #f59e0b;
            --error-color: #ef4444;
        }

        * {
            box-sizing: border-box;
            margin: 0;
            padding: 0;
        }

        body {
            font-family: 'Inter', -apple-system, BlinkMacSystemFont, sans-serif;
            background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%);
            color: var(--text-primary);
            line-height: 1.6;
            min-height: 100vh;
        }

        /* Header */
        .page-header {
            background: linear-gradient(135deg, var(--panadite-primary) 0%, var(--panadite-secondary) 100%);
            color: white;
            padding: 2rem 0;
            margin-bottom: 2rem;
            box-shadow: 0 4px 20px rgba(37, 99, 235, 0.15);
        }

        .header-content {
            max-width: 1200px;
            margin: 0 auto;
            padding: 0 1rem;
            text-align: center;
        }

        .page-title {
            font-size: 2.5rem;
            font-weight: 700;
            margin-bottom: 0.5rem;
            text-shadow: 0 2px 4px rgba(0,0,0,0.1);
        }

        .page-subtitle {
            font-size: 1.1rem;
            opacity: 0.9;
            font-weight: 400;
        }

        /* Main Container */
        .container {
            max-width: 1200px;
            margin: 0 auto;
            padding: 0 1rem 2rem;
        }

        /* Stats Overview */
        .stats-overview {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 1.5rem;
            margin-bottom: 3rem;
        }

        .stat-card {
            background: white;
            border-radius: 16px;
            padding: 2rem;
            box-shadow: 0 4px 20px rgba(0,0,0,0.08);
            border: 2px solid transparent;
            transition: all 0.3s ease;
            position: relative;
            overflow: hidden;
        }

        .stat-card::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 4px;
            background: linear-gradient(90deg, var(--panadite-primary), var(--panadite-secondary));
        }

        .stat-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 8px 30px rgba(37, 99, 235, 0.15);
            border-color: var(--panadite-light);
        }

        .stat-icon {
            width: 60px;
            height: 60px;
            background: linear-gradient(135deg, var(--panadite-light) 0%, rgba(37, 99, 235, 0.1) 100%);
            border-radius: 12px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 1.8rem;
            color: var(--panadite-primary);
            margin-bottom: 1rem;
        }

        .stat-number {
            font-size: 2.5rem;
            font-weight: 700;
            color: var(--panadite-primary);
            margin-bottom: 0.5rem;
        }

        .stat-label {
            font-size: 1rem;
            color: var(--text-secondary);
            font-weight: 500;
        }

        /* Course Grid */
        .courses-section {
            margin-bottom: 3rem;
        }

        .section-header {
            display: flex;
            align-items: center;
            gap: 1rem;
            margin-bottom: 2rem;
            padding-bottom: 1rem;
            border-bottom: 2px solid var(--border-color);
        }

        .section-title {
            font-size: 1.8rem;
            font-weight: 600;
            color: var(--text-primary);
        }

        .section-icon {
            width: 40px;
            height: 40px;
            background: var(--panadite-primary);
            border-radius: 10px;
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-size: 1.2rem;
        }

        .courses-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(400px, 1fr));
            gap: 2rem;
        }

        /* Course Card */
        .course-card {
            background: white;
            border-radius: 20px;
            overflow: hidden;
            box-shadow: 0 6px 25px rgba(0,0,0,0.08);
            transition: all 0.3s ease;
            border: 2px solid transparent;
            position: relative;
        }

        .course-card:hover {
            transform: translateY(-8px);
            box-shadow: 0 12px 40px rgba(37, 99, 235, 0.15);
            border-color: var(--panadite-light);
        }

        .course-image {
            width: 100%;
            height: 200px;
            background: linear-gradient(135deg, var(--panadite-primary) 0%, var(--panadite-secondary) 100%);
            background-size: cover;
            background-position: center;
            position: relative;
            display: flex;
            align-items: center;
            justify-content: center;
        }

        .course-image::before {
            content: '';
            position: absolute;
            inset: 0;
            background: rgba(0,0,0,0.2);
            z-index: 1;
        }

        .course-image i {
            font-size: 3rem;
            color: white;
            z-index: 2;
        }

        .course-content {
            padding: 2rem;
        }

        .course-title {
            font-size: 1.4rem;
            font-weight: 600;
            color: var(--text-primary);
            margin-bottom: 0.75rem;
            line-height: 1.3;
        }

        .course-description {
            color: var(--text-secondary);
            font-size: 0.95rem;
            margin-bottom: 1.5rem;
            line-height: 1.5;
            display: -webkit-box;
            -webkit-line-clamp: 2;
            -webkit-box-orient: vertical;
            overflow: hidden;
        }

        /* Activity Stats */
        .activity-stats {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 1rem;
            margin-bottom: 2rem;
            padding: 1.5rem;
            background: var(--panadite-light);
            border-radius: 12px;
            border: 1px solid rgba(37, 99, 235, 0.2);
        }

        .activity-stat {
            text-align: center;
        }

        .activity-stat-number {
            font-size: 1.8rem;
            font-weight: 700;
            color: var(--panadite-primary);
            display: block;
        }

        .activity-stat-label {
            font-size: 0.85rem;
            color: var(--text-secondary);
            font-weight: 500;
            margin-top: 0.25rem;
        }

        /* Activity Buttons */
        .activity-buttons {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 1rem;
        }

        .activity-btn {
            padding: 1rem 1.5rem;
            border: none;
            border-radius: 12px;
            font-weight: 600;
            font-size: 0.95rem;
            cursor: pointer;
            transition: all 0.3s ease;
            text-decoration: none;
            text-align: center;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 0.5rem;
            position: relative;
            overflow: hidden;
        }

        .activity-btn::before {
            content: '';
            position: absolute;
            inset: 0;
            background: linear-gradient(45deg, transparent 30%, rgba(255,255,255,0.3) 50%, transparent 70%);
            transform: translateX(-100%);
            transition: transform 0.5s ease;
        }

        .activity-btn:hover::before {
            transform: translateX(100%);
        }

        .quiz-btn {
            background: linear-gradient(135deg, var(--panadite-primary) 0%, var(--panadite-secondary) 100%);
            color: white;
            box-shadow: 0 4px 15px rgba(37, 99, 235, 0.3);
        }

        .quiz-btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(37, 99, 235, 0.4);
        }

        .assignment-btn {
            background: linear-gradient(135deg, var(--panadite-accent) 0%, #f97316 100%);
            color: white;
            box-shadow: 0 4px 15px rgba(245, 158, 11, 0.3);
        }

        .assignment-btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(245, 158, 11, 0.4);
        }

        .activity-btn:disabled {
            opacity: 0.6;
            cursor: not-allowed;
            transform: none !important;
        }

        /* Empty State */
        .empty-state {
            text-align: center;
            padding: 4rem 2rem;
            color: var(--text-secondary);
        }

        .empty-icon {
            font-size: 4rem;
            color: var(--border-color);
            margin-bottom: 1rem;
        }

        .empty-title {
            font-size: 1.5rem;
            font-weight: 600;
            margin-bottom: 0.5rem;
            color: var(--text-primary);
        }

        .empty-description {
            font-size: 1rem;
            max-width: 400px;
            margin: 0 auto;
        }

        /* Mobile Responsive */
        @media (max-width: 768px) {
            .page-title {
                font-size: 2rem;
            }
            
            .courses-grid {
                grid-template-columns: 1fr;
            }
            
            .stats-overview {
                grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            }
            
            .course-content {
                padding: 1.5rem;
            }
            
            .activity-stats {
                padding: 1rem;
            }
        }

        @media (max-width: 480px) {
            .container {
                padding: 0 0.5rem 2rem;
            }
            
            .page-header {
                padding: 1.5rem 0;
            }
            
            .stats-overview {
                grid-template-columns: 1fr;
                gap: 1rem;
            }
            
            .stat-card {
                padding: 1.5rem;
            }
            
            .activity-buttons {
                grid-template-columns: 1fr;
            }
        }
    </style>
</head>
<body>
    <?php include '../student/components/sidebar.php'; ?>
    
    <div class="main-content">
        <!-- Page Header -->
        <div class="page-header">
            <div class="header-content">
                <h1 class="page-title">
                    <i class="bx bx-task"></i>
                    Student Activities
                </h1>
                <p class="page-subtitle">Access your quizzes and assignments across all enrolled courses</p>
            </div>
        </div>

        <div class="container">
            <!-- Stats Overview -->
            <div class="stats-overview">
                <?php
                $total_courses = count($enrolled_courses);
                $total_quizzes = array_sum(array_column($enrolled_courses, 'quiz_count'));
                $total_assignments = array_sum(array_column($enrolled_courses, 'assignment_count'));
                $completed_quizzes = array_sum(array_column($enrolled_courses, 'completed_quizzes'));
                $completed_assignments = array_sum(array_column($enrolled_courses, 'completed_assignments'));
                ?>
                
                <div class="stat-card">
                    <div class="stat-icon">
                        <i class="bx bx-book"></i>
                    </div>
                    <div class="stat-number"><?= $total_courses ?></div>
                    <div class="stat-label">Enrolled Courses</div>
                </div>
                
                <div class="stat-card">
                    <div class="stat-icon">
                        <i class="bx bx-help-circle"></i>
                    </div>
                    <div class="stat-number"><?= $completed_quizzes ?>/<?= $total_quizzes ?></div>
                    <div class="stat-label">Quizzes Completed</div>
                </div>
                
                <div class="stat-card">
                    <div class="stat-icon">
                        <i class="bx bx-task"></i>
                    </div>
                    <div class="stat-number"><?= $completed_assignments ?>/<?= $total_assignments ?></div>
                    <div class="stat-label">Assignments Completed</div>
                </div>
            </div>

            <!-- Courses Section -->
            <div class="courses-section">
                <div class="section-header">
                    <div class="section-icon">
                        <i class="bx bx-collection"></i>
                    </div>
                    <h2 class="section-title">Your Course Activities</h2>
                </div>

                <?php if (empty($enrolled_courses)): ?>
                    <div class="empty-state">
                        <div class="empty-icon">
                            <i class="bx bx-book-open"></i>
                        </div>
                        <h3 class="empty-title">No Courses Found</h3>
                        <p class="empty-description">
                            You are not currently enrolled in any courses. Please contact your administrator to get enrolled in courses.
                        </p>
                    </div>
                <?php else: ?>
                    <div class="courses-grid">
                        <?php foreach ($enrolled_courses as $course): ?>
                            <div class="course-card">
                                <div class="course-image" <?php if (!empty($course['image'])): ?>style="background-image: url('<?= htmlspecialchars($course['image']) ?>')"<?php endif; ?>>
                                    <?php if (empty($course['image'])): ?>
                                        <i class="bx bx-book"></i>
                                    <?php endif; ?>
                                </div>
                                
                                <div class="course-content">
                                    <h3 class="course-title"><?= htmlspecialchars($course['course_title']) ?></h3>
                                    <p class="course-description"><?= htmlspecialchars($course['description']) ?></p>
                                    
                                    <!-- Activity Statistics -->
                                    <div class="activity-stats">
                                        <div class="activity-stat">
                                            <span class="activity-stat-number"><?= $course['quiz_count'] ?></span>
                                            <span class="activity-stat-label">Available Quizzes</span>
                                        </div>
                                        <div class="activity-stat">
                                            <span class="activity-stat-number"><?= $course['assignment_count'] ?></span>
                                            <span class="activity-stat-label">Available Assignments</span>
                                        </div>
                                    </div>
                                    
                                    <!-- Activity Buttons -->
                                    <div class="activity-buttons">
                                        <a href="quizzes.php?course_id=<?= $course['course_id'] ?>" 
                                           class="activity-btn quiz-btn"
                                           <?= $course['quiz_count'] == 0 ? 'style="opacity: 0.6; cursor: not-allowed;"' : '' ?>>
                                            <i class="bx bx-help-circle"></i>
                                            View Quizzes
                                        </a>
                                        
                                        <a href="assignments.php?course_id=<?= $course['course_id'] ?>" 
                                           class="activity-btn assignment-btn"
                                           <?= $course['assignment_count'] == 0 ? 'style="opacity: 0.6; cursor: not-allowed;"' : '' ?>>
                                            <i class="bx bx-task"></i>
                                            View Assignments
                                        </a>
                                    </div>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <script>
        // Add smooth animations and interactions
        document.addEventListener('DOMContentLoaded', function() {
            // Animate stat cards on scroll
            const observerOptions = {
                threshold: 0.1,
                rootMargin: '0px'
            };

            const observer = new IntersectionObserver((entries) => {
                entries.forEach(entry => {
                    if (entry.isIntersecting) {
                        entry.target.style.animation = 'fadeInUp 0.6s ease forwards';
                    }
                });
            }, observerOptions);

            // Observe all animated elements
            document.querySelectorAll('.stat-card, .course-card').forEach(el => {
                observer.observe(el);
            });

            // Disable clicks on inactive buttons
            document.querySelectorAll('.activity-btn[style*="opacity: 0.6"]').forEach(btn => {
                btn.addEventListener('click', function(e) {
                    e.preventDefault();
                    return false;
                });
            });
        });

        // Add keyframe animations
        const style = document.createElement('style');
        style.textContent = `
            @keyframes fadeInUp {
                from {
                    opacity: 0;
                    transform: translateY(30px);
                }
                to {
                    opacity: 1;
                    transform: translateY(0);
                }
            }
        `;
        document.head.appendChild(style);
    </script>
</body>
</html>
