<?php
// Set content type first to ensure proper JSON response
header('Content-Type: application/json');

// Set error reporting for production (hide errors from JSON response)
error_reporting(E_ALL & ~E_NOTICE & ~E_WARNING);
ini_set('display_errors', 0);

// Initialize session with proper configuration for both local and live servers
if (!session_id()) {
    ini_set('session.save_path', sys_get_temp_dir());
    session_start();
}

// Include database and authentication
require_once '../../config/db_connect.php';

// Simple authentication check using session
if (!isset($_SESSION['user_id']) || !isset($_SESSION['user_role']) || $_SESSION['user_role'] !== 'student') {
    echo json_encode(['error' => 'Authentication required']);
    exit;
}

header('Content-Type: application/json');

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['error' => 'Method not allowed']);
    exit;
}

// Get POST data
$video_id = intval($_POST['video_id'] ?? 0);
$lesson_id = intval($_POST['lesson_id'] ?? 0);
$course_id = intval($_POST['course_id'] ?? 0);
$progress_percentage = intval($_POST['progress_percentage'] ?? 0);
$current_time = intval($_POST['current_time'] ?? 0);
$completed = intval($_POST['completed'] ?? 0);
$student_id = $_SESSION['user_id'];

// DEBUG: Log all received data
error_log("🐛 VIDEO PROGRESS UPDATE - Received data: " . json_encode([
    'video_id' => $video_id,
    'lesson_id' => $lesson_id,
    'course_id' => $course_id,
    'progress_percentage' => $progress_percentage,
    'current_time' => $current_time,
    'completed' => $completed,
    'completed_raw' => $_POST['completed'] ?? 'NOT_SET',
    'student_id' => $student_id
]));

// Use video_id if available, otherwise fall back to lesson_id
$tracking_id = $video_id ? $video_id : $lesson_id;

// Validate required fields
if (!$tracking_id || !$course_id) {
    echo json_encode([
        'error' => 'Missing required fields',
        'debug' => [
            'video_id' => $video_id,
            'lesson_id' => $lesson_id,
            'tracking_id' => $tracking_id,
            'course_id' => $course_id
        ]
    ]);
    exit;
}

// Check database connection first
if (!$conn) {
    echo json_encode(['error' => 'Database connection failed']);
    exit;
}

// Verify student is enrolled in the course
$enrollment_check = $conn->prepare("
    SELECT enrollment_id 
    FROM enrollments 
    WHERE student_id = ? AND course_id = ?
");

if (!$enrollment_check) {
    echo json_encode([
        'error' => 'Database prepare failed', 
        'details' => $conn->error,
        'debug' => 'enrollment check prepare failed'
    ]);
    exit;
}

$enrollment_check->bind_param("ii", $student_id, $course_id);

if (!$enrollment_check->execute()) {
    echo json_encode([
        'error' => 'Database execute failed', 
        'details' => $enrollment_check->error,
        'debug' => 'enrollment check execute failed'
    ]);
    exit;
}

$enrollment_result = $enrollment_check->get_result();
if ($enrollment_result->num_rows === 0) {
    echo json_encode(['error' => 'Not enrolled in this course']);
    exit;
}

try {
    // Check if progress record exists
    $check_progress = $conn->prepare("
        SELECT id, last_position, completed 
        FROM video_progress 
        WHERE student_id = ? AND video_id = ?
    ");
    
    if (!$check_progress) {
        echo json_encode([
            'error' => 'Database prepare failed', 
            'details' => $conn->error,
            'debug' => 'video progress check prepare failed'
        ]);
        exit;
    }
    
    $check_progress->bind_param("ii", $student_id, $tracking_id);
    
    if (!$check_progress->execute()) {
        echo json_encode([
            'error' => 'Database execute failed', 
            'details' => $check_progress->error,
            'debug' => 'video progress check execute failed'
        ]);
        exit;
    }
    
    $existing_progress = $check_progress->get_result()->fetch_assoc();

    if ($existing_progress) {
        // Update existing progress - FIXED to include completion_percentage
        $update_progress = $conn->prepare("
            UPDATE video_progress 
            SET last_position = ?, 
                completed = ?,
                completion_percentage = ?,
                last_watched = NOW()
            WHERE student_id = ? AND video_id = ?
        ");
        
        if (!$update_progress) {
            echo json_encode([
                'error' => 'Update prepare failed', 
                'details' => $conn->error,
                'debug' => 'video progress update prepare failed'
            ]);
            exit;
        }
        
        $update_progress->bind_param("iiiii", $current_time, $completed, $progress_percentage, $student_id, $tracking_id);
        
        if (!$update_progress->execute()) {
            echo json_encode([
                'error' => 'Update execute failed', 
                'details' => $update_progress->error,
                'debug' => 'video progress update execute failed'
            ]);
            exit;
        }
        
        $action = 'updated';
    } else {
        // Insert new progress record - FIXED to include completion_percentage
        $insert_progress = $conn->prepare("
            INSERT INTO video_progress 
            (student_id, video_id, last_position, completed, completion_percentage) 
            VALUES (?, ?, ?, ?, ?)
        ");
        
        if (!$insert_progress) {
            echo json_encode([
                'error' => 'Insert prepare failed', 
                'details' => $conn->error,
                'debug' => 'video progress insert prepare failed'
            ]);
            exit;
        }
        
        $insert_progress->bind_param("iiiii", $student_id, $tracking_id, $current_time, $completed, $progress_percentage);
        
        if (!$insert_progress->execute()) {
            echo json_encode([
                'error' => 'Insert execute failed', 
                'details' => $insert_progress->error,
                'debug' => 'video progress insert execute failed'
            ]);
            exit;
        }
        
        $action = 'created';
    }

    // Calculate overall course progress - use hard-coded 7 lessons (as shown in UI)
    $total_lessons = 7; // Course shows 7 lessons in UI
    
    // Get completed videos count (same JOIN as existing system)  
    $course_progress_query = $conn->prepare("
        SELECT COUNT(DISTINCT vp.video_id) as completed_lessons
        FROM video_progress vp
        JOIN videos v ON vp.video_id = v.video_id 
        WHERE vp.student_id = ? AND v.course_id = ? AND vp.completed = 1
    ");
    
    if (!$course_progress_query) {
        error_log("Course progress prepare failed: " . $conn->error);
        echo json_encode([
            'error' => 'Course progress prepare failed',
            'details' => $conn->error,
            'debug' => 'course progress query prepare failed'
        ]);
        exit;
    }
    
    $course_progress_query->bind_param("ii", $student_id, $course_id);
    
    if (!$course_progress_query->execute()) {
        echo json_encode([
            'error' => 'Course progress execute failed',
            'details' => $course_progress_query->error,
            'debug' => 'course progress query execute failed'
        ]);
        exit;
    }
    
    $course_progress = $course_progress_query->get_result()->fetch_assoc();
    $lessons_completed = intval($course_progress['completed_lessons']);
    
    // Use the SAME calculation as existing system in course-detail.php (completed / total * 100)
    $overall_progress = $total_lessons > 0 ? round(($lessons_completed / $total_lessons) * 100) : 0;
    
    // Enhanced debug logging with course progress calculation values  
    error_log("🎯 COURSE PROGRESS DEBUG - Total: $total_lessons, Completed: $lessons_completed, Overall: $overall_progress%");
    error_log("🔍 COURSE PROGRESS CALCULATION: $lessons_completed completed videos ÷ $total_lessons total lessons × 100 = $overall_progress%");

    echo json_encode([
        'success' => true,
        'action' => $action,
        'video_progress' => round($overall_progress, 1),
        'course_progress' => round($overall_progress, 1),
        'completed_videos' => $lessons_completed,
        'total_videos' => $total_lessons,
        'message' => "Progress updated successfully",
        'debug_data' => [
            'received_data' => [
                'video_id' => $video_id,
                'lesson_id' => $lesson_id,
                'course_id' => $course_id,
                'progress_percentage' => $progress_percentage,
                'current_time' => $current_time,
                'completed' => $completed,
                'completed_raw' => $_POST['completed'] ?? 'NOT_SET',
                'student_id' => $student_id,
                'tracking_id' => $tracking_id
            ],
            'database_action' => $action,
            'completion_status' => $completed ? 'COMPLETED' : 'NOT_COMPLETED',
            'course_progress_debug' => [
                'total_lessons' => $total_lessons,
                'completed_lessons' => $lessons_completed,
                'overall_progress_calculated' => $overall_progress,
                'calculation_method' => 'completed_videos / 7_lessons * 100',
                'raw_query_result' => $course_progress
            ]
        ]
    ]);

} catch (Exception $e) {
    error_log("Video progress update error: " . $e->getMessage());
    echo json_encode([
        'success' => false,
        'error' => 'Failed to update progress',
        'details' => $e->getMessage(),
        'debug_data' => [
            'received_data' => [
                'video_id' => $_POST['video_id'] ?? 'missing',
                'completed' => $_POST['completed'] ?? 'missing',
                'progress_percentage' => $_POST['progress_percentage'] ?? 'missing'
            ],
            'error_occurred' => true
        ]
    ]);
}
?>
