<?php
// AJAX Help Search Handler
// Note: Session should be started by functions.php or session_start.php
require_once '../auth/functions.php';

// Set JSON header
header('Content-Type: application/json');

// Debug session information
error_log('AJAX Search Debug - Session ID: ' . session_id());
error_log('AJAX Search Debug - Session vars: ' . print_r($_SESSION, true));

// Get search query
$searchQuery = isset($_POST['query']) ? trim($_POST['query']) : '';

// Debug information (remove in production)
error_log('AJAX Search - Query: ' . $searchQuery);

if (empty($searchQuery)) {
    echo json_encode(['success' => false, 'message' => 'No search query provided']);
    exit;
}

if (strlen($searchQuery) < 2) {
    echo json_encode(['success' => false, 'message' => 'Search query too short']);
    exit;
}

// Initialize comprehensive help topics (same as in help.php)
$topics = [
    [
        'id' => 'getting-started',
        'title' => 'Getting Started',
        'icon' => 'bx-play-circle',
        'description' => 'Essential guides to begin your learning journey',
        'articles' => [
            [
                'title' => 'Complete Student Dashboard Overview',
                'content' => 'Navigate your dashboard to access courses, assignments, notifications, and progress tracking. Your dashboard is your learning command center.',
                'views' => 892
            ],
            [
                'title' => 'Setting Up Your Student Profile',
                'content' => 'Complete your profile to get personalized learning recommendations and connect with instructors.',
                'views' => 734
            ]
        ]
    ],
    [
        'id' => 'courses',
        'title' => 'Course Enrollment & Management',
        'icon' => 'bx-book-open',
        'description' => 'Everything about finding, enrolling, and managing your courses',
        'articles' => [
            [
                'title' => 'How to Enroll in Courses',
                'content' => 'Step-by-step guide to enroll in courses and start your learning journey.',
                'views' => 1245
            ],
            [
                'title' => 'Accessing Course Materials',
                'content' => 'Learn how to access and download course materials, videos, and resources.',
                'views' => 967
            ],
            [
                'title' => 'Understanding Course Progress Tracking',
                'content' => 'Monitor your learning progress and completion status.',
                'views' => 543
            ]
        ]
    ],
    [
        'id' => 'library',
        'title' => 'Library & Resources',
        'icon' => 'bx-library',
        'description' => 'Access and manage your learning resources effectively',
        'articles' => [
            [
                'title' => 'Using the Digital Library',
                'content' => 'Access thousands of educational resources, books, and reference materials.',
                'views' => 678
            ],
            [
                'title' => 'Managing Your Downloads',
                'content' => 'Track and re-download all your course materials and library resources.',
                'views' => 445
            ]
        ]
    ],
    [
        'id' => 'assignments',
        'title' => 'Assignments & Submissions',
        'icon' => 'bx-task',
        'description' => 'Complete and submit your assignments successfully',
        'articles' => [
            [
                'title' => 'Submitting Assignments',
                'content' => 'Complete guide to submitting assignments and tracking your submissions.',
                'views' => 834
            ],
            [
                'title' => 'Viewing Assignment Feedback',
                'content' => 'Access instructor feedback and grades on your submitted work.',
                'views' => 567
            ]
        ]
    ],
    [
        'id' => 'communication',
        'title' => 'Communication & Support',
        'icon' => 'bx-chat',
        'description' => 'Connect with instructors and get help when needed',
        'articles' => [
            [
                'title' => 'Contacting Your Instructors',
                'content' => 'Various ways to communicate with your course instructors.',
                'views' => 723
            ],
            [
                'title' => 'Getting Technical Support',
                'content' => 'When you need help with platform issues or technical problems.',
                'views' => 892
            ]
        ]
    ],
    [
        'id' => 'notifications',
        'title' => 'Notifications & Updates',
        'icon' => 'bx-bell',
        'description' => 'Stay updated with important announcements and activities',
        'articles' => [
            [
                'title' => 'Managing Your Notifications',
                'content' => 'Control what notifications you receive and how you receive them.',
                'views' => 456
            ]
        ]
    ],
    [
        'id' => 'certificates',
        'title' => 'Certificates & Achievements',
        'icon' => 'bx-award',
        'description' => 'Earn and manage your course completion certificates',
        'articles' => [
            [
                'title' => 'Earning Course Certificates',
                'content' => 'Complete courses and earn digital certificates to showcase your achievements.',
                'views' => 634
            ]
        ]
    ],
    [
        'id' => 'troubleshooting',
        'title' => 'Troubleshooting & FAQ',
        'icon' => 'bx-help-circle',
        'description' => 'Quick solutions to common problems',
        'articles' => [
            [
                'title' => 'Common Download Issues',
                'content' => 'Solve problems when downloading course materials or library resources.',
                'views' => 789
            ],
            [
                'title' => 'Login and Access Problems',
                'content' => 'Troubleshoot account access and login issues.',
                'views' => 543
            ],
            [
                'title' => 'Browser Compatibility',
                'content' => 'Ensure optimal platform performance with supported browsers.',
                'views' => 321
            ]
        ]
    ]
];

// Perform search
$searchResults = [];
foreach ($topics as $topic) {
    foreach ($topic['articles'] as $article) {
        // Search in title and content
        if (stripos($article['title'], $searchQuery) !== false || 
            stripos($article['content'], $searchQuery) !== false ||
            stripos($topic['title'], $searchQuery) !== false) {
            
            $searchResults[] = [
                'title' => $article['title'],
                'content' => $article['content'],
                'views' => $article['views'],
                'topic' => $topic['title'],
                'topic_id' => $topic['id'],
                'relevance' => calculateRelevance($article['title'] . ' ' . $article['content'], $searchQuery)
            ];
        }
    }
}

// Sort by relevance
usort($searchResults, function($a, $b) {
    return $b['relevance'] - $a['relevance'];
});

// Limit results to 10
$searchResults = array_slice($searchResults, 0, 10);

// Return JSON response
echo json_encode([
    'success' => true,
    'query' => $searchQuery,
    'results' => $searchResults,
    'count' => count($searchResults)
]);

function calculateRelevance($text, $query) {
    $text = strtolower($text);
    $query = strtolower($query);
    
    // Exact match in title gets highest score
    if (stripos($text, $query) !== false) {
        $score = 100;
        
        // Bonus for exact word match
        if (preg_match('/\b' . preg_quote($query, '/') . '\b/i', $text)) {
            $score += 50;
        }
        
        // Position bonus (earlier = better)
        $position = stripos($text, $query);
        $score += (100 - $position);
        
        return $score;
    }
    
    // Partial match
    $words = explode(' ', $query);
    $score = 0;
    foreach ($words as $word) {
        if (stripos($text, $word) !== false) {
            $score += 10;
        }
    }
    
    return $score;
}
?>
