<?php
/**
 * AJAX Message Handler
 * Panadite Academy
 */

// Include necessary files
require_once '../includes/session_start.php';
require_once '../config/db_connect.php';

// Set content type to JSON
header('Content-Type: application/json');

// Check user authentication
if (!isset($_SESSION['user_id'])) {
    echo json_encode(['success' => false, 'message' => 'Not logged in']);
    exit;
}

// Support both session variable formats for role
$isStudent = false;
if (isset($_SESSION['role']) && $_SESSION['role'] === 'student') {
    $isStudent = true;
} elseif (isset($_SESSION['user_role']) && $_SESSION['user_role'] === 'student') {
    $isStudent = true;
}

if (!$isStudent) {
    echo json_encode(['success' => false, 'message' => 'Unauthorized access']);
    exit;
}

$userId = $_SESSION['user_id'];

// Handle AJAX actions
$action = isset($_POST['action']) ? $_POST['action'] : '';

switch ($action) {
    case 'send_message':
        sendMessage($conn, $userId);
        break;
        
    case 'mark_read':
        markMessagesAsRead($conn, $userId);
        break;
        
    default:
        echo json_encode(['success' => false, 'message' => 'Invalid action']);
        exit;
}

/**
 * Send a new message
 */
function sendMessage($conn, $userId) {
    // Validate input
    if (!isset($_POST['recipient_id']) || !isset($_POST['message'])) {
        echo json_encode(['success' => false, 'message' => 'Missing required fields']);
        exit;
    }
    
    $recipientId = (int)$_POST['recipient_id'];
    $messageText = trim($_POST['message']);
    
    if (empty($messageText)) {
        echo json_encode(['success' => false, 'message' => 'Message cannot be empty']);
        exit;
    }
    
    try {
        // Verify recipient is a valid contact (teacher of student's enrolled course)
        $verifyQuery = "SELECT COUNT(*) as count 
                      FROM users u
                      JOIN teachers t ON u.user_id = t.user_id
                      JOIN courses c ON t.teacher_id = c.teacher_id
                      JOIN enrollments e ON c.course_id = e.course_id
                      WHERE e.student_id = ? AND u.user_id = ? AND u.user_role = 'teacher'";
        
        $stmt = $conn->prepare($verifyQuery);
        $stmt->bind_param("ii", $userId, $recipientId);
        $stmt->execute();
        $result = $stmt->get_result();
        $row = $result->fetch_assoc();
        
        if ($row['count'] == 0) {
            echo json_encode(['success' => false, 'message' => 'Invalid recipient']);
            exit;
        }
        
        // Insert the message
        $insertQuery = "INSERT INTO messages (sender_id, recipient_id, message_text, sent_at, is_read) 
                      VALUES (?, ?, ?, NOW(), 0)";
        
        $stmt = $conn->prepare($insertQuery);
        $stmt->bind_param("iis", $userId, $recipientId, $messageText);
        $success = $stmt->execute();
        
        if ($success) {
            $messageId = $stmt->insert_id;
            
            // Get the message details for response
            $messageQuery = "SELECT 
                            m.message_id, 
                            m.sender_id, 
                            m.recipient_id, 
                            m.message_text, 
                            m.sent_at, 
                            m.is_read,
                            u.first_name as sender_first_name,
                            u.last_name as sender_last_name
                        FROM messages m
                        JOIN users u ON m.sender_id = u.user_id
                        WHERE m.message_id = ?";
            
            $stmt = $conn->prepare($messageQuery);
            $stmt->bind_param("i", $messageId);
            $stmt->execute();
            $result = $stmt->get_result();
            $message = $result->fetch_assoc();
            
            echo json_encode([
                'success' => true, 
                'message' => 'Message sent successfully',
                'data' => [
                    'message_id' => $message['message_id'],
                    'sender_id' => $message['sender_id'],
                    'recipient_id' => $message['recipient_id'],
                    'message_text' => $message['message_text'],
                    'sent_at' => $message['sent_at'],
                    'sender_name' => $message['sender_first_name'] . ' ' . $message['sender_last_name']
                ]
            ]);
        } else {
            echo json_encode(['success' => false, 'message' => 'Failed to send message']);
        }
    } catch (Exception $e) {
        error_log("Send message error: " . $e->getMessage());
        echo json_encode(['success' => false, 'message' => 'An error occurred']);
    }
}

/**
 * Mark messages as read
 */
function markMessagesAsRead($conn, $userId) {
    if (!isset($_POST['contact_id'])) {
        echo json_encode(['success' => false, 'message' => 'Missing contact ID']);
        exit;
    }
    
    $contactId = (int)$_POST['contact_id'];
    
    try {
        $markReadQuery = "UPDATE messages 
                        SET is_read = 1 
                        WHERE sender_id = ? AND recipient_id = ? AND is_read = 0";
        
        $stmt = $conn->prepare($markReadQuery);
        $stmt->bind_param("ii", $contactId, $userId);
        $stmt->execute();
        
        echo json_encode(['success' => true, 'message' => 'Messages marked as read']);
    } catch (Exception $e) {
        error_log("Mark messages read error: " . $e->getMessage());
        echo json_encode(['success' => false, 'message' => 'An error occurred']);
    }
}
?>
