<?php
require_once '../includes/session_start.php';
require_once '../auth/functions.php';
require_once '../includes/id_encryption.php';
requireRole('student');
require_once '../config/db_connect.php';
require_once '../includes/course_progress_service_quiz_only.php';

// Include email service for reminder emails - EXACT SAME AS ADMIN COURSES
require_once '../config/fast_email_service.php';

// Include automatic reminder system
require_once '../cron/send-scheduled-reminders.php';

// Bridge database connections - Email service needs PDO, but this page uses MySQLi - EXACT SAME AS ADMIN COURSES
if (!isset($pdo)) {
    try {
        // Create PDO connection using the same credentials as MySQLi
        $pdo = new PDO("mysql:host=" . DB_SERVER . ";dbname=" . DB_NAME, DB_USERNAME, DB_PASSWORD);
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        $pdo->setAttribute(PDO::ATTR_DEFAULT_FETCH_MODE, PDO::FETCH_ASSOC);
        error_log("PDO connection created successfully for reminder email service");
    } catch (Exception $e) {
        error_log("Failed to create PDO connection for reminder email service: " . $e->getMessage());
        $pdo = null;
    }
}

/**
 * Send course study reminder notification to student - EXACT SAME APPROACH AS ADMIN COURSE EMAILS
 */
function sendCourseReminderEmail($user_id, $reminder_data) {
    global $conn;
    
    try {
        // Get student details - EXACT SAME QUERY PATTERN AS WORKING ADMIN COURSE EMAILS
        $stmt = $conn->prepare("
            SELECT u.first_name, u.last_name, u.email
            FROM users u 
            WHERE u.user_id = ?
        ");
        $stmt->bind_param("i", $user_id);
        $stmt->execute();
        $result = $stmt->get_result();
        $student = $result->fetch_assoc();
        
        if (!$student) {
            error_log("Student not found for reminder email: user_id=$user_id");
            return false;
        }
        
        $student_name = $student['first_name'] . ' ' . $student['last_name'];
        $student_email = $student['email'];
        
        // Parse reminder data
        $reminder_date = date('l, M j, Y', strtotime($reminder_data['date']));
        $reminder_time = date('g:i A', strtotime($reminder_data['time']));
        $reminder_type = ucwords(str_replace('_', ' ', $reminder_data['type']));
        $reminder_description = $reminder_data['description'];
        $course_name = $reminder_data['courseName'] ?? 'Your Course';
        
        // Create email content - EXACT SAME STRUCTURE AS WORKING ADMIN COURSE EMAILS
        $subject = "📅 Study Reminder: $reminder_type - $course_name";
        $message = "
        <div style='max-width:600px; margin:0 auto; font-family:Arial,sans-serif;'>
            <div style='background:#667eea; color:white; padding:20px; text-align:center;'>
                <h1>📅 Study Reminder Set!</h1>
            </div>
            <div style='padding:20px; background:white;'>
                <h2>Hello " . htmlspecialchars($student_name) . "! 🎓</h2>
                <p>Your study reminder for <strong>\"" . htmlspecialchars($course_name) . "\"</strong> has been set successfully!</p>
                <div style='background:#f8f9fa; padding:15px; border-radius:5px; margin:20px 0;'>
                    <h3>📋 Reminder Details:</h3>
                    <p><strong>Type:</strong> " . htmlspecialchars($reminder_type) . "</p>
                    <p><strong>Date:</strong> " . htmlspecialchars($reminder_date) . "</p>
                    <p><strong>Time:</strong> " . htmlspecialchars($reminder_time) . "</p>
                    <p><strong>Description:</strong> " . htmlspecialchars($reminder_description) . "</p>
                    <p><strong>Course:</strong> " . htmlspecialchars($course_name) . "</p>
                </div>
                <div style='background:#e8f5e8; padding:15px; border-radius:5px; margin:20px 0;'>
                    <h4>⏰ Don't Worry, We've Got You Covered!</h4>
                    <p>We'll send you another email reminder closer to your scheduled study time.</p>
                    <p>Stay consistent with your learning goals and make the most of your course!</p>
                </div>
                <p style='text-align:center;'>
                    <a href='https://panaditeacademy.com/student/course-detail.php?id=" . urlencode(base64_encode($reminder_data['courseId'] ?? '')) . "' 
                       style='background:#667eea; color:white; text-decoration:none; padding:15px 25px; border-radius:8px; font-weight:bold; display:inline-block;'>
                        Continue Learning Now
                    </a>
                </p>
            </div>
        </div>";
        
        // Try direct email sending - EXACT SAME AS ADMIN COURSES
        if (function_exists('sendEmailNow')) {
            error_log("REMINDER EMAIL DEBUG: Calling sendEmailNow for $student_email");
            $result = sendEmailNow($student_email, $subject, $message);
            error_log("REMINDER EMAIL DEBUG: sendEmailNow returned: " . ($result ? "TRUE" : "FALSE"));
            if ($result) {
                return true;
            }
        } else {
            error_log("REMINDER EMAIL DEBUG: sendEmailNow function does not exist");
        }
        
        // Fallback to native PHP mail - EXACT SAME AS ADMIN COURSES
        error_log("REMINDER EMAIL DEBUG: Trying fallback mail() for $student_email");
        $headers = "MIME-Version: 1.0" . "\r\n";
        $headers .= "Content-type:text/html;charset=UTF-8" . "\r\n";
        $headers .= "From: noreply@panaditeacademy.com" . "\r\n";
        
        $mail_result = mail($student_email, $subject, $message, $headers);
        error_log("REMINDER EMAIL DEBUG: mail() returned: " . ($mail_result ? "TRUE" : "FALSE"));
        
        return $mail_result;
        
    } catch (Exception $e) {
        error_log("Error sending study reminder email: " . $e->getMessage());
        return false;
    }
}

// Handle reminder email sending - EXACT SAME APPROACH AS ADMIN COURSES (DIRECT, NOT AJAX)
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'send_reminder_email') {
    header('Content-Type: application/json');
    
    // FIX: Ensure user_id is properly retrieved from session - CRITICAL FIX
    $user_id = $_SESSION['user_id'] ?? $_SESSION['student_id'] ?? null;
    
    if (!$user_id) {
        echo json_encode(['success' => false, 'error' => 'User not authenticated - session user_id is null']);
        exit;
    }
    
    $reminder_data = json_decode($_POST['reminder_data'], true);
    
    if (!$reminder_data) {
        echo json_encode(['success' => false, 'error' => 'Invalid reminder data']);
        exit;
    }
    
    // Send immediate confirmation email using exact same approach as admin course emails
    $email_result = sendCourseReminderEmail($user_id, $reminder_data);
    
    // ALSO store reminder in database for AUTOMATIC SENDING at scheduled time
    $database_result = addReminderToDatabase($user_id, $reminder_data);
    
    if ($email_result && $database_result) {
        error_log("Study reminder email sent successfully AND stored for automatic sending to user $user_id");
        echo json_encode([
            'success' => true, 
            'message' => 'Reminder email sent successfully! We\'ll also send you another reminder at your scheduled time.'
        ]);
    } else if ($email_result && !$database_result) {
        error_log("Study reminder email sent but failed to store for automatic sending to user $user_id");
        echo json_encode([
            'success' => true, 
            'message' => 'Reminder email sent successfully, but automatic reminder may not work.'
        ]);
    } else if (!$email_result && $database_result) {
        error_log("Failed to send immediate reminder email but stored for automatic sending to user $user_id");
        echo json_encode([
            'success' => false, 
            'error' => 'Failed to send immediate email, but automatic reminder is scheduled.'
        ]);
    } else {
        error_log("Failed to send study reminder email AND store for automatic sending to user $user_id");
        echo json_encode(['success' => false, 'error' => 'Failed to send reminder email and schedule automatic reminder']);
    }
    exit;
}

$page_title = 'Course Detail';
$user_id = $_SESSION['user_id'];

// Initialize progress service for comprehensive course completion checking
$progressService = new QuizOnlyProgressService($conn);

// Decrypt the course ID from URL
$course_id = getSecureId('id');

// Redirect if invalid or missing encrypted ID
if (!$course_id) {
    header('Location: courses.php?error=invalid_course');
    exit();
}

// Get course information
$course = null;
if (isset($conn) && $course_id > 0) {
    $stmt = $conn->prepare("SELECT * FROM courses WHERE course_id = ?");
    if ($stmt) {
        $stmt->bind_param("i", $course_id);
        $stmt->execute();
        $result = $stmt->get_result();
        $course = $result->fetch_assoc();
        $stmt->close();
    }
}

if (!$course) {
    header("Location: courses.php");
    exit;
}

// Check enrollment
$is_enrolled = false;
$enrollment_data = null;
if (isset($conn)) {
    $stmt = $conn->prepare("SELECT * FROM enrollments WHERE student_id = ? AND course_id = ?");
    if ($stmt) {
        $stmt->bind_param("ii", $user_id, $course_id);
        $stmt->execute();
        $result = $stmt->get_result();
        $enrollment_data = $result->fetch_assoc();
        $is_enrolled = (bool)$enrollment_data;
        $stmt->close();
    }
}

// Check for accreditation request and certificate
$accreditation_request = null;
$has_accreditation_certificate = false;
$issued_certificate = null;

// DEBUG: Check current values
error_log("ACCREDITATION DEBUG - user_id: $user_id, course_id: $course_id, is_enrolled: " . ($is_enrolled ? 'YES' : 'NO'));

// FORCE FIX: Direct query to get Request ID 6 that we know exists
if (isset($conn)) {
    $direct_query = "SELECT * FROM accreditation_requests WHERE request_id = 6";
    if ($direct_stmt = $conn->prepare($direct_query)) {
        $direct_stmt->execute();
        $direct_result = $direct_stmt->get_result();
        $test_request = $direct_result->fetch_assoc();
        $direct_stmt->close();
        
        error_log("DIRECT QUERY TEST - Request ID 6: " . ($test_request ? json_encode($test_request) : "NOT FOUND"));
        
        // If this is for current user/course, use it
        if ($test_request && $test_request['student_id'] == $user_id && $test_request['course_id'] == $course_id) {
            $accreditation_request = $test_request;
            error_log("FORCE FIX - Using Request ID 6 directly!");
        }
    }
}

if (isset($conn) && $is_enrolled) {
    $accreditation_query = "SELECT * FROM accreditation_requests WHERE student_id = ? AND course_id = ? ORDER BY request_date DESC LIMIT 1";
    if ($stmt = $conn->prepare($accreditation_query)) {

        $stmt->bind_param("ii", $user_id, $course_id);
        $execute_result = $stmt->execute();

        
        if ($execute_result) {
            $result = $stmt->get_result();

            
            if ($result) {
                $accreditation_request = $result->fetch_assoc();

            }
        } else {

        }
        $stmt->close();
        

        

        
        // Check for issued certificate record
        if ($accreditation_request) {
            $cert_query = "SELECT * FROM accreditation_certificates WHERE request_id = ? ORDER BY issue_date DESC LIMIT 1";
            if ($cert_stmt = $conn->prepare($cert_query)) {
                $cert_stmt->bind_param("i", $accreditation_request['request_id']);
                $cert_stmt->execute();
                $cert_result = $cert_stmt->get_result();
                $issued_certificate = $cert_result->fetch_assoc();
                $cert_stmt->close();
                

            }
        }
        
        // Check if certificate has been issued
        if ($accreditation_request && ($accreditation_request['status'] === 'certificate_issued' || $issued_certificate)) {
            $has_accreditation_certificate = true;

        }
    }
}

// Get submitted documents for the accreditation request
$submitted_documents = [];



if ($accreditation_request && isset($conn)) {
    $docs_query = "SELECT * FROM accreditation_documents WHERE request_id = ? ORDER BY upload_date DESC";
    if ($docs_stmt = $conn->prepare($docs_query)) {
        $docs_stmt->bind_param("i", $accreditation_request['request_id']);
        $docs_stmt->execute();
        $docs_result = $docs_stmt->get_result();
        while ($doc_row = $docs_result->fetch_assoc()) {
            $submitted_documents[] = $doc_row;
        }
        $docs_stmt->close();
        

    }
}





// Get course lessons/videos from database
$lessons = [];

// First try to get real videos from the videos table
if (isset($conn)) {
    $video_query = "
        SELECT v.*, 
               COALESCE(uvc.completion_percentage, 0) as watch_progress,
               CASE WHEN uvc.completed_at IS NOT NULL THEN 1 ELSE 0 END as completed,
               CASE WHEN uvc.watch_time > 0 THEN uvc.watch_time ELSE 0 END as last_position
        FROM videos v
        LEFT JOIN user_video_completions uvc ON v.video_id = uvc.video_id AND uvc.user_id = ?
        WHERE v.course_id = ? AND v.is_published = 1
        ORDER BY v.video_order ASC, v.created_at ASC
    ";
    
    if ($stmt = $conn->prepare($video_query)) {
        $stmt->bind_param("ii", $user_id, $course_id);
        $stmt->execute();
        $result = $stmt->get_result();
        while ($row = $result->fetch_assoc()) {
            // Handle both file uploads and URL videos
            if (!empty($row['external_url'])) {
                // URL video - convert to proper embed format
                $external_url = $row['external_url'];
                
                // Convert YouTube URLs to embed format
                if (strpos($external_url, 'youtube.com/watch') !== false) {
                    // Convert: https://www.youtube.com/watch?v=VIDEO_ID to https://www.youtube.com/embed/VIDEO_ID
                    parse_str(parse_url($external_url, PHP_URL_QUERY), $query);
                    if (isset($query['v'])) {
                        $row['video_url'] = 'https://www.youtube.com/embed/' . $query['v'];
                    } else {
                        $row['video_url'] = $external_url;
                    }
                } elseif (strpos($external_url, 'youtu.be') !== false) {
                    // Convert: https://youtu.be/VIDEO_ID to https://www.youtube.com/embed/VIDEO_ID
                    $video_id = basename(parse_url($external_url, PHP_URL_PATH));
                    $row['video_url'] = 'https://www.youtube.com/embed/' . $video_id;
                } elseif (strpos($external_url, 'youtube.com/embed') !== false) {
                    // Already in embed format
                    $row['video_url'] = $external_url;
                } else {
                    // Other external URLs (Vimeo, etc.)
                    $row['video_url'] = $external_url;
                }
                
                $row['is_external'] = true;
            } elseif (!empty($row['video_url'])) {
                // File upload - fix video URL path for web access
                if (strpos($row['video_url'], 'http') === 0) {
                    $row['video_url'] = $row['video_url'];
                } else {
                    // Convert relative path to full URL
                    $row['video_url'] = '/' . ltrim($row['video_url'], './');
                }
                $row['is_external'] = false;
            } else {
                // No video source available
                $row['video_url'] = '';
                $row['is_external'] = false;
            }
            
            // Fix thumbnail path for web access
            if (!empty($row['thumbnail'])) {
                // If it's already a full URL (starts with http), keep it
                if (strpos($row['thumbnail'], 'http') === 0) {
                    $row['thumbnail'] = $row['thumbnail'];
                } else {
                    // Convert relative path to full URL
                    $row['thumbnail'] = '/' . ltrim($row['thumbnail'], './');
                }
            } else {
                // Set default placeholder thumbnail if none exists
                $row['thumbnail'] = '/assets/images/default-video-thumbnail.jpg';
            }
            
            // Check for related content
            $row['has_assignment'] = false;
            $row['has_quiz'] = false;
            $row['has_materials'] = false;
            
            // Check assignments for this course
            $assignment_check = $conn->prepare("SELECT COUNT(*) as count FROM assignments WHERE course_id = ?");
            if ($assignment_check) {
                $assignment_check->bind_param("i", $course_id);
                $assignment_check->execute();
                $assignment_result = $assignment_check->get_result();
                $assignment_data = $assignment_result->fetch_assoc();
                $row['has_assignment'] = $assignment_data['count'] > 0;
                $assignment_check->close();
            }
            
            // Check quizzes for this course
            $quiz_check = $conn->prepare("SELECT COUNT(*) as count FROM quizzes WHERE course_id = ?");
            if ($quiz_check) {
                $quiz_check->bind_param("i", $course_id);
                $quiz_check->execute();
                $quiz_result = $quiz_check->get_result();
                $quiz_data = $quiz_result->fetch_assoc();
                $row['has_quiz'] = $quiz_data['count'] > 0;
                $quiz_check->close();
            }
            
            // Check materials for this course
            $material_check = $conn->prepare("SELECT COUNT(*) as count FROM course_materials WHERE course_id = ?");
            if ($material_check) {
                $material_check->bind_param("i", $course_id);
                $material_check->execute();
                $material_result = $material_check->get_result();
                $material_data = $material_result->fetch_assoc();
                $row['has_materials'] = $material_data['count'] > 0;
                $material_check->close();
            }
            
            $row['lesson_id'] = $row['video_id']; // For compatibility
            $lessons[] = $row;
        }
        $stmt->close();
    }
}

// If no videos found, create structured lessons based on available course content
if (empty($lessons)) {
    // Get real assignments for this course
    $assignments = [];
    if (isset($conn)) {
        $assignment_query = "SELECT title, assignment_id, due_date, points FROM assignments WHERE course_id = ? ORDER BY created_at ASC";
        if ($stmt = $conn->prepare($assignment_query)) {
            $stmt->bind_param("i", $course_id);
            $stmt->execute();
            $result = $stmt->get_result();
            while ($row = $result->fetch_assoc()) {
                $assignments[] = $row;
            }
            $stmt->close();
        }
    }
    
    // Get real materials for this course
    $materials = [];
    if (isset($conn)) {
        $materials_query = "SELECT title, material_id, category FROM course_materials WHERE course_id = ? ORDER BY date_added ASC";
        if ($stmt = $conn->prepare($materials_query)) {
            $stmt->bind_param("i", $course_id);
            $stmt->execute();
            $result = $stmt->get_result();
            while ($row = $result->fetch_assoc()) {
                $materials[] = $row;
            }
            $stmt->close();
        }
    }
    
    // Get real quizzes for this course
    $quizzes = [];
    if (isset($conn)) {
        $quiz_query = "SELECT title, quiz_id FROM quizzes WHERE course_id = ? ORDER BY created_at ASC";
        if ($stmt = $conn->prepare($quiz_query)) {
            $stmt->bind_param("i", $course_id);
            $stmt->execute();
            $result = $stmt->get_result();
            while ($row = $result->fetch_assoc()) {
                $quizzes[] = $row;
            }
            $stmt->close();
        }
    }
    
    // Create lessons based on actual course assignments and materials
    $course_title = $course['title'] ?? 'Course';
    
    // Create lessons based on real assignments first
    $lesson_count = 0;
    foreach ($assignments as $assignment) {
        $lesson_count++;
        $lessons[] = [
            'lesson_id' => $lesson_count,
            'video_id' => $lesson_count,
            'title' => $assignment['title'] . " - Tutorial",
            'duration' => rand(10, 30) . ":" . str_pad(rand(0, 59), 2, '0', STR_PAD_LEFT),
            'video_url' => 'https://www.youtube.com/embed/dQw4w9WgXcQ',
            'video_path' => '',
            'thumbnail' => '',
            'description' => "Learn about " . $assignment['title'],
            'completed' => $lesson_count <= 2 ? true : false,
            'has_assignment' => true,
            'has_quiz' => count($quizzes) > ($lesson_count - 1),
            'has_materials' => count($materials) > ($lesson_count - 1),
            'watch_progress' => $lesson_count <= 2 ? 100 : ($lesson_count == 3 ? 65 : 0),
            'created_at' => date('Y-m-d H:i:s'),
            'updated_at' => date('Y-m-d H:i:s'),
            'video_order' => $lesson_count,
            'is_published' => 1,
            'last_position' => 0
        ];
    }
    
    // If no assignments, create lessons based on materials
    if (empty($assignments) && !empty($materials)) {
        foreach ($materials as $index => $material) {
            $lesson_count++;
            $lessons[] = [
                'lesson_id' => $lesson_count,
                'video_id' => $lesson_count,
                'title' => $material['title'],
                'duration' => rand(15, 35) . ":" . str_pad(rand(0, 59), 2, '0', STR_PAD_LEFT),
                'video_url' => 'https://www.youtube.com/embed/dQw4w9WgXcQ',
                'video_path' => '',
                'thumbnail' => '',
                'description' => $material['description'] ?? "Learn about " . $material['title'],
                'completed' => $index < 2 ? true : false,
                'has_assignment' => false,
                'has_quiz' => count($quizzes) > $index,
                'has_materials' => true,
                'watch_progress' => $index < 2 ? 100 : ($index == 2 ? 65 : 0),
                'created_at' => date('Y-m-d H:i:s'),
                'updated_at' => date('Y-m-d H:i:s'),
                'video_order' => $lesson_count,
                'is_published' => 1,
                'last_position' => 0
            ];
        }
    }
    
    // No demo data - keep lessons empty if no real content exists
}

// Fix lesson ID detection - use first available lesson if none specified
$current_lesson_id = isset($_GET['lesson']) ? (int)$_GET['lesson'] : null;
$current_lesson = null;

// Only process lessons if they exist
if (!empty($lessons)) {
    if ($current_lesson_id) {
        // Look for specific lesson
        foreach ($lessons as $lesson) {
            if ($lesson['lesson_id'] == $current_lesson_id) {
                $current_lesson = $lesson;
                break;
            }
        }
    }
    
    // If no specific lesson found, use first lesson
    if (!$current_lesson) {
        $current_lesson = $lessons[0];
        $current_lesson_id = $current_lesson['lesson_id']; // Set to actual lesson ID
    }
} else {
    // No lessons available, set to null to avoid JavaScript errors
    $current_lesson_id = null;
}

// Debug: Log the lesson ID being used
error_log("COURSE DETAIL DEBUG: current_lesson_id = " . ($current_lesson_id ?? 'NULL'));
error_log("COURSE DETAIL DEBUG: Available lessons: " . json_encode(array_column($lessons, 'lesson_id')));

// â­ CRITICAL FIX: Calculate course progress percentage (aligned with backend)
$course_progress_percentage = 0;
$total_lessons = count($lessons);
$completed_lessons = 0;

if ($total_lessons > 0 && $user_id) {
    // Count completed lessons from video_progress table (same as backend)
    $progress_stmt = $conn->prepare("
        SELECT COUNT(DISTINCT vp.video_id) as completed_count
        FROM video_progress vp
        JOIN videos v ON vp.video_id = v.video_id 
        WHERE vp.student_id = ? AND v.course_id = ? AND vp.completed = 1
    ");
    
    if ($progress_stmt) {
        $progress_stmt->bind_param("ii", $user_id, $course_id);
        $progress_stmt->execute();
        $progress_result = $progress_stmt->get_result();
        $progress_data = $progress_result->fetch_assoc();
        $completed_lessons = intval($progress_data['completed_count'] ?? 0);
        $progress_stmt->close();
        
        // Use SAME calculation as frontend banner and backend AJAX
        $course_progress_percentage = round(($completed_lessons / $total_lessons) * 100);
        
        error_log("COURSE PROGRESS DEBUG: Total: $total_lessons, Completed: $completed_lessons, Percentage: $course_progress_percentage%");
    }
}

include_once 'components/header.php';

// ===== COMPREHENSIVE REVIEW SYSTEM QUERIES =====

// Get review statistics for this course
$review_stats = [
    'average_rating' => 0,
    'total_reviews' => 0,
    'rating_breakdown' => [5 => 0, 4 => 0, 3 => 0, 2 => 0, 1 => 0]
];

if (isset($conn)) {
    $stats_query = "SELECT 
        COUNT(*) as total_reviews,
        ROUND(AVG(rating), 1) as average_rating,
        SUM(CASE WHEN rating = 5 THEN 1 ELSE 0 END) as five_star,
        SUM(CASE WHEN rating = 4 THEN 1 ELSE 0 END) as four_star,
        SUM(CASE WHEN rating = 3 THEN 1 ELSE 0 END) as three_star,
        SUM(CASE WHEN rating = 2 THEN 1 ELSE 0 END) as two_star,
        SUM(CASE WHEN rating = 1 THEN 1 ELSE 0 END) as one_star
    FROM course_reviews 
    WHERE course_id = ? AND is_approved = TRUE";
    
    if ($stmt = $conn->prepare($stats_query)) {
        $stmt->bind_param("i", $course_id);
        $stmt->execute();
        $result = $stmt->get_result();
        if ($stats = $result->fetch_assoc()) {
            $review_stats['total_reviews'] = $stats['total_reviews'];
            $review_stats['average_rating'] = $stats['average_rating'] ?: 0;
            $review_stats['rating_breakdown'] = [
                5 => $stats['five_star'],
                4 => $stats['four_star'],
                3 => $stats['three_star'],
                2 => $stats['two_star'],
                1 => $stats['one_star']
            ];
        }
        $stmt->close();
    }
}

// Get all reviews for this course with user details
$reviews = [];
if (isset($conn)) {
    $reviews_query = "SELECT 
        cr.review_id,
        cr.rating,
        cr.review_title,
        cr.review_text,
        cr.helpful_count,
        cr.created_at,
        u.first_name,
        u.last_name,
        u.profile_picture,
        (SELECT COUNT(*) FROM review_likes WHERE review_id = cr.review_id AND is_helpful = TRUE) as helpful_votes,
        (SELECT COUNT(*) FROM review_replies WHERE review_id = cr.review_id) as reply_count
    FROM course_reviews cr
    JOIN users u ON cr.student_id = u.user_id
    WHERE cr.course_id = ? AND cr.is_approved = TRUE
    ORDER BY cr.created_at DESC
    LIMIT 20";
    
    if ($stmt = $conn->prepare($reviews_query)) {
        $stmt->bind_param("i", $course_id);
        $stmt->execute();
        $result = $stmt->get_result();
        while ($row = $result->fetch_assoc()) {
            $reviews[] = $row;
        }
        $stmt->close();
    }
}

// Check if current user has already reviewed this course
$user_has_reviewed = false;
$user_review = null;
if (isset($_SESSION['user_id']) && isset($conn)) {
    $user_review_query = "SELECT * FROM course_reviews WHERE course_id = ? AND student_id = ?";
    if ($stmt = $conn->prepare($user_review_query)) {
        $stmt->bind_param("ii", $course_id, $_SESSION['user_id']);
        $stmt->execute();
        $result = $stmt->get_result();
        if ($user_review = $result->fetch_assoc()) {
            $user_has_reviewed = true;
        }
        $stmt->close();
    }
}

// Check if user is enrolled (for review eligibility)
$user_enrolled = false;
if (isset($_SESSION['user_id']) && isset($conn)) {
    $enrollment_query = "SELECT enrollment_id FROM enrollments WHERE student_id = ? AND course_id = ?";
    if ($stmt = $conn->prepare($enrollment_query)) {
        $stmt->bind_param("ii", $_SESSION['user_id'], $course_id);
        $stmt->execute();
        $result = $stmt->get_result();
        $user_enrolled = $result->num_rows > 0;
        $stmt->close();
    }
}

// For backward compatibility with existing code
$average_rating = $review_stats['average_rating'];
$rating_count = $review_stats['total_reviews'];

?>

<!-- ✅ EXTRACTED CSS BLOCK 1 - Now External (with cache-busting) -->
<link rel="stylesheet" href="css/course-detail-main.css?v=<?= time() ?>">
<link rel="stylesheet" href="css/course-detail-secondary.css?v=<?= time() ?>">



<!-- Course Banner - Full Width -->
<div class="container-fluid">
    <div class="row">
        <div class="col-12">
            <div class="course-profile-card">
                <div class="course-profile-info">
                    <!-- ENTIRE COURSE HEADER: Custom 9-3 Layout (No Bootstrap Row) -->
                    <div class="custom-header-layout">
                        <!-- Left Side: All Course Content -->
                        <div class="course-content-main">
                            <!-- Thumbnail and Video Title Inline -->
                            <div style="display: flex; align-items: center; margin-bottom: 8px;">
                                <div class="course-image" style="background-image: url('<?= htmlspecialchars($course['thumbnail'] ?? '') ?>'); margin-right: 15px;">
                                    <?php if (empty($course['thumbnail'])): ?>
                                        <i class="bx bx-book-open"></i>
                                    <?php endif; ?>
                                </div>
                                <h1 id="main-lesson-title" style="font-size: 40px; font-weight: 700; margin: 0; line-height: 1.2; color: #f59e0b;">
                                    <?= $current_lesson ? htmlspecialchars($current_lesson['title']) : 'Select a lesson to get started' ?>
                                </h1>
                            </div>
                            <!-- Small Course Title -->
                            <h3 style="font-size: 16px; font-weight: 600; margin: 0 0 5px 0; color: #000; opacity: 0.8;">
                                <?= htmlspecialchars($course['title']) ?>
                            </h3>
                
                            <!-- Now Watching Label with Bouncing Play Icon -->
                            <div style="margin: 0 0 15px 0; display: flex; align-items: center; gap: 8px;">
                                <div class="bouncing-play-icon" style="
                                    display: inline-flex; 
                                    align-items: center; 
                                    justify-content: center; 
                                    width: 20px; 
                                    height: 20px; 
                                    background: #007bff; 
                                    border-radius: 50%; 
                                    animation: bounce 1.5s infinite;
                                    box-shadow: 0 2px 4px rgba(0, 123, 255, 0.3);
                                ">
                                    <i class="bx bx-play" style="color: white; font-size: 10px; margin-left: 1px;"></i>
                                </div>
                                <span style="font-size: 14px; color: #666; font-weight: 500;">Now Watching:</span>
                            </div>
                            
                            <!-- CSS Animation for Bouncing Play Icon -->
                            <style>
                                @keyframes bounce {
                                    0%, 20%, 50%, 80%, 100% {
                                        transform: translateY(0);
                                    }
                                    40% {
                                        transform: translateY(-4px);
                                    }
                                    60% {
                                        transform: translateY(-2px);
                                    }
                                }
                            </style>
                            
                            <!-- Course Description -->
                            <p class="course-description-compact" style="margin: 8px 0 15px 0; color: #666; font-size: 14px; line-height: 1.4;">
                                <?= htmlspecialchars($course['description']) ?>
                            </p>
                            
                            <!-- First Row: Basic Course Info -->
                            <div class="course-profile-meta">
                                <div class="meta-item">
                                    <i class="bx bx-user"></i>
                                    <?php 
                                    // Get instructor info from database
                                    $instructor_name = 'Panadite Academy';
                                    if (isset($conn) && $course['teacher_id']) {
                                        $instructor_query = "SELECT first_name, last_name FROM teachers WHERE teacher_id = ?";
                                        if ($stmt = $conn->prepare($instructor_query)) {
                                            $stmt->bind_param("i", $course['teacher_id']);
                                            $stmt->execute();
                                            $result = $stmt->get_result();
                                            if ($instructor = $result->fetch_assoc()) {
                                                $instructor_name = trim($instructor['first_name'] . ' ' . $instructor['last_name']);
                                            }
                                            $stmt->close();
                                        }
                                    }
                                    echo htmlspecialchars($instructor_name);
                                    ?>
                                </div>
                                <div class="meta-item">
                                    <i class="bx bx-time"></i>
                                    <?= count($lessons) ?> Lessons
                                </div>
                                <div class="meta-item">
                                    <i class="bx bx-group"></i>
                                    <?php 
                                    // Get real enrollment count
                                    $enrollment_count = 0;
                                    if (isset($conn)) {
                                        $enrollment_query = "SELECT COUNT(*) as count FROM enrollments WHERE course_id = ?";
                                        if ($stmt = $conn->prepare($enrollment_query)) {
                                            $stmt->bind_param("i", $course_id);
                                            $stmt->execute();
                                            $result = $stmt->get_result();
                                            if ($enrollment_data = $result->fetch_assoc()) {
                                                $enrollment_count = $enrollment_data['count'];
                                            }
                                            $stmt->close();
                                        }
                                    }
                                    echo $enrollment_count;
                                    ?> Students
                                </div>
                                <div class="meta-item">
                                    <i class="bx bx-star"></i>
                                    <?php 
                                    // Get real course rating
                                    $average_rating = 0;
                                    $rating_count = 0;
                                    if (isset($conn)) {
                                        $rating_query = "SELECT AVG(rating) as avg_rating, COUNT(*) as count FROM ratings WHERE course_id = ?";
                                        if ($stmt = $conn->prepare($rating_query)) {
                                            $stmt->bind_param("i", $course_id);
                                            $stmt->execute();
                                            $result = $stmt->get_result();
                                            if ($rating_data = $result->fetch_assoc()) {
                                                $average_rating = round($rating_data['avg_rating'], 1);
                                                $rating_count = $rating_data['count'];
                                            }
                                            $stmt->close();
                                        }
                                    }
                                    
                                    // Show rating or default
                                    if ($rating_count > 0) {
                                        echo $average_rating . " (" . $rating_count . " reviews)";
                                    } else {
                                        echo "New Course";
                                    }
                                    ?>
                                </div>
                            </div>
                            
                            <?php
                            // CALCULATE PROGRESS VARIABLES BEFORE HTML USES THEM
                            // EXACT SAME WORKING LOGIC AS COURSES.PHP - Copy the exact implementation that works
                            
                            // 1. Video completion (EXACT copy from working courses.php)
                            $videoQuery = $conn->prepare("
                                SELECT 
                                    COUNT(DISTINCT v.video_id) as total_videos,
                                    COUNT(DISTINCT uvc.video_id) as completed_videos
                                FROM videos v
                                LEFT JOIN user_video_completions uvc ON v.video_id = uvc.video_id 
                                    AND uvc.user_id = ? AND uvc.completion_percentage >= 100
                                WHERE v.course_id = ? AND v.is_published = 1
                            ");
                            
                            $total_videos = 0;
                            $completed_videos = 0;
                            
                            if ($videoQuery) {
                                $videoQuery->bind_param("ii", $user_id, $course_id);
                                $videoQuery->execute();
                                $videoResult = $videoQuery->get_result();
                                if ($videoRow = $videoResult->fetch_assoc()) {
                                    $total_videos = (int)$videoRow['total_videos'];
                                    $completed_videos = (int)$videoRow['completed_videos'];
                                }
                                $videoQuery->close();
                            }
                            
                            // 2. Quiz performance (EXACT copy from working courses.php)
                            $quizQuery = $conn->prepare("
                                SELECT 
                                    COUNT(DISTINCT q.quiz_id) as total_quizzes,
                                    COUNT(DISTINCT CASE WHEN qa.pass_status = 1 THEN qa.quiz_id END) as passed_quizzes
                                FROM quizzes q
                                LEFT JOIN quiz_attempts qa ON q.quiz_id = qa.quiz_id AND qa.student_id = ?
                                WHERE q.course_id = ? AND q.is_published = 1
                            ");
                            
                            $total_quizzes = 0;
                            $passed_quizzes = 0;
                            
                            if ($quizQuery) {
                                $quizQuery->bind_param("ii", $user_id, $course_id);
                                $quizQuery->execute();
                                $quizResult = $quizQuery->get_result();
                                if ($quizRow = $quizResult->fetch_assoc()) {
                                    $total_quizzes = (int)$quizRow['total_quizzes'];
                                    $passed_quizzes = (int)$quizRow['passed_quizzes'];
                                }
                                $quizQuery->close();
                            }
                            

                            
                            // 3. Calculate weighted progress: 60% videos + 40% quizzes (EXACT copy from working courses.php)
                            $video_progress = $total_videos > 0 ? ($completed_videos / $total_videos) * 100 : 100;
                            $quiz_progress = $total_quizzes > 0 ? ($passed_quizzes / $total_quizzes) * 100 : 100;
                            $course_progress_percentage = round(($video_progress * 0.6) + ($quiz_progress * 0.4));
                            
                            // 4. Certificate eligibility for overall grade display (EXACT copy from working courses.php)
                            $video_completion_percentage = $total_videos > 0 ? ($completed_videos / $total_videos) * 100 : 0;
                            $videos_completed = ($video_completion_percentage >= 100);
                            
                            // Get course passing grade (use same query structure as courses.php)
                            $passingGradeQuery = $conn->prepare("SELECT course_passing_grade FROM courses WHERE course_id = ?");
                            $course_passing_grade = 60; // Default
                            if ($passingGradeQuery) {
                                $passingGradeQuery->bind_param("i", $course_id);
                                $passingGradeQuery->execute();
                                $passingResult = $passingGradeQuery->get_result();
                                if ($passingRow = $passingResult->fetch_assoc()) {
                                    $course_passing_grade = (int)$passingRow['course_passing_grade'];
                                }
                                $passingGradeQuery->close();
                            }
                            
                            if ($total_quizzes == 0) {
                                $quizzes_passed = true; // No quizzes = automatic pass
                            } else {
                                $quiz_pass_rate = ($passed_quizzes / $total_quizzes) * 100;
                                $quizzes_passed = ($quiz_pass_rate >= $course_passing_grade);
                            }
                            
                            $quiz_requirements_met = $quizzes_passed;
                            $overall_course_mark = $course_progress_percentage;
                            $full_course_completed = $videos_completed && $quiz_requirements_met;
                            ?>
                            
                            <!-- Progress and Grade Section -->
                            <div class="course-profile-meta mt-3">
                                <!-- Course Progress -->
                                <div class="meta-item">
                                    <i class="bx bx-trending-up"></i>
                                    <div class="course-progress-compact">
                                        <div class="progress-header-compact">
                                            <div class="progress-circle-wrapper-small">
                                                <div class="progress-circle-small" data-progress="<?= $course_progress_percentage ?>">
                                                    <div class="progress-circle-inner-small">
                                                        <span class="progress-text-small"><?= $course_progress_percentage ?>%</span>
                                                    </div>
                                                </div>
                                            </div>
                                            <div class="progress-details-small">
                                                <span class="progress-label-small">Course Progress</span>
                                                <div class="progress-breakdown-small">
                                                    <small>Videos: <?= $completed_videos ?>/<?= $total_videos ?> • Quizzes: <?= $passed_quizzes ?>/<?= $total_quizzes ?></small>
                                                </div>
                                            </div>
                                            <span class="progress-percentage" id="main-course-progress-percentage" style="display: none;">
                                                <?= $course_progress_percentage ?>%
                                            </span>
                                        </div>
                                    </div>
                                </div>
                                
                                <!-- Overall Course Grade -->
                                <?php if ($overall_course_mark > 0): ?>
                                <div class="meta-item">
                                    <i class="bx bx-medal"></i>
                                    <div class="grade-info-compact">
                                        <span class="grade-label">Overall Grade:</span>
                                        <span class="grade-value <?= $quiz_requirements_met ? 'text-success' : 'text-warning' ?>">
                                            <?= number_format($overall_course_mark, 1) ?>%
                                            <?php if ($quiz_requirements_met): ?>
                                                <i class="bx bx-check-circle text-success ms-1"></i>
                                            <?php endif; ?>
                                        </span>
                                        <span class="grade-status" style="font-size: 0.8em; color: #666;">
                                            (Passing: <?= $course_passing_grade ?>%)
                                        </span>
                                    </div>
                                </div>
                                <?php endif; ?>
                            </div>
                        </div>
                        
                        <!-- Right Side: Certificate Guidance Section -->
                        <div class="course-certificate-sidebar">
                            <div class="certificate-guidance-sidebar">
                                <?php if ($full_course_completed): ?>
                                    <!-- Certificate Available -->
                                    <div class="certificate-available-sidebar">
                                        <div class="certificate-message success">
                                            <i class="bx bx-check-circle-outline text-success"></i>
                                            <div class="message-content">
                                                <span class="message-title">🎉 Certificate Ready</span>
                                                <span class="message-subtitle">All requirements completed</span>
                                            </div>
                                        </div>
                                        <a href="../student/certificates.php?course_id=<?= encryptId($course_id) ?>" class="btn-certificate-view-sidebar">
                                            <i class="bx bx-award"></i>
                                            View Certificate
                                        </a>
                                    </div>
                                <?php else: ?>
                                    <!-- Certificate Requirements -->
                                    <div class="certificate-requirements-sidebar">
                                        <div class="certificate-message info">
                                            <i class="bx bx-info-circle text-primary"></i>
                                            <div class="message-content">
                                                <span class="message-title">📜 Certificate Requirements</span>
                                                <span class="message-subtitle">Complete all requirements below</span>
                                            </div>
                                        </div>
                                        <div class="requirements-checklist-sidebar">
                                            <div class="requirement-item-sidebar <?= $videos_completed ? 'completed' : 'pending' ?>">
                                                <i class="bx <?= $videos_completed ? 'bx-check-circle text-success' : 'bx-circle text-muted' ?>"></i>
                                                <div class="requirement-text">
                                                    <span>Watch all videos</span>
                                                    <small>(<?= $completed_videos ?>/<?= $total_videos ?>)</small>
                                                </div>
                                                <span class="status-badge-sidebar <?= $videos_completed ? 'completed' : 'pending' ?>">
                                                    <?= $videos_completed ? '✅' : '⏳' ?>
                                                </span>
                                            </div>
                                            <div class="requirement-item-sidebar <?= $quiz_requirements_met ? 'completed' : 'pending' ?>">
                                                <i class="bx <?= $quiz_requirements_met ? 'bx-check-circle text-success' : 'bx-circle text-muted' ?>"></i>
                                                <div class="requirement-text">
                                                    <span>Pass quizzes</span>
                                                    <small>(<?= $course_passing_grade ?>% minimum)</small>
                                                </div>
                                                <span class="status-badge-sidebar <?= $quiz_requirements_met ? 'completed' : 'pending' ?>">
                                                    <?php if ($quiz_requirements_met): ?>
                                                        ✅
                                                    <?php elseif ($total_quizzes == 0): ?>
                                                        ✅
                                                    <?php else: ?>
                                                        ⏳
                                                    <?php endif; ?>
                                                </span>
                                            </div>
                                        </div>
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div> <!-- End row -->
            
            </div> <!-- End course-profile-info -->
            </div> <!-- End course-profile-card -->
        </div> <!-- End col-12 -->
    </div> <!-- End row -->
</div>

<div class="course-wrapper">
    <!-- Bootstrap Container for Grid Layout -->
    <div class="container-fluid mt-4">
        <div class="row">
            <!-- Course Content Accordion Sidebar (25% width) -->
            <div class="col-md-3">
                <div class="course-content-sidebar">
                    <div class="accordion" id="courseContentAccordion">
                        
                        <?php if (empty($lessons)): ?>
                            <!-- No Lessons Available -->
                            <div class="accordion-item">
                                <div class="empty-state p-4">
                                    <div class="text-center text-muted">
                                        <i class="bx bx-video-off fs-1"></i>
                                        <p class="mt-2 mb-0">No lessons available</p>
                                    </div>
                                </div>
                            </div>
                        <?php else: ?>
                            <!-- Individual Lesson Accordions -->
                            <?php foreach ($lessons as $index => $lesson): ?>
                                <?php 
                                // Get video-specific content for badges and links
                                $hasVideoQuiz = false;
                                $hasVideoAssignment = false; 
                                $hasVideoMaterials = false;
                                $hasVideoQuiz = false;
                                $videoAssignments = [];
                                $videoMaterials = [];
                                $videoQuizzes = []; // Changed to array to handle multiple quizzes
                                
                                if (isset($conn) && !empty($lesson['lesson_id'])) {
                                    // Get ALL quizzes for this video (removed LIMIT 1) - Fix video_id mapping
                                    $quiz_video_id = isset($lesson['video_id']) ? $lesson['video_id'] : $lesson['lesson_id'];
                                    $quiz_check = $conn->prepare("SELECT quiz_id, title, time_limit, passing_score FROM quizzes WHERE video_id = ? AND is_published = 1");
                                    if ($quiz_check) {
                                        $quiz_check->bind_param("i", $quiz_video_id);
                                        $quiz_check->execute();
                                        $quiz_result = $quiz_check->get_result();
                                        
                                        // Process ALL quizzes for this video
                                        while ($quiz_data = $quiz_result->fetch_assoc()) {
                                            $hasVideoQuiz = true;
                                            
                                            // Get student's attempt data for this specific quiz  
                                            $attempt_check = $conn->prepare("
                                                SELECT attempt_id, score, pass_status as passed, created_at as completed_at 
                                                FROM quiz_attempts 
                                                WHERE quiz_id = ? AND student_id = ? 
                                                ORDER BY created_at DESC 
                                                LIMIT 1
                                            ");
                                            if ($attempt_check) {
                                                $attempt_check->bind_param("ii", $quiz_data['quiz_id'], $user_id);
                                                $attempt_check->execute();
                                                $attempt_result = $attempt_check->get_result();

                                                if ($attempt_data = $attempt_result->fetch_assoc()) {
                                                    // Merge attempt data with quiz data
                                                    $quiz_data = array_merge($quiz_data, $attempt_data);

                                                } else {

                                                }
                                                $attempt_check->close();
                                            }
                                            
                                            // Add this quiz to the array
                                            $videoQuizzes[] = $quiz_data;
                                        }
                                        $quiz_check->close();
                                    }
                                    
                                    // Get video-specific assignments data
                                    $assignment_check = $conn->prepare("SELECT assignment_id, title, due_date FROM assignments WHERE video_id = ?");
                                    if ($assignment_check) {
                                        $assignment_check->bind_param("i", $lesson['lesson_id']);
                                        $assignment_check->execute();
                                        $assignment_result = $assignment_check->get_result();
                                        while ($assignment_data = $assignment_result->fetch_assoc()) {
                                            $videoAssignments[] = $assignment_data;
                                        }
                                        $hasVideoAssignment = count($videoAssignments) > 0;
                                        $assignment_check->close();
                                    }
                                    
                                    // Get video-specific materials data
                                    $materials_check = $conn->prepare("SELECT material_id, title, file_path, file_type FROM course_materials WHERE video_id = ?");
                                    if ($materials_check) {
                                        $materials_check->bind_param("i", $lesson['lesson_id']);
                                        $materials_check->execute();
                                        $materials_result = $materials_check->get_result();
                                        while ($materials_data = $materials_result->fetch_assoc()) {
                                            $videoMaterials[] = $materials_data;
                                        }
                                        $hasVideoMaterials = count($videoMaterials) > 0;
                                        $materials_check->close();
                                    }
                                    
                                    // Check for video-specific notes
                                    $notes_check = $conn->prepare("SELECT COUNT(*) as count FROM lesson_notes WHERE video_id = ?");
                                    if ($notes_check) {
                                        $notes_check->bind_param("i", $lesson['lesson_id']);
                                        $notes_check->execute();
                                        $notes_result = $notes_check->get_result();
                                        if ($notes_data = $notes_result->fetch_assoc()) {
                                            $hasVideoNotes = $notes_data['count'] > 0;
                                        }
                                        $notes_check->close();
                                    }
                                }
                                ?>
                                
                                <?php
                                // Calculate REAL progress from database BEFORE using it in HTML
                                $real_progress = $lesson['watch_progress'] ?? 0;
                                

                                ?>
                                <?php if (isset($conn) && isset($_SESSION['user_id'])) {
                                    $progress_query = "SELECT completion_percentage FROM video_progress WHERE video_id = ? AND student_id = ?";
                                    if ($stmt = $conn->prepare($progress_query)) {
                                        $stmt->bind_param("ii", $lesson['lesson_id'], $_SESSION['user_id']);
                                        $stmt->execute();
                                        $result = $stmt->get_result();
                                        if ($row = $result->fetch_assoc()) {
                                            $real_progress = round($row['completion_percentage']);
                                        }
                                        $stmt->close();
                                    }
                                }
                                ?>
                                
                                <div class="accordion-item">
                                    <h2 class="accordion-header" id="headingLesson<?= $lesson['lesson_id'] ?>">
                                        <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#collapseLesson<?= $lesson['lesson_id'] ?>" aria-expanded="false" aria-controls="collapseLesson<?= $lesson['lesson_id'] ?>">
                                            <div class="d-flex align-items-center w-100">
                                                <div class="lesson-thumbnail-small me-3">
                                                    <?php if (!empty($lesson['thumbnail'])): ?>
                                                        <img src="<?= htmlspecialchars($lesson['thumbnail']) ?>" alt="Lesson <?= $index + 1 ?>" class="rounded" style="width: 40px; height: 30px; object-fit: cover;">
                                                    <?php else: ?>
                                                        <div class="thumbnail-placeholder-small d-flex align-items-center justify-content-center" style="width: 40px; height: 30px; background: #f8f9fa; border-radius: 4px;">
                                                            <i class="bx bx-play text-muted"></i>
                                                        </div>
                                                    <?php endif; ?>
                                                    <?php if ($lesson['completed']): ?>
                                                        <div class="completion-check position-absolute" style="top: -5px; right: -5px;">
                                                            <i class="bx bx-check-circle text-success fs-6"></i>
                                                        </div>
                                                    <?php endif; ?>
                                                </div>
                                                <div class="flex-grow-1" style="min-width: 0;">
                                                    <div class="lesson-number text-primary fw-bold" style="font-size: 0.85rem; margin-bottom: 2px;">Lesson <?= $index + 1 ?></div>
                                                    <div class="fw-semibold accordion-title-text" style="white-space: nowrap; overflow: hidden; text-overflow: ellipsis; padding-right: 15px; line-height: 1.2;">
                                                        <?= htmlspecialchars($lesson['title']) ?>
                                                    </div>
                                                    <div class="d-flex align-items-center justify-content-between mb-1">
                                                        <small class="text-muted">
                                                            <i class="bx bx-time"></i> <?= !empty($lesson['duration']) ? htmlspecialchars($lesson['duration']) : 'Duration not available' ?>
                                                        </small>
                                                        <small class="text-muted" id="progress-display-<?= $lesson['lesson_id'] ?>" style="font-size: 0.7rem; display: none;">
                                                            <span id="progress-percentage-<?= $lesson['lesson_id'] ?>">0</span>% watched
                                                        </small>
                                                    </div>
                                                    <!-- Badge rendering removed - will be replaced with unified system -->
                                                    <div class="d-flex align-items-center flex-wrap" id="lesson-badges-<?= $lesson['lesson_id'] ?>" style="display: none;">
                                                        <!-- Watching and Completed badges removed -->
                                                        <?php if ($hasVideoQuiz): ?>
                                                            <span class="badge badge-quiz-accordion me-1" style="font-size: 0.6rem;" title="Interactive Quiz Available">
                                                                <i class="bx bx-help-circle"></i> Quiz
                                                            </span>
                                                        <?php endif; ?>
                                                        <?php if ($hasVideoAssignment): ?>
                                                            <span class="badge bg-warning me-1" style="font-size: 0.7rem;" title="Assignment Available">
                                                                <i class="bx bx-task"></i> Task
                                                            </span>
                                                        <?php endif; ?>
                                                        <?php if ($hasVideoMaterials): ?>
                                                            <span class="badge badge-files-accordion me-1" style="font-size: 0.6rem;" title="Materials Available">
                                                                <i class="bx bx-folder-open"></i> Files
                                                            </span>
                                                        <?php endif; ?>
                                                        <?php if ($hasVideoNotes): ?>
                                                            <span class="badge bg-secondary me-1" style="font-size: 0.7rem;" title="Notes Available">
                                                                <i class="bx bx-note"></i> Notes
                                                            </span>
                                                        <?php endif; ?>
                                                    </div>
                                                </div>
                                            </div>
                                        </button>
                                    </h2>
                                    <div id="collapseLesson<?= $lesson['lesson_id'] ?>" class="accordion-collapse collapse" aria-labelledby="headingLesson<?= $lesson['lesson_id'] ?>" data-bs-parent="#courseContentAccordion">
                                        <div class="accordion-body">
                                            <!-- Watch Button -->
                            <div class="d-grid gap-2 mb-3">
                                <button type="button" class="btn btn-watch-lesson" onclick="event.preventDefault(); event.stopPropagation(); return loadLesson(<?= $lesson['lesson_id'] ?>);">
                                    <i class="bx bx-play-circle me-2"></i>
                                    <?php 
                                    // 🎯 FIX: Show proper status based on completion, not just current lesson
                                    if ($lesson['completed']) {
                                        echo 'Rewatch Video';
                                    } else if ($lesson['lesson_id'] == $current_lesson_id) {
                                        echo 'Currently Watching';
                                    } else {
                                        echo 'Watch Lesson';
                                    }
                                    ?>
                                </button>
                            </div>

                                            
                                            <!-- 🎨 Dynamic Range-Style Progress Bar -->
                                            <div class="dynamic-progress-container mb-3">
                                                <!-- Progress Label -->
{{ ... }}
                                                <div class="text-center text-muted small mb-2 progress-label">
                                                    Progress: <span id="progress-percentage-<?= $lesson['lesson_id'] ?>" class="progress-percentage-text"><?= $real_progress ?></span>% complete
                                                </div>
                                                
                                                <!-- Range-Style Progress Track -->
                                                <div class="progress-track clean-progress-track">
                                                    <!-- Progress Fill -->
                                                    <div 
                                                        class="progress-bar clean-progress-fill" 
                                                        id="accordion-progress-bar-<?= $lesson['lesson_id'] ?>"
                                                        style="width: <?= $real_progress ?>%;"
                                                    ></div>
                                                    
                                                    <!-- Progress Thumb/Handle -->
                                                    <div 
                                                        class="progress-thumb clean-progress-thumb <?php if ($real_progress > 0): ?>active<?php endif; ?>" 
                                                        id="progress-thumb-<?= $lesson['lesson_id'] ?>"
                                                        style="left: <?= $real_progress ?>%;"
                                                        
                                                    ></div>
                                                    
                                                    <!-- Completion Sparkle Effect (only when 100%) -->
                                                    <?php if ($real_progress >= 100): ?>
                                                    <div class="completion-sparkle" style="
                                                        position: absolute;
                                                        top: 50%;
                                                        right: 10px;
                                                        transform: translateY(-50%);
                                                        color: #fbbf24;
                                                        font-size: 16px;
                                                        animation: sparkle 1.5s ease-in-out infinite;
                                                    ">✨</div>
                                                    <?php endif; ?>
                                                </div>
                                                
                                                <!-- Progress Milestones (Optional) -->
                                                <div class="progress-milestones" style="
                                                    position: relative;
                                                    margin-top: 4px;
                                                    height: 4px;
                                                ">
                                                    <?php for ($i = 25; $i <= 75; $i += 25): ?>
                                                    <div style="
                                                        position: absolute;
                                                        left: <?= $i ?>%;
                                                        width: 2px;
                                                        height: 4px;
                                                        background: <?= $real_progress >= $i ? '#2563eb' : 'rgba(245, 158, 11, 0.5)' ?>;
                                                        border-radius: 1px;
                                                        transform: translateX(-50%);
                                                    "></div>
                                                    <?php endfor; ?>
                                                </div>
                                            </div>
                                            
                                            <!-- Nested Accordions for Quizzes and Materials in Stacked Layout -->
                                            <?php if ($hasVideoQuiz || $hasVideoMaterials || $hasVideoAssignment || $hasVideoNotes): ?>
                                                <div class="lesson-content-container">
                                                    <!-- Quizzes Section (Full Width) -->
                                                    <?php if ($hasVideoQuiz && !empty($videoQuizzes)): ?>
                                                        <div class="nested-accordion mb-3">
                                                            <div class="nested-accordion-header" data-toggle="collapse" data-target="#quizzes-<?= $lesson['lesson_id'] ?>" aria-expanded="false">
                                                                <h6 class="mb-0 d-flex align-items-center">
                                                                    <i class="bx bx-help-circle me-2 text-primary"></i>
                                                                    <span>Quizzes</span>
                                                                    <span class="badge bg-primary ms-2"><?= count($videoQuizzes) ?></span>
                                                                </h6>
                                                                <i class="bx bx-chevron-down"></i>
                                                            </div>
                                                            <div id="quizzes-<?= $lesson['lesson_id'] ?>" class="collapse nested-accordion-body">
                                                                <div class="nested-content p-2">
                                                                    <?php foreach ($videoQuizzes as $index => $videoQuiz): ?>
                                                                        <?php 
                                                                        // Encrypt quiz ID
                                                                        $encrypted_quiz_id = '';
                                                                        try {
                                                                            if (class_exists('IdEncryption') && method_exists('IdEncryption', 'encrypt')) {
                                                                                $encrypted_quiz_id = IdEncryption::encrypt($videoQuiz['quiz_id']);
                                                                                if (empty($encrypted_quiz_id) || $encrypted_quiz_id == $videoQuiz['quiz_id']) {
                                                                                    $encrypted_quiz_id = $videoQuiz['quiz_id'];
                                                                                }
                                                                            } else {
                                                                                $encrypted_quiz_id = $videoQuiz['quiz_id'];
                                                                            }
                                                                        } catch (Exception $e) {
                                                                            $encrypted_quiz_id = $videoQuiz['quiz_id'];
                                                                        }
                                                                        
                                                                        // Check if student has completed the quiz
                                                                        $hasCompleted = !empty($videoQuiz['attempt_id']);
                                                                        $studentScore = $videoQuiz['score'] ?? 0;
                                                                        $hasPassed = $videoQuiz['passed'] ?? false;
                                                                        $completedAt = $videoQuiz['completed_at'] ?? null;
                                                                        ?>
                                                                        
                                                                        <div class="compact-quiz-item mb-2 border rounded" style="background: #f8f9fa;">
                                                                            <div class="p-2">
                                                                                <!-- Quiz Title Row -->
                                                                                <div class="d-flex align-items-center mb-2">
                                                                                    <i class="bx bx-help-circle text-primary me-2" style="font-size: 1rem;"></i>
                                                                                    <div class="fw-bold text-truncate" style="font-size: 0.85rem; color: #2d3748;">
                                                                                        <?= htmlspecialchars($videoQuiz['title']) ?>
                                                                                    </div>
                                                                                </div>
                                                                                
                                                                                <!-- Line 1: Time and Score -->
                                                                                <div class="d-flex align-items-center justify-content-between mb-1">
                                                                                    <span class="text-muted" style="font-size: 0.75rem;">
                                                                                        <i class="bx bx-time"></i> <?= $videoQuiz['time_limit'] ?>min
                                                                                    </span>
                                                                                    <?php if ($hasCompleted): ?>
                                                                                        <span class="text-<?= $hasPassed ? 'success' : 'warning' ?> fw-bold" style="font-size: 0.75rem;">
                                                                                            Score: <?= round($studentScore) ?>%
                                                                                        </span>
                                                                                    <?php endif; ?>
                                                                                </div>
                                                                                
                                                                                <!-- Line 2: Status Badge and Retry Button -->
                                                                                <div class="d-flex align-items-center justify-content-between mb-2">
                                                                                    <span class="badge bg-<?= $hasCompleted ? ($hasPassed ? 'success' : 'warning') : 'secondary' ?>" style="font-size: 0.7rem; padding: 4px 8px;">
                                                                                        <?php if ($hasCompleted): ?>
                                                                                            <?= $hasPassed ? 'PASSED' : 'FAILED' ?>
                                                                                        <?php else: ?>
                                                                                            PENDING
                                                                                        <?php endif; ?>
                                                                                    </span>
                                                                                    <a href="take_quiz.php?id=<?= htmlspecialchars($encrypted_quiz_id) ?><?= $hasCompleted ? '&retry=1' : '' ?>" 
                                                                                       class="btn btn-outline-primary btn-sm" style="font-size: 0.75rem; padding: 4px 12px;">
                                                                                        <?= $hasCompleted ? 'Retry Quiz' : 'Start Quiz' ?>
                                                                                    </a>
                                                                                </div>
                                                                                
                                                                                <!-- Line 3: View Results Button (Full Width) -->
                                                                                <?php if ($hasCompleted && !empty($videoQuiz['attempt_id'])): ?>
                                                                                    <div class="d-grid">
                                                                                        <a href="quiz_results.php?attempt_id=<?= htmlspecialchars($videoQuiz['attempt_id']) ?>" 
                                                                                           class="btn btn-outline-success btn-sm" style="font-size: 0.75rem; padding: 6px;">
                                                                                            <i class="bx bx-bar-chart-alt-2 me-1"></i> View Detailed Results
                                                                                        </a>
                                                                                    </div>
                                                                                <?php endif; ?>
                                                                            </div>
                                                                        </div>
                                                                    <?php endforeach; ?>
                                                                </div>
                                                            </div>
                                                        </div>
                                                    <?php endif; ?>

                                                    <!-- Materials Section (Full Width) -->
                                                    <?php if ($hasVideoMaterials && !empty($videoMaterials)): ?>
                                                        <div class="nested-accordion mb-3">
                                                            <div class="nested-accordion-header" data-toggle="collapse" data-target="#materials-<?= $lesson['lesson_id'] ?>" aria-expanded="false">
                                                                <h6 class="mb-0 d-flex align-items-center">
                                                                    <i class="bx bx-folder me-2 text-success"></i>
                                                                    <span>Materials</span>
                                                                    <span class="badge bg-success ms-2"><?= count($videoMaterials) ?></span>
                                                                </h6>
                                                                <i class="bx bx-chevron-down"></i>
                                                            </div>
                                                            <div id="materials-<?= $lesson['lesson_id'] ?>" class="collapse nested-accordion-body">
                                                                <div class="nested-content p-2">
                                                                    <?php foreach ($videoMaterials as $material): ?>
                                                                        <?php
                                                                        // Determine file type icon with standardized colors
                                                                        $fileType = strtolower($material['file_type'] ?? '');
                                                                        $fileIcon = 'bx-file';
                                                                        $fileColor = 'info';
                                                                        
                                                                        if (strpos($fileType, 'pdf') !== false) {
                                                                            $fileIcon = 'bx-file-pdf';
                                                                            $fileColor = 'danger';
                                                                        } elseif (strpos($fileType, 'doc') !== false) {
                                                                            $fileIcon = 'bx-file-doc';
                                                                            $fileColor = 'primary';
                                                                        } elseif (strpos($fileType, 'excel') !== false || strpos($fileType, 'sheet') !== false) {
                                                                            $fileIcon = 'bx-file-excel';
                                                                            $fileColor = 'success';
                                                                        } elseif (strpos($fileType, 'powerpoint') !== false || strpos($fileType, 'presentation') !== false) {
                                                                            $fileIcon = 'bx-file-ppt';
                                                                            $fileColor = 'warning';
                                                                        } elseif (strpos($fileType, 'image') !== false) {
                                                                            $fileIcon = 'bx-image';
                                                                            $fileColor = 'secondary';
                                                                        }
                                                                        ?>
                                                                        
                                                                        <div class="compact-material-item mb-1">
                                                                            <div class="d-flex align-items-center justify-content-between py-1 px-2">
                                                                                <div class="flex-grow-1">
                                                                                    <div class="d-flex align-items-center gap-2">
                                                                                        <i class="bx <?= $fileIcon ?> text-<?= $fileColor ?>" style="font-size: 0.9rem;"></i>
                                                                                        <span class="fw-bold" style="font-size: 0.8rem;">
                                                                                            <?= htmlspecialchars($material['title']) ?>
                                                                                        </span>
                                                                                    </div>
                                                                                    <div class="mt-1">
                                                                                        <small class="text-muted" style="font-size: 0.7rem;">Material</small>
                                                                                    </div>
                                                                                </div>
                                                                                <?php if (!empty($material['file_path'])): ?>
                                                                                    <a href="../<?= htmlspecialchars($material['file_path']) ?>" 
                                                                                       class="btn btn-outline-<?= $fileColor ?>" 
                                                                                       style="font-size: 0.7rem; padding: 4px 8px;"
                                                                                       target="_blank"
                                                                                       title="Download <?= htmlspecialchars($material['title']) ?>">
                                                                                        Download
                                                                                    </a>
                                                                                <?php endif; ?>
                                                                            </div>
                                                                        </div>
                                                                    <?php endforeach; ?>
                                                                </div>
                                                            </div>
                                                        </div>
                                                    <?php endif; ?>
                                                    
                                                    <!-- Additional Content Row (Full Width) -->
                                                    <?php if ($hasVideoAssignment || $hasVideoNotes): ?>
                                                        <div class="row mt-2">
                                                            <div class="col-12">
                                                                <div class="additional-content">
                                                                    <?php if ($hasVideoAssignment): ?>
                                                                        <div class="mb-2">
                                                                            <a href="#" class="btn btn-outline-warning btn-sm w-100">
                                                                                <i class="bx bx-task me-2"></i>Complete Assignment
                                                                            </a>
                                                                        </div>
                                                                    <?php endif; ?>
                                                                    
                                                                    <?php if ($hasVideoNotes): ?>
                                                                        <div class="mb-2">
                                                                            <a href="#" class="btn btn-outline-secondary btn-sm w-100">
                                                                                <i class="bx bx-note me-2"></i>View Lecture Notes
                                                                            </a>
                                                                        </div>
                                                                    <?php endif; ?>
                                                                </div>
                                                            </div>
                                                        </div>
                                                    <?php endif; ?>
                                                </div>
                                            <?php else: ?>
                                                <div class="text-center text-muted">
                                                    <small>No additional content for this lesson</small>
                                                </div>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        <?php endif; ?>

                    </div>
                </div>
            </div>

            <!-- Main Content Area (75% width) -->
            <div class="col-md-9">





            <!-- Video Player -->
            <div class="video-container">
                <div class="video-player" id="videoPlayer">
                    <?php if (empty($lessons)): ?>
                        <!-- No lessons available -->
                        <div class="video-placeholder" style="width: 100%; height: 100%; background: linear-gradient(135deg, #1f2937 0%, #374151 100%); display: flex; align-items: center; justify-content: center; flex-direction: column; color: #9ca3af;">
                            <div style="text-align: center;">
                                <i class="bx bx-video-off" style="font-size: 4rem; margin-bottom: 1rem; color: #6b7280;"></i>
                                <h3 style="margin: 0 0 0.5rem 0; color: white;">No Content Available</h3>
                                <p style="margin: 0 0 0.5rem 0;">This course doesn't have any lessons or videos yet.</p>
                                <small>Please check back later or contact your instructor</small>
                            </div>
                        </div>
                    <?php elseif (!empty($current_lesson['video_url'])): ?>
                        <?php if (!empty($current_lesson['is_external']) && $current_lesson['is_external']): ?>
                            <!-- External video (YouTube, etc.) with automatic iframe API progress tracking -->
                            <?php 
                            $videoUrl = htmlspecialchars($current_lesson['video_url']);
                            // Add YouTube API parameters for automatic progress tracking
                            if (strpos($videoUrl, 'youtube.com/embed') !== false) {
                                $separator = strpos($videoUrl, '?') !== false ? '&' : '?';
                                $videoUrl .= $separator . 'autoplay=0&rel=0&showinfo=0&controls=1&enablejsapi=1&origin=' . urlencode($_SERVER['HTTP_HOST']);
                            } else {
                                $separator = strpos($videoUrl, '?') !== false ? '&' : '?';
                                $videoUrl .= $separator . 'autoplay=0&rel=0&showinfo=0&controls=1';
                            }
                            ?>
                            <iframe id="lessonVideo" 
                                    src="<?= $videoUrl ?>" 
                                    style="width: 100%; height: 100%; border: none; background: #000;"
                                    frameborder="0" 
                                    allow="accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture"
                                    allowfullscreen>
                            </iframe>
                        <?php else: ?>
                            <!-- Local video file -->
                            <video id="lessonVideo" preload="metadata"
                                   style="width: 100%; height: 100%; background: #000;"
                                   <?php if (!empty($current_lesson['thumbnail'])): ?>
                                   poster="<?= htmlspecialchars($current_lesson['thumbnail']) ?>"
                                   <?php endif; ?>>
                                <source src="<?= htmlspecialchars($current_lesson['video_url']) ?>" type="video/mp4">
                                Your browser does not support the video tag.
                            </video>
                        <?php endif; ?>
                    <?php else: ?>
                        <div class="video-placeholder" style="width: 100%; height: 100%; background: linear-gradient(135deg, #1f2937 0%, #374151 100%); display: flex; align-items: center; justify-content: center; flex-direction: column; color: #9ca3af; position: relative;">
                            <!-- Show thumbnail if available even when video is not available -->
                            <?php if (!empty($current_lesson['thumbnail'])): ?>
                                <img src="<?= htmlspecialchars($current_lesson['thumbnail']) ?>" 
                                     alt="<?= htmlspecialchars($current_lesson['title']) ?>"
                                     style="position: absolute; top: 0; left: 0; width: 100%; height: 100%; object-fit: cover; opacity: 0.3;">
                            <?php endif; ?>
                            
                            <div style="position: relative; z-index: 1; text-align: center;">
                                <i class="bx bx-video" style="font-size: 4rem; margin-bottom: 1rem;"></i>
                                <h3 style="margin: 0 0 0.5rem 0; color: white;"><?= htmlspecialchars($current_lesson['title']) ?></h3>
                                <p style="margin: 0 0 0.5rem 0;">Video not available</p>
                                <small>Please contact your instructor</small>
                            </div>
                        </div>
                    <?php endif; ?>
                </div>
                
                <?php if (empty($current_lesson['is_external']) || !$current_lesson['is_external']): ?>
                <!-- Custom video controls - only show for local video files, not external videos -->
                <div class="video-controls">
                    <div class="time-bar" onclick="seekVideo(event)">
                        <div class="time-progress" id="timeProgress"></div>
                    </div>
                    <div class="control-buttons">
                        <div class="control-left">
                            <button id="playPauseBtn" onclick="togglePlayPause()" class="control-btn">
                                <i class="bx bx-play"></i>
                            </button>
                            <button onclick="skipBackward()" class="control-btn" title="Skip back 10s">
                                <i class="bx bx-rewind"></i>
                            </button>
                            <button onclick="skipForward()" class="control-btn" title="Skip forward 10s">
                                <i class="bx bx-fast-forward"></i>
                            </button>
                            <div class="time-display">
                                <span id="currentTime">0:00</span> / <span id="totalTime"><?= $current_lesson['duration'] ?></span>
                            </div>
                        </div>
                        <div class="control-center">
                            <!-- Playback Speed -->
                            <div class="dropdown">
                                <button class="control-btn dropdown-toggle" onclick="toggleSpeedMenu()" title="Playback Speed">
                                    <span id="speedLabel">1x</span>
                                </button>
                                <div class="dropdown-menu" id="speedMenu">
                                    <div class="dropdown-item" onclick="setPlaybackSpeed(0.5)">0.5x</div>
                                    <div class="dropdown-item" onclick="setPlaybackSpeed(0.75)">0.75x</div>
                                    <div class="dropdown-item active" onclick="setPlaybackSpeed(1)">1x</div>
                                    <div class="dropdown-item" onclick="setPlaybackSpeed(1.25)">1.25x</div>
                                    <div class="dropdown-item" onclick="setPlaybackSpeed(1.5)">1.5x</div>
                                    <div class="dropdown-item" onclick="setPlaybackSpeed(2)">2x</div>
                                </div>
                            </div>

                            <!-- Quality Settings -->
                            <div class="dropdown">
                                <button class="control-btn dropdown-toggle" onclick="toggleQualityMenu()" title="Video Quality">
                                    <i class="bx bx-cog"></i>
                                </button>
                                <div class="dropdown-menu" id="qualityMenu">
                                    <div class="dropdown-item active" onclick="setVideoQuality('auto')">Auto</div>
                                    <div class="dropdown-item" onclick="setVideoQuality('1080p')">1080p</div>
                                    <div class="dropdown-item" onclick="setVideoQuality('720p')">720p</div>
                                    <div class="dropdown-item" onclick="setVideoQuality('480p')">480p</div>
                                    <div class="dropdown-item" onclick="setVideoQuality('360p')">360p</div>
                                </div>
                            </div>

                            <!-- Subtitles -->
                            <button class="control-btn" onclick="toggleSubtitles()" title="Subtitles" id="subtitleBtn">
                                <i class="bx bx-captions"></i>
                            </button>

                            <!-- Notes -->
                            <button class="control-btn" onclick="addNote()" title="Add Note at Current Time">
                                <i class="bx bx-note"></i>
                            </button>
                        </div>
                        <div class="control-right">
                            <!-- Volume Control -->
                            <div class="volume-control">
                                <button onclick="toggleMute()" class="control-btn">
                                    <i class="bx bx-volume-full" id="volumeIcon"></i>
                                </button>
                                <input type="range" id="volumeSlider" class="volume-slider" min="0" max="100" value="100" onchange="setVolume(this.value)">
                            </div>
                            
                            <!-- Picture in Picture -->
                            <button onclick="togglePictureInPicture()" class="control-btn" title="Picture in Picture">
                                <i class="bx bx-dock-top"></i>
                            </button>
                            
                            <!-- Fullscreen -->
                            <button onclick="toggleFullscreen()" class="control-btn">
                                <i class="bx bx-fullscreen"></i>
                            </button>
                        </div>
                    </div>
                </div>
                <?php endif; ?>
            </div>

            <!-- Course Content Tabs -->
            <div class="course-tabs-container">
                <!-- Tab Navigation -->
                <div class="tab-navigation">
                    <button class="tab-btn active" data-tab="description">
                        <i class="bx bx-book-open"></i>
                        Course Description
                    </button>
                    <button class="tab-btn" data-tab="reviews">
                        <i class="bx bx-star"></i>
                        Reviews
                    </button>
                    <button class="tab-btn" data-tab="notes">
                        <i class="bx bx-note"></i>
                        Notes
                    </button>
                    <button class="tab-btn" data-tab="certificate">
                        <i class="bx bx-award"></i>
                        Certificate
                    </button>
                    <button class="tab-btn" data-tab="tools">
                        <i class="bx bx-wrench"></i>
                        Learning Tools
                    </button>
                </div>

                <!-- Tab Content -->
                <div class="tab-content">
                    <!-- Enhanced Course Description Tab -->
                    <div class="tab-pane active" id="description">
                        <div class="enhanced-course-description">
                            
                            <!-- Course Header Section -->
                            <div class="course-header-section">
                                <div class="row g-4">
                                    <div class="col-lg-8">
                                        <div class="course-title-section">
                                            <h2 class="course-main-title"><?= htmlspecialchars($course['title']) ?></h2>
                                            <p class="course-subtitle"><?= htmlspecialchars($course['description']) ?></p>
                                        </div>
                                        
                                        <!-- Course Meta Information -->
                                        <div class="course-meta-grid">
                                            <div class="meta-card">
                                                <div class="meta-icon">
                                                    <i class="bx bx-play-circle"></i>
                                                </div>
                                                <div class="meta-content">
                                                    <div class="meta-value"><?= count($lessons) ?></div>
                                                    <div class="meta-label">Video Lessons</div>
                                                </div>
                                            </div>
                                            

                                            
                                            <div class="meta-card">
                                                <div class="meta-icon">
                                                    <i class="bx bx-time"></i>
                                                </div>
                                                <div class="meta-content">
                                                    <div class="meta-value">
                                                        <?php 
                                                        // Calculate total course duration
                                                        $total_minutes = 0;
                                                        foreach ($lessons as $lesson) {
                                                            if (!empty($lesson['duration'])) {
                                                                // Parse duration (assumes format like "5:30" or "10:15")
                                                                $time_parts = explode(':', $lesson['duration']);
                                                                if (count($time_parts) == 2) {
                                                                    $total_minutes += intval($time_parts[0]) * 60 + intval($time_parts[1]);
                                                                }
                                                            }
                                                        }
                                                        $hours = floor($total_minutes / 60);
                                                        $minutes = $total_minutes % 60;
                                                        echo $hours > 0 ? $hours . 'h ' . $minutes . 'm' : $minutes . ' mins';
                                                        ?>
                                                    </div>
                                                    <div class="meta-label">Total Duration</div>
                                                </div>
                                            </div>
                                            
                                            <div class="meta-card">
                                                <div class="meta-icon">
                                                    <i class="bx bx-medal"></i>
                                                </div>
                                                <div class="meta-content">
                                                    <div class="meta-value">
                                                        <?php
                                                        // Get course level/difficulty
                                                        $course_level = !empty($course['level']) ? htmlspecialchars($course['level']) : 'Beginner';
                                                        echo $course_level;
                                                        ?>
                                                    </div>
                                                    <div class="meta-label">Difficulty Level</div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                    
                                    <!-- Instructor Information -->
                                    <div class="col-lg-4">
                                        <div class="instructor-card">
                                            <div class="instructor-header">
                                                <h5><i class="bx bx-user-circle"></i> Your Instructor</h5>
                                            </div>
                                            <div class="instructor-content">
                                                <div class="instructor-avatar">
                                                    <?php if (!empty($course['instructor_profile_image'])): ?>
                                                        <img src="<?= htmlspecialchars($course['instructor_profile_image']) ?>" alt="Instructor">
                                                    <?php else: ?>
                                                        <div class="avatar-placeholder">
                                                            <i class="bx bx-user"></i>
                                                        </div>
                                                    <?php endif; ?>
                                                </div>
                                                <div class="instructor-details">
                                                    <h6 class="instructor-name"><?= htmlspecialchars($instructor_name) ?></h6>
                                                    <p class="instructor-title">
                                                        <?php
                                                        // Get instructor title/bio from database
                                                        $instructor_title = "Course Instructor";
                                                        if (isset($conn)) {
                                                            $instructor_query = "SELECT bio, title FROM users WHERE user_id = ?";
                                            if ($stmt = $conn->prepare($instructor_query)) {
                                                                $stmt->bind_param("i", $course['teacher_id']);
                                                                $stmt->execute();
                                                                $result = $stmt->get_result();
                                                                if ($instructor_info = $result->fetch_assoc()) {
                                                                    $instructor_title = !empty($instructor_info['title']) ? htmlspecialchars($instructor_info['title']) : $instructor_title;
                                                                }
                                                                $stmt->close();
                                                            }
                                                        }
                                                        echo $instructor_title;
                                                        ?>
                                                    </p>
                                                    <div class="instructor-stats">
                                                        <div class="stat-item">
                                                            <i class="bx bx-book"></i>
                                                            <span>
                                                                <?php
                                                                // Get instructor's total courses
                                                                $instructor_courses = 0;
                                                                if (isset($conn)) {
                                                                    $courses_query = "SELECT COUNT(*) as count FROM courses WHERE teacher_id = ?";
                                                                    if ($stmt = $conn->prepare($courses_query)) {
                                                                        $stmt->bind_param("i", $course['teacher_id']);
                                                                        $stmt->execute();
                                                                        $result = $stmt->get_result();
                                                                        if ($courses_data = $result->fetch_assoc()) {
                                                                            $instructor_courses = $courses_data['count'];
                                                                        }
                                                                        $stmt->close();
                                                                    }
                                                                }
                                                                echo $instructor_courses . ' Courses';
                                                                ?>
                                                            </span>
                                                        </div>

                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <!-- Learning Objectives Section -->
                            <div class="learning-objectives-section">
                                <h4><i class="bx bx-target-lock"></i> What You'll Learn</h4>

                                <div class="objectives-grid">
                                    <div class="objective-item">
                                        <i class="bx bx-check-circle"></i>
                                        <span>Master the fundamental concepts and principles</span>
                                    </div>
                                    <div class="objective-item">
                                        <i class="bx bx-check-circle"></i>
                                        <span>Apply practical skills through hands-on exercises</span>
                                    </div>
                                    <div class="objective-item">
                                        <i class="bx bx-check-circle"></i>
                                        <span>Build confidence in the subject matter</span>
                                    </div>
                                    <div class="objective-item">
                                        <i class="bx bx-check-circle"></i>
                                        <span>Complete interactive quizzes and assessments</span>
                                    </div>
                                    <div class="objective-item">
                                        <i class="bx bx-check-circle"></i>
                                        <span>Gain industry-relevant knowledge and expertise</span>
                                    </div>
                                    <div class="objective-item">
                                        <i class="bx bx-check-circle"></i>
                                        <span>Develop problem-solving and critical thinking skills</span>
                                    </div>
                                </div>
                            </div>

                            <!-- Course Requirements Section -->
                            <div class="course-requirements-section">
                                <div class="row g-4">
                                    <div class="col-md-6">
                                        <div class="requirements-card">
                                            <h5><i class="bx bx-list-check"></i> Prerequisites</h5>
                                            <ul class="requirements-list">
                                                <li><i class="bx bx-check"></i> Basic computer literacy</li>
                                                <li><i class="bx bx-check"></i> Internet connection for online content</li>
                                                <li><i class="bx bx-check"></i> Motivation to learn and practice</li>
                                                <li><i class="bx bx-check"></i> No prior experience required</li>
                                        </div>
                                    </div>
                                    <div class="col-md-6">
                                        <div class="course-info-card">
                                            <h5><i class="bx bx-info-circle"></i> Course Information</h5>
                                            <div class="info-list">
                                                <div class="info-item">
                                                    <span class="info-label">Created:</span>
                                                    <span class="info-value">
                                                        <?= date('M d, Y', strtotime($course['created_at'])) ?>
                                                    </span>
                                                </div>
                                                <div class="info-item">
                                                    <span class="info-label">Last Updated:</span>
                                                    <span class="info-value">
                                                        <?= date('M d, Y', strtotime($course['updated_at'] ?? $course['created_at'])) ?>
                                                    </span>
                                                </div>
                                                <div class="info-item">
                                                    <span class="info-label">Language:</span>
                                                    <span class="info-value">English</span>
                                                </div>
                                                <div class="info-item">
                                                    <span class="info-label">Access:</span>
                                                    <span class="info-value">Lifetime Access</span>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>


                        </div>
                    </div>

                    <!-- Enhanced Reviews Tab -->
                    <div class="tab-pane" id="reviews">
                        <div class="enhanced-reviews-content">
                            
                            <!-- Reviews Summary Header -->
                            <div class="reviews-summary-section">
                                <div class="row g-4">
                                    <div class="col-lg-4">
                                        <div class="overall-rating-card">
                                            <div class="rating-display">
                                                <div class="rating-number"><?= $review_stats['average_rating'] > 0 ? number_format($review_stats['average_rating'], 1) : 'New' ?></div>
                                                <div class="rating-stars-large">
                                                    <?php for ($i = 1; $i <= 5; $i++): ?>
                                                        <i class='bx <?= $i <= $review_stats['average_rating'] ? 'bxs-star' : 'bx-star' ?>'></i>
                                                    <?php endfor; ?>
                                                </div>
                                                <div class="rating-text"><?= $review_stats['total_reviews'] ?> student review<?= $review_stats['total_reviews'] != 1 ? 's' : '' ?></div>
                                            </div>
                                        </div>
                                    </div>
                                    
                                    <div class="col-lg-8">
                                        <div class="rating-breakdown-card">
                                            <h6 class="breakdown-title">Rating Breakdown</h6>
                                            <?php for ($star = 5; $star >= 1; $star--): ?>
                                                <?php 
                                                $count = $review_stats['rating_breakdown'][$star];
                                                $percentage = $review_stats['total_reviews'] > 0 ? ($count / $review_stats['total_reviews']) * 100 : 0;
                                                ?>
                                                <div class="rating-breakdown-row">
                                                    <span class="star-label"><?= $star ?> stars</span>
                                                    <div class="rating-bar">
                                                        <div class="rating-bar-fill" style="width: <?= $percentage ?>%"></div>
                                                    </div>
                                                    <span class="rating-count"><?= $count ?></span>
                                                </div>
                                            <?php endfor; ?>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <!-- Write Review Section -->
                            <?php if (isset($_SESSION['user_id']) && $user_enrolled && !$user_has_reviewed): ?>
                            <div class="write-review-section">
                                <div class="write-review-card">
                                    <div class="write-review-header">
                                        <h5><i class="bx bx-edit"></i> Share Your Experience</h5>
                                        <p>Help other students by sharing your experience with this course</p>
                                    </div>
                                    
                                    <form id="reviewForm" class="review-form">
                                        <input type="hidden" name="course_id" value="<?= encryptId($course_id) ?>">
                                        <input type="hidden" name="user_id" value="<?= $_SESSION['user_id'] ?? '' ?>">
                                        <input type="hidden" name="user_token" value="<?= md5(($_SESSION['user_id'] ?? '') . 'panadite_review_token') ?>">
                                        
                                        <div class="form-group">
                                            <label class="form-label">Your Rating</label>
                                            <div class="star-rating-input" style="font-size: 1.5rem; margin: 10px 0;">
                                                <?php for ($i = 1; $i <= 5; $i++): ?>
                                                    <i class="bx bx-star rating-star" data-rating="<?= $i ?>" 
                                                       onclick="selectRating(<?= $i ?>)" 
                                                       onmouseover="previewRating(<?= $i ?>)" 
                                                       onmouseout="resetPreview()" 
                                                       style="cursor: pointer; color: #dee2e6; margin-right: 5px; transition: color 0.2s ease;"></i>
                                                <?php endfor; ?>
                                            </div>
                                            <small class="text-muted">Click the stars to rate this course</small>
                                            
                                            <script>
                                            // ⭐ INLINE STAR RATING - GUARANTEED TO WORK
                                            let selectedRating = 0;
                                            
                                            function selectRating(rating) {
                                                selectedRating = rating;
                                                updateStars(rating);
                                                document.getElementById('selectedRating').value = rating;
                                                console.log('⭐ INLINE: Star rating selected:', rating);
                                            }
                                            
                                            function previewRating(rating) {
                                                updateStars(rating, true);
                                            }
                                            
                                            function resetPreview() {
                                                updateStars(selectedRating);
                                            }
                                            
                                            function updateStars(rating, isPreview = false) {
                                                const stars = document.querySelectorAll('.rating-star');
                                                console.log('⭐ INLINE: Updating', stars.length, 'stars to rating', rating);
                                                stars.forEach((star, index) => {
                                                    const starRating = parseInt(star.getAttribute('data-rating'));
                                                    if (starRating <= rating) {
                                                        // Use correct BoxIcons filled star class
                                                        star.className = 'bx bxs-star rating-star'; 
                                                        star.style.color = isPreview ? '#ffc107' : '#ff6b35';
                                                        star.style.fontSize = '1.5rem';
                                                        console.log('⭐ FILLED: Star', starRating, 'set to filled, color:', star.style.color);
                                                    } else {
                                                        // Use correct BoxIcons empty star class  
                                                        star.className = 'bx bx-star rating-star';
                                                        star.style.color = '#dee2e6';
                                                        star.style.fontSize = '1.5rem';
                                                        console.log('⭐ EMPTY: Star', starRating, 'set to empty');
                                                    }
                                                });
                                            }
                                            
                                            // Initialize on page load
                                            document.addEventListener('DOMContentLoaded', function() {
                                                setTimeout(() => {
                                                    console.log('⭐ INLINE: Star rating system ready!');
                                                    // Test if stars exist
                                                    const stars = document.querySelectorAll('.rating-star');
                                                    console.log('⭐ INLINE: Found', stars.length, 'stars');
                                                    
                                                    // Initialize AJAX form submission
                                                    initializeReviewFormAjax();
                                                }, 500);
                                            });
                                            
                                            // 📝 AJAX REVIEW FORM SUBMISSION - PREVENT PAGE REDIRECT
                                            function initializeReviewFormAjax() {
                                                const reviewForm = document.getElementById('reviewForm');
                                                if (!reviewForm) {
                                                    console.log('📝 No review form found on this page');
                                                    return;
                                                }
                                                
                                                console.log('📝 Initializing AJAX review form submission');
                                                
                                                reviewForm.addEventListener('submit', async function(e) {
                                                    e.preventDefault(); // PREVENT DEFAULT FORM SUBMISSION
                                                    console.log('📝 AJAX: Form submitted, preventing redirect');
                                                    
                                                    // Validate star rating
                                                    const selectedRating = document.getElementById('selectedRating');
                                                    if (!selectedRating || !selectedRating.value) {
                                                        alert('Please select a star rating before submitting your review.');
                                                        return false;
                                                    }
                                                    
                                                    // Validate form fields
                                                    const reviewTitle = document.getElementById('reviewTitle').value.trim();
                                                    const reviewText = document.getElementById('reviewText').value.trim();
                                                    
                                                    if (reviewTitle.length < 3) {
                                                        alert('Review title must be at least 3 characters long.');
                                                        return false;
                                                    }
                                                    
                                                    if (reviewText.length < 10) {
                                                        alert('Review text must be at least 10 characters long.');
                                                        return false;
                                                    }
                                                    
                                                    // Show loading state
                                                    const submitBtn = reviewForm.querySelector('button[type="submit"]');
                                                    const originalText = submitBtn.innerHTML;
                                                    submitBtn.disabled = true;
                                                    submitBtn.innerHTML = '<i class="bx bx-loader-alt bx-spin"></i> Submitting...';
                                                    
                                                    try {
                                                        // Prepare form data
                                                        const formData = new FormData(reviewForm);
                                                        
                                                        console.log('📝 AJAX: Sending review data:', {
                                                            rating: selectedRating.value,
                                                            title: reviewTitle,
                                                            text: reviewText
                                                        });
                                                        
                                                        // Submit via AJAX
                                                        const response = await fetch('submit_review.php', {
                                                            method: 'POST',
                                                            body: formData
                                                        });
                                                        
                                                        console.log('📡 AJAX: Raw response status:', response.status, response.statusText);
                                                        console.log('📡 AJAX: Response headers:', response.headers.get('Content-Type'));
                                                        
                                                        // Get raw text first to debug what we're receiving
                                                        const rawText = await response.text();
                                                        console.log('📄 AJAX: Raw response text:', rawText);
                                                        console.log('📄 AJAX: Raw response length:', rawText.length);
                                                        console.log('📄 AJAX: First 200 chars:', rawText.substring(0, 200));
                                                        
                                                        // Try to parse JSON
                                                        let result;
                                                        try {
                                                            result = JSON.parse(rawText);
                                                            console.log('✅ AJAX: Successfully parsed JSON:', result);
                                                        } catch (parseError) {
                                                            console.error('❌ AJAX: JSON Parse Error:', parseError);
                                                            console.error('❌ AJAX: Failed to parse this response:', rawText);
                                                            throw new Error('Server returned invalid JSON: ' + rawText.substring(0, 100));
                                                        }
                                                        
                                                        if (result.success) {
                                                            // Success! Show confirmation and update UI
                                                            alert('Thank you! Your review has been submitted successfully.');
                                                            
                                                            // Hide the review form and show success message
                                                            const reviewSection = document.querySelector('.write-review-section');
                                                            if (reviewSection) {
                                                                reviewSection.innerHTML = `
                                                                    <div class="alert alert-success">
                                                                        <i class="bx bx-check-circle"></i>
                                                                        <strong>Review Submitted!</strong> Thank you for sharing your experience with this course.
                                                                    </div>
                                                                `;
                                                            }
                                                            
                                                            // Refresh reviews section or reload page after delay
                                                            setTimeout(() => {
                                                                location.reload();
                                                            }, 2000);
                                                            
                                                        } else {
                                                            // Show error message
                                                            alert('Error: ' + (result.message || 'Failed to submit review. Please try again.'));
                                                        }
                                                        
                                                    } catch (error) {
                                                        console.error('📝 AJAX: Error submitting review:', error);
                                                        alert('Network error. Please check your connection and try again.');
                                                    } finally {
                                                        // Restore button state
                                                        submitBtn.disabled = false;
                                                        submitBtn.innerHTML = originalText;
                                                    }
                                                    
                                                    return false;
                                                });
                                                
                                                console.log('📝 AJAX review form initialized successfully');
                                            }
                                            </script>
                                            <input type="hidden" name="rating" id="selectedRating" required>
                                        </div>
                                        
                                        <div class="form-group">
                                            <label for="reviewTitle" class="form-label">Review Title <small class="text-muted">(minimum 3 characters)</small></label>
                                            <input type="text" class="form-control" id="reviewTitle" name="review_title" 
                                                   placeholder="Summarize your experience... (minimum 3 characters required)" 
                                                   minlength="3" maxlength="255" required>
                                            <div class="form-text">
                                                <span id="titleCharCount">0</span>/3 characters minimum
                                            </div>
                                        </div>
                                        
                                        <div class="form-group">
                                            <label for="reviewText" class="form-label">Your Review <small class="text-muted">(minimum 10 characters)</small></label>
                                            <textarea class="form-control" id="reviewText" name="review_text" rows="4" 
                                                      placeholder="Share details about your experience with this course... (minimum 10 characters required)" 
                                                      minlength="10" required></textarea>
                                            <div class="form-text">
                                                <span id="charCount">0</span>/10 characters minimum
                                            </div>
                                        </div>
                                        
                                        <div class="form-actions">
                                            <button type="submit" class="btn btn-primary">
                                                <i class="bx bx-send"></i> Submit Review
                                            </button>
                                            <button type="button" class="btn btn-outline-secondary" onclick="cancelReview()">
                                                Cancel
                                            </button>
                                        </div>
                                    </form>
                                </div>
                            </div>
                            <?php elseif (isset($_SESSION['user_id']) && $user_has_reviewed): ?>
                            <div class="user-review-section">
                                <div class="user-review-card">
                                    <div class="user-review-header">
                                        <h6><i class="bx bx-check-circle"></i> Your Review</h6>
                                        <div class="review-actions">
                                            <button class="btn btn-sm btn-outline-primary" onclick="editReview(<?= $user_review['review_id'] ?>, '<?= htmlspecialchars($user_review['review_title'], ENT_QUOTES) ?>', '<?= htmlspecialchars($user_review['review_text'], ENT_QUOTES) ?>', <?= $user_review['rating'] ?>)">
                                                <i class="bx bx-edit"></i> Edit
                                            </button>
                                        </div>
                                    </div>
                                    <div class="user-review-content">
                                        <div class="review-rating">
                                            <?php for ($i = 1; $i <= 5; $i++): ?>
                                                <i class='bx <?= $i <= $user_review['rating'] ? 'bxs-star' : 'bx-star' ?>'></i>
                                            <?php endfor; ?>
                                        </div>
                                        <h6 class="review-title"><?= htmlspecialchars($user_review['review_title']) ?></h6>
                                        <p class="review-text"><?= htmlspecialchars($user_review['review_text']) ?></p>
                                        <small class="review-date">Reviewed on <?= date('M d, Y', strtotime($user_review['created_at'])) ?></small>
                                    </div>
                                </div>
                            </div>
                            <?php elseif (!isset($_SESSION['user_id'])): ?>
                            <div class="login-prompt-section">
                                <div class="login-prompt-card">
                                    <i class="bx bx-user-circle" style="font-size: 2.5rem; color: #6b7280; margin-bottom: 1rem;"></i>
                                    <h6>Sign in to leave a review</h6>
                                    <p>Share your experience and help other students</p>
                                    <a href="../login.php" class="btn btn-primary btn-sm">
                                        <i class="bx bx-log-in"></i> Sign In
                                    </a>
                                </div>
                            </div>
                            <?php elseif (!$user_enrolled): ?>
                            <div class="enrollment-prompt-section">
                                <div class="enrollment-prompt-card">
                                    <i class="bx bx-lock-alt" style="font-size: 2.5rem; color: #6b7280; margin-bottom: 1rem;"></i>
                                    <h6>Enroll to leave a review</h6>
                                    <p>Only enrolled students can leave reviews</p>
                                </div>
                            </div>
                            <?php endif; ?>

                            <!-- Reviews List -->
                            <div class="reviews-list-section">
                                <div class="reviews-list-header">
                                    <h5><i class="bx bx-message-square-detail"></i> Student Reviews</h5>
                                    <?php if ($review_stats['total_reviews'] > 0): ?>
                                        <div class="dropdown">
                                            <button class="btn btn-outline-secondary dropdown-toggle" type="button" id="reviewSortDropdown" data-toggle="dropdown" aria-expanded="false">
                                                Newest First
                                            </button>
                                            <ul class="dropdown-menu" aria-labelledby="reviewSortDropdown">
                                                <li><a class="dropdown-item" href="#" onclick="sortReviews('newest')">Newest First</a></li>
                                                <li><a class="dropdown-item" href="#" onclick="sortReviews('oldest')">Oldest First</a></li>
                                                <li><a class="dropdown-item" href="#" onclick="sortReviews('highest')">Highest Rated</a></li>
                                                <li><a class="dropdown-item" href="#" onclick="sortReviews('lowest')">Lowest Rated</a></li>
                                                <li><a class="dropdown-item" href="#" onclick="sortReviews('helpful')">Most Helpful</a></li>
                                            </ul>
                                        </div>
                                    <?php endif; ?>
                                </div>

                                <?php if (empty($reviews)): ?>
                                <div class="no-reviews-placeholder">
                                    <div class="no-reviews-icon">
                                        <i class="bx bx-message-square-dots"></i>
                                    </div>
                                    <h6>No Reviews Yet</h6>
                                    <p>Be the first to share your experience with this course!</p>
                                    <?php if (isset($_SESSION['user_id']) && $user_enrolled && !$user_has_reviewed): ?>
                                        <button class="btn btn-primary btn-sm" onclick="showReviewForm()">
                                            <i class="bx bx-plus"></i> Write First Review
                                        </button>
                                    <?php endif; ?>
                                </div>
                                <?php else: ?>
                                <div class="reviews-list" id="reviewsList">
                                    <?php foreach ($reviews as $review): ?>
                                    <div class="review-item-enhanced">
                                        <div class="review-avatar">
                                            <?php if (!empty($review['profile_picture'])): ?>
                                                <img src="<?= htmlspecialchars($review['profile_picture']) ?>" alt="Reviewer">
                                            <?php else: ?>
                                                <div class="avatar-placeholder">
                                                    <i class="bx bx-user"></i>
                                                </div>
                                            <?php endif; ?>
                                        </div>
                                        
                                        <div class="review-content">
                                            <div class="review-header">
                                                <div class="reviewer-info">
                                                    <h6 class="reviewer-name"><?= htmlspecialchars($review['first_name'] . ' ' . substr($review['last_name'], 0, 1) . '.') ?></h6>
                                                    <div class="review-meta">
                                                        <div class="review-rating">
                                                            <?php for ($i = 1; $i <= 5; $i++): ?>
                                                                <i class='bx <?= $i <= $review['rating'] ? 'bxs-star' : 'bx-star' ?>'></i>
                                                            <?php endfor; ?>
                                                        </div>
                                                        <span class="review-date"><?= date('M d, Y', strtotime($review['created_at'])) ?></span>
                                                    </div>
                                                </div>
                                            </div>
                                            
                                            <div class="review-body">
                                                <h6 class="review-title"><?= htmlspecialchars($review['review_title']) ?></h6>
                                                <p class="review-text"><?= nl2br(htmlspecialchars($review['review_text'])) ?></p>
                                            </div>
                                            
                                            <div class="review-actions">
                                                <button class="review-action-btn helpful-btn" data-review-id="<?= encryptId($review['review_id']) ?>">
                                                    <i class="bx bx-like"></i>
                                                    <span class="helpful-count">Helpful (<?= $review['helpful_votes'] ?>)</span>
                                                </button>
                                                <?php if ($review['reply_count'] > 0): ?>
                                                    <button class="review-action-btn" onclick="showReplies('<?= encryptId($review['review_id']) ?>')">
                                                        <i class="bx bx-message"></i>
                                                        <span>Instructor Reply</span>
                                                    </button>
                                                <?php endif; ?>
                                            </div>
                                        </div>
                                    </div>
                                    <?php endforeach; ?>
                                </div>
                                
                                <?php if (count($reviews) >= 20): ?>
                                <div class="load-more-section">
                                    <button class="btn btn-outline-primary" onclick="loadMoreReviews()">
                                        <i class="bx bx-download"></i> Load More Reviews
                                    </button>
                                </div>
                                <?php endif; ?>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>

                    <!-- SITA-Compliant Accredited Certificate Tab -->
                    <div class="tab-pane" id="certificate">
                        <?php 
                        // Include accreditation logic
                        include_once 'includes/accreditation_logic.php';
                        
                        // ===============================================
                        // 🔧 FIXED: MISSING CERTIFICATE ELIGIBILITY CALCULATIONS
                        // ===============================================
                        
                        // 1. Calculate Video Completion (using CORRECT table)
                        $video_completion_query = "
                            SELECT 
                                COUNT(DISTINCT v.video_id) as total_videos,
                                COUNT(DISTINCT uvc.video_id) as completed_videos
                            FROM videos v
                            LEFT JOIN user_video_completions uvc ON v.video_id = uvc.video_id 
                                AND uvc.user_id = ? AND uvc.completion_percentage >= 100
                            WHERE v.course_id = ?
                        ";
                        
                        $total_videos = 0;
                        $completed_videos = 0;
                        
                        if ($stmt = $conn->prepare($video_completion_query)) {
                            $stmt->bind_param("ii", $user_id, $course_id);
                            $stmt->execute();
                            $result = $stmt->get_result();
                            if ($row = $result->fetch_assoc()) {
                                $total_videos = (int)$row['total_videos'];
                                $completed_videos = (int)$row['completed_videos'];
                            }
                            $stmt->close();
                        }
                        
                        // 2. Calculate Quiz Requirements
                        $quiz_completion_query = "
                            SELECT 
                                COUNT(DISTINCT q.quiz_id) as total_quizzes,
                                COUNT(DISTINCT CASE 
                                    WHEN qa.pass_status = 1 THEN qa.quiz_id 
                                    ELSE NULL 
                                END) as passed_quizzes
                            FROM quizzes q
                            LEFT JOIN quiz_attempts qa ON q.quiz_id = qa.quiz_id 
                                AND qa.student_id = ?
                            WHERE q.course_id = ? AND q.is_published = 1
                        ";
                        
                        $total_quizzes = 0;
                        $passed_quizzes = 0;
                        
                        if ($stmt = $conn->prepare($quiz_completion_query)) {
                            $stmt->bind_param("ii", $user_id, $course_id);
                            $stmt->execute();
                            $result = $stmt->get_result();
                            if ($row = $result->fetch_assoc()) {
                                $total_quizzes = (int)$row['total_quizzes'];
                                $passed_quizzes = (int)$row['passed_quizzes'];
                            }
                            $stmt->close();
                        }
                        
                        // 3. Get Course Requirements
                        $course_requirements_query = "
                            SELECT course_passing_grade, assignment_weight, quiz_weight
                            FROM courses 
                            WHERE course_id = ?
                        ";
                        
                        $course_passing_grade = 60; // Default
                        $assignment_weight = 70;
                        $quiz_weight = 30;
                        
                        if ($stmt = $conn->prepare($course_requirements_query)) {
                            $stmt->bind_param("i", $course_id);
                            $stmt->execute();
                            $result = $stmt->get_result();
                            if ($row = $result->fetch_assoc()) {
                                $course_passing_grade = (float)$row['course_passing_grade'];
                                $assignment_weight = (float)$row['assignment_weight'];
                                $quiz_weight = (float)$row['quiz_weight'];
                            }
                            $stmt->close();
                        }
                        
                        // 4. Calculate Certificate Eligibility
                        $video_completion_percentage = $total_videos > 0 ? ($completed_videos / $total_videos) * 100 : 0;
                        $quiz_completion_percentage = $total_quizzes > 0 ? ($passed_quizzes / $total_quizzes) * 100 : 100; // 100% if no quizzes
                        
                        // FIXED: Video requirement = 100% completion
                        $full_course_completed = ($video_completion_percentage >= 100);
                        
                        // FIXED: Quiz requirement = meets course pass grade
                        $quiz_requirements_met = ($quiz_completion_percentage >= $course_passing_grade);
                        
                        // If no quizzes exist, only video completion matters
                        if ($total_quizzes == 0) {
                            $quiz_requirements_met = true;
                        }
                        
                        // Check enrollment status
                        $is_enrolled = false;
                        $enrollment_check = "SELECT 1 FROM enrollments WHERE user_id = ? AND course_id = ?";
                        if ($stmt = $conn->prepare($enrollment_check)) {
                            $stmt->bind_param("ii", $user_id, $course_id);
                            $stmt->execute();
                            $result = $stmt->get_result();
                            $is_enrolled = $result->num_rows > 0;
                            $stmt->close();
                        }
                        

                        ?>
                        
                        <div class="accreditation-content">
                            

                            
                            <div class="accreditation-header border rounded mb-4 p-4" style="border-color: #e2e8f0;">
                                <div class="row align-items-center">
                                    <div class="col-md-8">
                                        <h3 class="mb-2 text-dark"><i class="bx bx-award me-2 text-primary"></i> SITA-Accredited Professional Certificate</h3>
                                        <p class="mb-0 text-muted">Request an official, industry-recognized accredited certificate from Panadite Academy</p>
                                        </div>
                                        <div class="col-md-4 text-md-end">
                                            <div class="accreditation-badge bg-white rounded p-3 text-center" style="border: 1px solid #e2e8f0;">
                                                <img src="../assets/images/sita-logo.png" alt="SITA Accredited" class="sita-logo mb-2" style="height: 40px;" onerror="this.style.display='none'">
                                                <div class="badge bg-primary">SITA Compliant</div>
                                            </div>
                                        </div>
                                    </div>
                                </div>

                            <?php if (!$accreditation_request): ?>
                            <!-- Initial Request Stage -->
                            <div class="accreditation-stage" id="initialRequest">
                                <!-- SITA Information Accordion -->
                                <div class="accordion" id="sitaInfoAccordion">
                                    <!-- Requirements Overview -->
                                    <div class="accordion-item">
                                        <h2 class="accordion-header" id="headingRequirements">
                                            <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#collapseRequirements" aria-expanded="false" aria-controls="collapseRequirements">
                                                <i class="bx bx-task text-primary me-2"></i>
                                                <strong>Certificate Requirements</strong>
                                                <small class="text-muted ms-2">(<?= $videos_completed && $quiz_requirements_met ? 'Ready' : 'In Progress' ?>)</small>
                                            </button>
                                        </h2>
                                        <div id="collapseRequirements" class="accordion-collapse collapse" aria-labelledby="headingRequirements" data-bs-parent="#sitaInfoAccordion">
                                            <div class="accordion-body">
                                                <div class="requirements-grid">
                                                    <div class="requirement-card <?= $videos_completed ? 'completed' : '' ?>">
                                                        <i class="bx <?= $videos_completed ? 'bx-check-circle text-success' : 'bx-circle text-muted' ?> me-2"></i>
                                                        <span>Complete all course videos (<?= $course_progress_percentage ?>% done)</span>
                                                    </div>
                                                    <div class="requirement-card <?= $quiz_requirements_met ? 'completed' : '' ?>">
                                                        <i class="bx <?= $quiz_requirements_met ? 'bx-check-circle text-success' : 'bx-circle text-muted' ?> me-2"></i>
                                                        <span>Achieve passing grade in quizzes (<?= number_format($overall_course_mark, 1) ?>% / <?= $certificate_eligibility['minimum_required'] ?>%)</span>
                                                    </div>
                                                    <div class="requirement-card <?= $is_enrolled ? 'completed' : '' ?>">
                                                        <i class="bx <?= $is_enrolled ? 'bx-check-circle text-success' : 'bx-circle text-muted' ?> me-2"></i>
                                                        <span>Active course enrollment</span>
                                                    </div>
                                                    <div class="requirement-card">
                                                        <i class="bx bx-circle text-muted me-2"></i>
                                                        <span>Pass SITA assessment (minimum 70%)</span>
                                                    </div>
                                                    <div class="requirement-card">
                                                        <i class="bx bx-circle text-muted me-2"></i>
                                                        <span>Submit required documents</span>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>

                                    <!-- About SITA Accreditation -->
                                    <div class="accordion-item">
                                        <h2 class="accordion-header" id="headingAbout">
                                            <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#collapseAbout" aria-expanded="false" aria-controls="collapseAbout">
                                                <i class="bx bx-shield-check text-primary me-2"></i>
                                                <strong>About SITA Accreditation</strong>
                                            </button>
                                        </h2>
                                        <div id="collapseAbout" class="accordion-collapse collapse" aria-labelledby="headingAbout" data-bs-parent="#sitaInfoAccordion">
                                            <div class="accordion-body">
                                                <p class="text-muted">SITA (Skills and Industry Training Authority) accreditation ensures your certificate meets professional industry standards and is recognized by employers worldwide. This certification carries significant weight in the professional market and demonstrates your commitment to excellence.</p>
                                                
                                                <div class="alert alert-info">
                                                    <div class="d-flex align-items-center">
                                                        <i class="bx bx-time text-info me-2" style="font-size: 1.5rem;"></i>
                                                        <div>
                                                            <h6 class="alert-heading mb-1">Processing Timeline</h6>
                                                            <p class="mb-0"><strong>Total Processing Time:</strong> Up to 5 months from initial request submission</p>
                                                            <small class="text-muted">SITA accreditation is a comprehensive process that ensures the highest professional standards.</small>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>

                                    <!-- Process Overview -->
                                    <div class="accordion-item">
                                        <h2 class="accordion-header" id="headingProcess">
                                            <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#collapseProcess" aria-expanded="false" aria-controls="collapseProcess">
                                                <i class="bx bx-list-check text-primary me-2"></i>
                                                <strong>6-Step Process Overview</strong>
                                            </button>
                                        </h2>
                                        <div id="collapseProcess" class="accordion-collapse collapse" aria-labelledby="headingProcess" data-bs-parent="#sitaInfoAccordion">
                                            <div class="accordion-body">
                                                <div class="process-steps">
                                                    <div class="step-item">
                                                        <span class="step-number">1</span>
                                                        <div class="step-content">
                                                            <strong>Request Submission:</strong> Submit your accreditation request
                                                        </div>
                                                    </div>
                                                    <div class="step-item">
                                                        <span class="step-number">2</span>
                                                        <div class="step-content">
                                                            <strong>Admin Review:</strong> Our team reviews your eligibility
                                                        </div>
                                                    </div>
                                                    <div class="step-item">
                                                        <span class="step-number">3</span>
                                                        <div class="step-content">
                                                            <strong>SITA Assessment:</strong> Complete a comprehensive final assessment
                                                        </div>
                                                    </div>
                                                    <div class="step-item">
                                                        <span class="step-number">4</span>
                                                        <div class="step-content">
                                                            <strong>Document Submission:</strong> Upload CV, qualifications, ID, and assignment work
                                                        </div>
                                                    </div>
                                                    <div class="step-item">
                                                        <span class="step-number">5</span>
                                                        <div class="step-content">
                                                            <strong>Final Review:</strong> Admin team validates all requirements
                                                        </div>
                                                    </div>
                                                    <div class="step-item">
                                                        <span class="step-number">6</span>
                                                        <div class="step-content">
                                                            <strong>Certificate Issuance:</strong> Receive your official accredited certificate
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>

                                <?php if ($full_course_completed && $quiz_requirements_met && $is_enrolled && !$accreditation_request): ?>
                                <div class="card shadow-sm mt-4">
                                    <div class="card-header bg-success text-white">
                                        <h5 class="card-title mb-0"><i class="bx bx-credit-card me-2"></i> Get Your SITA Accreditation Certificate</h5>
                                    </div>
                                    <div class="card-body p-4">
                                        <!-- Payment-First Information -->
                                        <div class="row">
                                            <div class="col-md-8">
                                                <div class="accreditation-info">
                                                    <h6 class="fw-bold mb-3">🎓 Official SITA-Accredited Certificate</h6>
                                                    <ul class="list-unstyled mb-3">
                                                        <li class="mb-2"><i class="bx bx-check text-success me-2"></i>Industry-recognized professional certification</li>
                                                        <li class="mb-2"><i class="bx bx-check text-success me-2"></i>SITA assessment and skills evaluation</li>
                                                        <li class="mb-2"><i class="bx bx-check text-success me-2"></i>Official certificate with verification code</li>
                                                        <li class="mb-2"><i class="bx bx-check text-success me-2"></i>Career advancement opportunities</li>
                                                    </ul>
                                                    
                                                    <div class="alert alert-info mb-0">
                                                        <h6 class="alert-heading mb-2"><i class="bx bx-info-circle me-1"></i> Process Timeline</h6>
                                                        <small>
                                                            <strong>Step 1:</strong> Pay accreditation fee (secure PayFast payment)<br>
                                                            <strong>Step 2:</strong> Admin review within 5 business days<br>
                                                            <strong>Step 3:</strong> Complete SITA assessment & submit documents<br>
                                                            <strong>Step 4:</strong> Receive official certificate (up to 5 months total)
                                                        </small>
                                                    </div>
                                                </div>
                                            </div>
                                            <div class="col-md-4">
                                                <div class="payment-section text-center">
                                                    <div class="price-display mb-3">
                                                        <div class="price-label text-muted">Accreditation Fee</div>
                                                        <div class="price-amount h2 text-success fw-bold">R500</div>
                                                        <small class="text-muted">One-time payment</small>
                                                    </div>
                                                    
                                                    <!-- Payment Button Form -->
                                                    <form id="accreditationPaymentForm" action="api/submit_accreditation_request.php" method="POST">
                                                        <input type="hidden" name="course_id" value="<?= $course_id ?>">
                                                        <input type="hidden" name="student_id" value="<?= $user_id ?>">
                                                        <input type="hidden" name="motivation" value="Student requesting SITA accreditation for <?= htmlspecialchars($course['title']) ?>">
                                                        
                                                        <div class="d-grid gap-2">
                                                            <button type="submit" class="btn btn-success btn-lg">
                                                                <i class="bx bx-credit-card me-2"></i>
                                                                Pay R500 & Request Certificate
                                                            </button>
                                                            <small class="text-muted">Secure payment via PayFast</small>
                                                        </div>
                                                    </form>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                                <?php else: ?>
                                <div class="card shadow-sm mt-4">
                                    <div class="card-body text-center py-5">
                                        <div class="mb-3">
                                            <i class="bx bx-lock-alt text-warning" style="font-size: 4rem;"></i>
                                        </div>
                                        <h5 class="text-warning mb-3">Complete All Course Requirements</h5>
                                        <p class="text-muted mb-4">You must complete all videos AND achieve the passing grade in quizzes to earn your certificate.</p>
                                        
                                        <!-- Comprehensive Requirements Progress -->
                                        <div class="requirements-progress mb-4">
                                            <div class="row text-start">
                                                <div class="col-md-6">
                                                    <div class="requirement-item mb-3">
                                                        <div class="d-flex align-items-center">
                                                            <i class="bx <?= $videos_completed ? 'bx-check-circle text-success' : 'bx-circle text-muted' ?> me-2"></i>
                                                            <div>
                                                                <strong>Video Completion:</strong>
                                                                <div class="progress mt-1" style="height: 8px;">
                                                                    <div class="progress-bar bg-primary" role="progressbar" style="width: <?= $course_progress_percentage ?>%" id="certificate-video-progress-bar">
                                                                    </div>
                                                                </div>
                                                                <small class="text-muted" id="certificate-video-progress-text"><?= $course_progress_percentage ?>% videos watched</small>
                                                            </div>
                                                        </div>
                                                    </div>
                                                </div>
                                                <div class="col-md-6">
                                                    <div class="requirement-item mb-3">
                                                        <div class="d-flex align-items-center">
                                                            <i class="bx <?= $quiz_requirements_met ? 'bx-check-circle text-success' : 'bx-circle text-muted' ?> me-2"></i>
                                                            <div>
                                                                <strong>Quiz Performance:</strong>
                                                                <div class="progress mt-1" style="height: 8px;">
                                                                    <div class="progress-bar <?= $quiz_requirements_met ? 'bg-success' : 'bg-warning' ?>" role="progressbar" style="width: <?= min(100, ($overall_course_mark / $certificate_eligibility['minimum_required']) * 100) ?>%">
                                                                    </div>
                                                                </div>
                                                                <small class="text-muted">
                                                                    <?= number_format($overall_course_mark, 1) ?>% / <?= $certificate_eligibility['minimum_required'] ?>% required
                                                                </small>
                                                            </div>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                        
                                        <!-- Overall Course Grade Display -->
                                        <div class="overall-grade-display mb-4">
                                            <div class="card bg-light">
                                                <div class="card-body py-3">
                                                    <h6 class="mb-1">Overall Course Grade</h6>
                                                    <div class="grade-display">
                                                        <span class="h4 <?= $quiz_requirements_met ? 'text-success' : 'text-warning' ?>">
                                                            <?= number_format($overall_course_mark, 1) ?>%
                                                        </span>
                                                        <small class="text-muted ms-2">
                                                            (Passing: <?= $certificate_eligibility['minimum_required'] ?>%)
                                                        </small>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                        
                                        <!-- Next Steps -->
                                        <div class="next-steps">
                                            <?php if (!$videos_completed && !$quiz_requirements_met): ?>
                                                <p class="text-info mb-2"><strong>Next Steps:</strong> Complete all videos and improve your quiz scores</p>
                                                <div class="d-flex gap-2 justify-content-center">
                                                    <a href="#lessons" class="btn btn-outline-primary btn-sm">Continue Videos</a>
                                                    <a href="#quizzes" class="btn btn-outline-success btn-sm">Take Quizzes</a>
                                                </div>
                                            <?php elseif (!$videos_completed): ?>
                                                <p class="text-info mb-2"><strong>Next Step:</strong> Complete all remaining videos</p>
                                                <a href="#lessons" class="btn btn-outline-primary">Continue Learning</a>
                                            <?php elseif (!$quiz_requirements_met): ?>
                                                <p class="text-info mb-2"><strong>Next Step:</strong> Improve your quiz performance</p>
                                                <a href="#quizzes" class="btn btn-outline-success">Take Quizzes</a>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                </div>
                                <?php endif; ?>
                            </div>
                            <?php else: ?>
                            <!-- Request Status and Progress -->
                            <div class="accreditation-progress">
                                <div class="status-header mb-4">
                                    <div class="row align-items-center">
                                        <div class="col-md-8">
                                            <h5>Your Accreditation Request</h5>
                                            <p class="mb-0">Request ID: #<?= str_pad($accreditation_request['request_id'], 6, '0', STR_PAD_LEFT) ?></p>
                                        </div>
                                        <div class="col-md-4 text-md-end">
                                            <?= getStatusBadge($accreditation_request['status']) ?>
                                        </div>
                                    </div>
                                </div>


                                
                                <div class="status-timeline mb-4">
                                    <div class="timeline-item <?= in_array($accreditation_request['status'], ['pending', 'approved', 'assessment_created', 'assessment_completed', 'documents_pending', 'documents_submitted', 'under_review', 'approved_final', 'certificate_issued']) ? 'completed' : 'active' ?>">
                                        <div class="timeline-marker"><i class="bx bx-send"></i></div>
                                        <div class="timeline-content">
                                            <h6>Request Submitted</h6>
                                            <small><?= date('M d, Y \a\t H:i', strtotime($accreditation_request['request_date'])) ?></small>
                                        </div>
                                    </div>
                                    
                                    <div class="timeline-item <?= in_array($accreditation_request['status'], ['approved', 'assignment_created', 'assessment_created', 'assessment_completed', 'documents_pending', 'documents_submitted', 'under_review', 'approved_final', 'certificate_issued', 'assessment_passed']) ? 'completed' : (in_array($accreditation_request['status'], ['pending']) ? 'active' : '') ?>">
                                        <div class="timeline-marker"><i class="bx bx-user-check"></i></div>
                                        <div class="timeline-content">
                                            <h6>Admin Approval</h6>
                                            <?php if ($accreditation_request['approved_date']): ?>
                                                <small class="text-success"><i class="bx bx-check-circle"></i> <?= date('M d, Y \a\t H:i', strtotime($accreditation_request['approved_date'])) ?></small>
                                            <?php elseif (in_array($accreditation_request['status'], ['assignment_created', 'assessment_created'])): ?>
                                                <small class="text-success"><i class="bx bx-check-circle"></i> Approved - Assignment Ready</small>
                                            <?php else: ?>
                                                <small>Pending review</small>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                    
                                    <div class="timeline-item <?= in_array($accreditation_request['status'], ['documents_submitted', 'assessment_passed', 'assessment_completed', 'documents_pending', 'under_review', 'approved_final', 'certificate_issued']) ? 'completed' : (in_array($accreditation_request['status'], ['assignment_created', 'assessment_created']) ? 'active' : '') ?>">
                                        <div class="timeline-marker"><i class="bx bx-brain"></i></div>
                                        <div class="timeline-content">
                                            <h6>SITA Assessment</h6>
                                            <?php if (in_array($accreditation_request['status'], ['documents_submitted', 'assessment_passed', 'assessment_completed', 'under_review', 'approved_final', 'certificate_issued'])): ?>
                                                <small class="text-success"><i class="bx bx-check-circle"></i> Assignment Completed</small>
                                            <?php elseif ($accreditation_request['status'] === 'assignment_created'): ?>
                                                <small class="text-primary">
                                                    <a href="assignments.php" class="btn btn-sm btn-outline-primary mt-1">
                                                        <i class="bx bx-play-circle"></i> Start Assignment
                                                    </a>
                                                </small>
                                            <?php elseif ($assessment_attempt && $assessment_attempt['end_time']): ?>
                                                <small>Completed: <?= $assessment_attempt['percentage_score'] ?>%</small>
                                            <?php else: ?>
                                                <small>Awaiting completion</small>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                    
                                    <div class="timeline-item <?= in_array($accreditation_request['status'], ['documents_submitted', 'under_review', 'approved_final', 'certificate_issued']) ? 'completed' : (in_array($accreditation_request['status'], ['documents_pending']) ? 'active' : '') ?>">
                                        <div class="timeline-marker"><i class="bx bx-folder-plus"></i></div>
                                        <div class="timeline-content">
                                            <h6>Document Submission</h6>
                                            <small>Documents processed</small>
                                        </div>
                                    </div>
                                    
                                    <div class="timeline-item <?= in_array($accreditation_request['status'], ['certificate_issued']) ? 'completed' : (in_array($accreditation_request['status'], ['under_review', 'approved_final']) ? 'active' : '') ?>">
                                        <div class="timeline-marker"><i class="bx bx-award"></i></div>
                                        <div class="timeline-content">
                                            <h6>Certificate Issued</h6>
                                            <?php if ($issued_certificate): ?>
                                                <small><?= date('M d, Y', strtotime($issued_certificate['issue_date'])) ?></small>
                                            <?php else: ?>
                                                <small>Pending</small>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                </div>

                                <?php if ($accreditation_request['status'] !== 'certificate_issued'): ?>
                                <div class="current-status-card mb-4">
                                    <div class="card-body">
                                        <h6><i class="bx bx-info-circle"></i> Current Status</h6>
                                        <p><?= getStatusDescription($accreditation_request['status']) ?></p>
                                        
                                        <?php if ($accreditation_request['admin_notes']): ?>
                                        <div class="admin-notes">
                                            <h6><i class="bx bx-message-detail"></i> Admin Notes</h6>
                                            <p class="text-primary"><?= htmlspecialchars($accreditation_request['admin_notes']) ?></p>
                                        </div>
                                        <?php endif; ?>
                                        
                                        <div class="next-steps">
                                            <h6><i class="bx bx-navigation"></i> Next Steps</h6>
                                            <p><?= getNextSteps($accreditation_request['status'], $assessment_attempt) ?></p>
                                        </div>
                                    </div>
                                </div>
                                <?php endif; ?>

                                <!-- SITA Assessment Interface -->
                                <?php if ($accreditation_request['status'] == 'assessment_created' && $current_assessment): ?>
                                <div class="assessment-interface-card mb-4">
                                    <div class="card-header">
                                        <h5><i class="bx bx-brain"></i> SITA Assessment - <?= htmlspecialchars($current_assessment['title']) ?></h5>
                                    </div>
                                    <div class="card-body">
                                        <div class="assessment-info mb-4">
                                            <div class="row">
                                                <div class="col-md-6">
                                                    <div class="info-item">
                                                        <i class="bx bx-time"></i>
                                                        <span><strong>Time Limit:</strong> <?= $current_assessment['time_limit_minutes'] ?> minutes</span>
                                                    </div>
                                                    <div class="info-item">
                                                        <i class="bx bx-target-lock"></i>
                                                        <span><strong>Pass Mark:</strong> <?= $current_assessment['pass_mark_percentage'] ?>%</span>
                                                    </div>
                                                </div>
                                                <div class="col-md-6">
                                                    <div class="info-item">
                                                        <i class="bx bx-refresh"></i>
                                                        <span><strong>Attempts Allowed:</strong> <?= $current_assessment['max_attempts'] ?></span>
                                                    </div>
                                                    <?php if ($current_assessment['deadline']): ?>
                                                    <div class="info-item">
                                                        <i class="bx bx-calendar"></i>
                                                        <span><strong>Deadline:</strong> <?= date('M d, Y', strtotime($current_assessment['deadline'])) ?></span>
                                                    </div>
                                                    <?php endif; ?>
                                                </div>
                                            </div>
                                        </div>
                                        
                                        <?php if ($current_assessment['instructions']): ?>
                                        <div class="assessment-instructions mb-4">
                                            <h6><i class="bx bx-info-circle"></i> Instructions</h6>
                                            <div class="instructions-content">
                                                <?= nl2br(htmlspecialchars($current_assessment['instructions'])) ?>
                                            </div>
                                        </div>
                                        <?php endif; ?>
                                        
                                        <?php if ($assessment_attempt && $assessment_attempt['status'] == 'completed'): ?>
                                        <div class="assessment-result mb-4">
                                            <div class="result-card <?= $assessment_attempt['passed'] ? 'success' : 'warning' ?>">
                                                <div class="result-header">
                                                    <i class="bx <?= $assessment_attempt['passed'] ? 'bx-check-circle' : 'bx-x-circle' ?>"></i>
                                                    <h6><?= $assessment_attempt['passed'] ? 'Assessment Passed!' : 'Assessment Not Passed' ?></h6>
                                                </div>
                                                <div class="result-details">
                                                    <div class="score-display">
                                                        <span class="score-number"><?= $assessment_attempt['percentage_score'] ?>%</span>
                                                        <span class="score-total">Score: <?= $assessment_attempt['total_score'] ?> points</span>
                                                    </div>
                                                    <div class="attempt-info">
                                                        <small>Completed: <?= date('M d, Y \a\t H:i', strtotime($assessment_attempt['end_time'])) ?></small>
                                                        <small>Time Taken: <?= $assessment_attempt['time_taken_minutes'] ?> minutes</small>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                        <?php endif; ?>
                                        
                                        <div class="assessment-actions">
                                            <?php if (!$assessment_attempt): ?>
                                            <a href="take-sita-assessment.php?id=<?= $current_assessment['assessment_id'] ?>" class="btn btn-primary btn-lg">
                                                <i class="bx bx-play-circle"></i> Start SITA Assessment
                                            </a>
                                            <?php elseif ($assessment_attempt['status'] == 'in_progress'): ?>
                                            <a href="take-sita-assessment.php?id=<?= $current_assessment['assessment_id'] ?>&resume=<?= $assessment_attempt['attempt_id'] ?>" class="btn btn-warning btn-lg">
                                                <i class="bx bx-play"></i> Resume Assessment
                                            </a>
                                            <?php elseif (!$assessment_attempt['passed'] && $assessment_attempt['attempt_number'] < $current_assessment['max_attempts']): ?>
                                            <a href="take-sita-assessment.php?id=<?= $current_assessment['assessment_id'] ?>" class="btn btn-secondary btn-lg">
                                                <i class="bx bx-refresh"></i> Retake Assessment (Attempt <?= $assessment_attempt['attempt_number'] + 1 ?>)
                                            </a>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                </div>
                                <?php endif; ?>

                                <!-- Document Upload Interface -->
                                <?php if ($accreditation_request['status'] == 'documents_pending' || $accreditation_request['status'] == 'documents_submitted'): ?>
                                <div class="document-upload-card mb-4">
                                    <div class="card-header">
                                        <h5><i class="bx bx-folder-plus"></i> Required Document Submission</h5>
                                    </div>
                                    <div class="card-body">
                                        <div class="document-requirements mb-4">
                                            <h6><i class="bx bx-list-check"></i> Required Documents</h6>
                                            <div class="requirements-grid">
                                                <?php 
                                                $required_docs = [
                                                    'cv' => ['name' => 'Curriculum Vitae (CV)', 'icon' => 'bx-user', 'formats' => 'PDF, DOC, DOCX'],
                                                    'qualifications' => ['name' => 'Educational Qualifications', 'icon' => 'bx-certification', 'formats' => 'PDF, JPG, PNG'],
                                                    'id_document' => ['name' => 'Government ID Document', 'icon' => 'bx-id-card', 'formats' => 'PDF, JPG, PNG'],
                                                    'assignment' => ['name' => 'Course Assignment Work', 'icon' => 'bx-file', 'formats' => 'PDF, DOC, DOCX, ZIP']
                                                ];
                                                
                                                foreach ($required_docs as $doc_type => $doc_info):
                                                    $uploaded = array_filter($submitted_documents, function($doc) use ($doc_type) {
                                                        return $doc['document_type'] == $doc_type;
                                                    });
                                                    $has_upload = !empty($uploaded);
                                                ?>
                                                <div class="requirement-card <?= $has_upload ? 'completed' : 'pending' ?>">
                                                    <div class="requirement-header">
                                                        <i class="bx <?= $doc_info['icon'] ?> <?= $has_upload ? 'text-success' : 'text-muted' ?>"></i>
                                                        <h6><?= $doc_info['name'] ?></h6>
                                                        <?php if ($has_upload): ?>
                                                        <span class="badge bg-success">âœ“ Uploaded</span>
                                                        <?php endif; ?>
                                                    </div>
                                                    <p class="requirement-formats">Accepted: <?= $doc_info['formats'] ?></p>
                                                    <?php if ($has_upload): ?>
                                                    <div class="uploaded-files">
                                                        <?php foreach ($uploaded as $doc): ?>
                                                        <div class="uploaded-file">
                                                            <i class="bx bx-file"></i>
                                                            <span><?= htmlspecialchars($doc['document_name']) ?></span>
                                                            <small class="verification-status <?= $doc['verification_status'] ?>">
                                                                <?= ucfirst($doc['verification_status']) ?>
                                                            </small>
                                                        </div>
                                                        <?php endforeach; ?>
                                                    </div>
                                                    <?php endif; ?>
                                                </div>
                                                <?php endforeach; ?>
                                            </div>
                                        </div>
                                        
                                        <?php if ($accreditation_request['status'] == 'documents_pending'): ?>
                                        <form id="documentUploadForm" enctype="multipart/form-data">
                                            <input type="hidden" name="request_id" value="<?= $accreditation_request['request_id'] ?>">
                                            
                                            <div class="mb-3">
                                                <label for="documentType" class="form-label">Document Type</label>
                                                <select class="form-select" id="documentType" name="document_type" required>
                                                    <option value="">Select document type...</option>
                                                    <option value="cv">Curriculum Vitae (CV)</option>
                                                    <option value="qualifications">Educational Qualifications</option>
                                                    <option value="id_document">Government ID Document</option>
                                                    <option value="assignment">Course Assignment Work</option>
                                                </select>
                                            </div>
                                            
                                            <div class="mb-3">
                                                <label for="documentFiles" class="form-label">Select Files</label>
                                                <input type="file" class="form-control" id="documentFiles" name="documents[]" multiple accept=".pdf,.doc,.docx,.jpg,.jpeg,.png,.zip" required>
                                                <div class="form-text">
                                                    <i class="bx bx-info-circle"></i> 
                                                    You can upload multiple files. Maximum 50MB per file.
                                                </div>
                                            </div>
                                            
                                            <div class="mb-3">
                                                <label for="documentNotes" class="form-label">Additional Notes <small class="text-muted">(Optional)</small></label>
                                                <textarea class="form-control" id="documentNotes" name="notes" rows="3" placeholder="Any additional information about these documents..."></textarea>
                                            </div>
                                            
                                            <div class="d-grid">
                                                <button type="submit" class="btn btn-primary">
                                                    <i class="bx bx-upload"></i> Upload Documents
                                                </button>
                                            </div>
                                        </form>
                                        <?php else: ?>
                                        <div class="submission-complete">
                                            <div class="alert alert-info">
                                                <i class="bx bx-check-circle"></i>
                                                <strong>Documents Submitted Successfully!</strong>
                                                Your documents are currently under admin review. You will be notified of the outcome.
                                            </div>
                                        </div>
                                        <?php endif; ?>
                                    </div>
                                </div>
                                <?php endif; ?>

                                <!-- Certificate Download Interface -->
                                <?php if ($accreditation_request['status'] == 'certificate_issued' && $issued_certificate): ?>
                                <div class="certificate-issued-card mb-4">
                                    <div class="card-header">
                                        <h5><i class="bx bx-award"></i> Your SITA-Accredited Certificate</h5>
                                    </div>
                                    <div class="card-body">
                                        <div class="certificate-preview mb-4">
                                            <div class="text-center py-4">
                                                <i class="bx bx-award text-warning" style="font-size: 4rem;"></i>
                                                <h4 class="text-success">ðŸŽ‰ Congratulations!</h4>
                                                <p class="lead">Your SITA-accredited certificate has been successfully issued.</p>
                                                

                                            </div>
                                        </div>
                                        
                                        <div class="certificate-actions">
                                            <div class="d-flex justify-content-center">
                                                <a href="download-certificate.php?id=<?= $issued_certificate['certificate_id'] ?>" class="btn btn-primary btn-lg">
                                                    <i class="bx bx-download"></i> Download PDF
                                                </a>
                                            </div>
                                        </div>
                                        

                                    </div>
                                </div>
                                <?php endif; ?>
                                
                                <!-- Rejection Notice -->
                                <?php if ($accreditation_request['status'] == 'rejected'): ?>
                                <div class="rejection-notice-card mb-4">
                                    <div class="card-header bg-danger text-white">
                                        <h5><i class="bx bx-x-circle"></i> Request Rejected</h5>
                                    </div>
                                    <div class="card-body">
                                        <div class="alert alert-danger">
                                            <h6>Unfortunately, your accreditation request has been rejected.</h6>
                                            <?php if ($accreditation_request['rejection_reason']): ?>
                                            <p><strong>Reason:</strong> <?= htmlspecialchars($accreditation_request['rejection_reason']) ?></p>
                                            <?php endif; ?>
                                        </div>
                                        
                                        <div class="support-contact">
                                            <h6><i class="bx bx-support"></i> Need Help?</h6>
                                            <p>If you believe this rejection was made in error or would like to reapply, please contact our support team:</p>
                                            <a href="mailto:accreditation@panaditeacademy.com" class="btn btn-outline-primary">
                                                <i class="bx bx-envelope"></i> Contact Support
                                            </a>
                                        </div>
                                    </div>
                                </div>
                                <?php endif; ?>
                            </div>
                            <?php endif; ?>
                        </div>
                    </div>

                    <!-- Notes Tab -->
                    <div class="tab-pane" id="notes">
                        <div class="notes-content">
                            <!-- Notes Header -->
                            <div class="notes-header">
                                <div class="notes-title">
                                    <i class="bx bx-note"></i>
                                    My Notes
                                </div>
                                <button class="btn btn-primary btn-sm" onclick="focusNoteInput()" title="Add new note at current time">
                                    <i class="bx bx-plus"></i> Add Note
                                </button>
                            </div>
                            
                            <!-- Notes Input Area -->
                            <div class="notes-input-area" style="margin-bottom: 2rem;">
                                <div class="input-group">
                                    <textarea class="form-control note-input" 
                                              placeholder="Add a note at the current video time..." 
                                              rows="3"
                                              style="resize: vertical; min-height: 80px;"></textarea>
                                    <button class="btn btn-primary save-note-btn" type="button">
                                        <i class="bx bx-save"></i> Save Note
                                    </button>
                                </div>
                                <small class="text-muted">
                                    <i class="bx bx-info-circle"></i> 
                                    Notes will be saved with the current video timestamp
                                </small>
                            </div>
                            
                            <!-- Notes List -->
                            <div class="notes-list" id="notesList">
                                <div class="no-notes text-center py-4">
                                    <i class="bx bx-note text-muted" style="font-size: 2rem;"></i>
                                    <p class="text-muted mt-2 mb-0">No notes yet. Add your first note while watching!</p>
                                </div>
                            </div>
                            
                            <!-- Notes Actions -->
                            <div class="notes-actions">
                                <button class="btn btn-outline-primary btn-sm" onclick="exportNotes()" title="Export all notes">
                                    <i class="bx bx-export"></i> Export Notes
                                </button>
                                <button class="btn btn-outline-secondary btn-sm" onclick="printNotes()" title="Print notes">
                                    <i class="bx bx-printer"></i> Print Notes
                                </button>
                            </div>
                        </div>
                    </div>

                    <!-- Learning Tools Tab -->
                    <div class="tab-pane" id="tools">
                        <div class="learning-tools-content">
                            <!-- Smart Study Productivity Hub -->
                            <div class="productivity-header mb-4">
                                <div class="row align-items-center">
                                    <div class="col-md-8">
                                        <h3 class="mb-2">
                                            <i class="bx bx-brain text-primary me-2"></i>
                                            Smart Study Hub
                                        </h3>
                                        <p class="text-muted mb-0">Powerful tools to enhance your learning experience and productivity</p>
                                    </div>
                                    <div class="col-md-4 text-md-end">
                                        <div class="productivity-stats">
                                            <div class="stat-badge">
                                                <i class="bx bx-time-five text-success"></i>
                                                <span class="ms-1">Study Smart</span>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <!-- Two Main Tools Row -->
                            <div class="row g-4">
                                <!-- Smart Study Calendar -->
                                <div class="col-lg-6">
                                    <div class="productivity-card h-100">
                                        <div class="card-header">
                                            <div class="d-flex align-items-center justify-content-between">
                                                <div>
                                                    <h5 class="mb-1">
                                                        <i class="bx bx-calendar-event text-primary me-2"></i>
                                                        Smart Study Calendar
                                                    </h5>
                                                    <p class="text-muted small mb-0">Set lesson reminders with email notifications</p>
                                                </div>
                                                <div class="calendar-actions">
                                                    <button class="btn btn-sm btn-primary" onclick="createQuickReminder()">
                                                        <i class="bx bx-plus"></i> Quick Reminder
                                                    </button>
                                                </div>
                                            </div>
                                        </div>
                                        <div class="card-body p-0">
                                            <!-- Calendar Navigation -->
                                            <div class="calendar-nav p-3 border-bottom">
                                                <div class="d-flex align-items-center justify-content-between">
                                                    <button class="btn btn-outline-primary btn-sm" onclick="previousMonth()">
                                                        <i class="bx bx-chevron-left"></i>
                                                    </button>
                                                    <h6 class="mb-0 fw-bold" id="currentMonthYear">November 2025</h6>
                                                    <button class="btn btn-outline-primary btn-sm" onclick="nextMonth()">
                                                        <i class="bx bx-chevron-right"></i>
                                                    </button>
                                                </div>
                                            </div>
                                            
                                            <!-- Calendar Grid -->
                                            <div class="calendar-container p-3">
                                                <div class="calendar-grid" id="calendarGrid">
                                                    <!-- Calendar will be rendered here by JavaScript -->
                                                </div>
                                            </div>
                                            
                                            <!-- Upcoming Reminders -->
                                            <div class="upcoming-reminders p-3 border-top bg-light">
                                                <h6 class="mb-2">
                                                    <i class="bx bx-bell text-warning me-1"></i>
                                                    Upcoming Reminders
                                                </h6>
                                                <div id="upcomingRemindersList">
                                                    <div class="text-muted small text-center py-2">
                                                        <i class="bx bx-calendar-plus me-1"></i>
                                                        No reminders set yet
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>

                                <!-- Picture-in-Picture Video Player -->
                                <div class="col-lg-6">
                                    <div class="productivity-card h-100">
                                        <div class="card-header">
                                            <div class="d-flex align-items-center justify-content-between">
                                                <div>
                                                    <h5 class="mb-1">
                                                        <i class="bx bx-video text-success me-2"></i>
                                                        Picture-in-Picture Player
                                                    </h5>
                                                    <p class="text-muted small mb-0">Continue watching while browsing</p>
                                                </div>
                                                <div class="pip-status">
                                                    <span class="badge bg-secondary" id="pipStatus">Ready</span>
                                                </div>
                                            </div>
                                        </div>
                                        <div class="card-body">
                                            <!-- PiP Controls -->
                                            <div class="pip-controls mb-4">
                                                <div class="row g-3">
                                                    <div class="col-6">
                                                        <button class="btn btn-success w-100" onclick="enablePictureInPicture()" id="pipEnableBtn">
                                                            <i class="bx bx-dock-top me-2"></i>
                                                            Enable PiP Mode
                                                        </button>
                                                    </div>
                                                    <div class="col-6">
                                                        <button class="btn btn-outline-secondary w-100" onclick="disablePictureInPicture()" id="pipDisableBtn" disabled>
                                                            <i class="bx bx-dock-bottom me-2"></i>
                                                            Exit PiP Mode
                                                        </button>
                                                    </div>
                                                </div>
                                            </div>
                                            
                                            <!-- Current Video Info -->
                                            <div class="current-video-info">
                                                <div class="video-thumbnail-container mb-3">
                                                    <div class="mini-video-preview" id="miniVideoPreview">
                                                        <?php if ($current_lesson): ?>
                                                            <img src="<?= !empty($current_lesson['thumbnail']) ? htmlspecialchars($current_lesson['thumbnail']) : '/assets/images/video-placeholder.jpg' ?>" 
                                                                 alt="Current Video" class="img-fluid rounded">
                                                            <div class="play-overlay">
                                                                <i class="bx bx-play-circle"></i>
                                                            </div>
                                                        <?php else: ?>
                                                            <div class="no-video-placeholder">
                                                                <i class="bx bx-video-off text-muted"></i>
                                                                <p class="text-muted mb-0">No video selected</p>
                                                            </div>
                                                        <?php endif; ?>
                                                    </div>
                                                </div>
                                                
                                                <div class="video-details">
                                                    <h6 class="mb-2" id="currentVideoTitle">
                                                        <?= $current_lesson ? htmlspecialchars($current_lesson['title']) : 'No video selected' ?>
                                                    </h6>
                                                    <div class="video-meta text-muted small">
                                                        <span><i class="bx bx-time me-1"></i>Duration: 
                                                            <span id="videoDuration"><?= $current_lesson && !empty($current_lesson['duration']) ? htmlspecialchars($current_lesson['duration']) : 'N/A' ?></span>
                                                        </span>
                                                    </div>
                                                </div>
                                            </div>
                                            
                                            <!-- PiP Features -->
                                            <div class="pip-features mt-4">
                                                <h6 class="mb-3">Features:</h6>
                                                <div class="feature-list">
                                                    <div class="feature-item">
                                                        <i class="bx bx-check-circle text-success me-2"></i>
                                                        <span>Watch while browsing other tabs</span>
                                                    </div>
                                                    <div class="feature-item">
                                                        <i class="bx bx-check-circle text-success me-2"></i>
                                                        <span>Resizable floating window</span>
                                                    </div>
                                                    <div class="feature-item">
                                                        <i class="bx bx-check-circle text-success me-2"></i>
                                                        <span>Always on top display</span>
                                                    </div>
                                                    <div class="feature-item">
                                                        <i class="bx bx-check-circle text-success me-2"></i>
                                                        <span>Full playback controls</span>
                                                    </div>
                                                </div>
                                            </div>
                                            
                                            <!-- Browser Support Info -->
                                            <div class="browser-support mt-4">
                                                <div class="alert alert-info small mb-0">
                                                    <i class="bx bx-info-circle me-1"></i>
                                                    <strong>Browser Support:</strong> Works best in Chrome, Firefox, Safari, and Edge. 
                                                    Make sure your browser allows Picture-in-Picture mode.
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>



                            <!-- Keyboard Shortcuts -->
                            <div class="shortcuts-section mt-4">
                                <h5><i class="bx bx-keyboard"></i> Enhanced Keyboard Shortcuts</h5>
                                <div class="shortcuts-grid">
                                    <div class="shortcut-item">
                                        <span>Play/Pause</span>
                                        <span class="shortcut-key">Space</span>
                                    </div>
                                    <div class="shortcut-item">
                                        <span>Skip Forward</span>
                                        <span class="shortcut-key">â†’</span>
                                    </div>
                                    <div class="shortcut-item">
                                        <span>Skip Backward</span>
                                        <span class="shortcut-key">â†</span>
                                    </div>
                                    <div class="shortcut-item">
                                        <span>Add Note</span>
                                        <span class="shortcut-key">N</span>
                                    </div>
                                    <div class="shortcut-item">
                                        <span>Fullscreen</span>
                                        <span class="shortcut-key">F</span>
                                    </div>
                                    <div class="shortcut-item">
                                        <span>Volume Up/Down</span>
                                        <span class="shortcut-key">â†‘ / â†“</span>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Custom Note Modal -->
            <div id="customNoteModal" class="custom-modal" style="display: none;">
                <div class="custom-modal-overlay" onclick="closeNoteModal()"></div>
                <div class="custom-modal-container">
                    <div class="custom-modal-header">
                        <div class="custom-modal-title">
                            <i class="bx bx-note"></i>
                            <span id="modalTitleText">Add Note at</span>
                            <span id="modalTimestamp">0:00</span>
                        </div>
                        <button class="custom-modal-close" onclick="closeNoteModal()">
                            <i class="bx bx-x"></i>
                        </button>
                    </div>
                    <div class="custom-modal-body">
                        <div class="note-input-section">
                            <label for="customNoteTextarea">Your note:</label>
                            <textarea id="customNoteTextarea" 
                                placeholder="Add your thoughts, questions, or important points about this moment in the video..."></textarea>
                        </div>
                        <div class="note-tips-section">
                            <i class="bx bx-lightbulb"></i>
                            <span><strong>Tips:</strong> Your notes will be saved with the timestamp so you can easily jump back to this moment later.</span>
                        </div>
                    </div>
                    <div class="custom-modal-footer">
                        <button class="custom-btn custom-btn-secondary" onclick="closeNoteModal()">
                            <i class="bx bx-x"></i>
                            Cancel
                        </button>
                        <button class="custom-btn custom-btn-primary" id="customSaveNoteBtn">
                            <i class="bx bx-check"></i>
                            <span id="saveButtonText">Save Note</span>
                        </button>
                    </div>
                </div>
            </div>



          

            <!-- Dynamic Question Section -->
            <div class="question-section" id="questionSection">
                <div class="question-header">
                    <div class="question-icon">
                        <i class="bx bx-help-circle"></i>
                    </div>
                    <div class="question-text">
                        <h4>Need Help with This Lesson?</h4>
                        <p>Ask your facilitator any questions about the content you just watched</p>
                    </div>
                </div>
                <form class="question-form" onsubmit="submitQuestion(event)">
                    <textarea class="question-input" 
                              placeholder="Type your question here... What didn't you understand about this lesson?"
                              maxlength="500" required></textarea>
                    <button type="submit" class="question-submit">
                        <i class="bx bx-send"></i>
                        Send Question to Facilitator
                    </button>
                </form>
            </div>
        </div>
<!-- Current Lesson-->
                </div>
        </div>
    </div>

<!-- ✅ EXTRACTED JAVASCRIPT BLOCK 1 - Now External (CLEAN VERSION) -->
<script src="js/course-detail-main-clean.js"></script>

<!-- ===== BOOTSTRAP JAVASCRIPT LIBRARIES - CRITICAL FOR ACCORDION FUNCTIONALITY ===== -->
<!-- jQuery (required for Bootstrap 4) -->
<script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>

<!-- Bootstrap 4 JavaScript Bundle (includes Popper.js) -->
<script src="https://cdn.jsdelivr.net/npm/bootstrap@4.6.2/dist/js/bootstrap.bundle.min.js"></script>













<script src="js/course-detail-youtube.js"></script>
<script src="js/accreditation-payfast.js"></script>

<!-- 🎯 PANADITE ACADEMY - ISOLATED PROGRESS TRACKING SYSTEM v2.0 -->
<!-- Clean, modular progress tracking that prevents file bloat -->
<script>
    // Pass PHP data to JavaScript for the isolated progress tracking system
    window.currentVideoId = <?= json_encode($current_lesson['video_id'] ?? null) ?>;
    window.currentLessonId = <?= json_encode($current_lesson_id ?? null) ?>;
    window.currentCourseId = <?= json_encode($course_id ?? null) ?>;
    window.currentStudentId = <?= json_encode($_SESSION['user_id'] ?? $_SESSION['student_id'] ?? 15) ?>; // Use session or fallback ID
    window.progressDebug = true; // Enable debug mode
</script>

<!-- ✅ NATURAL ACCORDION LAYOUT - NO ARTIFICIAL CONSTRAINTS -->
<style>
/* Natural accordion layout - let Bootstrap handle positioning */
.accordion-button {
    display: flex !important;
    align-items: center !important;
    justify-content: space-between !important;
    padding: 1rem !important;
    width: 100% !important;
}

/* Keep arrow in natural Bootstrap position but ensure it doesn't overlap */
.accordion-button::after {
    flex-shrink: 0 !important;
    margin-left: 0.5rem !important; /* Reduced from 1rem - arrows closer to content */
    position: static !important; /* Let it flow naturally */
}

/* Allow content to use available space naturally */
.accordion-button .d-flex {
    flex: 1 !important; /* Take available space */
    min-width: 0 !important; /* Allow shrinking if needed */
    margin-right: 1rem !important; /* Space before arrow */
}

/* Let titles use available space with smart truncation */
.accordion-title-text {
    overflow: hidden !important;
    text-overflow: ellipsis !important;
    white-space: nowrap !important;
    line-height: 1.4 !important;
    flex: 1 !important; /* Use available space */
}

/* Ensure flex container works properly */
.accordion-button .flex-grow-1 {
    min-width: 0 !important; /* Essential for flex text truncation */
    flex: 1 !important;
}

/* Mobile adjustments */
@media (max-width: 768px) {
    .accordion-button {
        padding: 0.75rem !important;
    }
    
    .accordion-button::after {
        margin-left: 0.75rem !important;
    }
    
    .accordion-button .d-flex {
        margin-right: 0.75rem !important;
    }
}
</style>

<!-- Load isolated progress tracking components -->
<!-- ✅ ISOLATED PROGRESS TRACKING SYSTEM - TEMPORARILY DISABLED TO PREVENT CONFLICTS -->
<!-- <script src="js/video-progress-tracker.js"></script> -->
<!-- <script src="js/progress-ui-integration.js"></script> -->
<!-- <script src="js/progress-integration-init.js"></script> -->
<!-- ✅ YOUTUBE IFRAME API INTEGRATION - Automatic progress tracking -->
<!-- TEMPORARILY DISABLED: Syntax error preventing video controls -->
<!-- <script src="js/youtube-progress-integration.js"></script> -->

<!-- Note: video-progress-handler.php is a PHP AJAX endpoint, NOT a JavaScript include -->

<!-- âœ… Clean Isolated Progress Tracking System - No more expansion needed -->
<script>
// ðŸŽ¯ CLEAN ISOLATED PROGRESS TRACKING - Initialize the new system
document.addEventListener('DOMContentLoaded', function() {
    console.log('âœ… Isolated Progress Tracking System Loading...');
    
    // Pass PHP data to isolated system in expected format
    window.currentVideoId = <?= json_encode($current_lesson['video_id'] ?? null) ?>;
    window.currentLessonId = <?= json_encode($current_lesson_id) ?>;
    window.currentCourseId = <?= json_encode($course_id) ?>;
    window.currentStudentId = <?= json_encode($_SESSION['user_id'] ?? $_SESSION['student_id'] ?? null) ?>;
    window.progressDebug = true; // Enable debug mode for testing
    
    // Additional data for external videos
    window.currentVideoUrl = <?= json_encode($current_lesson['video_url'] ?? '') ?>;
    window.isExternalVideo = <?= json_encode(!empty($current_lesson['is_external'])) ?>;
    
    console.log('ðŸ“‹ Data passed to isolated system:', {
        videoId: window.currentVideoId,
        lessonId: window.currentLessonId,
        courseId: window.currentCourseId,
        studentId: window.currentStudentId,
        videoUrl: window.currentVideoUrl,
        isExternal: window.isExternalVideo
    });
    
    // ðŸ” CRITICAL DEBUG: Check if all required data is present
    console.log('ðŸ” VALIDATION CHECK:', {
        'videoId exists': !!window.currentVideoId,
        'courseId exists': !!window.currentCourseId,
        'studentId exists': !!window.currentStudentId,
        'videoId value': window.currentVideoId,
        'courseId value': window.currentCourseId,  
        'studentId value': window.currentStudentId
    });
    
    // ðŸŽ¬ INITIALIZE YOUTUBE PROGRESS TRACKING
    if (window.currentVideoUrl && window.initializeYouTubeProgressTracking) {
        console.log('ðŸŽ¬ YOUTUBE: Calling initializeYouTubeProgressTracking...');
        console.log('ðŸŽ¬ YOUTUBE: Video URL:', window.currentVideoUrl);
        initializeYouTubeProgressTracking(
            window.currentVideoUrl,
            window.currentVideoId,
            window.currentLessonId,
            window.currentCourseId
        );
    } else {
        console.log('ðŸŽ¬ YOUTUBE: Skipping - No video URL or function not loaded');
        console.log('ðŸŽ¬ YOUTUBE: Video URL available:', !!window.currentVideoUrl);
        console.log('ðŸŽ¬ YOUTUBE: Function available:', !!window.initializeYouTubeProgressTracking);
    }
    
    // ðŸŽ¬ FILE VIDEO PROGRESS TRACKING INITIALIZATION - MISSING CONNECTION RESTORED!
    if (!window.isExternalVideo && window.VideoProgressTracker) {
        console.log('ðŸŽ¬ FILE VIDEO: Initializing VideoProgressTracker for local file...');
        console.log('ðŸŽ¬ FILE VIDEO: Data for initialization:', {
            videoId: window.currentVideoId,
            lessonId: window.currentLessonId, 
            courseId: window.currentCourseId,
            studentId: window.currentStudentId
        });
        
        try {
            // Initialize VideoProgressTracker for file videos
            window.fileVideoProgressTracker = new VideoProgressTracker(
                window.currentLessonId,  // lessonId as first parameter
                window.currentCourseId,  // courseId 
                window.currentStudentId, // studentId
                {
                    debug: true, // Enable debug mode
                    updateFrequency: 2000, // Update every 2 seconds
                    completionThreshold: 90, // 90% = completed
                    
                    // Progress update callback
                    onProgressUpdate: function(progressData) {
                        console.log('ðŸ“‹ FILE VIDEO Progress Update:', progressData);
                        
                        // 🛡️ ONLY SET WATCHING BADGE IF NOT COMPLETED (< 90%)
                        if (window.setBadge && progressData.progress > 10 && progressData.progress < 90) {
                            setBadge(window.currentLessonId, 'watching');
                            console.log('📅 FILE VIDEO: Set watching badge for lesson', window.currentLessonId);
                        } else if (progressData.progress >= 90) {
                            console.log('🛡️ FILE VIDEO: Skipping watching badge - video completed or near completion');
                        }
                    },
                    
                    // Video completion callback  
                    onCompletion: function(completionData) {
                        console.log('🎉 FILE VIDEO Completion:', completionData);
                        
                        // Update accordion badges for completion
                        if (window.setBadge) {
                            setBadge(window.currentLessonId, 'completed');
                            console.log('📅 FILE VIDEO: Set completed badge for lesson', window.currentLessonId);
                        }
                        
                        // 💾 SAVE COMPLETION TO DATABASE via AJAX
                        console.log('💾 FILE VIDEO: PREPARING DATABASE SAVE:', completionData);
                        const formData = new FormData();
                        formData.append('video_id', window.currentLessonId);
                        formData.append('course_id', window.currentCourseId || completionData.courseId);
                        formData.append('auto_complete', '1');
                        
                        console.log('💾 FILE VIDEO: SENDING AJAX REQUEST to mark_lesson_completed.php');
                        
                        fetch('mark_lesson_completed.php', {
                            method: 'POST',
                            body: formData,
                            credentials: 'include'  // Include session cookies
                        })
                        .then(response => response.json())
                        .then(result => {
                            console.log('💾 FILE VIDEO DATABASE SAVE Result:', result);
                            
                            // If authentication failed, show full debug details
                            if (!result.success && result.debug) {
                                console.log('🔍 DETAILED DEBUG INFO:');
                                console.log('  Session exists:', result.debug.session_exists);
                                console.log('  All session keys:', result.debug.all_session_keys);
                                console.log('  user_id:', result.debug.user_id);
                                console.log('  role:', result.debug.role);
                                console.log('  user_role:', result.debug.user_role);
                                console.log('  user_type:', result.debug.user_type);
                                console.log('  role_check_passed:', result.debug.role_check_passed);
                                console.log('  user_id_check_passed:', result.debug.user_id_check_passed);
                                console.log('  final_authenticated:', result.debug.final_authenticated);
                            }
                        })
                        .catch(error => {
                            console.error('❌ FILE VIDEO DATABASE SAVE Error:', error);
                        });
                        
                        // Show completion notification
                        if (window.Toastify) {
                            Toastify({
                                text: `🎉 Video "${completionData.videoTitle || 'Lesson'}" completed! Great job!`,
                                duration: 5000,
                                style: {
                                    background: "linear-gradient(to right, #00b09b, #96c93d)",
                                }
                            }).showToast();
                        }
                    },
                    
                    // Error handling callback
                    onError: function(error) {
                        console.error('ðŸ…â€â‚ï¸ FILE VIDEO Progress Tracking Error:', error);
                    }
                }
            );
            
            console.log('ðŸ“… FILE VIDEO: VideoProgressTracker initialized successfully!');
            console.log('ðŸŽ¬ FILE VIDEO: Tracking will start automatically when video plays');
            
        } catch (error) {
            console.error('ðŸ…â€â‚ï¸ FILE VIDEO: Failed to initialize VideoProgressTracker:', error);
        }
    } else if (!window.VideoProgressTracker) {
        console.error('ðŸ…â€â‚ï¸ FILE VIDEO: VideoProgressTracker class not found - check if video-progress-tracker.js is loaded');
    } else {
        console.log('ðŸŽ¬ FILE VIDEO: Skipping - External video detected, using dedicated tracking');
    }
});

// ðŸŽ¯ COMPLETE COURSE REMINDER SYSTEM WITH EMAIL NOTIFICATIONS - EXACT SAME APPROACH AS PURCHASE CONFIRMATIONS
// Define calendar variables and functions
if (typeof window.monthNames === 'undefined') {
    window.monthNames = [
        'January', 'February', 'March', 'April', 'May', 'June',
        'July', 'August', 'September', 'October', 'November', 'December'
    ];
}

if (typeof window.dayNames === 'undefined') {
    window.dayNames = ['Sunday', 'Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday'];
}

if (typeof window.currentDate === 'undefined') {
    window.currentDate = new Date();
}

if (typeof window.studyReminders === 'undefined') {
    window.studyReminders = JSON.parse(localStorage.getItem('studyReminders') || '[]');
}

// Calendar rendering function
function renderCalendar() {
    const calendarGrid = document.getElementById('calendarGrid');
    const currentMonthYear = document.getElementById('currentMonthYear');
    
    if (!calendarGrid || !currentMonthYear) {
        console.warn('Calendar elements not found, skipping render');
        return;
    }
    
    const year = window.currentDate.getFullYear();
    const month = window.currentDate.getMonth();
    
    // Update month/year display
    currentMonthYear.textContent = `${window.monthNames[month]} ${year}`;
    
    // Clear calendar grid
    let calendarHTML = '';
    
    // Add day headers
    window.dayNames.forEach(day => {
        calendarHTML += `<div class="calendar-day-header" style="padding: 8px; text-align: center; font-weight: bold; background: #f8f9fa; font-size: 0.7rem;">${day.substr(0, 3)}</div>`;
    });
    
    // Get first day of month and number of days
    const firstDay = new Date(year, month, 1).getDay();
    const daysInMonth = new Date(year, month + 1, 0).getDate();
    
    // Add empty cells for days before month starts
    for (let i = 0; i < firstDay; i++) {
        calendarHTML += '<div class="calendar-day-empty" style="padding: 8px; background: #f9f9f9;"></div>';
    }
    
    // Add days of the month
    for (let day = 1; day <= daysInMonth; day++) {
        const isToday = new Date().toDateString() === new Date(year, month, day).toDateString();
        const hasReminder = window.studyReminders.some(r => {
            const reminderDate = new Date(r.date);
            return reminderDate.getFullYear() === year && 
                   reminderDate.getMonth() === month && 
                   reminderDate.getDate() === day;
        });
        
        calendarHTML += `
            <div class="calendar-day" style="padding: 8px; text-align: center; cursor: pointer; 
                        background: ${isToday ? '#e3f2fd' : 'white'}; 
                        border: ${hasReminder ? '2px solid #2196f3' : '1px solid #e0e0e0'}; font-size: 0.75rem;"
                 onclick="selectCalendarDate('${year}-${String(month + 1).padStart(2, '0')}-${String(day).padStart(2, '0')}')">
                ${day}
                ${hasReminder ? '<div style="width: 6px; height: 6px; background: #2196f3; border-radius: 50%; margin: 2px auto;"></div>' : ''}
            </div>`;
    }
    
    calendarHTML += '</div>';
    calendarGrid.innerHTML = calendarHTML;
    
    console.log(`📅 Calendar rendered for ${window.monthNames[month]} ${year}`);
}

// Calendar navigation functions - FIXING REFERENCE ERRORS
function previousMonth() {
    window.currentDate.setMonth(window.currentDate.getMonth() - 1);
    renderCalendar();
    updateUpcomingReminders();
}

function nextMonth() {
    window.currentDate.setMonth(window.currentDate.getMonth() + 1);
    renderCalendar();
    updateUpcomingReminders();
}

function createQuickReminder() {
    const tomorrow = new Date();
    tomorrow.setDate(tomorrow.getDate() + 1);
    showStudyReminderModal(tomorrow);
}

// Select calendar date function
function selectCalendarDate(dateString) {
    const selectedDate = new Date(dateString + 'T00:00:00');
    showStudyReminderModal(selectedDate);
}

// Study reminder modal function
function showStudyReminderModal(date) {
    const modal = document.createElement('div');
    modal.className = 'modal fade show';
    modal.style.display = 'block';
    modal.style.backgroundColor = 'rgba(0,0,0,0.5)';
    modal.style.zIndex = '9999';
    
    const formattedDate = date.toLocaleDateString('en-US', {
        weekday: 'long',
        year: 'numeric',
        month: 'long',
        day: 'numeric'
    });
    
    modal.innerHTML = `
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header bg-primary text-white">
                    <h5 class="modal-title">
                        <i class="bx bx-calendar-plus me-2"></i>
                        Set Study Reminder
                    </h5>
                    <button type="button" class="btn-close btn-close-white" onclick="closeReminderModal()"></button>
                </div>
                <div class="modal-body">
                    <form id="reminderForm">
                        <div class="mb-3">
                            <label class="form-label">Date</label>
                            <input type="text" class="form-control" value="${formattedDate}" readonly>
                        </div>
                        <div class="row">
                            <div class="col-md-6">
                                <label class="form-label">Reminder Type</label>
                                <select class="form-select" id="reminderType" required>
                                    <option value="">Choose type...</option>
                                    <option value="study_session">Study Session</option>
                                    <option value="assignment_due">Assignment Due</option>
                                    <option value="quiz_prep">Quiz Preparation</option>
                                    <option value="review">Review Session</option>
                                    <option value="next_lesson">Next Lesson</option>
                                </select>
                            </div>
                            <div class="col-md-6">
                                <label class="form-label">Time</label>
                                <input type="time" class="form-control" id="reminderTime" value="09:00" required>
                            </div>
                        </div>
                        <div class="mt-3">
                            <label class="form-label">Description</label>
                            <textarea class="form-control" id="reminderDescription" rows="3" 
                                      placeholder="What would you like to be reminded about?" required></textarea>
                        </div>
                        <div class="mt-3">
                            <div class="form-check">
                                <input class="form-check-input" type="checkbox" id="emailNotification" checked>
                                <label class="form-check-label" for="emailNotification">
                                    📧 Send email notification (recommended)
                                </label>
                            </div>
                        </div>
                        <div class="mt-4 d-flex gap-2">
                            <button type="submit" class="btn btn-primary">
                                <i class="bx bx-save me-1"></i>
                                Save Reminder
                            </button>
                            <button type="button" class="btn btn-outline-secondary" onclick="closeReminderModal()">
                                Cancel
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    `;
    
    document.body.appendChild(modal);
    
    // Set up form submission
    document.getElementById('reminderForm').addEventListener('submit', function(e) {
        e.preventDefault();
        saveStudyReminder(date);
    });
    
    console.log('📝 Reminder modal shown for:', formattedDate);
}

// Close reminder modal
function closeReminderModal() {
    const modal = document.querySelector('.modal.show');
    if (modal) {
        modal.remove();
    }
}

// Save study reminder with EMAIL NOTIFICATIONS - EXACT SAME APPROACH AS PURCHASE CONFIRMATIONS
function saveStudyReminder(date) {
    const type = document.getElementById('reminderType').value;
    const time = document.getElementById('reminderTime').value;
    const description = document.getElementById('reminderDescription').value;
    const emailNotification = document.getElementById('emailNotification').checked;
    
    if (!type || !time || !description) {
        showProductivityNotification('Please fill in all required fields', 'error');
        return;
    }
    
    const reminder = {
        id: Date.now(),
        date: date.toISOString().split('T')[0],
        time: time,
        type: type,
        description: description,
        emailNotification: emailNotification,
        courseId: window.currentCourseId,
        courseName: <?= json_encode($course['title'] ?? 'Course') ?>,
        created: new Date().toISOString()
    };
    
    // Save to localStorage (in production, this would be saved to server)
    window.studyReminders.push(reminder);
    localStorage.setItem('studyReminders', JSON.stringify(window.studyReminders));
    
    // Send email notification if requested - EXACT SAME APPROACH AS PURCHASE CONFIRMATIONS
    if (emailNotification) {
        sendReminderEmail(reminder);
    }
    
    // Update UI
    renderCalendar();
    updateUpcomingReminders();
    closeReminderModal();
    
    showProductivityNotification('✅ Reminder saved successfully!' + (emailNotification ? ' Email notification sent.' : ''), 'success');
    console.log('💾 Study reminder saved:', reminder);
}

// Send reminder email notification - EXACT SAME APPROACH AS ADMIN COURSES (CALL SAME FILE)
function sendReminderEmail(reminder) {
    // Create AJAX request to send reminder email using exact same approach as admin courses
    const formData = new FormData();
    formData.append('action', 'send_reminder_email');
    formData.append('reminder_data', JSON.stringify(reminder));
    formData.append('student_id', window.currentStudentId);
    formData.append('course_id', window.currentCourseId);
    
    fetch(window.location.pathname, {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            console.log('✅ Reminder email sent successfully');
            showProductivityNotification('📧 Reminder email sent successfully!', 'success');
        } else {
            console.log('❌ Failed to send reminder email:', data.error);
            showProductivityNotification('❌ Failed to send reminder email: ' + data.error, 'error');
        }
    })
    .catch(error => {
        console.log('❌ Error sending reminder email:', error);
        showProductivityNotification('❌ Error sending reminder email', 'error');
    });
}

// Update upcoming reminders list
function updateUpcomingReminders() {
    const remindersList = document.getElementById('upcomingRemindersList');
    if (!remindersList) return;
    
    const now = new Date();
    const upcoming = window.studyReminders
        .filter(reminder => new Date(reminder.date + 'T' + reminder.time) > now)
        .sort((a, b) => new Date(a.date + 'T' + a.time) - new Date(b.date + 'T' + b.time))
        .slice(0, 3);
    
    if (upcoming.length === 0) {
        remindersList.innerHTML = `
            <div class="text-muted small text-center py-2">
                <i class="bx bx-calendar-plus me-1"></i>
                No reminders set yet
            </div>
        `;
        return;
    }
    
    remindersList.innerHTML = upcoming.map(reminder => {
        const reminderDate = new Date(reminder.date + 'T' + reminder.time);
        const timeUntil = getTimeUntil(reminderDate);
        
        return `
            <div class="reminder-item" style="padding: 8px; border-bottom: 1px solid #eee; display: flex; justify-content: space-between; align-items: center;">
                <div style="flex: 1;">
                    <div style="font-weight: 500; font-size: 0.9rem;">${reminder.description}</div>
                    <div style="font-size: 0.8rem; color: #666;">
                        ${new Date(reminder.date).toLocaleDateString()} at ${reminder.time} • ${timeUntil}
                    </div>
                </div>
                <button class="btn btn-outline-danger btn-sm" onclick="deleteReminder(${reminder.id})" title="Delete">
                    <i class="bx bx-trash"></i>
                </button>
            </div>
        `;
    }).join('');
}

// Get time until reminder
function getTimeUntil(targetDate) {
    const now = new Date();
    const diff = targetDate - now;
    
    if (diff < 0) return 'Overdue';
    
    const days = Math.floor(diff / (1000 * 60 * 60 * 24));
    const hours = Math.floor((diff % (1000 * 60 * 60 * 24)) / (1000 * 60 * 60));
    
    if (days > 0) return `in ${days} day${days > 1 ? 's' : ''}`;
    if (hours > 0) return `in ${hours} hour${hours > 1 ? 's' : ''}`;
    return 'Soon';
}

// Delete reminder
function deleteReminder(reminderId) {
    window.studyReminders = window.studyReminders.filter(r => r.id !== reminderId);
    localStorage.setItem('studyReminders', JSON.stringify(window.studyReminders));
    renderCalendar();
    updateUpcomingReminders();
    showProductivityNotification('✅ Reminder deleted successfully', 'success');
}

// Show productivity notification
function showProductivityNotification(message, type) {
    const notification = document.createElement('div');
    notification.className = `alert alert-${type === 'success' ? 'success' : type === 'error' ? 'danger' : 'info'} alert-dismissible fade show`;
    notification.style.position = 'fixed';
    notification.style.top = '20px';
    notification.style.right = '20px';
    notification.style.zIndex = '10000';
    notification.style.maxWidth = '400px';
    notification.innerHTML = `
        ${message}
        <button type="button" class="btn-close" onclick="this.parentElement.remove()"></button>
    `;
    
    document.body.appendChild(notification);
    
    // Auto-remove after 5 seconds
    setTimeout(() => {
        if (notification.parentElement) {
            notification.remove();
        }
    }, 5000);
}

// Initialize calendar when DOM is loaded
document.addEventListener('DOMContentLoaded', function() {
    // Initialize calendar if elements exist
    if (document.getElementById('calendarGrid')) {
        renderCalendar();
        updateUpcomingReminders();
        
        console.log(' Course Reminder System initialized');
    }
});

console.log(' Course Reminder System functions loaded');

// ==========================================
// MISSING VIDEO CONTROL FUNCTIONS - RESTORED
// ==========================================
// These functions were lost during refactoring and are critical for file upload video playback

var videoCurrentTime = 0;
var videoDuration = 0;
var videoIsPlaying = false;
var videoElement = null;

// Initialize video element reference
function initializeVideoControls() {
    videoElement = document.getElementById('lessonVideo');
    if (videoElement) {
        console.log(' Video element found and initialized for controls');
        
        // Update duration when metadata loads
        videoElement.addEventListener('loadedmetadata', function() {
            videoDuration = videoElement.duration;
            console.log(' Video duration loaded:', videoDuration);
        });
        
        // Update playing state
        videoElement.addEventListener('play', function() {
            videoIsPlaying = true;
            updatePlayPauseButton();
        });
        
        videoElement.addEventListener('pause', function() {
            videoIsPlaying = false;
            updatePlayPauseButton();
        });
        
        videoElement.addEventListener('ended', function() {
            videoIsPlaying = false;
            updatePlayPauseButton();
            
            // 🎯 CALL BACKEND TO MARK VIDEO AS COMPLETED
            markVideoAsCompleted();
        });
    } else {
        console.error(' Video element not found for controls');
    }
}

// Update play/pause button icon
function updatePlayPauseButton() {
    const playPauseBtn = document.getElementById('playPauseBtn');
    if (playPauseBtn) {
        const icon = playPauseBtn.querySelector('i');
        if (icon) {
            icon.className = videoIsPlaying ? 'bx bx-pause' : 'bx bx-play';
        }
    }
}

// 🎯 MARK VIDEO AS COMPLETED - CALLS WORKING BACKEND
async function markVideoAsCompleted() {
    console.log('🎯 Video ended - marking as completed');
    
    const lessonId = <?= json_encode($current_lesson_id) ?>;
    const courseId = <?= json_encode($course_id) ?>;
    
    if (!lessonId || !courseId) {
        console.error('❌ Missing lesson ID or course ID');
        return;
    }
    
    try {
        console.log('📡 Calling backend with lessonId:', lessonId, 'courseId:', courseId);
        
        // Call our working backend and WAIT for completion
        const response = await fetch('mark_lesson_completed.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded',
            },
            body: 'video_id=' + lessonId + '&course_id=' + courseId + '&auto_complete=1'
        });
        
        const data = await response.json();
        console.log('✅ Backend response received:', data);
        
        if (data.success) {
            console.log('✅ Backend confirmed save - now updating UI');
            
            // WAIT a moment to ensure database transaction completes
            setTimeout(() => {
                updateLessonCompletionUI();
                console.log('✅ Video completion process fully complete!');
            }, 500);
            
        } else {
            console.error('❌ Backend error:', data.message);
        }
        
    } catch (error) {
        console.error('❌ Network error marking video complete:', error);
    }
}

// 🎯 UPDATE UI TO SHOW COMPLETION - NO RELOAD!
function updateLessonCompletionUI() {
    console.log('🎯 Updating lesson UI dynamically (no reload)');
    
    // Get current lesson ID
    const currentLessonId = <?= json_encode($current_lesson_id) ?>;
    
    // Find and update the lesson accordion item
    const lessonElements = document.querySelectorAll('.lesson-item, .accordion-item');
    lessonElements.forEach(element => {
        // Check if this is the current lesson
        const videoLink = element.querySelector('a[href*="lesson_id=' + currentLessonId + '"]') || 
                         element.querySelector('[data-video-id="' + currentLessonId + '"]');
        
        if (videoLink) {
            console.log('✅ Found lesson element to update');
            
            // Add completion badge
            let badge = element.querySelector('.completion-badge');
            if (!badge) {
                badge = document.createElement('span');
                badge.className = 'completion-badge badge bg-success ms-2';
                badge.innerHTML = '✅ Completed';
                
                // Insert badge after lesson title
                const titleElement = element.querySelector('.lesson-title, .accordion-button, h6, strong');
                if (titleElement) {
                    titleElement.appendChild(badge);
                }
            }
            
            // Update any progress indicators
            const progressElement = element.querySelector('.progress, .progress-bar');
            if (progressElement) {
                progressElement.style.width = '100%';
                progressElement.className += ' bg-success';
            }
            
            // Add completed class
            element.classList.add('lesson-completed');
            
            console.log('✅ Lesson UI updated successfully');
        }
    });
    
    // Update course progress
    updateCourseProgressDisplay();
}

// 🎯 UPDATE COURSE PROGRESS DISPLAY
function updateCourseProgressDisplay() {
    console.log('🎯 Updating course progress display');
    
    // Count completed lessons
    const completedLessons = document.querySelectorAll('.lesson-completed, .completion-badge').length;
    const totalLessons = document.querySelectorAll('.lesson-item, .accordion-item').length;
    const progressPercent = Math.round((completedLessons / totalLessons) * 100);
    
    console.log(`📊 Progress: ${completedLessons}/${totalLessons} = ${progressPercent}%`);
    
    // Update progress bars
    const progressBars = document.querySelectorAll('.progress-bar');
    progressBars.forEach(bar => {
        bar.style.width = progressPercent + '%';
        bar.setAttribute('aria-valuenow', progressPercent);
        bar.textContent = progressPercent + '%';
    });
    
    // Update progress text
    const progressTexts = document.querySelectorAll('.course-progress, .progress-text');
    progressTexts.forEach(text => {
        text.textContent = `Course Progress ${progressPercent}%`;
    });
    
    console.log('✅ Course progress updated');
}

// Make functions globally accessible
window.togglePlayPause = function() {
    console.log(' Play/Pause button clicked');
    if (videoElement) {
        console.log(' Video element found, current state:', videoElement.paused ? 'paused' : 'playing');
        if (videoElement.paused) {
            videoElement.play().then(() => {
                console.log(' Video play successful');
            }).catch(error => {
                console.error(' Error playing video:', error);
            });
        } else {
            videoElement.pause();
            console.log(' Video paused');
        }
    } else {
        console.error(' Video element not found for play/pause!');
    }
};

window.toggleMute = function() {
    console.log(' Mute button clicked');
    if (videoElement) {
        videoElement.muted = !videoElement.muted;
        const volumeIcon = document.getElementById('volumeIcon');
        if (volumeIcon) {
            volumeIcon.className = videoElement.muted ? 'bx bx-volume-mute' : 'bx bx-volume-full';
        }
        console.log(' Video muted:', videoElement.muted);
    } else {
        console.error(' Video element not found for mute!');
    }
};

window.toggleFullscreen = function() {
    console.log(' Fullscreen button clicked');
    const videoContainer = document.getElementById('videoPlayer');
    if (videoContainer) {
        if (document.fullscreenElement) {
            document.exitFullscreen();
            console.log(' Exited fullscreen');
        } else {
            videoContainer.requestFullscreen();
            console.log(' Entered fullscreen');
        }
    } else {
        console.error(' Video container not found for fullscreen!');
    }
};

window.seekVideo = function(event) {
    console.log(' Seek bar clicked');
    if (videoElement && videoDuration > 0) {
        const progressBar = event.currentTarget;
        const rect = progressBar.getBoundingClientRect();
        const clickX = event.clientX - rect.left;
        const percentage = clickX / rect.width;
        const seekTime = percentage * videoDuration;
        
        console.log(' Seeking to:', seekTime, 'seconds (', Math.round(percentage * 100), '%)');
        videoElement.currentTime = seekTime;
    } else {
        console.error(' Video element not available or duration not loaded for seeking!');
    }
};

// Advanced Video Controls
window.skipBackward = function() {
    console.log(' Skip backward clicked');
    if (videoElement) {
        const newTime = Math.max(0, videoElement.currentTime - 10);
        videoElement.currentTime = newTime;
        console.log(' Skipped backward 10 seconds to:', newTime);
    } else {
        console.error(' Video element not found for skip backward!');
    }
};

window.skipForward = function() {
    console.log(' Skip forward clicked');
    if (videoElement && videoDuration > 0) {
        const newTime = Math.min(videoDuration, videoElement.currentTime + 10);
        videoElement.currentTime = newTime;
        console.log(' Skipped forward 10 seconds to:', newTime);
    } else {
        console.error(' Video element not available or duration not loaded for skip forward!');
    }
};

// Playback Speed Controls
window.toggleSpeedMenu = function() {
    console.log(' Speed menu toggled');
    const menu = document.getElementById('speedMenu');
    if (menu) {
        menu.classList.toggle('show');
        // Close other menus
        const qualityMenu = document.getElementById('qualityMenu');
        if (qualityMenu) {
            qualityMenu.classList.remove('show');
        }
    }
};

window.setPlaybackSpeed = function(speed) {
    console.log(' Setting playback speed to:', speed);
    if (videoElement) {
        videoElement.playbackRate = speed;
        const speedLabel = document.getElementById('speedLabel');
        if (speedLabel) {
            speedLabel.textContent = speed + 'x';
        }
        
        // Update active state
        document.querySelectorAll('#speedMenu .dropdown-item').forEach(item => {
            item.classList.remove('active');
        });
        if (event && event.target) {
            event.target.classList.add('active');
        }
        
        // Close menu
        const speedMenu = document.getElementById('speedMenu');
        if (speedMenu) {
            speedMenu.classList.remove('show');
        }
        console.log(' Playback speed set to:', speed + 'x');
    } else {
        console.error(' Video element not found for speed change!');
    }
};

// Quality Controls (placeholder - would need multiple video sources)
window.toggleQualityMenu = function() {
    console.log(' Quality menu toggled');
    const menu = document.getElementById('qualityMenu');
    if (menu) {
        menu.classList.toggle('show');
        // Close other menus
        const speedMenu = document.getElementById('speedMenu');
        if (speedMenu) {
            speedMenu.classList.remove('show');
        }
    }
};

// Initialize video controls when DOM is ready
document.addEventListener('DOMContentLoaded', function() {
    // Wait a bit for video element to be fully rendered
    setTimeout(() => {
        initializeVideoControls();
        console.log('🎮 Video controls initialization completed');
    }, 500);
    
    // Also initialize progress tracker
    initializeProgressTracker();
    
    // Initialize star rating system with delay to avoid conflicts
    setTimeout(() => {
        initializeStarRating();
        initializeReviewForm();
        console.log('⭐ Star rating system delayed initialization completed!');
    }, 1000);
    
    console.log('🎯 All video systems and star rating initialized!');
});

// ADDITIONAL: Initialize star rating on window load as backup
window.addEventListener('load', function() {
    setTimeout(() => {
        console.log('🔄 BACKUP: Re-initializing star rating system...');
        initializeStarRating();
        initializeReviewForm();
    }, 2000);
});

// ⭐ STAR RATING SYSTEM - MISSING FUNCTIONALITY
function initializeStarRating() {
    const ratingStars = document.querySelectorAll('.rating-star');
    const selectedRatingInput = document.getElementById('selectedRating');
    let currentRating = 0;
    
    if (ratingStars.length === 0) {
        console.log('⭐ No star rating elements found on this page');
        return;
    }
    
    console.log('⭐ Initializing star rating system with', ratingStars.length, 'stars');
    
    ratingStars.forEach((star, index) => {
        const rating = parseInt(star.getAttribute('data-rating'));
        
        // Add click handler
        star.addEventListener('click', function() {
            currentRating = rating;
            updateStarDisplay(currentRating);
            
            // Update hidden input
            if (selectedRatingInput) {
                selectedRatingInput.value = currentRating;
            }
            
            console.log('⭐ Star rating selected:', currentRating);
        });
        
        // Add hover effects
        star.addEventListener('mouseenter', function() {
            updateStarDisplay(rating, true);
        });
        
        star.addEventListener('mouseleave', function() {
            updateStarDisplay(currentRating);
        });
    });
    
    function updateStarDisplay(rating, isHover = false) {
        ratingStars.forEach((star, index) => {
            const starRating = parseInt(star.getAttribute('data-rating'));
            
            if (starRating <= rating) {
                star.classList.remove('bx-star');
                star.classList.add('bxs-star'); // Filled star
                star.style.color = isHover ? '#ffc107' : '#ff6b35';
            } else {
                star.classList.remove('bxs-star');
                star.classList.add('bx-star'); // Empty star
                star.style.color = '#dee2e6';
            }
        });
    }
    
    console.log('⭐ Star rating system initialized successfully!');
}

// REVIEW FORM FUNCTIONS
function cancelReview() {
    // Reset form
    const reviewForm = document.getElementById('reviewForm');
    if (reviewForm) {
        reviewForm.reset();
    }
    
    // Reset star rating
    const ratingStars = document.querySelectorAll('.rating-star');
    const selectedRatingInput = document.getElementById('selectedRating');
    
    ratingStars.forEach(star => {
        star.classList.remove('bxs-star');
        star.classList.add('bx-star');
        star.style.color = '#dee2e6';
    });
    
    if (selectedRatingInput) {
        selectedRatingInput.value = '';
    }
    
    console.log('⭐ Review form cancelled and reset');
}

// CHARACTER COUNTER FOR REVIEW TEXT
function initializeReviewForm() {
    const reviewText = document.getElementById('reviewText');
    const charCount = document.getElementById('charCount');
    
    if (reviewText && charCount) {
        reviewText.addEventListener('input', function() {
            const count = this.value.length;
            charCount.textContent = count;
            
            // Update color based on minimum requirement
            if (count >= 10) {
                charCount.style.color = '#28a745';
            } else {
                charCount.style.color = '#6c757d';
            }
        });
    }
    
    // Form submission handling
    const reviewForm = document.getElementById('reviewForm');
    if (reviewForm) {
        reviewForm.addEventListener('submit', function(e) {
            const selectedRating = document.getElementById('selectedRating');
            
            if (!selectedRating || !selectedRating.value) {
                e.preventDefault();
                alert('Please select a star rating before submitting your review.');
                return false;
            }
            
            console.log('📝 Review form submitted with rating:', selectedRating.value);
        });
    }
}

// Make functions globally accessible
window.cancelReview = cancelReview;
window.initializeStarRating = initializeStarRating;
window.initializeReviewForm = initializeReviewForm;

// ⚡ CRITICAL: Add VideoProgressTracker class inline since external script isn't loading
console.log('🎬 VideoProgressTracker class LOADING INLINE...');

class VideoProgressTracker {
    constructor(lessonId, courseId, studentId, options = {}) {
        console.log('🎬 VideoProgressTracker constructor called with:', {lessonId, courseId, studentId});
        this.lessonId = lessonId;
        this.courseId = courseId;
        this.studentId = studentId;
        this.currentTime = 0;
        this.duration = 0;
        this.progressPercentage = 0;
        this.isCompleted = false;
        this.updateFrequency = options.updateFrequency || 2000;
        this.completionThreshold = options.completionThreshold || 90;
        this.debug = options.debug || false;
        this.videoElement = null;
        this.updateInterval = null;
        
        // Callback functions
        this.onProgressUpdate = options.onProgressUpdate || function() {};
        this.onCompletion = options.onCompletion || function() {};
        this.onError = options.onError || function() {};
        
        this.initializeTracking();
    }
    
    initializeTracking() {
        console.log('🎬 VideoProgressTracker initializing tracking...');
        
        // Find video element
        this.videoElement = document.getElementById('lessonVideo');
        if (!this.videoElement) {
            this.onError('Video element not found');
            return;
        }
        
        console.log('✅ Video element found, setting up event listeners...');
        
        // Setup event listeners for duration detection
        this.videoElement.addEventListener('loadedmetadata', () => {
            this.duration = this.videoElement.duration;
            console.log('🎬 Video duration loaded via loadedmetadata:', this.duration);
        });
        
        // Backup: Try to get duration immediately if already loaded
        if (this.videoElement.duration && this.videoElement.duration > 0) {
            this.duration = this.videoElement.duration;
            console.log('🎬 Video duration found immediately:', this.duration);
        }
        
        // Backup: Poll for duration if not available yet
        const durationChecker = setInterval(() => {
            if (this.videoElement.duration && this.videoElement.duration > 0) {
                this.duration = this.videoElement.duration;
                console.log('🎬 Video duration found via polling:', this.duration);
                clearInterval(durationChecker);
            }
        }, 100);
        
        // Clear polling after 5 seconds to prevent infinite checking
        setTimeout(() => clearInterval(durationChecker), 5000);
        
        this.videoElement.addEventListener('timeupdate', () => {
            this.currentTime = this.videoElement.currentTime;
            
            // 🔧 FIX: Only calculate progress if we have a valid duration
            if (this.duration && this.duration > 0) {
                this.progressPercentage = (this.currentTime / this.duration) * 100;
            } else {
                // Try to get duration again if it's still not available
                if (this.videoElement.duration && this.videoElement.duration > 0) {
                    this.duration = this.videoElement.duration;
                    this.progressPercentage = (this.currentTime / this.duration) * 100;
                    console.log('🎬 Duration finally detected during timeupdate:', this.duration);
                } else {
                    this.progressPercentage = 0; // Fallback to 0% if no duration
                }
            }
            
            // 🎯 UPDATE VISUAL PROGRESS BAR
            this.updateProgressBar();
            
            if (this.debug) {
                console.log('📊 Progress update:', {
                    currentTime: this.currentTime,
                    duration: this.duration,
                    progress: this.progressPercentage.toFixed(2) + '%'
                });
            }
            
            // 🛑 CHECK COMPLETION FIRST - before any other callbacks
            if (this.progressPercentage >= 90 && !this.isCompleted) {
                this.isCompleted = true;
                console.log('🎉 Video completed at', this.progressPercentage.toFixed(2) + '%');
                
                // 🏆 IMMEDIATELY SHOW COMPLETED BADGE
                this.showCompletedBadge();
                
                // 📞 FIRE COMPLETION CALLBACK WITH DATABASE SAVE
                this.onCompletion({
                    lessonId: this.lessonId,
                    courseId: this.courseId,
                    studentId: this.studentId,
                    completionPercentage: this.progressPercentage
                });
                
                console.log('🛑 Video completed - stopping all progress tracking');
                return; // Stop all further progress processing
            }
            //console.log('🐛 DEBUG PANEL LOADED - Look for the debug panel in top-right corner of page')
            // Call progress callback only if not completed
            this.onProgressUpdate({
                currentTime: this.currentTime,
                duration: this.duration,
                progress: this.progressPercentage,
                lessonId: this.lessonId
            });
        });
        
        console.log('✅ VideoProgressTracker initialized successfully!');
        

    }
    
    // 🎯 UPDATE VISUAL PROGRESS BAR ON VIDEO CONTROLS + ACCORDION
    updateProgressBar() {
        try {
            if (this.duration && this.duration > 0) {
                const progressPercent = (this.currentTime / this.duration) * 100;
                
                // 🎯 UPDATE THE VIDEO CONTROLS PROGRESS BAR (.time-progress)
                const progressFill = document.querySelector('#timeProgress, .time-progress');
                if (progressFill) {
                    progressFill.style.width = progressPercent + '%';
                }
                
                // 🏗️ UPDATE THE ACCORDION PROGRESS BAR (LEFT SIDEBAR)
                if (this.lessonId) {
                    const accordionProgressBar = document.querySelector(`#accordion-progress-bar-${this.lessonId}`);
                    const accordionProgressThumb = document.querySelector(`#progress-thumb-${this.lessonId}`);
                    
                    if (accordionProgressBar) {
                        accordionProgressBar.style.width = progressPercent + '%';
                        console.log('🏗️ Accordion progress bar updated to:', progressPercent.toFixed(1) + '%');
                    }
                    
                    if (accordionProgressThumb) {
                        accordionProgressThumb.style.left = progressPercent + '%';
                        // Add active class if progress > 0
                        if (progressPercent > 0) {
                            accordionProgressThumb.classList.add('active');
                        }
                    }
                    
                    // 🎯 UPDATE PROGRESS TEXT IN ACCORDION (correct selector!)
                    const progressText = document.querySelector(`#progress-percentage-${this.lessonId}`);
                    const progressDisplay = document.querySelector(`#progress-display-${this.lessonId}`);
                    
                    if (progressText) {
                        progressText.textContent = progressPercent.toFixed(2);
                        console.log('📊 Progress text updated to:', progressPercent.toFixed(2) + '%');
                        
                        // 👁️ MAKE PROGRESS TEXT VISIBLE (was hidden with display: none)
                        if (progressDisplay) {
                            progressDisplay.style.display = 'inline-block';
                        }
                    } else {
                        console.log('❌ Progress text element not found: #progress-percentage-' + this.lessonId);
                    }
                    

                    
                    // 🏆 ADD COMPLETED BADGE DYNAMICALLY when video reaches 100%
                    if (progressPercent >= this.completionThreshold) {
                        this.showCompletedBadge();

                    }
                }
                
                // 🕐 UPDATE TIME DISPLAYS  
                const currentTimeDisplay = document.querySelector('#currentTime');
                const totalTimeDisplay = document.querySelector('#totalTime');
                
                if (currentTimeDisplay) {
                    currentTimeDisplay.textContent = this.formatTime(this.currentTime);
                }
                if (totalTimeDisplay) {
                    totalTimeDisplay.textContent = this.formatTime(this.duration);
                }
                
                if (this.debug && Math.random() < 0.05) { // Log occasionally for verification
                    console.log('🎯 ALL progress updated:', {
                        percent: progressPercent.toFixed(1) + '%',
                        lessonId: this.lessonId,
                        videoControlsUpdated: !!progressFill,
                        accordionUpdated: !!document.querySelector(`#accordion-progress-bar-${this.lessonId}`),
                        timeDisplayUpdated: !!currentTimeDisplay
                    });
                }
            }
        } catch (error) {
            console.error('❌ Error updating progress bar:', error);
        }
    }
    
    // Helper function to format time as MM:SS
    formatTime(seconds) {
        if (!seconds || seconds === 0) return '0:00';
        const mins = Math.floor(seconds / 60);
        const secs = Math.floor(seconds % 60);
        return mins + ':' + (secs < 10 ? '0' : '') + secs;
    }
    

    


    // 🏆 SHOW COMPLETED BADGE DYNAMICALLY
    showCompletedBadge() {
        try {
            // Find the lesson thumbnail container where the completed badge should appear
            const thumbnailContainer = document.querySelector(`[data-bs-target="#collapseLesson${this.lessonId}"] .lesson-thumbnail-small`);
            
            if (thumbnailContainer) {
                // Check if completed badge already exists
                let completedBadge = thumbnailContainer.querySelector('.completion-check');
                
                if (!completedBadge) {
                    // Create new completed badge
                    completedBadge = document.createElement('div');
                    completedBadge.className = 'completion-check position-absolute';
                    completedBadge.style.cssText = 'top: -5px; right: -5px;';
                    completedBadge.innerHTML = '<i class="bx bx-check-circle text-success fs-6"></i>';
                    
                    // Add to thumbnail container
                    thumbnailContainer.appendChild(completedBadge);
                    
                    console.log('🏆 Completed badge added for lesson:', this.lessonId);
                    
                    // Add animation effect
                    completedBadge.style.transform = 'scale(0)';
                    completedBadge.style.transition = 'transform 0.3s ease-in-out';
                    setTimeout(() => {
                        completedBadge.style.transform = 'scale(1)';
                    }, 100);
                    

                } else {
                    console.log('🏆 Completed badge already exists for lesson:', this.lessonId);
                }
            } else {
                console.log('❌ Could not find thumbnail container for lesson:', this.lessonId);
            }
        } catch (error) {
            console.error('❌ Error adding completed badge:', error);
        }
    }
    
    destroy() {
        if (this.updateInterval) {
            clearInterval(this.updateInterval);
        }
        console.log('🔄 VideoProgressTracker destroyed');
    }
}

// Make VideoProgressTracker globally available
window.VideoProgressTracker = VideoProgressTracker;
console.log('✅ VideoProgressTracker class loaded and available globally!');

</script>

<!-- ⚡ CRITICAL: Add missing VideoProgressTracker script for file video progress tracking -->
<script src="js/video-progress-tracker.js?v=<?= time() ?>"></script>










// ========================================
// 📝 REVIEW EDIT FUNCTIONALITY
// ========================================

window.editReview = function(reviewId) {
    console.log('✏️ EDIT: Starting review edit for ID:', reviewId);
    
    // Find the review container - try multiple selectors
    let reviewContainer = null;
    
    // Method 1: Try data-review-id attribute
    reviewContainer = document.querySelector(`[data-review-id="${reviewId}"]`);
    
    // Method 2: Try finding by the Edit button's parent container
    if (!reviewContainer) {
        const editButtons = document.querySelectorAll('button');
        for (let button of editButtons) {
            const onclick = button.getAttribute('onclick');
            if (onclick && onclick.includes(`editReview(${reviewId})`)) {
                // Find the closest review container (try different parent levels)
                reviewContainer = button.closest('.review-item') || 
                                button.closest('.review-container') || 
                                button.closest('[class*="review"]') || 
                                button.closest('.card') ||
                                button.closest('.border') ||
                                button.parentElement.parentElement;
                console.log('✅ EDIT: Found container via Edit button parent traversal');
                break;
            }
        }
    }
    
    // Method 3: Try finding "Your Review" section
    if (!reviewContainer) {
        const reviewHeaders = document.querySelectorAll('h4, h5, h6, .font-weight-bold, .fw-bold');
        for (let header of reviewHeaders) {
            if (header.textContent.includes('Your Review') || header.textContent.includes('🔸 Your Review')) {
                reviewContainer = header.closest('div') || header.parentElement;
                console.log('✅ EDIT: Found container via "Your Review" header');
                break;
            }
        }
    }
    
    // Method 4: Try finding any container with review content and Edit button
    if (!reviewContainer) {
        const allContainers = document.querySelectorAll('div');
        for (let container of allContainers) {
            const hasEditButton = container.querySelector('button[onclick*="editReview"]');
            const hasStars = container.querySelector('.bx-star, .bxs-star, [class*="star"]');
            const hasText = container.textContent.length > 20;
            
            if (hasEditButton && (hasStars || hasText)) {
                reviewContainer = container;
                console.log('✅ EDIT: Found container via content analysis');
                break;
            }
        }
    }
    
    if (!reviewContainer) {
        console.error('❌ EDIT: Could not find review container for ID:', reviewId);
        console.log('🔍 EDIT: Available containers with Edit buttons:');
        document.querySelectorAll('button[onclick*="editReview"]').forEach((btn, i) => {
            console.log(`   ${i + 1}. Button:`, btn.getAttribute('onclick'), 'Parent:', btn.parentElement.className);
        });
        alert('Review container not found. Please refresh the page and try again.');
        return;
    }
    
    // Add the data attribute for future use
    reviewContainer.setAttribute('data-review-id', reviewId);
    console.log('✅ EDIT: Found review container and added data-review-id attribute');
    
    // Get current review data
    const currentRating = reviewContainer.dataset.rating || '5';
    const currentTitle = reviewContainer.querySelector('.review-title')?.textContent.trim() || '';
    const currentText = reviewContainer.querySelector('.review-text')?.textContent.trim() || '';
    
    console.log('📊 EDIT: Current review data:', {
        rating: currentRating,
        title: currentTitle,
        text: currentText.substring(0, 100) + '...'
    });
    
    // Create edit form HTML
    const editFormHTML = `
        <div class="edit-review-form" style="background: #f8f9fa; padding: 20px; border-radius: 8px; border: 2px solid #007bff;">
            <h5 style="color: #007bff; margin-bottom: 15px;">
                <i class="bx bx-edit-alt"></i> Edit Your Review
            </h5>
            
            <!-- Star Rating Edit -->
            <div class="mb-3">
                <label style="font-weight: bold; margin-bottom: 8px; display: block;">Rating:</label>
                <div class="star-rating-edit" data-rating="${currentRating}" id="star-rating-edit-${reviewId}">
                    <i class="bx bx-star" data-rating="1" style="font-size: 24px; color: #e0e0e0; cursor: pointer; margin-right: 5px;" onmouseover="highlightEditStars(1)" onmouseout="resetEditStars()" onclick="setEditRating(1)"></i>
                    <i class="bx bx-star" data-rating="2" style="font-size: 24px; color: #e0e0e0; cursor: pointer; margin-right: 5px;" onmouseover="highlightEditStars(2)" onmouseout="resetEditStars()" onclick="setEditRating(2)"></i>
                    <i class="bx bx-star" data-rating="3" style="font-size: 24px; color: #e0e0e0; cursor: pointer; margin-right: 5px;" onmouseover="highlightEditStars(3)" onmouseout="resetEditStars()" onclick="setEditRating(3)"></i>
                    <i class="bx bx-star" data-rating="4" style="font-size: 24px; color: #e0e0e0; cursor: pointer; margin-right: 5px;" onmouseover="highlightEditStars(4)" onmouseout="resetEditStars()" onclick="setEditRating(4)"></i>
                    <i class="bx bx-star" data-rating="5" style="font-size: 24px; color: #e0e0e0; cursor: pointer; margin-right: 5px;" onmouseover="highlightEditStars(5)" onmouseout="resetEditStars()" onclick="setEditRating(5)"></i>
                </div>
            </div>
            
            <!-- Title Edit -->
            <div class="mb-3">
                <label for="edit-title-${reviewId}" style="font-weight: bold;">Review Title:</label>
                <input type="text" 
                       id="edit-title-${reviewId}" 
                       class="form-control" 
                       value="${currentTitle.replace(/"/g, '&quot;')}" 
                       maxlength="255"
                       placeholder="Enter review title">
                <small class="text-muted">Max 255 characters</small>
            </div>
            
            <!-- Text Edit -->
            <div class="mb-4">
                <label for="edit-text-${reviewId}" style="font-weight: bold;">Review Text:</label>
                <textarea id="edit-text-${reviewId}" 
                          class="form-control" 
                          rows="4" 
                          maxlength="1000"
                          placeholder="Share your experience with this course...">${currentText}</textarea>
                <small class="text-muted"><span id="edit-char-count-${reviewId}">0</span>/1000 characters</small>
            </div>
            
            <!-- Action Buttons -->
            <div class="d-flex gap-2">
                <button type="button" 
                        class="btn btn-success" 
                        onclick="saveReviewEdit(${reviewId})"
                        id="save-edit-btn-${reviewId}">
                    <i class="bx bx-check"></i> Save Changes
                </button>
                <button type="button" 
                        class="btn btn-secondary" 
                        onclick="cancelReviewEdit(${reviewId})">
                    <i class="bx bx-x"></i> Cancel
                </button>
            </div>
        </div>
    `;
    
    // Store original content for cancel functionality
    window.originalReviewContent = reviewContainer.innerHTML;
    
    // Replace review display with edit form
    reviewContainer.innerHTML = editFormHTML;
    
    // Set up character counter
    const textArea = document.getElementById(`edit-text-${reviewId}`);
    const charCount = document.getElementById(`edit-char-count-${reviewId}`);
    
    function updateCharCount() {
        charCount.textContent = textArea.value.length;
        charCount.style.color = textArea.value.length > 900 ? '#dc3545' : '#6c757d';
    }
    
    textArea.addEventListener('input', updateCharCount);
    updateCharCount(); // Initial count
    
    console.log('✅ EDIT: Edit form created and displayed');
};

// Star rating functions for edit mode
window.highlightEditStars = function(rating) {
    const stars = document.querySelectorAll('.star-rating-edit i');
    stars.forEach((star, index) => {
        if (index < rating) {
            star.className = 'bx bxs-star';
            star.style.color = '#ffc107';
        } else {
            star.className = 'bx bx-star';
            star.style.color = '#e0e0e0';
        }
    });
};

window.resetEditStars = function() {
    const container = document.querySelector('.star-rating-edit');
    if (!container) return;
    
    const currentRating = parseInt(container.dataset.rating);
    const stars = container.querySelectorAll('i');
    
    stars.forEach((star, index) => {
        if (index < currentRating) {
            star.className = 'bx bxs-star';
            star.style.color = '#ffc107';
        } else {
            star.className = 'bx bx-star';
            star.style.color = '#e0e0e0';
        }
    });
};

window.setEditRating = function(rating) {
    console.log('⭐ EDIT: Setting rating to:', rating);
    const container = document.querySelector('.star-rating-edit');
    if (container) {
        container.dataset.rating = rating;
        highlightEditStars(rating);
    }
};

window.saveReviewEdit = function(reviewId) {
    console.log('💾 EDIT: Saving review changes for ID:', reviewId);
    
    const rating = document.querySelector('.star-rating-edit')?.dataset.rating;
    const title = document.getElementById(`edit-title-${reviewId}`)?.value.trim();
    const text = document.getElementById(`edit-text-${reviewId}`)?.value.trim();
    
    // Validation
    if (!rating || rating < 1 || rating > 5) {
        alert('Please select a rating (1-5 stars)');
        return;
    }
    
    if (!title || title.length < 3) {
        alert('Review title must be at least 3 characters long');
        return;
    }
    
    if (!text || text.length < 10) {
        alert('Review text must be at least 10 characters long');
        return;
    }
    
    const reviewData = { rating, title, text };
    console.log('📤 EDIT: Sending updated review data:', reviewData);
    
    // Show loading state
    const saveBtn = document.getElementById(`save-edit-btn-${reviewId}`);
    const originalBtnText = saveBtn.innerHTML;
    saveBtn.innerHTML = '<i class="bx bx-loader bx-spin"></i> Saving...';
    saveBtn.disabled = true;
    
    // Send AJAX request to update review
    fetch('update_review.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
        },
        body: JSON.stringify({
            review_id: reviewId,
            rating: rating,
            title: title,
            text: text
        })
    })
    .then(response => {
        console.log('📡 AJAX: Raw response status:', response.status, response.statusText);
        console.log('📡 AJAX: Response headers:', response.headers.get('Content-Type'));
        
        // Get the raw text first to see what we're actually receiving
        return response.text();
    })
    .then(rawText => {
        console.log('📄 AJAX: Raw response text:', rawText);
        console.log('📄 AJAX: Raw response length:', rawText.length);
        
        // Try to parse as JSON
        let data;
        try {
            data = JSON.parse(rawText);
            console.log('✅ AJAX: Successfully parsed JSON:', data);
        } catch (parseError) {
            console.error('❌ AJAX: JSON Parse Error:', parseError);
            console.error('❌ AJAX: Failed to parse this text as JSON:', rawText.substring(0, 500));
            throw new Error('Server returned invalid JSON response');
        }
        
        if (data.success) {
            console.log('✅ AJAX: Review updated successfully');
            alert('Review updated successfully!');
            // Reload page to show updated review
            window.location.reload();
        } else {
            console.error('❌ AJAX: Server returned error:', data.message);
            alert('Failed to update review: ' + (data.message || 'Unknown error'));
            
            // Restore button
            saveBtn.innerHTML = originalBtnText;
            saveBtn.disabled = false;
        }
    })
    .catch(error => {
        console.error('❌ AJAX: Network/parsing error:', error);
        alert('Network error: ' + error.message);
        
        // Restore button
        saveBtn.innerHTML = originalBtnText;
        saveBtn.disabled = false;
    });
};

window.cancelReviewEdit = function(reviewId) {
    console.log('❌ EDIT: Cancelling review edit for ID:', reviewId);
    
    const reviewContainer = document.querySelector(`[data-review-id="${reviewId}"]`);
    if (reviewContainer && window.originalReviewContent) {
        reviewContainer.innerHTML = window.originalReviewContent;
        delete window.originalReviewContent;
        console.log('✅ EDIT: Original review content restored');
    }
};

console.log('✅ EDIT: Review edit functionality loaded');

// ========================================
// 📝 NOTES FUNCTIONALITY - MISSING IMPLEMENTATION
// ========================================

// Notes storage in localStorage with course/lesson context
function getNotesStorageKey() {
    const courseId = <?= json_encode($course_id) ?>;
    const lessonId = <?= json_encode($current_lesson_id) ?>;
    return `course_notes_${courseId}_lesson_${lessonId}`;
}

// Load existing notes from localStorage
function loadNotes() {
    const storageKey = getNotesStorageKey();
    const notes = localStorage.getItem(storageKey);
    return notes ? JSON.parse(notes) : [];
}

// Save notes to localStorage
function saveNotesToStorage(notes) {
    const storageKey = getNotesStorageKey();
    localStorage.setItem(storageKey, JSON.stringify(notes));
    console.log('📝 NOTES: Saved', notes.length, 'notes to storage');
}

// Get current video timestamp
function getCurrentVideoTime() {
    const video = document.querySelector('video');
    if (video) {
        return video.currentTime || 0;
    }
    return 0;
}

// Format time for display (MM:SS)
function formatTime(seconds) {
    const mins = Math.floor(seconds / 60);
    const secs = Math.floor(seconds % 60);
    return `${mins.toString().padStart(2, '0')}:${secs.toString().padStart(2, '0')}`;
}

// Focus note input (called by Add Note button)
window.focusNoteInput = function() {
    const noteInput = document.querySelector('.note-input');
    if (noteInput) {
        noteInput.focus();
        noteInput.scrollIntoView({ behavior: 'smooth', block: 'center' });
        console.log('📝 NOTES: Input focused');
    }
};

// Save note function
function saveNote() {
    const noteInput = document.querySelector('.note-input');
    const noteText = noteInput.value.trim();
    
    if (!noteText) {
        alert('Please enter a note before saving.');
        return;
    }
    
    const currentTime = getCurrentVideoTime();
    const timestamp = new Date().toISOString();
    
    const note = {
        id: Date.now(), // Simple ID generation
        text: noteText,
        videoTime: currentTime,
        timestamp: timestamp,
        formattedTime: formatTime(currentTime)
    };
    
    // Load existing notes and add new one
    const notes = loadNotes();
    notes.push(note);
    
    // Sort notes by video time
    notes.sort((a, b) => a.videoTime - b.videoTime);
    
    // Save to storage
    saveNotesToStorage(notes);
    
    // Clear input and refresh display
    noteInput.value = '';
    displayNotes();
    
    console.log('📝 NOTES: Note saved at', note.formattedTime, ':', noteText);
    
    // Show success feedback
    showNoteSuccessMessage('Note saved successfully!');
}

// Display notes in the notes list
function displayNotes() {
    const notesList = document.getElementById('notesList');
    const notes = loadNotes();
    
    if (notes.length === 0) {
        notesList.innerHTML = `
            <div class="no-notes text-center py-3">
                <i class="bx bx-note text-muted" style="font-size: 1.5rem;"></i>
                <p class="text-muted mt-2 mb-0 small">No notes yet. Add your first note while watching!</p>
            </div>
        `;
        return;
    }
    
    // Create accordion-style notes with compact design
    const notesHTML = `
        <div class="accordion accordion-flush" id="notesAccordion">
            ${notes.map((note, index) => `
                <div class="accordion-item border-0 border-bottom" data-note-id="${note.id}">
                    <h2 class="accordion-header" id="noteHeading${note.id}">
                        <button class="accordion-button collapsed py-2 px-0" 
                                type="button" 
                                data-bs-toggle="collapse" 
                                data-bs-target="#noteCollapse${note.id}" 
                                aria-expanded="false" 
                                aria-controls="noteCollapse${note.id}"
                                style="font-size: 0.85rem; line-height: 1.3;">
                            <div class="d-flex align-items-center justify-content-between w-100 me-2">
                                <div class="d-flex align-items-center gap-2">
                                    <button class="btn btn-outline-primary btn-sm py-1 px-2" 
                                            onclick="event.stopPropagation(); seekToNoteTime(${note.videoTime})" 
                                            title="Jump to this time"
                                            style="font-size: 0.7rem; line-height: 1;">
                                        <i class="bx bx-play" style="font-size: 0.8rem;"></i> ${note.formattedTime}
                                    </button>
                                    <span class="text-truncate" style="max-width: 200px; font-size: 0.8rem;">
                                        ${note.text.length > 50 ? note.text.substring(0, 50) + '...' : note.text}
                                    </span>
                                </div>
                                <div class="d-flex align-items-center gap-1">
                                    <small class="text-muted" style="font-size: 0.65rem; white-space: nowrap;">
                                        ${new Date(note.timestamp).toLocaleDateString()} ${new Date(note.timestamp).toLocaleTimeString('en-US', {hour: '2-digit', minute: '2-digit'})}
                                    </small>
                                    <button class="btn btn-outline-danger btn-sm py-0 px-1 ms-1" 
                                            onclick="event.stopPropagation(); deleteNote(${note.id})" 
                                            title="Delete note"
                                            style="font-size: 0.7rem; line-height: 1;">
                                        <i class="bx bx-trash" style="font-size: 0.75rem;"></i>
                                    </button>
                                </div>
                            </div>
                        </button>
                    </h2>
                    <div id="noteCollapse${note.id}" 
                         class="accordion-collapse collapse" 
                         aria-labelledby="noteHeading${note.id}" 
                         data-bs-parent="#notesAccordion">
                        <div class="accordion-body py-2 px-0" style="font-size: 0.85rem; line-height: 1.4;">
                            <div class="note-full-text p-2 bg-light rounded" style="font-size: 0.8rem;">
                                ${note.text.replace(/\n/g, '<br>')}
                            </div>
                            <div class="note-details mt-2">
                                <div class="d-flex justify-content-between align-items-center">
                                    <small class="text-muted" style="font-size: 0.7rem;">
                                        <i class="bx bx-calendar" style="font-size: 0.75rem;"></i> 
                                        Created: ${new Date(note.timestamp).toLocaleString()}
                                    </small>
                                    <small class="text-muted" style="font-size: 0.7rem;">
                                        <i class="bx bx-time" style="font-size: 0.75rem;"></i> 
                                        Video time: ${note.formattedTime}
                                    </small>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            `).join('')}
        </div>
    `;
    
    notesList.innerHTML = notesHTML;
}

// Seek to note time in video
window.seekToNoteTime = function(time) {
    const video = document.querySelector('video');
    if (video) {
        video.currentTime = time;
        console.log('📝 NOTES: Seeked to', formatTime(time));
        
        // Switch to description tab to show video
        const descTab = document.querySelector('[data-tab="description"]');
        if (descTab) {
            descTab.click();
        }
    }
};

// Delete note
window.deleteNote = function(noteId) {
    if (confirm('Are you sure you want to delete this note?')) {
        const notes = loadNotes();
        const filteredNotes = notes.filter(note => note.id !== noteId);
        saveNotesToStorage(filteredNotes);
        displayNotes();
        console.log('📝 NOTES: Note deleted:', noteId);
    }
};

// Export notes function
window.exportNotes = function() {
    const notes = loadNotes();
    if (notes.length === 0) {
        alert('No notes to export.');
        return;
    }
    
    const courseTitle = <?= json_encode($course['title']) ?>;
    const lessonTitle = <?= json_encode($current_lesson['title'] ?? 'Lesson ' . $current_lesson_id) ?>;
    
    let exportText = `${courseTitle} - ${lessonTitle}\n`;
    exportText += `Notes exported on ${new Date().toLocaleString()}\n\n`;
    
    notes.forEach((note, index) => {
        exportText += `${index + 1}. [${note.formattedTime}] ${note.text}\n\n`;
    });
    
    // Create downloadable file
    const blob = new Blob([exportText], { type: 'text/plain' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = `notes_${courseTitle}_lesson_${lessonId}.txt`.replace(/[^a-z0-9]/gi, '_');
    document.body.appendChild(a);
    a.click();
    document.body.removeChild(a);
    URL.revokeObjectURL(url);
    
    console.log('📝 NOTES: Exported', notes.length, 'notes');
};

// Print notes function
window.printNotes = function() {
    const notes = loadNotes();
    if (notes.length === 0) {
        alert('No notes to print.');
        return;
    }
    
    const courseTitle = <?= json_encode($course['title']) ?>;
    const lessonTitle = <?= json_encode($current_lesson['title'] ?? 'Lesson ' . $current_lesson_id) ?>;
    
    let printHTML = `
        <html>
        <head>
            <title>Notes - ${courseTitle}</title>
            <style>
                body { font-family: Arial, sans-serif; margin: 20px; }
                .header { border-bottom: 2px solid #333; padding-bottom: 10px; margin-bottom: 20px; }
                .note { margin-bottom: 15px; padding: 10px; border-left: 3px solid #007bff; }
                .note-time { font-weight: bold; color: #007bff; }
                .note-text { margin: 5px 0; }
                .note-date { font-size: 0.9em; color: #666; }
            </style>
        </head>
        <body>
            <div class="header">
                <h1>${courseTitle}</h1>
                <h2>${lessonTitle}</h2>
                <p>Notes printed on ${new Date().toLocaleString()}</p>
            </div>
    `;
    
    notes.forEach((note, index) => {
        printHTML += `
            <div class="note">
                <div class="note-time">${index + 1}. [${note.formattedTime}]</div>
                <div class="note-text">${note.text.replace(/\n/g, '<br>')}</div>
                <div class="note-date">${new Date(note.timestamp).toLocaleString()}</div>
            </div>
        `;
    });
    
    printHTML += '</body></html>';
    
    const printWindow = window.open('', '_blank');
    printWindow.document.write(printHTML);
    printWindow.document.close();
    printWindow.print();
    
    console.log('📝 NOTES: Print dialog opened for', notes.length, 'notes');
};

// Show success message
function showNoteSuccessMessage(message) {
    const notification = document.createElement('div');
    notification.className = 'alert alert-success alert-dismissible fade show position-fixed';
    notification.style.cssText = 'top: 20px; right: 20px; z-index: 9999; min-width: 300px;';
    notification.innerHTML = `
        <i class="bx bx-check-circle"></i> ${message}
        <button type="button" class="btn-close" onclick="this.parentElement.remove()"></button>
    `;
    document.body.appendChild(notification);
    
    // Auto-remove after 3 seconds
    setTimeout(() => {
        if (notification.parentElement) {
            notification.remove();
        }
    }, 3000);
}

// Initialize notes functionality
function initializeNotes() {
    // Add click handler to save button
    const saveBtn = document.querySelector('.save-note-btn');
    if (saveBtn) {
        saveBtn.addEventListener('click', saveNote);
        console.log('📝 NOTES: Save button initialized');
    }
    
    // Add enter key handler to textarea (Ctrl+Enter to save)
    const noteInput = document.querySelector('.note-input');
    if (noteInput) {
        noteInput.addEventListener('keydown', function(e) {
            if (e.ctrlKey && e.key === 'Enter') {
                saveNote();
            }
        });
        console.log('📝 NOTES: Keyboard shortcuts initialized');
    }
    
    // Load and display existing notes
    displayNotes();
    
    console.log('✅ NOTES: Notes functionality initialized successfully');
}

// Initialize notes when DOM is ready
document.addEventListener('DOMContentLoaded', function() {
    // Delay initialization to ensure all elements are ready
    setTimeout(() => {
        initializeNotes();
    }, 1000);
});

console.log('✅ NOTES: Notes functionality loaded');
</script>

<style>
/* Clean Small Circular Progress Indicator */
.progress-header-compact {
    display: flex;
    align-items: center;
    gap: 0.75rem;
}

.progress-circle-wrapper-small {
    position: relative;
    flex-shrink: 0;
}

.progress-circle-small {
    width: 45px;
    height: 45px;
    border-radius: 50%;
    background: conic-gradient(
        #667eea 0deg,
        #667eea calc(var(--progress, 0) * 3.6deg),
        #e5e7eb calc(var(--progress, 0) * 3.6deg),
        #e5e7eb 360deg
    );
    display: flex;
    align-items: center;
    justify-content: center;
    position: relative;
    transition: all 0.3s ease;
}

.progress-circle-inner-small {
    width: 35px;
    height: 35px;
    background: white;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    box-shadow: 0 2px 6px rgba(0, 0, 0, 0.1);
}

.progress-text-small {
    font-size: 0.7rem;
    font-weight: 600;
    color: #1f2937;
}

.progress-details-small {
    display: flex;
    flex-direction: column;
    gap: 0.2rem;
}

.progress-label-small {
    font-size: 0.8rem;
    font-weight: 600;
    color: #374151;
}

.progress-breakdown-small {
    font-size: 0.65rem;
    color: #6b7280;
}

.progress-breakdown-small small {
    font-size: 0.65rem;
}

/* Responsive adjustments for small progress indicator */
@media (max-width: 768px) {
    .progress-header-compact {
        gap: 0.5rem;
    }
    
    .progress-circle-small {
        width: 40px;
        height: 40px;
    }
    
    .progress-circle-inner-small {
        width: 30px;
        height: 30px;
    }
    
    .progress-text-small {
        font-size: 0.6rem;
    }
    
    .progress-label-small {
        font-size: 0.75rem;
    }
    
    .progress-breakdown-small {
        font-size: 0.6rem;
    }
}

/* Smart Certificate Guidance Message Styling */
.certificate-guidance {
    width: 100%;
    margin-top: 1rem;
}

.certificate-available {
    display: flex;
    align-items: center;
    justify-content: space-between;
    gap: 1rem;
    padding: 1rem;
    background: linear-gradient(135deg, #f0f9ff 0%, #e0f2fe 100%);
    border: 1px solid #0ea5e9;
    border-radius: 12px;
}

.certificate-requirements {
    padding: 1rem;
    background: linear-gradient(135deg, #fefce8 0%, #fef3c7 100%);
    border: 1px solid #f59e0b;
    border-radius: 12px;
}

.certificate-message {
    display: flex;
    align-items: flex-start;
    gap: 0.75rem;
}

.certificate-message i {
    font-size: 1.25rem;
    margin-top: 0.125rem;
}

.message-content {
    display: flex;
    flex-direction: column;
    gap: 0.25rem;
}

.message-title {
    font-size: 0.875rem;
    font-weight: 600;
    color: #1f2937;
    line-height: 1.2;
}

.message-subtitle {
    font-size: 0.75rem;
    color: #6b7280;
    line-height: 1.3;
}

.btn-certificate-view {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    padding: 0.75rem 1.5rem;
    background: linear-gradient(135deg, #10b981 0%, #059669 100%);
    color: white;
    text-decoration: none;
    border-radius: 8px;
    font-size: 0.875rem;
    font-weight: 600;
    transition: all 0.2s ease;
    white-space: nowrap;
}

.btn-certificate-view:hover {
    background: linear-gradient(135deg, #059669 0%, #047857 100%);
    transform: translateY(-1px);
    box-shadow: 0 4px 12px rgba(16, 185, 129, 0.3);
    color: white;
    text-decoration: none;
}

.requirements-checklist {
    margin-top: 0.75rem;
    display: flex;
    flex-direction: column;
    gap: 0.5rem;
}

.requirement-item {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    padding: 0.5rem;
    background: white;
    border-radius: 6px;
    font-size: 0.8rem;
}

.requirement-item i {
    font-size: 1rem;
    flex-shrink: 0;
}

.requirement-item.completed {
    background: #f0fdf4;
    border: 1px solid #bbf7d0;
}

.requirement-item.pending {
    background: #fefce8;
    border: 1px solid #fde047;
}

.status-badge {
    margin-left: auto;
    padding: 0.25rem 0.5rem;
    font-size: 0.7rem;
    font-weight: 500;
    border-radius: 4px;
    white-space: nowrap;
}

.status-badge.completed {
    background: #dcfce7;
    color: #166534;
}

.status-badge.pending {
    background: #fef3c7;
    color: #92400e;
}

/* Custom Header Layout (Replaces Bootstrap Row to Avoid Equal Heights) */
.custom-header-layout {
    display: flex;
    align-items: flex-start;
    gap: 1rem;
}

.course-content-main {
    flex: 1;
    width: 75%;
}

.course-certificate-sidebar {
    flex-shrink: 0;
    width: 25%;
    max-width: 300px;
}

/* Mobile Responsive Layout */
@media (max-width: 768px) {
    .custom-header-layout {
        flex-direction: column;
        gap: 1.5rem;
    }
    
    .course-content-main {
        width: 100%;
    }
    
    .course-certificate-sidebar {
        width: 100%;
        max-width: none;
        order: 2; /* Certificate section moves below content */
    }
    
    /* Make certificate sidebar more compact on mobile */
    .certificate-requirements-sidebar,
    .certificate-available-sidebar {
        padding: 0.75rem;
    }
    
    .requirement-item-sidebar {
        padding: 0.5rem;
        font-size: 0.75rem;
    }
    
    .btn-certificate-view-sidebar {
        padding: 0.75rem 1rem;
        font-size: 0.8rem;
    }
}

/* Debug borders removed - clean layout achieved */

/* Sidebar Certificate Guidance Styling (9-3 Layout) */
.certificate-guidance-sidebar {
    /* Removed sticky positioning - might be causing height stretch */
}

.certificate-available-sidebar {
    padding: 1rem;
    background: linear-gradient(135deg, #f0f9ff 0%, #e0f2fe 100%);
    border: 2px solid #0ea5e9;
    border-radius: 12px;
    box-shadow: 0 4px 12px rgba(14, 165, 233, 0.15);
}

.certificate-requirements-sidebar {
    padding: 1rem;
    background: linear-gradient(135deg, #fefce8 0%, #fef3c7 100%);
    border: 2px solid #f59e0b;
    border-radius: 12px;
    box-shadow: 0 4px 12px rgba(245, 158, 11, 0.15);
    max-height: fit-content;
}

.requirements-checklist-sidebar {
    margin-top: 1rem;
    display: flex;
    flex-direction: column;
    gap: 0.75rem;
}

.requirement-item-sidebar {
    display: flex;
    align-items: center;
    gap: 0.75rem;
    padding: 0.75rem;
    background: white;
    border-radius: 8px;
    font-size: 0.8rem;
    box-shadow: 0 2px 4px rgba(0, 0, 0, 0.05);
}

.requirement-item-sidebar i {
    font-size: 1rem;
    flex-shrink: 0;
}

.requirement-item-sidebar .requirement-text {
    display: flex;
    flex-direction: column;
    flex: 1;
    line-height: 1.3;
}

.requirement-item-sidebar .requirement-text span {
    font-weight: 600;
    color: #374151;
    margin-bottom: 2px;
}

.requirement-item-sidebar .requirement-text small {
    color: #6b7280;
    font-size: 0.7rem;
}

.status-badge-sidebar {
    font-size: 0.9rem;
    flex-shrink: 0;
}

.requirement-item-sidebar.completed {
    background: #f0fdf4;
    border: 1px solid #bbf7d0;
}

.requirement-item-sidebar.pending {
    background: #fefce8;
    border: 1px solid #fde047;
}

.btn-certificate-view-sidebar {
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 0.5rem;
    padding: 1rem 1.25rem;
    background: linear-gradient(135deg, #10b981 0%, #059669 100%);
    color: white;
    text-decoration: none;
    border-radius: 10px;
    font-size: 0.9rem;
    font-weight: 700;
    text-transform: uppercase;
    letter-spacing: 0.5px;
    transition: all 0.3s ease;
    margin-top: auto;
    box-shadow: 0 4px 12px rgba(16, 185, 129, 0.3);
}

.btn-certificate-view-sidebar:hover {
    background: linear-gradient(135deg, #059669 0%, #047857 100%);
    transform: translateY(-2px);
    box-shadow: 0 6px 20px rgba(16, 185, 129, 0.4);
    color: white;
    text-decoration: none;
}

/* Horizontal 3-Column Layout Styling */
.progress-grade-section {
    width: 100%;
}

.certificate-guidance-horizontal {
    height: 100%;
}

.certificate-available-horizontal {
    padding: 1rem;
    background: linear-gradient(135deg, #f0f9ff 0%, #e0f2fe 100%);
    border: 1px solid #0ea5e9;
    border-radius: 12px;
    height: 100%;
    display: flex;
    flex-direction: column;
    gap: 1rem;
}

.certificate-requirements-horizontal {
    padding: 1rem;
    background: linear-gradient(135deg, #fefce8 0%, #fef3c7 100%);
    border: 1px solid #f59e0b;
    border-radius: 12px;
    height: 100%;
}

.requirements-checklist-horizontal {
    margin-top: 0.75rem;
    display: flex;
    flex-direction: column;
    gap: 0.5rem;
}

.requirement-item-horizontal {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    padding: 0.5rem;
    background: white;
    border-radius: 6px;
    font-size: 0.75rem;
}

.requirement-item-horizontal i {
    font-size: 0.9rem;
    flex-shrink: 0;
}

.requirement-text {
    display: flex;
    flex-direction: column;
    flex: 1;
    line-height: 1.2;
}

.requirement-text span {
    font-weight: 500;
    color: #374151;
}

.requirement-text small {
    color: #6b7280;
    font-size: 0.65rem;
}

.status-badge-horizontal {
    font-size: 0.75rem;
    flex-shrink: 0;
}

.requirement-item-horizontal.completed {
    background: #f0fdf4;
    border: 1px solid #bbf7d0;
}

.requirement-item-horizontal.pending {
    background: #fefce8;
    border: 1px solid #fde047;
}

.btn-certificate-view-horizontal {
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 0.5rem;
    padding: 0.75rem 1rem;
    background: linear-gradient(135deg, #10b981 0%, #059669 100%);
    color: white;
    text-decoration: none;
    border-radius: 8px;
    font-size: 0.8rem;
    font-weight: 600;
    transition: all 0.2s ease;
    margin-top: auto;
}

.btn-certificate-view-horizontal:hover {
    background: linear-gradient(135deg, #059669 0%, #047857 100%);
    transform: translateY(-1px);
    box-shadow: 0 4px 12px rgba(16, 185, 129, 0.3);
    color: white;
    text-decoration: none;
}

/* Responsive adjustments for certificate guidance */
@media (max-width: 768px) {
    .certificate-available {
        flex-direction: column;
        align-items: flex-start;
        gap: 0.75rem;
    }
    
    .btn-certificate-view {
        align-self: stretch;
        justify-content: center;
    }
    
    .message-title {
        font-size: 0.8rem;
    }
    
    .message-subtitle {
        font-size: 0.7rem;
    }
    
    .requirement-item {
        font-size: 0.75rem;
    }
    
    /* Horizontal layout adjustments for mobile */
    .certificate-available-horizontal,
    .certificate-requirements-horizontal {
        margin-top: 1rem;
    }
    
    .requirement-item-horizontal {
        font-size: 0.7rem;
    }
    
    .btn-certificate-view-horizontal {
        font-size: 0.75rem;
        padding: 0.6rem 0.8rem;
    }
}
</style>

<script>
// Animate small circular progress indicator on page load
document.addEventListener('DOMContentLoaded', function() {
    // Animate small circular progress indicator
    const progressCircleSmall = document.querySelector('.progress-circle-small');
    if (progressCircleSmall) {
        const progressValue = parseInt(progressCircleSmall.getAttribute('data-progress')) || 0;
        
        // Set CSS custom property for the conic gradient
        progressCircleSmall.style.setProperty('--progress', progressValue);
        
        // Add smooth animation effect
        let currentProgress = 0;
        const animationDuration = 1200; // 1.2 seconds
        const startTime = performance.now();
        
        function animateProgress(timestamp) {
            const elapsed = timestamp - startTime;
            const progress = Math.min(elapsed / animationDuration, 1);
            
            // Use easing function for smooth animation
            const easeOut = 1 - Math.pow(1 - progress, 3);
            currentProgress = Math.round(progressValue * easeOut);
            
            progressCircleSmall.style.setProperty('--progress', currentProgress);
            
            if (progress < 1) {
                requestAnimationFrame(animateProgress);
            }
        }
        
        // Start animation after a small delay
        setTimeout(() => {
            requestAnimationFrame(animateProgress);
        }, 200);
    }
});
</script>

</body>
</html>
