<?php
/**
 * Course Progress API Endpoint
 * Provides real-time course progress data for students
 */

header('Content-Type: application/json');

// Start session and check authentication
require_once '../../includes/session_start.php';
require_once '../../auth/functions.php';

// Ensure only students can access this
if (!isLoggedIn() || !hasRole('student')) {
    http_response_code(401);
    echo json_encode(['error' => 'Authentication required']);
    exit;
}

$student_id = $_SESSION['user_id'];
if (!$student_id) {
    http_response_code(401);
    echo json_encode(['error' => 'User not authenticated']);
    exit;
}

// Include database and progress service
require_once '../../config/db_connect.php';
require_once '../../includes/course_progress_service.php';

try {
    $progressService = new CourseProgressService($conn);
    
    // Check what action is requested
    $action = $_GET['action'] ?? 'all_courses';
    
    switch ($action) {
        case 'single_course':
            if (!isset($_GET['course_id'])) {
                http_response_code(400);
                echo json_encode(['error' => 'Course ID required']);
                exit;
            }
            
            $course_id = intval($_GET['course_id']);
            $progress = $progressService->calculateCourseProgress($student_id, $course_id);
            
            echo json_encode([
                'success' => true,
                'course_id' => $course_id,
                'progress' => $progress
            ]);
            break;
            
        case 'all_courses':
            $allProgress = $progressService->getAllCoursesProgress($student_id);
            
            echo json_encode([
                'success' => true,
                'student_id' => $student_id,
                'courses_progress' => $allProgress
            ]);
            break;
            
        case 'certificate_eligibility':
            if (!isset($_GET['course_id'])) {
                http_response_code(400);
                echo json_encode(['error' => 'Course ID required']);
                exit;
            }
            
            $course_id = intval($_GET['course_id']);
            $minimum = isset($_GET['minimum']) ? floatval($_GET['minimum']) : 80.0;
            
            $eligibility = $progressService->isCertificateEligible($student_id, $course_id, $minimum);
            
            echo json_encode([
                'success' => true,
                'course_id' => $course_id,
                'eligibility' => $eligibility
            ]);
            break;
            
        case 'progress_summary':
            // Get summary statistics for dashboard
            $allProgress = $progressService->getAllCoursesProgress($student_id);
            
            $summary = [
                'total_courses' => count($allProgress),
                'completed_courses' => 0,
                'in_progress_courses' => 0,
                'not_started_courses' => 0,
                'average_progress' => 0,
                'certificates_eligible' => 0
            ];
            
            $totalProgress = 0;
            foreach ($allProgress as $courseId => $courseData) {
                $progress = $courseData['progress'];
                $totalProgress += $progress['overall_percentage'];
                
                switch ($progress['completion_status']) {
                    case 'completed':
                        $summary['completed_courses']++;
                        break;
                    case 'in_progress':
                    case 'near_completion':
                        $summary['in_progress_courses']++;
                        break;
                    default:
                        $summary['not_started_courses']++;
                        break;
                }
                
                if ($progress['overall_percentage'] >= 80) {
                    $summary['certificates_eligible']++;
                }
            }
            
            $summary['average_progress'] = $summary['total_courses'] > 0 
                ? round($totalProgress / $summary['total_courses'], 2) 
                : 0;
            
            echo json_encode([
                'success' => true,
                'student_id' => $student_id,
                'summary' => $summary,
                'detailed_progress' => $allProgress
            ]);
            break;
            
        default:
            http_response_code(400);
            echo json_encode(['error' => 'Invalid action']);
            break;
    }
    
} catch (Exception $e) {
    error_log("Course Progress API Error: " . $e->getMessage());
    http_response_code(500);
    echo json_encode([
        'error' => 'Internal server error', 
        'message' => $e->getMessage()
    ]);
}
?>
