<?php
/**
 * 🔍 Debug Accreditation Request API
 * Identifies what's causing the JSON parsing error
 */

// Start output buffering to catch any accidental output
ob_start();

require_once '../../auth/functions.php';
requireRole('student');
require_once '../../config/db_connect.php';

header('Content-Type: application/json');

$debug_info = [];
$response = ['success' => false, 'message' => '', 'debug' => []];

try {
    $debug_info[] = "✅ Auth and DB connection successful";
    
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        throw new Exception('Method not allowed - use POST');
    }
    
    $student_id = $_SESSION['user_id'] ?? null;
    $course_id = intval($_POST['course_id'] ?? 0);
    $motivation = trim($_POST['motivation'] ?? '');
    
    $debug_info[] = "✅ Basic parameters: student_id=$student_id, course_id=$course_id";
    
    if (!$course_id) {
        throw new Exception('Invalid course ID');
    }
    
    // Test database access
    $test_query = $conn->query("SHOW TABLES LIKE 'accreditation_requests'");
    if ($test_query->num_rows == 0) {
        throw new Exception('❌ accreditation_requests table does not exist');
    }
    $debug_info[] = "✅ accreditation_requests table exists";
    
    // Check table structure
    $columns_query = $conn->query("DESCRIBE accreditation_requests");
    $columns = [];
    while ($col = $columns_query->fetch_assoc()) {
        $columns[] = $col['Field'];
    }
    $debug_info[] = "✅ Table columns: " . implode(', ', $columns);
    
    // Check required columns
    $required_columns = ['student_id', 'course_id', 'status', 'payment_amount', 'payment_status', 'payment_reference'];
    $missing_columns = array_diff($required_columns, $columns);
    if (!empty($missing_columns)) {
        throw new Exception('❌ Missing columns: ' . implode(', ', $missing_columns));
    }
    $debug_info[] = "✅ All required columns exist";
    
    // Test enrollment check
    $enrollment_check = $conn->prepare("SELECT * FROM enrollments WHERE student_id = ? AND course_id = ?");
    $enrollment_check->bind_param("ii", $student_id, $course_id);
    $enrollment_check->execute();
    $enrollment = $enrollment_check->get_result()->fetch_assoc();
    if (!$enrollment) {
        throw new Exception('❌ Not enrolled in course');
    }
    $debug_info[] = "✅ Student enrolled in course";
    
    // Check existing requests
    $existing_check = $conn->prepare("SELECT * FROM accreditation_requests WHERE student_id = ? AND course_id = ? AND status NOT IN ('rejected', 'certificate_issued', 'cancelled')");
    $existing_check->bind_param("ii", $student_id, $course_id);
    $existing_check->execute();
    $existing_request = $existing_check->get_result()->fetch_assoc();
    
    if ($existing_request) {
        $debug_info[] = "⚠️ Existing request found: ID={$existing_request['request_id']}, Status={$existing_request['status']}";
        throw new Exception('You already have an active accreditation request');
    }
    $debug_info[] = "✅ No existing active requests";
    
    // Test PayFast files existence
    $payfast_file = '../../payments/payfast.php';
    if (!file_exists($payfast_file)) {
        throw new Exception("❌ PayFast file missing: $payfast_file");
    }
    $debug_info[] = "✅ PayFast file exists";
    
    $encryption_file = '../includes/id_encryption.php';
    if (!file_exists($encryption_file)) {
        throw new Exception("❌ Encryption file missing: $encryption_file");
    }
    $debug_info[] = "✅ Encryption file exists";
    
    // Test basic insertion capability
    $test_insert = $conn->prepare("
        INSERT INTO accreditation_requests (
            student_id, course_id, motivation_statement, request_date, status, 
            payment_amount, payment_status, created_at, updated_at
        ) VALUES (?, ?, ?, NOW(), 'pending_payment', 500.00, 'pending', NOW(), NOW())
    ");
    $test_insert->bind_param("iis", $student_id, $course_id, $motivation);
    
    if (!$test_insert->execute()) {
        throw new Exception('❌ Database insert failed: ' . $conn->error);
    }
    
    $request_id = $conn->insert_id;
    $debug_info[] = "✅ Request created successfully with ID: $request_id";
    
    // Clean up test record
    $conn->query("DELETE FROM accreditation_requests WHERE request_id = $request_id");
    $debug_info[] = "✅ Test record cleaned up";
    
    $response['success'] = true;
    $response['message'] = 'Debug completed successfully - API should work fine!';
    
} catch (Exception $e) {
    $response['message'] = $e->getMessage();
    $debug_info[] = "❌ Error: " . $e->getMessage();
}

// Capture any output that shouldn't be there
$unwanted_output = ob_get_clean();
if ($unwanted_output) {
    $debug_info[] = "⚠️ Unwanted output detected: " . substr($unwanted_output, 0, 200);
}

$response['debug'] = $debug_info;
$response['timestamp'] = date('Y-m-d H:i:s');

echo json_encode($response, JSON_PRETTY_PRINT);
?>
