<?php
/**
 * Delete Course Enrollment API Endpoint
 * Handles permanent deletion of course enrollment and progress
 */

header('Content-Type: application/json');

// Start session and check authentication
require_once '../../includes/session_start.php';
require_once '../../auth/functions.php';

// Initialize response
$response = [
    'success' => false,
    'message' => ''
];

try {
    // Ensure only students can access this
    if (!isLoggedIn() || !hasRole('student')) {
        throw new Exception('Authentication required');
    }

    // Get JSON input
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input || !isset($input['course_id'])) {
        throw new Exception('Invalid request data');
    }

    $courseId = intval($input['course_id']);
    $studentId = $_SESSION['user_id'];

    if ($courseId <= 0 || $studentId <= 0) {
        throw new Exception('Invalid course or student ID');
    }

    // Connect to database
    require_once '../../config/db_connect.php';
    
    if (!isset($conn)) {
        throw new Exception('Database connection failed');
    }

    // Check if student is enrolled in this course
    $checkQuery = "SELECT enrollment_id FROM enrollments WHERE student_id = ? AND course_id = ?";
    $checkStmt = $conn->prepare($checkQuery);
    $checkStmt->bind_param("ii", $studentId, $courseId);
    $checkStmt->execute();
    $checkResult = $checkStmt->get_result();

    if ($checkResult->num_rows === 0) {
        throw new Exception('You are not enrolled in this course');
    }

    // Begin transaction
    $conn->begin_transaction();

    try {
        // Delete related data (video progress, quiz attempts, etc.)
        $tables = [
            'video_progress' => 'student_id = ? AND course_id = ?',
            'quiz_attempts' => 'student_id = ? AND course_id = ?',
            'assignment_submissions' => 'student_id = ? AND course_id = ?',
            'enrollments' => 'student_id = ? AND course_id = ?'
        ];

        foreach ($tables as $table => $where) {
            $deleteQuery = "DELETE FROM $table WHERE $where";
            $deleteStmt = $conn->prepare($deleteQuery);
            $deleteStmt->bind_param("ii", $studentId, $courseId);
            $deleteStmt->execute();
        }

        // Commit transaction
        $conn->commit();

        $response['success'] = true;
        $response['message'] = 'Course enrollment and all progress data deleted successfully';

    } catch (Exception $e) {
        // Rollback transaction
        $conn->rollback();
        throw new Exception('Failed to delete course data: ' . $e->getMessage());
    }

} catch (Exception $e) {
    $response['message'] = $e->getMessage();
}

echo json_encode($response);
?>
