<?php
/**
 * Dynamic Search API for Student Portal
 * Searches across courses, teachers, categories, and content
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST');
header('Access-Control-Allow-Headers: Content-Type');

require_once '../../config/db_connect.php';

// Get search parameters
$query = $_GET['q'] ?? $_POST['q'] ?? '';
$category = $_GET['category'] ?? $_POST['category'] ?? '';
$teacher = $_GET['teacher'] ?? $_POST['teacher'] ?? '';
$price_min = $_GET['price_min'] ?? $_POST['price_min'] ?? '';
$price_max = $_GET['price_max'] ?? $_POST['price_max'] ?? '';
$sort = $_GET['sort'] ?? $_POST['sort'] ?? 'relevance';
$limit = intval($_GET['limit'] ?? $_POST['limit'] ?? 20);

try {
    $results = [];
    
    if (empty($query) && empty($category) && empty($teacher)) {
        // Return popular/featured courses if no search query
        $sql = "
            SELECT DISTINCT
                c.course_id,
                c.title,
                c.description,
                c.thumbnail,
                c.price,
                c.created_at,
                cat.category_name,
                CONCAT(u.first_name, ' ', u.last_name) as teacher_name,
                u.profile_picture as teacher_avatar,
                COUNT(DISTINCT e.enrollment_id) as enrollment_count,
                COUNT(DISTINCT v.video_id) as video_count,
                AVG(v.duration) as avg_duration,
                'course' as result_type
            FROM courses c
            LEFT JOIN categories cat ON c.category_id = cat.category_id
            LEFT JOIN users u ON c.teacher_id = u.user_id
            LEFT JOIN enrollments e ON c.course_id = e.course_id
            LEFT JOIN videos v ON c.course_id = v.course_id
            WHERE c.is_published = 1
            GROUP BY c.course_id
            ORDER BY enrollment_count DESC, c.created_at DESC
            LIMIT ?
        ";
        
        $stmt = $pdo->prepare($sql);
        $stmt->execute([$limit]);
        $results['courses'] = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
    } else {
        // Dynamic search with filters
        $searchResults = [];
        
        // Build WHERE clause for filters
        $whereConditions = ["c.is_published = 1"];
        $params = [];
        
        if (!empty($query)) {
            $whereConditions[] = "(
                c.title LIKE ? OR 
                c.description LIKE ? OR 
                cat.category_name LIKE ? OR 
                CONCAT(u.first_name, ' ', u.last_name) LIKE ? OR
                v.title LIKE ? OR
                v.description LIKE ?
            )";
            $searchTerm = "%$query%";
            $params = array_merge($params, [$searchTerm, $searchTerm, $searchTerm, $searchTerm, $searchTerm, $searchTerm]);
        }
        
        if (!empty($category)) {
            $whereConditions[] = "cat.category_name = ?";
            $params[] = $category;
        }
        
        if (!empty($teacher)) {
            $whereConditions[] = "CONCAT(u.first_name, ' ', u.last_name) LIKE ?";
            $params[] = "%$teacher%";
        }
        
        if (!empty($price_min)) {
            $whereConditions[] = "c.price >= ?";
            $params[] = floatval($price_min);
        }
        
        if (!empty($price_max)) {
            $whereConditions[] = "c.price <= ?";
            $params[] = floatval($price_max);
        }
        
        $whereClause = implode(' AND ', $whereConditions);
        
        // Order clause
        $orderClause = "ORDER BY ";
        switch ($sort) {
            case 'price_low':
                $orderClause .= "c.price ASC";
                break;
            case 'price_high':
                $orderClause .= "c.price DESC";
                break;
            case 'newest':
                $orderClause .= "c.created_at DESC";
                break;
            case 'popular':
                $orderClause .= "enrollment_count DESC";
                break;
            case 'rating':
                $orderClause .= "avg_rating DESC";
                break;
            default: // relevance
                if (!empty($query)) {
                    $orderClause .= "
                        CASE 
                            WHEN c.title LIKE ? THEN 1
                            WHEN cat.category_name LIKE ? THEN 2
                            WHEN CONCAT(u.first_name, ' ', u.last_name) LIKE ? THEN 3
                            ELSE 4
                        END,
                        enrollment_count DESC
                    ";
                    $params = array_merge($params, ["%$query%", "%$query%", "%$query%"]);
                } else {
                    $orderClause .= "enrollment_count DESC, c.created_at DESC";
                }
        }
        
        // Main search query
        $sql = "
            SELECT DISTINCT
                c.course_id,
                c.title,
                c.description,
                c.thumbnail,
                c.price,
                c.created_at,
                cat.category_name,
                CONCAT(u.first_name, ' ', u.last_name) as teacher_name,
                u.profile_picture as teacher_avatar,
                u.user_id as teacher_id,
                COUNT(DISTINCT e.enrollment_id) as enrollment_count,
                COUNT(DISTINCT v.video_id) as video_count,
                ROUND(AVG(CASE WHEN v.duration REGEXP '^[0-9]+$' THEN CAST(v.duration AS UNSIGNED) ELSE 0 END), 1) as avg_duration,
                GROUP_CONCAT(DISTINCT v.title ORDER BY v.video_id LIMIT 3) as sample_videos,
                COALESCE(AVG(r.rating), 0) as avg_rating,
                COUNT(DISTINCT r.rating_id) as review_count,
                'course' as result_type
            FROM courses c
            LEFT JOIN categories cat ON c.category_id = cat.category_id
            LEFT JOIN users u ON c.teacher_id = u.user_id
            LEFT JOIN enrollments e ON c.course_id = e.course_id
            LEFT JOIN videos v ON c.course_id = v.course_id
            LEFT JOIN course_ratings r ON c.course_id = r.course_id
            WHERE $whereClause
            GROUP BY c.course_id
            $orderClause
            LIMIT ?
        ";
        
        $params[] = $limit;
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        $searchResults = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Process results for better display
        foreach ($searchResults as &$course) {
            // Format price
            $course['formatted_price'] = $course['price'] == 0 ? 'Free' : '$' . number_format($course['price'], 2);
            
            // Format duration
            $course['formatted_duration'] = $course['avg_duration'] ? round($course['avg_duration']) . ' min avg' : 'N/A';
            
            // Format thumbnail
            if (empty($course['thumbnail']) || !file_exists('../../' . $course['thumbnail'])) {
                $course['thumbnail'] = '../../assets/images/default-course.jpg';
            } else {
                $course['thumbnail'] = '../../' . $course['thumbnail'];
            }
            
            // Format teacher avatar
            if (empty($course['teacher_avatar']) || !file_exists('../../' . $course['teacher_avatar'])) {
                $course['teacher_avatar'] = '../../assets/images/default-avatar.png';
            } else {
                $course['teacher_avatar'] = '../../' . $course['teacher_avatar'];
            }
            
            // Format rating
            $course['formatted_rating'] = $course['avg_rating'] > 0 ? number_format($course['avg_rating'], 1) : 'No ratings';
            
            // Generate course URL
            $course['course_url'] = '../course-detail.php?id=' . $course['course_id'];
            
            // Sample videos as array
            $course['sample_videos'] = $course['sample_videos'] ? explode(',', $course['sample_videos']) : [];
        }
        
        $results['courses'] = $searchResults;
    }
    
    // Get filter options for the UI
    if (empty($query)) {
        // Get categories
        $catStmt = $pdo->prepare("
            SELECT category_name, COUNT(c.course_id) as course_count
            FROM categories cat
            LEFT JOIN courses c ON cat.category_id = c.category_id AND c.is_published = 1
            GROUP BY cat.category_id, cat.category_name
            HAVING course_count > 0
            ORDER BY course_count DESC
        ");
        $catStmt->execute();
        $results['categories'] = $catStmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Get teachers
        $teacherStmt = $pdo->prepare("
            SELECT CONCAT(u.first_name, ' ', u.last_name) as teacher_name, 
                   u.profile_picture, 
                   COUNT(c.course_id) as course_count
            FROM users u
            INNER JOIN courses c ON u.user_id = c.teacher_id AND c.is_published = 1
            WHERE u.user_role = 'teacher'
            GROUP BY u.user_id
            HAVING course_count > 0
            ORDER BY course_count DESC
        ");
        $teacherStmt->execute();
        $results['teachers'] = $teacherStmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Get price ranges
        $priceStmt = $pdo->prepare("
            SELECT MIN(price) as min_price, MAX(price) as max_price
            FROM courses 
            WHERE is_published = 1 AND price > 0
        ");
        $priceStmt->execute();
        $priceRange = $priceStmt->fetch(PDO::FETCH_ASSOC);
        $results['price_range'] = $priceRange;
    }
    
    // Add search metadata
    $results['search_info'] = [
        'query' => $query,
        'total_results' => count($results['courses'] ?? []),
        'search_time' => microtime(true) - $_SERVER['REQUEST_TIME_FLOAT'],
        'filters_applied' => [
            'category' => $category,
            'teacher' => $teacher,
            'price_min' => $price_min,
            'price_max' => $price_max,
            'sort' => $sort
        ]
    ];
    
    echo json_encode([
        'success' => true,
        'data' => $results
    ]);

} catch (Exception $e) {
    echo json_encode([
        'success' => false,
        'error' => 'Search failed: ' . $e->getMessage()
    ]);
}
?>
