<?php
/**
 * API: Get Student Enrolled Courses with Grade Data
 * Returns enrolled courses with calculated grades and progress
 */

header('Content-Type: application/json');
error_reporting(E_ALL);
ini_set('display_errors', 0); // Don't display errors in JSON response

try {
    require_once '../../auth/functions.php';
    requireRole('student');
    require_once '../../config/db_connect.php';
    
    $student_id = $_SESSION['user_id'];
    
    // Get enrolled courses with course details (simplified query first)
    $query = "
        SELECT 
            c.course_id,
            c.title,
            c.thumbnail,
            u.first_name as instructor_first,
            u.last_name as instructor_last,
            e.enrolled_date
        FROM courses c
        JOIN enrollments e ON c.course_id = e.course_id
        JOIN users u ON c.teacher_id = u.user_id
        WHERE e.student_id = ?
        ORDER BY e.enrolled_date DESC
    ";
    
    $stmt = $conn->prepare($query);
    if (!$stmt) {
        throw new Exception("SQL prepare failed: " . $conn->error);
    }
    
    $stmt->bind_param("i", $student_id);
    $stmt->execute();
    $result = $stmt->get_result();
    
    $courses = [];
    
    while ($course = $result->fetch_assoc()) {
        $course_id = $course['course_id'];
        
        // Calculate REAL video completion using FIXED logic (same as certificates)
        $videoQuery = $conn->prepare("
            SELECT 
                COUNT(DISTINCT v.video_id) as total_videos,
                COUNT(DISTINCT uvc.video_id) as completed_videos
            FROM videos v
            LEFT JOIN user_video_completions uvc ON v.video_id = uvc.video_id 
                AND uvc.user_id = ? AND uvc.completion_percentage >= 100
            WHERE v.course_id = ? AND v.is_published = 1
        ");
        
        $total_videos = 0;
        $completed_videos = 0;
        
        if ($videoQuery) {
            $videoQuery->bind_param("ii", $student_id, $course_id);
            $videoQuery->execute();
            $videoResult = $videoQuery->get_result();
            if ($videoRow = $videoResult->fetch_assoc()) {
                $total_videos = (int)$videoRow['total_videos'];
                $completed_videos = (int)$videoRow['completed_videos'];
            }
            $videoQuery->close();
        }
        
        // Calculate REAL quiz performance using FIXED logic
        $quizQuery = $conn->prepare("
            SELECT 
                COUNT(DISTINCT q.quiz_id) as total_quizzes,
                COUNT(DISTINCT CASE WHEN qa.pass_status = 1 THEN qa.quiz_id END) as passed_quizzes,
                c.course_passing_grade
            FROM quizzes q
            LEFT JOIN quiz_attempts qa ON q.quiz_id = qa.quiz_id AND qa.student_id = ?
            LEFT JOIN courses c ON q.course_id = c.course_id
            WHERE q.course_id = ? AND q.is_published = 1
            GROUP BY c.course_passing_grade
        ");
        
        $total_quizzes = 0;
        $passed_quizzes = 0;
        $course_passing_grade = 60;
        
        if ($quizQuery) {
            $quizQuery->bind_param("ii", $student_id, $course_id);
            $quizQuery->execute();
            $quizResult = $quizQuery->get_result();
            if ($quizRow = $quizResult->fetch_assoc()) {
                $total_quizzes = (int)$quizRow['total_quizzes'];
                $passed_quizzes = (int)$quizRow['passed_quizzes'];
                $course_passing_grade = (float)$quizRow['course_passing_grade'] ?: 60;
            }
            $quizQuery->close();
        }
        
        // Calculate overall course progress: 60% videos + 40% quizzes
        $video_progress = $total_videos > 0 ? ($completed_videos / $total_videos) * 100 : 100;
        $quiz_progress = $total_quizzes > 0 ? ($passed_quizzes / $total_quizzes) * 100 : 100;
        $overall_progress = round(($video_progress * 0.6) + ($quiz_progress * 0.4));
        
        // Determine final grade and status
        $final_grade = $overall_progress;
        $letter_grade = 'F';
        $grade_class = 'grade-fail';
        $grade_text = 'Needs Work';
        
        if ($final_grade >= 90) {
            $letter_grade = 'A';
            $grade_class = 'grade-excellent';
            $grade_text = 'Excellent';
        } elseif ($final_grade >= 80) {
            $letter_grade = 'B';
            $grade_class = 'grade-good';
            $grade_text = 'Good';
        } elseif ($final_grade >= 70) {
            $letter_grade = 'C';
            $grade_class = 'grade-average';
            $grade_text = 'Average';
        } elseif ($final_grade >= 60) {
            $letter_grade = 'D';
            $grade_class = 'grade-poor';
            $grade_text = 'Poor';
        }
        
        // Certificate eligibility (100% videos + quiz pass rate >= course pass grade)
        $videos_completed = ($video_progress >= 100);
        $quizzes_passed = ($total_quizzes == 0) ? true : (($passed_quizzes / $total_quizzes * 100) >= $course_passing_grade);
        $certificate_eligible = ($videos_completed && $quizzes_passed);
        
        // Progress status
        $progress_status = 'Not Started';
        if ($overall_progress >= 100) {
            $progress_status = 'Completed';
        } elseif ($overall_progress >= 75) {
            $progress_status = 'Nearly Done';
        } elseif ($overall_progress >= 25) {
            $progress_status = 'In Progress';
        } elseif ($overall_progress > 0) {
            $progress_status = 'Started';
        }
        
        $courses[] = [
            'course_id' => $course['course_id'],
            'title' => $course['title'],
            'thumbnail' => $course['thumbnail'] ? $course['thumbnail'] : null,
            'instructor' => trim($course['instructor_first'] . ' ' . $course['instructor_last']),
            'enrolled_date' => $course['enrolled_date'],
            'final_grade' => $final_grade,
            'letter_grade' => $letter_grade,
            'grade_class' => $grade_class,
            'grade_text' => $grade_text,
            'progress_percentage' => $overall_progress,
            'progress_status' => $progress_status,
            'certificate_eligible' => $certificate_eligible,
            // Additional data for dashboard
            'total_videos' => $total_videos,
            'completed_videos' => $completed_videos,
            'total_quizzes' => $total_quizzes,
            'passed_quizzes' => $passed_quizzes,
            'video_progress' => round($video_progress, 1),
            'quiz_progress' => round($quiz_progress, 1)
        ];
    }
    
    echo json_encode([
        'success' => true,
        'courses' => $courses,
        'total_enrolled' => count($courses)
    ]);
    
} catch (Exception $e) {
    echo json_encode([
        'success' => false,
        'error' => 'Failed to fetch courses: ' . $e->getMessage()
    ]);
}
?>
