<?php
// API endpoint to send a message

// Apply advanced session fix first - before ANY output
require_once '../../includes/advanced_session.php';

// Include authentication functions
require_once '../../auth/functions.php';

// Log session for debugging
error_log('API Session with advanced fix: ' . print_r($_SESSION, true));

// Debugging session info
$debug_session = ["user_id" => $_SESSION['user_id'] ?? null, "user_role" => $_SESSION['user_role'] ?? null];

// Direct check for API authentication - don't use requireRole which would redirect
if (!isset($_SESSION['user_id']) || !isset($_SESSION['user_role']) || $_SESSION['user_role'] !== 'student') {
    // Return JSON response for API
    echo json_encode([
        'status' => 'error',
        'message' => 'Unauthorized access. Please log in as a student.',
        'debug_session' => $debug_session
    ]);
    exit;
}

// Initialize response array
$response = [
    'status' => 'error',
    'message' => '',
    'data' => null
];

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Validate required fields
    $requiredFields = ['recipient_id', 'message'];
    $missingFields = [];
    
    foreach ($requiredFields as $field) {
        if (!isset($_POST[$field]) || empty(trim($_POST[$field]))) {
            $missingFields[] = $field;
        }
    }
    
    if (!empty($missingFields)) {
        $response['message'] = 'Missing required fields: ' . implode(', ', $missingFields);
    } else {
        // Get and sanitize inputs
        $senderId = $_SESSION['user_id'];
        $recipientId = intval($_POST['recipient_id']);
        $message = trim($_POST['message']);
        $courseId = isset($_POST['course_id']) ? intval($_POST['course_id']) : 0;
        
        try {
            // Connect to database
            require_once '../../config/db_connect.php';
            
            if ($conn) {
                // First check if the recipient is a teacher/facilitator
                $isTeacherQuery = "SELECT 1 FROM users u 
                                 WHERE u.user_id = ? AND u.user_role = 'teacher'";
                $stmt = $conn->prepare($isTeacherQuery);
                $stmt->bind_param("i", $recipientId);
                $stmt->execute();
                $teacherResult = $stmt->get_result();
                
                if ($teacherResult->num_rows > 0) {
                    // Teacher found, allow messaging
                    // Insert the message using the correct column names from the messages table
                    $insertQuery = "INSERT INTO messages (sender_id, recipient_id, message_text, is_read, sent_at) 
                                   VALUES (?, ?, ?, 0, NOW())";
                    $stmt = $conn->prepare($insertQuery);
                    $stmt->bind_param("iis", $senderId, $recipientId, $message);
                    
                    if ($stmt->execute()) {
                        $messageId = $conn->insert_id;
                        
                        // Get the inserted message with sender details
                        $messageQuery = "SELECT 
                                        m.message_id,
                                        m.sender_id,
                                        u.first_name as sender_first_name,
                                        u.last_name as sender_last_name,
                                        u.profile_picture as sender_profile_picture,
                                        m.recipient_id,
                                        m.message_text,
                                        m.is_read,
                                        m.sent_at
                                      FROM messages m
                                      JOIN users u ON m.sender_id = u.user_id
                                      WHERE m.message_id = ?";
                        $stmt = $conn->prepare($messageQuery);
                        $stmt->bind_param("i", $messageId);
                        $stmt->execute();
                        $msgResult = $stmt->get_result();
                        
                        if ($msgData = $msgResult->fetch_assoc()) {
                            // Already have sent_at from query, no need to rename
                            $response['status'] = 'success';
                            $response['message'] = 'Message sent successfully';
                            $response['data'] = $msgData;
                        } else {
                            $response['status'] = 'success';
                            $response['message'] = 'Message sent but could not fetch details';
                        }
                    } else {
                        $response['status'] = 'error';
                        $response['message'] = 'Could not send message';
                    }
                }
            }
        } catch (Exception $e) {
            $response['message'] = 'Error: ' . $e->getMessage();
        }
    }
}

// Return JSON response
header('Content-Type: application/json');
echo json_encode($response);
exit;
?>
