<?php
require_once '../includes/session_start.php';
require_once '../auth/functions.php';

// Set the missing session variables that requireRole needs
if (!isset($_SESSION['logged_in']) && isset($_SESSION['user_id'])) {
    $_SESSION['logged_in'] = true;
}
if (!isset($_SESSION['user_role']) && isset($_SESSION['role'])) {
    $_SESSION['user_role'] = $_SESSION['role'];
}

requireRole('student');
require_once '../config/db_connect.php';

$student_id = $_SESSION['user_id'];
$request_id = isset($_GET['request_id']) ? intval($_GET['request_id']) : 0;

if (!$request_id) {
    header('Location: assignments.php');
    exit();
}

// Verify that this accreditation request belongs to the current student - Allow both assignment_created and documents_submitted
$verify_query = "SELECT ar.*, c.title as course_title 
                FROM accreditation_requests ar 
                JOIN courses c ON ar.course_id = c.course_id 
                WHERE ar.request_id = ? AND ar.student_id = ? AND ar.status IN ('assignment_created', 'documents_submitted')";

$verify_stmt = $conn->prepare($verify_query);
$verify_stmt->bind_param("ii", $request_id, $student_id);
$verify_stmt->execute();
$verify_result = $verify_stmt->get_result();

if ($verify_result->num_rows === 0) {
    header('Location: assignments.php');
    exit();
}

$request_data = $verify_result->fetch_assoc();

// Get already submitted documents
$docs_query = "SELECT * FROM accreditation_documents WHERE request_id = ? ORDER BY document_type";
$docs_stmt = $conn->prepare($docs_query);
$docs_stmt->bind_param("i", $request_id);
$docs_stmt->execute();
$submitted_docs = $docs_stmt->get_result()->fetch_all(MYSQLI_ASSOC);

// Set current page for navigation
$currentPage = 'assignments';

// Include modern header with sidebar and navbar
include_once __DIR__ . '/components/header.php';
?>

<!-- SITA Assignment Submission Content -->
<div class="sita-assignment-content">
    <style>
        :root {
            --panadite-primary: #2563eb;
            --panadite-secondary: #1d4ed8;
            --panadite-light: #dbeafe;
            --panadite-dark: #1e40af;
            --panadite-accent: #f59e0b;
        }
        
        .document-card { 
            border: 1px solid #dee2e6; 
            border-radius: 8px; 
            margin-bottom: 20px; 
            background: white;
            box-shadow: 0 1px 3px rgba(0,0,0,0.1);
        }
        .document-card.uploaded { 
            border-color: #28a745; 
            background-color: #f8fff9; 
        }
        .document-card.pending { 
            border-color: #ffc107; 
            background-color: #fffbf0; 
        }
        .upload-area { 
            border: 2px dashed #ced4da; 
            border-radius: 8px; 
            padding: 40px; 
            text-align: center; 
            transition: all 0.3s;
            cursor: pointer;
        }
        .upload-area:hover { 
            border-color: var(--panadite-primary); 
            background-color: var(--panadite-light); 
        }
        .upload-area.dragover { 
            border-color: var(--panadite-primary); 
            background-color: #e3f2fd; 
        }
        .file-info { 
            font-size: 0.9rem; 
            color: #6c757d; 
        }
        .progress-indicator { 
            height: 6px; 
            background: #e9ecef; 
            border-radius: 3px; 
            overflow: hidden; 
        }
        .progress-indicator .progress-bar { 
            height: 100%; 
            background: linear-gradient(90deg, var(--panadite-primary), var(--panadite-secondary)); 
            border-radius: 3px; 
        }
        .assignment-header {
            background: linear-gradient(135deg, var(--panadite-primary) 0%, var(--panadite-secondary) 100%);
            color: white;
            border-radius: 12px;
            margin-bottom: 24px;
        }
    </style>
</head>
<body>
    <?php include_once 'components/sidebar.php'; ?>
    
    <div class="page-header">
        <div class="header-content">
            <h1 class="page-title">
                <i class="bx bx-award"></i>
                SITA Assignment Submission
            </h1>
            <p class="page-subtitle">Submit required documents for your accreditation assessment</p>
        </div>
    </div>

    <div class="container">
        <!-- Assignment Header -->
        <div class="assignment-header p-4 rounded mb-4">
            <div class="row align-items-center">
                <div class="col-md-8">
                    <h2><i class="bx bx-award me-2"></i>SITA Assignment Submission</h2>
                    <p class="mb-0">Course: <strong><?= htmlspecialchars($request_data['course_title']) ?></strong></p>
                    <small>Request ID: #<?= str_pad($request_id, 6, '0', STR_PAD_LEFT) ?></small>
                </div>
                <div class="col-md-4 text-md-end">
                    <div class="badge bg-warning fs-6 px-3 py-2">
                        <i class="bx bx-time"></i> Assignment Phase
                    </div>
                </div>
            </div>
        </div>

        <div class="row">
            <div class="col-md-8">
                <!-- Assignment Instructions -->
                <div class="card mb-4">
                    <div class="card-header bg-primary text-white">
                        <h5 class="mb-0"><i class="bx bx-info-circle me-2"></i>Assignment Instructions</h5>
                    </div>
                    <div class="card-body">
                        <?php if ($request_data['assignment_instructions']): ?>
                            <div class="assignment-content">
                                <?= nl2br(htmlspecialchars($request_data['assignment_instructions'])) ?>
                            </div>
                        <?php else: ?>
                            <p class="text-muted">Please complete the following document submissions for your SITA accreditation assessment.</p>
                        <?php endif; ?>
                        
                        <div class="alert alert-info mt-3">
                            <h6><i class="bx bx-lightbulb me-1"></i> Important Notes:</h6>
                            <ul class="mb-0">
                                <li>Submit all required documents to proceed with your accreditation</li>
                                <li>Ensure all files are clear, legible, and in accepted formats</li>
                                <li>Your submission will be reviewed by our SITA assessors</li>
                                <li>You'll receive feedback within 5-10 business days</li>
                            </ul>
                        </div>
                    </div>
                </div>

                <!-- Document Submission Form -->
                <div class="card">
                    <div class="card-header">
                        <h5 class="mb-0"><i class="bx bx-upload me-2"></i>Document Submissions</h5>
                    </div>
                    <div class="card-body">
                        <form id="documentSubmissionForm" enctype="multipart/form-data">
                            <input type="hidden" name="request_id" value="<?= $request_id ?>">
                            
                            <?php
                            $required_docs = [
                                'cv' => [
                                    'name' => 'Curriculum Vitae (CV)', 
                                    'icon' => 'bx-user',
                                    'description' => 'Your updated professional CV/Resume',
                                    'formats' => ['PDF', 'DOC', 'DOCX'],
                                    'required' => true
                                ],
                                'qualifications' => [
                                    'name' => 'Educational Qualifications', 
                                    'icon' => 'bx-certification',
                                    'description' => 'Certificates, diplomas, degrees',
                                    'formats' => ['PDF', 'JPG', 'PNG'],
                                    'required' => true
                                ],
                                'id_document' => [
                                    'name' => 'Government ID Document', 
                                    'icon' => 'bx-id-card',
                                    'description' => 'National ID, passport, or driver\'s license',
                                    'formats' => ['PDF', 'JPG', 'PNG'],
                                    'required' => true
                                ],
                                'assignment_work' => [
                                    'name' => 'Assignment Submission', 
                                    'icon' => 'bx-file-blank',
                                    'description' => 'Your completed SITA assignment work',
                                    'formats' => ['PDF', 'DOC', 'DOCX', 'ZIP'],
                                    'required' => true
                                ]
                            ];

                            foreach ($required_docs as $doc_type => $doc_info):
                                $uploaded_doc = null;
                                // Map UI document types to database types for matching
                                $db_doc_type = ($doc_type === 'assignment_work') ? 'assignment' : $doc_type;
                                
                                foreach ($submitted_docs as $doc) {
                                    if ($doc['document_type'] === $db_doc_type) {
                                        $uploaded_doc = $doc;
                                        break;
                                    }
                                }
                                $is_uploaded = !empty($uploaded_doc);
                            ?>
                            
                            <div class="document-card <?= $is_uploaded ? 'uploaded' : 'pending' ?>">
                                <div class="card-body">
                                    <div class="row align-items-center">
                                        <div class="col-md-6">
                                            <div class="d-flex align-items-center">
                                                <i class="bx <?= $doc_info['icon'] ?> fs-2 me-3 <?= $is_uploaded ? 'text-success' : 'text-warning' ?>"></i>
                                                <div>
                                                    <h6 class="mb-1">
                                                        <?= $doc_info['name'] ?>
                                                        <?php if ($doc_info['required']): ?>
                                                            <span class="text-danger">*</span>
                                                        <?php endif; ?>
                                                    </h6>
                                                    <small class="text-muted"><?= $doc_info['description'] ?></small>
                                                    <br><small class="file-info">Accepts: <?= implode(', ', $doc_info['formats']) ?> | Max: 10MB</small>
                                                </div>
                                            </div>
                                        </div>
                                        <div class="col-md-6">
                                            <?php if ($is_uploaded): ?>
                                                <div class="d-flex align-items-center justify-content-between">
                                                    <div>
                                                        <i class="bx bx-check-circle text-success me-1"></i>
                                                        <span class="text-success fw-bold">Uploaded</span>
                                                        <br><small><?= htmlspecialchars($uploaded_doc['document_name']) ?></small>
                                                        <br><small class="text-muted"><?= date('M d, Y', strtotime($uploaded_doc['upload_date'])) ?></small>
                                                    </div>
                                                    <button type="button" class="btn btn-outline-primary btn-sm" onclick="replaceDocument('<?= $doc_type ?>')">
                                                        <i class="bx bx-refresh"></i> Replace
                                                    </button>
                                                </div>
                                            <?php else: ?>
                                                <div class="upload-area" onclick="triggerFileInput('<?= $doc_type ?>')">
                                                    <i class="bx bx-cloud-upload fs-1 text-muted mb-2"></i>
                                                    <p class="mb-1">Click to upload or drag & drop</p>
                                                    <small class="text-muted">Max file size: 10MB</small>
                                                    <input type="file" 
                                                           id="file_<?= $doc_type ?>" 
                                                           name="documents[<?= $doc_type ?>]"
                                                           accept=".<?= strtolower(implode(',.',$doc_info['formats'])) ?>"
                                                           style="display: none;"
                                                           onchange="handleFileSelect(this, '<?= $doc_type ?>')">
                                                </div>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            
                            <?php endforeach; ?>

                            <div class="d-grid gap-2 d-md-flex justify-content-md-end mt-4">
                                <a href="assignments.php" class="btn btn-outline-secondary me-md-2">
                                    <i class="bx bx-arrow-back"></i> Back to Assignments
                                </a>
                                <button type="submit" class="btn btn-success btn-lg">
                                    <i class="bx bx-send"></i> Submit All Documents
                                </button>
                            </div>
                        </form>
                    </div>
                </div>
            </div>

            <div class="col-md-4">
                <!-- Progress Summary -->
                <div class="card mb-4">
                    <div class="card-header">
                        <h6 class="mb-0"><i class="bx bx-chart-line me-2"></i>Submission Progress</h6>
                    </div>
                    <div class="card-body">
                        <?php
                        $total_docs = count($required_docs);
                        $uploaded_count = count($submitted_docs);
                        $progress = $total_docs > 0 ? ($uploaded_count / $total_docs) * 100 : 0;
                        ?>
                        <div class="text-center mb-3">
                            <div class="h2 mb-1"><?= $uploaded_count ?>/<?= $total_docs ?></div>
                            <small class="text-muted">Documents Submitted</small>
                        </div>
                        <div class="progress-indicator mb-2">
                            <div class="progress-bar" style="width: <?= $progress ?>%"></div>
                        </div>
                        <small class="text-muted"><?= round($progress) ?>% Complete</small>
                        
                        <?php if ($uploaded_count === $total_docs): ?>
                            <div class="alert alert-success mt-3">
                                <i class="bx bx-check-circle me-1"></i>
                                All documents submitted! Ready for assessment.
                            </div>
                        <?php endif; ?>
                    </div>
                </div>

                <!-- Help & Support -->
                <div class="card">
                    <div class="card-header">
                        <h6 class="mb-0"><i class="bx bx-help-circle me-2"></i>Need Help?</h6>
                    </div>
                    <div class="card-body">
                        <p class="small text-muted mb-3">Having trouble with your submission?</p>
                        <div class="d-grid gap-2">
                            <a href="../contact.php" class="btn btn-outline-primary btn-sm">
                                <i class="bx bx-envelope"></i> Contact Support
                            </a>
                            <a href="assignments.php" class="btn btn-outline-secondary btn-sm">
                                <i class="bx bx-download"></i> Re-download Assignment
                            </a>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        function triggerFileInput(docType) {
            document.getElementById('file_' + docType).click();
        }

        function handleFileSelect(input, docType) {
            if (input.files && input.files[0]) {
                const file = input.files[0];
                const fileSize = file.size / 1024 / 1024; // MB
                
                if (fileSize > 10) {
                    alert('File size must be less than 10MB');
                    input.value = '';
                    return;
                }
                
                // Update UI to show selected file WITHOUT removing the input
                const uploadArea = input.parentElement;
                
                // Hide the original upload area content but keep the input
                const originalContent = uploadArea.querySelector('.upload-content');
                if (originalContent) {
                    originalContent.style.display = 'none';
                } else {
                    // Create a wrapper for original content if it doesn't exist
                    const children = Array.from(uploadArea.children);
                    const contentWrapper = document.createElement('div');
                    contentWrapper.className = 'upload-content';
                    children.forEach(child => {
                        if (child.tagName !== 'INPUT') {
                            contentWrapper.appendChild(child);
                        }
                    });
                    uploadArea.insertBefore(contentWrapper, input);
                    contentWrapper.style.display = 'none';
                }
                
                // Add file display without destroying the input
                let fileDisplay = uploadArea.querySelector('.file-display');
                if (!fileDisplay) {
                    fileDisplay = document.createElement('div');
                    fileDisplay.className = 'file-display text-center';
                    uploadArea.insertBefore(fileDisplay, input);
                }
                
                fileDisplay.innerHTML = `
                    <i class="bx bx-file fs-1 text-primary mb-2"></i>
                    <p class="mb-1 text-primary fw-bold">${file.name}</p>
                    <small class="text-muted">${(fileSize).toFixed(2)} MB</small>
                    <br><button type="button" class="btn btn-sm btn-outline-danger mt-2" onclick="clearFileSelection('${docType}')">
                        <i class="bx bx-x"></i> Remove
                    </button>
                `;
                fileDisplay.style.display = 'block';
            }
        }

        function clearFileSelection(docType) {
            const input = document.getElementById('file_' + docType);
            const uploadArea = input.parentElement;
            
            // Clear the file input
            input.value = '';
            
            // Hide file display and show original upload content
            const fileDisplay = uploadArea.querySelector('.file-display');
            const originalContent = uploadArea.querySelector('.upload-content');
            
            if (fileDisplay) {
                fileDisplay.style.display = 'none';
            }
            if (originalContent) {
                originalContent.style.display = 'block';
            }
        }

        function replaceDocument(docType) {
            triggerFileInput(docType);
        }

        // Form submission
        document.getElementById('documentSubmissionForm').addEventListener('submit', function(e) {
            e.preventDefault();
            
            const formData = new FormData(this);
            const submitBtn = e.target.querySelector('button[type="submit"]');
            
            submitBtn.disabled = true;
            submitBtn.innerHTML = '<i class="bx bx-loader-alt bx-spin"></i> Submitting...';
            
            fetch('api/submit_assignment_documents.php', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    // Show success notification on same page without redirect
                    const successHtml = `
                        <div class="alert alert-success alert-dismissible fade show" role="alert" style="position: fixed; top: 20px; right: 20px; z-index: 9999; min-width: 400px;">
                            <div class="d-flex align-items-center">
                                <i class="bx bx-check-circle me-2 fs-4"></i>
                                <div>
                                    <strong>🎉 Documents Submitted Successfully!</strong><br>
                                    <small>${data.message || 'Your SITA accreditation documents are under review (3-5 months processing time). Email notification sent!'}</small>
                                </div>
                            </div>
                            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                        </div>
                    `;
                    document.body.insertAdjacentHTML('beforeend', successHtml);
                    
                    // Auto-hide notification after 8 seconds
                    setTimeout(() => {
                        const alert = document.querySelector('.alert-success');
                        if (alert) {
                            alert.remove();
                        }
                    }, 8000);
                    
                    // Reset form if all documents submitted
                    if (data.total_submitted >= 4) {
                        document.getElementById('documentSubmissionForm').reset();
                        // Hide all file displays
                        document.querySelectorAll('.file-display').forEach(display => {
                            display.style.display = 'none';
                        });
                        document.querySelectorAll('.upload-content').forEach(content => {
                            content.style.display = 'block';
                        });
                    }
                } else {
                    // Show error notification on same page
                    const errorHtml = `
                        <div class="alert alert-danger alert-dismissible fade show" role="alert" style="position: fixed; top: 20px; right: 20px; z-index: 9999; min-width: 400px;">
                            <div class="d-flex align-items-center">
                                <i class="bx bx-error-circle me-2 fs-4"></i>
                                <div>
                                    <strong>❌ Submission Failed</strong><br>
                                    <small>${data.message || 'Failed to submit documents'}</small>
                                </div>
                            </div>
                            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                        </div>
                    `;
                    document.body.insertAdjacentHTML('beforeend', errorHtml);
                }
            })
            .catch(error => {
                console.error('Error:', error);
                alert('An error occurred while submitting documents');
            })
            .finally(() => {
                submitBtn.disabled = false;
                submitBtn.innerHTML = '<i class="bx bx-upload"></i> Submit Documents';
            });
        });

        // Drag and drop functionality
        document.querySelectorAll('.upload-area').forEach(area => {
            area.addEventListener('dragover', function(e) {
                e.preventDefault();
                this.classList.add('dragover');
            });
            
            area.addEventListener('dragleave', function(e) {
                e.preventDefault();
                this.classList.remove('dragover');
            });
            
            area.addEventListener('drop', function(e) {
                e.preventDefault();
                this.classList.remove('dragover');
                
                const files = e.dataTransfer.files;
                if (files.length > 0) {
                    const docType = this.querySelector('input[type="file"]').id.replace('file_', '');
                    const input = document.getElementById('file_' + docType);
                    input.files = files;
                    handleFileSelect(input, docType);
                }
            });
        });
    </script>
</body>
</html>
