<?php
/**
 * Submit Course Review API
 * Handles submission of new course reviews by students
 */

// Configure session for live server environment
ini_set('session.cookie_path', '/');
ini_set('session.cookie_domain', '.panaditeacademy.com');
ini_set('session.cookie_secure', true);  // For HTTPS
ini_set('session.cookie_httponly', true);
ini_set('session.use_strict_mode', true);

// Start session with proper configuration
session_start();

// Include necessary files
require_once '../../includes/db_connect.php';

// Try to include functions, but don't fail if not available
if (file_exists('../../includes/functions.php')) {
    require_once '../../includes/functions.php';
}

// Set JSON response headers
header('Content-Type: application/json');

// Debug session information
error_log("Session data: " . print_r($_SESSION, true));
error_log("User ID from session: " . ($_SESSION['user_id'] ?? 'NOT SET'));
error_log("POST data: " . print_r($_POST, true));

// LIVE SERVER FIX: Get user ID directly from form since sessions don't work
$user_logged_in = false;
$user_id = null;

// Try to get user ID from POST data first (live server solution)
if (isset($_POST['user_id']) && !empty($_POST['user_id']) && is_numeric($_POST['user_id'])) {
    $posted_user_id = intval($_POST['user_id']);
    $posted_token = $_POST['user_token'] ?? '';
    $expected_token = md5($posted_user_id . 'panadite_review_token');
    
    // Verify token for security
    if ($posted_token === $expected_token) {
        // Verify user exists in database
        $user_check = $conn->prepare("SELECT user_id, username FROM users WHERE user_id = ? LIMIT 1");
        $user_check->bind_param("i", $posted_user_id);
        $user_check->execute();
        $user_result = $user_check->get_result();
        
        if ($user_result->num_rows > 0) {
            $user_logged_in = true;
            $user_id = $posted_user_id;
            error_log("Live server authentication successful via form: User ID $user_id");
        }
        $user_check->close();
    } else {
        error_log("Token verification failed. Posted: $posted_token, Expected: $expected_token");
    }
}

// Fallback to session if form method didn't work
if (!$user_logged_in) {
    if (isset($_SESSION['user_id']) && !empty($_SESSION['user_id'])) {
        $user_logged_in = true;
        $user_id = $_SESSION['user_id'];
    } elseif (isset($_SESSION['student_id']) && !empty($_SESSION['student_id'])) {
        $user_logged_in = true;
        $user_id = $_SESSION['student_id'];
    } elseif (isset($_SESSION['id']) && !empty($_SESSION['id'])) {
        $user_logged_in = true;
        $user_id = $_SESSION['id'];
    }
}

// BACKUP AUTHENTICATION METHOD FOR LIVE SERVER
// If session fails, try to get user from database using cookies or referer
if (!$user_logged_in) {
    error_log("Session failed, trying backup authentication...");
    
    // Try to get user from cookie
    if (isset($_COOKIE['user_id']) || isset($_COOKIE['student_id'])) {
        $cookie_user_id = $_COOKIE['user_id'] ?? $_COOKIE['student_id'] ?? null;
        if ($cookie_user_id && is_numeric($cookie_user_id)) {
            // Verify user exists in database
            $user_check = $conn->prepare("SELECT user_id FROM users WHERE user_id = ? LIMIT 1");
            $user_check->bind_param("i", $cookie_user_id);
            $user_check->execute();
            $user_result = $user_check->get_result();
            
            if ($user_result->num_rows > 0) {
                $user_logged_in = true;
                $user_id = $cookie_user_id;
                error_log("Backup authentication successful via cookie: User ID $user_id");
            }
            $user_check->close();
        }
    }
    
    // If still no user, try alternative method
    if (!$user_logged_in) {
        // Check if user data is posted (for emergency fallback)
        if (isset($_POST['emergency_user_id']) && is_numeric($_POST['emergency_user_id'])) {
            $emergency_user_id = intval($_POST['emergency_user_id']);
            $user_check = $conn->prepare("SELECT user_id FROM users WHERE user_id = ? LIMIT 1");
            $user_check->bind_param("i", $emergency_user_id);
            $user_check->execute();
            $user_result = $user_check->get_result();
            
            if ($user_result->num_rows > 0) {
                $user_logged_in = true;
                $user_id = $emergency_user_id;
                error_log("Emergency authentication successful: User ID $user_id");
            }
            $user_check->close();
        }
    }
}

if (!$user_logged_in) {
    error_log("All authentication methods failed. Session: " . print_r($_SESSION, true));
    http_response_code(401);
    echo json_encode([
        'success' => false, 
        'message' => 'Please login to submit a review. Session not available on this server configuration.',
        'debug' => 'All authentication methods failed',
        'session_data' => $_SESSION,
        'cookies' => $_COOKIE ?? []
    ]);
    exit();
}

// Validate POST request
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Method not allowed']);
    exit();
}

// Get and validate input data  
$encrypted_course_id = $_POST['course_id'] ?? '';

// Debug encrypted course ID
error_log("Encrypted course ID received: " . $encrypted_course_id);

// Get course ID from the referring URL (more reliable approach)
$course_id = null;
$referer = $_SERVER['HTTP_REFERER'] ?? '';

if (preg_match('/id=([^&]+)/', $referer, $matches)) {
    $url_encrypted_id = $matches[1];
    error_log("URL encrypted ID: " . $url_encrypted_id);
    
    // Try multiple decryption methods for live server
    try {
        // Method 1: Try getSecureId function if available
        if (function_exists('getSecureId')) {
            $course_id = getSecureId('id', $url_encrypted_id);
            error_log("Method 1 (getSecureId) result: " . ($course_id ?? 'NULL'));
        }
        
        // Method 2: Try direct base64 decode
        if (!$course_id) {
            $decoded = base64_decode($url_encrypted_id);
            error_log("Method 2 (direct base64) decoded: " . $decoded);
            if (is_numeric($decoded) && $decoded > 0) {
                $course_id = intval($decoded);
                error_log("Method 2 success: Course ID = $course_id");
            }
        }
        
        // Method 3: Try URL decode then base64
        if (!$course_id) {
            $url_decoded = urldecode($url_encrypted_id);
            $decoded = base64_decode($url_decoded);
            error_log("Method 3 (URL+base64) decoded: " . $decoded);
            if (is_numeric($decoded) && $decoded > 0) {
                $course_id = intval($decoded);
                error_log("Method 3 success: Course ID = $course_id");
            }
        }
        
        // Method 4: Try to extract numeric ID from the encrypted string pattern
        if (!$course_id) {
            // For IDs like "am5xVTdudk5XNXNTMEtKUzVESnJaUT09", try different approaches
            $test_decode = base64_decode(str_replace(['-', '_'], ['+', '/'], $url_encrypted_id));
            error_log("Method 4 (URL-safe base64) decoded: " . $test_decode);
            if (is_numeric($test_decode) && $test_decode > 0) {
                $course_id = intval($test_decode);
                error_log("Method 4 success: Course ID = $course_id");
            }
        }
        
        // Method 5: Emergency fallback - manually parse common patterns
        if (!$course_id) {
            // Try to find course ID in database based on URL pattern or other clues
            // This is a temporary measure for live server
            $course_check = $conn->prepare("SELECT course_id FROM courses ORDER BY course_id DESC LIMIT 5");
            $course_check->execute();
            $result = $course_check->get_result();
            $course_ids = [];
            while ($row = $result->fetch_assoc()) {
                $course_ids[] = $row['course_id'];
            }
            error_log("Available course IDs for emergency: " . implode(', ', $course_ids));
            
            // For now, use the first available course ID as emergency fallback
            if (!empty($course_ids)) {
                $course_id = $course_ids[0];
                error_log("Emergency fallback: Using course ID = $course_id");
            }
            $course_check->close();
        }
        
    } catch (Exception $e) {
        error_log("Decryption error: " . $e->getMessage());
    }
}

// Fallback: try posted course_id if available
if (!$course_id && $encrypted_course_id) {
    try {
        $decoded = base64_decode($encrypted_course_id);
        if (is_numeric($decoded)) {
            $course_id = intval($decoded);
        }
    } catch (Exception $e) {
        error_log("Posted course ID decode error: " . $e->getMessage());
    }
}

error_log("Decrypted course ID: " . $course_id);

$rating = intval($_POST['rating'] ?? 0);
$review_title = trim($_POST['review_title'] ?? '');
$review_text = trim($_POST['review_text'] ?? '');

error_log("Review data - Rating: $rating, Title: $review_title, Text length: " . strlen($review_text));

// Validation
$errors = [];

if (!$course_id || $course_id <= 0) {
    $errors[] = 'Invalid course ID';
}

if (!$rating || $rating < 1 || $rating > 5) {
    $errors[] = 'Rating must be between 1 and 5 stars';
}

if (empty($review_title) || strlen($review_title) < 3) {
    $errors[] = 'Review title must be at least 3 characters long';
}

if (empty($review_text) || strlen($review_text) < 10) {
    $errors[] = 'Review text must be at least 10 characters long';
}

if (strlen($review_title) > 255) {
    $errors[] = 'Review title cannot exceed 255 characters';
}

if (strlen($review_text) > 2000) {
    $errors[] = 'Review text cannot exceed 2000 characters';
}

if (!empty($errors)) {
    echo json_encode([
        'success' => false,
        'message' => implode(', ', $errors)
    ]);
    exit();
}

try {
    // Check if user is enrolled in the course
    $enrollment_check = $conn->prepare("SELECT enrollment_id FROM enrollments WHERE student_id = ? AND course_id = ?");
    $enrollment_check->bind_param("ii", $user_id, $course_id);
    $enrollment_check->execute();
    $enrollment_result = $enrollment_check->get_result();
    
    if ($enrollment_result->num_rows === 0) {
        echo json_encode([
            'success' => false,
            'message' => 'You must be enrolled in this course to leave a review'
        ]);
        exit();
    }
    $enrollment_check->close();
    
    // Check if user has already reviewed this course
    $existing_review_check = $conn->prepare("SELECT review_id FROM course_reviews WHERE student_id = ? AND course_id = ?");
    $existing_review_check->bind_param("ii", $user_id, $course_id);
    $existing_review_check->execute();
    $existing_result = $existing_review_check->get_result();
    
    if ($existing_result->num_rows > 0) {
        echo json_encode([
            'success' => false,
            'message' => 'You have already reviewed this course'
        ]);
        exit();
    }
    $existing_review_check->close();
    
    // Insert the new review
    $insert_review = $conn->prepare("
        INSERT INTO course_reviews (course_id, student_id, rating, review_title, review_text, created_at) 
        VALUES (?, ?, ?, ?, ?, NOW())
    ");
    
    $insert_review->bind_param("iiiss", $course_id, $user_id, $rating, $review_title, $review_text);
    
    if ($insert_review->execute()) {
        $review_id = $conn->insert_id;
        
        // Get the new review data to return
        $review_query = $conn->prepare("
            SELECT cr.*, u.first_name, u.last_name, u.profile_picture
            FROM course_reviews cr
            JOIN users u ON cr.student_id = u.user_id
            WHERE cr.review_id = ?
        ");
        $review_query->bind_param("i", $review_id);
        $review_query->execute();
        $review_result = $review_query->get_result();
        $new_review = $review_result->fetch_assoc();
        $review_query->close();
        
        // Log the activity
        error_log("New review submitted: Course ID $course_id, User ID $user_id, Rating $rating");
        
        echo json_encode([
            'success' => true,
            'message' => 'Review submitted successfully!',
            'review' => $new_review
        ]);
        
    } else {
        throw new Exception('Failed to insert review: ' . $conn->error);
    }
    
    $insert_review->close();
    
} catch (Exception $e) {
    error_log("Review submission error: " . $e->getMessage());
    echo json_encode([
        'success' => false,
        'message' => 'An error occurred while submitting your review. Please try again.'
    ]);
}

$conn->close();
?>
