<?php
/**
 * Toggle Helpful Vote API
 * Handles helpful/unhelpful votes on course reviews
 */

// Configure session for live server environment
ini_set('session.cookie_path', '/');
ini_set('session.cookie_domain', '.panaditeacademy.com');
ini_set('session.cookie_secure', true);  // For HTTPS
ini_set('session.cookie_httponly', true);
ini_set('session.use_strict_mode', true);

session_start();
require_once '../../includes/db_connect.php';

// Try to include functions, but don't fail if not available
if (file_exists('../../includes/functions.php')) {
    require_once '../../includes/functions.php';
}

// Set JSON response headers
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST');
header('Access-Control-Allow-Headers: Content-Type');

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    http_response_code(401);
    echo json_encode(['success' => false, 'message' => 'Please login to vote']);
    exit();
}

// Validate POST request
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Method not allowed']);
    exit();
}

// Get and validate input data
$user_id = $_SESSION['user_id'];
$encrypted_review_id = $_POST['review_id'] ?? '';
$review_id = getSecureId('review_id', $encrypted_review_id); // Decrypt the review ID

if (!$review_id) {
    echo json_encode([
        'success' => false,
        'message' => 'Invalid review ID'
    ]);
    exit();
}

try {
    // Check if user has already voted on this review
    $existing_vote_check = $conn->prepare("SELECT like_id, is_helpful FROM review_likes WHERE review_id = ? AND user_id = ?");
    $existing_vote_check->bind_param("ii", $review_id, $user_id);
    $existing_vote_check->execute();
    $existing_result = $existing_vote_check->get_result();
    $existing_vote = $existing_result->fetch_assoc();
    $existing_vote_check->close();
    
    if ($existing_vote) {
        // Remove existing vote (toggle off)
        $delete_vote = $conn->prepare("DELETE FROM review_likes WHERE review_id = ? AND user_id = ?");
        $delete_vote->bind_param("ii", $review_id, $user_id);
        $delete_vote->execute();
        $delete_vote->close();
        $message = 'Vote removed';
    } else {
        // Add new helpful vote
        $insert_vote = $conn->prepare("INSERT INTO review_likes (review_id, user_id, is_helpful, created_at) VALUES (?, ?, TRUE, NOW())");
        $insert_vote->bind_param("ii", $review_id, $user_id);
        $insert_vote->execute();
        $insert_vote->close();
        $message = 'Marked as helpful';
    }
    
    // Get updated helpful count
    $count_query = $conn->prepare("SELECT COUNT(*) as helpful_count FROM review_likes WHERE review_id = ? AND is_helpful = TRUE");
    $count_query->bind_param("i", $review_id);
    $count_query->execute();
    $count_result = $count_query->get_result();
    $count_data = $count_result->fetch_assoc();
    $helpful_count = $count_data['helpful_count'];
    $count_query->close();
    
    // Update the review's helpful_count field
    $update_review = $conn->prepare("UPDATE course_reviews SET helpful_count = ? WHERE review_id = ?");
    $update_review->bind_param("ii", $helpful_count, $review_id);
    $update_review->execute();
    $update_review->close();
    
    echo json_encode([
        'success' => true,
        'message' => $message,
        'helpful_count' => $helpful_count
    ]);
    
} catch (Exception $e) {
    error_log("Helpful vote error: " . $e->getMessage());
    echo json_encode([
        'success' => false,
        'message' => 'An error occurred while processing your vote'
    ]);
}

$conn->close();
?>
