<?php
/**
 * Unenroll Course API Endpoint
 * Handles student unenrollment from courses
 */

header('Content-Type: application/json');

// Start session and check authentication
require_once '../../includes/session_start.php';
require_once '../../auth/functions.php';

// Initialize response
$response = [
    'success' => false,
    'message' => ''
];

try {
    // Ensure only students can access this
    if (!isLoggedIn() || !hasRole('student')) {
        throw new Exception('Authentication required');
    }

    // Get JSON input
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input || !isset($input['course_id'])) {
        throw new Exception('Invalid request data');
    }

    $courseId = intval($input['course_id']);
    $studentId = $_SESSION['user_id'];

    if ($courseId <= 0 || $studentId <= 0) {
        throw new Exception('Invalid course or student ID');
    }

    // Connect to database
    require_once '../../config/db_connect.php';
    
    if (!isset($conn)) {
        throw new Exception('Database connection failed');
    }

    // Check if student is enrolled in this course
    $checkQuery = "SELECT enrollment_id FROM enrollments WHERE student_id = ? AND course_id = ?";
    $checkStmt = $conn->prepare($checkQuery);
    $checkStmt->bind_param("ii", $studentId, $courseId);
    $checkStmt->execute();
    $checkResult = $checkStmt->get_result();

    if ($checkResult->num_rows === 0) {
        throw new Exception('You are not enrolled in this course');
    }

    // Delete enrollment record
    $deleteQuery = "DELETE FROM enrollments WHERE student_id = ? AND course_id = ?";
    $deleteStmt = $conn->prepare($deleteQuery);
    $deleteStmt->bind_param("ii", $studentId, $courseId);

    if ($deleteStmt->execute()) {
        $response['success'] = true;
        $response['message'] = 'Successfully unenrolled from course';
    } else {
        throw new Exception('Failed to unenroll from course');
    }

} catch (Exception $e) {
    $response['message'] = $e->getMessage();
}

echo json_encode($response);
?>
