<?php
require_once __DIR__ . '/../auth/functions.php';
requireRole('student');

require_once __DIR__ . '/../config/db_connect.php';

$student_id = $_SESSION['user_id'];
$message = '';
$messageType = '';

// Handle form submission
if ($_POST) {
    if (isset($_POST['action'])) {
        if ($_POST['action'] === 'submit_assessment') {
            $assessment_id = intval($_POST['assessment_id']);
            
            // Verify this assessment belongs to the current student
            $verify_stmt = $conn->prepare("SELECT * FROM sita_assessments WHERE assessment_id = ? AND student_id = ? AND status = 'pending'");
            $verify_stmt->bind_param("ii", $assessment_id, $student_id);
            $verify_stmt->execute();
            $assessment = $verify_stmt->get_result()->fetch_assoc();
            
            if ($assessment) {
                // Process answers
                $all_answered = true;
                $answers_saved = 0;
                
                // Get all questions for this assessment
                $questions_stmt = $conn->prepare("SELECT question_id FROM sita_assessment_questions WHERE assessment_id = ? ORDER BY question_order");
                $questions_stmt->bind_param("i", $assessment_id);
                $questions_stmt->execute();
                $questions_result = $questions_stmt->get_result();
                
                while ($question = $questions_result->fetch_assoc()) {
                    $question_id = $question['question_id'];
                    $answer_key = "answer_" . $question_id;
                    
                    if (isset($_POST[$answer_key]) && !empty(trim($_POST[$answer_key]))) {
                        // Save or update answer
                        $answer_stmt = $conn->prepare("INSERT INTO sita_assessment_answers (assessment_id, question_id, student_id, answer_text) VALUES (?, ?, ?, ?) ON DUPLICATE KEY UPDATE answer_text = VALUES(answer_text), submitted_date = NOW()");
                        $answer_text = trim($_POST[$answer_key]);
                        $answer_stmt->bind_param("iiis", $assessment_id, $question_id, $student_id, $answer_text);
                        
                        if ($answer_stmt->execute()) {
                            $answers_saved++;
                        }
                    } else {
                        $all_answered = false;
                    }
                }
                
                if ($all_answered && $answers_saved > 0) {
                    // Update assessment status to submitted
                    $update_stmt = $conn->prepare("UPDATE sita_assessments SET status = 'submitted', submitted_date = NOW() WHERE assessment_id = ?");
                    $update_stmt->bind_param("i", $assessment_id);
                    $update_stmt->execute();
                    
                    // Update accreditation request status
                    $status_stmt = $conn->prepare("UPDATE accreditation_requests SET status = 'assessment_completed' WHERE request_id = ?");
                    $status_stmt->bind_param("i", $assessment['accreditation_request_id']);
                    $status_stmt->execute();
                    
                    // Log audit trail
                    $audit_stmt = $conn->prepare("INSERT INTO sita_assessment_audit_log (assessment_id, student_id, action, details) VALUES (?, ?, 'assessment_submitted', 'Student completed and submitted SITA assessment')");
                    $audit_stmt->bind_param("ii", $assessment_id, $student_id);
                    $audit_stmt->execute();
                    
                    $message = 'Assessment submitted successfully! You will be notified about the next steps.';
                    $messageType = 'success';
                } else {
                    $message = 'Please answer all questions before submitting.';
                    $messageType = 'error';
                }
            } else {
                $message = 'Invalid assessment or assessment already submitted.';
                $messageType = 'error';
            }
        }
    }
}

// Get student's SITA assessments
$assessments_query = "SELECT sa.*, 
                             c.title as course_title,
                             ar.request_date,
                             ar.status as request_status
                      FROM sita_assessments sa
                      JOIN accreditation_requests ar ON sa.accreditation_request_id = ar.request_id
                      JOIN courses c ON sa.course_id = c.course_id
                      WHERE sa.student_id = ?
                      ORDER BY sa.created_date DESC";

$assessments_stmt = $conn->prepare($assessments_query);
$assessments_stmt->bind_param("i", $student_id);
$assessments_stmt->execute();
$assessments_result = $assessments_stmt->get_result();
$assessments = [];

while ($row = $assessments_result->fetch_assoc()) {
    $assessments[] = $row;
}

// Get current assessment if viewing/editing one
$current_assessment = null;
$assessment_questions = [];
$student_answers = [];

if (isset($_GET['id'])) {
    $assessment_id = intval($_GET['id']);
    
    // Get assessment details
    $current_stmt = $conn->prepare("SELECT sa.*, c.title as course_title FROM sita_assessments sa JOIN courses c ON sa.course_id = c.course_id WHERE sa.assessment_id = ? AND sa.student_id = ?");
    $current_stmt->bind_param("ii", $assessment_id, $student_id);
    $current_stmt->execute();
    $current_assessment = $current_stmt->get_result()->fetch_assoc();
    
    if ($current_assessment) {
        // Get questions
        $questions_stmt = $conn->prepare("SELECT * FROM sita_assessment_questions WHERE assessment_id = ? ORDER BY question_order");
        $questions_stmt->bind_param("i", $assessment_id);
        $questions_stmt->execute();
        $questions_result = $questions_stmt->get_result();
        
        while ($question = $questions_result->fetch_assoc()) {
            $assessment_questions[] = $question;
        }
        
        // Get existing answers
        $answers_stmt = $conn->prepare("SELECT question_id, answer_text FROM sita_assessment_answers WHERE assessment_id = ? AND student_id = ?");
        $answers_stmt->bind_param("ii", $assessment_id, $student_id);
        $answers_stmt->execute();
        $answers_result = $answers_stmt->get_result();
        
        while ($answer = $answers_result->fetch_assoc()) {
            $student_answers[$answer['question_id']] = $answer['answer_text'];
        }
    }
}

function getStatusBadgeClass($status) {
    switch ($status) {
        case 'pending': return 'bg-warning';
        case 'in_progress': return 'bg-info';
        case 'submitted': return 'bg-primary';
        case 'graded': return 'bg-secondary';
        case 'passed': return 'bg-success';
        case 'failed': return 'bg-danger';
        default: return 'bg-secondary';
    }
}

function getStatusText($status) {
    switch ($status) {
        case 'pending': return 'Ready to Start';
        case 'in_progress': return 'In Progress';
        case 'submitted': return 'Submitted';
        case 'graded': return 'Graded';
        case 'passed': return 'Passed';
        case 'failed': return 'Failed';
        default: return ucfirst($status);
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= $current_assessment ? 'SITA Assessment - ' . htmlspecialchars($current_assessment['course_title']) : 'My SITA Assessments' ?> - Panadite Academy</title>
    
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css">
    <link href='https://unpkg.com/boxicons@2.1.4/css/boxicons.min.css' rel='stylesheet'>
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    
    <style>
        body {
            font-family: 'Poppins', sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
        }
        
        .main-container {
            padding: 2rem 0;
        }
        
        .assessment-card {
            background: white;
            border-radius: 16px;
            box-shadow: 0 8px 32px rgba(0, 0, 0, 0.1);
            overflow: hidden;
            margin-bottom: 2rem;
            transition: all 0.3s ease;
        }
        
        .assessment-card:hover {
            transform: translateY(-4px);
            box-shadow: 0 12px 40px rgba(0, 0, 0, 0.15);
        }
        
        .assessment-header {
            background: linear-gradient(135deg, #2563eb 0%, #1d4ed8 100%);
            color: white;
            padding: 2rem;
            text-align: center;
        }
        
        .assessment-body {
            padding: 2rem;
        }
        
        .question-card {
            background: #f8fafc;
            border: 2px solid #e2e8f0;
            border-radius: 12px;
            padding: 1.5rem;
            margin-bottom: 1.5rem;
            transition: all 0.3s ease;
        }
        
        .question-card:focus-within {
            border-color: #2563eb;
            box-shadow: 0 0 0 3px rgba(37, 99, 235, 0.1);
        }
        
        .question-number {
            background: #2563eb;
            color: white;
            width: 30px;
            height: 30px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: 600;
            margin-bottom: 1rem;
        }
        
        .form-control {
            border: 2px solid #e2e8f0;
            border-radius: 8px;
            padding: 0.75rem;
            font-size: 0.95rem;
            line-height: 1.6;
        }
        
        .form-control:focus {
            border-color: #2563eb;
            box-shadow: 0 0 0 3px rgba(37, 99, 235, 0.1);
        }
        
        .word-count {
            font-size: 0.8rem;
            color: #64748b;
            text-align: right;
            margin-top: 0.5rem;
        }
        
        .progress-indicator {
            background: #f1f5f9;
            border-radius: 10px;
            padding: 1rem;
            margin-bottom: 2rem;
        }
        
        .progress-bar {
            height: 8px;
            border-radius: 4px;
        }
        
        .btn-primary {
            background: linear-gradient(135deg, #2563eb 0%, #1d4ed8 100%);
            border: none;
            border-radius: 8px;
            padding: 0.75rem 2rem;
            font-weight: 600;
            transition: all 0.3s ease;
        }
        
        .btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(37, 99, 235, 0.3);
        }
        
        .status-badge {
            font-size: 0.8rem;
            padding: 0.5rem 1rem;
            border-radius: 20px;
            font-weight: 600;
        }
        
        .assessment-info {
            background: #f0f9ff;
            border: 2px solid #0ea5e9;
            border-radius: 12px;
            padding: 1.5rem;
            margin-bottom: 2rem;
        }
        
        .assessment-info h5 {
            color: #0369a1;
            margin-bottom: 1rem;
        }
        
        .back-link {
            color: #2563eb;
            text-decoration: none;
            font-weight: 500;
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
            margin-bottom: 2rem;
        }
        
        .back-link:hover {
            color: #1d4ed8;
        }
    </style>
</head>
<body>
    <?php include 'components/header.php'; ?>
    
    <div class="main-container">
        <div class="container">
            <?php if ($message): ?>
                <div class="alert alert-<?= $messageType === 'success' ? 'success' : 'danger' ?> alert-dismissible fade show" role="alert">
                    <i class="bx <?= $messageType === 'success' ? 'bx-check-circle' : 'bx-error-circle' ?> me-2"></i>
                    <?= htmlspecialchars($message) ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
            <?php endif; ?>
            
            <?php if ($current_assessment): ?>
                <!-- Single Assessment View -->
                <a href="assessments.php" class="back-link">
                    <i class="bx bx-arrow-back"></i> Back to My Assessments
                </a>
                
                <div class="assessment-card">
                    <div class="assessment-header">
                        <h2><i class="bx bx-award me-2"></i><?= htmlspecialchars($current_assessment['title']) ?></h2>
                        <p class="mb-0 opacity-90"><?= htmlspecialchars($current_assessment['course_title']) ?></p>
                        <span class="badge <?= getStatusBadgeClass($current_assessment['status']) ?> mt-2">
                            <?= getStatusText($current_assessment['status']) ?>
                        </span>
                    </div>
                    
                    <div class="assessment-body">
                        <?php if ($current_assessment['status'] === 'submitted' || $current_assessment['status'] === 'graded' || $current_assessment['status'] === 'passed' || $current_assessment['status'] === 'failed'): ?>
                            <!-- Assessment already submitted - show read-only view -->
                            <div class="assessment-info">
                                <h5><i class="bx bx-info-circle me-2"></i>Assessment Status</h5>
                                <p class="mb-2"><strong>Status:</strong> <?= getStatusText($current_assessment['status']) ?></p>
                                <p class="mb-2"><strong>Submitted:</strong> <?= date('F j, Y g:i A', strtotime($current_assessment['submitted_date'])) ?></p>
                                <?php if ($current_assessment['score']): ?>
                                    <p class="mb-2"><strong>Score:</strong> <?= $current_assessment['score'] ?>%</p>
                                <?php endif; ?>
                                <?php if ($current_assessment['feedback']): ?>
                                    <p class="mb-0"><strong>Feedback:</strong> <?= nl2br(htmlspecialchars($current_assessment['feedback'])) ?></p>
                                <?php endif; ?>
                            </div>
                            
                            <h4>Your Submitted Answers</h4>
                            <?php foreach ($assessment_questions as $index => $question): ?>
                                <div class="question-card">
                                    <div class="question-number"><?= $index + 1 ?></div>
                                    <h6 class="mb-3"><?= nl2br(htmlspecialchars($question['question_text'])) ?></h6>
                                    <div class="bg-white p-3 border rounded">
                                        <strong>Your Answer:</strong><br>
                                        <?= nl2br(htmlspecialchars($student_answers[$question['question_id']] ?? 'No answer provided')) ?>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        <?php else: ?>
                            <!-- Assessment form for completion -->
                            <div class="assessment-info">
                                <h5><i class="bx bx-info-circle me-2"></i>Assessment Instructions</h5>
                                <p><?= nl2br(htmlspecialchars($current_assessment['description'])) ?></p>
                                <ul class="mb-0">
                                    <li>Answer all questions thoroughly and professionally</li>
                                    <li>Each question has a <?= $assessment_questions[0]['max_words'] ?? 500 ?> word limit</li>
                                    <li>You can save your progress and return later</li>
                                    <li>Make sure to submit when complete</li>
                                    <li>You need <?= $current_assessment['passing_score'] ?>% to pass</li>
                                </ul>
                            </div>
                            
                            <form method="POST" id="assessmentForm">
                                <input type="hidden" name="action" value="submit_assessment">
                                <input type="hidden" name="assessment_id" value="<?= $current_assessment['assessment_id'] ?>">
                                
                                <?php foreach ($assessment_questions as $index => $question): ?>
                                    <div class="question-card">
                                        <div class="question-number"><?= $index + 1 ?></div>
                                        <h6 class="mb-3"><?= nl2br(htmlspecialchars($question['question_text'])) ?></h6>
                                        <textarea 
                                            name="answer_<?= $question['question_id'] ?>" 
                                            class="form-control" 
                                            rows="8" 
                                            placeholder="Enter your detailed answer here..."
                                            maxlength="<?= $question['max_words'] * 7 ?>"
                                            data-max-words="<?= $question['max_words'] ?>"
                                            required
                                        ><?= htmlspecialchars($student_answers[$question['question_id']] ?? '') ?></textarea>
                                        <div class="word-count">
                                            <span class="current-words">0</span> / <?= $question['max_words'] ?> words
                                        </div>
                                    </div>
                                <?php endforeach; ?>
                                
                                <div class="d-flex justify-content-between align-items-center mt-4">
                                    <div class="form-check">
                                        <input class="form-check-input" type="checkbox" id="confirmSubmit" required>
                                        <label class="form-check-label" for="confirmSubmit">
                                            I confirm that my answers are complete and ready for submission
                                        </label>
                                    </div>
                                    <button type="submit" class="btn btn-primary btn-lg">
                                        <i class="bx bx-send me-2"></i>Submit Assessment
                                    </button>
                                </div>
                            </form>
                        <?php endif; ?>
                    </div>
                </div>
            <?php else: ?>
                <!-- Assessments List View -->
                <div class="row">
                    <div class="col-12">
                        <div class="assessment-card">
                            <div class="assessment-header">
                                <h2><i class="bx bx-award me-2"></i>My SITA Assessments</h2>
                                <p class="mb-0 opacity-90">Complete your accreditation assessments below</p>
                            </div>
                            
                            <div class="assessment-body">
                                <?php if (count($assessments) > 0): ?>
                                    <div class="row">
                                        <?php foreach ($assessments as $assessment): ?>
                                            <div class="col-md-6 col-lg-4 mb-4">
                                                <div class="card h-100">
                                                    <div class="card-body">
                                                        <h5 class="card-title"><?= htmlspecialchars($assessment['course_title']) ?></h5>
                                                        <p class="card-text"><?= htmlspecialchars(substr($assessment['description'], 0, 100)) ?>...</p>
                                                        <div class="mb-3">
                                                            <span class="badge <?= getStatusBadgeClass($assessment['status']) ?>">
                                                                <?= getStatusText($assessment['status']) ?>
                                                            </span>
                                                        </div>
                                                        <div class="small text-muted mb-3">
                                                            <div><i class="bx bx-calendar me-1"></i>Created: <?= date('M j, Y', strtotime($assessment['created_date'])) ?></div>
                                                            <?php if ($assessment['due_date']): ?>
                                                                <div><i class="bx bx-time me-1"></i>Due: <?= date('M j, Y', strtotime($assessment['due_date'])) ?></div>
                                                            <?php endif; ?>
                                                            <?php if ($assessment['score']): ?>
                                                                <div><i class="bx bx-trophy me-1"></i>Score: <?= $assessment['score'] ?>%</div>
                                                            <?php endif; ?>
                                                        </div>
                                                        <?php if ($assessment['status'] === 'pending' || $assessment['status'] === 'in_progress'): ?>
                                                            <a href="assessments.php?id=<?= $assessment['assessment_id'] ?>" class="btn btn-primary">
                                                                <i class="bx bx-edit me-1"></i>
                                                                <?= $assessment['status'] === 'pending' ? 'Start Assessment' : 'Continue Assessment' ?>
                                                            </a>
                                                        <?php else: ?>
                                                            <a href="assessments.php?id=<?= $assessment['assessment_id'] ?>" class="btn btn-outline-primary">
                                                                <i class="bx bx-show me-1"></i>View Assessment
                                                            </a>
                                                        <?php endif; ?>
                                                    </div>
                                                </div>
                                            </div>
                                        <?php endforeach; ?>
                                    </div>
                                <?php else: ?>
                                    <div class="text-center py-5">
                                        <i class="bx bx-award" style="font-size: 4rem; color: #cbd5e1;"></i>
                                        <h4 class="text-muted mt-3">No SITA Assessments Available</h4>
                                        <p class="text-muted">Submit an accreditation request from your course page to get started.</p>
                                        <a href="dashboard.php" class="btn btn-primary">
                                            <i class="bx bx-home me-2"></i>Go to Dashboard
                                        </a>
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                </div>
            <?php endif; ?>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Word count functionality
        document.addEventListener('DOMContentLoaded', function() {
            const textareas = document.querySelectorAll('textarea[data-max-words]');
            
            textareas.forEach(textarea => {
                const maxWords = parseInt(textarea.dataset.maxWords);
                const wordCountDisplay = textarea.parentNode.querySelector('.current-words');
                
                function updateWordCount() {
                    const text = textarea.value.trim();
                    const words = text === '' ? 0 : text.split(/\s+/).length;
                    wordCountDisplay.textContent = words;
                    
                    // Color coding
                    const parent = wordCountDisplay.parentNode;
                    if (words > maxWords) {
                        parent.style.color = '#dc2626';
                        parent.innerHTML = `<strong>${words} / ${maxWords} words (${words - maxWords} over limit)</strong>`;
                    } else if (words > maxWords * 0.8) {
                        parent.style.color = '#f59e0b';
                        parent.innerHTML = `${words} / ${maxWords} words`;
                    } else {
                        parent.style.color = '#64748b';
                        parent.innerHTML = `${words} / ${maxWords} words`;
                    }
                }
                
                textarea.addEventListener('input', updateWordCount);
                updateWordCount(); // Initial count
            });
            
            // Form submission validation
            const form = document.getElementById('assessmentForm');
            if (form) {
                form.addEventListener('submit', function(e) {
                    const textareas = form.querySelectorAll('textarea[required]');
                    let hasOverLimit = false;
                    
                    textareas.forEach(textarea => {
                        const maxWords = parseInt(textarea.dataset.maxWords);
                        const text = textarea.value.trim();
                        const words = text === '' ? 0 : text.split(/\s+/).length;
                        
                        if (words > maxWords) {
                            hasOverLimit = true;
                        }
                    });
                    
                    if (hasOverLimit) {
                        e.preventDefault();
                        alert('Please ensure all answers are within the word limit before submitting.');
                        return false;
                    }
                    
                    // Confirmation dialog
                    if (!confirm('Are you sure you want to submit this assessment? You cannot modify your answers after submission.')) {
                        e.preventDefault();
                        return false;
                    }
                });
            }
        });
    </script>
</body>
</html>
