/**
 * Panadite Academy - Dashboard Animations and Interactions
 */

document.addEventListener('DOMContentLoaded', function() {
    // Initialize animations for elements as they come into view
    initializeAnimations();
    
    // Initialize counters for stat numbers
    initializeCounters();
    
    // Add hover effects to cards
    initializeCardEffects();
    
    // Initialize any tooltips
    initializeTooltips();
});

/**
 * Initialize animations for elements as they scroll into view
 */
function initializeAnimations() {
    // Select all elements that should animate when scrolled into view
    const animatedElements = document.querySelectorAll('.stat-card, .course-card, .schedule-card, .activity-item');
    
    // Create an intersection observer
    const observer = new IntersectionObserver((entries) => {
        entries.forEach(entry => {
            // Add animation class when element comes into view
            if (entry.isIntersecting) {
                entry.target.style.opacity = '1';
                entry.target.style.transform = 'translateY(0)';
                observer.unobserve(entry.target);
            }
        });
    }, {
        threshold: 0.1,
        rootMargin: '0px 0px -50px 0px'
    });
    
    // Observe each element
    animatedElements.forEach(el => {
        observer.observe(el);
    });
}

/**
 * Initialize counters for stat numbers to count up
 */
function initializeCounters() {
    const statNumbers = document.querySelectorAll('.stat-number');
    
    statNumbers.forEach(statNumber => {
        const finalValue = parseInt(statNumber.textContent);
        let startValue = 0;
        const duration = 1000; // ms
        const frameRate = 30; // frames per second
        const totalFrames = duration / (1000 / frameRate);
        const incrementPerFrame = finalValue / totalFrames;
        let currentFrame = 0;
        
        const countUp = setInterval(() => {
            currentFrame++;
            const currentValue = Math.round(incrementPerFrame * currentFrame);
            
            if (currentValue >= finalValue) {
                statNumber.textContent = finalValue;
                clearInterval(countUp);
                return;
            }
            
            statNumber.textContent = currentValue;
        }, 1000 / frameRate);
    });
}

/**
 * Add hover effects to cards
 */
function initializeCardEffects() {
    const cards = document.querySelectorAll('.card');
    
    cards.forEach(card => {
        card.addEventListener('mouseenter', function() {
            this.style.transform = 'translateY(-5px)';
            this.style.boxShadow = '0 10px 30px rgba(0, 0, 0, 0.1)';
            this.style.transition = 'all 0.3s ease';
        });
        
        card.addEventListener('mouseleave', function() {
            this.style.transform = '';
            this.style.boxShadow = '';
        });
    });
}

/**
 * Initialize tooltips for better UX
 */
function initializeTooltips() {
    const tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
    if (typeof bootstrap !== 'undefined') {
        tooltipTriggerList.map(function (tooltipTriggerEl) {
            return new bootstrap.Tooltip(tooltipTriggerEl);
        });
    }
}

/**
 * Update the dashboard data in real-time (for demo purposes)
 */
function initializeRealtimeUpdates() {
    // This could be used for websocket connections to update dashboard data
    // in real-time without page refresh
    console.log('Real-time dashboard updates initialized');
}
